// Copyright 2024 Christian Mazakas.
// Distributed under the Boost Software License, Version 1.0.
// https://www.boost.org/LICENSE_1_0.txt

#define _CRT_SECURE_NO_WARNINGS

#ifdef _MSC_VER
# pragma warning(disable: 4309) // truncation of constant value
#endif

#include <boost/hash2/sha2.hpp>
#include <boost/core/lightweight_test.hpp>
#include <cstdint>
#include <string>
#include <cstddef>
#include <cstdio>
#include <vector>
#include <tuple>

std::string from_hex( char const* str )
{
    auto f = []( char c ) { return ( c >= 'a' ? c - 'a' + 10 : c - '0' ); };

    std::string s;
    while( *str != '\0' )
    {
        s.push_back( static_cast<char>( ( f( str[ 0 ] ) << 4 ) + f( str[ 1 ] ) ) );
        str += 2;
    }
    return s;
}

template<class H> std::string digest( std::string const & k, std::string const & s )
{
    H h( k.data(), k.size() );

    h.update( s.data(), s.size() );

    return to_string( h.result() );
}

static void hmac_sha_256()
{
    using boost::hash2::hmac_sha2_256;

    // Test vectors from https://en.wikipedia.org/wiki/Hash-based_message_authentication_code
    BOOST_TEST_EQ( digest<hmac_sha2_256>( "key", "The quick brown fox jumps over the lazy dog" ), std::string( "f7bc83f430538424b13298e6aa6fb143ef4d59a14946175997479dbc2d1a3cd8" ) );

    // Test vectors from https://www.rfc-editor.org/rfc/rfc4868#section-2.7
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b"), "Hi There" ), std::string( "b0344c61d8db38535ca8afceaf0bf12b881dc200c9833da726e9376c2e32cff7" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( "Jefe", "what do ya want for nothing?" ), std::string( "5bdcc146bf60754e6a042426089575c75a003f089d2739839dec58b964ec3843" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), from_hex( "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd" ) ), std::string( "773ea91e36800e46854db8ebd09181a72959098b3ef8c122d9635514ced565fe" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "0102030405060708090a0b0c0d0e0f10111213141516171819"), from_hex( "cdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcd" ) ), std::string( "82558a389a443c0ea4cc819899f2083a85f0faa3e578f8077a2e3ff46729665b" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), std::string( "Test Using Larger Than Block-Size Key - Hash Key First" ) ), std::string( "60e431591ee0b67f0d8a26aacbf5b77f8e0bc6213728c5140546040f0ee37f54" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), std::string( "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm." ) ), std::string( "9b09ffa71b942fcb27635fbcd5b0e944bfdc63644f0713938a7f51535c3a35e2" ) );

    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b"), "Hi There" ), std::string( "198a607eb44bfbc69903a0f1cf2bbdc5ba0aa3f3d9ae3c1c7a3b1696a0b68cf7" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( "JefeJefeJefeJefeJefeJefeJefeJefe", "what do ya want for nothing?" ), std::string( "167f928588c5cc2eef8e3093caa0e87c9ff566a14794aa61648d81621a2a40c6" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), from_hex( "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd" ) ), std::string( "cdcb1220d1ecccea91e53aba3092f962e549fe6ce9ed7fdc43191fbde45c30b0" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20" ), from_hex( "cdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcd" ) ), std::string( "372efcf9b40b35c2115b1346903d2ef42fced46f0846e7257bb156d3d7b30d3f" ) );

    // use selected samples from the download available here: https://csrc.nist.gov/Projects/cryptographic-algorithm-validation-program/Message-Authentication
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "9779d9120642797f1747025d5b22b7ac607cab08e1758f2f3a46c8be1e25c53b8c6a8f58ffefa176" ), from_hex( "b1689c2591eaf3c9e66070f8a77954ffb81749f1b00346f9dfe0b2ee905dcc288baf4a92de3f4001dd9f44c468c3d07d6c6ee82faceafc97c2fc0fc0601719d2dcd0aa2aec92d1b0ae933c65eb06a03c9c935c2bad0459810241347ab87e9f11adb30415424c6c7f5f22a003b8ab8de54f6ded0e3ab9245fa79568451dfa258e" ) ), std::string( "769f00d3e6a6cc1fb426a14a4f76c6462e6149726e0dee0ec0cf97a16605ac8b" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "09675f2dcc4783b599f18fb765583668a0fd8ae4096f6fcdc60d4f35b4130fbefcd542ffe7459d2a" ), from_hex( "0cf2198c31376f5c8915660137725f2bbc180a986e5a7bda27fa81593a4a339bab92cbc39fb2b8581108ee48c794812d845a72ce8008c9e915d9e330bbb90e9136aa53ba0e6693dd4046d6b03362dfb9edfa04c887153cc5de677aab8c7839d517035879679c29727e96c5426324a2575fbe678d6cc7fef5eb6cebd595cfddef" ) ), std::string( "6b142d4dfe217f1881aa0e6483b271dd5d43f70b85605953a0fef272ddde46ca" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "b763263dc4fc62b227cd3f6b4e9e358c21ca036ce396ab9259c1bedd2f5cd90297dc703c336eca3e358a4d6dc5" ), from_hex( "53cb09d0a788e4466d01588df6945d8728d9363f76cd012a10308dad562b6be093364892e8397a8d86f1d81a2096cfc8a1bbb26a1a75525ffebfcf16911dadd09e802aa8686acfd1e4524620254a6bca18dfa56e71417756e5a452fa9ae5aec5dc71591c11630e9defec49a4ecf85a14f60eb854657899972ea5bf6159cb9547" ) ), std::string( "737301dea93db6bcbadd7bf796693961317ca680b380416f12f466f06526b36b" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "9fe42dfac92a4a136fa7c9f6e331b5d3a61aa73035b53a8d2517be43721b31b215a96b9bd43798cb5e8febfa97" ), from_hex( "f9660fb784c14b5fbec280526a69c2294fba12aea163789bbe9f52a51b5aebb97d964f866c0d5e3be41820924fcf580db0725c7f210823cf7f45a0f964b14e5555070d1c3ddb2c281a80c7fbf72953031a4e771d7e521d578462cafae5a02ac8eb81f082e173ddadc8c41d964bbfda94f5180c8da28a8ebb33be77b0866fa798" ) ), std::string( "7786c155d10c741b63ec650b7b1aa3bfd71ac71881ad06ae98fb082f17e0caa0" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "2914da23e86a603cda1eede153be2431c2947cdaeed6a1ea801d18e2c218220ca682e40f0a51c4c13a31163cb730f83437bb7a88ecc903160956f0d483137d1d145ce948866ad57f2eca" ), from_hex( "6b8db9acdfd24150808a92368596557181d445e5a04e91112db2812b58035d72378d8bc00a1ef75ec373b81dc6f1f0a2ed96f302cf2eac8f42ca3df11e6ee678440a28b0dfab2a36eaf35bcbf3c759a71e47120f6c03292a3d6b9b111488a2259bead9a5e7e2a180fcf1c467947f59271cd0e8360035ce8b287fe2b3c3b95822" ) ), std::string( "4de7bab7fe9a0a9bf7b51a7cdf7d929f2b1c6ff4575fd527baba1efdf4254890" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_256>( from_hex( "4b7ab133efe99e02fc89a28409ee187d579e774f4cba6fc223e13504e3511bef8d4f638b9aca55d4a43b8fbd64cf9d74dcc8c9e8d52034898c70264ea911a3fd70813fa73b083371289b" ), from_hex( "138efc832c64513d11b9873c6fd4d8a65dbf367092a826ddd587d141b401580b798c69025ad510cff05fcfbceb6cf0bb03201aaa32e423d5200925bddfadd418d8e30e18050eb4f0618eb9959d9f78c1157d4b3e02cd5961f138afd57459939917d9144c95d8e6a94c8f6d4eef3418c17b1ef0b46c2a7188305d9811dccb3d99" ) ), std::string( "4f1ee7cb36c58803a8721d4ac8c4cf8cae5d8832392eed2a96dc59694252801b" ) );
}

static void hmac_sha_224()
{
    using boost::hash2::hmac_sha2_224;

    // use selected samples from the download available here: https://csrc.nist.gov/Projects/cryptographic-algorithm-validation-program/Message-Authentication
    // note: must use `std::make_tuple()` for gcc-5, gcc-4.8 because of https://www.open-std.org/jtc1/sc22/wg21/docs/papers/2015/n4387.html
    std::vector<std::tuple<char const*, char const*, char const*>> inputs =
    {                    /* expected mac */                                          /* key */                                                                                               /* msg */
        std::make_tuple( "5507c4c7c127046c04d161fdc75ec2f5122ed44e86679c39c69c5ae8", "82a289b0911f55c532e2571866fa354d973948ec7b899d573b833f00d53eb8d8bf65d50bcd8daaea54c5d628906e084a434d", "d06d296cad7d428b56b25c53fdfcf558b5bc4211ed31cfe63732678a4f23338b582225286881986247793056741c08c9698c0c05206ba3e4d692922a0f061d17da276e3d562b3b9075cba4bc003c5abbdc6106a68b3fe9cbf5b1bf01695dea38dfe6cc548064753c68117e2daa44345594a4fa9d3595e8c61df7b8b76410e315" ),
        std::make_tuple( "3613be648e6da86b363598f8d3c622239c70374b154e9758be6ac9da", "5ed770cab9aa8cf1107dcb4fed42a48832e6a30d5b83c8f003d62af7a9f3dfefb5372042b67fa9ccee975feb10f635d548a8", "5c42ded57c112cfd1365227f99e993790bb7fef88e6e57b056840af1c6de3ed6d89e56c0c03dbe02ccd9e4de8fd084b13e8d296a5aa2f1408f1790a5dd029775f558f774e603c0f7f8352fe1d8fbe8d1f615809a2c053ef4a0662371eb7f27b27b18c6da358cbb435f200c70be934643fbf70a704cbfc8325018cb3d3a7d76f3" ),
        std::make_tuple( "9d0e55b33e5b72be9f073e5f4ef4bd8a46d8c18a128ec33a09a536f1", "e8ed22a9cea119cb930ddca9c2675ac8dafd3b7fea7616585e5be5b819d6b6c3c4b9a37a65c10326331de2faa92aaff4d981", "3d2edc42edc46ae84ef03cb26b60d56bfbc0c4af36391e96e7372b93c370609bf9228d890a9b58c1dbffea130946e8e37c84a5873f6d96f18e18068e31030a6d0958b8a4e96e94f0a7934b9b7209a4a3654eb3185da050c608c1c7143ff8d12c9658c35c314406bcd933c7124eb1eff2dfa7562a39db335d7dbad8ed95569c1e" ),
        std::make_tuple( "14e6f0b573b44c2899b9fbb916b3f0c3b44fced9088fede4add62cbf", "faa2e1ec2658570a8d1b60376f7f5241ed72781cbe1c61e1230a7e171d8ced32dd5b8f597c7f606cec7008462f7cad3fc668", "db0b6f188d114902a58112a506c653f52099b0955cd56d49ba8731aeb8450a2977c2b21d199f11616d8302f1ce6d5ae34849279dfdc23546c758d77a1fb2e3423e35ff4381d51b35122b3f97c5c887a140ca6c7b5fad44acd7630cd519a6669f798ddef28f5d7168dda25001b37fe118504307954a3ca36530d05b276dc99a28" ),
        std::make_tuple( "1f7e48a7c46fd9fc87160d68914a52cb4032e671fba07dcc8f684d84", "167b50360501077b3b91f9f612716bad0d661c2d0f191d17cabfe3db79f99bc4d2f9bd3536e7e4322c83dd210d0bfdd2015f", "07c15fd37f8f92a60640f55a8819868d81bc37f41eb137959f467236ff995b15d7f3e9d98057dfc9775b513c06376450211c55df8fc7bdaf4a5513e9275972cdb92c9ca3738440f50e4878fb2aa396573858200397551987fd8a826f19fb21fc7714811aa33d3a9291af72813c86a4098e17a1468083d483828dc9ae1da8f10a" ),
        std::make_tuple( "1c4944843c3ee7f98ab52987e0d2d4494aa72548bfdc3602db45b5f4", "b34a6eba59e632e3c334df582fd0b03aca7f641c6914920a79943dca3998a86127f36bdab795c7424baf37f76018472305f0a983928386", "d03dd94e43d96ca459928aa96d2b81c35ed54566e33e66351fca406b7620727a9ee991f2f9d41da322deb306c6d085d9b509080a387decd6a6fe513232bb386a07063708975a72f72f9cc6e8cfa147f53af1ed8449ca8a6f8468ff62f384b084321b3559c47053e7a9542a1733a5af5f15155d9ebd2c2847e491b3c26385c20f" ),
        std::make_tuple( "eabba9f35ba39cfb9283390d5425687cdd70d4cb1fea433925647c79", "2fd5863ab5ac0109cd1cc8a6598e75d85811a84d0df14cd55e8b1cce7a5f65dfbe670deadaa8d43b2f06da067c5c6210baccd5ac44540a", "85c02d7cfab29f8adf0fa55ef36722a04757c8865053d2af3ba2f64e80aa958aba6e3625b655325cca2db00f686fd422f2c53423d0c98c2dc110b20c6e67cca1455cc0888401ecf994ec18ec9982a8814776169ef78ca0dafaa33e9a2df2d779cd92b4ee8d3c3529e655c33daf270584ed72573fec23787e8f638240e4d320da" ),
        std::make_tuple( "9d2df0d7aebb6b08e816bef8df1bcf7d907622c617449346113dd317", "b67787d13434effcc68528109442b9418df025de59e5788e8fad1fda29adfceb3506ba95cdbc5e9548181f17b24a787c7a537b596b2268", "59282658eba93c8d2997b78c201d29700adac276c58668bc83d90a49c2a8324e7f52daaba65a81b5ce557e37230793b7a3c9c5fa8e2d6bca17e586f660e1509ba5b818a2b8494c8aada2d867d621ae2bf2d8a625a13689c71be468ac22ee33da3523874407bec6932f466d2521b2a2565dd10dffe6e9e93df69f5b74f0160293" ),
        std::make_tuple( "7d3cf647380947283bbc71d6a6f0f2aaeb1bd74a2cb0ec7b9a02c962", "112bb3c6f8d2e4f85235a7325d2cbb457336f9b9d3263619a4196b9237ab14bd8d6bd5b8198c637bc6dda2693cfb0b265fc451a9191044", "db2a58b070862fa950c80788b5cf8d5d443e31e492e5cda2e67c87314197f8d99bd68e60705c2e227b831e64c2ce23e5d220e6b64c2b9c4ca02ae6d419d3f6c61a87f4bda7cda08626fe387667429c01e3bab08941ff02741fccad810ef77d5cdea653701dbfcbfcdaa01d69ae1141dbfe7657067a4a4543e9c8a2e229ebd3c2" ),
        std::make_tuple( "b942bae8786a0addc6c61e27a776429b2a110fa913711b835aca58e4", "9dae3b463c86d64ce598ba9a567ca83783265e402d20c642b5176ee7339938d5c31d53d31f44692d735e1a5a6c30aadd92f5f669d9a2ad", "a7052e9a8f361a959f3f988e98e722a95c9e38f60a11a5a36bfb03708d3e8162ad32e4bc5a3dcc2bd8a43941c3a04d24f4b55488a93f01dadf4efabe03b398bcca5be7365edcf59eba4e49e6575cb137089a41e5221a98a3e13a65f307bf2ae1c62840f1e03c1c560de6d0649d42d4485ef807f7c0c9f16437674bf6e18ac139" ),
        std::make_tuple( "695c328dc858a46be7ac8b8debf58dd9a9bd72d7a408a43e6bbc69a3", "f3aa51ee90aa06e98e2388df7a3af2cd697f2a528f2a14140da405600b4a7b10076d0bb26c9d9ae66727d008f6dcca0f42140fb52ef3c6d930eb262160822de3", "2c070e5eb0a7efbfc40b234314c055fc436ce1e2300539b3784287fd3c4f947824c5e89aa3d933667dd4eb8587c33797ae6f0ccb3b8f95ad563dd940c8a7984d050775fa69d55b9ec23a19d40fc94fcf876a8eeddd96bb8ec3cab426ad353691d7b4ffa789012f119a28f45b333a6649d544b06f9d8271fcc7d162d7726338ed" ),
        std::make_tuple( "ee63e86aac85f36961d697451288cf780041eafe462ca6557035d92a", "a27f5ba4a0d5a80cb4ef9bb18d4ffc4ad487680cb5a8f6c69cfe1b293cfacb67d670e101b803c1d904dd3c8fb2a3b40ad8a78055296091adbc1856d810e5deb8", "802ed28873c2160562fb9162246b60d75e2ffb8cecc5bb08831e9aec94f43be1735e2acfb8a26450e64a1a644a7ab26626800f3e0fad12d9946360d620706682229a9b076a7b05e0d690ff902f8c39944ef2009683e2c0e33733c71fcfc34b38d40baf9851f7029f7df4c9509a4afe4a1dac58fa0f0dcd10b3b6562f6996f2e9" ),
        std::make_tuple( "5b25f50fd3ba3f628058d49d28cd6ba02fdda45d842d935ff9a8ead6", "489bd8b1eaa33c50e65f35208b752111d3896607e6b010e9e027bcf4d445b45611ae5bf8a4cacc9b1e75be2697f8ab3ef2383118632eb88b2f97556ab3edee10", "4b2b843db0ae2327dea304821a4d31bd65b55b52a34222f9fc89911d8366e88c154c9f7284d9a788f5aef389877d37e63663f0bafe79a40043a66b0470b7ae17dfb12f87f96549ce9e467a0ac7781ead69297d769d2408a0ffa3e059536598756f013c64557a92619f139fed20656d7cc8ae0c6ec86cd740f72bf804749b2f0e" ),
        std::make_tuple( "d6c24c2c68c3c879b141677702852fac78c88a39701de6bb8a466fee", "21556df80e1bba77a079a3f3c86ebcffdbcc209ee53c62604182c7ce6602b20804835fad5034d4fb9bec15382ab69850428df8bf8264edb1766a7c68688b6bdf", "908049137194295e174f2a0565dd5737dc8a5e3fb283416224e14f060de3531ab67b0bb1f00ddbf06073c32b1b448f4b73564d73108104e342a6a31c95f03844a65a62cd367209527d5c4cc1c019bbbf260ac748c8af769607b55c452230c6b4082538ae6a4b1a4a1512ae0f7fe5455c9facb307029600451c1560cadc2a6531" ),
        std::make_tuple( "929a3b33729126cea761fb2d23baf40757e77e0b3c5536dcceb68433", "f86fe94d5c6ac0accbb652d7eda9aa7559cf380f098ddbcf0a043f87783869f4aa729045aae32fd7e88958432f574b9ad79047ef3d74a016701e12ac2dad5557", "8cc53f92c6522670b1408bf15ef09e97131b7e12920ba1119e0b7558bc1c471237e9656966fd7f55313453b1d26f8a94d98cdcde73967b11eb38af637178bd111ba20ef529818d5b8dd55111ab027cbd5d0663733aadef87180843a0cf1b9e0e40d6835e9647182bcc0dcd3770103dbaa66f59f96fbac31b2b6278385d6db2b4" ),
        std::make_tuple( "0ae2eceff877842b3d77b4210d391f42151b696006c6974722f0bf46", "d83efe16354e973367648ce9c3127b6f7957b7da1914ed3456bf08cf095b51d879a8db2fd9971e761200e269c3975f6d861da26b76d332deee34f42b93e34c3294", "0df219910ce5b0bb03e05b21dd747a7413866d661ff4a66d347c02c42cd959b4c22e541101a87e07fc5f9e17c07e455c87a4a3132e8a9e5ae9e52638368ec2de0fe06f2c6461ab77c8a5cd8a6b7097ea481b0e5f9f3989afa67a09ad7ccfc8725ece15eb1d595e94cf7a8f82c5929ba7beb357525a4a8ff5d201542be5e393f9" ),
        std::make_tuple( "69269527fe9a121b9098c2472e1250577c66e1ef48a432059488f3ea", "b74637baa33d5a7443304b85aa7260f48c788b01008d397184c67926423961f2926aadadbf06aa850ed45396fce8b9bbc81aa8a331b68727817580dacaace1e270", "0c671581ca7648fc5717d4559b9e83b568a757af5d64872b0b35c4a25779c0911fa5f12721939341181cec9927e8d33a7b8573382db74d9d650f7ca6c5bf3d3005adae6ddac6d6db1d95e6772914be2e85c2b394c3f590d8a772daf88c3b593104a8d3babfb91bb1aeab4b5667c94d77cc85a02be7c758c3198567fe74941c7c" ),
        std::make_tuple( "818bdeb99de0067263ad93afcc03b7bb91e076d16d1cb2f2762636d9", "411356543a5c2819773ff2a93fb94894f575bf6eea4b5d7675084d74466dd7e8f77a9060e4ecce2a3d045f2b598a7b0f4d12f6a3be7eba7ef3c475e72ef18ff33d", "d1287e9a5a8f7f32ff9bb5c9d0a461acb7815474b8b358da78e1cbccfc858dc0b4e41d48b843fee651586022bbfffcdb292e40db0a2bacae8bfc7cb35d208773b87c6732a098dac4670b68a27334463d412b7b92aaf741ff6502ae55ceca932ee5d8a5819af6da73489d16707fad077b06604dc22081df760fd9aad9d7eb7b99" ),
        std::make_tuple( "55b681a0747e9f2db56762ff44129ae29c94f5ffb933156197dbc655", "f105508d6a45d37a80a7f963f6f927a9f5cf7594f3d2b248d95ba8f56441a04fcbef293ea3b916d60070c689de33a89bbf89fdd795ade081e129848b1da6be0b58", "eeabda87856e3ba98a57d619be748ff5508a2cbf50b66818665042cac4149b0c598400024788c584985853202c9bc262f6d1ab5269ee373f2c477cbb15b20ae755e9c0f1f030e5da7138747ddb42b6cde287165e5bd83227f2822e861b744d844ba4c25b67aec3357dbee037b2b193be7b7718e810bc7e571ec7ff31bca28d6b" ),
        std::make_tuple( "ef9157092c101ffe29e2770231cdb0721d3b90d2d0c08344eef32d70", "67f86b8af7c8bc9bfb568522f9dcadbf23990354e5197566bab882c64362f2709a9cc4b3e4c8689225a42e6b5915ffd6c0234bd07944421952dd47dadf06a18bc9", "9f552fdb9eb3464a75fe354c3c7e748dc257eb63d1b958a5b358f0c0f41c49076a19774b73c176ac7e58c8aecda95e9079f8bcd9b6375933587d447c96b30d1b0e91e3900ac9fa4f028130cf6207efde8cc05537216292567d63928b80b2dd4e94a4bbd2eb319bdd0960b5d68300ef200699fda69aa2cdb728f422153a7f7f48" ),
        std::make_tuple( "39eeb4ce5ac153717476531a8b0c1659bb77fb6bfea052c9b90e86e9", "f67ab51d4988b3e9d55f05b379acb03c84d498a42993982d2de3d243e31f44b95a689d56d5484e907b95e39750989495646f17344bb0b962cc4fad51ddc2a4c1d0", "69d557b16e874038598b25f616afeb4f4a900be7dd0d38b5b6fb4259c51a3aaf4748d7a445f518485ed72b25c7df8ed0906b74bd29bd6a5724ac3a503c990f3697a5db484821f68718470810862728a80ce34599a41fc5bd8bb46dd845a4812ae1532c457ef4211d0e41835e5a6f030247614822571c930c727ba397e723d6b3" ),
        std::make_tuple( "e51184d7cae852dd4464bf04cf3dc35b201d286a995d8c045ffb025c", "a629f7b9614bd8800a9a53196555fbd55174389105dea58b9adcb674cace9ec6f71d70576e70842fee231c9664f54772f90006d04eefaec1066b8d6786d6d04ff8", "8ed85b5f96565939cca9c21728f9ef816229019b59af6d9fb5a3b98acb158016c2a4bce9f44c450c9780949e9d79cdcad8cdfdf662439f3d3ce66620355836780e6973afdbe87f30acba7e1c6d285d3f0b7d1242d9d1252231206fe15e9f227bb94f091b13d2118f537e09f960b0a6c84262cae784000534dc785d340e914a05" ),
        std::make_tuple( "e9169a8bed8cac987699ee1e93910c43f6285d5ededf37b4ea99fbda", "fb396dfb0055d6c7cba8ff2275b9e569fa6189e38ed2eaed35d730548e5e2a03ef33fadeb70e0fede0d054bdd40aa70382bf8d83b06b2430934359959dbf5a7396", "73db1e155da94a056e175a070f7b994a888e2c6fc220f4e1510ab565b16377f2d0f72cc2cdf494a900e75ea63ae790b84e1028f2d03403d21cccffcf9a355d31411969255a2b5c118717812503b897332b8c77c03e66f65b61630d2de61f37d2cd291b678bb46be40b8f0cacf357776328eb5f009420942327b627a81b8bcf7c" ),
    };

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<hmac_sha2_224>( from_hex( std::get<1>( input ) ), from_hex( std::get<2>( input) ) ), std::string( std::get<0>( input ) ) );
    }
}

static void hmac_sha_512()
{
    using boost::hash2::hmac_sha2_512;

    // Test vectors from https://en.wikipedia.org/wiki/HMAC#Examples
    BOOST_TEST_EQ( digest<hmac_sha2_512>( "key", "The quick brown fox jumps over the lazy dog" ), std::string( "b42af09057bac1e2d41708e48a902e09b5ff7f12ab428a4fe86653c73dd248fb82f948a549f7b791a5b41915ee4d1ec3935357e4e2317250d0372afa2ebeeb3a" ) );

    // Test vectors from https://www.rfc-editor.org/rfc/rfc4868#section-2.7
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b" ), "Hi There" ), std::string( "87aa7cdea5ef619d4ff0b4241a1d6cb02379f4e2ce4ec2787ad0b30545e17cdedaa833b7d6b8a702038b274eaea3f4e4be9d914eeb61f1702e696c203a126854" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( "Jefe", "what do ya want for nothing?" ), std::string( "164b7a7bfcf819e2e395fbe73b56e0a387bd64222e831fd610270cd7ea2505549758bf75c05a994a6d034f65f8f0e6fdcaeab1a34d4a6b4b636e070a38bce737" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), from_hex( "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd" ) ), std::string( "fa73b0089d56a284efb0f0756c890be9b1b5dbdd8ee81a3655f83e33b2279d39bf3e848279a722c806b485a47e67c807b946a337bee8942674278859e13292fb" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "0102030405060708090a0b0c0d0e0f10111213141516171819" ), from_hex( "cdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcd" ) ), std::string( "b0ba465637458c6990e5a8c5f61d4af7e576d97ff94b872de76f8050361ee3dba91ca5c11aa25eb4d679275cc5788063a5f19741120c4f2de2adebeb10a298dd" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), "Test Using Larger Than Block-Size Key - Hash Key First" ), std::string( "80b24263c7c1a3ebb71493c1dd7be8b49b46d1f41b4aeec1121b013783f8f3526b56d037e05f2598bd0fd2215d6a1e5295e64f73f63f0aec8b915a985d786598" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm." ), std::string( "e37b6a775dc87dbaa4dfa9f96e5e3ffddebd71f8867289865df5a32d20cdc944b6022cac3c4982b10d5eeb55c3e4de15134676fb6de0446065c97440fa8c6a58" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b" ), "Hi There" ), std::string( "637edc6e01dce7e6742a99451aae82df23da3e92439e590e43e761b33e910fb8ac2878ebd5803f6f0b61dbce5e251ff8789a4722c1be65aea45fd464e89f8f5b" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( "JefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefe", "what do ya want for nothing?" ), std::string( "cb370917ae8a7ce28cfd1d8f4705d6141c173b2a9362c15df235dfb251b154546aa334ae9fb9afc2184932d8695e397bfa0ffb93466cfcceaae38c833b7dba38" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), from_hex( "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd" ) ), std::string( "2ee7acd783624ca9398710f3ee05ae41b9f9b0510c87e49e586cc9bf961733d8623c7b55cebefccf02d5581acc1c9d5fb1ff68a1de45509fbe4da9a433922655" ) );

    // must apply fix noted in the errata https://www.rfc-editor.org/errata/rfc4868
    BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f40" ), from_hex( "cdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcd" ) ), std::string( "5e6688e5a3daec826ca32eaea224eff5e700628947470e13ad01302561bab108b8c48cbc6b807dcfbd850521a685babc7eae4a2a2e660dc0e86b931d65503fd2" ) );

    // use selected samples from the download available here: https://csrc.nist.gov/Projects/cryptographic-algorithm-validation-program/Message-Authentication
    // note: must use `std::make_tuple()` for gcc-5, gcc-4.8 because of https://www.open-std.org/jtc1/sc22/wg21/docs/papers/2015/n4387.html
    std::vector<std::tuple<char const*, char const*, char const*>> inputs =
    {                    /* expected mac */                                                                                                                  /* key */                                                                                                                                                                                                   /* msg */
        std::make_tuple( "33c511e9bc2307c62758df61125a980ee64cefebd90931cb91c13742d4714c06de4003faf3c41c06aefc638ad47b21906e6b104816b72de6269e045a1f4429d4", "57c2eb677b5093b9e829ea4babb50bde55d0ad59fec34a618973802b2ad9b78e26b2045dda784df3ff90ae0f2cc51ce39cf54867320ac6f3ba2c6f0d72360480c96614ae66581f266c35fb79fd28774afd113fa5187eff9206d7cbe90dd8bf67c844e202", "2423dff48b312be864cb3490641f793d2b9fb68a7763b8e298c86f42245e4540eb01ae4d2d4500370b1886f23ca2cf9701704cad5bd21ba87b811daf7a854ea24a56565ced425b35e40e1acbebe03603e35dcf4a100e57218408a1d8dbcc3b99296cfea931efe3ebd8f719a6d9a15487b9ad67eafedf15559ca42445b0f9b42e" ),
        std::make_tuple( "4cc28818486bb9b1b52e333dde71f73acc227488453fd907c6b51d349d67af1df29a9f225532ce04f50395fed565e98d78978626df93462d3f012f7373347298", "7c98912c74421362e112a2f98fed9babe0057fc778b4453239aaf5ac724b725553539770a5bc8666b8e13d0e9ce36b2b934c8137c7f20b5f391f41cefaeed92e9df8206cec3049bcda0c05deb9e6549fada19aa2618ff560f892ce6e4782aeff41cf53a9", "74e8936d83bf3f16b8d03fb73384ed8f46bd32343f5df8358107e2fdda293afa103a2bffbd4030e75d96cc7ca6ec7c97188fea88d4eb63b7b14e8b8c8dee4f8de12e1cc6981d4e6e223fecc7c491924632c7aef45fd8ef1494bcfb06c074616b0f4cce8abd5d83f32d550661357b18e5bcede841882c869251db9a331ac456dd" ),
        std::make_tuple( "c0d6e13c5746369d49bef107cfc9a465627691320b8203233359e6a49659025ac96a6db6c4d460224f6aa1cb7a6b8df311e066f6109bd466cd9aee3058dbc5f0", "662ca8f53b97edd9bbd43b1f9e4ea49f2ac14417faee257aff93608bc49a85abf6913def235a2e76c2241ffa749a5da489595d25c6a8a2026563e12f5e3964e0e518ac9c34e45a938a6f503174a613f34b08737afe5d6fde11a45344e64d23b33ca83c23", "0c057a2b56cb7e651c6339e4c91a1a72d51af2a646de9dfd77e9e42c18b8a2b576f526b9fcedd90dfa442090a6e784bb614311793bb5fb39b8418842d586294746f1ea3c02320d6801ecf2ba44b13b60172d2d9693a158bc66947aacd7c5a14a0463905d6e80649db8c4770cac5e858a7f400da4568cfaae08498311265b50e5" ),
        std::make_tuple( "c34bf0931b2dd2e41956dc86996e1427379d0c89739b1c33fa3be5b0770673a20c5335c6d22c766826009938fe1f4d478b882b59a3b19fdf25bf18f043fbb3f7", "0cc5bca2025bd6030fe0818e0a61ecc730b2e5526da942c0d7897fa97bc1a8fb5dd77991ba9fc50890b014ce6118907b334f2265db6ad86e7b918a214ab3bdfe9378c711017834ca19aa6908081f87779ff0921c9c75d32e2bb77a28ac28881cb792ec4a", "c532714f570982993d4b22c7d07a1e79ff5a75c94eee75dc1fa222b630cad753664b30f3c99826b5cfe17c67dd875b9d0bd2390028e6ffe9fef36a2fd6adb13d3ffc69670cf4a67e9c0764a15e7925579315dbdb561f07b7da892394f4693e51d9abe65228034a1b2b26a01d5a3ac5cf208b2301e27fd86e3ecc159090e8c3b8" ),
        std::make_tuple( "771bf59b658cb17576761d078cf6b1474db746a2201d30ddf289fa708366a27d6a53959bb7eb2b963622b326edaae3dce086dc364c93c874e50089b69c5cf52d", "7d407fda74d3a127b2ed14c727d0e81a04f6789d20eeef629b670abdc18b1f41318e5eea3e86579c957dbccc20c4687d2b8ba16fc6af9a936ad33cc1dfb226ad5cb3f318f1bfbb43224fdca9d5c9faed6e0c44123849f9ea07162bd11bbdc49b48dac6ca", "eabd8db90e6d67a41f096e4369f77cd6ba23da4fcfa459120d9c9ef9725fbe9bcad80bce26292d6a8a927450e6946cab4756b2764f47073fe305a32a237ecb389f55a6c9c7874d60a44e21a7c64561b37ecdfc884db0a3e09b052328ac54f2ccd1fa07b4dbceef0fd5041e4ff3528374c5525f8eb028567d9f64c7fcf62a59de" ),
        std::make_tuple( "4459066109cb11e6870fa9c6bfd251adfa304c0a2928ca915049704972edc560cc7c0bc38249e9101aae2f7d4da62eaff83fb07134efc277de72b9e4ab360425", "c367aeb5c02b727883ffe2a4ceebf911b01454beb328fb5d57fc7f11bf744576aba421e2a63426ea8109bd28ff21f53cd2bf1a11c6c989623d6ec27cdb0bbf458250857d819ff84408b4f3dce08b98b1587ee59683af8852a0a5f55bda3ab5e132b4010e", "1a7331c8ff1b748e3cee96952190fdbbe4ee2f79e5753bbb368255ee5b19c05a4ed9f1b2c72ff1e9b9cb0348205087befa501e7793770faf0606e9c901836a9bc8afa00d7db94ee29eb191d5cf3fc3e8da95a0f9f4a2a7964289c3129b512bd890de8700a9205420f28a8965b6c67be28ba7fe278e5fcd16f0f22cf2b2eacbb9" ),
        std::make_tuple( "b1c34ea9d837b4e0b0771792384fb5f5b9bb5af7226d461b5ca81ce8079c6472c5c44624a640f01960c8a94f6aaca5324c0da2cbfdbcb077cbdca7f6c6a38e75", "52d3e26c59df9bf3f5c01e311fd6611b895dbf6e8e918ff16916fbfaa6981033d7af119e880511d775bac09afa078684ca22ce1ee462a517c3a483d1d5ed68202f512b4e7f130f62420d98a137529d5613139dcf76bf57a81e6e944c5b8048b8c281d982", "2485736977ef55a55abeba3b8e857ee2fa5beb144324e46f9e12625be26b25ede28ca30bf92e45d1e6e8d234daf52be5d0383a781d7d25c64802c7901b366065fac08bc574c3718618603d778a7dd044d6c5b59903f0578aec4571334b5dc79b172914df1037438c9830e14cc4a6d3c5b30c44be1e06e28331e44a8b9968c059" ),
        std::make_tuple( "d5e6ef77772459874a73fc4f7665cd3ade20468bea1ecdac41142ff32350b8cb15828612050046299f08ccc486acef0d0c04e0f8feb29045ef7e3a3db093d512", "ce1e3b693ce203166bf045472fdd1457c8f6591a0ad41912bb30f6e63df8f90f6ca18ed5cac0d07adad407b5c9666f6253553c77e56bda3aff3379b1dd0fc95a5685021d04da287fa5e28d18c11697478ee7c3241052eab684b5c467ffe1aab45370a029", "e2542c06864dd3a0499493e144a97fe04006b68c83a5dd4ceac3cde238e48895ae17728fdc7bbe84b6022694ea75df7371b8ccc76450f2d112222c504f7d1fa20f5b712d33e436fda234abae9c5e278d4bb14efa9b3a88114c89b28946b813db2caa91a73391245435b38cf8016d3f77f678a6eed06b8852c181c754c49d4a88" ),
        std::make_tuple( "d8bf5ff4392938534a7962c64985f163ce7c95e6c05f93cf704106f9bda7c9ae963f5ea87f73626f67ed3146e8611ca62ef2eeb4f9a13847dc6e7ffbe3d851a4", "3699d9cd078a20ec0c96eb01aa60df6cd5dcb554260eea8e2e15b7c00b6943c638611854aaf8d3dd18d020b49a77e67275eabf973557ab74fd2705481c3cb6a9e077a825af7e7e2a53bc822396a9dee40f4b10483bfd9818d06ff32f4deecddcd6e57388", "9186eaa3b8e785dd2a747297bdbdd4f5532a47b7008c21686ff7f8d881d464cd383205f6d45dc8203bb267ac9eb12f415a5406be1c9fac73497941909dba08dd12856aac03d83e0d916147404694fe70f8fa929ef0cc2edb4cc07abaa2236405e62820af8e806d0af32a1b3afb8dcaeaf5c4f43dc4392e074075aa3ed93601ab" ),
        std::make_tuple( "2180018c7e9c3287c3d2928fdf36dda80be4fa21d3a879c0f617eb0e43c58836b0cd714a8081652f8dca9a01925a4f3ec5dbe07b5160be7b1ac58ea623952293", "b0c0a896096bc42bd0c5ea646779a4f1ce541f9cbc04df29ef20b180c069e10efa50ae68ecb8fb31bdfc473f0034dff988b452037ed6261eb0fac9492ccaca2c0ec349b796f1ad077ef995898a5d106160fc100d9ad81c451a1c46269d5e5d90932163cd", "debcf190ea6ab2358636af5cfe4b3a9bdc1bce160bf350aa3cd3956b897e255158cd3e2e83481ce3b6f778d418764f992d48e4f7fb6d080e6b3799d3f35949c17241a0cc5ba84597166779e6a38ce45681ad944cce7c432baf9cd8caf2b33125f2c12052bbb0b3b76f2cb97be9b4813a9ff1e5fdcd478769d0ab5b36cfb466e3" ),
        std::make_tuple( "a481e713cdc81ca5afa0efcb16e35cd20d01aa449958fd2eaede2e25a5ba540beafba2fab4adfef2e146b4c1b2a1832e93dd373d63fa90bb61490f6568191f65", "13fb1ed6389f32d1de3139cb04bcdd53525c9889b85379d3535a25d290351c95938a3d0cdaf38dbf1d5234bf7965c8ddce9ace1b66247e60d74ec7702a0f931a3cdf4cb465ca9fc458c380004a3a6e7957f1f813210b8038ba663fcdc42a8965d6a252b5224bf249552b2575bf64568db4091d58323006c3c34994d3a5", "88ad812fd34e55c809e817199604b4a7f7feae42cdc4c9e930db08e845a3d74313db8a57926706bf0551be758a0fe239f004d237c849d9e4bfac18292bf9c0c3e37985ea54b94f30d18c32ad2b53a059827cddb95a49b4bef1d369ead14eeeb4a18e592e40ca96e515a15908a05a57cd5570b611ab4ec23f7057e1725f29c9de" ),
        std::make_tuple( "b6cad1ca5ba505498a8f66a9422bf539426a8a55334fab9c6b9e08e3a5179d157d1efa0f91d5c5e26ffa43f5c1cb7ca5f906ce4f0efcf4e871820b8353e890e4", "fd5070362296c40d65b105d5ab4653fe34e0200516933f3eeae03ed0c5d9f6016a8560b4bd86ab2f7bf98b22299ed3e54a394602d538aaf3e6951f2db4feaf5dc33426f15bb124da388d709083a28f5701ef96c28b3a3c75bef9332ef373b90771236af5e25d589504345d28a19ab0dbc1c9b74d1ee21c4bd8d423de6a", "8d2e68d7e9846cfa30d931a38efb59bcced53a14164b3163d2653888eeb0bb1448e1a80c65bcc6eb633447e72ed4a075f75d980fe2b19f35ffef62b27ce09c2019922faedb427321057fce19448d85962a08d1baddc936d1110e108e33d46f97e7882445b5df1ca4ff03edc237efaf264f1c0d9e705d9b3eee076ba57c56db82" ),
        std::make_tuple( "e913fccc25f84a1390ef0b0f86050fe9e61146c1b4ef0b4b60d187baa36b61458bbda925b96e99cf8a4330291690417fe194cb9bc2b6232d43e046d13f038ed3", "0f67caceedf8cce39a7223d32ae1b6badae2c2ab01bf75f543dbb8a408514c4b2cd81801f9eac1aa52257c7830f120b1536380b23161f734330744ce204a98bbd9dbeede484e9b03937c26689526597c8edb98e6191b72c95fbc76468b8d48437e3af46bc36f8176c540caab3fc989a3f511b54fa2350e3d31c2f6162f", "3173a712ed715b321a849279be6ba237fc90dec0e48b0e1290e81cb86c2a10eed50f3e05e616be098e3f1da2e6125238c2e2f45a8753aa613c1ae250e304c6ff093a6b799cc34ca2fd4af81d5622076c2e8752593a27649576e12ba075ccf3e8f57b9635b77fec448e2e89b2b5a3a81d65066285a70e24f868c35f6ecb8981f2" ),
        std::make_tuple( "be73d57ab3a7c5bb305451922fa9b11d0ae938bec886fac5645a8b72de93770d96465291da2bdd5d11692f2cbfae69adb36ae714f1e5cf43b9d2841ddeba4ca6", "5c8589b3845970145e12b34713455eb6b5ceb132242024e42fd9a886fc9a30d3aa15703b3467e4dc99a915ae3ac118fd837e571dcde5945983c52a2c849296b4c96f3763488d52f818b459add51f6db2468d21db3d958196bf3a531f65bf9cf702bd66928672c14f235c08aeae0665be472397d43cd9f3822b5fa9c767", "7f2544a478961dd012ac705e12e74587e2df6b2ee1406a0bfe13b908853f31eb384fc236a275654ba8cf51d461ea726228851754ef97ebfbd69326fcfcea04f594d177633c4618c38b4d64f7e98025eca4c62e7a40634b8b0e317880002c51d0bb34caf2ee0f207ee2ae108f53b1466b7f2d18667cac7403ce75175d390e00c5" ),
        std::make_tuple( "165e1cc47a7433270f1673e1cee581001708195a3471c9ec71333ce87299e72abc5dd490d43676d10c1375125d4e7fc673adb15342ddb7e72eb24ff36f54f82c", "7198b12a22014154578f5236b5a0d4cbe29aab3bf289bea2220a4a13c9677ddd8b03400cd0f954337400a069c192c9929a4d04e0f89928999b8b081af909ff1c7b2fcc36e8f2a0a32103764610f6a3ebdba4aed05dce6164d603204ee1c37ab1a7e4feae5335502bc6627cd38b17895e0d6852130771918b3d393db76a", "9b2e0306e73662c94377e5a99fa9b63f034ba91c8b95645eb0afc381c8207b77d089fdfa6a939d212226f331f5dc81b614b597d3e50c74d74bc9ae1027d9d4e41fdb511d9e1c93851bc66db39f54ba248149448e7422e470c589eaeeddcbbedc0d8cc8198bba8e1eb7bf1cd6a3fba9b1d37177f441c0fb53a4563ec1f2e08387" ),
        std::make_tuple( "5702e9898e96cf220c6778cc1d07469f13ef0c2ec0e335ddd1791ff265ce865d200e4e6238b52eaa9be880adfc9076202625bf83bc07d861401a17d4a51d84a8", "b5ffe36117589646ac097327e4147dc9fb13f7346fd97a6fb756448b32bbd3e25edfb3a14ae194844e109fe1f9070ae84b1cea2924fc4957fd8f794bd622a74b6fc4b38dbbf040f7926d0729a67370bdb80f63cd0cc85bb61a83bd1c86a4692f52768e2c53cdc226e1ea5206d39ed6d1abb309290d87d81776fab9072a", "1e6b0799e857a1efb3cb0aaadf74c78c31d5e1c72547dd1d863eed463bcf6892646f78cfa6fe136dc2042ce06d3a2a465c4c994a9edd1f482ecbb2b2c9b509b2fdbb501083852057ce87ae33e483431e6d4fec3b09d87282e7678c1e9423541310d8f82427f6b2f4feddfa6bed57fa5b8c6642641141bd15d999e353442031ff" ),
        std::make_tuple( "9c87ca3b8ee3849224833b477333d4fb1bea3d8167fda445dc677d0d70b9ba85d0c4fd79909703481fbfe99caba7299da8c514a7a0799d59ab6f9e3f1f63a372", "b8100034c20a0b6e423c9f6c541fe9fca08fde8ce7177191db6f0929344332fe0036926e3a2720ea9fb3909cbdfb114d4da73b85c22b43f42248721015d6d5003cc235b9c35349673ab12b0ea0a70971c1a81d33c310df3cfbad795e057e7b3813bad05b8514f1acba3e580a26ac3830a59a6bdc0f50ab310da506592b", "29aaac0418f6bb3890902888219720b59878f226d06c7e8bcb42e9c0015e96f4f802520a15cccf3fb280540e7108b251cfb97aa8fcd86d1eea5d340aa3f65234e14f5639d89155315729978e0fca914732b513374138c3c01f74cab36964cd740a1b1f59094d3554a6115ad2a6e5a3e2ebf3269a479367b692101383faaff1fc" ),
        std::make_tuple( "7e983cd601ff5837e7d170f3092e914e076c21b31761eb7b9ec211e3506758d8d1395ea914c0350afdd6827c0283ea4af188cf30c1fdf075e41363fbdbb29eed", "41f4749cded6e44c11b8118c38cd71cb95a26f9eff01bbbdd716e44e3ed02867858a8bcee5eb2603710ac28048d6a53f0fb6ac7d9f6c9abefa3fb01184597e95706ef83c789ecccfd19df3325e1186ea243bd4dcfedab157914c115583f7d5fee8e7e46efdb87eb819b7cd2be044bdd4ba7b0e438413a89285852ea4a371d5abd63e77edde02e3c731a178f23838", "5c74212dfc2a80d30c39d680327d2488838c35d6503c1a8c4366d7eaabe95c7115f1b7481c7987de820eb4d17fd65d0f58d2123b346044522c04f98ea167c48ad2a0f5a8adb30db0e65775b947fd6f4d470d4cc8dd73e001965a332ce63779ffbb0a441458e1f98f619d800032f8408b75c74b46f47dd5e2abf1eb8e22616218" ),
        std::make_tuple( "562b5ef3d5cec882a2f54f8169612dba2b033325ce5ed924024e7806c745de9e7612dbcfcbb95ea3fdb93de9c6460a866bd412b45eaba5139939fe43d20f9315", "fe27bbc87755aacc37f667f8ca37f8888fc9dc530fe4f8f38e8cd426e01307747edff012d96da707ee96338d1b11feba313a865fca115431dd8632268ff499224ceb69d31732dcd91e0cbd2b92bbd5b6b543a74735705daab81a0114b8a8f0be91d38cd3d8ad328cefe16c99d63c67c4446ca7d1f708f9a848d1a9b60238f6907420c3d9c5e48f67889ca7a1909c", "99841c3e4a41b53c30267dc056e7e9b8f9994494dfbba363ea761c38ec2433d3bd10957d8b7c093472e9a3084c923ac5cb3a1dd2c5270259ce6f3fa80c723dd847a829ac409decbb44395ed20045b694972b4663f2fd658458b9ac7d3ecc65c260d4409110aa481bcea016e41a07446c86f5250f0f45b32aaddec97f293993db" ),
        std::make_tuple( "38ca18d60f180fd2a40e342272190d9b84ba37bbccf59b29bcbdb08762a90e1f8b28349ca634a6f955cc08c96835ee70a2267444fde88b45b8e313b0daf6e12f", "29f8eb9fc8ab58fef681f9faaf934e992d42046f0ccd2fc9ab23d42bd5f5aafda110218196eaa408137a1b66ee4db5a35cd7e4f31107a9e8a81e11e744c000d9784b2d2264696ed721e1362b60b35b2b4d631dacba95658179da4af109cad9687653166c7a503ed3e85d4f334aeaca9bc98fb8804e9febfae70086316c3ac01162cde4461fc89c642f977065f71d", "3ac2ffbc5b6b2334809232c0f8151ed379a8634d70d3f5a1963a7637c421ad0f082f34a8f872702046a4c69c95ad0cea8b683e6528aa731956810f28c1b9396de8a5905e751c1937c9c17c55dc8771df447575ac93a7c161e6967cdabb9930cc03ab7ba8796e07c23170bbb274ad33facb566eea5ad1c7c16f0127155bc77875" ),
        std::make_tuple( "24d2dd3d082e6556dbe27381640837a23e5d4a4d6822066cd09217a677068e5b8901c1eca7da77a9595be271abfa76f9d40656cfbae050ff6d8ddedb0f4c82ed", "345479ae901adbac7223f5f9edc419bb64665cba4e3684b7371e28ff07f3124087f0e89a21630cf9e8a6c0a3d8518e0d5eaee7f31b6d0aa7e59927aa0ecbc479e99e61a98b625736cf1506199d8f2f186bfc9fe2038f0e5b87754635b30888c063462b035581860b2f571083c4e5c6859338cdb09004597b2899cdc87f1224bdfcd08fcf07275f1f1156260ad5bd", "e4b38e556aa285688979a55eeacd7d953f1ee0ab8109444c7cc068488eb83ae9aca1f783a59b944caba75d6e0f5bdc5b4cdbfc6147046e7ed5ea4c757e85fc2181a7580a17310b36fc873e422c4175b1ea24b3830750e50961ba7df9aadd5ebe6badf81148cdb4cd850192ffc9e6103d22e14f3a4a557197291945fb9a292665" ),
        std::make_tuple( "b6e82d35182ec417bb33d9230a55690f8720d32191cb5cd46bfd591421911727a0f8ff64ba6e16f25aa10669a85bf2ba74d84a754ed947335b7a17af0297accc", "2aa1d94ec83ce7c3c75c6bc847759b085234fd44b407d8f80ddfe93c243556e87e4be8fb30b4743ef1169a24732fb2f5f416042b10c3371dd9d20dda29844d58370700ce69f7df5e69240df77b96027a0ecec71b904f690b875da854de05ef047c5d898d1c0d116c580e2a0906b271dec8e5b0dcdfb2550a40092270eabf253376d6eb01f0fff1afe55d5b21bd8c", "acf624e86580af11d0d23c19df6969fe2ec2cdc737bfd00bc54dc0b2ab4421ffb58f44cfdf8c1b1bc5b54bc45b818390de850c6f0adfa2048ed48360bdb8c511860eec5ba6f1bcc51cb34cd8ddc35c23cad4e882df3bfea0ad99ccbb0abbfda707be461622773b16bd1268dbcff89dbfdaf789871d9d8ae80ae4c44afa1571cb" ),
        std::make_tuple( "d336f2002c558eb518c773608387bd500704156043b76104eca2309afa67d69ad9b00e6b83417e088d3f93435922d4e8242e9631f962cd9fc258f3505305d636", "cea946542b91ca50e2afecba73cf546ce1383d82668ecb6265f79ffaa07daa49abb43e21a19c6b2b15c8882b4bc01085a8a5b00168139dcb8f4b2bbe22929ce196d43532898d98a3b0ea4d63112ba25e724bb50711e3cf55954cf30b4503b73d785253104c2df8c19b5b63e92bd6b1ff2573751ec9c508085f3f206c719aa4643776bf425344348cbf63f1450389", "f3ac4422cc724378100d7515ddfbf3fe340002b7976c43acd69c2acf26c3b18173eb4eb6f73622540c6a73dd3eac5c4ea58cc34772428c6bc7370c0accc8c1feff4640d2cb416e2a5d06f35eb366ec69f5b9e0020923f6086216652318182ba93ec702be701a90c0abe9dee261b00b16cd9042318596e9494e401b62333d594a" ),
        std::make_tuple( "39d94c4e1c8456bed8637e592e4231854df3a6ffce98463e4a85c477d9fd34d27035cdfccfcfd385d91e4e38e8c75d9ff941de80742e985baa9c94dbec5a6837", "ef71b7b3ca0f904dc50447ae548096b2b3603b312a5e59d490851b270ee99aef259401bdf2c3efc3b1531ce78176401666aa30db94ec4a30eb281494bef5205dd87f3350c1c4a56f3d040b12167214391b30b121697a7915e9224b871a3c355f111a9493be7b7df870ff5c589bdedbc4dada062b3072ac2c93590829ab26a09dd74d6eaf714e3e07532c57e09921", "50ec304fa342839457d7eb28791b671ba5c425f711c3a351cc76149d481f0547179540fff239f054ff2c078454bfdd92b72b199aa783d562a1e6fd319cf9f8e4d6948b3ed2bcfa80a1d270396209a060051eada0544347f3335c1872266d5e6c1553d9b54cb3e740c631eef0abe2faac1703a7b21deb422d0c3e2b09f0647d06" ),
        std::make_tuple( "d675982ccc457324e24a8ac6db3710b38e5f18c5057730cb7ea2a37b4ba44c41dde0874e43836cc95e97ff0b3ac10410497f9664177b0e576be8c508ab1c7857", "e5606f31ca4d0f5d62730f443f6db0edd8224f1881eaf27f9af3215d06e2f72ddfbd78b467082541422ece34e323a8bd45489fe6db8fedd4c9dfec4954ba286e971db9d078a7d0a8dbfe8f5f166f1e51a4d4fbd21dbb916e65c40d75244b6db87747d98de672371995abfacebe983a325e8f0ae22fb706d7d76a2be95fdeec91e60581f397b1831cd8fcb688c4e7", "082e7b4604dbd3608df7932475e4279bb288688ef998cceb8e16d9695a18e06f3ecce733a7b9e71f62473878b2824941a01b945d93afd1f5204c6a19233230aa0fd64c77822d78a61d266f569279a182fe9f2c287a2108abec16817724e7ebe32456915bbebfeebe659d20053d4f9926741d1837d576d7d79a7b06ca82c279e3" ),
        std::make_tuple( "3780ef695742f09a160c8dd7d35e2758b08284e8150934d222db31df2767d40d7c815c526ecee5f787030c8dc5f050c419ec6ea7563650dcce1480892d3088e6", "8a0349d4d1ed8c4af533e9e83468b5859bb68237798038171346684499c9dc2b5970730533eb2ca04d1680630820f58d32ecf0bd7db7cab72ffc27651c94831cd1220e2113aeba6c889092abb3904d8a264b2332f2d9df0f63ac36d7eabb57c85be0c331587f5f330d69c7c91f00e606de9bc49ec22c9ea815203ca2ed867fb65d743a3beca6427f4669c9c432b7", "035f55033df01f670015a828eff154a245e8ca7474b0b3330cabbe5fdd74e89560b8fa075347532aa46ae7ae907888b30ca4653a6419d0d9224944b43181a6a842c1cbc96fcc3b0f1e7b344c2956f2613c652eb27e44e5d773765a9521fb5e0c7125cf31d9a75f7f38ef96ea01b61b159cd52fc4095a7a94c7db0aeaf40a9929" ),
        std::make_tuple( "086f674d778db491e73b6fbc5126233c6b6e1f066963356d49ea386d9c0868ad25bf6edad0371cde87cea94a18c6dba47535dfce2e40d2246ab17980495d656c", "f78343071f61ee7d9f791bd53132e6d557928bcfe4b214bebf6f3592e46374c7ab148c3c4d6a1443a4675cf4321298c865b440631947b6b05f2c2a337d1cbb9b3661de974b4604eb41cc77c3659e85470e47e16f22a34619db935d59cbf5e1101ed401c020db069eff1035e9d1bff77bd8b3379e05ac0c20bc0e98aad7d7304dedd3bc5ed4136184649b5e0f7e5b", "d63b50b54e1536e35d5f3c6e29f1e49a78ca43fa22b31232c71f0300bd56517e4cd29ba11ee9f206f1ad31ee8f118c87004d6c6dfe837b70a9a2fa987c8b5b6680720c5dbf8791c1fcd6d59fa16cc20df9bc0fb39f41598a376476e45b9f06add8e34af01b373a9ce6a3d189484cacb6cbe0d3d5ef34d709d72c1dee43dc79da" ),
    };

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<hmac_sha2_512>( from_hex( std::get<1>( input ) ), from_hex( std::get<2>( input) ) ), std::string( std::get<0>( input ) ) );
    }
}

static void hmac_sha_384()
{
    using boost::hash2::hmac_sha2_384;

    // Test vectors from https://www.rfc-editor.org/rfc/rfc4868#section-2.7
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b" ), "Hi There" ), std::string( "afd03944d84895626b0825f4ab46907f15f9dadbe4101ec682aa034c7cebc59cfaea9ea9076ede7f4af152e8b2fa9cb6" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( "Jefe", "what do ya want for nothing?" ), std::string( "af45d2e376484031617f78d2b58a6b1b9c7ef464f5a01b47e42ec3736322445e8e2240ca5e69e2c78b3239ecfab21649" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), from_hex( "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd" ) ), std::string( "88062608d3e6ad8a0aa2ace014c8a86f0aa635d947ac9febe83ef4e55966144b2a5ab39dc13814b94e3ab6e101a34f27" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "0102030405060708090a0b0c0d0e0f10111213141516171819" ), from_hex( "cdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcd" ) ), std::string( "3e8a69b7783c25851933ab6290af6ca77a9981480850009cc5577c6e1f573b4e6801dd23c4a7d679ccf8a386c674cffb" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), "Test Using Larger Than Block-Size Key - Hash Key First" ), std::string( "4ece084485813e9088d2c63a041bc5b44f9ef1012a2b588f3cd11f05033ac4c60c2ef6ab4030fe8296248df163f44952" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm." ), std::string( "6617178e941f020d351e2f254e8fd32c602420feb0b8fb9adccebb82461e99c5a678cc31e799176d3860e6110c46523e" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b" ), "Hi There" ), std::string( "b6a8d5636f5c6a7224f9977dcf7ee6c7fb6d0c48cbdee9737a959796489bddbc4c5df61d5b3297b4fb68dab9f1b582c2" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( "JefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefeJefe", "what do ya want for nothing?" ), std::string( "2c7353974f1842fd66d53c452ca42122b28c0b594cfb184da86a368e9b8e16f5349524ca4e82400cbde0686d403371c9" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" ), from_hex( "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd" ) ), std::string( "809f439be00274321d4a538652164b53554a508184a0c3160353e3428597003d35914a18770f9443987054944b7c4b4a" ) );
    BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f200a0b0c0d0e0f10111213141516171819" ), from_hex( "cdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcdcd" ) ), std::string( "5b540085c6e6358096532b2493609ed1cb298f774f87bb5c2ebf182c83cc7428707fb92eab2536a5812258228bc96687" ) );

    // use selected samples from the download available here: https://csrc.nist.gov/Projects/cryptographic-algorithm-validation-program/Message-Authentication
    // note: must use `std::make_tuple()` for gcc-5, gcc-4.8 because of https://www.open-std.org/jtc1/sc22/wg21/docs/papers/2015/n4387.html
    std::vector<std::tuple<char const*, char const*, char const*>> inputs =
    {                    /* expected mac */                                                                                  /* key */                                                                                                                                                                                                   /* msg */
        std::make_tuple( "7cf5a06156ad3de5405a5d261de90275f9bb36de45667f84d08fbcb308ca8f53a419b07deab3b5f8ea231c5b036f8875", "5eab0dfa27311260d7bddcf77112b23d8b42eb7a5d72a5a318e1ba7e7927f0079dbb701317b87a3340e156dbcee28ec3a8d9", "f41380123ccbec4c527b425652641191e90a17d45e2f6206cf01b5edbe932d41cc8a2405c3195617da2f420535eed422ac6040d9cd65314224f023f3ba730d19db9844c71c329c8d9d73d04d8c5f244aea80488292dc803e772402e72d2e9f1baba5a6004f0006d822b0b2d65e9e4a302dd4f776b47a972250051a701fab2b70" ),
        std::make_tuple( "4bb4eb2db2cc921b159b78a2bb9edc1608bb2a1ca9873b411ae30a63386e462f9f69d9f5fc838ff1818748aab74da94f", "f86902e5e5db478ec6e278692728a812c4cd8745f90a7d9f7915f5a94345fc12d2770a3c94b01ffb9e0412999eb6261d11a0", "e0bcacbe96dad6f60e51129f35acd03e12276a91fa13fc15037c75cabb0aee3a19253bb8b35cc0e63208867a032c8f4150a066642f6ff9ea197dab7e9d6da67255c16e051a43bce174a489e85464693006f11a4c61135dce4187040937eb4d1c7eda6e2c315771f0bc6f4273911a07151c63afd3f8c8cec963e4a8f5ef4b8b3e" ),
        std::make_tuple( "a7d1b0081f4884bfa7b30903251fa2803d96822a2160772a473568d41ed3adc77c1b4950a22a14ef5b335a2842046cf2", "0d729c7e630560f8e8dbeb745e021f1fcab754bf6c8af0f4f4bc37bdf3ec653f90f2fac27c0c56fd2838859335e05cdd2cbc", "009b71ebbb05a82c8c13388f65c4b3b57164c5e5ed0eafae3970f7ce2fce5e53b34913008b6226f272ba3f84671b906f4bbc0836abc1cac1db02dd47440d2572fe6e92bea22e97235999def339443a05d24fdd1303e8cb6113108e9ffca22b110a9156eda571e2056b397ba5834797bf2da4aa459d8a99d70923729e01945a78" ),
        std::make_tuple( "6cb9ff908bb72c45f49da3f4a8c5265983c1f4d811dfc098cc3ffa58aa6a6d0ba18183455cee62992b4c3a6cb53c5c16", "5405e1f28569138976b03675efcb88ce373e92c84080fe381ea0a853bd2ef8b8d4596eb115557eb5a7009f32d5c49f8a08bf", "3a7f4dfecb333e44f0258d33d69ceff70ac07e948f5c6acd2ee3ec450db17ce48394447c2b8bd4ac37cd1705070d866cfbfd9bbf851189bc82852a1ecaa13908053cc1a7c17fd514c2eb8277999d027279d132bd45a6470ded21bc3bfb738ea2697facc9b697a9423986b5d1b4d4cbd12c452cd4f8a30b8a19bd8ae4c9b0e578" ),
        std::make_tuple( "58c6f879aa6ee523dc374a01e541f02f4c3dffb948b071468d2b242bbd358d8614aa7fad660348e61828cea1b758b91e", "6c075056122218f595bb28753e87ae6334a0adc24336e85bdb8202545cfce30490ce5e067988108c4e158bf6c0eeb4a4818d", "0b9b52ec0e46793a179589513f117a956fcf98717339373f5010a268d8b254cc7b996b6460255925b59bdd28436d320945bc868d7a1bad87799617fc45f3ff852137e6f5a56c403c12a26d8be334eab9a44cc9e607a95e5e35a03cbf6261605fa47cabe805a49645d7b221c247ed0c6e35884ac4436cb38b2c38f9fb5886dd21" ),
        std::make_tuple( "c550fa53514da34f15e7f98ea87226ab6896cdfae25d3ec2335839f755cdc9a4992092e70b7e5bd422784380b6396cf5", "d137f3e6cc4af28554beb03ba7a97e60c9d3959cd3bb08068edbf68d402d0498c6ee0ae9e3a20dc7d8586e5c352f605cee19", "64a884670d1c1dff555483dcd3da305dfba54bdc4d817c33ccb8fe7eb2ebf623624103109ec41644fa078491900c59a0f666f0356d9bc0b45bcc79e5fc9850f4543d96bc68009044add0838ac1260e80592fbc557b2ddaf5ed1b86d3ed8f09e622e567f1d39a340857f6a850cceef6060c48dac3dd0071fe68eb4ed2ed9aca01" ),
        std::make_tuple( "9891feb5bae6e7f517783322943d7074275f405e54412f3f27ab11f5fdfb57bb54067852789b55dd21107c11a809836e", "5f76d094bd102281900a1c1973a1d7a6da880c85f8382022e0885d24ea7d83e458637f24391c7afe8eb9151aac47f324e1eb", "07b2eb870ed51f53b01cfbe9b01469d96f69f7cc03aa8768884b4774bbf0e43dc5e9fb3854f8d1e8f05b43028d0600d05c31f5d3e71209ac5dec0b3d2a48e80bcb8222e6e1887dcd84dffc0cd9e70ba7e75ffefba62bbaa35955b2991e53169dc48846dbdc0d8dc813b00c4dbf3df427b7fdf98e687d0314259026d1bf97d4bd" ),
        std::make_tuple( "af65cf67f4b82a1f1ef5ff3c7b99266daff3e134059f5ff26261fbfd6114f13d69c1cf1195b93cc9814255ca9a3e9267", "e339f1caaf630eea2e88fa5989bec005c780464f935d2e6140edfbc20a0975279f3b4e8dde740584defd0bce32b025286b68", "edd19cc1f4e6185c362787bf3cc3e93a2a21990c4f2380cbf9ad88068e615328d8ad2e9fa108403e807807e0f258064fac58b76430c5c76f9f0b82d8ce788d079b42d38282ccc1556b25b6b7c6d1b5eadacd5e635acd1c2b3ac89f85c0b68dd1de9974b0bfba0469a0357b8ce1b760d55c877452f5c3e6c872edc6548ea6bc4e" ),
        std::make_tuple( "b075d10aaf54eec1a14ed42d77a9d961452b480bc6f655995dbaf7acab6befbe7bf6945fc0b28ec1898b3a912b539a7f", "fa0f033f3293389e5a9312c6046757eee20fda34e4c8b4d8a1ddd2a8f8c0b0bbf7596362fb9edd5c19646d49bd74856191e2", "ab6d18f2c0756519657eab694e73f35969a324c279d0c994008214c863a12395494383e744c3f30db3136ab1e7535b0e84a2375d2a34349055a44d43828c89f09454816275471c6c00ec9eda1f93e1a8302ee172ae627f2c0dde7e13e5ecd2e353451a4b9d2a6b9e14b04fcb3f5c0d8f35ef2be94b8332cfefd7b11fd4e884d8" ),
        std::make_tuple( "b50fa5b15b51f7b9eace48a79bbf3e392150fef2448500e56a1b25cd5f41b3b159872985b67a6170503e0ace217e6ad5", "19d7b971259fdbdbb4cc0977adb704917d2b3c68b97796df05c26d75349fd777b1e229751db1bdf239def25f0774754b2c8f", "5ade9519477fbfb754e1427586417618202ec140cfec68ed2517a4414888f988ec7dc3757ef312aea257b78ce05e9f1b9a32606942ce12fdcaab60a55b9cde7549f69e0c47f5fe19d75bc31e055a6c7455b4c090fa21aa8448f146c86cf23c3b5b944d65084b2bffda358eb55dfd842a38ff083d5fadd78b89317f34e554b9e3" ),
        std::make_tuple( "e82eeb7f4b7415a4c95dc82c46bb59715fda4e0bdaf64a7fb3af3c7058ec7d2a172b8293057b72f9664454e7dee11d95", "bfe6bb4c9b171b93d28e9f8f86b88dbe509c66eed41818a1986d75b616fee4460f5456cd23667c8a9f1738289601519d33716a534db235", "5b7a078f980bb8919743bbce52fd0ba3c22083d2b0254e28c8d3a05def4da33bd64fb502cfb5d00ce03d49ad168dbe5d1c784a190c7dfa0685908558fe1e37725a4b2f4ebc7eca209c1f5f361b9f2d2393b9911c73f87da24a7a256221f3fb590ef4de3b066e8e16f3726432063a403d4f6dc2a48b9fbd443d17e84200d6d737" ),
        std::make_tuple( "830b4a798f85c448b3d54abfee61b376597f65666d83a21052cb3f4466f44747431607bc659c91cb520308fbf4fcdb58", "4cf54eb8cf7bd421ddb0586ac4fab9c478cdaedd89cc5a195332211f7571b9988419843300fa1ded868d318f48909078bbf1839c8fed61", "d22f194a1af33cd8cdffe9967f677acb68500d6cbbf77a3f34f58840f0c16044827641dc43d6767ce98f85dd5cbeaa9fc5b28333e7f20df8b281cfa4156055e61555e04a1ceb5c5c93ba9210b2e89f6197f0a53996a2c091d16c3cd908d7059ab2545e5a4c39d6c0f10778f82bee43590993da4571107c51b83c35a6702e56a8" ),
        std::make_tuple( "a136b0ce4e8239ae848c5dd84a708027b0083efbb61589b2c30764def27e2e1b221ffb4d18af81f27c35dccf0683bbeb", "de281ac95941750111396ad0e152be30ebf8c47c2dfe8bd5562b56c5d7f54c223c8c7135bd1ad51c10490c6d8207dd1c46863e9f0d8681", "ee6922e96d48b6e0b52a1cc6f3ddfc7b1cd11786237a1fa36ff6cc7b8cf02cb221c43b2622e3b376467a25c2bc4e49b21aea93ba96fd069c78fe4d2ecf11085d632e6472ec80ee94810ba98e53925dd39bb833bfd1657f201f6bddf5156a7f769ef1c55433e50e414e27cfb32271e2e58981201f3e7a31384c87a359c6c13a03" ),
        std::make_tuple( "856c255d058752cfde3e74df1c4a34986e6f7ad5a34a06c157f59d8d6b2039918c4201dd8b2d58617911555802d7c799", "790bd8d4e9ff691a6bb8c0cd64d68d31195961e2c46363b9e63f44f3dea37590bb3297fa4f4533e784ebab80f42f70c3b39976266ff7e5", "4d53f5dfe10ca3880f44fbc2d9bf3d1f97479f4f49ac3f432c085d918b47c71bd8e794ecfbfd1562ea7909927aaa10e87505630eec22c10fce07adbdbaa9d65012b0d74f517b20b49e64344ba145165b953bfd889ea94ff85d80cdb5c2be0b52223524d28a8eabc5528de5b4f59735bf2071d785084ad14401fc27414d8f4f6b" ),
        std::make_tuple( "8d6895c2b80f3ff362dfe355252578dfa31596c8f8b028ee9cd1a689807b6207a216e3516f3d782234d65eb1f9597f68", "15e3ca0e8d9f69f7bc8f63f05b57586c1cb59eb33ac37d3f54769f667b6bc23a8ae3b0aeb50c71fa6582bf404415ba6f93cddd68c42b31", "929172a72b23de3936037ed526238ba70aa8a695c04cb4c56500ec5f0fd52e94bf1c97df91cc528a3625b175a4937c130cd589b50d5e8c0b27fc31e772b7684590fc0089e0fb7807f44d12c846da288b26fac30bf7da6a809b2238682cd43287ed77d72d8076f8b759c3c95872bf391a405e7f1fe786f52daab92a6aa5f43263" ),
        std::make_tuple( "fd24205b960f569042ec8db3c4e442eaa4eb38272aa36f40327ac9f5e3753d3ee8eeefa28ec5fb1e4e1c8815416515d9", "dfd7e7fa6dc49cac285dac27f0968e1a8bacdaceea5cabffad5247ebe9e402a828005fbab16379e50a0ca140bcf1498b56f021c2b4087f", "004c38356decec9ee5ac710126e804b8f6d947eaad0ce6a4f5137b2cce7aaf92d5fda287c5f78e723e1ea7c1c56e457d0d634422746babadd19b8df66da12d404ffa650fbd65493aff970740663012df8dee7fe3df20af3d595e2e549178dba0f8ac49745c1c661bb17f5a271016b20600e1bed6f514377a16c525ee43e700d7" ),
        std::make_tuple( "36768ec85d992222b67619454918b6ad9369bd002d5a1d08079dbefd35c3ae6d28f0475747afad3c3c64981b20e7b665", "8c354e398566a4b9bbe215f1ad7b40d70e698710753365fe56d301fa311503de0940a6c718f80e19c163ddc3c68b01c4bb03cfeebbce56", "a116269f6dc522d0abd6cd740c0879e7902b4a0b9bdfea334b9e7f3f09fdda085e2ef1fe08ca900f4ecd4bcb90e3b7c622e2da3e5b97dffee8c50e82666bfc5c3146ffd77697e5d99026e60e9187d6421e9ca00f815befb9f9d12e565467b332c0653771dfb48af619e88484367e3f232c6183c635b3822a25cbc601fb7a6750" ),
        std::make_tuple( "7bfaebd1b8b1950274b815c26d4b7d11bfebf2fc1e0bfe4ef0a3e36196ebb64dbea0e2aa0f342c4fa021f82faeca66ef", "1fd04f50423cccd5f884f5954c19c8a4c6efbbb0342ea24da92bd28ac79687c83af583cc8550f812012ea86422c14bacfd5e3107073028", "c050ea6c6b8bc0971c1c64e8b8df91397ab8f0a1f2823c004d48a4d8d6b8705fbdd4e8d217eb710b27c8fa56dc2996819a736a323ea3ca7d5c889fb6ca300c8463a0513705c7ef5cdcb50d8ee3091a8fa7a8b4974fb5c8ebd9bdfbb2a6314904391aec453c0880b1c34b6437d566638b29c194772d9e7e724c4e807371a57107" ),
        std::make_tuple( "8b4258be4c094aa4056f332ede8c733772664b088ba22ef8caae7cefd77eceb35e83af8d9c1283cbbffe4a372b699c21", "b619d9d07461c11bc9fb66117d61ed90001366bbffdbff583556777584b0d65244af5c7bdbf3b7358d7c791b966cc809760e57398d1896ace72d26cc59a6904fcd92365edafb8af7986c7d90b2af3bfdbcdb1593c78fbe8e3378bbb0c519152bf9cb51c19a02a12a8fd35cb6f8b3ac337a828711d6c8e0c4c13e1e6af090cdae", "5a81e711adfe5077dd8c8b57c95e8e1f3de39f4fc448c523bd3e7c72b1fdacd6e489dc0d2a34a39ffc6460c1cb962b7a94a30c04b5426a75ffcfc69f0c4ba934d3a3da2e7935d56d6b9079a2a97b016d653a35c2cc0ce19124f887a617c951ce4e58493b4209cc294f983cc20b16f63fd52e8451b1ad13bf5342275079818deb" ),
        std::make_tuple( "ec780a915ec7deeba2c8c9e2ab15c9762a3eb18fafa2d48a554ae1fe6c4459da1a54e2d58bdf06fea0740098eebbb699", "e48825a5503a6afe0bf9a240c67f27acd4a8f6993834645e03c80c72dd370cd2e10071a3ae18ef19bae9d697ea9a4118609190cd95361907a7fa1b58f499f3f5e79b935f12212f437dde399e3e6490244aa1f5e38ba9be2433b6ce924f6cc49e9f627321a5df9343fce1b59deb647d9a3ae00b234414ba7b4e020d67173be693", "8561865ac2ce12832746f82584a4b98e7f4c3ae2410e18196f4e3b475c62ae207d3cadbb1d490096519888db2f3f18e13bfb86f62216015cab8ea491ea734cd3b791a7e45e4f8e0b98d7955bba77e0372d4738161e0d5d84765d9e6a0d05a88e1aa89c5defa864e9e349462e8f14b9993d7a78cb9dbad69aba0551582ddf6958" ),
        std::make_tuple( "f9bf35de6f3888828b0acf7da8d2a8848c292a5ab9e6e2e3475d665d398e5ab424183e9beb4db0d7accc6315c53bf0de", "b9280949918c582fa63ac4c68cc7a0e74971435e08035aef12f98397f9e51713161146a736666a18918c95fafc536c325f5239b368ce3b8b81dd412300ca77c29ebc7f542b93c36a80130a81b4444a879318bd9b4b4f37156998b7c604f93c813870dad9d7db0c2d5a154ce21e62e1f0cd0df9a0194ff794b1730e076d380f56", "9b1d9b8060067187f19e09e2bfe17c11d0092dc1b7b25b06c43bf2e5786fa7153b65430651a43f230b8b92553af1c7c8e90852f14c8724d0f3087d7243ae02bb270c0f6203424e3dea6989bf1032bcbf82c96f7a140042dc9ba9cafb5d439be5c6daa69b5d28c65df600952828ed847e01162b65f964ddb225e78087ad769b1b" ),
        std::make_tuple( "8d032500ea94fd8686b465634d74047db687a1d1e56fc17100f40ea0f7e3288efbd016daffd5d3c32fc6988996292788", "1dd0942297cba8157069925114b832018fe31f35450ef6e2e382f22a950abdde7a9c8642553c5410851e9abf370b0b04da05af9ad32e373f72e68eff2534b0b4ec67bfce0dfd545009c3131bce3e826f2e4a1baf90019f64e6deae85df2295f65d77674e29d44fdde6a024ee431a3e4975f888fce54ed75b25b27a46299e13aa", "8428a75382510207600c10573c4c0f6056a74a0b7cfc33d7c559b2d9821cbd017258299e357abf24032c932d9e7991bf853259cd6b914d00e71de7b76dcd514c7d702f991a4cbfba7e0625150cc20098868a414280f43de515ff012ad83fe5d50d3a644a1a112a5038636f166e6e8474c077ab72a46c2c0eb5c9a53aabb8a225" ),
        std::make_tuple( "40d43eb4671e6840050be5fc38c2709eccfcc47388c380afd52331dff3868227377550420ff176ff97088cbf9c5aaa05", "3dc1c0cf6978292dc6389fe87e96335dcbf493165b528940f867c45f68f0db4092498f2fe33a516bf01304d75dfb03f2fd8566093a8481acb821ae5e352a4d90b4ad08d1dac65c3f98b554410a3398789f07a7f27bd461a11935c80c350b0745e916b41b16c3529005451a8e4ee7fbd5f9a1df38e4e9300964cb9ad22f3c1345", "b5e3eefd6fabee2f84ac6b7b98ca4493a1ef4e49410b49ad54591376c5e31ca4a5481a637ebdfe31e45226434e77d42524bedfa219e2a8e3bb0950db484628e2c462699f48dc262c9d2e7e750a5216e16985c735804b93b58fb3de8c26dcda6d39b8447b76104e66d6c8dcd77365f20f99cc6dcb1f4cf26af8df255105ad1b8f" ),
        std::make_tuple( "3802d4c92f767fda5187cfee5eb73ae20b6bce00e8cfdde4879a0f8906ccba8a0ae3d4fb4a7f1188c3ae9b1910bd07cb", "abcdb92d96bd11e85ebc4fbd6f17dc8417701e188cb59bf53153620615a6a7b8bcb4ad53231ce98afe49645144eb61eeeda84053fb0ea2abb7cdce333d23534b3a97efcdf5eab09d8c4b598180ca0733740f14b5324c3ce5ef70fe51d09e454caddd48421411af48912af7f72c9b177b340d64e73b55b2cb62a4bc09d7fec6e4", "b5ef6f15a59e24f357e578b50c0f6016e9c0b70c54d1ca42a15b19c9ee125c0b4bd5c5001f8385a4c70f91a87fd7a66786318a1d7d4796531026719ed9111e89811b39d029f57dcf08048a3955b6e50d671741c61b9ffdd3764eed468bfc5dd09fea53ed8c9c155c63ebebd2555626efd4996171f58107a819fe162acc7a22c7" ),
        std::make_tuple( "375d68bebc28b9bd0d7108bcb9cb779df000d15090ce6b188ebd3b80cab50c431929ec374c693ca4f08654ab378f3648", "56cac4042da649666fe7b1efd4f70f72b55435891687d728e35aee787a465be26a2bcd697a44e43eee59978e0536d39fc65d641bfcf819ac801123c8b605bfcf5a3615537d015fb6caf04dfb0e30a8d6f84537051dfcdfe33ea10d3bce73595aaf5563683dc601f88d139dddf6cbd83fd17473ef7e7c70f8a8737477c97770ac", "d292a7d9a314845d0468665d59ae01891dccaea88b59bd989eba0d903c8adba9ae40d89d540b7ece9701a38a1e794f8d45b5cbc7cbc5fa60caa7ecfaedb3b136dd376b79fc5ea7f05656eed1d5b196e4b40cde4335ef3b5e97f5ec41422dd41a46bf89f503d3a9cb29ae6dbbe41e8097501911a233a12798ebea34efa5bfee5d" ),
        std::make_tuple( "3ddf47cb13e49e5cf054bbcc073636f30b05dc9ad979a017cc8401d3f1f7f83e3034c41db41942a671b3d3dd706b3619", "a797132ce5971a3a9152f0e3521f9de381406a2be0c53ae189ac1da7244d23205df66eb096d28b84d80f03d3103f2c6378ff454e53d7c206dd62e59782af8e2c4ee654d5a52fd48734cf496527945e3d9c7440565871f669b9c0f5edec29ef9760b1ed227b779d4c8bdced21ffe6d87bd7e780ca59597e11060d7631cc85ee6a", "30accaec827d47ee5aa21603cf62b6c3ee29331cbb7d47e97334c5c614e437231f9cc6989dd15b78609b04a98b2299b355529f1609d76ce151458c80b59664d65fda158c212aa4a8b9de5761530062314f00c22d45a86a9246539da0655a9bab76d07f6f166f68a4ecd1d7d22ee458676698650776da8723d9c6becd2d1823ca" ),
        std::make_tuple( "4f1283e5446483a4f8237650572c8a7694d5f8e34086e81cea1f5bcfea345e33dff699a36aabaaf95fb2e0ec6e0e3b03", "657fcef962db04bd269ae5fef2cbd5e6558d072946d235e8706394d4cd250796769a926fbaaa121b6da42cfc82808474dd672f9362756af252bd8cded78d39b9ddf4d99e24824844934fcf25d03e54df0d83cdda2563fb2be73b54b8b1c4419d429589cfc9ea0dff41a3b7c20190adee8febca47b6264e5bd8e8d4aa8552850a", "989d5d6083c85c2b09be202c60f1277b8c5e471fca623b812fd05b218d42ab8896ac177e4437fd7c784cec64e1eeac701f4e7e682899a419eb152402346cca50d0486c0df11f7194d4519448a070e68592de12d7579ee56ab9640ec27eee22ac8d97e375532ac15965f4a13e671ddea32c388dd31e18065ee1e5a0c93370bb85" ),
    };

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<hmac_sha2_384>( from_hex( std::get<1>( input ) ), from_hex( std::get<2>( input) ) ), std::string( std::get<0>( input ) ) );
    }
}

static void hmac_sha_512_224()
{
    using boost::hash2::hmac_sha2_512_224;

    // use a copy-paste of the test vectors from the hmac_sha2_384 test suite
    // note: must use `std::make_tuple()` for gcc-5, gcc-4.8 because of https://www.open-std.org/jtc1/sc22/wg21/docs/papers/2015/n4387.html
    // note: the expected MACs were generated via OpenSSL with the following sample code:
    /*
        #include <openssl/hmac.h>

        for( auto const& input : inputs )
        {
            auto key = from_hex( std::get<1>( input ) );
            auto data = from_hex( std::get<2>( input ) );
            std::array<unsigned char, 28> hmac = {};
            unsigned md_len = 0;
            HMAC(EVP_sha512_224(), key.data(), key.size(), reinterpret_cast<unsigned char*>( data.data() ), data.size(), hmac.data(), &md_len );
            BOOST_TEST_EQ( digest<hmac_sha2_512_224>( from_hex( std::get<1>( input ) ), from_hex( std::get<2>( input) ) ), to_string( hmac ) );

            std::cout << to_string( hmac ) << std::endl;
        }
    */
    std::vector<std::tuple<char const*, char const*, char const*>> inputs =
    {                    /* expected mac */                                          /* key */                                                                                               /* msg */
        std::make_tuple( "d523897bf108288df23d9ee48a48dea0532c65aa828b65c692997345", "5eab0dfa27311260d7bddcf77112b23d8b42eb7a5d72a5a318e1ba7e7927f0079dbb701317b87a3340e156dbcee28ec3a8d9", "f41380123ccbec4c527b425652641191e90a17d45e2f6206cf01b5edbe932d41cc8a2405c3195617da2f420535eed422ac6040d9cd65314224f023f3ba730d19db9844c71c329c8d9d73d04d8c5f244aea80488292dc803e772402e72d2e9f1baba5a6004f0006d822b0b2d65e9e4a302dd4f776b47a972250051a701fab2b70" ),
        std::make_tuple( "cdae54f6834259598be68a76bfdee6b67484003363215375c42882fb", "f86902e5e5db478ec6e278692728a812c4cd8745f90a7d9f7915f5a94345fc12d2770a3c94b01ffb9e0412999eb6261d11a0", "e0bcacbe96dad6f60e51129f35acd03e12276a91fa13fc15037c75cabb0aee3a19253bb8b35cc0e63208867a032c8f4150a066642f6ff9ea197dab7e9d6da67255c16e051a43bce174a489e85464693006f11a4c61135dce4187040937eb4d1c7eda6e2c315771f0bc6f4273911a07151c63afd3f8c8cec963e4a8f5ef4b8b3e" ),
        std::make_tuple( "fb6503f3fd6390eaee15f3a8ea42271990f62a1c3a454cc69398f5d9", "0d729c7e630560f8e8dbeb745e021f1fcab754bf6c8af0f4f4bc37bdf3ec653f90f2fac27c0c56fd2838859335e05cdd2cbc", "009b71ebbb05a82c8c13388f65c4b3b57164c5e5ed0eafae3970f7ce2fce5e53b34913008b6226f272ba3f84671b906f4bbc0836abc1cac1db02dd47440d2572fe6e92bea22e97235999def339443a05d24fdd1303e8cb6113108e9ffca22b110a9156eda571e2056b397ba5834797bf2da4aa459d8a99d70923729e01945a78" ),
        std::make_tuple( "d54a0525b491e66deb9ec914ee4176d834f16f32f72e341852d16faf", "5405e1f28569138976b03675efcb88ce373e92c84080fe381ea0a853bd2ef8b8d4596eb115557eb5a7009f32d5c49f8a08bf", "3a7f4dfecb333e44f0258d33d69ceff70ac07e948f5c6acd2ee3ec450db17ce48394447c2b8bd4ac37cd1705070d866cfbfd9bbf851189bc82852a1ecaa13908053cc1a7c17fd514c2eb8277999d027279d132bd45a6470ded21bc3bfb738ea2697facc9b697a9423986b5d1b4d4cbd12c452cd4f8a30b8a19bd8ae4c9b0e578" ),
        std::make_tuple( "ef5c32514a7b25c72c7120c0e979e39cef7aa20d7fa45d8e8c6e3fe6", "6c075056122218f595bb28753e87ae6334a0adc24336e85bdb8202545cfce30490ce5e067988108c4e158bf6c0eeb4a4818d", "0b9b52ec0e46793a179589513f117a956fcf98717339373f5010a268d8b254cc7b996b6460255925b59bdd28436d320945bc868d7a1bad87799617fc45f3ff852137e6f5a56c403c12a26d8be334eab9a44cc9e607a95e5e35a03cbf6261605fa47cabe805a49645d7b221c247ed0c6e35884ac4436cb38b2c38f9fb5886dd21" ),
        std::make_tuple( "e89a7a1bbb56db52b59c9c8a6998af8192f11fb0f43f3be8f7b0b094", "d137f3e6cc4af28554beb03ba7a97e60c9d3959cd3bb08068edbf68d402d0498c6ee0ae9e3a20dc7d8586e5c352f605cee19", "64a884670d1c1dff555483dcd3da305dfba54bdc4d817c33ccb8fe7eb2ebf623624103109ec41644fa078491900c59a0f666f0356d9bc0b45bcc79e5fc9850f4543d96bc68009044add0838ac1260e80592fbc557b2ddaf5ed1b86d3ed8f09e622e567f1d39a340857f6a850cceef6060c48dac3dd0071fe68eb4ed2ed9aca01" ),
        std::make_tuple( "160c4b3be2a2c1b75e2436e5b72870ae75615ebda9aa6aa85ed1f770", "5f76d094bd102281900a1c1973a1d7a6da880c85f8382022e0885d24ea7d83e458637f24391c7afe8eb9151aac47f324e1eb", "07b2eb870ed51f53b01cfbe9b01469d96f69f7cc03aa8768884b4774bbf0e43dc5e9fb3854f8d1e8f05b43028d0600d05c31f5d3e71209ac5dec0b3d2a48e80bcb8222e6e1887dcd84dffc0cd9e70ba7e75ffefba62bbaa35955b2991e53169dc48846dbdc0d8dc813b00c4dbf3df427b7fdf98e687d0314259026d1bf97d4bd" ),
        std::make_tuple( "7f1a7a60257e535a86cab0cd1c2a30344469392a526fee4eff07338e", "e339f1caaf630eea2e88fa5989bec005c780464f935d2e6140edfbc20a0975279f3b4e8dde740584defd0bce32b025286b68", "edd19cc1f4e6185c362787bf3cc3e93a2a21990c4f2380cbf9ad88068e615328d8ad2e9fa108403e807807e0f258064fac58b76430c5c76f9f0b82d8ce788d079b42d38282ccc1556b25b6b7c6d1b5eadacd5e635acd1c2b3ac89f85c0b68dd1de9974b0bfba0469a0357b8ce1b760d55c877452f5c3e6c872edc6548ea6bc4e" ),
        std::make_tuple( "6c2ffc989dbdec0ae5f30e93111cafe8a8adbfc937aa5d2db9b65e23", "fa0f033f3293389e5a9312c6046757eee20fda34e4c8b4d8a1ddd2a8f8c0b0bbf7596362fb9edd5c19646d49bd74856191e2", "ab6d18f2c0756519657eab694e73f35969a324c279d0c994008214c863a12395494383e744c3f30db3136ab1e7535b0e84a2375d2a34349055a44d43828c89f09454816275471c6c00ec9eda1f93e1a8302ee172ae627f2c0dde7e13e5ecd2e353451a4b9d2a6b9e14b04fcb3f5c0d8f35ef2be94b8332cfefd7b11fd4e884d8" ),
        std::make_tuple( "364dfbbcb0320f444e357ca5b7c1ab7d24cad9c2509a3bee5f575e50", "19d7b971259fdbdbb4cc0977adb704917d2b3c68b97796df05c26d75349fd777b1e229751db1bdf239def25f0774754b2c8f", "5ade9519477fbfb754e1427586417618202ec140cfec68ed2517a4414888f988ec7dc3757ef312aea257b78ce05e9f1b9a32606942ce12fdcaab60a55b9cde7549f69e0c47f5fe19d75bc31e055a6c7455b4c090fa21aa8448f146c86cf23c3b5b944d65084b2bffda358eb55dfd842a38ff083d5fadd78b89317f34e554b9e3" ),
        std::make_tuple( "c2d589bdc491bc9c95de6a12c7743cbaa554f1935d21cb259761c345", "bfe6bb4c9b171b93d28e9f8f86b88dbe509c66eed41818a1986d75b616fee4460f5456cd23667c8a9f1738289601519d33716a534db235", "5b7a078f980bb8919743bbce52fd0ba3c22083d2b0254e28c8d3a05def4da33bd64fb502cfb5d00ce03d49ad168dbe5d1c784a190c7dfa0685908558fe1e37725a4b2f4ebc7eca209c1f5f361b9f2d2393b9911c73f87da24a7a256221f3fb590ef4de3b066e8e16f3726432063a403d4f6dc2a48b9fbd443d17e84200d6d737" ),
        std::make_tuple( "f5654bf901721992c085e5759fe95c5b6ca36aada64825b093ef16b9", "4cf54eb8cf7bd421ddb0586ac4fab9c478cdaedd89cc5a195332211f7571b9988419843300fa1ded868d318f48909078bbf1839c8fed61", "d22f194a1af33cd8cdffe9967f677acb68500d6cbbf77a3f34f58840f0c16044827641dc43d6767ce98f85dd5cbeaa9fc5b28333e7f20df8b281cfa4156055e61555e04a1ceb5c5c93ba9210b2e89f6197f0a53996a2c091d16c3cd908d7059ab2545e5a4c39d6c0f10778f82bee43590993da4571107c51b83c35a6702e56a8" ),
        std::make_tuple( "90af965b8f232609f153e279cc9c6f295e4ed2591dff459cef006153", "de281ac95941750111396ad0e152be30ebf8c47c2dfe8bd5562b56c5d7f54c223c8c7135bd1ad51c10490c6d8207dd1c46863e9f0d8681", "ee6922e96d48b6e0b52a1cc6f3ddfc7b1cd11786237a1fa36ff6cc7b8cf02cb221c43b2622e3b376467a25c2bc4e49b21aea93ba96fd069c78fe4d2ecf11085d632e6472ec80ee94810ba98e53925dd39bb833bfd1657f201f6bddf5156a7f769ef1c55433e50e414e27cfb32271e2e58981201f3e7a31384c87a359c6c13a03" ),
        std::make_tuple( "12c1945507a0741a58ff43bdd72ce28602d40320f416d8498aeca36d", "790bd8d4e9ff691a6bb8c0cd64d68d31195961e2c46363b9e63f44f3dea37590bb3297fa4f4533e784ebab80f42f70c3b39976266ff7e5", "4d53f5dfe10ca3880f44fbc2d9bf3d1f97479f4f49ac3f432c085d918b47c71bd8e794ecfbfd1562ea7909927aaa10e87505630eec22c10fce07adbdbaa9d65012b0d74f517b20b49e64344ba145165b953bfd889ea94ff85d80cdb5c2be0b52223524d28a8eabc5528de5b4f59735bf2071d785084ad14401fc27414d8f4f6b" ),
        std::make_tuple( "d98ade9918703b1cb172f02aab7cfbe758316febba2e07b81a54b87e", "15e3ca0e8d9f69f7bc8f63f05b57586c1cb59eb33ac37d3f54769f667b6bc23a8ae3b0aeb50c71fa6582bf404415ba6f93cddd68c42b31", "929172a72b23de3936037ed526238ba70aa8a695c04cb4c56500ec5f0fd52e94bf1c97df91cc528a3625b175a4937c130cd589b50d5e8c0b27fc31e772b7684590fc0089e0fb7807f44d12c846da288b26fac30bf7da6a809b2238682cd43287ed77d72d8076f8b759c3c95872bf391a405e7f1fe786f52daab92a6aa5f43263" ),
        std::make_tuple( "2620399c7c3bafe681474312e44ac5ebcff05993bca6cf2073220cff", "dfd7e7fa6dc49cac285dac27f0968e1a8bacdaceea5cabffad5247ebe9e402a828005fbab16379e50a0ca140bcf1498b56f021c2b4087f", "004c38356decec9ee5ac710126e804b8f6d947eaad0ce6a4f5137b2cce7aaf92d5fda287c5f78e723e1ea7c1c56e457d0d634422746babadd19b8df66da12d404ffa650fbd65493aff970740663012df8dee7fe3df20af3d595e2e549178dba0f8ac49745c1c661bb17f5a271016b20600e1bed6f514377a16c525ee43e700d7" ),
        std::make_tuple( "8617f408d67f48b79203786cddfcd54036046c145d8b25e8ba3ff6a5", "8c354e398566a4b9bbe215f1ad7b40d70e698710753365fe56d301fa311503de0940a6c718f80e19c163ddc3c68b01c4bb03cfeebbce56", "a116269f6dc522d0abd6cd740c0879e7902b4a0b9bdfea334b9e7f3f09fdda085e2ef1fe08ca900f4ecd4bcb90e3b7c622e2da3e5b97dffee8c50e82666bfc5c3146ffd77697e5d99026e60e9187d6421e9ca00f815befb9f9d12e565467b332c0653771dfb48af619e88484367e3f232c6183c635b3822a25cbc601fb7a6750" ),
        std::make_tuple( "1875d907e80cc8ab7301be4b28d20259857d77d006af28349f4f2693", "1fd04f50423cccd5f884f5954c19c8a4c6efbbb0342ea24da92bd28ac79687c83af583cc8550f812012ea86422c14bacfd5e3107073028", "c050ea6c6b8bc0971c1c64e8b8df91397ab8f0a1f2823c004d48a4d8d6b8705fbdd4e8d217eb710b27c8fa56dc2996819a736a323ea3ca7d5c889fb6ca300c8463a0513705c7ef5cdcb50d8ee3091a8fa7a8b4974fb5c8ebd9bdfbb2a6314904391aec453c0880b1c34b6437d566638b29c194772d9e7e724c4e807371a57107" ),
        std::make_tuple( "60411c09aa114827b00f3f6f992af8ffacc2119bac32e0611d03aaa8", "b619d9d07461c11bc9fb66117d61ed90001366bbffdbff583556777584b0d65244af5c7bdbf3b7358d7c791b966cc809760e57398d1896ace72d26cc59a6904fcd92365edafb8af7986c7d90b2af3bfdbcdb1593c78fbe8e3378bbb0c519152bf9cb51c19a02a12a8fd35cb6f8b3ac337a828711d6c8e0c4c13e1e6af090cdae", "5a81e711adfe5077dd8c8b57c95e8e1f3de39f4fc448c523bd3e7c72b1fdacd6e489dc0d2a34a39ffc6460c1cb962b7a94a30c04b5426a75ffcfc69f0c4ba934d3a3da2e7935d56d6b9079a2a97b016d653a35c2cc0ce19124f887a617c951ce4e58493b4209cc294f983cc20b16f63fd52e8451b1ad13bf5342275079818deb" ),
        std::make_tuple( "9eb96425de64568ec61e4c6996f201a741551789f859cae256a3b456", "e48825a5503a6afe0bf9a240c67f27acd4a8f6993834645e03c80c72dd370cd2e10071a3ae18ef19bae9d697ea9a4118609190cd95361907a7fa1b58f499f3f5e79b935f12212f437dde399e3e6490244aa1f5e38ba9be2433b6ce924f6cc49e9f627321a5df9343fce1b59deb647d9a3ae00b234414ba7b4e020d67173be693", "8561865ac2ce12832746f82584a4b98e7f4c3ae2410e18196f4e3b475c62ae207d3cadbb1d490096519888db2f3f18e13bfb86f62216015cab8ea491ea734cd3b791a7e45e4f8e0b98d7955bba77e0372d4738161e0d5d84765d9e6a0d05a88e1aa89c5defa864e9e349462e8f14b9993d7a78cb9dbad69aba0551582ddf6958" ),
        std::make_tuple( "35aa1dd700bc2687373df7666671f381ab8665ebe932683bee844ca5", "b9280949918c582fa63ac4c68cc7a0e74971435e08035aef12f98397f9e51713161146a736666a18918c95fafc536c325f5239b368ce3b8b81dd412300ca77c29ebc7f542b93c36a80130a81b4444a879318bd9b4b4f37156998b7c604f93c813870dad9d7db0c2d5a154ce21e62e1f0cd0df9a0194ff794b1730e076d380f56", "9b1d9b8060067187f19e09e2bfe17c11d0092dc1b7b25b06c43bf2e5786fa7153b65430651a43f230b8b92553af1c7c8e90852f14c8724d0f3087d7243ae02bb270c0f6203424e3dea6989bf1032bcbf82c96f7a140042dc9ba9cafb5d439be5c6daa69b5d28c65df600952828ed847e01162b65f964ddb225e78087ad769b1b" ),
        std::make_tuple( "bded11143c0a97a29170aa3e582d83623885eb130297bd5f7b627a0f", "1dd0942297cba8157069925114b832018fe31f35450ef6e2e382f22a950abdde7a9c8642553c5410851e9abf370b0b04da05af9ad32e373f72e68eff2534b0b4ec67bfce0dfd545009c3131bce3e826f2e4a1baf90019f64e6deae85df2295f65d77674e29d44fdde6a024ee431a3e4975f888fce54ed75b25b27a46299e13aa", "8428a75382510207600c10573c4c0f6056a74a0b7cfc33d7c559b2d9821cbd017258299e357abf24032c932d9e7991bf853259cd6b914d00e71de7b76dcd514c7d702f991a4cbfba7e0625150cc20098868a414280f43de515ff012ad83fe5d50d3a644a1a112a5038636f166e6e8474c077ab72a46c2c0eb5c9a53aabb8a225" ),
        std::make_tuple( "b9438d036efe5ac8bc2180b94e679ca99180ba53d369ba2c1ad9a892", "3dc1c0cf6978292dc6389fe87e96335dcbf493165b528940f867c45f68f0db4092498f2fe33a516bf01304d75dfb03f2fd8566093a8481acb821ae5e352a4d90b4ad08d1dac65c3f98b554410a3398789f07a7f27bd461a11935c80c350b0745e916b41b16c3529005451a8e4ee7fbd5f9a1df38e4e9300964cb9ad22f3c1345", "b5e3eefd6fabee2f84ac6b7b98ca4493a1ef4e49410b49ad54591376c5e31ca4a5481a637ebdfe31e45226434e77d42524bedfa219e2a8e3bb0950db484628e2c462699f48dc262c9d2e7e750a5216e16985c735804b93b58fb3de8c26dcda6d39b8447b76104e66d6c8dcd77365f20f99cc6dcb1f4cf26af8df255105ad1b8f" ),
        std::make_tuple( "8a10d9a2668389569176f92773333ca4ad57df1d51a9f057e5129b77", "abcdb92d96bd11e85ebc4fbd6f17dc8417701e188cb59bf53153620615a6a7b8bcb4ad53231ce98afe49645144eb61eeeda84053fb0ea2abb7cdce333d23534b3a97efcdf5eab09d8c4b598180ca0733740f14b5324c3ce5ef70fe51d09e454caddd48421411af48912af7f72c9b177b340d64e73b55b2cb62a4bc09d7fec6e4", "b5ef6f15a59e24f357e578b50c0f6016e9c0b70c54d1ca42a15b19c9ee125c0b4bd5c5001f8385a4c70f91a87fd7a66786318a1d7d4796531026719ed9111e89811b39d029f57dcf08048a3955b6e50d671741c61b9ffdd3764eed468bfc5dd09fea53ed8c9c155c63ebebd2555626efd4996171f58107a819fe162acc7a22c7" ),
        std::make_tuple( "a074568b8c99692f868a476f51ab1794c0cf3a6f15a7bcde47dcc421", "56cac4042da649666fe7b1efd4f70f72b55435891687d728e35aee787a465be26a2bcd697a44e43eee59978e0536d39fc65d641bfcf819ac801123c8b605bfcf5a3615537d015fb6caf04dfb0e30a8d6f84537051dfcdfe33ea10d3bce73595aaf5563683dc601f88d139dddf6cbd83fd17473ef7e7c70f8a8737477c97770ac", "d292a7d9a314845d0468665d59ae01891dccaea88b59bd989eba0d903c8adba9ae40d89d540b7ece9701a38a1e794f8d45b5cbc7cbc5fa60caa7ecfaedb3b136dd376b79fc5ea7f05656eed1d5b196e4b40cde4335ef3b5e97f5ec41422dd41a46bf89f503d3a9cb29ae6dbbe41e8097501911a233a12798ebea34efa5bfee5d" ),
        std::make_tuple( "3b367856625f0b0714147f2f587063005fa9d2af70563a76bfbe5bf5", "a797132ce5971a3a9152f0e3521f9de381406a2be0c53ae189ac1da7244d23205df66eb096d28b84d80f03d3103f2c6378ff454e53d7c206dd62e59782af8e2c4ee654d5a52fd48734cf496527945e3d9c7440565871f669b9c0f5edec29ef9760b1ed227b779d4c8bdced21ffe6d87bd7e780ca59597e11060d7631cc85ee6a", "30accaec827d47ee5aa21603cf62b6c3ee29331cbb7d47e97334c5c614e437231f9cc6989dd15b78609b04a98b2299b355529f1609d76ce151458c80b59664d65fda158c212aa4a8b9de5761530062314f00c22d45a86a9246539da0655a9bab76d07f6f166f68a4ecd1d7d22ee458676698650776da8723d9c6becd2d1823ca" ),
        std::make_tuple( "172c85b41f49798fb22b0bbc8c7ccb0e3a0e2ac51b63c8cf82ea146a", "657fcef962db04bd269ae5fef2cbd5e6558d072946d235e8706394d4cd250796769a926fbaaa121b6da42cfc82808474dd672f9362756af252bd8cded78d39b9ddf4d99e24824844934fcf25d03e54df0d83cdda2563fb2be73b54b8b1c4419d429589cfc9ea0dff41a3b7c20190adee8febca47b6264e5bd8e8d4aa8552850a", "989d5d6083c85c2b09be202c60f1277b8c5e471fca623b812fd05b218d42ab8896ac177e4437fd7c784cec64e1eeac701f4e7e682899a419eb152402346cca50d0486c0df11f7194d4519448a070e68592de12d7579ee56ab9640ec27eee22ac8d97e375532ac15965f4a13e671ddea32c388dd31e18065ee1e5a0c93370bb85" ),
    };

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<hmac_sha2_512_224>( from_hex( std::get<1>( input ) ), from_hex( std::get<2>( input) ) ), std::string( std::get<0>( input ) ) );
    }
}

static void hmac_sha_512_256()
{
    using boost::hash2::hmac_sha2_512_256;

    // use a copy-paste of the test vectors from the hmac_sha2_384 test suite
    // note: must use `std::make_tuple()` for gcc-5, gcc-4.8 because of https://www.open-std.org/jtc1/sc22/wg21/docs/papers/2015/n4387.html
    // note: the expected MACs were generated via OpenSSL with the following sample code:
    /*
        #include <openssl/hmac.h>

        for( auto const& input : inputs )
        {
            auto key = from_hex( std::get<1>( input ) );
            auto data = from_hex( std::get<2>( input ) );
            std::array<unsigned char, 32> hmac = {};
            unsigned md_len = 0;
            HMAC( EVP_sha512_256(), key.data(), key.size(), reinterpret_cast<unsigned char*>( data.data() ), data.size(), hmac.data(), &md_len );
            BOOST_TEST_EQ( digest<hmac_sha2_512_256>( from_hex( std::get<1>( input ) ), from_hex( std::get<2>( input) ) ), to_string( hmac ) );

            std::cout << to_string( hmac ) << std::endl;
        }
    */
    std::vector<std::tuple<char const*, char const*, char const*>> inputs =
    {                    /* expected mac */                                                  /* key */                                                                                               /* msg */
        std::make_tuple( "4a29f173a92ce52d31b808b136390659ce647b628161aff06ba944914a4685d0", "5eab0dfa27311260d7bddcf77112b23d8b42eb7a5d72a5a318e1ba7e7927f0079dbb701317b87a3340e156dbcee28ec3a8d9", "f41380123ccbec4c527b425652641191e90a17d45e2f6206cf01b5edbe932d41cc8a2405c3195617da2f420535eed422ac6040d9cd65314224f023f3ba730d19db9844c71c329c8d9d73d04d8c5f244aea80488292dc803e772402e72d2e9f1baba5a6004f0006d822b0b2d65e9e4a302dd4f776b47a972250051a701fab2b70" ),
        std::make_tuple( "e7f7f3df393ee0fb150f707becc57ac2e9f7bc63a17ea88e9fda4f746b06c668", "f86902e5e5db478ec6e278692728a812c4cd8745f90a7d9f7915f5a94345fc12d2770a3c94b01ffb9e0412999eb6261d11a0", "e0bcacbe96dad6f60e51129f35acd03e12276a91fa13fc15037c75cabb0aee3a19253bb8b35cc0e63208867a032c8f4150a066642f6ff9ea197dab7e9d6da67255c16e051a43bce174a489e85464693006f11a4c61135dce4187040937eb4d1c7eda6e2c315771f0bc6f4273911a07151c63afd3f8c8cec963e4a8f5ef4b8b3e" ),
        std::make_tuple( "bc602da84e72b56282488a0e7673a91d754e35aad44124505ac900562db31b66", "0d729c7e630560f8e8dbeb745e021f1fcab754bf6c8af0f4f4bc37bdf3ec653f90f2fac27c0c56fd2838859335e05cdd2cbc", "009b71ebbb05a82c8c13388f65c4b3b57164c5e5ed0eafae3970f7ce2fce5e53b34913008b6226f272ba3f84671b906f4bbc0836abc1cac1db02dd47440d2572fe6e92bea22e97235999def339443a05d24fdd1303e8cb6113108e9ffca22b110a9156eda571e2056b397ba5834797bf2da4aa459d8a99d70923729e01945a78" ),
        std::make_tuple( "f8a17b550f0806522f865a25e4c547e48292d5aed8b197b5ba4098ca59357c75", "5405e1f28569138976b03675efcb88ce373e92c84080fe381ea0a853bd2ef8b8d4596eb115557eb5a7009f32d5c49f8a08bf", "3a7f4dfecb333e44f0258d33d69ceff70ac07e948f5c6acd2ee3ec450db17ce48394447c2b8bd4ac37cd1705070d866cfbfd9bbf851189bc82852a1ecaa13908053cc1a7c17fd514c2eb8277999d027279d132bd45a6470ded21bc3bfb738ea2697facc9b697a9423986b5d1b4d4cbd12c452cd4f8a30b8a19bd8ae4c9b0e578" ),
        std::make_tuple( "0228803d6dbf2070a9fe844d892972d9d19bbcbdb925f2b320e9b3663cfc6ed5", "6c075056122218f595bb28753e87ae6334a0adc24336e85bdb8202545cfce30490ce5e067988108c4e158bf6c0eeb4a4818d", "0b9b52ec0e46793a179589513f117a956fcf98717339373f5010a268d8b254cc7b996b6460255925b59bdd28436d320945bc868d7a1bad87799617fc45f3ff852137e6f5a56c403c12a26d8be334eab9a44cc9e607a95e5e35a03cbf6261605fa47cabe805a49645d7b221c247ed0c6e35884ac4436cb38b2c38f9fb5886dd21" ),
        std::make_tuple( "315b198e75ee80a65329c0e21e375ae6f2ffaac4375f1fe1bcfea9da8c371364", "d137f3e6cc4af28554beb03ba7a97e60c9d3959cd3bb08068edbf68d402d0498c6ee0ae9e3a20dc7d8586e5c352f605cee19", "64a884670d1c1dff555483dcd3da305dfba54bdc4d817c33ccb8fe7eb2ebf623624103109ec41644fa078491900c59a0f666f0356d9bc0b45bcc79e5fc9850f4543d96bc68009044add0838ac1260e80592fbc557b2ddaf5ed1b86d3ed8f09e622e567f1d39a340857f6a850cceef6060c48dac3dd0071fe68eb4ed2ed9aca01" ),
        std::make_tuple( "21822f62774211b5682bd748ea0a68f19d7660d96e6793d050bb09dfb2bc8c19", "5f76d094bd102281900a1c1973a1d7a6da880c85f8382022e0885d24ea7d83e458637f24391c7afe8eb9151aac47f324e1eb", "07b2eb870ed51f53b01cfbe9b01469d96f69f7cc03aa8768884b4774bbf0e43dc5e9fb3854f8d1e8f05b43028d0600d05c31f5d3e71209ac5dec0b3d2a48e80bcb8222e6e1887dcd84dffc0cd9e70ba7e75ffefba62bbaa35955b2991e53169dc48846dbdc0d8dc813b00c4dbf3df427b7fdf98e687d0314259026d1bf97d4bd" ),
        std::make_tuple( "7db251b255994738a0d9edf75af60e25d10e35732dc7eb4b49fc1e2d0388dcd3", "e339f1caaf630eea2e88fa5989bec005c780464f935d2e6140edfbc20a0975279f3b4e8dde740584defd0bce32b025286b68", "edd19cc1f4e6185c362787bf3cc3e93a2a21990c4f2380cbf9ad88068e615328d8ad2e9fa108403e807807e0f258064fac58b76430c5c76f9f0b82d8ce788d079b42d38282ccc1556b25b6b7c6d1b5eadacd5e635acd1c2b3ac89f85c0b68dd1de9974b0bfba0469a0357b8ce1b760d55c877452f5c3e6c872edc6548ea6bc4e" ),
        std::make_tuple( "fa11b7b959a06549c627f9f65ba00d81882b8a1e197dfb80a317b68b20133c70", "fa0f033f3293389e5a9312c6046757eee20fda34e4c8b4d8a1ddd2a8f8c0b0bbf7596362fb9edd5c19646d49bd74856191e2", "ab6d18f2c0756519657eab694e73f35969a324c279d0c994008214c863a12395494383e744c3f30db3136ab1e7535b0e84a2375d2a34349055a44d43828c89f09454816275471c6c00ec9eda1f93e1a8302ee172ae627f2c0dde7e13e5ecd2e353451a4b9d2a6b9e14b04fcb3f5c0d8f35ef2be94b8332cfefd7b11fd4e884d8" ),
        std::make_tuple( "1f66b9ca4b8907f71606c63cbd1fdf6b64b5449cdb20fec46daee0e99e1fdddb", "19d7b971259fdbdbb4cc0977adb704917d2b3c68b97796df05c26d75349fd777b1e229751db1bdf239def25f0774754b2c8f", "5ade9519477fbfb754e1427586417618202ec140cfec68ed2517a4414888f988ec7dc3757ef312aea257b78ce05e9f1b9a32606942ce12fdcaab60a55b9cde7549f69e0c47f5fe19d75bc31e055a6c7455b4c090fa21aa8448f146c86cf23c3b5b944d65084b2bffda358eb55dfd842a38ff083d5fadd78b89317f34e554b9e3" ),
        std::make_tuple( "99ebb3df6dd8acc78dbc728adf5066b326eacfa6c1440b6a68d71501455bcdc6", "bfe6bb4c9b171b93d28e9f8f86b88dbe509c66eed41818a1986d75b616fee4460f5456cd23667c8a9f1738289601519d33716a534db235", "5b7a078f980bb8919743bbce52fd0ba3c22083d2b0254e28c8d3a05def4da33bd64fb502cfb5d00ce03d49ad168dbe5d1c784a190c7dfa0685908558fe1e37725a4b2f4ebc7eca209c1f5f361b9f2d2393b9911c73f87da24a7a256221f3fb590ef4de3b066e8e16f3726432063a403d4f6dc2a48b9fbd443d17e84200d6d737" ),
        std::make_tuple( "945fac2b3202d677b75240da82ede5b8627691ea6337b5247a61f62ea27db0e2", "4cf54eb8cf7bd421ddb0586ac4fab9c478cdaedd89cc5a195332211f7571b9988419843300fa1ded868d318f48909078bbf1839c8fed61", "d22f194a1af33cd8cdffe9967f677acb68500d6cbbf77a3f34f58840f0c16044827641dc43d6767ce98f85dd5cbeaa9fc5b28333e7f20df8b281cfa4156055e61555e04a1ceb5c5c93ba9210b2e89f6197f0a53996a2c091d16c3cd908d7059ab2545e5a4c39d6c0f10778f82bee43590993da4571107c51b83c35a6702e56a8" ),
        std::make_tuple( "5899794a68ff0424424614aa335b43774c7e18f09d0f8650f5f56c46e0d4178c", "de281ac95941750111396ad0e152be30ebf8c47c2dfe8bd5562b56c5d7f54c223c8c7135bd1ad51c10490c6d8207dd1c46863e9f0d8681", "ee6922e96d48b6e0b52a1cc6f3ddfc7b1cd11786237a1fa36ff6cc7b8cf02cb221c43b2622e3b376467a25c2bc4e49b21aea93ba96fd069c78fe4d2ecf11085d632e6472ec80ee94810ba98e53925dd39bb833bfd1657f201f6bddf5156a7f769ef1c55433e50e414e27cfb32271e2e58981201f3e7a31384c87a359c6c13a03" ),
        std::make_tuple( "3cdec8d01089e2829b467aa1ace32126c8938d0dad06c1aec95b36b1ad3430ca", "790bd8d4e9ff691a6bb8c0cd64d68d31195961e2c46363b9e63f44f3dea37590bb3297fa4f4533e784ebab80f42f70c3b39976266ff7e5", "4d53f5dfe10ca3880f44fbc2d9bf3d1f97479f4f49ac3f432c085d918b47c71bd8e794ecfbfd1562ea7909927aaa10e87505630eec22c10fce07adbdbaa9d65012b0d74f517b20b49e64344ba145165b953bfd889ea94ff85d80cdb5c2be0b52223524d28a8eabc5528de5b4f59735bf2071d785084ad14401fc27414d8f4f6b" ),
        std::make_tuple( "a393a6806446b54900f178407a108583a10eb7f4ba4b74a681f9829187a27360", "15e3ca0e8d9f69f7bc8f63f05b57586c1cb59eb33ac37d3f54769f667b6bc23a8ae3b0aeb50c71fa6582bf404415ba6f93cddd68c42b31", "929172a72b23de3936037ed526238ba70aa8a695c04cb4c56500ec5f0fd52e94bf1c97df91cc528a3625b175a4937c130cd589b50d5e8c0b27fc31e772b7684590fc0089e0fb7807f44d12c846da288b26fac30bf7da6a809b2238682cd43287ed77d72d8076f8b759c3c95872bf391a405e7f1fe786f52daab92a6aa5f43263" ),
        std::make_tuple( "aeeda14765b382fd574abf586f961491671b91991c9d63c7f7153ebc72c3124f", "dfd7e7fa6dc49cac285dac27f0968e1a8bacdaceea5cabffad5247ebe9e402a828005fbab16379e50a0ca140bcf1498b56f021c2b4087f", "004c38356decec9ee5ac710126e804b8f6d947eaad0ce6a4f5137b2cce7aaf92d5fda287c5f78e723e1ea7c1c56e457d0d634422746babadd19b8df66da12d404ffa650fbd65493aff970740663012df8dee7fe3df20af3d595e2e549178dba0f8ac49745c1c661bb17f5a271016b20600e1bed6f514377a16c525ee43e700d7" ),
        std::make_tuple( "62364cc4510cb698858b2877703c7319471228b1026e5b00d13bdf287f4a7166", "8c354e398566a4b9bbe215f1ad7b40d70e698710753365fe56d301fa311503de0940a6c718f80e19c163ddc3c68b01c4bb03cfeebbce56", "a116269f6dc522d0abd6cd740c0879e7902b4a0b9bdfea334b9e7f3f09fdda085e2ef1fe08ca900f4ecd4bcb90e3b7c622e2da3e5b97dffee8c50e82666bfc5c3146ffd77697e5d99026e60e9187d6421e9ca00f815befb9f9d12e565467b332c0653771dfb48af619e88484367e3f232c6183c635b3822a25cbc601fb7a6750" ),
        std::make_tuple( "9f8f55e54a8600c44187594fe038409e0085f5b1d4faaf5941304ae0b1ce228e", "1fd04f50423cccd5f884f5954c19c8a4c6efbbb0342ea24da92bd28ac79687c83af583cc8550f812012ea86422c14bacfd5e3107073028", "c050ea6c6b8bc0971c1c64e8b8df91397ab8f0a1f2823c004d48a4d8d6b8705fbdd4e8d217eb710b27c8fa56dc2996819a736a323ea3ca7d5c889fb6ca300c8463a0513705c7ef5cdcb50d8ee3091a8fa7a8b4974fb5c8ebd9bdfbb2a6314904391aec453c0880b1c34b6437d566638b29c194772d9e7e724c4e807371a57107" ),
        std::make_tuple( "138020161a88d3bbfc254f53aa090eb3dc0ebafd21aab4f25bc01f9906d27a14", "b619d9d07461c11bc9fb66117d61ed90001366bbffdbff583556777584b0d65244af5c7bdbf3b7358d7c791b966cc809760e57398d1896ace72d26cc59a6904fcd92365edafb8af7986c7d90b2af3bfdbcdb1593c78fbe8e3378bbb0c519152bf9cb51c19a02a12a8fd35cb6f8b3ac337a828711d6c8e0c4c13e1e6af090cdae", "5a81e711adfe5077dd8c8b57c95e8e1f3de39f4fc448c523bd3e7c72b1fdacd6e489dc0d2a34a39ffc6460c1cb962b7a94a30c04b5426a75ffcfc69f0c4ba934d3a3da2e7935d56d6b9079a2a97b016d653a35c2cc0ce19124f887a617c951ce4e58493b4209cc294f983cc20b16f63fd52e8451b1ad13bf5342275079818deb" ),
        std::make_tuple( "7e99af96aa7f317edc9f0c71f46177dcfcc221be74a1f9d5cf04aa670de4bd79", "e48825a5503a6afe0bf9a240c67f27acd4a8f6993834645e03c80c72dd370cd2e10071a3ae18ef19bae9d697ea9a4118609190cd95361907a7fa1b58f499f3f5e79b935f12212f437dde399e3e6490244aa1f5e38ba9be2433b6ce924f6cc49e9f627321a5df9343fce1b59deb647d9a3ae00b234414ba7b4e020d67173be693", "8561865ac2ce12832746f82584a4b98e7f4c3ae2410e18196f4e3b475c62ae207d3cadbb1d490096519888db2f3f18e13bfb86f62216015cab8ea491ea734cd3b791a7e45e4f8e0b98d7955bba77e0372d4738161e0d5d84765d9e6a0d05a88e1aa89c5defa864e9e349462e8f14b9993d7a78cb9dbad69aba0551582ddf6958" ),
        std::make_tuple( "c20a26e699e7b4ec75535f45fd01806bd4a44bb0506f0a71907d5df22f5e07ee", "b9280949918c582fa63ac4c68cc7a0e74971435e08035aef12f98397f9e51713161146a736666a18918c95fafc536c325f5239b368ce3b8b81dd412300ca77c29ebc7f542b93c36a80130a81b4444a879318bd9b4b4f37156998b7c604f93c813870dad9d7db0c2d5a154ce21e62e1f0cd0df9a0194ff794b1730e076d380f56", "9b1d9b8060067187f19e09e2bfe17c11d0092dc1b7b25b06c43bf2e5786fa7153b65430651a43f230b8b92553af1c7c8e90852f14c8724d0f3087d7243ae02bb270c0f6203424e3dea6989bf1032bcbf82c96f7a140042dc9ba9cafb5d439be5c6daa69b5d28c65df600952828ed847e01162b65f964ddb225e78087ad769b1b" ),
        std::make_tuple( "2db08360163820ef346ed0d18da488fb6d6790eaf43a38379794cc496f59a50d", "1dd0942297cba8157069925114b832018fe31f35450ef6e2e382f22a950abdde7a9c8642553c5410851e9abf370b0b04da05af9ad32e373f72e68eff2534b0b4ec67bfce0dfd545009c3131bce3e826f2e4a1baf90019f64e6deae85df2295f65d77674e29d44fdde6a024ee431a3e4975f888fce54ed75b25b27a46299e13aa", "8428a75382510207600c10573c4c0f6056a74a0b7cfc33d7c559b2d9821cbd017258299e357abf24032c932d9e7991bf853259cd6b914d00e71de7b76dcd514c7d702f991a4cbfba7e0625150cc20098868a414280f43de515ff012ad83fe5d50d3a644a1a112a5038636f166e6e8474c077ab72a46c2c0eb5c9a53aabb8a225" ),
        std::make_tuple( "b95a2031ca186985a680bc470ed68a47776c6cf2774d8ac84786d1da93064d3e", "3dc1c0cf6978292dc6389fe87e96335dcbf493165b528940f867c45f68f0db4092498f2fe33a516bf01304d75dfb03f2fd8566093a8481acb821ae5e352a4d90b4ad08d1dac65c3f98b554410a3398789f07a7f27bd461a11935c80c350b0745e916b41b16c3529005451a8e4ee7fbd5f9a1df38e4e9300964cb9ad22f3c1345", "b5e3eefd6fabee2f84ac6b7b98ca4493a1ef4e49410b49ad54591376c5e31ca4a5481a637ebdfe31e45226434e77d42524bedfa219e2a8e3bb0950db484628e2c462699f48dc262c9d2e7e750a5216e16985c735804b93b58fb3de8c26dcda6d39b8447b76104e66d6c8dcd77365f20f99cc6dcb1f4cf26af8df255105ad1b8f" ),
        std::make_tuple( "1c75a119ecb923f38f7020d524212bc535ab5aaed9192358ccb717005adf0cdc", "abcdb92d96bd11e85ebc4fbd6f17dc8417701e188cb59bf53153620615a6a7b8bcb4ad53231ce98afe49645144eb61eeeda84053fb0ea2abb7cdce333d23534b3a97efcdf5eab09d8c4b598180ca0733740f14b5324c3ce5ef70fe51d09e454caddd48421411af48912af7f72c9b177b340d64e73b55b2cb62a4bc09d7fec6e4", "b5ef6f15a59e24f357e578b50c0f6016e9c0b70c54d1ca42a15b19c9ee125c0b4bd5c5001f8385a4c70f91a87fd7a66786318a1d7d4796531026719ed9111e89811b39d029f57dcf08048a3955b6e50d671741c61b9ffdd3764eed468bfc5dd09fea53ed8c9c155c63ebebd2555626efd4996171f58107a819fe162acc7a22c7" ),
        std::make_tuple( "43dc9f5975c343e30ffe8576001d7a446769a3b2490b6090c24a1eb4a7c291fa", "56cac4042da649666fe7b1efd4f70f72b55435891687d728e35aee787a465be26a2bcd697a44e43eee59978e0536d39fc65d641bfcf819ac801123c8b605bfcf5a3615537d015fb6caf04dfb0e30a8d6f84537051dfcdfe33ea10d3bce73595aaf5563683dc601f88d139dddf6cbd83fd17473ef7e7c70f8a8737477c97770ac", "d292a7d9a314845d0468665d59ae01891dccaea88b59bd989eba0d903c8adba9ae40d89d540b7ece9701a38a1e794f8d45b5cbc7cbc5fa60caa7ecfaedb3b136dd376b79fc5ea7f05656eed1d5b196e4b40cde4335ef3b5e97f5ec41422dd41a46bf89f503d3a9cb29ae6dbbe41e8097501911a233a12798ebea34efa5bfee5d" ),
        std::make_tuple( "d081a3f2acd97b8b8cf31569e2b1e538bbf3074f64fd82a6d2ed457656e2d30d", "a797132ce5971a3a9152f0e3521f9de381406a2be0c53ae189ac1da7244d23205df66eb096d28b84d80f03d3103f2c6378ff454e53d7c206dd62e59782af8e2c4ee654d5a52fd48734cf496527945e3d9c7440565871f669b9c0f5edec29ef9760b1ed227b779d4c8bdced21ffe6d87bd7e780ca59597e11060d7631cc85ee6a", "30accaec827d47ee5aa21603cf62b6c3ee29331cbb7d47e97334c5c614e437231f9cc6989dd15b78609b04a98b2299b355529f1609d76ce151458c80b59664d65fda158c212aa4a8b9de5761530062314f00c22d45a86a9246539da0655a9bab76d07f6f166f68a4ecd1d7d22ee458676698650776da8723d9c6becd2d1823ca" ),
        std::make_tuple( "d9c3f666c3bbbf27a747ff203935d44c9b263495f034efb223f6ba133c8b9ce5", "657fcef962db04bd269ae5fef2cbd5e6558d072946d235e8706394d4cd250796769a926fbaaa121b6da42cfc82808474dd672f9362756af252bd8cded78d39b9ddf4d99e24824844934fcf25d03e54df0d83cdda2563fb2be73b54b8b1c4419d429589cfc9ea0dff41a3b7c20190adee8febca47b6264e5bd8e8d4aa8552850a", "989d5d6083c85c2b09be202c60f1277b8c5e471fca623b812fd05b218d42ab8896ac177e4437fd7c784cec64e1eeac701f4e7e682899a419eb152402346cca50d0486c0df11f7194d4519448a070e68592de12d7579ee56ab9640ec27eee22ac8d97e375532ac15965f4a13e671ddea32c388dd31e18065ee1e5a0c93370bb85" ),
    };

    for( auto const& input : inputs )
    {
        BOOST_TEST_EQ( digest<hmac_sha2_512_256>( from_hex( std::get<1>( input ) ), from_hex( std::get<2>( input) ) ), std::string( std::get<0>( input ) ) );
    }
}

int main()
{
    hmac_sha_256();
    hmac_sha_224();
    hmac_sha_512();
    hmac_sha_384();
    hmac_sha_512_224();
    hmac_sha_512_256();

    return boost::report_errors();
}
