% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dbs.R
\name{dbs}
\alias{dbs}
\title{Derivatives of B-Splines}
\usage{
dbs(
  x,
  derivs = 1L,
  df = NULL,
  knots = NULL,
  degree = 3L,
  intercept = FALSE,
  Boundary.knots = NULL,
  ...
)
}
\arguments{
\item{x}{The predictor variable.  Missing values are allowed and will be
returned as they are.}

\item{derivs}{A positive integer specifying the order of derivative.  The
default value is \code{1L} for the first derivative.}

\item{df}{Degree of freedom that equals to the column number of the returned
matrix.  One can specify \code{df} rather than \code{knots}, then the
function chooses \code{df - degree - as.integer(intercept)} internal
knots at suitable quantiles of \code{x} ignoring missing values and
those \code{x} outside of the boundary.  If internal knots are specified
via \code{knots}, the specified \code{df} will be ignored.}

\item{knots}{The internal breakpoints that define the splines.  The default
is \code{NULL}, which results in a basis for ordinary polynomial
regression.  Typical values are the mean or median for one knot,
quantiles for more knots.}

\item{degree}{A nonnegative integer specifying the degree of the piecewise
polynomial. The default value is \code{3} for cubic splines. Zero degree
is allowed for piecewise constant basis functions.}

\item{intercept}{If \code{TRUE}, the complete basis matrix will be returned.
Otherwise, the first basis will be excluded from the output.}

\item{Boundary.knots}{Boundary points at which to anchor the splines.  By
default, they are the range of \code{x} excluding \code{NA}.  If both
\code{knots} and \code{Boundary.knots} are supplied, the basis
parameters do not depend on \code{x}. Data can extend beyond
\code{Boundary.knots}.}

\item{...}{Optional arguments that are not used.}
}
\value{
A numeric matrix of \code{length(x)} rows and \code{df} columns if
    \code{df} is specified or \code{length(knots) + degree +
    as.integer(intercept)} columns if \code{knots} are specified instead.
    Attributes that correspond to the arguments specified are returned
    mainly for other functions in this package.
}
\description{
Produces the derivatives of given order of B-splines.
}
\details{
This function provides a more user-friendly interface and a more consistent
handling for \code{NA}'s than \code{splines::splineDesign()} for derivatives
of B-splines.  The implementation is based on the closed-form recursion
formula.  At knots, the derivative is defined to be the right derivative
except at the right boundary knot.
}
\examples{
library(splines2)
x <- seq.int(0, 1, 0.01)
knots <- c(0.2, 0.4, 0.7)
## the second derivative of cubic B-splines with three internal knots
dMat <- dbs(x, derivs = 2L, knots = knots, intercept = TRUE)

## compare with the results from splineDesign
ord <- attr(dMat, "degree") + 1L
bKnots <- attr(dMat, "Boundary.knots")
aKnots <- c(rep(bKnots[1L], ord), knots, rep(bKnots[2L], ord))
res <- splines::splineDesign(aKnots, x = x, derivs = 2L)
stopifnot(all.equal(res, dMat, check.attributes = FALSE))
}
\references{
De Boor, Carl. (1978). \emph{A practical guide to splines}.
Vol. 27. New York: Springer-Verlag.
}
\seealso{
\code{\link{bSpline}} for B-splines;
\code{\link{ibs}} for integrals of B-splines.
}
