@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-event-send: Trigger an ECaccess Event
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;

my %opt = ( environment => undef, delay => undef, at => undef, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  environment:s
	  delay:s
	  at:s
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $eventid  = $ARGV[0];
my $sequence = $ARGV[1];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No event-id specified!\n" ) if not($eventid);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No sequence specified!\n" ) if not($sequence);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Invalid sequence number!\n" ) if not( ( ( $sequence * 1 ) eq $sequence ) && abs($sequence) <= 2147483647 );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# Trigger the event
my $updates = $controlChannel->sendEvent(
	$token,
	$eventid,
	SOAP::Data->name(
		"request" => \SOAP::Data->value(
			SOAP::Data->name( 'env'   => $opt{environment} ),
			SOAP::Data->name( 'delay' => $opt{delay} ),
			SOAP::Data->name( 'at'    => $opt{at} ),
			SOAP::Data->name( 'seq'   => $sequence )
		)
	)
)->result;
print "Notification sent for ", $eventid, " (", $updates, " subscription(s) updated)\n";

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-event-send - Trigger an ECaccess Event

=head1 SYNOPSIS

B<ecaccess-event-send -version|-help|-manual>

B<ecaccess-event-send [-debug] [-environment> I<variables>B<] [-delay> I<duration>B<] [-at> I<date>B<]> I<event-id> I<sequence>

=head1 DESCRIPTION

Allow triggering the event specified by its I<event-id>. The I<sequence> number should always be greater
than the last I<sequence> submitted. The specified environment parameters are passed to the jobs before
submissions. The triggering of the event can also be delayed or started at a later time if required.
As a result the number of notification(s) updated is displayed.

=head1 ARGUMENTS

=over 8

=item I<event-id>

The identifier of the Event to trigger.

=item I<sequence>

The sequence number for the triggering (must be an integer between 0 and 2147483647).

=back

=head1 OPTIONS

=over 8

=item B<-environment> I<variables>

Specify which environment I<variables> to pass to the jobs. Multiple variables should be
separated by a semi-column (e.g. I<"PARAM1=xxx;PARAM2=yyy">).

=item B<-delay> I<duration>

Allow delaying the submission of the event (default: no delay). The I<duration> is
specified in [w]eeks, [d]ays, [h]ours, [m]inutes or [s]econds (e.g. I<1w> or I<2d>).

=item B<-at> I<date>

Specify the starting I<date> for the event (default: immediate submission). The
format of the date is "yyyy-MM-dd HH:mm" (e.g. I<"2011-01-19 12:17">).

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-event-send> I<167> I<2000>

Trigger the event I<167> with a sequence number of I<2000>.

B<ecaccess-event-send -delay> I<1d> I<167> I<2000>

Trigger the same event as in the previous example in 24 hours.

=head1 SEE ALSO

B<ecaccess-event-grant>, B<ecaccess-event-clear>, B<ecaccess-event-list>, B<ecaccess-event-delete>,
B<ecaccess-event-create> and B<ecaccess>.

=cut

__END__
:endofperl
