/*
 * This file is part of the Polkit-tqt project
 * Copyright (C) 2009 Jaroslav Reznik <jreznik@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#define POLKIT_AGENT_I_KNOW_API_IS_SUBJECT_TO_CHANGE 1
#include <polkitagent/polkitagent.h>

#include "listeneradapter_p.h"


namespace PolkitTQt
{

namespace Agent
{

//--------------------------------------
// ListenerAdapter
//--------------------------------------

ListenerAdapter *ListenerAdapter::m_theListenerAdapter = nullptr;

ListenerAdapter *ListenerAdapter::instance()
{
  if (!m_theListenerAdapter)
  {
    m_theListenerAdapter = new ListenerAdapter();
  }
  return m_theListenerAdapter;
}

ListenerAdapter::ListenerAdapter(TQObject *parent) : TQObject(parent)
{
}

Listener* ListenerAdapter::findListener(PolkitAgentListener *listener)
{
  for (Listener *&listItem : m_listeners)
  {
    if (listItem && listItem->listener() == listener)
    {
      return listItem;
    }
  }

  return nullptr;
}

void ListenerAdapter::polkit_tqt_listener_initiate_authentication(PolkitAgentListener *listener,
        const gchar *action_id, const gchar *message, const gchar *icon_name,
        PolkitDetails *details, const gchar *cookie, GList *identities,
        GCancellable *cancellable, GSimpleAsyncResult *result)
{
  tqDebug("polkit_tqt_listener_initiate_authentication callback for %p", listener);

  Identity::List idents;
  Details dets(details);
  GList *identity;
  for (identity = g_list_first(identities); identity != nullptr; identity = g_list_next(identity))
  {
    idents.append(Identity((PolkitIdentity*)identity->data));
  }

  Listener *list = findListener(listener);
  list->initiateAuthentication(TQString::fromUtf8(action_id), TQString::fromUtf8(message),
          TQString::fromUtf8(icon_name), dets, TQString::fromUtf8(cookie), idents,
          new AsyncResult(result));
}

gboolean ListenerAdapter::polkit_tqt_listener_initiate_authentication_finish(
        PolkitAgentListener *listener, GAsyncResult *res, GError **error)
{
  tqDebug("polkit_tqt_listener_initiate_authentication_finish callback for %p", listener);

  GSimpleAsyncResult *simple = G_SIMPLE_ASYNC_RESULT(res);
  if (g_simple_async_result_propagate_error(simple, error))
  {
    return false;
  }
  return true;
}

void ListenerAdapter::cancelled_cb(PolkitAgentListener *listener)
{
  tqDebug("cancelled_cb for %p", listener);

  Listener *list = findListener(listener);
  if (list)
  {
    list->cancelAuthentication();
  }
}

void ListenerAdapter::addListener(Listener *listener)
{
  tqDebug("Adding new listener %p for %p", listener, listener->listener());

  m_listeners.append(listener);
}

void ListenerAdapter::removeListener(Listener *listener)
{
  tqDebug("Removing listener %p", listener);

  // should be safe as we don't have more than one same listener registered in one time
  TQValueList<Listener*>::iterator listerIt = m_listeners.find(listener);
  if (listerIt != m_listeners.end())
  {
    m_listeners.remove(listerIt);
  }
}

}

}

#include "listeneradapter_p.moc"
