/***************************************************************************
 *   Copyright (C) 2005-2007 Nicolas Hadacek <hadacek@kde.org>             *
 *   Copyright (C) 2003-2004 Alain Gibaud <alain.gibaud@free.fr>           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "toplevel.h"

#include <tqpixmap.h>
#include <tqiconset.h>
#include <tqlayout.h>
#include <tqsplitter.h>
#include <tqstringlist.h>
#include <tqtimer.h>
#include <tqprogressbar.h>
#include <tqeventloop.h>
#include <tqapplication.h>
#include <tqtooltip.h>

#include <kstatusbar.h>
#include <tdelocale.h>
#include <kedittoolbar.h>
#include <tdeapplication.h>
#include <tdeconfig.h>
#include <tdeaction.h>
#include <tdecmdlineargs.h>
#include <tdeio/observer.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>
#include <khelpmenu.h>
#include <tdemenubar.h>

#include "gui_debug_manager.h"
#include "hex_editor.h"
#include "project_manager.h"
#include "project.h"
#include "global_config.h"
#include "editor.h"
#include "device_gui.h"
#include "text_editor.h"
#include "new_dialogs.h"
#include "common/global/process.h"
#include "common/gui/misc_gui.h"
#include "common/gui/purl_gui.h"
#include "devices/list/device_list.h"
#include "progs/base/prog_config.h"
#include "progs/list/prog_list.h"
#include "progs/gui/prog_group_ui.h"
#include "editor_manager.h"
#include "tools/list/compile_manager.h"
#include "object_view.h"
#include "config_gen.h"
#include "tools/list/compile_config.h"
#include "watch_view.h"
#include "coff/base/text_coff.h"
#include "tools/list/tool_list.h"
#include "breakpoint_view.h"
#include "main_global.h"
#include "gui_prog_manager.h"
#include "devices/gui/register_view.h"
#include "likeback.h"
#include "console.h"
#include "devices/base/device_group.h"
#include "tools/gui/toolchain_config_center.h"
#include "global_config.h"

//----------------------------------------------------------------------------
KDockWidget *MainWindow::createDock(const TQString &name, const TQPixmap &icon,
                                    const TQString &title, const DockPosition &position)
{
  KDockWidget *dock = createDockWidget(name, icon, 0, title);
  dock->setDockSite(KDockWidget::DockCenter);
  DockData ddata;
  ddata.title = title;
  ddata.position = position;
  ddata.dock = dock;
  ddata.action = new ViewMenuAction(dock);
  connect(ddata.action, TQ_SIGNAL(activated(TQWidget *)), TQ_SLOT(toggleToolView(TQWidget *)));
  _docks += ddata;
  initDockPosition(ddata);
  return dock;
}

MainWindow::MainWindow()
  : _configGenerator(0), _pikloopsProcess(0), _kfindProcess(0), _forceProgramAfterBuild(false)
{
  Q_ASSERT( Main::_toplevel==0 );
  Main::_toplevel = this;

// status bar
    _actionStatus = new TQLabel(statusBar());
    statusBar()->addWidget(_actionStatus);
    _actionProgress = new TQProgressBar(statusBar());
    statusBar()->addWidget(_actionProgress);
    _debugStatus = new TQLabel(statusBar());
    statusBar()->addWidget(_debugStatus, 0, true);
    _editorStatus = new TQLabel(statusBar());
    statusBar()->addWidget(_editorStatus, 0, true);
    _programmerStatus = new ProgrammerStatusWidget(statusBar());
    connect(_programmerStatus, TQ_SIGNAL(configure()), TQ_SLOT(configureProgrammer()));
    connect(_programmerStatus, TQ_SIGNAL(selected(const Programmer::Group &)), TQ_SLOT(selectProgrammer(const Programmer::Group &)));
    statusBar()->addWidget(_programmerStatus->widget(), 0, true);
    _toolStatus = new ToolStatusWidget(statusBar());
    connect(_toolStatus, TQ_SIGNAL(configureToolchain()), TQ_SLOT(configureToolchains()));
    connect(_toolStatus, TQ_SIGNAL(configure()), TQ_SLOT(configureProject()));
    connect(_toolStatus, TQ_SIGNAL(selected(const Tool::Group &)), TQ_SLOT(selectTool(const Tool::Group &)));
    statusBar()->addWidget(_toolStatus->widget(), 0, true);

// interface
    _mainDock = createDockWidget("main_dock_widget", TQPixmap());
    _mainDock->setDockSite(KDockWidget::DockCorner);
    _mainDock->setEnableDocking(KDockWidget::DockNone);
    setView(_mainDock);
    setMainDockWidget(_mainDock);

    TDEIconLoader loader;
    KDockWidget *dock = createDock("project_manager_dock_widget", PURL::icon(PURL::Project),
                                   i18n("Project Manager"), DockPosition(KDockWidget::DockLeft, 20));
    Main::_projectManager = new ProjectManager::View(dock);
    connect(Main::_projectManager, TQ_SIGNAL(guiChanged()), TQ_SLOT(updateGUI()));
    dock->setWidget(Main::_projectManager);

    dock = createDock("watch_view_dock_widget", loader.loadIcon("viewmag", TDEIcon::Small),
                      i18n("Watch View"), DockPosition("project_manager_dock_widget"));
    Main::_watchView = new Register::WatchView(dock);
    dock->setWidget(Main::_watchView);

    Main::_editorManager = new EditorManager(_mainDock);
    _mainDock->setWidget(Main::_editorManager);
    connect(Main::_editorManager, TQ_SIGNAL(guiChanged()), TQ_SLOT(updateGUI()));
    connect(Main::_editorManager, TQ_SIGNAL(modified(const PURL::Url &)), Main::_projectManager, TQ_SLOT(modified(const PURL::Url &)));
    connect(Main::_editorManager, TQ_SIGNAL(statusChanged(const TQString &)), _editorStatus, TQ_SLOT(setText(const TQString &)));

    dock = createDock("compile_log_dock_widget", loader.loadIcon("piklab_compile", TDEIcon::Small),
                      i18n("Compile Log"), DockPosition(KDockWidget::DockBottom, 80));
    Main::_compileLog = new Compile::LogWidget(dock);
    Main::_compileLog->setFocusPolicy(TQWidget::NoFocus);
    dock->setWidget(Main::_compileLog);

    dock = createDock("program_log_dock_widget", loader.loadIcon("piklab_burnchip", TDEIcon::Small),
                      i18n("Program Log"), DockPosition("compile_log_dock_widget"));
    _programLog = new Log::Widget(dock, "program_log");
    _programLog->setFocusPolicy(TQWidget::NoFocus);
    dock->setWidget(_programLog);

    dock = createDock("breakpoints_dock_widget", loader.loadIcon("piklab_breakpoint_active", TDEIcon::Small),
                      i18n("Breakpoints"), DockPosition("compile_log_dock_widget"));
    Main::_breakpointsView = new Breakpoint::View(dock);
    Main::_breakpointsView->setFocusPolicy(TQWidget::NoFocus);
    dock->setWidget(Main::_breakpointsView);

    dock = createDock("console_dock_widget", loader.loadIcon("konsole", TDEIcon::Small),
                      i18n("Konsole"), DockPosition("compile_log_dock_widget"));
    Main::_consoleView = new ConsoleView(dock);
    dock->setWidget(Main::_consoleView);

// managers
    Programmer::manager = new Programmer::GuiManager(this);
    Programmer::manager->setView(_programLog);
    connect(Programmer::manager, TQ_SIGNAL(actionMessage(const TQString &)), _actionStatus, TQ_SLOT(setText(const TQString &)));
    connect(Programmer::manager, TQ_SIGNAL(showProgress(bool)), TQ_SLOT(showProgress(bool)));
    connect(Programmer::manager, TQ_SIGNAL(setTotalProgress(uint)), TQ_SLOT(setTotalProgress(uint)));
    connect(Programmer::manager, TQ_SIGNAL(setProgress(uint)), TQ_SLOT(setProgress(uint)));

    Debugger::manager = new Debugger::GuiManager;
    connect(Debugger::manager, TQ_SIGNAL(targetStateChanged()), TQ_SLOT(updateGUI()));
    connect(Debugger::manager, TQ_SIGNAL(statusChanged(const TQString &)), _debugStatus, TQ_SLOT(setText(const TQString &)));
    connect(Debugger::manager, TQ_SIGNAL(actionMessage(const TQString &)), _actionStatus, TQ_SLOT(setText(const TQString &)));

    Main::_compileManager = new Compile::Manager(this);
    Main::_compileManager->setView(Main::_compileLog);
    connect(Main::_compileManager, TQ_SIGNAL(success()), TQ_SLOT(compileSuccess()));
    connect(Main::_compileManager, TQ_SIGNAL(failure()), TQ_SLOT(compileFailure()));
    connect(Main::_compileManager, TQ_SIGNAL(updateFile(const Compile::FileData &)),
            TQ_SLOT(updateFile(const Compile::FileData &)));

// actions
    // file actions
    TDEAction *a = KStdAction::openNew(this, TQ_SLOT(newSourceFile()), actionCollection());
    a->setText(i18n("&New Source File..."));
    (void)new TDEAction(i18n("New hex File..."), "document-new", 0, this, TQ_SLOT(newHexFile()),
                      actionCollection(), "file_new_hex");
    KStdAction::open(this, TQ_SLOT(openFile()), actionCollection());
    TDERecentFilesAction *recent = KStdAction::openRecent(this, TQ_SLOT(openRecentFile(const KURL &)), actionCollection());
    recent->setMaxItems(20);
    recent->loadEntries(tdeApp->config(), "recent-files");
    (void)new TDEAction(i18n("Save All"), 0, 0, Main::_editorManager, TQ_SLOT(saveAllFiles()),
                      actionCollection(), "file_save_all");
    KStdAction::close(Main::_editorManager, TQ_SLOT(closeCurrentEditor()), actionCollection());
    (void)new TDEAction(i18n("C&lose All"), 0, 0, Main::_editorManager, TQ_SLOT(closeAllEditors()),
                      actionCollection(), "file_close_all");
    (void)new TDEAction(i18n("Close All Others"), 0, 0, Main::_editorManager, TQ_SLOT(closeAllOtherEditors()),
                      actionCollection(), "file_closeother");
    KStdAction::quit(this, TQ_SLOT(close()), actionCollection());

    // edit actions

    // view actions
    (void)new TDEAction(i18n("Back"), "back", TQt::ALT + TQt::Key_Left,
                      Main::_editorManager, TQ_SLOT(goBack()), actionCollection(), "history_back");
    (void)new TDEAction(i18n("Forward"), "forward", TQt::ALT + TQt::Key_Right,
                      Main::_editorManager, TQ_SLOT(goForward()), actionCollection(), "history_forward");
    (void)new TDEAction(i18n("Switch to..."), 0, TQt::CTRL + TQt::Key_Slash,
                      Main::_editorManager, TQ_SLOT(switchToEditor()), actionCollection(), "file_switchto");
    (void)new TDEAction(i18n("Switch Header/Implementation"), 0, TQt::SHIFT + TQt::Key_F12,
                      Main::_editorManager, TQ_SLOT(switchHeaderImplementation()), actionCollection(), "view_switch_source");
    (void)new TDEAction(TQString(), 0, 0,
                      Debugger::manager, TQ_SLOT(toggleBreakpoint()), actionCollection(), "toggle_breakpoint");
    (void)new TDEAction(TQString(), 0, 0,
                      Debugger::manager, TQ_SLOT(toggleEnableBreakpoint()), actionCollection(), "enable_breakpoint");
    (void)new TDEAction(i18n("Show disassembly location"), 0, 0,
                      Debugger::manager, TQ_SLOT(showDisassemblyLocation()), actionCollection(), "show_disassembly_location");
    TDEActionMenu *toolViewsMenu = new TDEActionMenu( i18n("Tool Views"), 0, "view_tool_views");
    connect(toolViewsMenu->popupMenu(), TQ_SIGNAL(aboutToShow()), TQ_SLOT(updateToolViewsActions()));
    actionCollection()->insert(toolViewsMenu);
    a = new TDEAction(i18n("&Reset Layout"), 0, 0, this, TQ_SLOT(resetDockLayout()), actionCollection(), "view_reset_layout");
    toolViewsMenu->insert(a);
    toolViewsMenu->popupMenu()->insertSeparator();
    TQValueList<DockData>::iterator it;
    for(it=_docks.begin(); it!=_docks.end(); ++it) toolViewsMenu->insert((*it).action);

    // project actions
    (void)new TDEAction(i18n("New Project..."), "piklab_createproject", 0,
                      this, TQ_SLOT(newProject()), actionCollection(), "project_new");
    (void)new TDEAction(i18n("Open Project..."), "piklab_openproject", 0,
                      this, TQ_SLOT(openProject()), actionCollection(), "project_open");
    recent = new TDERecentFilesAction(i18n("Open Recent Project"), 0,
                      this, TQ_SLOT(openRecentProject(const KURL &)), actionCollection(), "project_open_recent");
    recent->setMaxItems(20);
    recent->loadEntries(tdeApp->config(), "recent-projects");
    (void)new TDEAction(i18n("Project Options..."), "configure", 0,
                      this, TQ_SLOT(configureProject()), actionCollection(), "project_options");
    (void)new TDEAction(i18n("Close Project"), "window-close", 0,
                      this, TQ_SLOT(closeProject()), actionCollection(), "project_close");
    (void)new TDEAction(i18n("Add Source File..."), "piklab_addfile", 0,
                      Main::_projectManager, TQ_SLOT(insertSourceFiles()), actionCollection(), "project_add_source_file");
    (void)new TDEAction(i18n("Add Object File..."), "piklab_addfile", 0,
                      Main::_projectManager, TQ_SLOT(insertObjectFiles()), actionCollection(), "project_add_object_file");
    (void)new TDEAction(i18n("Add Current File"), "piklab_addcurrentfile", 0,
                      Main::_projectManager, TQ_SLOT(insertCurrentFile()), actionCollection(), "project_add_current_file");

    // build actions
    (void)new TDEAction(i18n("&Build Project"), "piklab_compile", TQt::Key_F8,
                      this, TQ_SLOT(buildProject()), actionCollection(), "build_build_project");
    (void)new TDEAction(i18n("&Compile File"), 0, TQt::SHIFT + TQt::Key_F8,
                      this, TQ_SLOT(compileFile()), actionCollection(), "build_compile_file");
    (void)new TDEAction(i18n("Clean"), "trashcan_empty", 0,
                      this, TQ_SLOT(cleanBuild()), actionCollection(), "build_clean");
    (void)new TDEAction(i18n("Stop"), "process-stop", 0,
                      this, TQ_SLOT(stopBuild()), actionCollection(), "build_stop");

    // programmer actions
    (void)new TDEAction(i18n("&Connect"), "connect_creating", 0,
                      Programmer::manager, TQ_SLOT(connectDevice()), actionCollection(), "prog_connect");
    (void)new TDEToggleAction(i18n("Device Power"), "piklab_power", 0,
                            Programmer::manager, TQ_SLOT(toggleDevicePower()), actionCollection(), "prog_power");
    (void)new TDEAction(i18n("&Disconnect"), "connect_no", 0,
                      Programmer::manager, TQ_SLOT(disconnectDevice()), actionCollection(), "prog_disconnect");
    (void)new TDEAction(i18n("&Program"), "piklab_burnchip", TQt::SHIFT + TQt::Key_F5,
                      this, TQ_SLOT(program()), actionCollection(), "prog_program");
    (void)new TDEAction(i18n("&Verify"), "piklab_verifychip", TQt::SHIFT + TQt::Key_F6,
                      this, TQ_SLOT(verify()), actionCollection(), "prog_verify");
    (void)new TDEAction(i18n("&Read"), "piklab_readchip", TQt::SHIFT + TQt::Key_F7,
                      this, TQ_SLOT(read()), actionCollection(), "prog_read");
    (void)new TDEAction(i18n("&Erase"), "piklab_erasechip", 0,
                      this, TQ_SLOT(erase()), actionCollection(), "prog_erase");
    (void)new TDEAction(i18n("&Blank Check"), "piklab_blankcheck", 0,
                      this, TQ_SLOT(blankCheck()), actionCollection(), "prog_blank_check");
    (void)new TDEAction(i18n("&Run"), "launch", TQt::SHIFT + TQt::Key_F9,
                      Programmer::manager, TQ_SLOT(run()), actionCollection(), "prog_run");
    (void)new TDEAction(i18n("&Stop"), "piklab_stop", 0,
                      Programmer::manager, TQ_SLOT(halt()), actionCollection(), "prog_stop");
    (void)new TDEAction(i18n("R&estart"), "piklab_restart", 0,
                      Programmer::manager, TQ_SLOT(restart()), actionCollection(), "prog_restart");
    (void)new TDEAction(i18n("&Advanced..."), 0, 0,
                      Programmer::manager , TQ_SLOT(showAdvancedDialog()), actionCollection(), "prog_advanced");
    (void)new TDEAction(i18n("Settings..."), "configure", 0,
                      this, TQ_SLOT(showProgrammerSettings()), actionCollection(), "prog_settings");

    // debugger actions
    (void)new TDEAction(i18n("&Start"), "launch", TQt::SHIFT + TQt::Key_F9,
                      Programmer::manager, TQ_SLOT(restart()), actionCollection(), "debug_start");
    (void)new TDEAction(i18n("&Run"), "piklab_run", TQt::SHIFT + TQt::Key_F9,
                      Programmer::manager, TQ_SLOT(run()), actionCollection(), "debug_run");
    (void)new TDEAction(i18n("&Step"), "piklab_debug_step", 0,
                      Programmer::manager, TQ_SLOT(step()), actionCollection(), "debug_next");
    //(void)new TDEAction(i18n("Step &In"), "piklab_debug_stepin",
    //             0, this, TQ_SLOT(debugStepIn()), actionCollection(), "debug_step_in");
    //(void)new TDEAction(i18n("Step &Out"), "piklab_debug_stepout",
    //             0, this, TQ_SLOT(debugStepOut()), actionCollection(), "debug_step_out");
    (void)new TDEAction(i18n("&Break<Translators: it is the verb>", "&Halt"), "piklab_debughalt", 0,
                      Programmer::manager, TQ_SLOT(halt()), actionCollection(), "debug_halt");
    (void)new TDEAction(i18n("&Disconnect/Stop"), "piklab_stop", 0,
                      Programmer::manager, TQ_SLOT(disconnectDevice()), actionCollection(), "debug_stop");
    (void)new TDEAction(i18n("R&eset"), "piklab_restart", 0,
                      Programmer::manager, TQ_SLOT(restart()), actionCollection(), "debug_reset");
    (void)new TDEAction(i18n("Show Program Counter"), "piklab_program_counter", 0,
                      Debugger::manager, TQ_SLOT(showPC()), actionCollection(), "debug_show_pc");
    (void)new TDEAction(i18n("Clear All Breakpoints"), "remove", 0,
                      Debugger::manager, TQ_SLOT(clearBreakpoints()), actionCollection(), "debug_clear_breakpoints");
    (void)new TDEAction(i18n("Settings..."), "configure", 0,
                      this, TQ_SLOT(showDebuggerSettings()), actionCollection(), "debug_settings");

    // tools
    (void)new TDEAction(i18n("&Pikloops..."), 0, 0,
                      this, TQ_SLOT(runPikloops()), actionCollection(), "tools_pikloops");
    (void)new TDEAction(i18n("&Find Files..."), "edit-find", 0,
                      this, TQ_SLOT(runKfind()), actionCollection(), "tools_kfind");
    (void)new TDEAction(i18n("&Device Information..."), "application-vnd.tde.info", 0,
                      this, TQ_SLOT(showDeviceInfo()), actionCollection(), "tools_device_information");
    (void)new TDEAction(i18n("&Config Generator..."), 0, 0,
                      this, TQ_SLOT(configGenerator()), actionCollection(), "tools_config_generator");
    (void)new TDEAction(i18n("&Template Generator..."), 0, 0,
                      this, TQ_SLOT(templateGenerator()), actionCollection(), "tools_template_generator");

    // settings actions
    (void)new TDEAction(i18n("Configure Toolchains..."), 0, 0,
                      this, TQ_SLOT(configureToolchains()), actionCollection(), "options_configure_toolchains");
    (void)KStdAction::preferences(this, TQ_SLOT(configure()), actionCollection());

    // help
    (void)new TDEAction(i18n("Report Bug..."), "likeback_bug", 0,
                      LikeBack::instance(), TQ_SLOT(iFoundABug()), actionCollection(), "help_report_bug_piklab");

    setupGUI();
    readDockConfig();

    // LikeBack buttons
    menuBar()->insertItem(new TQLabel(menuBar())); // #### first widget is put left-most...
    MenuBarButton *button = new MenuBarButton("likeback_like", menuBar());
    TQToolTip::add(button, i18n("I like..."));
    connect(button, TQ_SIGNAL(clicked()), LikeBack::instance(), TQ_SLOT(iLike()));
    menuBar()->insertItem(button);
    button = new MenuBarButton("likeback_dislike", menuBar());
    TQToolTip::add(button, i18n("I do not like..."));
    connect(button, TQ_SIGNAL(clicked()), LikeBack::instance(), TQ_SLOT(iDoNotLike()));
    menuBar()->insertItem(button);
    button = new MenuBarButton("likeback_bug", menuBar());
    TQToolTip::add(button, i18n("I found a bug..."));
    connect(button, TQ_SIGNAL(clicked()), LikeBack::instance(), TQ_SLOT(iFoundABug()));
    menuBar()->insertItem(button);
    button = new MenuBarButton("configure", menuBar());
    TQToolTip::add(button, i18n("Configure email..."));
    connect(button, TQ_SIGNAL(clicked()), LikeBack::instance(), TQ_SLOT(askEMail()));
    menuBar()->insertItem(button);
    button = new MenuBarButton("help", menuBar());
    connect(button, TQ_SIGNAL(clicked()), LikeBack::instance(), TQ_SLOT(showWhatsThisMessage()));
    menuBar()->insertItem(button);

    TQTimer::singleShot(0, this, TQ_SLOT(initialLoading()));
}

MainWindow::~MainWindow()
{}

void MainWindow::readDockConfig()
{
  KDockMainWindow::readDockConfig(tdeApp->config(), "dock_config");

  // if there is a new dock: it is not displayed by default...
  TQMap<TQString, TQString> entries = tdeApp->config()->entryMap("dock_config");
  TQValueList<DockData>::iterator it;
  for(it=_docks.begin(); it!=_docks.end(); ++it) {
    TQMap<TQString, TQString>::const_iterator eit;
    for(eit=entries.begin(); eit!=entries.end(); ++eit)
      if ( eit.key().startsWith((*it).dock->name()) ) break;
    if ( eit==entries.end() ) initDockPosition(*it);
  }

  // readDockConfig also restore the names/tooltips: what if a new version of the application changes these names...
  for(it=_docks.begin(); it!=_docks.end(); ++it) (*it).dock->setTabPageLabel((*it).title);
  TQApplication::postEvent(this, new TQEvent(TQEvent::CaptionChange));
}

void MainWindow::initialLoading()
{
  ::PBusyCursor bc;
  TDECmdLineArgs *args = TDECmdLineArgs::parsedArgs();
  if ( args->count()!=0 ) { // opened urls provided on the command line
    for (int i = 0; i<args->count(); i++) {
      PURL::Url url(args->url(i));
      if ( url.fileType()==PURL::Project ) {
        if ( Main::_projectManager->project()==0 ) Main::_projectManager->openProject(url);
      } else Main::_editorManager->openEditor(url);
    }
  } else { // otherwise reopen last project/files
    Main::_projectManager->openProject(GlobalConfig::openedProject());
    PURL::UrlList files = GlobalConfig::openedFiles();
    PURL::UrlList::const_iterator it = files.begin();
    for (; it!=files.end(); ++it) Main::_editorManager->openEditor(*it);
  }
  updateGUI();
}

void MainWindow::openRecentFile(const KURL &kurl)
{
  Main::_editorManager->openFile(PURL::Url(kurl));
}

void MainWindow::configureToolbar()
{
  saveMainWindowSettings(TDEGlobal::config(), "MainWindow");
  KEditToolbar dlg(actionCollection());
  connect(&dlg, TQ_SIGNAL(newToolbarConfig()), TQ_SLOT(applyToolbarSettings()));
  dlg.exec();
}

void MainWindow::applyToolbarSettings()
{
  createGUI();
  applyMainWindowSettings(TDEGlobal::config(), "MainWindow");
}

void MainWindow::configure(ConfigCenter::Type showType)
{
  stopOperations();
  ConfigCenter dialog(showType, this);
  dialog.exec();
  Programmer::manager->clear();
  updateGUI();
  Debugger::manager->update(true);
}

void MainWindow::configureToolchains()
{
  stopOperations();
  ToolchainsConfigCenter dialog(Main::toolGroup(), this);
  dialog.exec();
  Programmer::manager->clear();
  updateGUI();
  Debugger::manager->update(true);
}

void MainWindow::selectProgrammer(const Programmer::Group &group)
{
  if ( group.name()==Main::programmerGroup().name() ) return;
  bool debugInitialized = Debugger::manager->coff();
  stopOperations();
  GlobalConfig::writeProgrammerGroup(group);
  Programmer::manager->clear();
  updateGUI();
  if (debugInitialized) Debugger::manager->init();
  else Debugger::manager->update(true);
}

void MainWindow::selectTool(const Tool::Group &group)
{
  if ( group.name()==Compile::Config::toolGroup(Main::project()).name() ) return;
  bool debugInitialized = Debugger::manager->coff();
  stopOperations();
  Compile::Config::setToolGroup(Main::project(), group);
  updateGUI();
  if (debugInitialized) Debugger::manager->init();
  else Debugger::manager->update(true);
}

void MainWindow::setDevice(const TQString &device)
{
  if ( device==i18n(Device::AUTO_DATA.label) ) Compile::Config::setDevice(Main::project(), Device::AUTO_DATA.name);
  else Compile::Config::setDevice(Main::project(), device);
  updateGUI();
}

void MainWindow::showDeviceInfo()
{
  DeviceChooser::Dialog d(Main::device(), (Main::project() ? DeviceChooser::Choose : DeviceChooser::ChooseWithAuto), this);
  if ( d.exec() ) {
    setDevice(d.device());
    updateGUI();
  }
}

void MainWindow::configGenerator()
{
  PURL::FileType ftype = (Main::currentEditor() ? Main::currentEditor()->fileType() : PURL::Nb_FileTypes);
  PURL::SourceFamily family = (ftype!=PURL::Nb_FileTypes ? ftype.data().sourceFamily : PURL::SourceFamily(PURL::SourceFamily::Nb_Types));
  PURL::ToolType type = (family!=PURL::SourceFamily::Nb_Types ? family.data().toolType : PURL::ToolType(PURL::ToolType::Assembler));
  ConfigGenerator dialog(this);
  dialog.set(Main::deviceData(), Main::toolGroup(), type);
  dialog.exec();
}

void MainWindow::templateGenerator()
{
  PURL::FileType ftype = (Main::currentEditor() ? Main::currentEditor()->fileType() : PURL::Nb_FileTypes);
  PURL::SourceFamily family = (ftype!=PURL::Nb_FileTypes ? ftype.data().sourceFamily : PURL::SourceFamily(PURL::SourceFamily::Nb_Types));
  PURL::ToolType type = (family!=PURL::SourceFamily::Nb_Types ? family.data().toolType : PURL::ToolType(PURL::ToolType::Assembler));
  TemplateGenerator dialog(this);
  dialog.set(Main::deviceData(), Main::toolGroup(), type);
  dialog.exec();
}

bool MainWindow::queryClose()
{
  if ( !stopOperations() ) return false;
  Main::setState(Main::Closing);
  // save list of opened editors
  PURL::UrlList toSave;
  const PURL::UrlList files = Main::_editorManager->files();
  PURL::UrlList::const_iterator it;
  for (it=files.begin(); it!=files.end(); ++it)
    if ( !Main::_projectManager->isExternalFile(*it) ) toSave.append(*it);
  GlobalConfig::writeOpenedFiles(toSave);
  // close editors
  if ( !Main::_editorManager->closeAllEditors() ) {
    Main::setState(Main::Idle);
    return false;
  }
  // save other settings
  ::PBusyCursor bc;
  writeDockConfig(tdeApp->config(), "dock_config");
  static_cast<TDERecentFilesAction *>(Main::action("project_open_recent"))->saveEntries(tdeApp->config(), "recent-projects");
  static_cast<TDERecentFilesAction *>(Main::action("file_open_recent"))->saveEntries(tdeApp->config(), "recent-files");
  GlobalConfig::writeOpenedProject(Main::project() ? Main::project()->url() : PURL::Url());
  if ( Main::project() ) Main::_projectManager->closeProject();
  return true;
}

void MainWindow::newSourceFile()
{
  NewFileDialog dialog(Main::project(), this);
  if ( dialog.exec()!=TQDialog::Accepted ) return;
  if ( !dialog.url().exists() && !dialog.url().create(*Main::_compileLog) ) return;
  Main::_editorManager->openEditor(dialog.url());
  if ( dialog.addToProject() ) Main::_projectManager->insertFile(dialog.url());
}

void MainWindow::newHexFile()
{
  if ( Main::device()==Device::AUTO_DATA.name ) {
    MessageBox::sorry(i18n("You need to specify a device to create a new hex file."), Log::Show);
    return;
  }
  TQString s;
  for (uint i=0; true; i++) {
    s = i18n("Hex") + (i==0 ? TQString() : TQString::number(i));
    if ( Main::_editorManager->findEditor(s)==0 ) break;
  }
  HexEditor *editor = new HexEditor(s, Main::_editorManager);
  editor->memoryRead();
  Main::_editorManager->addEditor(editor);
}

bool MainWindow::openFile()
{
  TQString filter;
  filter += PURL::sourceFilter(PURL::SimpleFilter);
  filter += "\n" + PURL::filter(PURL::Hex);
  filter += "\n" + PURL::projectFilter(PURL::SimpleFilter);
  filter += "\n*|" + i18n("All Files");
  PURL::Url url = PURL::getOpenUrl(":open_file", filter, this ,i18n("Open File"));
  if ( url.fileType()==PURL::Project || url.fileType()==PURL::PikdevProject ) {
    stopOperations();
    if ( !Main::_projectManager->openProject(url) ) return false;
    updateGUI();
    return true;
  }
  return Main::_editorManager->openFile(url);
}

void MainWindow::updateGUI()
{
  bool idle = ( Main::_state==Main::Idle );
  switch (Main::_state) {
  case Main::Closing: return;
  case Main::Idle:
    showProgress(false);
    break;
  case Main::Compiling:
    _actionStatus->setText(Main::_compileManager->label());
    showProgress(true);
    makeWidgetDockVisible(Main::_compileLog);
    break;
  case Main::Programming:
    makeWidgetDockVisible(_programLog);
    break;
  }

  // update editor actions
  Main::_editorManager->updateTitles();
  Main::action("file_save_all")->setEnabled(Main::currentEditor());
  Main::action("file_close")->setEnabled(Main::currentEditor());
  Main::action("file_close_all")->setEnabled(Main::currentEditor());
  Main::action("file_closeother")->setEnabled(Main::_editorManager->nbEditors()>1);
  Main::action("options_configure")->setEnabled(idle);
  PURL::FileType currentType = (Main::currentEditor() ? Main::currentEditor()->fileType() : PURL::Nb_FileTypes);
  bool isSource = (currentType==PURL::Nb_FileTypes ? false : currentType.data().group==PURL::Source);
  bool isHeader = (currentType==PURL::Nb_FileTypes ? false : currentType.data().group==PURL::Header);
  Main::action("view_switch_source")->setEnabled(isSource || isHeader);
  Main::action("history_back")->setEnabled(Main::editorManager().history().hasBack());
  Main::action("history_forward")->setEnabled(Main::editorManager().history().hasForward());
  Main::action("show_disassembly_location")->setEnabled(Debugger::manager->coff()!=0 && (isSource || isHeader));

  // update project
  bool inProject = ( Main::currentEditor() && (currentType==PURL::Nb_FileTypes || Main::currentEditor()->url().isEmpty() || Main::_projectManager->contains(Main::currentEditor()->url())) );
  if ( Main::project()==0 && !inProject ) {
    if ( Main::currentEditor()==0 ) Main::_projectManager->closeProject();
    else if ( isSource ) Main::_projectManager->setStandalone(Main::currentEditor()->url(), currentType);
    else {
      PURL::FileType type;
      PURL::Url purl = Main::_projectManager->standaloneGenerator(Main::currentEditor()->url(), type);
      if ( type!=PURL::Nb_FileTypes ) Main::_projectManager->setStandalone(purl, type);
      else if ( currentType==PURL::Hex ) Main::_projectManager->setStandalone(purl, PURL::Hex);
    }
  }
  if ( Main::currentEditor() ) Main::_projectManager->select(Main::currentEditor());

  // update project actions
  Main::action("project_new")->setEnabled(idle);
  Main::action("project_open")->setEnabled(idle);
  Main::action("project_close")->setEnabled(Main::project() && idle);
  Main::action("project_options")->setEnabled(Main::project() && idle);
  Main::action("project_add_source_file")->setEnabled(Main::project() && idle);
  Main::action("project_add_object_file")->setEnabled(Main::project() && idle);
  Main::action("project_add_current_file")->setEnabled(Main::project() && !inProject && idle && isSource);

  // update build actions
  static_cast<PopupButton *>(_toolStatus->widget())->setText(" " + Main::toolGroup().label() + " ");
  bool hexProject = ( Main::_projectManager->projectUrl().fileType()==PURL::Hex );
  bool customTool = Main::toolGroup().isCustom();
  Main::action("build_build_project")->setEnabled((Main::project() || (inProject && !hexProject) ) && idle);
  PURL::Url selected = Main::_projectManager->selectedUrl();
  bool isSelectedSource = ( !selected.isEmpty() && selected.data().group==PURL::Source );
  Main::action("build_compile_file")->setEnabled(!hexProject && (isSource || isSelectedSource) && idle && !customTool);
  Main::action("build_clean")->setEnabled((Main::project() || inProject) && idle && !customTool);
  Main::action("build_stop")->setEnabled(Main::_state==Main::Compiling);

  // update programmer status
  PortType ptype = Programmer::GroupConfig::portType(Main::programmerGroup());
  static_cast<PopupButton *>(_programmerStatus->widget())->setText(" " + Main::programmerGroup().statusLabel(ptype) + " ");
  TQFont f = font();
  bool supported = (Main::deviceData() ? Main::programmerGroup().isSupported(Main::deviceData()->name()) : false);
  f.setItalic(!supported);
  _programmerStatus->widget()->setFont(f);
  bool isProgrammer = ( Main::programmerGroup().properties() & ::Programmer::Programmer );
  PURL::Url purl = Main::_projectManager->projectUrl();
  bool hasHex = ( currentType==PURL::Hex || Main::_projectManager->contains(purl.toFileType(PURL::Hex)) );
  Main::action("prog_connect")->setEnabled(isProgrammer && idle);
  Main::action("prog_read")->setEnabled(isProgrammer && idle);
  Main::action("prog_program")->setEnabled(isProgrammer && hasHex && idle);
  Main::action("prog_verify")->setEnabled(isProgrammer && hasHex && idle);
  Main::action("prog_erase")->setEnabled(isProgrammer && idle);
  Main::action("prog_blank_check")->setEnabled(isProgrammer && idle);
  Programmer::State pstate = (Main::programmer() ? Main::programmer()->state() : Programmer::NotConnected);
  static_cast<TDEToggleAction *>(Main::action("prog_power"))->setEnabled(isProgrammer && idle && pstate!=Programmer::NotConnected);
  Main::action("prog_disconnect")->setEnabled(isProgrammer && idle && pstate!=Programmer::NotConnected);
  bool isDebugger = ( Main::programmerGroup().properties() & ::Programmer::Debugger );
  bool resetAvailable = ( Main::programmerGroup().properties() & Programmer::CanReleaseReset );
  Main::action("prog_run")->setEnabled(idle && resetAvailable && !isDebugger && pstate!=Programmer::Running);
  Main::action("prog_stop")->setEnabled(idle && !isDebugger && pstate==Programmer::Running);
  Main::action("prog_restart")->setEnabled(idle && !isDebugger && pstate==Programmer::Running);
  const Programmer::GroupUI *pgui = static_cast<const Programmer::GroupUI *>(Main::programmerGroup().gui());
  Main::action("prog_advanced")->setEnabled(idle && pgui->hasAdvancedDialog());

  // update debugger status
  Debugger::manager->updateDevice();
  Main::action("debug_start")->setEnabled(idle && isDebugger && pstate!=Programmer::Running && pstate!=Programmer::Halted);
  Main::action("debug_run")->setEnabled(idle && isDebugger && pstate!=Programmer::Running && !Debugger::manager->isStepping() );
  Main::action("debug_halt")->setEnabled(idle && isDebugger && (pstate==Programmer::Running || Debugger::manager->isStepping()) );
  Main::action("debug_stop")->setEnabled(idle && isDebugger && (pstate==Programmer::Running || pstate==Programmer::Halted));
  Main::action("debug_next")->setEnabled(idle && isDebugger && pstate!=Programmer::Running);
  Main::action("debug_reset")->setEnabled(idle && isDebugger && (pstate==Programmer::Running || pstate==Programmer::Halted));
  Main::action("debug_show_pc")->setEnabled(idle && isDebugger && Debugger::manager->coff()!=0 && Debugger::manager->pc().isInitialized() );

  Main::_projectManager->updateGUI();

  // caption
  TQString caption;
  if ( Main::project() ) caption += Main::project()->name();
  if ( Main::currentEditor() ) {
    if ( Main::project() ) caption += " - ";
    caption += Main::currentEditor()->url().filepath();
  }
  setCaption(tdeApp->makeStdCaption(caption));

  emit stateChanged();
}

void MainWindow::updateToolViewsActions()
{
  TQValueList<DockData>::iterator it;
  for(it=_docks.begin(); it!=_docks.end(); ++it) (*it).action->setChecked((*it).dock->mayBeHide());
}

void MainWindow::initDockPosition(const DockData &ddata)
{
  const DockPosition &pos = ddata.position;
  ddata.dock->manualDock(manager()->getDockWidgetFromName(pos.parent), pos.pos, pos.space);
}

void MainWindow::resetDockLayout()
{
  TQValueList<DockData>::iterator it;
  for (it=_docks.begin(); it!=_docks.end(); ++it) initDockPosition(*it);
}

void MainWindow::toggleToolView(TQWidget *widget)
{
  static_cast<KDockWidget *>(widget)->changeHideShowState();
}

void MainWindow::runKfind()
{
  if (_kfindProcess) return;
  _kfindProcess = new ::Process::StringOutput;
  TQString path;
  PURL::Url url = Main::projectManager().projectUrl();
  if ( !url.isEmpty() ) path = url.path();
  _kfindProcess->setup("kfind", path, false);
  connect(_kfindProcess, TQ_SIGNAL(done(int)), TQ_SLOT(kfindDone()));
  if ( !_kfindProcess->start(0) )
    MessageBox::sorry(i18n("Could not run \"kfind\""), Log::Show);
}

void MainWindow::kfindDone()
{
  delete _kfindProcess;
  _kfindProcess = 0;
}

void MainWindow::runPikloops()
{
  if (_pikloopsProcess) return;
  _pikloopsProcess = new ::Process::StringOutput;
  _pikloopsProcess->setup("pikloops", TQStringList(), false);
  connect(_pikloopsProcess, TQ_SIGNAL(done(int)), TQ_SLOT(pikloopsDone()));
  if ( !_pikloopsProcess->start(0) )
    MessageBox::detailedSorry(i18n("Could not run \"pikloops\""), i18n("The Pikloops utility (%1) is not installed in your system.").arg("http://pikloops.sourceforge.net"), Log::Show);
}

void MainWindow::pikloopsDone()
{
  delete _pikloopsProcess;
  _pikloopsProcess = 0;
}

//-----------------------------------------------------------------------------
void MainWindow::compileFile()
{
  Editor *e = Main::currentEditor();
  if ( e && e->isModified() ) e->save(); // buffer is systematically saved
  stopOperations();
  Main::_compileLog->clear();
  PURL::Url url = (e ? e->url() : Main::_projectManager->selectedUrl());
  bool generated = (e ? Main::_projectManager->isExternalFile(url) : false);
  if ( Main::project()==0 || !generated ) Main::_projectManager->removeExternalFiles();
  url = (!generated ? url : Main::_projectManager->projectUrl());
  if ( Main::_compileManager->compileFile(Compile::TodoItem(url, generated)) ) Main::setState(Main::Compiling);
  else compileFailure();
}

void MainWindow::buildProject()
{
  if ( Main::project()==0 ) {
    compileFile();
    return;
  }
  // save modified buffers
  PURL::UrlList files = Main::project()->absoluteFiles();
  PURL::UrlList::const_iterator it;
  for (it=files.begin(); it!=files.end(); ++it) {
    // save modified editors
    Editor *e = Main::_editorManager->findEditor(*it);
    if ( e && e->isModified() ) e->save();
  }
  bool tmp = _forceProgramAfterBuild;
  stopOperations();
  _forceProgramAfterBuild = tmp;
  Main::_compileLog->clear();
  Main::_projectManager->removeExternalFiles();
  Compile::LinkType ltype = (Main::programmerGroup().name()=="icd2_debugger" ? Compile::Icd2Linking : Compile::NormalLinking);
  if ( Main::_compileManager->buildProject(ltype) ) Main::setState(Main::Compiling);
  else compileFailure();
}

void MainWindow::compileFailure()
{
  _forceProgramAfterBuild = false;
  Main::setState(Main::Idle);
}

void MainWindow::compileSuccess()
{
  if ( !Main::_compileManager->compileOnly() ) {
    Main::_projectManager->setModified(false);
    if ( Main::project()->outputType()==Tool::OutputType::Executable ) {
      if ( Debugger::manager->init() ) {
        const TQStringList &included = Debugger::manager->coff()->filenames();
        TQStringList::const_iterator it;
        for (it=included.begin(); it!=included.end(); ++it) {
          PURL::Directory dir = (Main::project() ? Main::project()->directory() : Main::projectManager().projectUrl().directory());
          PURL::Url url = PURL::Url::fromPathOrUrl(*it).toAbsolute(dir);
          if ( !url.exists() ) continue;
          Main::_projectManager->addExternalFile(url, ProjectManager::Included);
        }
      }
      if ( _forceProgramAfterBuild || readConfigEntry(BaseGlobalConfig::ProgramAfterBuild).toBool() ) program();
    }
  }
  _forceProgramAfterBuild = false;
  Main::setState(Main::Idle);
}

void MainWindow::updateFile(const Compile::FileData &fdata)
{
  if ( fdata.actions & Compile::InProject ) {
    if ( fdata.actions & Compile::Generated ) Main::_projectManager->addExternalFile(fdata.url, ProjectManager::Generated);
    else if ( fdata.actions & Compile::Included ) Main::_projectManager->addExternalFile(fdata.url, ProjectManager::Included);
    else Q_ASSERT(false);
  }
  if ( fdata.actions & Compile::Show ) {
    Editor *e = Main::_editorManager->openEditor(fdata.url);
    if (e) e->setReadOnly(true);
  }
}

void MainWindow::cleanBuild()
{
  stopOperations();
  Main::_compileLog->clear();
  if ( Main::project() ) {
    Compile::LinkType ltype = (Main::programmerGroup().name()=="icd2_debugger" ? Compile::Icd2Linking : Compile::NormalLinking);
    Main::_compileManager->cleanProject(ltype);
  } else {
    PURL::FileType type;
    PURL::Url url = Main::_projectManager->standaloneGenerator(Main::currentEditor()->url(), type);
    Main::_compileManager->cleanFile(url);
  }
  Main::_projectManager->removeExternalFiles();
}

void MainWindow::stopBuild()
{
  Main::_compileManager->kill();
}

//-----------------------------------------------------------------------------
void MainWindow::keyPressEvent(TQKeyEvent *e)
{
  if ( e->key()==Key_Escape ) stopOperations();
}

bool MainWindow::stopOperations()
{
  if ( Main::_state==Main::Programming ) {
    _programLog->log(Log::LineType::Warning, i18n("Programming in progress. Cannot be aborted."));
    return false;
  }
  stopBuild();
  Programmer::manager->stop();
  Debugger::manager->clear();
  return true;
}

void MainWindow::newProject()
{
  stopOperations();
  Main::_projectManager->newProject();
  updateGUI();
  Main::_compileLog->clear();
}

void MainWindow::openProject()
{
  stopOperations();
  Main::_projectManager->openProject();
  updateGUI();
  Main::_compileLog->clear();
}

void MainWindow::openRecentProject(const KURL &url)
{
  stopOperations();
  Main::_projectManager->openProject(PURL::Url(url));
  updateGUI();
  Main::_compileLog->clear();
}

void MainWindow::configureProject()
{
  stopOperations();
  if ( Main::project()==0 ) configure(ConfigCenter::Standalone);
  else Main::_projectManager->editProject();
  updateGUI();
}

void MainWindow::closeProject()
{
  stopOperations();
  Main::_projectManager->closeProject();
  updateGUI();
  Main::_compileLog->clear();
}

//----------------------------------------------------------------------------
HexEditor *MainWindow::getHexEditor()
{
  if ( Main::_projectManager->isModified() || Main::_projectManager->needsRecompile() ) {
    MessageBox::Result res = MessageBox::Yes;
    if ( !readConfigEntry(BaseGlobalConfig::AutoRebuildModified).toBool() ) {
      res = MessageBox::questionYesNoCancel(i18n("The project hex file may not be up-to-date since some project files have been modified."),
                                            i18n("Recompile First"), i18n("Continue Anyway"));
      if ( res==MessageBox::Cancel ) return 0;
    }
    if ( res==MessageBox::Yes ) {
      _forceProgramAfterBuild = true;
      buildProject();
      return 0;
    }
  }
  if ( Main::currentEditor() && Main::currentEditor()->fileType()==PURL::Hex )
    return static_cast<HexEditor *>(Main::currentEditor());
  PURL::Url purl = Main::_projectManager->projectUrl();
  HexEditor *editor = static_cast<HexEditor *>(Main::_editorManager->openEditor(purl.toFileType(PURL::Hex)));
  if ( editor==0 ) return 0;
  editor->setReadOnly(true);
  return editor;
}

void MainWindow::erase()
{
  Programmer::manager->erase(Device::MemoryRange());
}

void MainWindow::blankCheck()
{
  Programmer::manager->blankCheck(Device::MemoryRange());
}

void MainWindow::program()
{
  HexEditor *editor = getHexEditor();
  if ( editor==0 ) return;
  if ( Main::programmerGroup().isDebugger() && !Main::_projectManager->contains(editor->url()) ) {
    MessageBox::sorry(i18n("It is not possible to start a debugging session with an hex file not generated with the current project."), Log::Show);
    return;
  }
  Programmer::manager->program(*editor->memory(), Device::MemoryRange());
}

void MainWindow::verify()
{
  HexEditor *editor = getHexEditor();
  if ( editor==0 ) return;
  Programmer::manager->verify(*editor->memory(), Device::MemoryRange());
}

void MainWindow::read()
{
  TQString s = i18n("Read");
  Editor *e = Main::_editorManager->findEditor(s);
  if (e) Main::_editorManager->closeEditor(e, true);
  HexEditor *editor = new HexEditor(s, Main::_editorManager);
  editor->setDevice();
  if ( Programmer::manager->read(*editor->memory(), Device::MemoryRange()) ) {
    editor->memoryRead();
    Main::_editorManager->addEditor(editor);
  } else delete editor;
}

void MainWindow::showProgress(bool show)
{
  if (show) {
    PBusyCursor::start();
    _actionStatus->show();
    _actionProgress->show();
  } else {
    PBusyCursor::stop();
    _actionStatus->hide();
    _actionProgress->hide();
  }
}

void MainWindow::setTotalProgress(uint nb)
{
  //TDEIO::Job *job = new TDEIO::SimpleJob(KURL(), 0, 0, false);
  //int id = Observer::self()->newJob(job, true);
  //Observer::self()->slotTotalSize(job, total);
  //Observer::self()->slotInfoMessage(job, "test");
  //tqDebug("set total steps: %i", total);
  _actionProgress->setTotalSteps(nb);
  _actionProgress->setProgress(0);
  TQApplication::eventLoop()->processEvents(TQEventLoop::ExcludeUserInput); // #### DANGER !!!!
}

void MainWindow::setProgress(uint nb)
{
  _actionProgress->setProgress(nb);
  TQApplication::eventLoop()->processEvents(TQEventLoop::ExcludeUserInput); // #### DANGER !!!!
}

#include "toplevel.moc"
