/*
 *  File : snippet_part.cpp
 *
 *  Author: Robert Gruber <rgruber@users.sourceforge.net>
 *
 *  Copyright: See COPYING file that comes with this distribution
 */
#include "snippet_part.h"

#include <tqwhatsthis.h>

#include <tdeaction.h>
#include <kiconloader.h>
#include <tdelocale.h>
#include <tdeaboutdata.h>
#include <tqvbox.h>
#include <kdialogbase.h>
#include <klineedit.h>
#include <tqcheckbox.h>
#include <tqbuttongroup.h>
#include <ktrader.h>

#include "kdevcore.h"
#include <kdevproject.h>
#include "kdevmainwindow.h"
#include <kdevgenericfactory.h>
#include <kdevplugininfo.h>

#include <kdebug.h>

#include "snippet_widget.h"
#include "snippetitem.h"
#include "snippetsettings.h"
#include "snippetconfig.h"

static const KDevPluginInfo pluginData("kdevsnippet");
typedef KDevGenericFactory<SnippetPart> snippetFactory;
K_EXPORT_COMPONENT_FACTORY( libkdevsnippet, snippetFactory( pluginData ) )

SnippetPart::SnippetPart(TQObject *parent, const char *name, const TQStringList& )
  : KDevPlugin(&pluginData, parent, name ? name : "SnippetPart" )
{
  setInstance(snippetFactory::instance());
  setXMLFile("kdevpart_snippet.rc");

  m_widget = new SnippetWidget(this);
  m_widget->setCaption(i18n("Code Snippets"));
  m_widget->setIcon(SmallIcon( info()->icon() ));

  TQWhatsThis::add(m_widget, i18n("<b>Code Snippet</b><p>This is a list of available snippets."));

  mainWindow()->embedSelectViewRight( m_widget, i18n("Code Snippets"), i18n("Insert a code snippet") );

  connect( core(), TQ_SIGNAL( configWidget( KDialogBase * ) ), this, TQ_SLOT( slotConfigWidget( KDialogBase * ) ) );
  
  /*The next two connects are used to check if certain SnippetGroups need to be opened 
    according to the languages supported by this project*/
  connect( core(), TQ_SIGNAL( projectOpened() ), m_widget, TQ_SLOT( languageChanged() ) );
  connect( core(), TQ_SIGNAL( languageChanged() ), m_widget, TQ_SLOT( languageChanged() ) );

  setupActions();
}

SnippetPart::~SnippetPart()
{
  if ( m_widget )
  {
    mainWindow()->removeView( m_widget );
  }

  delete m_widget;
}

void SnippetPart::setupActions()
{
  new TDEAction( i18n("Show Snippet Tree"), CTRL+ALT+SHIFT+Key_S, this, TQ_SLOT(slotShowView()), actionCollection(), "snippet_showview");
}

/*!
    \fn SnippetPart::aboutData()
 */
TDEAboutData* SnippetPart::aboutData()
{
  TDEAboutData *data = new TDEAboutData  ("snippetpart", I18N_NOOP("SnippetPart"), "1.1",
                                                           I18N_NOOP( "SnippetPart for TDevelop" ),
                                                           TDEAboutData::License_LGPL_V2,
                                                           I18N_NOOP( "(c) 2003" ), 0, "http://www.kdevelop.org");
  data->addAuthor ("Robert Gruber", I18N_NOOP("Author"), "rgruber@users.sourceforge.net", "http://kmp3indexer.sf.net");


  return data;
}


/*!
    \fn SnippetPart::slotConfigWidget( KDialogBase *dlg )
 */
void SnippetPart::slotConfigWidget( KDialogBase *dlg )
{
  TQVBox *vbox = dlg->addVBoxPage( i18n("Code Snippets"), i18n("Code Snippets"), BarIcon( info()->icon(), TDEIcon::SizeMedium ) );

  SnippetSettings * w = new SnippetSettings( m_widget, vbox );

  w->btnGroup->setButton(m_widget->getSnippetConfig()->getInputMethod());
  w->leDelimiter->setText(m_widget->getSnippetConfig()->getDelimiter());
  w->cbToolTip->setChecked(m_widget->getSnippetConfig()->useToolTips());
  w->btnGroupAutoOpen->setButton(m_widget->getSnippetConfig()->getAutoOpenGroups());
  connect( dlg, TQ_SIGNAL(okClicked()), w, TQ_SLOT(slotOKClicked()) );
}


TQStringList SnippetPart::getAllLanguages()
{
    TDETrader::OfferList languageSupportOffers =
        TDETrader::self()->query(TQString::fromLatin1("TDevelop/LanguageSupport"),
                               TQString::fromLatin1("[X-TDevelop-Version] == %1"
                               ).arg( TDEVELOP_PLUGIN_VERSION ));
    
    TQStringList languages;

    for (TDETrader::OfferList::ConstIterator it = languageSupportOffers.begin(); it != languageSupportOffers.end(); ++it)
    {
        TQString language = (*it)->property("X-TDevelop-Language").toString();
        languages.append(language);
        
        kdDebug(9035) << "Found language: " << (*it)->property("X-TDevelop-Language").toString() << endl <<
        "genericName(): " <<(*it)->genericName() << endl << 
        "comment(): " <<(*it)->comment() << endl << endl;
    }
    
    return languages;
}


TQStringList SnippetPart::getProjectLanguages()
{
    TQStringList languages;
    if (!projectDom())
        return languages;

    TQDomDocument m_projectDom = *projectDom();
    
    if (m_widget->getSnippetConfig()->getAutoOpenGroups() == 1)
        languages = DomUtil::readListEntry(m_projectDom, "/general/secondaryLanguages", "language");
    languages.prepend( DomUtil::readEntry(m_projectDom, "/general/primarylanguage") );
    return languages;
}

void SnippetPart::slotShowView()
{
    mainWindow()->raiseView( m_widget );
}


#include "snippet_part.moc"
