# Copyright 2024 The Chromium Authors
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
"""Codegen common to .h files."""

import common
import java_types


def class_accessors(sb, java_classes, module_name):
  split_arg = f'"{module_name}", ' if module_name else ''
  for java_class in java_classes:
    if java_class in (java_types.OBJECT_CLASS, java_types.STRING_CLASS):
      continue
    escaped_name = java_class.to_cpp()
    # #ifdef needed when multple .h files are #included that common classes.
    sb(f"""\
#ifndef {escaped_name}_clazz_defined
#define {escaped_name}_clazz_defined
""")
    # Uses std::atomic<> instead of "static jclass cached_class = ..." because
    # that moves the initialize-once logic into the helper method (smaller code
    # size).
    sb(f"""\
inline jclass {escaped_name}_clazz(JNIEnv* env) {{
  static const char kClassName[] = "{java_class.full_name_with_slashes}";
  static std::atomic<jclass> cached_class;
  return jni_zero::internal::LazyGetClass(env, kClassName, {split_arg}&cached_class);
}}
#endif

""")


def class_accessor_expression(java_class):
  if java_class == java_types.OBJECT_CLASS:
    return 'jni_zero::g_object_class'
  if java_class == java_types.STRING_CLASS:
    return 'jni_zero::g_string_class'

  return f'{java_class.to_cpp()}_clazz(env)'


def header_preamble(script_name,
                    java_class,
                    system_includes,
                    user_includes,
                    header_guard=None):
  if header_guard is None:
    header_guard = f'{java_class.to_cpp()}_JNI'
  sb = []
  sb.append(f"""\
// This file was generated by
//     {script_name}
// For
//     {java_class.full_name_with_dots}

#ifndef {header_guard}
#define {header_guard}

""")
  sb.extend(f'#include <{x}>\n' for x in system_includes)
  sb.append('\n')
  sb.extend(f'#include "{x}"\n' for x in user_includes)
  preamble = ''.join(sb)

  epilogue = f"""
#endif  // {header_guard}
"""
  return preamble, epilogue
