// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/cr_components/theme_color_picker/theme_color_picker_handler.h"

#include <functional>
#include <memory>
#include <string>
#include <vector>

#include "base/containers/fixed_flat_map.h"
#include "base/memory/raw_ptr.h"
#include "base/memory/raw_ptr_exclusion.h"
#include "base/test/mock_callback.h"
#include "base/test/scoped_feature_list.h"
#include "chrome/browser/new_tab_page/chrome_colors/generated_colors_info.h"
#include "chrome/browser/search/background/ntp_custom_background_service.h"
#include "chrome/browser/search/background/ntp_custom_background_service_observer.h"
#include "chrome/browser/themes/theme_service.h"
#include "chrome/browser/themes/theme_service_factory.h"
#include "chrome/browser/ui/color/chrome_color_id.h"
#include "chrome/browser/ui/webui/cr_components/theme_color_picker/customize_chrome_colors.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "chrome/test/base/testing_profile.h"
#include "components/keyed_service/core/keyed_service.h"
#include "content/public/browser/web_contents.h"
#include "content/public/test/browser_task_environment.h"
#include "content/public/test/test_web_contents_factory.h"
#include "extensions/browser/extension_registry.h"
#include "extensions/common/extension_builder.h"
#include "mojo/public/cpp/bindings/pending_receiver.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/skia/include/core/SkColor.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/mojom/themes.mojom.h"
#include "ui/base/ui_base_features.h"
#include "ui/color/color_provider.h"
#include "ui/color/color_provider_key.h"
#include "ui/color/dynamic_color/palette_factory.h"
#include "ui/native_theme/native_theme.h"

namespace content {
class BrowserContext;
}  // namespace content

namespace {

using testing::_;

ui::ColorProviderKey::SchemeVariant GetSchemeVariant(
    ui::mojom::BrowserColorVariant color_variant) {
  using BCV = ui::mojom::BrowserColorVariant;
  using SV = ui::ColorProviderKey::SchemeVariant;
  static constexpr auto kSchemeVariantMap = base::MakeFixedFlatMap<BCV, SV>({
      {BCV::kTonalSpot, SV::kTonalSpot},
      {BCV::kNeutral, SV::kNeutral},
      {BCV::kVibrant, SV::kVibrant},
      {BCV::kExpressive, SV::kExpressive},
  });
  return kSchemeVariantMap.at(color_variant);
}

class MockClient : public theme_color_picker::mojom::ThemeColorPickerClient {
 public:
  MockClient() = default;
  ~MockClient() override = default;

  mojo::PendingRemote<theme_color_picker::mojom::ThemeColorPickerClient>
  BindAndGetRemote() {
    DCHECK(!receiver_.is_bound());
    return receiver_.BindNewPipeAndPassRemote();
  }

  void FlushForTesting() { receiver_.FlushForTesting(); }

  MOCK_METHOD(void, SetTheme, (theme_color_picker::mojom::ThemePtr));

  mojo::Receiver<theme_color_picker::mojom::ThemeColorPickerClient> receiver_{
      this};
};

class MockNtpCustomBackgroundService : public NtpCustomBackgroundService {
 public:
  explicit MockNtpCustomBackgroundService(Profile* profile)
      : NtpCustomBackgroundService(profile) {}
  MOCK_METHOD(absl::optional<CustomBackground>, GetCustomBackground, ());
  MOCK_METHOD(void, ResetCustomBackgroundInfo, ());
  MOCK_METHOD(void, AddObserver, (NtpCustomBackgroundServiceObserver*));
};

class MockThemeService : public ThemeService {
 public:
  MockThemeService() : ThemeService(nullptr, theme_helper_) { set_ready(); }
  using ThemeService::NotifyThemeChanged;
  MOCK_METHOD(void, UseDefaultTheme, ());
  MOCK_CONST_METHOD0(UsingDefaultTheme, bool());
  MOCK_METHOD(void, UseDeviceTheme, (bool));
  MOCK_CONST_METHOD0(UsingSystemTheme, bool());
  MOCK_CONST_METHOD0(UsingExtensionTheme, bool());
  MOCK_CONST_METHOD0(UsingPolicyTheme, bool());
  MOCK_CONST_METHOD0(GetThemeID, std::string());
  MOCK_METHOD(void, BuildAutogeneratedThemeFromColor, (SkColor));
  MOCK_CONST_METHOD0(GetAutogeneratedThemeColor, SkColor());
  MOCK_CONST_METHOD0(GetBrowserColorScheme, ThemeService::BrowserColorScheme());
  MOCK_METHOD(void, SetUserColor, (absl::optional<SkColor>));
  MOCK_CONST_METHOD0(GetUserColor, absl::optional<SkColor>());
  MOCK_CONST_METHOD0(UsingDeviceTheme, bool());
  MOCK_CONST_METHOD0(GetBrowserColorVariant, ui::mojom::BrowserColorVariant());
  MOCK_METHOD(void,
              SetUserColorAndBrowserColorVariant,
              (SkColor, ui::mojom::BrowserColorVariant));
  MOCK_METHOD(void, SetIsGrayscale, (bool));
  MOCK_CONST_METHOD0(GetIsGrayscale, bool());

 private:
  ThemeHelper theme_helper_;
};

std::unique_ptr<TestingProfile> MakeTestingProfile() {
  TestingProfile::Builder profile_builder;
  profile_builder.AddTestingFactory(
      ThemeServiceFactory::GetInstance(),
      base::BindRepeating([](content::BrowserContext* context)
                              -> std::unique_ptr<KeyedService> {
        return std::make_unique<testing::NiceMock<MockThemeService>>();
      }));
  return profile_builder.Build();
}

}  // namespace

class ThemeColorPickerHandlerTest : public testing::Test {
 public:
  ThemeColorPickerHandlerTest()
      : profile_(MakeTestingProfile()),
        mock_ntp_custom_background_service_(profile_.get()),
        web_contents_(web_contents_factory_.CreateWebContents(profile_.get())),
        mock_theme_service_(static_cast<MockThemeService*>(
            ThemeServiceFactory::GetForProfile(profile_.get()))) {}

  void SetUp() override {
    EXPECT_CALL(mock_ntp_custom_background_service_, AddObserver)
        .Times(1)
        .WillOnce(
            testing::SaveArg<0>(&ntp_custom_background_service_observer_));
    handler_ = std::make_unique<ThemeColorPickerHandler>(
        mojo::PendingReceiver<
            theme_color_picker::mojom::ThemeColorPickerHandler>(),
        mock_client_.BindAndGetRemote(), &mock_ntp_custom_background_service_,
        web_contents_);
    mock_client_.FlushForTesting();
    EXPECT_EQ(handler_.get(), ntp_custom_background_service_observer_);

    scoped_feature_list_.Reset();
  }

  TestingProfile& profile() { return *profile_; }
  content::WebContents& web_contents() { return *web_contents_; }
  ThemeColorPickerHandler& handler() { return *handler_; }
  NtpCustomBackgroundServiceObserver& ntp_custom_background_service_observer() {
    return *ntp_custom_background_service_observer_;
  }
  MockThemeService& mock_theme_service() { return *mock_theme_service_; }

 protected:
  // NOTE: The initialization order of these members matters.
  content::BrowserTaskEnvironment task_environment_;
  std::unique_ptr<TestingProfile> profile_;
  testing::NiceMock<MockNtpCustomBackgroundService>
      mock_ntp_custom_background_service_;
  // This field is not a raw_ptr<> because it was filtered by the rewriter for:
  // #addr-of
  RAW_PTR_EXCLUSION NtpCustomBackgroundServiceObserver*
      ntp_custom_background_service_observer_;
  content::TestWebContentsFactory web_contents_factory_;
  raw_ptr<content::WebContents> web_contents_;
  testing::NiceMock<MockClient> mock_client_;
  raw_ptr<MockThemeService> mock_theme_service_;
  base::test::ScopedFeatureList scoped_feature_list_;
  std::unique_ptr<ThemeColorPickerHandler> handler_;
};

TEST_F(ThemeColorPickerHandlerTest, GetChromeColorsExtended) {
  std::vector<theme_color_picker::mojom::ChromeColorPtr> colors;
  base::MockCallback<ThemeColorPickerHandler::GetChromeColorsCallback> callback;
  EXPECT_CALL(callback, Run(testing::_))
      .Times(1)
      .WillOnce(testing::Invoke(
          [&colors](std::vector<theme_color_picker::mojom::ChromeColorPtr>
                        colors_arg) { colors = std::move(colors_arg); }));
  handler().GetChromeColors(false, true, callback.Get());

  auto num_colors = sizeof(chrome_colors::kGeneratedColorsInfo) /
                    sizeof(chrome_colors::kGeneratedColorsInfo[0]);
  ASSERT_EQ(num_colors, colors.size());
  for (size_t i = 0; i < num_colors; i++) {
    EXPECT_EQ(l10n_util::GetStringUTF8(
                  chrome_colors::kGeneratedColorsInfo[i].label_id),
              colors[i]->name);
    EXPECT_EQ(chrome_colors::kGeneratedColorsInfo[i].color, colors[i]->seed);
    EXPECT_EQ(GetAutogeneratedThemeColors(
                  chrome_colors::kGeneratedColorsInfo[i].color)
                  .active_tab_color,
              colors[i]->background);
    EXPECT_EQ(GetAutogeneratedThemeColors(
                  chrome_colors::kGeneratedColorsInfo[i].color)
                  .frame_color,
              colors[i]->foreground);
  }
}

TEST_F(ThemeColorPickerHandlerTest, GetChromeColors) {
  std::vector<theme_color_picker::mojom::ChromeColorPtr> colors;
  base::MockCallback<ThemeColorPickerHandler::GetChromeColorsCallback> callback;
  EXPECT_CALL(callback, Run(testing::_))
      .Times(1)
      .WillOnce(testing::Invoke(
          [&colors](std::vector<theme_color_picker::mojom::ChromeColorPtr>
                        colors_arg) { colors = std::move(colors_arg); }));
  handler().GetChromeColors(false, false, callback.Get());

  ASSERT_EQ(kCustomizeChromeColors.size(), colors.size());
  for (size_t i = 0; i < kCustomizeChromeColors.size(); i++) {
    EXPECT_EQ(l10n_util::GetStringUTF8(kCustomizeChromeColors[i].label_id),
              colors[i]->name);
    EXPECT_EQ(kCustomizeChromeColors[i].color, colors[i]->seed);
    EXPECT_EQ(GetAutogeneratedThemeColors(kCustomizeChromeColors[i].color)
                  .active_tab_color,
              colors[i]->background);
    EXPECT_EQ(GetAutogeneratedThemeColors(kCustomizeChromeColors[i].color)
                  .frame_color,
              colors[i]->foreground);
  }
}

TEST_F(ThemeColorPickerHandlerTest, GetChromeColorsGM3) {
  scoped_feature_list_.InitWithFeatures(
      {features::kChromeRefresh2023, features::kChromeWebuiRefresh2023}, {});
  std::vector<theme_color_picker::mojom::ChromeColorPtr> colors;
  base::MockCallback<ThemeColorPickerHandler::GetChromeColorsCallback> callback;
  EXPECT_CALL(callback, Run(testing::_))
      .Times(2)
      .WillRepeatedly(testing::Invoke(
          [&colors](std::vector<theme_color_picker::mojom::ChromeColorPtr>
                        colors_arg) { colors = std::move(colors_arg); }));

  // Test with Dark Mode on.
  handler().GetChromeColors(true, false, callback.Get());

  ASSERT_EQ(kDynamicCustomizeChromeColors.size(), colors.size());
  for (size_t i = 0; i < kDynamicCustomizeChromeColors.size(); i++) {
    auto palette = ui::GeneratePalette(
        kDynamicCustomizeChromeColors[i].color,
        GetSchemeVariant(kDynamicCustomizeChromeColors[i].variant));
    EXPECT_EQ(
        l10n_util::GetStringUTF8(kDynamicCustomizeChromeColors[i].label_id),
        colors[i]->name);
    EXPECT_EQ(kDynamicCustomizeChromeColors[i].color, colors[i]->seed);
    EXPECT_EQ(palette->primary().get(80), colors[i]->background);
    EXPECT_EQ(palette->primary().get(30), colors[i]->foreground);
    EXPECT_EQ(palette->secondary().get(50), colors[i]->base);
  }

  // Test with Dark Mode off.
  handler().GetChromeColors(false, false, callback.Get());

  ASSERT_EQ(kDynamicCustomizeChromeColors.size(), colors.size());
  for (size_t i = 0; i < kDynamicCustomizeChromeColors.size(); i++) {
    auto palette = ui::GeneratePalette(
        kDynamicCustomizeChromeColors[i].color,
        GetSchemeVariant(kDynamicCustomizeChromeColors[i].variant));
    EXPECT_EQ(
        l10n_util::GetStringUTF8(kDynamicCustomizeChromeColors[i].label_id),
        colors[i]->name);
    EXPECT_EQ(kDynamicCustomizeChromeColors[i].color, colors[i]->seed);
    EXPECT_EQ(palette->primary().get(40), colors[i]->background);
    EXPECT_EQ(palette->primary().get(90), colors[i]->foreground);
    EXPECT_EQ(palette->primary().get(80), colors[i]->base);
  }
}

enum class ThemeUpdateSource {
  kMojo,
  kThemeService,
  kNativeTheme,
  kCustomBackgroundService,
};

class ThemeColorPickerHandlerSetThemeTest
    : public ThemeColorPickerHandlerTest,
      public ::testing::WithParamInterface<ThemeUpdateSource> {
 public:
  void UpdateTheme() {
    switch (GetParam()) {
      case ThemeUpdateSource::kMojo:
        handler().UpdateTheme();
        break;
      case ThemeUpdateSource::kThemeService:
        mock_theme_service().NotifyThemeChanged();
        break;
      case ThemeUpdateSource::kNativeTheme:
        ui::NativeTheme::GetInstanceForNativeUi()->NotifyOnNativeThemeUpdated();
        break;
      case ThemeUpdateSource::kCustomBackgroundService:
        ntp_custom_background_service_observer()
            .OnCustomBackgroundImageUpdated();
        break;
    }
  }
};

TEST_P(ThemeColorPickerHandlerSetThemeTest, SetTheme) {
  theme_color_picker::mojom::ThemePtr theme;
  EXPECT_CALL(mock_client_, SetTheme)
      .Times(1)
      .WillOnce(
          testing::Invoke([&theme](theme_color_picker::mojom::ThemePtr arg) {
            theme = std::move(arg);
          }));
  CustomBackground custom_background;
  custom_background.custom_background_url = GURL("https://foo.com/img.png");
  custom_background.custom_background_attribution_line_1 = "foo line";
  custom_background.is_uploaded_image = false;
  custom_background.custom_background_main_color = SK_ColorGREEN;
  custom_background.collection_id = "test_collection";
  custom_background.daily_refresh_enabled = false;
  ON_CALL(mock_ntp_custom_background_service_, GetCustomBackground())
      .WillByDefault(testing::Return(absl::make_optional(custom_background)));
  ON_CALL(mock_theme_service(), GetAutogeneratedThemeColor())
      .WillByDefault(testing::Return(SK_ColorBLUE));
  ON_CALL(mock_theme_service(), GetUserColor())
      .WillByDefault(testing::Return(absl::optional<SkColor>()));
  ON_CALL(mock_theme_service(), UsingDefaultTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingSystemTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingPolicyTheme())
      .WillByDefault(testing::Return(true));
  ON_CALL(mock_theme_service(), GetIsGrayscale())
      .WillByDefault(testing::Return(true));
  ON_CALL(mock_theme_service(), UsingDeviceTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), GetBrowserColorVariant())
      .WillByDefault(testing::Return(ui::mojom::BrowserColorVariant::kNeutral));
  ui::NativeTheme::GetInstanceForNativeUi()->set_use_dark_colors(true);

  UpdateTheme();
  mock_client_.FlushForTesting();

  ASSERT_TRUE(theme);
  ASSERT_TRUE(theme->has_background_image);
  EXPECT_EQ(SK_ColorGREEN,
            theme->background_image_main_color.value_or(SK_ColorWHITE));
  EXPECT_TRUE(theme->is_dark_mode);
  EXPECT_EQ(SK_ColorBLUE, theme->seed_color);
  EXPECT_EQ(
      web_contents().GetColorProvider().GetColor(kColorNewTabPageBackground),
      theme->background_color);
  EXPECT_EQ(web_contents().GetColorProvider().GetColor(ui::kColorFrameActive),
            theme->foreground_color);
  EXPECT_EQ(web_contents().GetColorProvider().GetColor(kColorNewTabPageText),
            theme->color_picker_icon_color);
  EXPECT_TRUE(theme->colors_managed_by_policy);
  EXPECT_TRUE(theme->is_grey_baseline);
  EXPECT_EQ(theme->browser_color_variant,
            ui::mojom::BrowserColorVariant::kNeutral);
  EXPECT_FALSE(theme->follow_device_theme);
}

TEST_P(ThemeColorPickerHandlerSetThemeTest, SetThemeColorSchemeGM3) {
  scoped_feature_list_.InitWithFeatures(
      {features::kChromeRefresh2023, features::kChromeWebuiRefresh2023}, {});
  theme_color_picker::mojom::ThemePtr theme;
  EXPECT_CALL(mock_client_, SetTheme)
      .Times(2)
      .WillRepeatedly(
          testing::Invoke([&theme](theme_color_picker::mojom::ThemePtr arg) {
            theme = std::move(arg);
          }));

  // Set mocks needed for UpdateTheme.
  CustomBackground custom_background;
  custom_background.custom_background_url = GURL("https://foo.com/img.png");
  custom_background.custom_background_attribution_line_1 = "foo line";
  custom_background.is_uploaded_image = false;
  custom_background.custom_background_main_color = SK_ColorGREEN;
  custom_background.collection_id = "test_collection";
  custom_background.daily_refresh_enabled = false;
  ON_CALL(mock_ntp_custom_background_service_, GetCustomBackground())
      .WillByDefault(testing::Return(absl::make_optional(custom_background)));
  ON_CALL(mock_theme_service(), GetAutogeneratedThemeColor())
      .WillByDefault(testing::Return(SK_ColorBLUE));
  ON_CALL(mock_theme_service(), GetUserColor())
      .WillByDefault(testing::Return(SK_ColorGREEN));
  ON_CALL(mock_theme_service(), UsingDefaultTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingSystemTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingPolicyTheme())
      .WillByDefault(testing::Return(true));
  ON_CALL(mock_theme_service(), GetIsGrayscale())
      .WillByDefault(testing::Return(false));

  // Set BrowserColorScheme to System and system to dark mode.
  ON_CALL(mock_theme_service(), GetBrowserColorScheme())
      .WillByDefault(
          testing::Return(ThemeService::BrowserColorScheme::kSystem));
  ui::NativeTheme::GetInstanceForNativeUi()->set_use_dark_colors(true);

  UpdateTheme();
  mock_client_.FlushForTesting();

  // // Theme should be dark to match the system.
  EXPECT_TRUE(theme->is_dark_mode);

  // Set BrowserColorScheme to Light and leave system still on dark mode.
  ON_CALL(mock_theme_service(), GetBrowserColorScheme())
      .WillByDefault(testing::Return(ThemeService::BrowserColorScheme::kLight));

  UpdateTheme();
  mock_client_.FlushForTesting();

  EXPECT_FALSE(theme->is_dark_mode);
}

TEST_P(ThemeColorPickerHandlerSetThemeTest, UsingDeviceThemeGM3) {
  scoped_feature_list_.InitWithFeatures(
      {features::kChromeRefresh2023, features::kChromeWebuiRefresh2023}, {});
  theme_color_picker::mojom::ThemePtr theme;
  EXPECT_CALL(mock_client_, SetTheme)
      .Times(1)
      .WillRepeatedly(
          testing::Invoke([&theme](theme_color_picker::mojom::ThemePtr arg) {
            theme = std::move(arg);
          }));

  // Set mocks needed for UpdateTheme.
  CustomBackground custom_background;
  custom_background.custom_background_url = GURL("https://foo.com/img.png");
  custom_background.custom_background_attribution_line_1 = "foo line";
  custom_background.is_uploaded_image = false;
  custom_background.custom_background_main_color = SK_ColorGREEN;
  custom_background.collection_id = "test_collection";
  custom_background.daily_refresh_enabled = false;
  ON_CALL(mock_ntp_custom_background_service_, GetCustomBackground())
      .WillByDefault(testing::Return(absl::make_optional(custom_background)));
  ON_CALL(mock_theme_service(), UsingDefaultTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingSystemTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingPolicyTheme())
      .WillByDefault(testing::Return(true));
  ON_CALL(mock_theme_service(), GetIsGrayscale())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), GetBrowserColorScheme())
      .WillByDefault(testing::Return(ThemeService::BrowserColorScheme::kLight));

  // User color should be ignored as UsingDeviceTheme() will override it.
  ON_CALL(mock_theme_service(), GetUserColor())
      .WillByDefault(testing::Return(SK_ColorGREEN));
  ON_CALL(mock_theme_service(), UsingDeviceTheme())
      .WillByDefault(testing::Return(true));
  ui::NativeTheme::GetInstanceForNativeUi()->set_user_color(SK_ColorBLUE);

  UpdateTheme();
  mock_client_.FlushForTesting();

  EXPECT_EQ(SK_ColorBLUE, theme->seed_color);
}

TEST_P(ThemeColorPickerHandlerSetThemeTest, SetThemeWithDailyRefresh) {
  theme_color_picker::mojom::ThemePtr theme;
  EXPECT_CALL(mock_client_, SetTheme)
      .Times(1)
      .WillOnce(
          testing::Invoke([&theme](theme_color_picker::mojom::ThemePtr arg) {
            theme = std::move(arg);
          }));
  CustomBackground custom_background;
  custom_background.custom_background_url = GURL("https://foo.com/img.png");
  custom_background.daily_refresh_enabled = true;
  custom_background.collection_id = "test_collection";
  ON_CALL(mock_ntp_custom_background_service_, GetCustomBackground())
      .WillByDefault(testing::Return(absl::make_optional(custom_background)));

  UpdateTheme();
  mock_client_.FlushForTesting();

  ASSERT_TRUE(theme);
  ASSERT_TRUE(theme->has_background_image);
}

TEST_P(ThemeColorPickerHandlerSetThemeTest, SetUploadedImage) {
  theme_color_picker::mojom::ThemePtr theme;
  EXPECT_CALL(mock_client_, SetTheme)
      .Times(1)
      .WillOnce(
          testing::Invoke([&theme](theme_color_picker::mojom::ThemePtr arg) {
            theme = std::move(arg);
          }));
  CustomBackground custom_background;
  custom_background.custom_background_url = GURL("https://foo.com/img.png");
  custom_background.is_uploaded_image = true;
  ON_CALL(mock_ntp_custom_background_service_, GetCustomBackground())
      .WillByDefault(testing::Return(absl::make_optional(custom_background)));
  ON_CALL(mock_theme_service(), UsingDefaultTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingSystemTheme())
      .WillByDefault(testing::Return(false));

  UpdateTheme();
  mock_client_.FlushForTesting();

  ASSERT_TRUE(theme);
  ASSERT_TRUE(theme->has_background_image);
}

TEST_P(ThemeColorPickerHandlerSetThemeTest, SetThirdPartyTheme) {
  theme_color_picker::mojom::ThemePtr theme;
  EXPECT_CALL(mock_client_, SetTheme)
      .Times(1)
      .WillOnce(
          testing::Invoke([&theme](theme_color_picker::mojom::ThemePtr arg) {
            theme = std::move(arg);
          }));
  CustomBackground custom_background;
  custom_background.custom_background_url = GURL("https://foo.com/img.png");

  auto* extension_registry = extensions::ExtensionRegistry::Get(profile_.get());
  scoped_refptr<const extensions::Extension> extension;
  extension = extensions::ExtensionBuilder()
                  .SetManifest(base::Value::Dict()
                                   .Set("name", "Foo Extension")
                                   .Set("version", "1.0.0")
                                   .Set("manifest_version", 2))
                  .SetID("foo")
                  .Build();
  extension_registry->AddEnabled(extension);

  ON_CALL(mock_ntp_custom_background_service_, GetCustomBackground())
      .WillByDefault(testing::Return(absl::make_optional(custom_background)));
  ON_CALL(mock_theme_service(), UsingDefaultTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), UsingExtensionTheme())
      .WillByDefault(testing::Return(true));
  ON_CALL(mock_theme_service(), UsingSystemTheme())
      .WillByDefault(testing::Return(false));
  ON_CALL(mock_theme_service(), GetThemeID())
      .WillByDefault(testing::Return("foo"));

  UpdateTheme();
  mock_client_.FlushForTesting();

  ASSERT_TRUE(theme);
  ASSERT_TRUE(theme->has_background_image);
}

INSTANTIATE_TEST_SUITE_P(
    All,
    ThemeColorPickerHandlerSetThemeTest,
    ::testing::Values(ThemeUpdateSource::kMojo,
                      ThemeUpdateSource::kThemeService,
                      ThemeUpdateSource::kNativeTheme,
                      ThemeUpdateSource::kCustomBackgroundService));

TEST_F(ThemeColorPickerHandlerTest, SetDefaultColor) {
  EXPECT_CALL(mock_theme_service(), UseDefaultTheme).Times(1);

  handler().SetDefaultColor();
}

TEST_F(ThemeColorPickerHandlerTest, SetDefaultColorGM3) {
  scoped_feature_list_.InitWithFeatures(
      {features::kChromeRefresh2023, features::kChromeWebuiRefresh2023}, {});
  absl::optional<SkColor> color;
  EXPECT_CALL(mock_theme_service(), SetUserColor)
      .Times(1)
      .WillOnce(testing::SaveArg<0>(&color));
  EXPECT_CALL(mock_theme_service(), UseDeviceTheme(false)).Times(1);

  handler().SetDefaultColor();

  EXPECT_EQ(SK_ColorTRANSPARENT, color.value_or(SK_ColorWHITE));
}

TEST_F(ThemeColorPickerHandlerTest, SetGreyDefaultColor) {
  bool is_grey;
  EXPECT_CALL(mock_theme_service(), SetIsGrayscale)
      .Times(1)
      .WillOnce(testing::SaveArg<0>(&is_grey));

  handler().SetGreyDefaultColor();
  EXPECT_TRUE(is_grey);
}

TEST_F(ThemeColorPickerHandlerTest, SetGreyDefaultColorGM3) {
  scoped_feature_list_.InitWithFeatures(
      {features::kChromeRefresh2023, features::kChromeWebuiRefresh2023}, {});
  bool is_grey;
  EXPECT_CALL(mock_theme_service(), SetIsGrayscale)
      .Times(1)
      .WillOnce(testing::SaveArg<0>(&is_grey));
  EXPECT_CALL(mock_theme_service(), UseDeviceTheme(false)).Times(1);

  handler().SetGreyDefaultColor();
  EXPECT_TRUE(is_grey);
}

TEST_F(ThemeColorPickerHandlerTest, SetSeedColor) {
  SkColor color = SK_ColorWHITE;
  EXPECT_CALL(mock_theme_service(), BuildAutogeneratedThemeFromColor)
      .Times(1)
      .WillOnce(testing::SaveArg<0>(&color));

  handler().SetSeedColor(SK_ColorBLUE,
                         ui::mojom::BrowserColorVariant::kTonalSpot);

  EXPECT_EQ(SK_ColorBLUE, color);
}

TEST_F(ThemeColorPickerHandlerTest, SetSeedColorGM3) {
  scoped_feature_list_.InitWithFeatures(
      {features::kChromeRefresh2023, features::kChromeWebuiRefresh2023}, {});
  absl::optional<SkColor> color;
  ui::mojom::BrowserColorVariant variant;
  EXPECT_CALL(mock_theme_service(), SetUserColorAndBrowserColorVariant)
      .Times(1)
      .WillOnce(testing::DoAll(testing::SaveArg<0>(&color),
                               testing::SaveArg<1>(&variant)));
  EXPECT_CALL(mock_theme_service(), UseDeviceTheme(false)).Times(1);

  handler().SetSeedColor(SK_ColorBLUE,
                         ui::mojom::BrowserColorVariant::kTonalSpot);

  EXPECT_EQ(SK_ColorBLUE, color.value_or(SK_ColorWHITE));
  EXPECT_EQ(ui::mojom::BrowserColorVariant::kTonalSpot, variant);
}

TEST_F(ThemeColorPickerHandlerTest, RemoveBackgroundImage) {
  EXPECT_CALL(mock_ntp_custom_background_service_, ResetCustomBackgroundInfo)
      .Times(1);

  handler().RemoveBackgroundImage();
}
