// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_EXTENSIONS_API_PASSWORDS_PRIVATE_PASSWORDS_PRIVATE_UTILS_H_
#define CHROME_BROWSER_EXTENSIONS_API_PASSWORDS_PRIVATE_PASSWORDS_PRIVATE_UTILS_H_

#include <functional>
#include <string>
#include <unordered_map>

#include "base/containers/flat_map.h"
#include "chrome/common/extensions/api/passwords_private.h"
#include "components/password_manager/core/browser/ui/credential_ui_entry.h"
#include "url/gurl.h"

class Profile;

namespace password_manager {
struct CredentialUIEntry;
}  // namespace password_manager

namespace extensions {

// Obtains a collection of URLs from the passed in |credential|. This includes
// an origin URL used for internal logic, a human friendly string shown to the
// user as well as a URL that is linked to.
api::passwords_private::UrlCollection CreateUrlCollectionFromCredential(
    const password_manager::CredentialUIEntry& credential);

// Obtains a collection of URLs from the passed in |url|. This includes an
// origin URL used for internal logic, a human friendly string shown to the user
// as well as a URL that is linked to.
api::passwords_private::UrlCollection CreateUrlCollectionFromGURL(
    const GURL& url);

// Returns PasswordStoreSet for |credential|.
extensions::api::passwords_private::PasswordStoreSet StoreSetFromCredential(
    const password_manager::CredentialUIEntry& credential);

// This class is an id generator for an arbitrary key type. It is used by both
// PasswordManagerPresenter and PasswordCheckDelegate to create ids send to the
// UI. It is similar to base::IDMap, but has the following important
// differences:
// - IdGenerator owns a copy of the key data, so that clients don't need to
//   worry about dangling pointers.
// - Repeated calls to GenerateId with the same |credential| return the same ids
// and
//   replace the |credential|.
class IdGenerator {
 public:
  IdGenerator();
  ~IdGenerator();

  // This method generates an id corresponding to |credential|. Additionally it
  // remembers ids generated in the past, so that this method is idempotent.
  // Furthermore, it is guaranteed that different ids are returned for different
  // |credential| arguments. This implies GenerateId(a) == GenerateId(b) if and
  // only if a == b.
  int GenerateId(password_manager::CredentialUIEntry credential);

  // This method tries to return the key corresponding to |id|. In case |id| was
  // not generated by GenerateId() before, this method returns nullptr.
  // Otherwise it returns a pointer p to a key, such that |id| ==
  // GenerateId(*p).
  const password_manager::CredentialUIEntry* TryGetKey(int id) const;

 private:
  // Maps credential key to id.
  std::unordered_map<std::string, int> key_to_id_;
  // Maps id to the credential.
  std::unordered_map<int, password_manager::CredentialUIEntry>
      id_to_credential_;
  int next_id_ = 0;
};

}  // namespace extensions

#endif  // CHROME_BROWSER_EXTENSIONS_API_PASSWORDS_PRIVATE_PASSWORDS_PRIVATE_UTILS_H_
