#!/usr/bin/env sh
# Copyright (C) 2020 The Qt Company Ltd.
# SPDX-License-Identifier: LicenseRef-Qt-Commercial OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only

# Install tools for singing packages
# This script assume that OS is vanilla. Target machine dosen't have any signing certificates installed.

set -ex

cache="http://ci-files01-hki.intra.qt.io/input"
cacheSigningTools="$cache/mac/sign_tools"
targetFolder="/Users/qt"
keychains="$targetFolder/Library/Keychains"

Install() {

    url=$1
    targetFile=$2
    expectedSha1=$3

    echo "Fetching $targetFile from $url..."
    curl --retry 5 --retry-delay 10 --retry-max-time 60 "$url" -o "$targetFile"
    shasum "$targetFile" |grep "$expectedSha1"

}

# qt-license
sha1QtLicense="9d59241d16f68d914f1c7aa1dc23e05faa169e8d"
Install "$cache/semisecure/.qt-license" "$targetFolder/.qt-license" $sha1QtLicense

# Login keychain
sha1LoginKeychainPassword="aae58d00d0a1b179a09f21cfc67f9d16fb95ff36"
Install "$cacheSigningTools/login_keychain_password.txt" "$targetFolder/login_keychain_password.txt" "$sha1LoginKeychainPassword"
{ loginKeychainPassword=$(cat "$targetFolder/login_keychain_password.txt"); } 2> /dev/null
loginKeychain=$keychains/login.keychain

echo "Setting login.keychain as default keychain.."
security default-keychain -s $loginKeychain*
echo "Unlocking Login keychain with password.."
{ security unlock-keychain -p "$loginKeychainPassword" $loginKeychain*; } 2> /dev/null

echo "remove the "Lock after X minutes of inactivity" from login.keychain"
security set-keychain-settings $loginKeychain

# Apple Worldwide Developer Relations Certification Authority -> https://developer.apple.com/certificationauthority/AppleWWDRCA.cer
sha1AppleWWDRCA="ff6797793a3cd798dc5b2abef56f73edc9f83a64"
Install "$cacheSigningTools/AppleWWDRCA.cer" "$targetFolder/AppleWWDRCA.cer" $sha1AppleWWDRCA
sudo security add-certificates -k $loginKeychain* "$targetFolder/AppleWWDRCA.cer"

# Developer ID Certification Authority -> https://www.apple.com/certificateauthority/DeveloperIDCA.cer
sha1DeveloperIDCA="3b166c3b7dc4b751c9fe2afab9135641e388e186"
Install "$cacheSigningTools/DeveloperIDCA.cer" "$targetFolder/DeveloperIDCA.cer" $sha1DeveloperIDCA
sudo security add-certificates -k $loginKeychain* "$targetFolder/DeveloperIDCA.cer"

# Create script to unlock keychain 'security unlock-keychain -p 'password' Developer_ID_TheQtCompany.keychain'
sha1UnLockKeychain="4398870e3f558ad28c80566b5f70e24dc29ea724"
unlockKeychain=$targetFolder/unlock-keychain.sh
{ Install "$cacheSigningTools/unlock-keychain.sh" "$unlockKeychain" $sha1UnLockKeychain; } 2> /dev/null
sudo chmod 755 "$unlockKeychain"

# Codesigning requirements file. The bundle identifier in the requirements file should match the identifier of the application that is signed.
shaCsreq="2c3f00b1845a0f475673fd6934ba25ea51d1f910"
csreq=$targetFolder/csreq_qt_company.txt
Install "$cacheSigningTools/csreq_qt_company.txt" "$csreq" $shaCsreq
chmod 755 "$csreq"

# iOS signing tools
devIDKeychain="Developer_ID_TheQtCompany.keychain-db"
shaDevIdKeychain="972cca1879cdaeeb6042f9879756c748a8d1eddc"
Install "$cacheSigningTools/$devIDKeychain" "$keychains/$devIDKeychain" $shaDevIdKeychain
echo "Opening $devIDKeychain.."
open "$keychains/$devIDKeychain"

sha1DeveloperIDTheQtCompanyKeychainPassword="d758e067736bbda7a91ffaec66cd38afdaf68ea6"
Install "$cacheSigningTools/Developer_ID_TheQtCompany_keychain_password.txt" "$targetFolder/Developer_ID_TheQtCompany_keychain_password.txt" "$sha1DeveloperIDTheQtCompanyKeychainPassword"
{ DeveloperIDTheQtCompanyKeychainPassword=$(cat "$targetFolder/Developer_ID_TheQtCompany_keychain_password.txt"); } 2> /dev/null

echo "Unlocking $devIDKeychain with password.."
{ security unlock-keychain -p "$DeveloperIDTheQtCompanyKeychainPassword" $keychains/Developer_ID_TheQtCompany.keychain; } 2> /dev/null
security set-keychain-settings $keychains/Developer_ID_TheQtCompany.keychain

sha1Ios="aae58d00d0a1b179a09f21cfc67f9d16fb95ff36"
{ Install "$cacheSigningTools/ios_password.txt" "$targetFolder/ios_password.txt" $sha1Ios; } 2> /dev/null
{ iosPassword=$(cat "$targetFolder/ios_password.txt"); } 2> /dev/null

iPhoneDeveloper="iPhoneDeveloperCiTeam2020.p12"
shaIPhoneDeveloper="73136e44fed850f398e85908e4a73f20a6bdd8a2"
Install "$cacheSigningTools/latest_ios_cert_2020/$iPhoneDeveloper" "$targetFolder/$iPhoneDeveloper" $shaIPhoneDeveloper
echo "Importing $iPhoneDeveloper.."
{ security import $targetFolder/$iPhoneDeveloper -k $loginKeychain* -P "$iosPassword" -T /usr/bin/codesign; } 2> /dev/null

iPhoneDistribution="iPhoneDistributionCiTeam2020.p12"
shaIPhoneDistribution="36215c3cbc6bfee48e86aafbd53c56282fe36d8d"
Install "$cacheSigningTools/latest_ios_cert_2020/$iPhoneDistribution" "$targetFolder/$iPhoneDistribution" $shaIPhoneDistribution
echo "Importing $iPhoneDistribution.."
{ security import "$targetFolder/$iPhoneDistribution" -k $loginKeychain* -P "$iosPassword" -T /usr/bin/codesign; } 2> /dev/null

# Mobileprovision
echo "Creating directory $targetFolder/Library/MobileDevice/Provisioning Profiles.."
mkdir "$targetFolder/Library/MobileDevice"
mkdir "$targetFolder/Library/MobileDevice/Provisioning Profiles"
shaMobileprovision="ec9697276253ca7bf4401912236f0e6fde4214da"
Install "$cacheSigningTools/latest_ios_cert_2020/iOS_Dev_2020_citeam.mobileprovision" "$targetFolder/Library/MobileDevice/Provisioning Profiles/iOS_Dev_2020_citeam.mobileprovision" $shaMobileprovision

# Removing password files
rm -fr "$targetFolder/login_keychain_password.txt"

