// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_COMMON_CHROMEOS_EXTENSIONS_CHROMEOS_SYSTEM_EXTENSION_INFO_H_
#define CHROME_COMMON_CHROMEOS_EXTENSIONS_CHROMEOS_SYSTEM_EXTENSION_INFO_H_

#include <memory>
#include <string>

#include "base/containers/flat_set.h"
#include "components/web_package/signed_web_bundles/signed_web_bundle_id.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace chromeos {

namespace switches {

extern const char kTelemetryExtensionManufacturerOverrideForTesting[];
extern const char kTelemetryExtensionPwaOriginOverrideForTesting[];
extern const char kTelemetryExtensionIwaIdOverrideForTesting[];

}  // namespace switches

// Information related to a ChromeOS system extension.
struct ChromeOSSystemExtensionInfo {
  ChromeOSSystemExtensionInfo(
      const base::flat_set<std::string>& manufacturers,
      const absl::optional<std::string>& pwa_origin,
      const absl::optional<web_package::SignedWebBundleId>& iwa_id);
  ChromeOSSystemExtensionInfo(const ChromeOSSystemExtensionInfo&);
  ChromeOSSystemExtensionInfo& operator=(const ChromeOSSystemExtensionInfo&);
  ~ChromeOSSystemExtensionInfo();

  // The extension is allowed on devices from these manufacturers.
  base::flat_set<std::string> manufacturers;
  // The connected pwa origin. |nullopt| if no connected pwa.
  absl::optional<std::string> pwa_origin;
  // The connected iwa id. |nullopt| if no connected iwa.
  absl::optional<web_package::SignedWebBundleId> iwa_id;
};

// Check if |id| is a ChromeOS system extension id.
bool IsChromeOSSystemExtension(const std::string& id);

// Get the information of a ChromeOS system extension by id.
const ChromeOSSystemExtensionInfo& GetChromeOSExtensionInfoById(
    const std::string& id);

// Check if `manufacturer` provides any chromeos system extension.
bool IsChromeOSSystemExtensionProvider(const std::string& manufacturer);

// Exported for testing.
// A helper class to restore the allowlist after tests. This should be created
// before modifying base::CommandLine to avoid changing the original allowlist.
class ScopedChromeOSSystemExtensionInfo {
 public:
  virtual ~ScopedChromeOSSystemExtensionInfo() = default;

  // Creates a instance.
  static std::unique_ptr<ScopedChromeOSSystemExtensionInfo> CreateForTesting();

  // Applies the change from the related switches in base::CommandLine.
  virtual void ApplyCommandLineSwitchesForTesting() = 0;

 protected:
  ScopedChromeOSSystemExtensionInfo() = default;
};

}  // namespace chromeos

#endif  // CHROME_COMMON_CHROMEOS_EXTENSIONS_CHROMEOS_SYSTEM_EXTENSION_INFO_H_
