/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.plugin.insights.core.service;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.cluster.metadata.IndexMetadata;
import org.opensearch.common.Nullable;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.core.action.ActionListener;
import org.opensearch.plugin.insights.core.exporter.QueryInsightsExporter;
import org.opensearch.plugin.insights.core.exporter.QueryInsightsExporterFactory;
import org.opensearch.plugin.insights.core.metrics.OperationalMetric;
import org.opensearch.plugin.insights.core.metrics.OperationalMetricsCounter;
import org.opensearch.plugin.insights.core.reader.QueryInsightsReader;
import org.opensearch.plugin.insights.core.reader.QueryInsightsReaderFactory;
import org.opensearch.plugin.insights.core.service.grouper.MinMaxHeapQueryGrouper;
import org.opensearch.plugin.insights.core.service.grouper.QueryGrouper;
import org.opensearch.plugin.insights.core.utils.ExporterReaderUtils;
import org.opensearch.plugin.insights.rules.model.AggregationType;
import org.opensearch.plugin.insights.rules.model.Attribute;
import org.opensearch.plugin.insights.rules.model.GroupingType;
import org.opensearch.plugin.insights.rules.model.MetricType;
import org.opensearch.plugin.insights.rules.model.SearchQueryRecord;
import org.opensearch.plugin.insights.rules.model.healthStats.TopQueriesHealthStats;
import org.opensearch.plugin.insights.settings.QueryInsightsSettings;
import org.opensearch.telemetry.metrics.tags.Tags;
import org.opensearch.threadpool.ThreadPool;
import org.opensearch.transport.client.Client;
import reactor.util.annotation.NonNull;

public class TopQueriesService {
    public static final String TOP_QUERIES_EXPORTER_ID = "top_queries_exporter";
    public static final String TOP_QUERIES_READER_ID = "top_queries_reader";
    public static final String TOP_QUERIES_INDEX_TAG_VALUE = "top_n_queries";
    private static final String METRIC_TYPE_TAG = "metric_type";
    private static final String GROUPBY_TAG = "groupby";
    private final Logger logger = LogManager.getLogger();
    private boolean enabled = false;
    private final Client client;
    private final MetricType metricType;
    private int topNSize;
    private TimeValue windowSize;
    private long windowStart;
    private final PriorityBlockingQueue<SearchQueryRecord> topQueriesStore;
    private final AtomicReference<List<SearchQueryRecord>> topQueriesCurrentSnapshot;
    private final AtomicReference<List<SearchQueryRecord>> topQueriesHistorySnapshot;
    private final QueryInsightsExporterFactory queryInsightsExporterFactory;
    private final QueryInsightsReaderFactory queryInsightsReaderFactory;
    private final ThreadPool threadPool;
    private final QueryGrouper queryGrouper;
    private final Predicate<SearchQueryRecord> checkIfInternal = record -> {
        Map<Attribute, Object> attributes = record.getAttributes();
        Object indicesObject = attributes.get((Object)Attribute.INDICES);
        if (indicesObject instanceof Object[]) {
            Object[] indices = (Object[])indicesObject;
            return Arrays.stream(indices).noneMatch(index -> {
                if (index instanceof String) {
                    String indexString = (String)index;
                    return indexString.contains("top_queries");
                }
                return false;
            });
        }
        return true;
    };

    TopQueriesService(Client client, MetricType metricType, ThreadPool threadPool, QueryInsightsExporterFactory queryInsightsExporterFactory, QueryInsightsReaderFactory queryInsightsReaderFactory) {
        this.client = client;
        this.metricType = metricType;
        this.threadPool = threadPool;
        this.queryInsightsExporterFactory = queryInsightsExporterFactory;
        this.queryInsightsReaderFactory = queryInsightsReaderFactory;
        this.topNSize = 10;
        this.windowSize = QueryInsightsSettings.DEFAULT_WINDOW_SIZE;
        this.windowStart = -1L;
        this.topQueriesStore = new PriorityBlockingQueue(this.topNSize, (a, b) -> SearchQueryRecord.compare(a, b, metricType));
        this.topQueriesCurrentSnapshot = new AtomicReference(new ArrayList());
        this.topQueriesHistorySnapshot = new AtomicReference(new ArrayList());
        this.queryGrouper = new MinMaxHeapQueryGrouper(metricType, QueryInsightsSettings.DEFAULT_GROUPING_TYPE, AggregationType.AVERAGE, this.topQueriesStore, this.topNSize);
    }

    public void setTopNSize(int topNSize) {
        this.topNSize = topNSize;
        this.queryGrouper.updateTopNSize(topNSize);
    }

    public int getTopNSize() {
        return this.topNSize;
    }

    public void validateTopNSize(int size) {
        if (size < 1 || size > 100) {
            throw new IllegalArgumentException("Top N size setting for [" + String.valueOf(this.metricType) + "] should be between 1 and 100, was (" + size + ")");
        }
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public void setWindowSize(TimeValue windowSize) {
        this.windowSize = windowSize;
        this.windowStart = -1L;
    }

    public void setGrouping(GroupingType groupingType) {
        boolean changed = this.queryGrouper.setGroupingType(groupingType);
        if (changed) {
            this.drain();
        }
    }

    public void setMaxGroups(int maxGroups) {
        boolean changed = this.queryGrouper.setMaxGroups(maxGroups);
        if (changed) {
            this.drain();
        }
    }

    public void validateWindowSize(TimeValue windowSize) {
        if (windowSize.compareTo(QueryInsightsSettings.MAX_WINDOW_SIZE) > 0 || windowSize.compareTo(QueryInsightsSettings.MIN_WINDOW_SIZE) < 0) {
            throw new IllegalArgumentException("Window size setting for [" + String.valueOf(this.metricType) + "] should be between [" + String.valueOf(QueryInsightsSettings.MIN_WINDOW_SIZE) + "," + String.valueOf(QueryInsightsSettings.MAX_WINDOW_SIZE) + "]was (" + String.valueOf(windowSize) + ")");
        }
        if (!QueryInsightsSettings.VALID_WINDOW_SIZES_IN_MINUTES.contains(windowSize) && windowSize.getMinutes() % 60L != 0L) {
            throw new IllegalArgumentException("Window size setting for [" + String.valueOf(this.metricType) + "] should be multiple of 1 hour, or one of " + String.valueOf(QueryInsightsSettings.VALID_WINDOW_SIZES_IN_MINUTES) + ", was (" + String.valueOf(windowSize) + ")");
        }
    }

    public List<SearchQueryRecord> getTopQueriesRecords(@NonNull boolean includeLastWindow, @Nullable String from, @Nullable String to, @Nullable String id, @Nullable Boolean verbose) {
        OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.TOP_N_QUERIES_USAGE_COUNT, Tags.create().addTag(METRIC_TYPE_TAG, this.metricType.name()).addTag(GROUPBY_TAG, this.queryGrouper.getGroupingType().name()));
        ArrayList queries = new ArrayList(this.topQueriesCurrentSnapshot.get());
        if (includeLastWindow) {
            queries.addAll(this.topQueriesHistorySnapshot.get());
        }
        List<Object> filterQueries = queries;
        if (from != null && to != null) {
            ZonedDateTime start = ZonedDateTime.parse(from);
            ZonedDateTime end = ZonedDateTime.parse(to);
            Predicate<SearchQueryRecord> timeFilter = element -> start.toInstant().toEpochMilli() <= element.getTimestamp() && element.getTimestamp() <= end.toInstant().toEpochMilli();
            filterQueries = queries.stream().filter(this.checkIfInternal.and(timeFilter)).collect(Collectors.toList());
        }
        if (id != null) {
            filterQueries = filterQueries.stream().filter(record -> record.getId().equals(id)).collect(Collectors.toList());
        }
        if (Boolean.FALSE.equals(verbose)) {
            filterQueries = filterQueries.stream().map(SearchQueryRecord::copyAndSimplifyRecord).collect(Collectors.toList());
        }
        return Stream.of(filterQueries).flatMap(Collection::stream).sorted((a, b) -> SearchQueryRecord.compare(a, b, this.metricType) * -1).collect(Collectors.toList());
    }

    public void getTopQueriesRecordsFromIndex(String from, String to, String id, Boolean verbose, final ActionListener<List<SearchQueryRecord>> listener) {
        QueryInsightsReader reader = this.queryInsightsReaderFactory.getReader(TOP_QUERIES_READER_ID);
        if (reader == null) {
            listener.onResponse(new ArrayList());
            return;
        }
        try {
            reader.read(from, to, id, verbose, this.metricType, new ActionListener<List<SearchQueryRecord>>(){
                final /* synthetic */ TopQueriesService this$0;
                {
                    this.this$0 = this$0;
                }

                public void onResponse(List<SearchQueryRecord> records) {
                    try {
                        List filteredRecords = records.stream().filter(this.this$0.checkIfInternal).sorted((a, b) -> SearchQueryRecord.compare(a, b, this.this$0.metricType) * -1).collect(Collectors.toList());
                        listener.onResponse(filteredRecords);
                    }
                    catch (Exception e) {
                        this.this$0.logger.error("Failed to process records from index: ", (Throwable)e);
                        listener.onFailure(e);
                    }
                }

                public void onFailure(Exception e) {
                    this.this$0.logger.error("Failed to read from index: ", (Throwable)e);
                    listener.onFailure(e);
                }
            });
        }
        catch (Exception e) {
            this.logger.error("Failed to initiate read from index: ", (Throwable)e);
            listener.onFailure(e);
        }
    }

    void consumeRecords(List<SearchQueryRecord> records) {
        long currentWindowStart = this.calculateWindowStart(System.currentTimeMillis());
        ArrayList<SearchQueryRecord> recordsInLastWindow = new ArrayList<SearchQueryRecord>();
        ArrayList<SearchQueryRecord> recordsInThisWindow = new ArrayList<SearchQueryRecord>();
        for (SearchQueryRecord record : records) {
            if (!record.getMeasurements().containsKey(this.metricType)) continue;
            if (record.getTimestamp() < currentWindowStart) {
                recordsInLastWindow.add(record);
                continue;
            }
            recordsInThisWindow.add(record);
        }
        this.addToTopNStore(recordsInLastWindow);
        this.rotateWindowIfNecessary(currentWindowStart);
        this.addToTopNStore(recordsInThisWindow);
        ArrayList<SearchQueryRecord> newSnapShot = new ArrayList<SearchQueryRecord>(this.topQueriesStore);
        newSnapShot.sort((a, b) -> SearchQueryRecord.compare(a, b, this.metricType));
        this.topQueriesCurrentSnapshot.set(newSnapShot);
    }

    private void addToTopNStore(List<SearchQueryRecord> records) {
        if (this.queryGrouper.getGroupingType() != GroupingType.NONE) {
            for (SearchQueryRecord record : records) {
                this.queryGrouper.add(record);
            }
        } else {
            this.topQueriesStore.addAll(records);
            while (this.topQueriesStore.size() > this.topNSize) {
                this.topQueriesStore.poll();
            }
        }
    }

    private void rotateWindowIfNecessary(long newWindowStart) {
        if (this.windowStart < newWindowStart) {
            ArrayList<SearchQueryRecord> history = new ArrayList<SearchQueryRecord>();
            if (this.windowStart == newWindowStart - this.windowSize.getMillis()) {
                history.addAll(this.topQueriesStore);
            }
            this.topQueriesHistorySnapshot.set(history);
            this.topQueriesStore.clear();
            if (this.queryGrouper.getGroupingType() != GroupingType.NONE) {
                this.queryGrouper.drain();
            }
            this.topQueriesCurrentSnapshot.set(new ArrayList());
            this.windowStart = newWindowStart;
            for (SearchQueryRecord record : history) {
                record.setTopNTrue(this.metricType);
            }
            QueryInsightsExporter exporter = this.queryInsightsExporterFactory.getExporter(TOP_QUERIES_EXPORTER_ID);
            if (exporter != null) {
                this.threadPool.executor("query_insights_executor").execute(() -> exporter.export(history));
            }
        }
    }

    private long calculateWindowStart(long timestamp) {
        LocalDateTime currentTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.of("UTC"));
        LocalDateTime windowStartTime = currentTime.truncatedTo(ChronoUnit.HOURS);
        while (!windowStartTime.plusMinutes(this.windowSize.getMinutes()).isAfter(currentTime)) {
            windowStartTime = windowStartTime.plusMinutes(this.windowSize.getMinutes());
        }
        return windowStartTime.toInstant(ZoneOffset.UTC).getEpochSecond() * 1000L;
    }

    public List<SearchQueryRecord> getTopQueriesCurrentSnapshot() {
        return this.topQueriesCurrentSnapshot.get();
    }

    public void close() throws IOException {
    }

    private void drain() {
        this.topQueriesStore.clear();
        this.topQueriesHistorySnapshot.set(new ArrayList());
        this.topQueriesCurrentSnapshot.set(new ArrayList());
    }

    public TopQueriesHealthStats getHealthStats() {
        return new TopQueriesHealthStats(this.topQueriesStore.size(), this.queryGrouper.getHealthStats());
    }

    public static boolean isTopQueriesIndex(String indexName, IndexMetadata indexMetadata) {
        try {
            LocalDate date;
            if (indexMetadata == null || indexMetadata.mapping() == null) {
                return false;
            }
            Map sourceMap = Objects.requireNonNull(indexMetadata.mapping()).getSourceAsMap();
            if (sourceMap == null || !sourceMap.containsKey("_meta")) {
                return false;
            }
            Map metaMap = (Map)sourceMap.get("_meta");
            if (metaMap == null || !metaMap.containsKey("query_insights_feature_space")) {
                return false;
            }
            if (!metaMap.get("query_insights_feature_space").equals(TOP_QUERIES_INDEX_TAG_VALUE)) {
                return false;
            }
            String[] parts = indexName.split("-");
            if (parts.length != 3) {
                return false;
            }
            if (!"top_queries".equals(parts[0])) {
                return false;
            }
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy.MM.dd", Locale.ROOT);
            try {
                date = LocalDate.parse(parts[1], formatter);
            }
            catch (DateTimeParseException e) {
                return false;
            }
            String expectedHash = ExporterReaderUtils.generateLocalIndexDateHash(date);
            return parts[2].equals(expectedHash);
        }
        catch (Exception e) {
            return false;
        }
    }
}

