"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthentication = void 0;
var _querystring = require("querystring");
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
var _http = require("../../../../../../src/core/server/http");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SamlAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "type", _common.AuthType.SAML);
    // Check if we can get the previous tenant information from the expired cookie.
    _defineProperty(this, "redirectSAMlCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
  }
  generateNextUrl(request) {
    let path = (0, _http.getRedirectUrl)({
      request,
      basePath: this.coreSetup.http.basePath.serverBasePath,
      nextUrl: request.url.pathname || '/app/opensearch-dashboards'
    });
    if (request.url.search) {
      path += request.url.search;
    }
    return (0, _querystring.escape)(path);
  }
  async init() {
    this.createExtraStorage();
    const samlAuthRoutes = new _routes.SamlAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.securityClient, this.coreSetup);
    samlAuthRoutes.setupRoutes();
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.saml.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.saml.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.saml.extra_storage.cookie_prefix,
      additionalCookies: this.config.saml.extra_storage.additional_cookies,
      logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers[SamlAuthentication.AUTH_HEADER_NAME] ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }

  // SAML expiry time is set by the IDP and returned via the security backend. Keep alive should not modify this value.
  getKeepAliveExpiry(cookie, request) {
    return cookie.expiryTime;
  }
  getCookie(request, authInfo) {
    const authorizationHeaderValue = request.headers[SamlAuthentication.AUTH_HEADER_NAME];
    (0, _cookie_splitter.setExtraAuthStorage)(request, authorizationHeaderValue, this.getExtraAuthStorageOptions(this.logger));
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: _common.AuthType.SAML,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }

  // Can be improved to check if the token is expiring.
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    return cookie.authType === _common.AuthType.SAML && cookie.username && cookie.expiryTime && (((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue) || this.getExtraAuthStorageValue(request, cookie));
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectSAMlCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions(this.logger));
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const headers = {};
    if ((_cookie$credentials3 = cookie.credentials) !== null && _cookie$credentials3 !== void 0 && _cookie$credentials3.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        headers[SamlAuthentication.AUTH_HEADER_NAME] = extraAuthStorageValue;
      } catch (error) {
        this.logger.error(error);
        // @todo Re-throw?
        // throw error;
      }
    } else {
      var _cookie$credentials4;
      headers[SamlAuthentication.AUTH_HEADER_NAME] = (_cookie$credentials4 = cookie.credentials) === null || _cookie$credentials4 === void 0 ? void 0 : _cookie$credentials4.authHeaderValue;
    }
    return headers;
  }
}
exports.SamlAuthentication = SamlAuthentication;
_defineProperty(SamlAuthentication, "AUTH_HEADER_NAME", 'authorization');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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