"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;
var fs = _interopRequireWildcard(require("fs"));
var _wreck = _interopRequireDefault(require("@hapi/wreck"));
var _proxyAgent = require("proxy-agent");
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _helper = require("./helper");
var _object_properties_defined = require("../../../utils/object_properties_defined");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
var _http = require("../../../../../../src/core/server/http");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;
    super(config, sessionStorageFactory, router, esClient, core, logger);
    _defineProperty(this, "type", _common.AuthType.OPEN_ID);
    _defineProperty(this, "openIdAuthConfig", void 0);
    _defineProperty(this, "authHeaderName", void 0);
    _defineProperty(this, "openIdConnectUrl", void 0);
    _defineProperty(this, "wreckClient", void 0);
    _defineProperty(this, "wreckHttpsOption", {});
    _defineProperty(this, "redirectOIDCCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/openid/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;
    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }
    this.openIdAuthConfig.scope = scope;
  }
  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      this.createExtraStorage();
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info
      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }
  generateNextUrl(request) {
    let path = (0, _http.getRedirectUrl)({
      request,
      basePath: this.coreSetup.http.basePath.serverBasePath,
      nextUrl: request.url.pathname || '/app/opensearch-dashboards'
    });
    if (request.url.search) {
      path += request.url.search;
    }
    return escape(path);
  }
  createWreckClient() {
    var _this$config$openid3, _this$config$openid4, _this$config$openid5, _this$config$openid6, _this$config$openid7, _this$config$openid9;
    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      this.wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
      this.logger.debug(`Using CA Cert: ${this.config.openid.root_ca}`);
    }
    if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.pfx) {
      // Use PFX or PKCS12 if provided
      this.logger.debug(`Using PFX or PKCS12: ${this.config.openid.pfx}`);
      this.wreckHttpsOption.pfx = [fs.readFileSync(this.config.openid.pfx)];
    } else if ((_this$config$openid5 = this.config.openid) !== null && _this$config$openid5 !== void 0 && _this$config$openid5.certificate && (_this$config$openid6 = this.config.openid) !== null && _this$config$openid6 !== void 0 && _this$config$openid6.private_key) {
      // Use 'certificate' and 'private_key' if provided
      this.logger.debug(`Using Certificate: ${this.config.openid.certificate}`);
      this.logger.debug(`Using Private Key: ${this.config.openid.private_key}`);
      this.wreckHttpsOption.cert = [fs.readFileSync(this.config.openid.certificate)];
      this.wreckHttpsOption.key = [fs.readFileSync(this.config.openid.private_key)];
    } else {
      this.logger.debug(`Client certificates not provided. Mutual TLS will not be used to obtain endpoints.`);
    }
    // Check if passphrase is provided, use it for 'pfx' and 'key'
    if (((_this$config$openid7 = this.config.openid) === null || _this$config$openid7 === void 0 ? void 0 : _this$config$openid7.passphrase) !== '') {
      var _this$config$openid8;
      this.logger.debug(`Passphrase not provided for private key and/or pfx.`);
      this.wreckHttpsOption.passphrase = (_this$config$openid8 = this.config.openid) === null || _this$config$openid8 === void 0 ? void 0 : _this$config$openid8.passphrase;
    }
    if (((_this$config$openid9 = this.config.openid) === null || _this$config$openid9 === void 0 ? void 0 : _this$config$openid9.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);
      this.wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }
    this.logger.info((0, _object_properties_defined.getObjectProperties)(this.wreckHttpsOption, 'WreckHttpsOptions'));

    // Use proxy agent to allow usage of e.g. http_proxy environment variable
    const httpAgent = new _proxyAgent.ProxyAgent();
    const httpsAllowUnauthorizedAgent = new _proxyAgent.ProxyAgent({
      rejectUnauthorized: false
    });
    let httpsAgent = new _proxyAgent.ProxyAgent();
    if (Object.keys(this.wreckHttpsOption).length > 0) {
      httpsAgent = new _proxyAgent.ProxyAgent(this.wreckHttpsOption);
    }
    return _wreck.default.defaults({
      agents: {
        http: httpAgent,
        https: httpsAgent,
        httpsAllowUnauthorized: httpsAllowUnauthorizedAgent
      }
    });
  }
  getWreckHttpsOptions() {
    return this.wreckHttpsOption;
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.openid.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.openid.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger: this.logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, request.headers.authorization, this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }
  getKeepAliveExpiry(cookie, request) {
    return Date.now() + this.config.session.ttl;
  }

  // TODO: Add token expiration check here
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) && !this.getExtraAuthStorageValue(request, cookie)) {
      return false;
    }
    if (cookie.credentials.expiryTime > Date.now()) {
      return true;
    }

    // need to renew id token
    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid10, _this$config$openid11;
        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid10 = this.config.openid) === null || _this$config$openid10 === void 0 ? void 0 : _this$config$openid10.client_id,
          client_secret: (_this$config$openid11 = this.config.openid) === null || _this$config$openid11 === void 0 ? void 0 : _this$config$openid11.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);

        // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token
        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValueExtra: true,
            refresh_token: refreshTokenResponse.refreshToken,
            expiryTime: (0, _helper.getExpirationDate)(refreshTokenResponse)
          };
          (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${refreshTokenResponse.idToken}`, this.getExtraAuthStorageOptions());
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectOIDCCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
        // TODO Re-throw?
        // throw error;
      }
    }

    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }
    return header;
  }
}
exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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