"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerChatRoutes = registerChatRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _olly_chat_service = require("../services/chat/olly_chat_service");
var _agent_framework_storage_service = require("../services/storage/agent_framework_storage_service");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const llmRequestRoute = {
  path: _llm.ASSISTANT_API.SEND_MESSAGE,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.maybe(_configSchema.schema.string()),
      messages: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
      input: _configSchema.schema.object({
        type: _configSchema.schema.literal('input'),
        context: _configSchema.schema.object({
          appId: _configSchema.schema.maybe(_configSchema.schema.string()),
          content: _configSchema.schema.maybe(_configSchema.schema.string()),
          datasourceId: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        content: _configSchema.schema.string(),
        contentType: _configSchema.schema.literal('text'),
        promptPrefix: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const abortAgentExecutionRoute = {
  path: `${_llm.ASSISTANT_API.ABORT_AGENT_EXECUTION}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const regenerateRoute = {
  path: `${_llm.ASSISTANT_API.REGENERATE}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string(),
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationsRoute = {
  path: _llm.ASSISTANT_API.CONVERSATIONS,
  validate: {
    query: _configSchema.schema.object({
      perPage: _configSchema.schema.number({
        min: 0,
        defaultValue: 20
      }),
      page: _configSchema.schema.number({
        min: 0,
        defaultValue: 1
      }),
      sortOrder: _configSchema.schema.maybe(_configSchema.schema.string()),
      sortField: _configSchema.schema.maybe(_configSchema.schema.string()),
      fields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
      search: _configSchema.schema.maybe(_configSchema.schema.string()),
      searchFields: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])),
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const deleteConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const updateConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      title: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getTracesRoute = {
  path: `${_llm.ASSISTANT_API.TRACE}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const feedbackRoute = {
  path: `${_llm.ASSISTANT_API.FEEDBACK}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      satisfaction: _configSchema.schema.boolean()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const accountRoute = {
  path: `${_llm.ASSISTANT_API.ACCOUNT}`,
  validate: {}
};
function registerChatRoutes(router, routeOptions) {
  const createStorageService = async (context, dataSourceId) => new _agent_framework_storage_service.AgentFrameworkStorageService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }), routeOptions.messageParsers);
  const createChatService = async (context, dataSourceId) => new _olly_chat_service.OllyChatService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }));
  router.post(llmRequestRoute, async (context, request, response) => {
    var _outputs, _outputs2;
    const {
      messages = [],
      input,
      conversationId: conversationIdInRequestBody
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.requestLLM({
        messages,
        input,
        conversationId: conversationIdInRequestBody
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }

    /**
     * Retrieve latest interactions from memory
     */
    const conversationId = ((_outputs = outputs) === null || _outputs === void 0 ? void 0 : _outputs.conversationId) || conversationIdInRequestBody;
    const interactionId = ((_outputs2 = outputs) === null || _outputs2 === void 0 ? void 0 : _outputs2.interactionId) || '';
    try {
      if (!conversationId) {
        throw new Error('Not a valid conversation');
      }
      const resultPayload = {
        messages: [],
        interactions: [],
        conversationId
      };
      if (!conversationIdInRequestBody) {
        /**
         * If no conversationId is provided in request payload,
         * it means it is a brand new conversation,
         * need to fetch all the details including title.
         */
        const conversation = await storageService.getConversation(conversationId);
        resultPayload.interactions = conversation.interactions;
        resultPayload.messages = conversation.messages;
        resultPayload.title = conversation.title;
      } else {
        /**
         * Only response with the latest interaction.
         * It may have some issues in Concurrent case like a user may use two tabs to chat with Chatbot in one conversation.
         * But for now we will ignore this case, can be optimized by always fetching conversation if we need to take this case into consideration.
         */
        const interaction = await storageService.getInteraction(conversationId, interactionId);
        resultPayload.interactions = [interaction].filter(item => item);
        resultPayload.messages = resultPayload.interactions.length ? await storageService.getMessagesFromInteractions(resultPayload.interactions) : [];
      }
      resultPayload.messages.filter(message => message.type === 'input').forEach(msg => {
        // hide additional conetxt to how was it generated
        const index = msg.content.indexOf('answer question:');
        const len = 'answer question:'.length;
        if (index !== -1) {
          msg.content = msg.content.substring(index + len);
        }
      });
      return response.ok({
        body: resultPayload
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationsRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversations(request.query);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete(deleteConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.deleteConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(updateConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.updateConversation(request.params.conversationId, request.body.title);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getTracesRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getTraces(request.params.interactionId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post(abortAgentExecutionRoute, async (context, request, response) => {
    const chatService = await createChatService(context, request.query.dataSourceId);
    try {
      chatService.abortAgentExecution(request.body.conversationId);
      context.assistant_plugin.logger.info(`Abort agent execution: ${request.body.conversationId}`);
      return response.ok();
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(regenerateRoute, async (context, request, response) => {
    const {
      conversationId,
      interactionId
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.regenerate({
        conversationId,
        interactionId
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }

    /**
     * Retrieve latest interactions from memory
     */
    try {
      var _outputs3;
      const interaction = await storageService.getInteraction(conversationId, ((_outputs3 = outputs) === null || _outputs3 === void 0 ? void 0 : _outputs3.interactionId) || '');
      const finalInteractions = [interaction].filter(item => item);
      const messages = finalInteractions.length ? await storageService.getMessagesFromInteractions(finalInteractions) : [];
      return response.ok({
        body: {
          interactions: finalInteractions,
          messages,
          conversationId
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(feedbackRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const {
      interactionId
    } = request.params;
    try {
      const updateResponse = await storageService.updateInteraction(interactionId, {
        feedback: request.body
      });
      return response.ok({
        body: {
          ...updateResponse,
          success: true
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(accountRoute, async (context, request, response) => {
    try {
      var _auth$state$authInfo$, _auth$state;
      const auth = routeOptions.auth.get(request);
      return response.ok({
        body: {
          user_name: (_auth$state$authInfo$ = auth === null || auth === void 0 || (_auth$state = auth.state) === null || _auth$state === void 0 || (_auth$state = _auth$state.authInfo) === null || _auth$state === void 0 ? void 0 : _auth$state.user_name) !== null && _auth$state$authInfo$ !== void 0 ? _auth$state$authInfo$ : _llm.DEFAULT_USER_NAME
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.ok({
        body: {
          user_name: _llm.DEFAULT_USER_NAME
        }
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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