/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.cache.lru;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.OpenHAB;
import org.openhab.core.cache.lru.LRUMediaCacheEntry;
import org.openhab.core.storage.Storage;
import org.openhab.core.storage.StorageService;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
public class LRUMediaCache<V> {
    private final Logger logger = LoggerFactory.getLogger(LRUMediaCache.class);
    private static final String CACHE_FOLDER_NAME = "cache";
    final Map<String, @Nullable LRUMediaCacheEntry<V>> cachedResults;
    private final Map<String, Lock> lockByEntry = new ConcurrentHashMap<String, Lock>();
    protected final long maxCacheSize;
    private final Path cacheFolder;
    private Storage<V> storage;
    protected boolean cacheIsOK = true;

    public LRUMediaCache(@Reference StorageService storageService, long maxCacheSize, String pid, @Nullable ClassLoader clazzLoader) {
        this.storage = storageService.getStorage(pid, clazzLoader);
        this.cachedResults = Collections.synchronizedMap(new LinkedHashMap(20, 0.75f, true));
        this.cacheFolder = Path.of(OpenHAB.getUserDataFolder(), CACHE_FOLDER_NAME, pid);
        this.maxCacheSize = maxCacheSize;
        this.logger.debug("Creating cache folder '{}'", (Object)this.cacheFolder);
        try {
            Files.createDirectories(this.cacheFolder, new FileAttribute[0]);
            this.cleanCacheDirectory();
            this.loadAll();
        }
        catch (IOException e) {
            this.cacheIsOK = false;
            this.logger.warn("Cannot use cache directory", (Throwable)e);
        }
        if (this.getFreeSpace() < maxCacheSize * 2L) {
            this.cacheIsOK = false;
            this.logger.warn("Not enough space for the cache");
        }
        this.logger.debug("Using cache folder '{}'", (Object)this.cacheFolder);
    }

    private void cleanCacheDirectory() throws IOException {
        try {
            Throwable throwable = null;
            Object var2_4 = null;
            try (Stream<Path> files = Files.list(this.cacheFolder);){
                String fileName;
                ArrayList<Path> filesInCacheFolder = new ArrayList<Path>(files.toList());
                Iterator fileDeleterIterator = filesInCacheFolder.iterator();
                while (fileDeleterIterator.hasNext()) {
                    Path path = (Path)fileDeleterIterator.next();
                    File file = path.toFile();
                    if (file.length() != 0L) continue;
                    file.delete();
                    fileName = path.getFileName().toString();
                    this.storage.remove(fileName);
                    fileDeleterIterator.remove();
                }
                for (Path path : filesInCacheFolder) {
                    fileName = path.getFileName().toString();
                    V metadata = this.storage.get(fileName);
                    if (metadata != null) continue;
                    Files.delete(path);
                }
                for (Map.Entry entry : this.storage.stream().toList()) {
                    Path correspondingFile = this.cacheFolder.resolve((String)entry.getKey());
                    if (Files.exists(correspondingFile, new LinkOption[0])) continue;
                    this.storage.remove((String)entry.getKey());
                }
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            this.logger.warn("Cannot load the cache directory : {}", (Object)e.getMessage());
            this.cacheIsOK = false;
            return;
        }
    }

    private long getFreeSpace() {
        try {
            Path rootPath = Paths.get(new URI("file:///"));
            Path dirPath = rootPath.resolve(this.cacheFolder.getParent());
            FileStore dirFileStore = Files.getFileStore(dirPath);
            return dirFileStore.getUsableSpace();
        }
        catch (IOException | URISyntaxException e) {
            this.logger.error("Cannot compute free disk space for the cache. Reason: {}", (Object)e.getMessage());
            return 0L;
        }
    }

    public LRUMediaCacheEntry<V> get(String key, Supplier<LRUMediaCacheEntry<V>> supplier) {
        if (!this.cacheIsOK) {
            return supplier.get();
        }
        Lock lockForCurrentEntry = this.lockByEntry.computeIfAbsent(key, k -> new ReentrantLock());
        if (lockForCurrentEntry == null) {
            this.cacheIsOK = false;
            this.logger.error("Cannot compute lock within cache system. Shouldn't happen");
            return supplier.get();
        }
        lockForCurrentEntry.lock();
        try {
            LRUMediaCacheEntry<V> result = this.cachedResults.get(key);
            if (result != null && result.isFaulty()) {
                result.deleteFile();
                this.cachedResults.remove(key);
                result = null;
            }
            if (result == null) {
                this.logger.debug("Cache miss {}", (Object)key);
                result = supplier.get();
                this.put(result);
            }
            LRUMediaCacheEntry<V> lRUMediaCacheEntry = result;
            return lRUMediaCacheEntry;
        }
        finally {
            lockForCurrentEntry.unlock();
        }
    }

    protected void put(LRUMediaCacheEntry<V> result) {
        result.setCacheContext(this.cacheFolder, this.storage);
        this.cachedResults.put(result.getKey(), result);
        this.makeSpace();
    }

    private void loadAll() throws IOException {
        this.cachedResults.clear();
        this.storage.stream().map(entry -> new LRUMediaCacheEntry((String)entry.getKey())).forEach(this::put);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void makeSpace() {
        Map<String, LRUMediaCacheEntry<V>> map = this.cachedResults;
        synchronized (map) {
            Iterator<@Nullable LRUMediaCacheEntry<V>> iterator = this.cachedResults.values().iterator();
            Long currentCacheSize = this.cachedResults.values().stream().map(result -> result == null ? 0L : result.getCurrentSize()).reduce(0L, Long::sum);
            int attemptToDelete = 0;
            while (currentCacheSize > this.maxCacheSize && this.cachedResults.size() > 1 && attemptToDelete < 10) {
                ++attemptToDelete;
                LRUMediaCacheEntry<V> oldestEntry = iterator.next();
                if (oldestEntry != null) {
                    oldestEntry.deleteFile();
                    currentCacheSize = currentCacheSize - oldestEntry.getCurrentSize();
                    this.lockByEntry.remove(oldestEntry.getKey());
                }
                iterator.remove();
            }
        }
    }
}

