<?php

# This file is a part of RackTables, a datacenter and server room management
# framework. See accompanying file "COPYING" for the full copyright and
# licensing information.

function reloadDictionary ($rows_per_query = 25)
{
	global $dictionary;

	function buildInsert ($vlist)
	{
		$ret= 'INSERT INTO Dictionary (dict_key, chapter_id, dict_value, dict_sticky) VALUES ';
		$ret .= implode (', ', $vlist);
		return $ret;
	}

	// isNaturalNumber() depends on functions.php, which is not involved in an upgrade (see index.php).
	$rows_per_query = (int) $rows_per_query;
	if ($rows_per_query < 1)
		throw new InvalidArgException ('rows_per_query', $rows_per_query, 'must be a natural number');
	// Not only update existing stuff, but make sure all obsolete records are gone.
	$ret = array ("DELETE FROM Dictionary WHERE dict_key BETWEEN 1 AND 49999");
	$vlist = array();
	# Iterating through 50K possible valid indices is way too slow in PHP and
	# is likely to hit the default execution time limit of 30 seconds.
	foreach ($dictionary as $dict_key => $record)
	{
		$vlist[] = "(${dict_key}, ${record['chapter_id']}, '${record['dict_value']}', 'yes')";
		if (count ($vlist) == $rows_per_query)
		{
			$ret[] = buildInsert ($vlist);
			$vlist = array();
		}
	}
	if (count ($vlist))
		$ret[] = buildInsert ($vlist);
	return $ret;
}

function isInnoDBSupported ()
{
	global $dbxlink;
	// create a temp table and a trigger
	$dbxlink->query("DROP TABLE IF EXISTS `innodb_test`");
	$dbxlink->query("CREATE TABLE `innodb_test` (`id` int) ENGINE=InnoDB");
	$innodb_row = $dbxlink->query("SHOW TABLE STATUS LIKE 'innodb_test'")->fetch(PDO::FETCH_ASSOC);
	$dbxlink->query("CREATE TRIGGER `trigger_test` BEFORE INSERT ON `innodb_test` FOR EACH ROW BEGIN END");
	$trigger_row = $dbxlink->query("SELECT COUNT(*) AS count FROM information_schema.TRIGGERS WHERE TRIGGER_SCHEMA = SCHEMA() AND TRIGGER_NAME = 'trigger_test'")->fetch(PDO::FETCH_ASSOC);
	$dbxlink->query("DROP TABLE `innodb_test`");
	return $innodb_row['Engine'] == 'InnoDB' && $trigger_row['count'] == 1;
}

function platform_generic_test ($is_ok, $topic, $what_if_not = 'FAILED', $error_class = 'trerror')
{
	echo "<tr><th class=tdleft>${topic}</th>";
	if ($is_ok)
	{
		echo '<td class="trok tdleft">PASSED</td></tr>';
		return 0;
	}
	echo "<td class='${error_class} tdleft'>${what_if_not}</td></tr>";
	return 1;
}

function pcre8_with_properties()
{
	return FALSE === @preg_match ('/\p{L}/u', 'a') ? FALSE : TRUE;
}

// Check for PHP extensions.
function platform_is_ok ($test_innodb = FALSE)
{
	$extlist = array
	(
		// mandatory
		array
		(
			'name' => 'PDO',
			'comment' => 'PHP data objects',
		),
		array
		(
			'name' => 'pdo_mysql',
			'comment' => 'PDO MySQL driver',
		),
		array
		(
			'name' => 'pcre',
			'comment' => 'PCRE',
		),
		array
		(
			'name' => 'gd',
			'comment' => 'GD and image',
		),
		array
		(
			'name' => 'mbstring',
			'comment' => 'multibyte string',
		),
		array
		(
			'name' => 'json',
			'comment' => 'JSON',
		),
		array
		(
			'name' => 'bcmath',
			'comment' => 'arbitrary precision mathematics',
		),
		// optional
		array
		(
			'name' => 'snmp',
			'comment' => 'SNMP',
			'impact' => 'SNMP sync feature will not work',
		),
		array
		(
			'name' => 'ldap',
			'comment' => 'LDAP',
			'impact' => 'LDAP authentication will not work',
		),
		array
		(
			'name' => 'curl',
			'comment' => 'client URL library',
			'impact' => 'some plugins may not work',
		),
		array
		(
			'name' => 'pcntl',
			'comment' => 'process control',
			'impact' => '802.1Q parallel sync will not work',
		),
	);
	$nerrs = 0;
	echo "<table border=1 cellpadding=5>\n";
	$nerrs += platform_generic_test (version_compare (PHP_VERSION, '7.0.0', '>='), 'PHP version >= 7.0');
	foreach ($extlist as $e)
	{
		if (array_key_exists ('impact', $e))
		{
			// When an optional PHP extension is not available, display a warning and a message
			// with some additional information so that the user can decide if it is OK to proceed
			// without the feature(s) that depend on the extension.
			$what_if_not = "Not found ({$e['impact']}).";
			$error_class = 'trwarning';
			$c = 0;
		}
		else
		{
			// If a mandatory PHP extension is not available, just report the failure.
			$what_if_not = 'Not found.';
			$error_class = 'trerror';
			$c = 1;
		}
		$nerrs += $c * platform_generic_test
		(
			extension_loaded ($e['name']),
			"{$e['comment']} extension ({$e['name']})",
			$what_if_not,
			$error_class
		);
	}
	if ($test_innodb)
		$nerrs += platform_generic_test (isInnoDBSupported(), 'InnoDB support');
	$nerrs += platform_generic_test (pcre8_with_properties(), 'PCRE compiled with<br>--enable-unicode-properties');
	platform_generic_test
	(
		(! empty ($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off'),
		'accessed over HTTPS',
		'No! (all your passwords will be transmitted in cleartext)',
		'trwarning'
	);
	echo "</table>\n";
	return !$nerrs;
}

// FIXME: Use this in the installer instead of the hardcoded SQL.
function getConfigDefaults()
{
	return array
	(
		'MASSCOUNT' => '8',
		'MAXSELSIZE' => '30',
		'ROW_SCALE' => '2',
		'IPV4_ADDRS_PER_PAGE' => '256',
		'DEFAULT_RACK_HEIGHT' => '42',
		'DEFAULT_SLB_VS_PORT' => '',
		'DEFAULT_SLB_RS_PORT' => '',
		'DETECT_URLS' => 'no',
		'RACK_PRESELECT_THRESHOLD' => '1',
		'DEFAULT_IPV4_RS_INSERVICE' => 'no',
		'AUTOPORTS_CONFIG' => '4 = 1*33*kvm + 2*24*eth%u;15 = 1*446*kvm',
		'SHOW_EXPLICIT_TAGS' => 'yes',
		'SHOW_IMPLICIT_TAGS' => 'yes',
		'SHOW_AUTOMATIC_TAGS' => 'no',
		'DEFAULT_OBJECT_TYPE' => '4',
		'IPV4_AUTO_RELEASE' => '1',
		'SHOW_LAST_TAB' => 'yes',
		'EXT_IPV4_VIEW' => 'yes',
		'TREE_THRESHOLD' => '25',
		'IPV4_JAYWALK' => 'no',
		'ADDNEW_AT_TOP' => 'yes',
		'IPV4_TREE_SHOW_USAGE' => 'no',
		'PREVIEW_TEXT_MAXCHARS' => '10240',
		'PREVIEW_TEXT_ROWS' => '25',
		'PREVIEW_TEXT_COLS' => '80',
		'PREVIEW_IMAGE_MAXPXS' => '320',
		'VENDOR_SIEVE' => '',
		'IPV4LB_LISTSRC' => 'false',
		'IPV4OBJ_LISTSRC' => 'not ({$typeid_3} or {$typeid_9} or {$typeid_10} or {$typeid_11})',
		'IPV4NAT_LISTSRC' => '{$typeid_4} or {$typeid_7} or {$typeid_8} or {$typeid_798}',
		'ASSETWARN_LISTSRC' => '{$typeid_4} or {$typeid_7} or {$typeid_8}',
		'NAMEWARN_LISTSRC' => '{$typeid_4} or {$typeid_7} or {$typeid_8}',
		'RACKS_PER_ROW' => '12',
		'FILTER_PREDICATE_SIEVE' => '',
		'FILTER_DEFAULT_ANDOR' => 'and',
		'FILTER_SUGGEST_ANDOR' => 'yes',
		'FILTER_SUGGEST_TAGS' => 'yes',
		'FILTER_SUGGEST_PREDICATES' => 'yes',
		'FILTER_SUGGEST_EXTRA' => 'no',
		'DEFAULT_SNMP_COMMUNITY' => 'public',
		'IPV4_ENABLE_KNIGHT' => 'yes',
		'TAGS_TOPLIST_SIZE' => '50',
		'TAGS_QUICKLIST_SIZE' => '20',
		'TAGS_QUICKLIST_THRESHOLD' => '50',
		'ENABLE_MULTIPORT_FORM' => 'no',
		'DEFAULT_PORT_IIF_ID' => '1',
		'DEFAULT_PORT_OIF_IDS' => '1=24; 3=1078; 4=1077; 5=1079; 6=1080; 8=1082; 9=1084; 10=1588; 11=1668; 12=1589; 13=1590; 14=1591',
		'IPV4_TREE_RTR_AS_CELL' => 'no',
		'PROXIMITY_RANGE' => '0',
		'IPV4_TREE_SHOW_VLAN' => 'yes',
		'VLANSWITCH_LISTSRC' => '',
		'VLANNET_LISTSRC' => '',
		'DEFAULT_VDOM_ID' => '',
		'DEFAULT_VST_ID' => '',
		'STATIC_FILTER' => 'yes',
		'8021Q_DEPLOY_MINAGE' => '300',
		'8021Q_DEPLOY_MAXAGE' => '3600',
		'8021Q_DEPLOY_RETRY' => '10800',
		'8021Q_WRI_AFTER_CONFT_LISTSRC' => 'false',
		'8021Q_INSTANT_DEPLOY' => 'no',
		'ENABLE_BULKPORT_FORM' => 'yes',
		'CDP_RUNNERS_LISTSRC' => '',
		'LLDP_RUNNERS_LISTSRC' => '',
		'SHRINK_TAG_TREE_ON_CLICK' => 'yes',
		'MAX_UNFILTERED_ENTITIES' => '0',
		'SYNCDOMAIN_MAX_PROCESSES' => '0',
		'PORT_EXCLUSION_LISTSRC' => '{$typeid_3} or {$typeid_10} or {$typeid_11} or {$typeid_1505} or {$typeid_1506}',
		'FILTER_RACKLIST_BY_TAGS' => 'yes',
		'MGMT_PROTOS' => 'ssh: {$typeid_4}; telnet: {$typeid_8}',
		'SYNC_8021Q_LISTSRC' => '',
		'QUICK_LINK_PAGES' => 'depot,ipv4space,rackspace',
		'VIRTUAL_OBJ_CSV' => '1504,1505,1506,1507',
		'DATETIME_ZONE' => 'UTC',
		'DATETIME_FORMAT' => '%Y-%m-%d',
		'DATEONLY_FORMAT' => '%Y-%m-%d',
		'SEARCH_DOMAINS' => '',
		'8021Q_EXTSYNC_LISTSRC' => 'false',
		'8021Q_MULTILINK_LISTSRC' => 'false',
		'REVERSED_RACKS_LISTSRC' => 'false',
		'NEAREST_RACKS_CHECKBOX' => 'yes',
		'SHOW_OBJECTTYPE' => 'yes',
		'IPV4_TREE_SHOW_UNALLOCATED' => 'yes',
		'OBJECTLOG_PREVIEW_ENTRIES' => '5',
	);
}

$dictionary = array
(
	1 => array ('chapter_id' => 1, 'dict_value' => 'BlackBox'),
	2 => array ('chapter_id' => 1, 'dict_value' => 'PDU'),
	3 => array ('chapter_id' => 1, 'dict_value' => 'Shelf'),
	4 => array ('chapter_id' => 1, 'dict_value' => 'Server'),
	5 => array ('chapter_id' => 1, 'dict_value' => 'DiskArray'),
	6 => array ('chapter_id' => 1, 'dict_value' => 'TapeLibrary'),
	7 => array ('chapter_id' => 1, 'dict_value' => 'Router'),
	8 => array ('chapter_id' => 1, 'dict_value' => 'Network switch'),
	9 => array ('chapter_id' => 1, 'dict_value' => 'PatchPanel'),
	10 => array ('chapter_id' => 1, 'dict_value' => 'CableOrganizer'),
	11 => array ('chapter_id' => 1, 'dict_value' => 'spacer'),
	12 => array ('chapter_id' => 1, 'dict_value' => 'UPS'),
	13 => array ('chapter_id' => 1, 'dict_value' => 'Modem'),
	14 => array ('chapter_id' => 1, 'dict_value' => 'MediaConverter'),
	15 => array ('chapter_id' => 1, 'dict_value' => 'console'),
	42 => array ('chapter_id' => 11, 'dict_value' => 'noname/unknown'),
	43 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%305'),
	44 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%306'),
	45 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%306m'),
	46 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%326m'),
	47 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%330'),
	48 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%335'),
	49 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Ultra 10'),
	50 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Enterprise 420R'),
	51 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire X2100'),
	52 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire E4900'),
	53 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Netra X1'),
	54 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%346'),
	55 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%1650'),
	56 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%2850'),
	57 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V210'),
	58 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V240'),
	59 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%326'),
	60 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Netra t1 105'),
	61 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Enterprise 4500'),
	62 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%1950'),
	63 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%1550'),
	64 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Ultra 5'),
	65 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%2950'),
	66 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%650'),
	67 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%4600'),
	68 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3250'),
	69 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3455'),
	70 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3550'),
	71 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3650'),
	72 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3655'),
	73 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3650 T'),
	74 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3755'),
	75 => array ('chapter_id' => 11, 'dict_value' => 'IBM xSeries%GPASS%3850'),
	76 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire X4600'),
	77 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire X4500'),
	78 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire X4200'),
	79 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire X4100'),
	80 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire X2100 M2'),
	81 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire X2200 M2'),
	82 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V40z'),
	83 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V125'),
	84 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V215'),
	85 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V245'),
	86 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V445'),
	87 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V440'),
	88 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V490'),
	89 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V890'),
	90 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire E2900'),
	91 => array ('chapter_id' => 11, 'dict_value' => 'Sun%GPASS%Fire V1280'),
	92 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%185'),
	93 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%505'),
	94 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%505Q'),
	95 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%510'),
	96 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%510Q'),
	97 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%520'),
	98 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%520Q'),
	99 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%550'),
	100 => array ('chapter_id' => 11, 'dict_value' => 'IBM pSeries%GPASS%550Q'),
	101 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL140'),
	102 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL145'),
	103 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL320'),
	104 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL360'),
	105 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL380'),
	106 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL385'),
	107 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL580'),
	108 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL585'),
	109 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%ML110'),
	110 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%ML150'),
	111 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%ML310'),
	112 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%ML350'),
	113 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%ML370'),
	114 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%ML570'),
	115 => array ('chapter_id' => 12, 'dict_value' => 'noname/unknown'),
	116 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron WorkGroup'),
	117 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron II'),
	118 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron'),
	119 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron XL'),
	120 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron 350'),
	121 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 2402'),
	122 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 4802'),
	123 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge X424'),
	124 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2924XL'),
	125 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron SuperX'),
	126 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2912XL'),
	127 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron GS 648P'),
	128 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 2402-PREM'),
	129 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron GS 624P'),
	130 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron GS 624P-POE'),
	131 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron GS 648P-POE'),
	132 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron 4G'),
	133 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron 4G-SSL'),
	134 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge X448'),
	135 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge X424HF'),
	136 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge X424-POE'),
	137 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron SX 800'),
	138 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron SX 1600'),
	139 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst WS-C3560-8PC'),
	140 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-48TC-S'),
	141 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560-E'),
	142 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst Express 500-24LC'),
	143 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750-24TS'),
	144 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750-E'),
	145 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 4503'),
	146 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 6513'),
	147 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Catalyst 4948 | http://www.cisco.com/en/US/products/ps6026/index.html]]'),
	148 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6509-E%L9,1H%'),
	149 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6509-NEB-A%L9,1H%'),
	150 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6506-E%L6,1H%'),
	151 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6504-E%L4,1H%'),
	152 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6503-E%L3,1H%'),
	153 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6503%L3,1H%'),
	154 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6506%L6,1H%'),
	155 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6509-NEB%L9,1H%'),
	156 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 4506%L6,1H%'),
	157 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 4507R%L7,1H%'),
	158 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 4510R%L10,1H%'),
	159 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst Express 500-24PC'),
	160 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst Express 500-24TT'),
	161 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst Express 500G-12TC'),
// 1572 duplicated 162 later, so 162 isn't here any more
// 1710 duplicated 163 later, so 163 isn't here any more
	164 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24TT-L'),
	165 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-8TC-L'),
	166 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960G-48TC-L'),
	167 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960G-24TC-L'),
	168 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960G-8TC-L'),
	169 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560-24TS'),
	170 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560-48TS'),
	171 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560-24PS'),
	172 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560-48PS'),
	173 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560G-24TS'),
	174 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560G-48TS'),
	175 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560G-24PS'),
	176 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560G-48PS'),
	177 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750-48TS'),
	178 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750-24PS'),
	179 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750-48PS'),
	180 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750-24FS'),
	181 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-24T'),
	182 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-24TS'),
	183 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-24TS-1U'),
	184 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-48TS'),
	185 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-24PS'),
	186 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-48PS'),
	187 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-16TD'),
	188 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-12S'),
	189 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750G-24WS'),
	190 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%EdgeIron 2402CF'),
	191 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%EdgeIron 24G'),
	192 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%EdgeIron 4802CF'),
	193 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%EdgeIron 48G'),
	194 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%EdgeIron 24GS'),
	195 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%EdgeIron 48GS'),
	196 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%EdgeIron 8X10G'),
	197 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 4802-PREM'),
	198 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 12GCF'),
	199 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 12GCF-PREM'),
	200 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 9604'),
	201 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 9604-PREM'),
	202 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 2402-POE'),
	203 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Edge 4802-POE'),
	204 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Workgroup X424'),
	205 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron Workgroup X448'),
	206 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron 450'),
	207 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron 850'),
	208 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron GT C'),
	209 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%ServerIron GT E'),
	210 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2970G-24T'),
	211 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2970G-24TS'),
	212 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora C1'),
	213 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora C2'),
	214 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora C3'),
	215 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora C4'),
	216 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora C5'),
	217 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora C6'),
	218 => array ('chapter_id' => 13, 'dict_value' => 'Solaris%GSKIP%Solaris 8'),
	219 => array ('chapter_id' => 13, 'dict_value' => 'Solaris%GSKIP%Solaris 9'),
	220 => array ('chapter_id' => 13, 'dict_value' => 'Solaris%GSKIP%Solaris 10'),
	221 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows 2000'),
	222 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows XP'),
	223 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows 2003'),
	224 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows Vista'),
	225 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V1'),
	226 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V2'),
	227 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V3'),
	228 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V4'),
	229 => array ('chapter_id' => 13, 'dict_value' => 'ALT_Linux%GSKIP%ALTLinux Master 2.0'),
	230 => array ('chapter_id' => 13, 'dict_value' => 'ALT_Linux%GSKIP%ALTLinux Master 2.2'),
	231 => array ('chapter_id' => 13, 'dict_value' => 'ALT_Linux%GSKIP%ALTLinux Master 2.4'),
	232 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 7'),
	233 => array ('chapter_id' => 13, 'dict_value' => 'SUSE Enterprise%GSKIP%SLES10'),
	234 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 3.0 (woody) | http://debian.org/releases/woody/]]'),
	235 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 3.1 (sarge) | http://debian.org/releases/sarge/]]'),
	236 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 1.x'),
	237 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 2.x'),
	238 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 3.x'),
	239 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 4.x'),
	240 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 5.x'),
	241 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 6.x'),
	242 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 8'),
	243 => array ('chapter_id' => 13, 'dict_value' => 'ALT_Linux%GSKIP%ALTLinux Master 4.0'),
	244 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 12.0'),
	245 => array ('chapter_id' => 14, 'dict_value' => 'Foundry SLB'),
	246 => array ('chapter_id' => 14, 'dict_value' => 'Foundry WXM'),
	247 => array ('chapter_id' => 14, 'dict_value' => 'Foundry L2'),
	248 => array ('chapter_id' => 14, 'dict_value' => 'Foundry full L3'),
	249 => array ('chapter_id' => 14, 'dict_value' => 'Foundry basic L3'),
	250 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 11.2'),
	251 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 12.1'),
	252 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 12.2'),
	253 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 11.3'),
	254 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 12.0'),
	255 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 12.1'),
	256 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 12.2'),
	257 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 12.3'),
	258 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 12.4'),
	259 => array ('chapter_id' => 16, 'dict_value' => 'Foundry L3'),
	260 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 11.2'),
	261 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 11.3'),
	262 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%BigIron 4000'),
	263 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7609'),
	264 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2610XM'),
	265 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2611XM'),
	266 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%3620'),
	267 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%3640'),
	268 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2621XM'),
	269 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7206VXR'),
	270 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2651XM'),
	271 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7204VXR'),
	272 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2612'),
	273 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2620XM'),
	274 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2650XM'),
	275 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2691'),
	276 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7603'),
	277 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7606'),
	278 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7613'),
	279 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2801'),
	280 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2811'),
	281 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2821'),
	282 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%2851'),
	283 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%3725'),
	284 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%3745'),
	285 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%3825'),
	286 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%3845'),
	287 => array ('chapter_id' => 24, 'dict_value' => 'Juniper%GPASS%NetScreen 100'),
	288 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron MLX-4'),
	289 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron MLX-8'),
	290 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron MLX-16'),
	291 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron MLX-32'),
	292 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron XMR 4000'),
	293 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron XMR 8000'),
	294 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron XMR 16000'),
	295 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%NetIron XMR 32000'),
	296 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%BigIron RX-4'),
	297 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%BigIron RX-8'),
	298 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%BigIron RX-16'),
	299 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%1841'),
	300 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%1812'),
	301 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%1811'),
	302 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%1803'),
	303 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%1802'),
	304 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%1801'),
	305 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7202'),
	306 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7204'),
	307 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7206'),
	308 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%7604'),
	309 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%OSR-7609'),
	310 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%BigIron 8000'),
	311 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%BigIron 15000'),
	312 => array ('chapter_id' => 18, 'dict_value' => 'Sun%GPASS%StorEdge A1000'),
	313 => array ('chapter_id' => 18, 'dict_value' => 'Dell/EMC AX150'),
	314 => array ('chapter_id' => 18, 'dict_value' => 'EMC CLARiiON CX600'),
	315 => array ('chapter_id' => 18, 'dict_value' => 'Sun%GPASS%StorEdge D240'),
	316 => array ('chapter_id' => 18, 'dict_value' => 'EMC CLARiiON CX300'),
	317 => array ('chapter_id' => 18, 'dict_value' => 'Sun%GPASS%StorageTek 6140'),
	318 => array ('chapter_id' => 18, 'dict_value' => 'Sun%GPASS%StorageTek 3511'),
	319 => array ('chapter_id' => 18, 'dict_value' => 'Sun%GPASS%StorageTek 3510'),
	320 => array ('chapter_id' => 18, 'dict_value' => 'Sun%GPASS%StorageTek 3320'),
	321 => array ('chapter_id' => 18, 'dict_value' => 'Sun%GPASS%StorageTek 3120'),
	322 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%NX1950'),
	323 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%220S'),
	324 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%MD3000'),
	325 => array ('chapter_id' => 19, 'dict_value' => 'Dell PowerVault%GPASS%136T'),
	326 => array ('chapter_id' => 19, 'dict_value' => 'Sun%GPASS%StorageTek SL500'),
	327 => array ('chapter_id' => 19, 'dict_value' => 'Sun%GPASS%StorageTek L1400'),
	328 => array ('chapter_id' => 19, 'dict_value' => 'Sun%GPASS%StorageTek SL8500'),
	329 => array ('chapter_id' => 19, 'dict_value' => 'Sun%GPASS%StorageTek C4'),
	330 => array ('chapter_id' => 19, 'dict_value' => 'Dell PowerVault%GPASS%ML6000'),
	331 => array ('chapter_id' => 19, 'dict_value' => 'Dell PowerVault%GPASS%132T'),
	332 => array ('chapter_id' => 19, 'dict_value' => 'Dell PowerVault%GPASS%TL4000'),
	333 => array ('chapter_id' => 19, 'dict_value' => 'Dell PowerVault%GPASS%TL2000'),
	334 => array ('chapter_id' => 19, 'dict_value' => 'Dell PowerVault%GPASS%124T'),
	335 => array ('chapter_id' => 19, 'dict_value' => 'Sun%GPASS%StorageTek C2'),
	338 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2216'),
	339 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2224'),
	340 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2324'),
	341 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2708'),
	342 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2716'),
	343 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2724'),
	344 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2748'),
	345 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3424'),
	346 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3424P'),
	347 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3448'),
	348 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3448P'),
	349 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%5324'),
	350 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%6224'),
	351 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%6224P'),
	352 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%6224F'),
	353 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%6248'),
	354 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%6248P'),
	355 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%6850'),
	356 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%6950'),
	357 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R900'),
	358 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%4400'),
	359 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%2650'),
	360 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%2550'),
	361 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%750'),
	362 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%2450'),
	363 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%850'),
	364 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%1850'),
	365 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%860'),
	366 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%2900'),
	367 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%2970'),
	368 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%SC1435'),
	369 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6509%L9,1H%'),
	370 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 6524GS-8S'),
	371 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 6524GT-8S'),
	372 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 4503-E'),
	373 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 4506-E'),
	374 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 4507R-E'),
	375 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 4510R-E'),
	376 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750-24TE-M'),
	377 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Catalyst 4948-10GE | http://www.cisco.com/en/US/products/ps6230/index.html]]'),
	378 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 4924-10GE'),
	379 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24-S'),
	380 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950-24'),
	381 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst WS-C2950-12'),
	382 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950C-24'),
	383 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst WS-C2950G-24-DC'),
	384 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950SX-48'),
	385 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst WS-C2950SX-24'),
	386 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950T-24'),
	387 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950T-48'),
	388 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950G-12'),
	389 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950G-24'),
	390 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2950G-48'),
	391 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3508G XL'),
	392 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3512 XL'),
	393 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst WS-C3524-XL'),
	394 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3524 PWR XL'),
	395 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3548 XL'),
	396 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 2400-24TS-A'),
	397 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 2400-24TS-D'),
	398 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3550-12T'),
	399 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3550-12G'),
	400 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3550-24'),
	401 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3550-24 FX'),
	402 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3550-24 DC'),
	403 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3550-24 PWR'),
	404 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3550-48'),
	405 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3400G-12CS-A'),
	406 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3400G-12CS-D'),
	407 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3400G-2CS-A'),
	408 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3400-24TS-A'),
	409 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3400-24TS-D'),
	410 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3400-24FS-A'),
	411 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron GS 624XGP'),
	412 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron GS 624XGP-POE'),
	413 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron LS 624'),
	414 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%FastIron LS 648'),
	415 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%NetIron M2404F'),
	416 => array ('chapter_id' => 12, 'dict_value' => 'Foundry%GPASS%NetIron M2404C'),
	417 => array ('chapter_id' => 17, 'dict_value' => 'Foundry%GPASS%BigIron RX-32'),
	418 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 2.0 (hamm) | http://debian.org/releases/hamm/]]'),
	419 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 2.1 (slink) | http://debian.org/releases/slink/]]'),
	420 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 2.2 (potato) | http://debian.org/releases/potato/]]'),
	421 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 4.0 (etch) | http://debian.org/releases/etch/]]'),
	422 => array ('chapter_id' => 13, 'dict_value' => 'ALT_Linux%GSKIP%ALTLinux Server 4.0'),
	423 => array ('chapter_id' => 13, 'dict_value' => 'ALT_Linux%GSKIP%ALTLinux Sisyphus'),
	424 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 10.0'),
	425 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 10.1'),
	426 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 10.2'),
	427 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 10.3'),
	428 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 4.10'),
	429 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 5.04'),
	430 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 5.10'),
	431 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 6.06 LTS'),
	432 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 6.10'),
	433 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 7.04'),
	434 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 7.10'),
	435 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 8.04 LTS'),
	436 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V5'),
	437 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%210S'),
	438 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%221S'),
	441 => array ('chapter_id' => 13, 'dict_value' => '[[CentOS%GSKIP%CentOS V2 | http://www.centos.org/]]'),
	442 => array ('chapter_id' => 13, 'dict_value' => '[[CentOS%GSKIP%CentOS V3 | http://www.centos.org/]]'),
	443 => array ('chapter_id' => 13, 'dict_value' => '[[CentOS%GSKIP%CentOS V4 | http://www.centos.org/]]'),
	444 => array ('chapter_id' => 13, 'dict_value' => '[[CentOS%GSKIP%CentOS V5 | http://www.centos.org/]]'),
	445 => array ('chapter_id' => 1, 'dict_value' => 'KVM switch'),
	447 => array ('chapter_id' => 1, 'dict_value' => 'multiplexer'),
	448 => array ('chapter_id' => 21, 'dict_value' => 'Avocent DSR1021'),
	449 => array ('chapter_id' => 21, 'dict_value' => 'Avocent DSR1022'),
	450 => array ('chapter_id' => 21, 'dict_value' => 'Avocent DSR1024'),
	451 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent DSR1031 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentDSRAppliances.aspx]]'),
	452 => array ('chapter_id' => 21, 'dict_value' => 'Avocent DSR1020'),
	453 => array ('chapter_id' => 21, 'dict_value' => 'Avocent DSR2020'),
	454 => array ('chapter_id' => 21, 'dict_value' => 'Avocent DSR4020'),
	455 => array ('chapter_id' => 21, 'dict_value' => 'Avocent DSR8020'),
	456 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent DSR1030 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentDSRAppliances.aspx]]'),
	457 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent DSR2030 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentDSRAppliances.aspx]]'),
	458 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent DSR2035 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentDSRAppliances.aspx]]'),
	459 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent DSR4030 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentDSRAppliances.aspx]]'),
	460 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent DSR8030 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentDSRAppliances.aspx]]'),
	461 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent DSR8035 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentDSRAppliances.aspx]]'),
	462 => array ('chapter_id' => 21, 'dict_value' => 'Avocent AutoView 1415'),
	463 => array ('chapter_id' => 21, 'dict_value' => 'Avocent AutoView 1515'),
	464 => array ('chapter_id' => 21, 'dict_value' => 'Avocent AutoView 2015'),
	465 => array ('chapter_id' => 21, 'dict_value' => 'Avocent AutoView 2020'),
	466 => array ('chapter_id' => 21, 'dict_value' => 'Avocent AutoView 2030'),
	467 => array ('chapter_id' => 21, 'dict_value' => 'Avocent AutoView 3100'),
	468 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent AutoView 3200 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentAutoViewAppliances.aspx]]'),
	469 => array ('chapter_id' => 21, 'dict_value' => 'Avocent SwitchView 1000 4-port'),
	470 => array ('chapter_id' => 21, 'dict_value' => 'Avocent SwitchView 1000 8-port'),
	471 => array ('chapter_id' => 21, 'dict_value' => 'Avocent SwitchView 1000 16-port'),
	472 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-4E1 | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	473 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-4E1/ETS | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	474 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-4E1/M | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	475 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-8E1 | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	476 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-8E1/ETS | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	477 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-8E1/M | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	478 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-16E1 | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	479 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-16E1/ETS | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	480 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/S-16E1/M | http://www.cronyx.ru/hardware/fmux-ring.html]]'),
	481 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%E1-XL/S | http://www.cronyx.ru/hardware/e1xl-s.html]]'),
	482 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%E1-DXC/S | http://www.cronyx.ru/hardware/e1dxc-s.html]]'),
	483 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX-4-E2 | http://www.cronyx.ru/hardware/fmux4-e2.html]]'),
	484 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX-16-E3 | http://www.cronyx.ru/hardware/fmux16-e3.html]]'),
	485 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%FMUX/SAT | http://www.cronyx.ru/hardware/fmux-sat.html]]'),
	486 => array ('chapter_id' => 9999, 'dict_value' => '[[Cronyx%GPASS%E1-XL/S-IP | http://www.cronyx.ru/hardware/e1xl-ip.html]]'),
	487 => array ('chapter_id' => 17, 'dict_value' => '[[RAD%GPASS%FCD-IPM | http://www.rad.com/Article/0,6583,36426-E1_T1_or_Fractional_E1_T1_Modular_Access_Device_with_Integrated_Router,00.html]]'),
	488 => array ('chapter_id' => 9999, 'dict_value' => '[[RAD%GPASS%FCD-E1M | http://www.rad.com/Article/0,6583,36723-E1_T1_Modular_Access_Multiplexer,00.html]]'),
	489 => array ('chapter_id' => 9999, 'dict_value' => '[[RAD%GPASS%FCD-T1M | http://www.rad.com/Article/0,6583,36723-E1_T1_Modular_Access_Multiplexer,00.html]]'),
	490 => array ('chapter_id' => 9999, 'dict_value' => '[[RAD%GPASS%FCD-155E | http://www.rad.com/Article/0,6583,36276-Ethernet_over_SDH_SONET_ADM,00.html]]'),
	491 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS78 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20070319151852001&layerid=subClass2]]'),
	492 => array ('chapter_id' => 21, 'dict_value' => '[[Aten ACS1208A | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224111025006&layerid=subClass2]]'),
	493 => array ('chapter_id' => 21, 'dict_value' => '[[Aten ACS1216A | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224111953008&layerid=subClass2]]'),
	494 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS1754 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050217161051008&layerid=subClass2]]'),
	495 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS1758 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224093143008&layerid=subClass2]]'),
	496 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS9134 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20070130133658002&pid=20050217172845005&layerid=subClass2]]'),
	497 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS9138 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224094519006&layerid=subClass2]]'),
	498 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS1708 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=2005022410563008&layerid=subClass2]]'),
	499 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS1716 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224110022008&layerid=subClass2]]'),
	500 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS1004 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224100546008&layerid=subClass2]]'),
	501 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS228 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224114323008&layerid=subClass2]]'),
	502 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS428 | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224114721008&layerid=subClass2]]'),
	503 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS138A | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=20050224111458007&layerid=subClass2]]'),
	504 => array ('chapter_id' => 21, 'dict_value' => '[[Aten CS88A | http://www.aten.com/products/productItem.php?pcid=20070130111936003&psid=20070130133658002&pid=2005022411042006&layerid=subClass2]]'),
	505 => array ('chapter_id' => 21, 'dict_value' => '[[Aten KM0832 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131626002&pid=20060628154826001&layerid=subClass1]]'),
	506 => array ('chapter_id' => 21, 'dict_value' => '[[Aten KM0216 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411131626002&pid=20060417153950007&layerid=subClass1]]'),
	507 => array ('chapter_id' => 21, 'dict_value' => '[[Aten KM0432 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411131626002&pid=2006041715359007&layerid=subClass1]]'),
	508 => array ('chapter_id' => 21, 'dict_value' => '[[Aten KH1508 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411130954001&pid=20061101174038001&layerid=subClass1]]'),
	509 => array ('chapter_id' => 21, 'dict_value' => '[[Aten KH1516 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411130954001&pid=20061101175320001&layerid=subClass1]]'),
	510 => array ('chapter_id' => 21, 'dict_value' => '[[Aten KH0116 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411130954001&pid=20060411145734003&layerid=subClass1]]'),
	511 => array ('chapter_id' => 21, 'dict_value' => '[[Aten KH98 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=2007012911116003&pid=20061221104352001&layerid=subClass1]]'),
	512 => array ('chapter_id' => 23, 'dict_value' => '[[Aten KL1100 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=20071225113046001&layerid=subClass1]]'),
	513 => array ('chapter_id' => 23, 'dict_value' => '[[Aten KL1508 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411131050002&pid=20070710020717009&layerid=subClass1]]'),
	514 => array ('chapter_id' => 23, 'dict_value' => '[[Aten KL1516 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411131050002&pid=20070716232614001&layerid=subClass1]]'),
	515 => array ('chapter_id' => 23, 'dict_value' => '[[Aten KL9108 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=20060811153413009&layerid=subClass1]]'),
	516 => array ('chapter_id' => 23, 'dict_value' => '[[Aten KL9116 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411131050002&pid=2006081115384001&layerid=subClass1]]'),
	517 => array ('chapter_id' => 23, 'dict_value' => '[[Aten KL3116 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=20060913162532009&layerid=subClass1]]'),
	518 => array ('chapter_id' => 23, 'dict_value' => '[[Aten KL1116 | http://www.aten.com/products/productItem.php?pcid=2006041110563001&psid=20060411131050002&pid=20060420101520005&layerid=subClass1]]'),
	519 => array ('chapter_id' => 23, 'dict_value' => '[[Aten CS1208DL | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=2005022413505007&layerid=subClass1]]'),
	520 => array ('chapter_id' => 23, 'dict_value' => '[[Aten CS1216DL | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=2005022413505007&layerid=subClass1]]'),
	521 => array ('chapter_id' => 23, 'dict_value' => '[[Aten CS1200L | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=20050224140854008&layerid=subClass1]]'),
	522 => array ('chapter_id' => 23, 'dict_value' => '[[Aten CL1758 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=20051229164553003&layerid=subClass1]]'),
	523 => array ('chapter_id' => 23, 'dict_value' => '[[Aten CL1208 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=2005072215482&layerid=subClass1]]'),
	524 => array ('chapter_id' => 23, 'dict_value' => '[[Aten CL1216 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=2005072215482&layerid=subClass1]]'),
	525 => array ('chapter_id' => 23, 'dict_value' => '[[Aten CL1200 | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=20050722165040002&layerid=subClass1]]'),
	526 => array ('chapter_id' => 23, 'dict_value' => '[[Aten ACS1208AL | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=2005022413597003&layerid=subClass1]]'),
	527 => array ('chapter_id' => 23, 'dict_value' => '[[Aten ACS1216AL | http://www.aten.com/products/productItem.php?pcid=2005010513171002&psid=20060411131050002&pid=2005022413597003&layerid=subClass1]]'),
	528 => array ('chapter_id' => 9999, 'dict_value' => '[[Tainet%GPASS%MUXpro 8216 | http://www.tainet.net/Product/muxpro820_8216.htm]]'),
	529 => array ('chapter_id' => 9999, 'dict_value' => 'Tainet%GPASS%Mercury 3600+'),
	530 => array ('chapter_id' => 9999, 'dict_value' => 'Tainet%GPASS%Mercury 3820'),
	531 => array ('chapter_id' => 9999, 'dict_value' => 'Tainet%GPASS%Mercury 3630'),
	532 => array ('chapter_id' => 9999, 'dict_value' => 'Tainet%GPASS%Mercury 3630E'),
	533 => array ('chapter_id' => 9999, 'dict_value' => 'Tainet%GPASS%DSD-08A'),
	534 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL160'),
	535 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL180'),
	536 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL185'),
	537 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL365'),
	538 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL320s'),
	539 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL320p'),
	540 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%ML115'),
	541 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%S2410CP'),
	542 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%S50N'),
	543 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%S50V'),
	544 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%S25P'),
	545 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%C150'),
	546 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%C300'),
	547 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%E300'),
	548 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%E600'),
	549 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%E1200'),
	550 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%JGS524F'),
	551 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%JGS516'),
	552 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%JFS524'),
	553 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%JFS524F'),
	554 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%JGS524'),
	555 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS524'),
	556 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%JFS516'),
	557 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7224R'),
	558 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7248'),
	559 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7212'),
	560 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM726S'),
	561 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7248R'),
	562 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7224v1'),
	563 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM750S'),
	564 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM726'),
	565 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS724TP'),
	566 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS748TS'),
	567 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS724T'),
	568 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS728TP'),
	569 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS752TS'),
	570 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS728TS'),
	571 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS726T'),
	572 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS748TP'),
	573 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS724TS'),
	574 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS748T'),
	575 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS716T'),
	576 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS752TPS'),
	577 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS750T2'),
	578 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FS726TP'),
	579 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM7328PS'),
	580 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7352S'),
	581 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7324'),
	582 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM7326P'),
	583 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM7352PS'),
	584 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7328FS'),
	585 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7328Sv1'),
	586 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7312'),
	587 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM7328S'),
	588 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%FSM7352S'),
	589 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-6500 | http://www.dlink.com/products/?sec=0&pid=341]]'),
	590 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DWS-3227 | http://www.dlink.com/products/?sec=0&pid=506]]'),
	591 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DWS-3227P | http://www.dlink.com/products/?sec=0&pid=507]]'),
	592 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DWS-3250 | http://www.dlink.com/products/?sec=0&pid=468]]'),
	593 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DWS-1008 | http://www.dlink.com/products/?sec=0&pid=434]]'),
	594 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3612G | http://www.dlink.com/products/?sec=0&pid=557]]'),
	595 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3627 | http://www.dlink.com/products/?sec=0&pid=639]]'),
	596 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3650 | http://www.dlink.com/products/?sec=0&pid=640]]'),
	597 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3324SR | http://www.dlink.com/products/?sec=0&pid=294]]'),
	598 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3324SRi | http://www.dlink.com/products/?sec=0&pid=309]]'),
	599 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DXS-3326GSR | http://www.dlink.com/products/?sec=0&pid=339]]'),
	600 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DXS-3350SR | http://www.dlink.com/products/?sec=0&pid=340]]'),
	601 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3828 | http://www.dlink.com/products/?sec=0&pid=439]]'),
	602 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3828P | http://www.dlink.com/products/?sec=0&pid=440]]'),
	603 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3100-24 | http://www.dlink.com/products/?sec=0&pid=635]]'),
	604 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3100-24P | http://www.dlink.com/products/?sec=0&pid=636]]'),
	605 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3100-48 | http://www.dlink.com/products/?sec=0&pid=637]]'),
	606 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3100-48P | http://www.dlink.com/products/?sec=0&pid=638]]'),
	607 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DXS-3227 | http://www.dlink.com/products/?sec=0&pid=483]]'),
	608 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DXS-3227P | http://www.dlink.com/products/?sec=0&pid=497]]'),
	609 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DXS-3250 | http://www.dlink.com/products/?sec=0&pid=443]]'),
	610 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3024 | http://www.dlink.com/products/?sec=0&pid=404]]'),
	611 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3224TGR | http://www.dlink.com/products/?sec=0&pid=269]]'),
	612 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-3048 | http://www.dlink.com/products/?sec=0&pid=496]]'),
	613 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3228PA | http://www.dlink.com/products/?sec=0&pid=644]]'),
	614 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3028 | http://www.dlink.com/products/?sec=0&pid=630]]'),
	615 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3028P | http://www.dlink.com/products/?sec=0&pid=631]]'),
	616 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3052 | http://www.dlink.com/products/?sec=0&pid=632]]'),
	617 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3052P | http://www.dlink.com/products/?sec=0&pid=633]]'),
	618 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3010FA | http://www.dlink.com/products/?sec=0&pid=423]]'),
	619 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3010GA | http://www.dlink.com/products/?sec=0&pid=424]]'),
	620 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3010PA | http://www.dlink.com/products/?sec=0&pid=469]]'),
	621 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3226L | http://www.dlink.com/products/?sec=0&pid=298]]'),
	622 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3526 | http://www.dlink.com/products/?sec=0&pid=330]]'),
	623 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-3550 | http://www.dlink.com/products/?sec=0&pid=331]]'),
	624 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-1216T | http://www.dlink.com/products/?sec=0&pid=324]]'),
	625 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-1224T | http://www.dlink.com/products/?sec=0&pid=329]]'),
	626 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-1248T | http://www.dlink.com/products/?sec=0&pid=367]]'),
	627 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-1316 | http://www.dlink.com/products/?sec=0&pid=353]]'),
	628 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-1228 | http://www.dlink.com/products/?sec=0&pid=540]]'),
	629 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-1228P | http://www.dlink.com/products/?sec=0&pid=541]]'),
	630 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-1252 | http://www.dlink.com/products/?sec=0&pid=555]]'),
	631 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-1016D | http://www.dlink.com/products/?sec=0&pid=337]]'),
	632 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DGS-1024D | http://www.dlink.com/products/?sec=0&pid=338]]'),
	633 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DSS-24+ | http://www.dlink.com/products/?sec=0&pid=73]]'),
	634 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-1024D | http://www.dlink.com/products/?sec=0&pid=75]]'),
	635 => array ('chapter_id' => 12, 'dict_value' => '[[D-Link%GPASS%DES-1026G | http://www.dlink.com/products/?sec=0&pid=76]]'),
	636 => array ('chapter_id' => 21, 'dict_value' => '[[D-Link%GPASS%DKVM-16 | http://www.dlink.com/products/?sec=0&pid=228]]'),
	637 => array ('chapter_id' => 21, 'dict_value' => '[[D-Link%GPASS%DKVM-8E | http://www.dlink.com/products/?sec=0&pid=161]]'),
	638 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC702'),
	639 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC702-GE'),
	640 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%ISCOM4300'),
	641 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC953-FE4E1'),
	642 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC953-FX4E1'),
	643 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC953-FE8E1'),
	644 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC953-FX8E1'),
	645 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC953-8FE16E1'),
	646 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC953E-3FE16E1'),
	647 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC953-GESTM1'),
	648 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%OPCOM3100-155'),
	649 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%OPCOM3101-155'),
	650 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC831-120'),
	651 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC831-120-BL'),
	652 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC831-240'),
	653 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC831-240E'),
	654 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2801-480GE-BL'),
	655 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2801-120FE'),
	656 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2801-120FE-BL'),
	657 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2801-240FE'),
	658 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2801-240FE-BL'),
	659 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2801-240EFE'),
	660 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2811-120FE'),
	661 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2811-240FE'),
	662 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2811-240FE-BL'),
	663 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2811-480FE'),
	664 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2811-480FE-BL'),
	665 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2811-240EFE'),
	666 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2104-120'),
	667 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2304-120'),
	668 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2504-120'),
	669 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2104-240'),
	670 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2304-240'),
	671 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RCMS2504-240'),
	672 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC801-120B'),
	673 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC801-240B'),
	674 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC801-480B'),
	675 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC803-120B'),
	676 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC803-240B'),
	677 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC803-480B'),
	678 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC805-120B'),
	679 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC805-240B'),
	680 => array ('chapter_id' => 9999, 'dict_value' => 'Raisecom%GPASS%RC805-480B'),
	683 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%S2410P'),
	684 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X150-24t | http://www.extremenetworks.com/products/summit-x150.aspx]]'),
	685 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X150-48t | http://www.extremenetworks.com/products/summit-x150.aspx]]'),
	686 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X150-24p | http://www.extremenetworks.com/products/summit-x150.aspx]]'),
	687 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X250e-24t | http://www.extremenetworks.com/products/summit-x250e.aspx]]'),
	688 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X250e-48t | http://www.extremenetworks.com/products/summit-x250e.aspx]]'),
	689 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X250e-24p | http://www.extremenetworks.com/products/summit-x250e.aspx]]'),
	690 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X250e-48p | http://www.extremenetworks.com/products/summit-x250e.aspx]]'),
	691 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X250e-24x | http://www.extremenetworks.com/products/summit-x250e.aspx]]'),
	692 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X450-24t | http://www.extremenetworks.com/products/summit-x450.aspx]]'),
	693 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X450-24x | http://www.extremenetworks.com/products/summit-x450.aspx]]'),
	694 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X450a-24t | http://www.extremenetworks.com/products/summit-x450a.aspx]]'),
	695 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X450a-48t | http://www.extremenetworks.com/products/summit-x450a.aspx]]'),
	696 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X450a-24x | http://www.extremenetworks.com/products/summit-x450a.aspx]]'),
	697 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X450e-24p | http://www.extremenetworks.com/products/summit-x450e.aspx]]'),
	698 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X450e-48p | http://www.extremenetworks.com/products/summit-x450e.aspx]]'),
	699 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 200-24fx | http://www.extremenetworks.com/products/summit-200.aspx]]'),
	700 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 200-24 | http://www.extremenetworks.com/products/summit-200.aspx]]'),
	701 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 200-48 | http://www.extremenetworks.com/products/summit-200.aspx]]'),
	702 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 300-24 | http://www.extremenetworks.com/products/summit-300.aspx]]'),
	703 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 300-48 | http://www.extremenetworks.com/products/summit-300.aspx]]'),
	704 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 400-24p | http://www.extremenetworks.com/products/summit-400-24p.aspx]]'),
	705 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 400-24t | http://www.extremenetworks.com/products/summit-400-24t.aspx]]'),
	706 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit 400-48t | http://www.extremenetworks.com/products/summit-400-48t.aspx]]'),
	707 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit48si | http://www.extremenetworks.com/products/summit-48si.aspx]]'),
	708 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Alpine 3804 | http://www.extremenetworks.com/products/Alpine-3800.aspx]]'),
	709 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Alpine 3808 | http://www.extremenetworks.com/products/Alpine-3800.aspx]]'),
	710 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%BlackDiamond 6808 | http://www.extremenetworks.com/products/blackdiamond-6800.aspx]]'),
	711 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%BlackDiamond 8806 | http://www.extremenetworks.com/products/blackdiamond-8800.aspx]]'),
	712 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%BlackDiamond 8810 | http://www.extremenetworks.com/products/blackdiamond-8800.aspx]]'),
	713 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%BlackDiamond 10808 | http://www.extremenetworks.com/products/blackdiamond-10808.aspx]]'),
	714 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%BlackDiamond 12802R | http://www.extremenetworks.com/products/blackdiamond-12800r.aspx]]'),
	715 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%BlackDiamond 12804R | http://www.extremenetworks.com/products/blackdiamond-12800r.aspx]]'),
	716 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%BlackDiamond 12804C | http://www.extremenetworks.com/products/blackdiamond-12804c.aspx]]'),
	717 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%ASR 1002 | http://cisco.com/en/US/products/ps9436/index.html]]'),
	718 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%ASR 1004 | http://cisco.com/en/US/products/ps9437/index.html]]'),
	719 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%ASR 1006 | http://cisco.com/en/US/products/ps9438/index.html]]'),
	720 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 3.3 | http://www.openbsd.org/33.html]]'),
	721 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 3.4 | http://www.openbsd.org/34.html]]'),
	722 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 3.5 | http://www.openbsd.org/35.html]]'),
	723 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 3.6 | http://www.openbsd.org/36.html]]'),
	724 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 3.7 | http://www.openbsd.org/37.html]]'),
	725 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 3.8 | http://www.openbsd.org/38.html]]'),
	726 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 3.9 | http://www.openbsd.org/39.html]]'),
	727 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.0 | http://www.openbsd.org/40.html]]'),
	728 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.1 | http://www.openbsd.org/41.html]]'),
	729 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.2 | http://www.openbsd.org/42.html]]'),
	730 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.3 | http://www.openbsd.org/43.html]]'),
// 1018 duplicated 731 later, so 731 isn't here any more
	732 => array ('chapter_id' => 13, 'dict_value' => '[[FreeBSD%GSKIP%FreeBSD 7.0 | http://www.freebsd.org/releases/7.0R/announce.html]]'),
	733 => array ('chapter_id' => 13, 'dict_value' => '[[NetBSD%GSKIP%NetBSD 2.0 | http://netbsd.org/releases/formal-2.0/]]'),
	734 => array ('chapter_id' => 13, 'dict_value' => '[[NetBSD%GSKIP%NetBSD 2.1 | http://netbsd.org/releases/formal-2.0/NetBSD-2.1.html]]'),
	735 => array ('chapter_id' => 13, 'dict_value' => '[[NetBSD%GSKIP%NetBSD 3.0 | http://netbsd.org/releases/formal-3/]]'),
	736 => array ('chapter_id' => 13, 'dict_value' => '[[NetBSD%GSKIP%NetBSD 3.1 | http://netbsd.org/releases/formal-3/NetBSD-3.1.html]]'),
	737 => array ('chapter_id' => 13, 'dict_value' => '[[NetBSD%GSKIP%NetBSD 4.0 | http://netbsd.org/releases/formal-4/NetBSD-4.0.html]]'),
	738 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2016'),
	739 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2024'),
	740 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2126-G'),
	741 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2816'),
	742 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2824'),
	743 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2226 Plus'),
	744 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2426-PWR Plus'),
	745 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2250 Plus'),
	746 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2916-SFP Plus'),
	747 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2924-SFP Plus'),
	748 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2924-PWR Plus'),
	749 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%Baseline 2948-SFP Plus'),
	750 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%3870 24-port'),
	751 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%3870 48-port'),
	752 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4200 26-port'),
	753 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4200 28-port'),
	754 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4200 50-port'),
	755 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4200G 12-port'),
	756 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4200G 24-port'),
	757 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4200G PWR 24-port'),
	758 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4200G 48-port'),
	759 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4210 26-port'),
	760 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4210 52-port'),
	761 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4210 26-port PWR'),
	762 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%SS3 4400 48-port'),
	763 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%SS3 4400 24-port'),
	764 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%SS3 4400 PWR'),
	765 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%SS3 4400 SE 24-port'),
	766 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500 26-port'),
	767 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500 50-port'),
	768 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500 PWR 26-port'),
	769 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500 PWR 50-port'),
	770 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500G 24-port'),
	771 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500G 48-port'),
	772 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500G PWR 24-port'),
	773 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4500G PWR 48-port'),
	774 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500-EI 28-port'),
	775 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500-EI 52-port'),
	776 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500-EI 28-port PWR'),
	777 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500-EI 52-port PWR'),
	778 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500-EI 28-port FX'),
	779 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500G-EI 24-port'),
	780 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500G-EI 48-port'),
	781 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500G-EI PWR 24-port'),
	782 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500G-EI 48-port PWR'),
	783 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%5500G-EI 24-port SFP'),
	784 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%7754'),
	785 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%7757'),
	786 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%7758'),
	787 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%8807'),
	788 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%8810'),
	789 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%8814'),
	790 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 9'),
	791 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 11.x'),
	792 => array ('chapter_id' => 11, 'dict_value' => 'SGI%GPASS%Altix XE250'),
	793 => array ('chapter_id' => 11, 'dict_value' => 'SGI%GPASS%Altix XE310'),
	794 => array ('chapter_id' => 11, 'dict_value' => 'SGI%GPASS%Altix XE320'),
	795 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco (blade)%GPASS%Catalyst 3032-DEL | http://www.cisco.com/en/US/products/ps8772/index.html]]'),
	796 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 8.10'),
	797 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.4 | http://www.openbsd.org/44.html]]'),
	798 => array ('chapter_id' => 1, 'dict_value' => 'Network security'),
	799 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5505'),
	800 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5510'),
	801 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5520'),
	802 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5540'),
	803 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5550'),
	804 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5580-20'),
	805 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5580-40'),
	806 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%IDS 4215 | http://www.cisco.com/en/US/products/hw/vpndevc/ps4077/ps5367/index.html]]'),
	807 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%IDS 4240 | http://www.cisco.com/en/US/products/ps5768/index.html]]'),
	808 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%IDS 4255 | http://www.cisco.com/en/US/products/ps5769/index.html]]'),
	809 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%IDS 4260 | http://www.cisco.com/en/US/products/ps6751/index.html]]'),
	810 => array ('chapter_id' => 24, 'dict_value' => '[[Cisco%GPASS%IDS 4270 | http://www.cisco.com/en/US/products/ps9157/index.html]]'),
	811 => array ('chapter_id' => 24, 'dict_value' => 'Foundry%GPASS%SecureIron 100'),
	812 => array ('chapter_id' => 24, 'dict_value' => 'Foundry%GPASS%SecureIron 100C'),
	813 => array ('chapter_id' => 24, 'dict_value' => 'Foundry%GPASS%SecureIron 300'),
	814 => array ('chapter_id' => 24, 'dict_value' => 'Foundry%GPASS%SecureIron 300C'),
	815 => array ('chapter_id' => 24, 'dict_value' => 'Foundry%GPASS%SecureIronLS 100-4802'),
	816 => array ('chapter_id' => 24, 'dict_value' => 'Foundry%GPASS%SecureIronLS 300-32GC02'),
	817 => array ('chapter_id' => 24, 'dict_value' => 'Foundry%GPASS%SecureIronLS 300-32GC10G'),
	818 => array ('chapter_id' => 24, 'dict_value' => '[[D-Link%GPASS%DFL-1600 | http://www.dlink.com/products/?sec=0&pid=454]]'),
	819 => array ('chapter_id' => 24, 'dict_value' => '[[D-Link%GPASS%DFL-M510 | http://www.dlink.com/products/?sec=0&pid=455]]'),
	820 => array ('chapter_id' => 24, 'dict_value' => '[[Extreme Networks%GPASS%Sentriant AG200 | http://www.extremenetworks.com/products/sentriant-ag200.aspx]]'),
	821 => array ('chapter_id' => 24, 'dict_value' => '[[Extreme Networks%GPASS%Sentriant NG300 | http://www.extremenetworks.com/products/sentriant-ng300.aspx]]'),
	822 => array ('chapter_id' => 24, 'dict_value' => 'Force10%GPASS%P-Series'),
	823 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%SSG 140 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/ssg_140/index.html]]'),
	824 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%SSG 320 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/ssg_300_series/index.html]]'),
	825 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%SSG 350 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/ssg_300_series/index.html]]'),
	826 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%SSG 520 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/ssg_500_series/index.html]]'),
	827 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%SSG 550 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/ssg_500_series/index.html]]'),
	828 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%ISG 1000 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/isg_series_slash_gprs/index.html]]'),
	829 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%ISG 2000 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/isg_series_slash_gprs/index.html]]'),
	830 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%NetScreen 5200 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/isg_series_slash_gprs/index.html]]'),
	831 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%NetScreen 5400 | http://www.juniper.net/products_and_services/firewall_slash_ipsec_vpn/isg_series_slash_gprs/index.html]]'),
	832 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%SRX 5600 | http://www.juniper.net/products_and_services/srx_series/index.html]]'),
	833 => array ('chapter_id' => 24, 'dict_value' => '[[Juniper%GPASS%SRX 5800 | http://www.juniper.net/products_and_services/srx_series/index.html]]'),
	834 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%PRO 1260 | http://www.sonicwall.com/us/products/PRO_1260.html]]'),
	835 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%PRO 2040 | http://www.sonicwall.com/us/products/PRO_2040.html]]'),
	836 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%PRO 3060 | http://www.sonicwall.com/us/products/PRO_3060.html]]'),
	837 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%PRO 4060 | http://www.sonicwall.com/us/products/PRO_4060.html]]'),
	838 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%PRO 4100 | http://www.sonicwall.com/us/products/PRO_4100.html]]'),
	839 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%PRO 5060 | http://www.sonicwall.com/us/products/PRO_5060.html]]'),
	840 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA 240 | http://www.sonicwall.com/us/products/NSA_240.html]]'),
	841 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA 2400 | http://www.sonicwall.com/us/products/NSA_2400.html]]'),
	842 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA 3500 | http://www.sonicwall.com/us/products/NSA_3500.html]]'),
	843 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA 4500 | http://www.sonicwall.com/us/products/NSA_4500.html]]'),
	844 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA 5000 | http://www.sonicwall.com/us/products/NSA_5000.html]]'),
	845 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA E5500 | http://www.sonicwall.com/us/products/NSA_E5500.html]]'),
	846 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA E6500 | http://www.sonicwall.com/us/products/NSA_E6500.html]]'),
	847 => array ('chapter_id' => 24, 'dict_value' => '[[SonicWall%GPASS%NSA E7500 | http://www.sonicwall.com/us/products/NSA_E7500.html]]'),
	848 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%1400-24G'),
	849 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%1700-24'),
	850 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%1800-24G'),
	851 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2124'),
	852 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2312'),
	853 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2324'),
	854 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2510-24'),
	855 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2510-48'),
	856 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2510G-24'),
	857 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2510G-48'),
	858 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2512'),
	859 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2524'),
	860 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2610-24 J9085A'),
	861 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2610-24-PoE J9087A'),
	862 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2610-48 J9088A'),
	863 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2610-48-PoE J9089A'),
	864 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2626'),
	865 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2626-PWR'),
	866 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2650'),
	867 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2650-PWR'),
	868 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2810-24G'),
	869 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2810-48G J9022A'),
	870 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2824'),
	871 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2848'),
	872 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2900-24G (J9049A)'),
	873 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2900-48G'),
	874 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%3400cl-24G'),
	875 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%3400cl-48G'),
	876 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%3500yl-24G-PWR'),
	877 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%3500yl-48G-PWR'),
	878 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%4202vl-72'),
	879 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%4204vl'),
	880 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%4204vl-48GS'),
	881 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%4208vl'),
	882 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%4208vl-72GS'),
	883 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%4208vl-96'),
	884 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%5304xl'),
	885 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%5308xl'),
	886 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%5348xl'),
	887 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%5372xl'),
	888 => array ('chapter_id' => 30, 'dict_value' => 'HP ProCurve%GPASS%5406zl J8697A%L4,2H%'),
	889 => array ('chapter_id' => 30, 'dict_value' => 'HP ProCurve%GPASS%5406zl-48G J8699A%L4,2H%'),
	890 => array ('chapter_id' => 30, 'dict_value' => 'HP ProCurve%GPASS%5412zl J8698A%L7,2H%'),
	891 => array ('chapter_id' => 30, 'dict_value' => 'HP ProCurve%GPASS%5412zl-96G J8700A%L7,2H%'),
	892 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6108'),
	893 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6200yl-24G-mGBIC'),
	894 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6400cl'),
	895 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6410cl'),
	896 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%8108fl'),
	897 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%8116fl'),
	898 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%8212zl'),
	899 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX3200-24P | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	900 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX3200-24T | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	901 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX3200-48P | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	902 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX3200-48T | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	903 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX4200-24F | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	904 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX4200-24P | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	905 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX4200-24T | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	906 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX4200-48P | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	907 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX4200-48T | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	908 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX8208 | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	909 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%E120 BSR | http://www.juniper.net/products_and_services/e_series_broadband_service/index.html]]'),
	910 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%E320 BSR | http://www.juniper.net/products_and_services/e_series_broadband_service/index.html]]'),
	911 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%ERX-310 | http://www.juniper.net/products_and_services/e_series_broadband_service/index.html]]'),
	912 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%ERX-705 | http://www.juniper.net/products_and_services/e_series_broadband_service/index.html]]'),
	913 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%ERX-710 | http://www.juniper.net/products_and_services/e_series_broadband_service/index.html]]'),
	914 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%ERX-1410 | http://www.juniper.net/products_and_services/e_series_broadband_service/index.html]]'),
	915 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%ERX-1440 | http://www.juniper.net/products_and_services/e_series_broadband_service/index.html]]'),
	916 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%J2320 | http://www.juniper.net/products_and_services/j_series_services_routers/index.html]]'),
	917 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%J2350 | http://www.juniper.net/products_and_services/j_series_services_routers/index.html]]'),
	918 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%J4350 | http://www.juniper.net/products_and_services/j_series_services_routers/index.html]]'),
	919 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%J6350 | http://www.juniper.net/products_and_services/j_series_services_routers/index.html]]'),
	920 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%M7i | http://www.juniper.net/products_and_services/m_series_routing_portfolio/index.html]]'),
	921 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%M10i | http://www.juniper.net/products_and_services/m_series_routing_portfolio/index.html]]'),
	922 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%M40e | http://www.juniper.net/products_and_services/m_series_routing_portfolio/index.html]]'),
	923 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%M120 | http://www.juniper.net/products_and_services/m_series_routing_portfolio/index.html]]'),
	924 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%M320 | http://www.juniper.net/products_and_services/m_series_routing_portfolio/index.html]]'),
	925 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX240 | http://www.juniper.net/products-services/routing/mx-series/mx240]]'),
	926 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX480 | http://www.juniper.net/products-services/routing/mx-series/mx480]]'),
	927 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX960 | http://www.juniper.net/products-services/routing/mx-series/mx960]]'),
	928 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%T320 | http://www.juniper.net/products_and_services/t_series_core_platforms/index.html]]'),
	929 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%T640 | http://www.juniper.net/products_and_services/t_series_core_platforms/index.html]]'),
	930 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%T1600 | http://www.juniper.net/products_and_services/t_series_core_platforms/index.html]]'),
	931 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%TX Matrix | http://www.juniper.net/products_and_services/t_series_core_platforms/index.html]]'),
	932 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 10'),
	933 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 11.1'),
	934 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP WebAccelerator 4500 | http://www.f5.com/pdf/products/big-ip-webaccelerator-ds.pdf]]'),
	935 => array ('chapter_id' => 30, 'dict_value' => 'F5%GPASS%VIPRION 2400%L2,2H%'),
	936 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 1500 | http://www.f5.com/pdf/products/big-ip-platforms-2007-ds.pdf]]'),
	937 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 1600 | http://www.f5.com/pdf/products/big-ip-platforms-ds.pdf]]'),
	938 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 3400 | http://www.f5.com/pdf/products/big-ip-platforms-2007-ds.pdf]]'),
	939 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 3600 | http://www.f5.com/pdf/products/big-ip-platforms-ds.pdf]]'),
	940 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 6400 | http://www.f5.com/pdf/products/big-ip-platforms-2007-ds.pdf]]'),
	941 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 6800 | http://www.f5.com/pdf/products/big-ip-platforms-2007-ds.pdf]]'),
	942 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 6900 | http://www.f5.com/pdf/products/big-ip-platforms-ds.pdf]]'),
	943 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 8400 | http://www.f5.com/pdf/products/big-ip-platforms-ds.pdf]]'),
	944 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 8800 | http://www.f5.com/pdf/products/big-ip-platforms-ds.pdf]]'),
	945 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%ARX 500 | http://www.f5.com/pdf/products/arx-series-ds.pdf]]'),
	946 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%ARX 1000 | http://www.f5.com/pdf/products/arx-series-ds.pdf]]'),
	947 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%ARX 4000 | http://www.f5.com/pdf/products/arx-series-ds.pdf]]'),
	948 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%ARX 6000 | http://www.f5.com/pdf/products/arx-series-ds.pdf]]'),
	949 => array ('chapter_id' => 17, 'dict_value' => '[[F5%GPASS%WANJet 300 | http://www.f5.com/pdf/products/wanjet-hardware-ds.pdf]]'),
	950 => array ('chapter_id' => 17, 'dict_value' => '[[F5%GPASS%WANJet 500 | http://www.f5.com/pdf/products/wanjet-hardware-ds.pdf]]'),
	951 => array ('chapter_id' => 24, 'dict_value' => '[[F5%GPASS%FirePass 1200 | http://www.f5.com/pdf/products/firepass-hardware-ds.pdf]]'),
	952 => array ('chapter_id' => 24, 'dict_value' => '[[F5%GPASS%FirePass 4100 | http://www.f5.com/pdf/products/firepass-hardware-ds.pdf]]'),
	953 => array ('chapter_id' => 24, 'dict_value' => '[[F5%GPASS%FirePass 4300 | http://www.f5.com/pdf/products/firepass-hardware-ds.pdf]]'),
	954 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 5.0 (lenny) | http://debian.org/releases/lenny/]]'),
	955 => array ('chapter_id' => 11, 'dict_value' => 'SGI%GPASS%Altix XE270'),
	956 => array ('chapter_id' => 11, 'dict_value' => 'SGI%GPASS%Altix XE340'),
	957 => array ('chapter_id' => 11, 'dict_value' => 'SGI%GPASS%Altix XE500'),
	958 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 2148T | http://cisco.com/en/US/products/ps10118/index.html]]'),
	959 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 5010 | http://cisco.com/en/US/products/ps9711/index.html]]'),
	960 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 5020 | http://cisco.com/en/US/products/ps9710/index.html]]'),
	961 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 7010 | http://cisco.com/en/US/products/ps9512/index.html]]'),
	962 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 7018 | http://cisco.com/en/US/products/ps10098/index.html]]'),
	963 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 4.0'),
	964 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 4.1'),
	965 => array ('chapter_id' => 1, 'dict_value' => 'Wireless'),
	966 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%2106'),
	967 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%2112'),
	968 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%2125'),
	969 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%4402'),
	970 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%4404'),
	971 => array ('chapter_id' => 25, 'dict_value' => '[[Cisco%GPASS%Aironet 1140 | http://cisco.com/en/US/products/ps10092/index.html]]'),
	972 => array ('chapter_id' => 25, 'dict_value' => '[[Cisco%GPASS%Aironet 1200 | http://cisco.com/en/US/products/hw/wireless/ps430/ps4076/index.html]]'),
	973 => array ('chapter_id' => 25, 'dict_value' => '[[Cisco%GPASS%Aironet 1230 AG | http://cisco.com/en/US/products/ps6132/index.html]]'),
	974 => array ('chapter_id' => 25, 'dict_value' => '[[Cisco%GPASS%Aironet 1240 AG | http://cisco.com/en/US/products/ps6521/index.html]]'),
	975 => array ('chapter_id' => 25, 'dict_value' => '[[Cisco%GPASS%Aironet 1250 | http://cisco.com/en/US/products/ps8382/index.html]]'),
	976 => array ('chapter_id' => 25, 'dict_value' => '[[Cisco%GPASS%Aironet 1520 | http://cisco.com/en/US/products/ps8368/index.html]]'),
	977 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%AP150'),
	978 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%OAP180'),
	979 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%AP201'),
	980 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%AP208'),
	981 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%AP250'),
	982 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%AP300'),
	983 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%RS4000'),
	984 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%MC500'),
	985 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%MC1000'),
	986 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%MC3000'),
	987 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%MC4100'),
	988 => array ('chapter_id' => 25, 'dict_value' => 'Foundry%GPASS%MC5000'),
	989 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R410'),
	990 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R610'),
	991 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R710'),
	992 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R805'),
	993 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R905'),
	994 => array ('chapter_id' => 31, 'dict_value' => 'Dell PowerEdge%GPASS%M1000e%L2,8V%'),
	995 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%MD1000'),
	996 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%MD1120'),
	997 => array ('chapter_id' => 18, 'dict_value' => 'Dell EqualLogic PS5000'),
	998 => array ('chapter_id' => 18, 'dict_value' => 'Dell EqualLogic PS6000'),
	999 => array ('chapter_id' => 18, 'dict_value' => 'EMC CLARiiON CX4-120 SPE'),
	1000 => array ('chapter_id' => 18, 'dict_value' => 'EMC CLARiiON CX4-240 SPE'),
	1001 => array ('chapter_id' => 18, 'dict_value' => 'EMC CLARiiON CX4-480 SPE'),
	1002 => array ('chapter_id' => 18, 'dict_value' => 'EMC CLARiiON CX4-960 SPE'),
	1003 => array ('chapter_id' => 18, 'dict_value' => 'EMC CLARiiON CX4 DAE'),
	1004 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%300'),
	1005 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%4900'),
	1006 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%5000'),
	1007 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%5100'),
	1008 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%5300'),
	1009 => array ('chapter_id' => 26, 'dict_value' => '[[Cisco%GPASS%MDS 9124 | http://www.cisco.com/en/US/products/ps7079/index.html]]'),
	1010 => array ('chapter_id' => 26, 'dict_value' => '[[Cisco%GPASS%MDS 9134 | http://www.cisco.com/en/US/products/ps8414/index.html]]'),
	1011 => array ('chapter_id' => 26, 'dict_value' => 'QLogic%GPASS%1400'),
	1012 => array ('chapter_id' => 26, 'dict_value' => 'QLogic%GPASS%3800'),
	1013 => array ('chapter_id' => 26, 'dict_value' => 'QLogic%GPASS%5600Q'),
	1014 => array ('chapter_id' => 26, 'dict_value' => 'QLogic%GPASS%5800V'),
	1015 => array ('chapter_id' => 26, 'dict_value' => 'QLogic%GPASS%9000'),
	1016 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%ASR 9006 | http://cisco.com/en/US/products/ps10075/index.html]]'),
	1017 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%ASR 9010 | http://cisco.com/en/US/products/ps10076/index.html]]'),
	1018 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Catalyst 4900M | http://www.cisco.com/en/US/products/ps9310/index.html]]'),
	1019 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Catalyst 4928-10GE | http://www.cisco.com/en/US/products/ps9903/index.html]]'),
	1022 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron CX 624S | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-cx-series/overview.page]]'),
	1023 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron CX 624S-HPOE | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-cx-series/overview.page]]'),
	1024 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron CX 648S | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-cx-series/overview.page]]'),
	1025 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron CX 648S-HPOE | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-cx-series/overview.page]]'),
	1026 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 624 | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1027 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 624-POE | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1028 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 624G | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1029 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 624G-POE | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1030 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 648 | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1031 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 648-POE | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1032 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 648G | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1033 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FastIron WS 648G-POE | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fastiron-ws-series/overview.page]]'),
	1034 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%NetIron CES 2024C | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/service-provider/product-details/netiron-ces-2000-series/overview.page]]'),
	1035 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%NetIron CES 2024F | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/service-provider/product-details/netiron-ces-2000-series/overview.page]]'),
	1036 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%NetIron CES 2048C | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/service-provider/product-details/netiron-ces-2000-series/overview.page]]'),
	1037 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%NetIron CES 2048F | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/service-provider/product-details/netiron-ces-2000-series/overview.page]]'),
	1038 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%NetIron CES 2048CX | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/service-provider/product-details/netiron-ces-2000-series/overview.page]]'),
	1039 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%NetIron CES 2048FX | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/service-provider/product-details/netiron-ces-2000-series/overview.page]]'),
	1040 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ServerIron ADX 1000 | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/application-switching/product-details/serveriron-adx-series/overview.page]]'),
	1041 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ServerIron ADX 4000 | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/application-switching/product-details/serveriron-adx-series/overview.page]]'),
	1042 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ServerIron ADX 8000 | http://www.brocade.com/products-solutions/products/ethernet-switches-routers/application-switching/product-details/serveriron-adx-series/overview.page]]'),
	1043 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ServerIron 4G-SSL-FIPS | http://www.brocade.com/sites/dotcom/products-solutions/products/ethernet-switches-routers/application-switching/product-details/serveriron-4g-application-switches/index.page]]'),
	1044 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%TurboIron 24X | http://www.brocade.com/sites/dotcom/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/turboiron-24x-switch/index.page]]'),
	1045 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 11'),
	1046 => array ('chapter_id' => 13, 'dict_value' => '[[NetBSD%GSKIP%NetBSD 5.0 | http://netbsd.org/releases/formal-5/NetBSD-5.0.html]]'),
	1047 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.5 | http://www.openbsd.org/45.html]]'),
	1048 => array ('chapter_id' => 13, 'dict_value' => '[[Solaris%GSKIP%OpenSolaris 2008.05 | http://opensolaris.org/os/project/indiana/resources/relnotes/200805/x86/]]'),
	1049 => array ('chapter_id' => 13, 'dict_value' => '[[Solaris%GSKIP%OpenSolaris 2008.11 | http://opensolaris.org/os/project/indiana/resources/relnotes/200811/x86/]]'),
	1050 => array ('chapter_id' => 13, 'dict_value' => '[[Solaris%GSKIP%OpenSolaris 2009.06 | http://opensolaris.org/os/project/indiana/resources/relnotes/200906/x86/]]'),
	1051 => array ('chapter_id' => 13, 'dict_value' => '[[Gentoo%GSKIP%Gentoo 2006.0 | http://www.gentoo.org/proj/en/releng/release/2006.0/2006.0.xml]]'),
	1052 => array ('chapter_id' => 13, 'dict_value' => '[[Gentoo%GSKIP%Gentoo 2007.0 | http://www.gentoo.org/proj/en/releng/release/2007.0/2007.0-press-release.txt]]'),
	1053 => array ('chapter_id' => 13, 'dict_value' => '[[Gentoo%GSKIP%Gentoo 2008.0 | http://www.gentoo.org/proj/en/releng/release/2008.0/index.xml]]'),
	1054 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 9.04'),
	1055 => array ('chapter_id' => 1, 'dict_value' => 'FC switch'),
	1056 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst CBS3030-DEL'),
	1057 => array ('chapter_id' => 13, 'dict_value' => '[[FreeBSD%GSKIP%FreeBSD 7.1 | http://www.freebsd.org/releases/7.1R/relnotes.html]]'),
	1058 => array ('chapter_id' => 13, 'dict_value' => '[[FreeBSD%GSKIP%FreeBSD 7.2 | http://www.freebsd.org/releases/7.2R/relnotes.html]]'),
	1059 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R200'),
	1060 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R300'),
	1061 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2808'),
	1062 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2816'),
	1063 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2824'),
	1064 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%2848'),
	1065 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3524'),
	1066 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3524P'),
	1067 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3548'),
	1068 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3548P'),
	1069 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%5424'),
	1070 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%5448'),
	1071 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%Silkworm 2400'),
	1072 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%Silkworm 2800'),
	1073 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%Silkworm 3200'),
	1074 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%Silkworm 3800'),
	1075 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%Silkworm 3900'),
	1076 => array ('chapter_id' => 26, 'dict_value' => 'Brocade%GPASS%Silkworm 4100'),
	1085 => array ('chapter_id' => 12, 'dict_value' => 'Nortel%GPASS%BES50GE-12T PWR'),
	1086 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%4000M'),
	1088 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%FAS2020 | http://www.netapp.com/us/products/storage-systems/fas2000/]]'),
	1089 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%FAS2050 | http://www.netapp.com/us/products/storage-systems/fas2000/]]'),
	1090 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3020'),
	1091 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3040'),
	1092 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3050'),
	1093 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3070'),
	1094 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%FAS3140 | http://www.netapp.com/us/products/storage-systems/fas3100/]]'),
	1095 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%FAS3160 | http://www.netapp.com/us/products/storage-systems/fas3100/]]'),
	1096 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%FAS3170 | http://www.netapp.com/us/products/storage-systems/fas3100/]]'),
	1097 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS6030'),
	1098 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%FAS6040 | http://www.netapp.com/us/products/storage-systems/fas6000/]]'),
	1099 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS6070'),
	1100 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%FAS6080 | http://www.netapp.com/us/products/storage-systems/fas6000/]]'),
	1101 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%V3140 | http://www.netapp.com/us/products/storage-systems/v3100/]]'),
	1102 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%V3160 | http://www.netapp.com/us/products/storage-systems/v3100/]]'),
	1103 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%V3170 | http://www.netapp.com/us/products/storage-systems/v3100/]]'),
	1104 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%V6030 | http://www.netapp.com/us/products/storage-systems/v6000/]]'),
	1105 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%V6040 | http://www.netapp.com/us/products/storage-systems/v6000/]]'),
	1106 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%V6070 | http://www.netapp.com/us/products/storage-systems/v6000/]]'),
	1107 => array ('chapter_id' => 18, 'dict_value' => '[[NetApp%GPASS%V6080 | http://www.netapp.com/us/products/storage-systems/v6000/]]'),
	1108 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%DS14mk2 AT'),
	1109 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%DS14mk2 FC'),
	1110 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%DS14mk4 FC'),
	1111 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7152 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7152]]'),
	1112 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7155 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7155]]'),
	1113 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7175 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7175]]'),
	1114 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7526 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7526]]'),
	1115 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7551 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7551]]'),
	1116 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7552 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7552]]'),
	1117 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7553 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7553]]'),
	1118 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7554 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7554]]'),
	1119 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7555 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7555]]'),
	1120 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7557 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7557]]'),
	1121 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7585 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7585]]'),
	1122 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7586 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7586]]'),
	1123 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7611 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7611]]'),
	1124 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7631 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7631]]'),
	1125 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7820 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7820]]'),
	1126 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7821 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7821]]'),
	1127 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7822 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7822]]'),
	1128 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7850 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7850]]'),
	1129 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7851 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7851]]'),
	1130 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7852 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7852]]'),
	1131 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7853 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7853]]'),
	1132 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7854 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7854]]'),
	1133 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7855A | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7855A]]'),
	1134 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7856 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7856]]'),
	1135 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7856A | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7856A]]'),
	1136 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7857 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7857]]'),
	1137 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7920 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7920]]'),
	1138 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7921 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7921]]'),
	1139 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7922 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7922]]'),
	1140 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7950 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7950]]'),
	1141 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7951 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7951]]'),
	1142 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7952 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7952]]'),
	1143 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7953 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7953]]'),
	1144 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7954 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7954]]'),
	1145 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7957 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7957]]'),
	1146 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP9559 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP9559]]'),
	1147 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP9565 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP9565]]'),
	1148 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP9568 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP9568]]'),
	1149 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP9572 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP9572]]'),
	1150 => array ('chapter_id' => 30, 'dict_value' => '[[Cisco%GPASS%Catalyst 6509-V-E%L1,9V% | http://www.cisco.com/en/US/products/ps9306/index.html]]'),
	1151 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7902J | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7902J]]'),
	1152 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7930J | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7930J]]'),
	1153 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7932J | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7932J]]'),
	1154 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7900 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7900]]'),
	1155 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7901 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7901]]'),
	1156 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7902 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7902]]'),
	1157 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7930 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7930]]'),
	1158 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7931 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7931]]'),
	1159 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7932 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7932]]'),
	1160 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7911 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7911]]'),
	1161 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7940 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7940]]'),
	1162 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7941 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7941]]'),
	1163 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7960 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7960]]'),
	1164 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7961 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7961]]'),
	1165 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7968 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7968]]'),
	1166 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7990 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7990]]'),
	1167 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7991 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7991]]'),
	1168 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP7998 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP7998]]'),
	1169 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3108FG2'),
	1170 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3124GT-HR2'),
	1171 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia4348GT'),
	1172 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia4348GT-PSR'),
	1173 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia4328GT'),
	1174 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia4224GT-PSR'),
	1175 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia2248G2'),
	1176 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia2124GT2'),
	1177 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia2124GT-SS2'),
	1178 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia2124-SS2'),
	1179 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia13000-24GX-PSR'),
	1180 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia13000-48X'),
	1181 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3424GT-SS'),
	1182 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3424GT-PoE'),
	1183 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3248G-PSR2'),
	1184 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3248G2'),
	1185 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3124GT-PSR2'),
	1186 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia3124GT2'),
	1187 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia18020'),
	1188 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia18008'),
	1189 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia18005'),
	1190 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia8007'),
	1191 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia8004'),
	1192 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia6148G-PSR'),
	1193 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%Apresia6148GT-PSR'),
	1194 => array ('chapter_id' => 12, 'dict_value' => 'Hitachi Cable%GPASS%VXC-1024FE'),
	1301 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-BR1410A'),
	1302 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-BR1310G'),
	1303 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1252AG'),
	1304 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1252G'),
	1305 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1242AG'),
	1306 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1242G'),
	1307 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1231G'),
	1308 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1232AG'),
	1309 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1131AG'),
	1310 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1131G'),
	1311 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1121G'),
	1312 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP521G'),
	1313 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-WLC2106'),
	1314 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-WLC526'),
	1315 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-WLC4402'),
	1317 => array ('chapter_id' => 13, 'dict_value' => 'SUSE Enterprise%GSKIP%SLES11'),
	1318 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows Server 2008'),
	1319 => array ('chapter_id' => 13, 'dict_value' => 'SlackWare%GSKIP%Slackware 13.0'),
	1320 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 9.10'),
	1321 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5328C-EI-24S'),
	1323 => array ('chapter_id' => 1, 'dict_value' => 'Voice/video'),
	1324 => array ('chapter_id' => 28, 'dict_value' => 'Cisco%GPASS%MCS 7816'),
	1325 => array ('chapter_id' => 28, 'dict_value' => 'Cisco%GPASS%MCS 7825'),
	1326 => array ('chapter_id' => 28, 'dict_value' => 'Cisco%GPASS%MCS 7835'),
	1327 => array ('chapter_id' => 17, 'dict_value' => '[[ Cisco%GPASS%2901 | http://www.cisco.com/en/US/products/ps10539/index.html]]'),
	1328 => array ('chapter_id' => 17, 'dict_value' => '[[ Cisco%GPASS%2911 | http://www.cisco.com/en/US/products/ps10540/index.html]]'),
	1329 => array ('chapter_id' => 17, 'dict_value' => '[[ Cisco%GPASS%2921 | http://www.cisco.com/en/US/products/ps10543/index.html]]'),
	1330 => array ('chapter_id' => 17, 'dict_value' => '[[ Cisco%GPASS%2951 | http://www.cisco.com/en/US/products/ps10544/index.html]]'),
	1331 => array ('chapter_id' => 13, 'dict_value' => 'ALT_Linux%GSKIP%ALTLinux 5'),
	1332 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 12 | http://docs.fedoraproject.org/release-notes/f12/en-US/html/]]'),
	1333 => array ('chapter_id' => 13, 'dict_value' => 'Gentoo%GSKIP%Gentoo 10.0'),
	1334 => array ('chapter_id' => 13, 'dict_value' => 'Gentoo%GSKIP%Gentoo 10.1'),
	1335 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5328C-EI'),
	1336 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5328C-PWR-EI'),
	1337 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5352C-EI'),
	1338 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5352C-PWR-EI'),
	1339 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5328C-SI'),
	1340 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5328C-PWR-SI'),
	1341 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5352C-SI'),
	1342 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5352C-PWR-SI'),
	1343 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5324TP-SI'),
	1344 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5324TP-PWR-SI'),
	1345 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5348TP-SI'),
	1346 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5348TP-PWR-SI'),
	1347 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24PC-L'),
	1348 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2350-48TD'),
	1349 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2910-48G J9147A'),
	1350 => array ('chapter_id' => 14, 'dict_value' => 'ExtremeXOS 10'),
	1351 => array ('chapter_id' => 14, 'dict_value' => 'ExtremeXOS 11'),
	1352 => array ('chapter_id' => 14, 'dict_value' => 'ExtremeXOS 12'),
	1353 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X480-24x | http://extremenetworks.com/products/summit-X480.aspx]]'),
	1354 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X480-48t | http://extremenetworks.com/products/summit-X480.aspx]]'),
	1355 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X480-48x | http://extremenetworks.com/products/summit-X480.aspx]]'),
	1356 => array ('chapter_id' => 12, 'dict_value' => '[[Extreme Networks%GPASS%Summit X650 | http://extremenetworks.com/products/summit-x650.aspx]]'),
	1357 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S9303'),
	1358 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S9306'),
	1359 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S9312'),
	1360 => array ('chapter_id' => 14, 'dict_value' => 'Huawei VRP 5.3'),
	1361 => array ('chapter_id' => 14, 'dict_value' => 'Huawei VRP 5.5'),
	1362 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%FCX 648 | http://www.brocade.com/sites/dotcom/products-solutions/products/ethernet-switches-routers/enterprise-mobility/product-details/fcx-series-data-center/index.page ]]'),
	1363 => array ('chapter_id' => 14, 'dict_value' => 'IronWare 5'),
	1364 => array ('chapter_id' => 14, 'dict_value' => 'IronWare 7'),
	1365 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 4.2'),
	1366 => array ('chapter_id' => 14, 'dict_value' => 'JunOS 9'),
	1367 => array ('chapter_id' => 14, 'dict_value' => 'JunOS 10'),
	1368 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX80 | http://www.juniper.net/products-services/routing/mx-series/mx80]]'),
	1369 => array ('chapter_id' => 14, 'dict_value' => 'Huawei VRP 5.7'),
	1370 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-48PST-L'),
	1371 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8024L2'),
	1372 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8124PL2'),
	1373 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8126L2'),
	1374 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8150L2'),
	1375 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8612XL3'),
	1376 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8708L2'),
	1377 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8824M'),
	1378 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8848M'),
	1379 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8926EM'),
	1380 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%8950EM'),
	1381 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R910'),
	1382 => array ('chapter_id' => 18, 'dict_value' => 'Dell PowerVault%GPASS%MD1220'),
	1383 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-48TD-L'),
	1384 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-24TD-L'),
	1385 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-48FPD-L'),
	1386 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-48LPD-L'),
	1387 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-24PD-L'),
	1388 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-48TS-L'),
	1389 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-24TS-L'),
	1390 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-48TS-S'),
	1391 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-24TS-S'),
	1392 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-48FPS-L'),
	1393 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-48LPS-L'),
	1394 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-24PS-L'),
	1395 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 6.0 (squeeze) | http://debian.org/releases/squeeze/]]'),
	1396 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V6'),
	1397 => array ('chapter_id' => 1, 'dict_value' => 'Power supply chassis'),
	1398 => array ('chapter_id' => 1, 'dict_value' => 'Power supply'),
	1400 => array ('chapter_id' => 34, 'dict_value' => 'Cisco%GPASS%RPS 2300'),
	1401 => array ('chapter_id' => 34, 'dict_value' => 'D-Link%GPASS%DPS-800'),
	1402 => array ('chapter_id' => 34, 'dict_value' => 'D-Link%GPASS%DPS-900'),
	1403 => array ('chapter_id' => 35, 'dict_value' => 'Cisco%GPASS%RPS 675'),
	1404 => array ('chapter_id' => 35, 'dict_value' => 'Cisco%GPASS%C3K-PWR-750WAC'),
	1405 => array ('chapter_id' => 35, 'dict_value' => 'Cisco%GPASS%C3K-PWR-1150WAC'),
	1406 => array ('chapter_id' => 35, 'dict_value' => 'D-Link%GPASS%DPS-200'),
	1407 => array ('chapter_id' => 35, 'dict_value' => 'D-Link%GPASS%DPS-500'),
	1408 => array ('chapter_id' => 35, 'dict_value' => 'D-Link%GPASS%DPS-510'),
	1409 => array ('chapter_id' => 35, 'dict_value' => 'D-Link%GPASS%DPS-600'),
	1410 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 5.0'),
	1411 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 5.1'),
	1412 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 5548P | http://cisco.com/en/US/products/ps11215/index.html]]'),
	1413 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 2224TP | http://cisco.com/en/US/products/ps11045/index.html]]'),
	1414 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 2248TP | http://cisco.com/en/US/products/ps10783/index.html]]'),
	1415 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 2232PP | http://cisco.com/en/US/products/ps10784/index.html]]'),
	1416 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 8.x'),
	1417 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL3.x | https://www.scientificlinux.org/]]'),
	1418 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL4.x | https://www.scientificlinux.org/]]'),
	1419 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL5.1 | https://www.scientificlinux.org/]]'),
	1420 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL5.2 | https://www.scientificlinux.org/]]'),
	1421 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL5.3 | https://www.scientificlinux.org/]]'),
	1422 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL5.4 | https://www.scientificlinux.org/]]'),
	1423 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2600-8-PWR'),
	1467 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1141N'),
	1468 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1262N'),
	1470 => array ('chapter_id' => 12, 'dict_value' => '[[Force10%GPASS%S55 | http://www.force10networks.com/products/s55.asp]]'),
	1471 => array ('chapter_id' => 12, 'dict_value' => '[[Force10%GPASS%S60 | http://www.force10networks.com/products/s60.asp]]'),
	1472 => array ('chapter_id' => 12, 'dict_value' => '[[Force10%GPASS%S4810 | http://www.force10networks.com/products/s4810.asp]]'),
	1473 => array ('chapter_id' => 31, 'dict_value' => 'IBM%GPASS%BladeCenter S'),
	1474 => array ('chapter_id' => 31, 'dict_value' => 'IBM%GPASS%BladeCenter H%L1,14V%'),
	1475 => array ('chapter_id' => 31, 'dict_value' => 'IBM%GPASS%BladeCenter E%L1,14V%'),
	1476 => array ('chapter_id' => 31, 'dict_value' => 'IBM%GPASS%BladeCenter T'),
	1477 => array ('chapter_id' => 31, 'dict_value' => 'IBM%GPASS%BladeCenter HT'),
	1478 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%HS12'),
	1479 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%HS20'),
	1480 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%HS21'),
	1481 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%HS21 XM'),
	1482 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%HS22'),
	1483 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%HS22V'),
	1484 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%HX5'),
	1485 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%JS12'),
	1486 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%JS20'),
	1487 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%JS21'),
	1488 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%JS22'),
	1489 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%JS23'),
	1490 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%JS43'),
	1491 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%LS20'),
	1492 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%LS21'),
	1493 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%LS22'),
	1494 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%LS41'),
	1495 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%LS42'),
	1496 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%PS700'),
	1497 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%PS701'),
	1498 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%PS702'),
	1499 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%PS703'),
	1500 => array ('chapter_id' => 29, 'dict_value' => 'No'),
	1501 => array ('chapter_id' => 29, 'dict_value' => 'Yes'),
	1502 => array ('chapter_id' => 1, 'dict_value' => 'Server chassis'),
	1503 => array ('chapter_id' => 1, 'dict_value' => 'Network chassis'),
	1504 => array ('chapter_id' => 1, 'dict_value' => 'VM'),
	1505 => array ('chapter_id' => 1, 'dict_value' => 'VM Cluster'),
	1506 => array ('chapter_id' => 1, 'dict_value' => 'VM Resource Pool'),
	1507 => array ('chapter_id' => 1, 'dict_value' => 'VM Virtual Switch'),
	1508 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESX 3.5'),
	1509 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 3.5'),
	1510 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESX 4.0'),
	1511 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 4.0'),
	1512 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESX 4.1'),
	1513 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 4.1'),
	1514 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 4.0'),
	1515 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 5.0'),
	1516 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 5.5'),
	1517 => array ('chapter_id' => 31, 'dict_value' => 'Dell PowerEdge%GPASS%1855%L1,10V%'),
	1518 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%1955'),
	1519 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M605'),
	1520 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M610'),
	1521 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M610x'),
	1522 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M710%L2,1%'),
	1523 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M805%L2,1%'),
	1524 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M905%L2,1%'),
	1525 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M910%L2,1%'),
	1526 => array ('chapter_id' => 26, 'dict_value' => 'Brocade (blade)%GPASS%McDATA 3014'),
	1527 => array ('chapter_id' => 26, 'dict_value' => 'Brocade (blade)%GPASS%McDATA 4314'),
	1528 => array ('chapter_id' => 26, 'dict_value' => 'Brocade (blade)%GPASS%McDATA 4416'),
	1529 => array ('chapter_id' => 26, 'dict_value' => 'Brocade (blade)%GPASS%M4424'),
	1530 => array ('chapter_id' => 26, 'dict_value' => 'Brocade (blade)%GPASS%M5424'),
	1531 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect (blade)%GPASS%5316M'),
	1532 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect (blade)%GPASS%M6220'),
	1533 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect (blade)%GPASS%M8024'),
	1534 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%Catalyst 3130G'),
	1535 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%Catalyst 3130X'),
	1536 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6148-GE-TX'),
	1537 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6148A-GE-45AF'),
	1538 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6148A-GE-TX'),
	1539 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6408A-GBIC'),
	1540 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6416-GBIC'),
	1541 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6516A-GBIC'),
	1542 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6548-GE-TX'),
	1543 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6548-GE-45AF'),
	1544 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6704-10GE'),
	1545 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6708-10G-3C'),
	1546 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6708-10G-3CXL'),
	1547 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6716-10GT-3C'),
	1548 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6716-10GT-3CXL'),
	1549 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6724-SFP'),
	1550 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6748-GE-TX'),
	1551 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6748-SFP'),
	1552 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-SUP720-3B'),
	1553 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-XSUP720-3BXL'),
	1554 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-SUP32-GE-3B'),
	1555 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-SUP32-10GE-3B'),
	1556 => array ('chapter_id' => 32, 'dict_value' => 'VMware%GPASS%Standard vSwitch'),
	1557 => array ('chapter_id' => 32, 'dict_value' => 'VMware%GPASS%Distributed vSwitch'),
	1558 => array ('chapter_id' => 32, 'dict_value' => 'Cisco%GPASS%Nexus 1000V'),
	1559 => array ('chapter_id' => 33, 'dict_value' => 'NS-OS 4.0'),
	1560 => array ('chapter_id' => 1, 'dict_value' => 'Rack'),
	1561 => array ('chapter_id' => 1, 'dict_value' => 'Row'),
	1562 => array ('chapter_id' => 1, 'dict_value' => 'Location'),
	1563 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%PS704'),
	1564 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%QS21'),
	1565 => array ('chapter_id' => 11, 'dict_value' => 'IBM BladeCenter%GPASS%QS22'),
	1566 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%SMC6110L2'),
	1567 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%SMC6128L2'),
	1568 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%SMC6128PL2'),
	1569 => array ('chapter_id' => 12, 'dict_value' => 'SMC%GPASS%SMC6152L2'),
	1570 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2610-24/12-PoE J9086A'),
	1571 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2910-24G'),
	1572 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-48TT-L'),
	1573 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-48TT-S'),
	1574 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560E-48TD'),
	1575 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560E-24TD'),
	1576 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560X-24T'),
	1577 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560X-48T'),
	1578 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560X-24P'),
	1579 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560X-48P'),
	1580 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560X-48PF'),
	1581 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750X-24T'),
	1582 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750X-48T'),
	1583 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750X-24P'),
	1584 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750X-48P'),
	1585 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750X-48PF'),
	1586 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750X-12S'),
	1587 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3750X-24S'),
	1589 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1261N'),
	1590 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-48TC-L'),
	1591 => array ('chapter_id' => 14, 'dict_value' => 'Force10 SFTOS 2'),
	1592 => array ('chapter_id' => 14, 'dict_value' => 'Force10 FTOS 6'),
	1593 => array ('chapter_id' => 14, 'dict_value' => 'Force10 FTOS 7'),
	1594 => array ('chapter_id' => 14, 'dict_value' => 'Force10 FTOS 8'),
	1595 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 13 | http://docs.fedoraproject.org/release-notes/f13/en-US/html/]]'),
	1596 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 14 | http://docs.fedoraproject.org/release-notes/f14/en-US/html/]]'),
	1597 => array ('chapter_id' => 16, 'dict_value' => 'JunOS 10'),
	1598 => array ('chapter_id' => 16, 'dict_value' => 'JunOS 11'),
	1599 => array ('chapter_id' => 16, 'dict_value' => 'JunOS 12'),
	1600 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2910-48G-PoE+ J9148A'),
	1601 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7328Sv2'),
	1602 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7224v2'),
	1605 => array ('chapter_id' => 12, 'dict_value' => 'HP GbE2c w/SFP'),
	1606 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2360-48TD'),
	1607 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560E-12D'),
	1608 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 5.0'),
	1609 => array ('chapter_id' => 17, 'dict_value' => 'Fortinet%GPASS%Fortigate 310B'),
	1610 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7124S'),
	1611 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%3348'),
	1612 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-48'),
	1613 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2309TP-EI'),
	1614 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2309TP-SI'),
	1615 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2318TP-EI'),
	1616 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2318TP-SI'),
	1617 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2326TP-EI'),
	1618 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2326TP-SI'),
	1619 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2352P-EI'),
	1620 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2309TP-PWR-EI'),
	1621 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2326TP-PWR-EI'),
	1622 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%5224'),
	1623 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%6024F'),
	1624 => array ('chapter_id' => 12, 'dict_value' => 'Linksys%GPASS%SRW2048'),
	1625 => array ('chapter_id' => 30, 'dict_value' => 'HP ProCurve%GPASS%5406zl-48G PoE+ J9447A%L4,2H%'),
	1626 => array ('chapter_id' => 30, 'dict_value' => 'HP ProCurve%GPASS%5412zl-96G PoE+ J9448A%L7,2H%'),
	1627 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl Management Module J8726A'),
	1628 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 24 1Gb-PoE J8702A'),
	1629 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 20 1Gb + 4 Mini-GBIC J8705A'),
	1630 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 24 Mini-GBIC J8706A'),
	1631 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 4 10GbE X2 J8707A'),
	1632 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 4 10GbE CX4 J8708A'),
	1633 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 24 1Gb-PoE+ J9307A'),
	1634 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 20 1Gb-PoE+ + 4 Mini-GBIC J9308A'),
	1635 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 4 10GbE SFP+ J9309A'),
	1636 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve (blade)%GPASS%5400zl 24 100Mb PoE+ J9478A'),
	1637 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6600-24G J9263A'),
	1638 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6600-24G-4XG J9264A'),
	1639 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6600-24XG J9265A'),
	1640 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6600-48G J9451A'),
	1641 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%6600-48G-4XG J9452A'),
	1643 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 6.0'),
	1644 => array ('chapter_id' => 1, 'dict_value' => 'serial console server'),
	1645 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6150 | http://www.moxa.com/product/NPort_6150.htm]]'),
	1646 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6610-8 | http://www.moxa.com/product/NPort_6650.htm]]'),
	1647 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6610-16 | http://www.moxa.com/product/NPort_6650.htm]]'),
	1648 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6610-32 | http://www.moxa.com/product/NPort_6650.htm]]'),
	1649 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6650-8 | http://www.moxa.com/product/NPort_6650.htm]]'),
	1650 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6650-16 | http://www.moxa.com/product/NPort_6650.htm]]'),
	1651 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6650-32 | http://www.moxa.com/product/NPort_6650.htm]]'),
	1652 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%CN2510-8 | http://www.moxa.com/product/CN2510.htm]]'),
	1653 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%CN2510-16 | http://www.moxa.com/product/CN2510.htm]]'),
	1654 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%CN2610-8 | http://www.moxa.com/product/CN2610.htm]]'),
	1655 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%CN2610-16 | http://www.moxa.com/product/CN2610.htm]]'),
	1656 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%CN2650-8 | http://www.moxa.com/product/CN2610.htm]]'),
	1657 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%CN2650-16 | http://www.moxa.com/product/CN2610.htm]]'),
	1658 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6250 | http://www.moxa.com/product/NPort_6250.htm]]'),
	1659 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort 6450 | http://www.moxa.com/product/NPort_6450.htm]]'),
	1660 => array ('chapter_id' => 36, 'dict_value' => '[[Moxa%GPASS%NPort S8458 | http://www.moxa.com/product/NPort_S8458_Series.htm]]'),
	1665 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL6.x | https://www.scientificlinux.org/]]'),
	1666 => array ('chapter_id' => 13, 'dict_value' => '[[SciLin%GSKIP%SL5.x | https://www.scientificlinux.org/]]'),
	1667 => array ('chapter_id' => 13, 'dict_value' => '[[CentOS%GSKIP%CentOS V6 | http://www.centos.org/]]'),
	1672 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R510'),
	1673 => array ('chapter_id' => 37, 'dict_value' => 'Cisco Aironet IOS 12.3'),
	1674 => array ('chapter_id' => 37, 'dict_value' => 'Cisco Aironet IOS 12.4'),
	1675 => array ('chapter_id' => 14, 'dict_value' => 'Arista EOS 4'),
	1676 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C1100'),
	1677 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C2100'),
	1678 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C5125'),
	1679 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C5220'),
	1680 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C6100'),
	1681 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C6105'),
	1682 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C6145'),
	1683 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge C%GPASS%C6220'),
	1684 => array ('chapter_id' => 31, 'dict_value' => 'Dell PowerEdge C%GPASS%C410x%L1,10V%'),
	1685 => array ('chapter_id' => 31, 'dict_value' => 'Dell PowerEdge C%GPASS%C5000%L1,12V%'),
	1686 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R210'),
	1687 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R310'),
	1688 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R415'),
	1689 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R515'),
	1690 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R620'),
	1691 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R715'),
	1692 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R720'),
	1693 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R720xd'),
	1694 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R810'),
	1695 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R815'),
	1696 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M620'),
	1697 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M710HD'),
	1698 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M915%L2,1%'),
	1699 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect (blade)%GPASS%M6348'),
	1700 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect (blade)%GPASS%M8428'),
	1701 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 16 | http://docs.fedoraproject.org/en-US/Fedora/16/html/Release_Notes/]]'),
	1702 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%8024'),
	1703 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%8024F'),
	1704 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 10.04 LTS'),
	1705 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 10.10'),
	1706 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 11.04'),
	1707 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 11.10'),
	1708 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 12.04 LTS'),
	1709 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 7 (wheezy) | http://debian.org/releases/wheezy/]]'),
	1710 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24TC-L'),
	1711 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2520-24-PoE J9138A'),
	1712 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R210 II'),
	1713 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.6 | http://www.openbsd.org/46.html]]'),
	1714 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.7 | http://www.openbsd.org/47.html]]'),
	1715 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.8 | http://www.openbsd.org/48.html]]'),
	1716 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 4.9 | http://www.openbsd.org/49.html]]'),
	1717 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.0 | http://www.openbsd.org/50.html]]'),
	1718 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.1 | http://www.openbsd.org/51.html]]'),
	1719 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2924M-XL'),
	1720 => array ('chapter_id' => 12, 'dict_value' => 'Allied Telesis%GPASS%AT9924T'),
	1721 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560E-12SD'),
	1722 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7148S'),
	1723 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7124SX'),
	1724 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7148SX'),
	1725 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7124FX'),
	1726 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7048T-A'),
	1727 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050T-64'),
	1728 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050T-52'),
	1729 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050Q-16'),
	1730 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050S-64'),
	1731 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050S-52'),
	1732 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 17 | http://docs.fedoraproject.org/en-US/Fedora/17/html/Release_Notes/]]'),
	1733 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 12.x'),
	1734 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 9.x'),
	1735 => array ('chapter_id' => 31, 'dict_value' => 'Cisco%GPASS%UCS 5108 Blade Chassis%L4,2H%'),
	1736 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B200 M1'),
	1737 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B200 M2'),
	1738 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B200 M3'),
	1739 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B230 M1'),
	1740 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B230 M2'),
	1741 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B250 M1'),
	1742 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B250 M2'),
	1743 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B440 M2'),
	1744 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B420 M3'),
	1745 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B22  M3'),
	1746 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C200 M2'),
	1747 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C210 M2'),
	1748 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C250 M2'),
	1749 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C260 M2'),
	1750 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C460 M2'),
	1751 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C22  M3'),
	1752 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C220 M3'),
	1753 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C24  M3'),
	1754 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C240 M3'),
	1755 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%UCS 6120 Fabric Interconnect'),
	1756 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%UCS 6140 Fabric Interconnect'),
	1757 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%UCS 6248 Fabric Interconnect'),
	1758 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%UCS 6296 Fabric Interconnect'),
	1759 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS XR 4.2'),
	1760 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-28TP-EI-24S'),
	1761 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-28TP-EI-MC-AC'),
	1762 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-28TP-PWR-EI'),
	1763 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28C-HI'),
	1764 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28C-PWR-EI'),
	1765 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-48TP-PWR-SI'),
	1766 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-52C-EI'),
	1767 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52P-PWR-LI'),
	1768 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700S-28P-LI'),
	1769 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE5850-48T4S2Q-EI'),
	1770 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S6700-24-EI'),
	1771 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S6700-48-EI'),
	1772 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6850-48S4Q-EI'),
	1773 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6850-48T4Q-EI'),
	1774 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S7703'),
	1775 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S7706'),
	1776 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S7712'),
	1777 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S9703'),
	1778 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S9706'),
	1779 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S9712'),
	1780 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE12804'),
	1781 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE12808'),
	1782 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE12812'),
	1783 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-52'),
	1784 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-24P'),
	1785 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-10'),
	1786 => array ('chapter_id' => 14, 'dict_value' => 'Marvell ROS 1.1'),
	1787 => array ('chapter_id' => 1, 'dict_value' => 'Management interface'),
	1788 => array ('chapter_id' => 38, 'dict_value' => 'Cisco%GPASS%UCS Domain'),
	1789 => array ('chapter_id' => 38, 'dict_value' => 'Generic%GPASS%Switch stack'),
	1790 => array ('chapter_id' => 38, 'dict_value' => 'VMware%GPASS%vSphere instance'),
	1791 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%5524'),
	1792 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%5548'),
	1793 => array ('chapter_id' => 12, 'dict_value' => 'TP-Link%GPASS%TL-SG5426'),
	1794 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GSM7352Sv2'),
	1795 => array ('chapter_id' => 25, 'dict_value' => 'Motorola%GPASS%RFS 4000'),
	1796 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2948G-L3'),
	1797 => array ('chapter_id' => 12, 'dict_value' => 'D-Link%GPASS%DGS-1210-10P'),
	1798 => array ('chapter_id' => 12, 'dict_value' => 'D-Link%GPASS%DGS-1210-16'),
	1799 => array ('chapter_id' => 12, 'dict_value' => 'D-Link%GPASS%DGS-1210-24'),
	1800 => array ('chapter_id' => 12, 'dict_value' => 'D-Link%GPASS%DGS-1210-48'),
	1801 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X670-48x'),
	1802 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X670V-48x'),
	1803 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X670V-48t'),
	1804 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560V2-24TS'),
	1805 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560V2-48TS'),
	1806 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560V2-24PS'),
	1807 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560V2-48PS'),
	1808 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560V2-24TS-SD'),
	1809 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 5.2'),
	1810 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5240'),
	1811 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5820'),
	1812 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows Server 2008 R2'),
	1813 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 12.10'),
	1814 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX100 S7 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx100/index.html]]'),
	1815 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX200 S7 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx200/index.html]]'),
	1816 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX300 S6 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx300/index.html]]'),
	1817 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX300 S7 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx300/index.html]]'),
	1818 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX350 S7 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx350/index.html]]'),
	1819 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX500 S7 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx500/index.html]]'),
	1820 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX600 S6 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx600/index.html]]'),
	1821 => array ('chapter_id' => 11, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY RX900 S2 | http://www.fujitsu.com/fts/products/computing/servers/primergy/rack/rx900/index.html]]'),
	1822 => array ('chapter_id' => 31, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY BX400 S1%L1,9V% | http://www.fujitsu.com/fts/products/computing/servers/primergy/blades/bx400/index.html]]'),
	1823 => array ('chapter_id' => 31, 'dict_value' => '[[Fujitsu%GSKIP%PRIMERGY BX900 S2%L2,9V% | http://www.fujitsu.com/fts/products/computing/servers/primergy/blades/bx900s2/index.html]]'),
	1824 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3690 X5 | http://www-03.ibm.com/systems/x/hardware/enterprise/x3690x5/index.html]]'),
	1825 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3850 X5 | http://www-03.ibm.com/systems/x/hardware/enterprise/x3850x5/index.html]]'),
	1826 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3950 X5 | http://www-03.ibm.com/systems/x/hardware/enterprise/x3850x5/index.html]]'),
	1827 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3750 M4 | http://www-03.ibm.com/systems/x/hardware/rack/x3750m4/index.html]]'),
	1828 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3650 M4 | http://www-03.ibm.com/systems/x/hardware/rack/x3650m4/index.html]]'),
	1829 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3630 M4 | http://www-03.ibm.com/systems/x/hardware/rack/x3630m4/index.html]]'),
	1830 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3550 M4 | http://www-03.ibm.com/systems/x/hardware/rack/x3550m4/index.html]]'),
	1831 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3530 M4 | http://www-03.ibm.com/systems/x/hardware/rack/x3530m4/index.html]]'),
	1832 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3250 M4 | http://www-03.ibm.com/systems/x/hardware/rack/x3250m4/index.html]]'),
	1833 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3755 M3 | http://www-03.ibm.com/systems/x/hardware/rack/x3755m3/index.html]]'),
	1834 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3650 M3 | http://www-03.ibm.com/systems/x/hardware/rack/x3650m3/index.html]]'),
	1835 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3630 M3 | http://www-03.ibm.com/systems/x/hardware/rack/x3630m3/index.html]]'),
	1836 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3620 M3 | http://www-03.ibm.com/systems/x/hardware/rack/x3620m3/index.html]]'),
	1837 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3550 M3 | http://www-03.ibm.com/systems/x/hardware/rack/x3550m3/index.html]]'),
	1838 => array ('chapter_id' => 11, 'dict_value' => '[[IBM xSeries%GPASS%x3250 M3 | http://www-03.ibm.com/systems/x/hardware/rack/x3250m3/index.html]]'),
	1839 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent%GPASS%AutoView 3008 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentAutoViewAppliances.aspx]]'),
	1840 => array ('chapter_id' => 21, 'dict_value' => '[[Avocent%GPASS%AutoView 3016 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/DigitalKVMAppliances/Pages/AvocentAutoViewAppliances.aspx]]'),
	1841 => array ('chapter_id' => 23, 'dict_value' => '[[Avocent%GPASS%ECS19PWRUSB | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/LCDConsoleTrays/Pages/AvocentLCDConsoleTray.aspx]]'),
	1842 => array ('chapter_id' => 23, 'dict_value' => '[[Avocent%GPASS%ECS19UWRUSB | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/LCDConsoleTrays/Pages/AvocentLCDConsoleTray.aspx]]'),
	1843 => array ('chapter_id' => 23, 'dict_value' => '[[Avocent%GPASS%ECS17KMMP | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/LCDConsoleTrays/Pages/AvocentLCDConsoleTray.aspx]]'),
	1844 => array ('chapter_id' => 23, 'dict_value' => '[[Avocent%GPASS%ECS17KMM | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/LCDConsoleTrays/Pages/AvocentLCDConsoleTray.aspx]]'),
	1845 => array ('chapter_id' => 23, 'dict_value' => '[[Avocent%GPASS%AP17KMMP | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/LCDConsoleTrays/Pages/AvocentLCDConsoleTray.aspx]]'),
	1846 => array ('chapter_id' => 23, 'dict_value' => '[[Avocent%GPASS%LCD17 | http://www.emersonnetworkpower.com/en-US/Products/InfrastructureManagement/LCDConsoleTrays/Pages/AvocentLCDConsoleTray.aspx]]'),
	1847 => array ('chapter_id' => 21, 'dict_value' => '[[APC%GPASS%AP5201 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP5201]]'),
	1848 => array ('chapter_id' => 21, 'dict_value' => '[[APC%GPASS%AP5202 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP5202]]'),
	1849 => array ('chapter_id' => 35, 'dict_value' => '[[Cisco%GPASS%RPS 2300 | http://www.cisco.com/en/US/products/ps7130/index.html]]'),
	1850 => array ('chapter_id' => 19, 'dict_value' => '[[NEC%GPASS%LL009F | http://www.nec.com/en/global/prod/tapestorage/index_009.html]]'),
	1851 => array ('chapter_id' => 19, 'dict_value' => '[[NEC%GPASS%T30A | http://www.nec.com/en/global/prod/tapestorage/index_t30a.html]]'),
	1852 => array ('chapter_id' => 19, 'dict_value' => '[[NEC%GPASS%T60A | http://www.nec.com/en/global/prod/tapestorage/index_t60a.html]]'),
	1853 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES F16F-R4031 | http://www.infortrend.com/global/products/models/ES%20F16F-R4031]]'),
	1854 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES F16F-R4840 | http://www.infortrend.com/global/products/models/ES%20F16F-R4840]]'),
	1855 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES F16F-S4031 | http://www.infortrend.com/global/products/models/ES%20F16F-S4031]]'),
	1856 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES F16F-S4840 | http://www.infortrend.com/global/products/models/ES%20F16F-S4840]]'),
	1857 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S24F-G1440 | http://www.infortrend.com/global/products/models/ES%20S24F-G1440]]'),
	1858 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S24F-G1840 | http://www.infortrend.com/global/products/models/ES%20S24F-G1840]]'),
	1859 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S24F-R1440 | http://www.infortrend.com/global/products/models/ES%20S24F-R1440]]'),
	1860 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S24F-R1840 | http://www.infortrend.com/global/products/models/ES%20S24F-R1840]]'),
	1861 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S16F-G1840 | http://www.infortrend.com/global/products/models/ES%20S16F-G1840]]'),
	1862 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S16F-R1840 | http://www.infortrend.com/global/products/models/ES%20S16F-R1840]]'),
	1863 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S12F-G1842 | http://www.infortrend.com/global/products/models/ES%20S12F-G1842]]'),
	1864 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S12F-R1840 | http://www.infortrend.com/global/products/models/ES%20S12F-R1840]]'),
	1865 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A08F-G2422 | http://www.infortrend.com/global/products/models/ES%20A08F-G2422]]'),
	1866 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A12F-G2422 | http://www.infortrend.com/global/products/models/ES%20A12F-G2422]]'),
	1867 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A12E-G2121 | http://www.infortrend.com/global/products/models/ES%20A12E-G2121]]'),
	1868 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A08S-C2133 | http://www.infortrend.com/global/products/models/ES%20A08S-C2133]]'),
	1869 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A08S-C2134 | http://www.infortrend.com/global/products/models/ES%20A08S-C2134]]'),
	1870 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A16S-G2130 | http://www.infortrend.com/global/products/models/ES%20A16S-G2130]]'),
	1871 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A08S-G2130 | http://www.infortrend.com/global/products/models/ES%20A08S-G2130]]'),
	1872 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A12S-G2130 | http://www.infortrend.com/global/products/models/ES%20A12S-G2130]]'),
	1873 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S16U-G1440 | http://www.infortrend.com/global/products/models/ES%20S16U-G1440]]'),
	1874 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S12U-G1440 | http://www.infortrend.com/global/products/models/ES%20S12U-G1440]]'),
	1875 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A24U-G2421 | http://www.infortrend.com/global/products/models/ES%20A24U-G2421]]'),
	1876 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A08U-G2421 | http://www.infortrend.com/global/products/models/ES%20A08U-G2421]]'),
	1877 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A12U-G2421 | http://www.infortrend.com/global/products/models/ES%20A12U-G2421]]'),
	1878 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A04U-G2421 | http://www.infortrend.com/global/products/models/ES%20A04U-G2421]]'),
	1879 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES F16F-J4000-R | http://www.infortrend.com/global/products/models/ES%20F16F-J4000-R]]'),
	1880 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES B12S-J1000-R | http://www.infortrend.com/global/products/models/ES%20B12S-J1000-R]]'),
	1881 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES B12S-J1000-S | http://www.infortrend.com/global/products/models/ES%20B12S-J1000-S]]'),
	1882 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES A16F-J2430-G | http://www.infortrend.com/global/products/models/ES%20A16F-J2430-G]]'),
	1883 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S16S-J1000-R1 | http://www.infortrend.com/global/products/models/ES%20S16S-J1000-R1]]'),
	1884 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S16S-J1000-S1 | http://www.infortrend.com/global/products/models/ES%20S16S-J1000-S1]]'),
	1885 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S12S-J1000-G | http://www.infortrend.com/global/products/models/ES%20S12S-J1000-G]]'),
	1886 => array ('chapter_id' => 18, 'dict_value' => '[[Infortrend%GPASS%ES S12S-J1002-R | http://www.infortrend.com/global/products/models/ES%20S12S-J1002-R]]'),
	1887 => array ('chapter_id' => 12, 'dict_value' => '[[IBM%GPASS%RackSwitch G8052 | http://www-03.ibm.com/systems/networking/switches/rack/g8052/]]'),
	1888 => array ('chapter_id' => 12, 'dict_value' => '[[IBM%GPASS%RackSwitch G8264 | http://www-03.ibm.com/systems/networking/switches/rack/g8264/]]'),
	1889 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ICX-6610-48-PE | http://www.brocade.com/products/all/switches/product-details/icx-6610-switch/index.page]]'),
	1890 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ICX-6650-48-E-ADV | http://www.brocade.com/products/all/switches/product-details/icx-6650-switch/index.page]]'),
	1891 => array ('chapter_id' => 12, 'dict_value' => '[[Fortinet%GPASS%Fortigate 3140B| http://www.fortinet.com/products/fortigate/3140B.html]]'),
	1892 => array ('chapter_id' => 12, 'dict_value' => '[[Fortinet%GPASS%Fortigate 300C | http://www.fortinet.com/products/fortigate/300C.html]]'),
	1893 => array ('chapter_id' => 12, 'dict_value' => '[[Fortinet%GPASS%Fortigate 800C | http://www.fortinet.com/products/fortigate/800C.html]]'),
	1894 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24TC-S'),
	1895 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24LT-L'),
	1896 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-48PST-S'),
	1897 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24PC-S'),
	1898 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-24LC-S'),
	1899 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-8TC-S'),
	1900 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960PD-8TT-L'),
	1901 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 15.0'),
	1902 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-F48FPS-L'),
	1903 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-F48LPS-L'),
	1904 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-F24PS-L'),
	1905 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-F48TS-L'),
	1906 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-F24TS-L'),
	1907 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-F48TS-S'),
	1908 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960S-F24TS-S'),
	1909 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-18TP-EI'),
	1910 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-18TP-SI'),
	1911 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-26TP-EI'),
	1912 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-26TP-PWR-EI'),
	1913 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-26TP-SI'),
	1914 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-52P-EI'),
	1915 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-52P-PWR-EI'),
	1916 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-9TP-EI'),
	1917 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-9TP-PWR-EI'),
	1918 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2700-9TP-SI'),
	1919 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2710-52P-PWR-SI'),
	1920 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S2710-52P-SI'),
	1921 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-26C-HI'),
	1922 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-28TP-EI'),
	1923 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-28TP-EI-MC'),
	1924 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-28TP-PWR-SI'),
	1925 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-28TP-SI'),
	1926 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-52P-EI'),
	1927 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-52P-EI-24S'),
	1928 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-52P-EI-48S'),
	1929 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-52P-PWR-EI'),
	1930 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-52P-PWR-SI'),
	1931 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S3700-52P-SI'),
	1932 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-10P-LI'),
	1933 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-10P-PWR-LI'),
	1934 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-24TP-PWR-SI'),
	1935 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-24TP-SI'),
	1936 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-26X-SI-12S'),
	1937 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28C-EI'),
	1938 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28C-EI-24S'),
	1939 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28C-HI-24S'),
	1940 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28C-PWR-SI'),
	1941 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28C-SI'),
	1942 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28P-LI'),
	1943 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28P-PWR-LI'),
	1944 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28X-LI'),
	1945 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28X-PWR-LI'),
	1946 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-48TP-SI'),
	1947 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52C-EI'),
	1948 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52C-PWR-EI'),
	1949 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52C-PWR-SI'),
	1950 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52C-SI'),
	1951 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52P-LI'),
	1952 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52X-LI'),
	1953 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52X-PWR-LI'),
	1954 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700S-52P-LI'),
	1955 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-28C-EI'),
	1956 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-28C-LI'),
	1957 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-28C-PWR-EI'),
	1958 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-28C-PWR-LI'),
	1959 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-52C-LI'),
	1960 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-52C-PWR-EI'),
	1961 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5710-52C-PWR-LI'),
	1962 => array ('chapter_id' => 12, 'dict_value' => 'Force10%GPASS%S4820T'),
	1963 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 15.1'),
	1964 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1041N'),
	1965 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-AP1042N'),
	1966 => array ('chapter_id' => 12, 'dict_value' => 'Linksys%GPASS%SRW224G4'),
	1967 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2520-8-PoE J9137A'),
	1968 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%K6%L3,2H%'),
	1969 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%K10%L5,2H%'),
	1970 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%SSA130'),
	1971 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%SSA150'),
	1972 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%SSA180'),
	1973 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%S1%L1,1H%'),
	1974 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%S3%L3,1H%'),
	1975 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%S4%L4,1H%'),
	1976 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%S6%L6,1H%'),
	1977 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%S8%L8,1H%'),
	1978 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%A4H124-24'),
	1979 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%A4H124-24P'),
	1980 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%A4H124-24FX'),
	1981 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%A4H124-48'),
	1982 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%A4H124-48P'),
	1983 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%A4H254-8F8T'),
	1984 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5G124-24'),
	1985 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5G124-24P2'),
	1986 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5G124-48'),
	1987 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5G124-48P2'),
	1988 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5K125-24'),
	1989 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5K125-24P2'),
	1990 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5K125-48'),
	1991 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%B5K125-48P2'),
	1992 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5G124-24'),
	1993 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5G124-24P2'),
	1994 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5G124-48'),
	1995 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5G124-48P2'),
	1996 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5K125-24'),
	1997 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5K125-24P2'),
	1998 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5K125-48'),
	1999 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5K125-48P2'),
	2000 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%C5K175-24'),
	2001 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%7124-24'),
	2002 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%7124-24T'),
	2003 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%7124-48'),
	2004 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%7124-48T'),
	2005 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08G20G2-08'),
	2006 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08G20G2-08P'),
	2007 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08G20G4-24'),
	2008 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08G20G4-24P'),
	2009 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08G20G4-48'),
	2010 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08G20G4-48P'),
	2011 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08H20G4-24'),
	2012 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08H20G4-24P'),
	2013 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08H20G4-48'),
	2014 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%08H20G4-48P'),
	2015 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%D2G124-12'),
	2016 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%D2G124-12P'),
	2017 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%G3G124-24'),
	2018 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%G3G124-24P'),
	2019 => array ('chapter_id' => 12, 'dict_value' => 'Enterasys%GPASS%G3G170-24'),
	2020 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%N1%L1,1H%'),
	2021 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%N3%L3,1H%'),
	2022 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%N5%L5,1H%'),
	2023 => array ('chapter_id' => 30, 'dict_value' => 'Enterasys%GPASS%N7%L1,7V%'),
	2024 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%877'),
	2025 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%878'),
	2026 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Catalyst 4948E | http://www.cisco.com/en/US/products/ps10947/index.html]]'),
	2027 => array ('chapter_id' => 14, 'dict_value' => 'Huawei VRP 8.5'),
	2028 => array ('chapter_id' => 14, 'dict_value' => 'Cisco NX-OS 6.1'),
	2029 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Catalyst C2960CG-8TC-L | http://www.cisco.com/en/US/products/ps6406/index.html]]'),
	2030 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME-3400EG-2CS-A'),
	2031 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M520'),
	2032 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M820%L2,1%'),
	2033 => array ('chapter_id' => 30, 'dict_value' => 'F5%GPASS%VIPRION 4480%L4,1H%'),
	2034 => array ('chapter_id' => 30, 'dict_value' => 'F5%GPASS%VIPRION 4800%L1,8V%'),
	2035 => array ('chapter_id' => 12, 'dict_value' => 'F5 (blade)%GPASS%VIPRION 2100'),
	2036 => array ('chapter_id' => 12, 'dict_value' => 'F5 (blade)%GPASS%VIPRION 4200'),
	2037 => array ('chapter_id' => 12, 'dict_value' => 'F5 (blade)%GPASS%VIPRION 4300'),
	2038 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Catalyst WS-CBS3012-IBM/-I | http://www.cisco.com/en/US/products/ps8766/index.html]]'),
	2039 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB1000U'),
	2040 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB1100'),
	2041 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB1100Hx2'),
	2042 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB1100AH'),
	2043 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB1100AHx2'),
	2044 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB1200'),
	2045 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB2011L-RM'),
	2046 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB2011iL-RM'),
	2047 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB2011UAS-RM'),
	2048 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1016-12G'),
	2049 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1036-8G-2S+'),
	2050 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1036-12G-4S'),
	2051 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7150S-24'),
	2052 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7150S-52'),
	2053 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7150S-64'),
	2054 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7100S'),
	2055 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050T-36'),
	2056 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050QX-32'),
	2057 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX-128'),
	2058 => array ('chapter_id' => 12, 'dict_value' => '[[TP-Link%GPASS%TL-SL5428E | http://www.tp-link.com/en/products/details/?model=TL-SL5428E]]'),
	2059 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3560CG-8PC-S'),
	2060 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 18 | http://docs.fedoraproject.org/en-US/Fedora/18/html/Release_Notes/]]'),
	2061 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 19 | http://docs.fedoraproject.org/en-US/Fedora/19/html/Release_Notes/]]'),
	2062 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS125-24G-1S-RM'),
	2063 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows Server 2012'),
	2064 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows Server 2012 R2'),
	2065 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X770'),
	2066 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7504'),
	2067 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7508'),
	2068 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7304'),
	2069 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7308'),
	2070 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7316'),
	2071 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7250QX-64'),
	2072 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1012'),
	2073 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1016'),
	2074 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1024'),
	2075 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1035'),
	2076 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1036'),
	2077 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5248'),
	2078 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-3290'),
	2079 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-3295'),
	2080 => array ('chapter_id' => 14, 'dict_value' => 'Huawei VRP 5.11'),
	2081 => array ('chapter_id' => 14, 'dict_value' => 'Huawei VRP 5.12'),
	2082 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 15.1'),
	2083 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%ASR 9001 | http://cisco.com/en/US/products/ps12074/index.html]]'),
	2084 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%ASR 9922 | http://cisco.com/en/US/products/ps11755/index.html]]'),
	2085 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 6001 | http://www.cisco.com/en/US/products/ps12869/index.html]]'),
	2086 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 6004 | http://www.cisco.com/en/US/products/ps12807/index.html]]'),
	2087 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE5810-48T4S-EI'),
	2088 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE5810-24T4S-EI'),
	2089 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-3780'),
	2090 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-3922'),
	2091 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF302-08MP'),
	2092 => array ('chapter_id' => 12, 'dict_value' => '3Com%GPASS%4510G 24-port'),
	2093 => array ('chapter_id' => 12, 'dict_value' => 'Linksys%GPASS%SRW248G4'),
	2094 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 1924'),
	2095 => array ('chapter_id' => 12, 'dict_value' => 'Allied Telesis%GPASS%AT-GS950/24'),
	2096 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2910-24G-PoE J9146A'),
	2097 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%8132'),
	2098 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%8132F'),
	2099 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%8164'),
	2100 => array ('chapter_id' => 12, 'dict_value' => 'Dell PowerConnect%GPASS%8164F'),
	2101 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.2 | http://www.openbsd.org/52.html]]'),
	2102 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.3 | http://www.openbsd.org/53.html]]'),
	2103 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.4 | http://www.openbsd.org/54.html]]'),
	2104 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 10.x'),
	2105 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 5.1'),
	2106 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 5.5'),
	2107 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 13.04'),
	2108 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 13.10'),
	2109 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 14.04 LTS'),
	2110 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-48FPD-L'),
	2111 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-48LPD-L'),
	2112 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-24PD-L'),
	2113 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-48TD-L'),
	2114 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-24TD-L'),
	2115 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-48FPS-L'),
	2116 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-48LPS-L'),
	2117 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-24PS-L'),
	2118 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-24PSQ-L'),
	2119 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-48TS-L'),
	2120 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-24TS-L'),
	2121 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-48TS-LL'),
	2122 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960X-24TS-LL'),
	2123 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-48FPD-I'),
	2124 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-48LPD-I'),
	2125 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-24PD-I'),
	2126 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-48TD-I'),
	2127 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-24TD-I'),
	2128 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-48FPS-I'),
	2129 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-48LPS-I'),
	2130 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-24PS-I'),
	2131 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-48TS-I'),
	2132 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960XR-24TS-I'),
	2133 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 48PST-L'),
	2134 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 24PC-L'),
	2135 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 24LC-L'),
	2136 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 48TC-L'),
	2137 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 24TC-L'),
	2138 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 48PST-S'),
	2139 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 24PC-S'),
	2140 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 24LC-S'),
	2141 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 24TC-S'),
	2142 => array ('chapter_id' => 14, 'dict_value' => 'Cisco IOS 15.2'),
	2143 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V7'),
	2144 => array ('chapter_id' => 38, 'dict_value' => 'Cisco%GPASS%Wireless Controller'),
	2145 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-3297'),
	2146 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-3930'),
	2147 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R420'),
	2148 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX4200-24PX | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	2149 => array ('chapter_id' => 12, 'dict_value' => '[[Juniper%GPASS%EX4200-48PX | http://www.juniper.net/products_and_services/ex_series/index.html]]'),
	2150 => array ('chapter_id' => 25, 'dict_value' => 'Cisco%GPASS%AIR-LAP1142N'),
	2151 => array ('chapter_id' => 14, 'dict_value' => 'JunOS 11'),
	2152 => array ('chapter_id' => 14, 'dict_value' => 'JunOS 12'),
	2153 => array ('chapter_id' => 9999, 'dict_value' => '[[Alcatel-Lucent%GPASS%1642 EMC | http://www.alcatel-lucent.com/products/1642-edge-multiplexer-compact]]'),
	2154 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%Z9000'),
	2155 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%Z9500'),
	2156 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N4032'),
	2157 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N4032F'),
	2158 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N4064'),
	2159 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N4064F'),
	2160 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N3024'),
	2161 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N3024F'),
	2162 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N3024P'),
	2163 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N3048'),
	2164 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N3048P'),
	2165 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N2024'),
	2166 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N2024P'),
	2167 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N2048'),
	2168 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%N2048P'),
	2169 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5220F-24T2XW'),
	2170 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5220F-20S2XW'),
	2171 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5459-48GT-4X2Q'),
	2172 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5459-48XP-4Q'),
	2173 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1009-8G-1S'),
	2174 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1009-8G-1S-1S+'),
	2175 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1016-12S-1S+'),
	2176 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%1910-24G (JE006A)'),
	2177 => array ('chapter_id' => 13, 'dict_value' => '[[Univention%GSKIP%Univention Corporate Server 3.2 (borgfeld) | http://docs.univention.de/release-notes-3.2-2-de.html]]'),
	2178 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP8941 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP8941]]'),
	2179 => array ('chapter_id' => 27, 'dict_value' => '[[APC%GPASS%AP8959EU3 | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=AP8959EU3]]'),
	2180 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 3.0 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_3.0]]'),
	2181 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 3.1 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_3.1]]'),
	2182 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 3.2 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_3.2]]'),
	2183 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5512-X'),
	2184 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5515-X'),
	2185 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5525-X'),
	2186 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5545-X'),
	2187 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5555-X'),
	2188 => array ('chapter_id' => 24, 'dict_value' => 'Cisco%GPASS%ASA 5585-X'),
	2189 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-24T'),
	2190 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-48T'),
	2191 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 2000S | http://www.f5.com/pdf/products/big-ip-platforms-datasheet.pdf]]'),
	2192 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 2200S | http://www.f5.com/pdf/products/big-ip-platforms-datasheet.pdf]]'),
	2193 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 3900 | http://www.f5.com/pdf/products/big-ip-platforms-datasheet.pdf]]'),
	2194 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 4000S | http://www.f5.com/pdf/products/big-ip-platforms-datasheet.pdf]]'),
	2195 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 4200V | http://www.f5.com/pdf/products/big-ip-platforms-datasheet.pdf]]'),
	2196 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 5000S | http://www.f5.com/pdf/products/big-ip-platforms-datasheet.pdf]]'),
	2197 => array ('chapter_id' => 12, 'dict_value' => '[[F5%GPASS%BIG-IP 5200V | http://www.f5.com/pdf/products/big-ip-platforms-datasheet.pdf]]'),
	2198 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXII-108'),
	2199 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXII-116'),
	2200 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXII-132'),
	2201 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-108'),
	2202 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-116'),
	2203 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-132'),
	2204 => array ('chapter_id' => 23, 'dict_value' => 'Dell%GPASS%18.5in LED KMM'),
	2205 => array ('chapter_id' => 36, 'dict_value' => 'Raritan%GPASS%Dominion SX4'),
	2206 => array ('chapter_id' => 36, 'dict_value' => 'Raritan%GPASS%Dominion SX8'),
	2207 => array ('chapter_id' => 36, 'dict_value' => 'Raritan%GPASS%Dominion SXA-8'),
	2208 => array ('chapter_id' => 36, 'dict_value' => 'Raritan%GPASS%Dominion SXA-16'),
	2209 => array ('chapter_id' => 36, 'dict_value' => 'Raritan%GPASS%Dominion SXA-16-DL'),
	2210 => array ('chapter_id' => 36, 'dict_value' => 'Raritan%GPASS%Dominion SXA-16-DLM'),
	2211 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%CGS-2520-24TC'),
	2212 => array ('chapter_id' => 12, 'dict_value' => 'Linksys%GPASS%SRW2024P'),
	2213 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%2920-48G J9728A'),
	2214 => array ('chapter_id' => 12, 'dict_value' => 'Edge-Core%GPASS%AS6700-32X'),
	2215 => array ('chapter_id' => 12, 'dict_value' => 'Edge-Core%GPASS%AS6701-32X'),
	2216 => array ('chapter_id' => 12, 'dict_value' => 'Edge-Core%GPASS%AS5610-52X'),
	2217 => array ('chapter_id' => 12, 'dict_value' => 'Edge-Core%GPASS%AS5600-52X'),
	2218 => array ('chapter_id' => 12, 'dict_value' => 'Edge-Core%GPASS%AS4600-54T'),
	2219 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 2960-Plus 48TC-S'),
	2220 => array ('chapter_id' => 31, 'dict_value' => 'Cisco%GPASS%UCS 5108 AC2 Blade Chassis%L4,2H%'),
	2221 => array ('chapter_id' => 31, 'dict_value' => 'Cisco%GPASS%UCS 5108 DC2 Blade Chassis%L4,2H%'),
	2222 => array ('chapter_id' => 31, 'dict_value' => 'Cisco%GPASS%UCS 5108 HVDC Blade Chassis%L4,2H%'),
	2223 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B260 M4'),
	2224 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B460 M4'),
	2225 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B200 M4'),
	2226 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE7850-32Q-EI'),
	2227 => array ('chapter_id' => 30, 'dict_value' => 'Cisco%GPASS%Catalyst 6807-XL%L7,1H%'),
	2228 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%VS-S2T-10G'),
	2229 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%VS-S2T-10G-XL'),
	2230 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6908-10G-2T'),
	2231 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6908-10G-2TXL'),
	2232 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6904-40G-2T'),
	2233 => array ('chapter_id' => 12, 'dict_value' => 'Cisco (blade)%GPASS%WS-X6904-40G-2TXL'),
	2234 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 3.3 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_3.3]]'),
	2235 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 9504 | http://www.cisco.com/c/en/us/products/switches/nexus-9504-switch/index.html]]'),
	2236 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 9508 | http://www.cisco.com/c/en/us/products/switches/nexus-9508-switch/index.html]]'),
	2237 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 9516 | http://www.cisco.com/c/en/us/products/switches/nexus-9516-switch/index.html]]'),
	2238 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%1910-48G (JE009A)'),
	2239 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ICX-6430-48 | http://www.brocade.com/products/all/switches/product-details/icx-6430-and-6450-switches/index.page]]'),
	2240 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ICX-6450-48 | http://www.brocade.com/products/all/switches/product-details/icx-6430-and-6450-switches/index.page]]'),
	2241 => array ('chapter_id' => 12, 'dict_value' => '[[IBM%GPASS%RackSwitch G8000 | http://www-03.ibm.com/systems/networking/switches/rack/g8000/]]'),
	2242 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%1810G-24 (J9450A)'),
	2243 => array ('chapter_id' => 17, 'dict_value' => 'Huawei%GPASS%NE20E-S4'),
	2244 => array ('chapter_id' => 17, 'dict_value' => 'Huawei%GPASS%NE20E-S8'),
	2245 => array ('chapter_id' => 17, 'dict_value' => 'Huawei%GPASS%NE20E-S16'),
	2246 => array ('chapter_id' => 17, 'dict_value' => 'Huawei%GPASS%NE40E-X3'),
	2247 => array ('chapter_id' => 17, 'dict_value' => 'Huawei%GPASS%NE40E-X8'),
	2248 => array ('chapter_id' => 17, 'dict_value' => 'Huawei%GPASS%NE40E-X16'),
	2249 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX5 | http://www.juniper.net/products-services/routing/mx-series/mx5]]'),
	2250 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX10 | http://www.juniper.net/products-services/routing/mx-series/mx10]]'),
	2251 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX40 | http://www.juniper.net/products-services/routing/mx-series/mx40]]'),
	2252 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX104 | http://www.juniper.net/products-services/routing/mx-series/mx104]]'),
	2253 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX2010 | http://www.juniper.net/products-services/routing/mx-series/mx2010]]'),
	2254 => array ('chapter_id' => 17, 'dict_value' => '[[Juniper%GPASS%MX2020 | http://www.juniper.net/products-services/routing/mx-series/mx2020]]'),
	2255 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SE-64'),
	2256 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SE-68'),
	2257 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SE-72'),
	2258 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX-64'),
	2259 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX-72'),
	2260 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX-96'),
	2261 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050TX-48'),
	2262 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050TX-64'),
	2263 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050TX-72'),
	2264 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050TX-96'),
	2265 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050TX-128'),
	2266 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050QX-32S'),
	2267 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7010T-48'),
	2268 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 20'),
	2269 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE 13.x'),
	2270 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%892'),
	2271 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%891'),
	2272 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%888'),
	2273 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%887V'),
	2274 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%887VA'),
	2275 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%887VA-W'),
	2276 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%886VA'),
	2277 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%886VA-W'),
	2278 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%881'),
	2279 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%880 3G'),
	2280 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%880G'),
	2281 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%867VAE'),
	2282 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%866VAE'),
	2283 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%861'),
	2284 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%819'),
	2285 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%812 CiFi'),
	2286 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%800M'),
	2287 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%C892FSP'),
	2288 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%C881W'),
	2289 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-5101'),
	2290 => array ('chapter_id' => 12, 'dict_value' => 'Pica8%GPASS%P-5401'),
	2291 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%S5000'),
	2292 => array ('chapter_id' => 12, 'dict_value' => 'Dell%GPASS%S6000'),
	2293 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X430-8p'),
	2294 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X430-24p'),
	2295 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X430-24t'),
	2296 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X430-48t'),
	2297 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-8t'),
	2298 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-8p'),
	2299 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-24t'),
	2300 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-24tDC'),
	2301 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-24p'),
	2302 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-24x'),
	2303 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-48t'),
	2304 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-48tDC'),
	2305 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-48p'),
	2306 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-24t-10G'),
	2307 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-24p-10G'),
	2308 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X440-24x-10G'),
	2309 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-24t'),
	2310 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-48t'),
	2311 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-24x'),
	2312 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-48x'),
	2313 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-24p'),
	2314 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-48p'),
	2315 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-24t-10GE4'),
	2316 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-48t-10GE4'),
	2317 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-24x-10GE4'),
	2318 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-48x-10GE4'),
	2319 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-24p-10GE4'),
	2320 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-48p-10GE4'),
	2321 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-24t-GE4'),
	2322 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-48t-GE4'),
	2323 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-24p-GE4'),
	2324 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X460-G2-48p-GE4'),
	2325 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X670-G2-48x-4q'),
	2326 => array ('chapter_id' => 12, 'dict_value' => 'Extreme Networks%GPASS%Summit X670-G2-72x'),
	2327 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3548 | http://www.cisco.com/c/en/us/products/switches/nexus-3548-switch/index.html]]'),
	2328 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3524 | http://www.cisco.com/c/en/us/products/switches/nexus-3524-switch/index.html]]'),
	2329 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3172 | http://www.cisco.com/c/en/us/products/switches/nexus-3172-switch/index.html]]'),
	2330 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3164Q | http://www.cisco.com/c/en/us/products/switches/nexus-3164q-switch/index.html]]'),
	2331 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3132Q | http://www.cisco.com/c/en/us/products/switches/nexus-3132q-switch/index.html]]'),
	2332 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3064 | http://www.cisco.com/c/en/us/products/switches/nexus-3064-switch/index.html]]'),
	2333 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3048 | http://www.cisco.com/c/en/us/products/switches/nexus-3048-switch/index.html]]'),
	2334 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 3016 | http://www.cisco.com/c/en/us/products/switches/nexus-3016-switch/index.html]]'),
	2335 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-50'),
	2336 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-50P'),
	2337 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-50FP'),
	2338 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-26'),
	2339 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-26P'),
	2340 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-26FP'),
	2341 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-18'),
	2342 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-10FP'),
	2343 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-08'),
	2344 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG200-08P'),
	2345 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF200-24'),
	2346 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF200-24P'),
	2347 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF200-24FP'),
	2348 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF200-48'),
	2349 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF200-48P'),
	2350 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG220-50P'),
	2351 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG220-50'),
	2352 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG220-26P'),
	2353 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG220-26'),
	2354 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF220-48P'),
	2355 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF220-48'),
	2356 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF220-24P'),
	2357 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF220-24'),
	2358 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF302-08PP'),
	2359 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF302-08MPP'),
	2360 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-10PP'),
	2361 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-10MPP'),
	2362 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-24PP'),
	2363 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-48PP'),
	2364 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-28PP'),
	2365 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-08'),
	2366 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-48P'),
	2367 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-10MP'),
	2368 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-10P'),
	2369 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-28P'),
	2370 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-28'),
	2371 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-20'),
	2372 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF302-08P'),
	2373 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-24'),
	2374 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF302-08'),
	2375 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF300-24MP'),
	2376 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-10SFP'),
	2377 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-28MP'),
	2378 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-52P'),
	2379 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG300-52MP'),
	2380 => array ('chapter_id' => 13, 'dict_value' => 'SUSE Enterprise%GSKIP%SLES12'),
	2381 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS226-24G-2S+RM'),
	2382 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE5850-48T4S2Q-HI'),
	2384 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL120'),
	2385 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge (blade)%GPASS%M630'),
	2386 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R430'),
	2387 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R530'),
	2388 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R630'),
	2389 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R730'),
	2390 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R730xd'),
	2391 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3210'),
	2392 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3220'),
	2393 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3240'),
	2394 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS3270'),
	2395 => array ('chapter_id' => 12, 'dict_value' => 'Juniper%GPASS%EX2200-24P-4G'),
	2396 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%E2620-24-PoE J9625A'),
	2397 => array ('chapter_id' => 14, 'dict_value' => 'JunOS 13'),
	2398 => array ('chapter_id' => 14, 'dict_value' => 'JunOS 14'),
	2399 => array ('chapter_id' => 14, 'dict_value' => 'JunOS 15'),
	2400 => array ('chapter_id' => 16, 'dict_value' => 'JunOS 13'),
	2401 => array ('chapter_id' => 16, 'dict_value' => 'JunOS 14'),
	2402 => array ('chapter_id' => 16, 'dict_value' => 'JunOS 15'),
	2403 => array ('chapter_id' => 13, 'dict_value' => 'SUSE Enterprise%GSKIP%SLES9'),
	2404 => array ('chapter_id' => 13, 'dict_value' => '[[CentOS%GSKIP%CentOS V7 | http://www.centos.org/]]'),
	2405 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 8 (Jessie) | http://debian.org/releases/jessie/]]'),
	2406 => array ('chapter_id' => 13, 'dict_value' => '[[Gentoo%GSKIP%Gentoo | http://gentoo.org]]'),
	2407 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF500-24'),
	2408 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF500-24P'),
	2409 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF500-24MP'),
	2410 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF500-48'),
	2411 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF500-48P'),
	2412 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SF500-48MP'),
	2413 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500-28'),
	2414 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500-28P'),
	2415 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500-28MPP'),
	2416 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500-52'),
	2417 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500-52P'),
	2418 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500-52MP'),
	2419 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500X-24'),
	2420 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500X-24P'),
	2421 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500X-24MPP'),
	2422 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500X-48'),
	2423 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500X-48P'),
	2424 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500X-48MP'),
	2425 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%SG500XG-8F8T'),
	2426 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 9332PQ'),
	2427 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 9372PX'),
	2428 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 9372PX-E'),
	2429 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 9372TX'),
	2430 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 9372TX-E'),
	2431 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 9396PX'),
	2432 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 9396TX'),
	2433 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 93120TX'),
	2434 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 93128TX'),
	2435 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R220'),
	2436 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNXe1600'),
	2437 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNXe3200'),
	2438 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-200'),
	2439 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-500'),
	2440 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-3020'),
	2441 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-3050'),
	2442 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-3060'),
	2443 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-5020'),
	2444 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-5050'),
	2445 => array ('chapter_id' => 24, 'dict_value' => 'Palo Alto Networks%GPASS%PA-5060'),
	2446 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 6.0'),
	2447 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-216'),
	2448 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-232'),
	2449 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-416'),
	2450 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-432'),
	2451 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-464'),
	2452 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-808'),
	2453 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-832'),
	2454 => array ('chapter_id' => 21, 'dict_value' => 'Raritan%GPASS%Dominion KXIII-864'),
	2455 => array ('chapter_id' => 21, 'dict_value' => 'TrippLite%GPASS%B051-000'),
	2456 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS2220'),
	2457 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS2240-2'),
	2458 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS2240-4'),
	2459 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS2520'),
	2460 => array ('chapter_id' => 18, 'dict_value' => '[[ NetApp%GPASS%FAS2552 | http://mysupport.netapp.com/documentation/docweb/index.html?productID=61619 ]]'),
	2461 => array ('chapter_id' => 18, 'dict_value' => 'NetApp%GPASS%FAS2554'),
	2462 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-24G-4SFP+ (JG932A)'),
	2463 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-24G-SFP-4SFP+ (JG933A)'),
	2464 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-48G-4SFP+ (JG934A)'),
	2465 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-24G-PoE+-4SFP+ (JG936A)'),
	2466 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-48G-PoE+-4SFP+ (JG937A)'),
	2467 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-24G-2SFP+-2XGT (JG938A)'),
	2468 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-48G-2SFP+-2XGT (JG939A)'),
	2469 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-24G-PoE+-2SFP+-2XGT (JG940A)'),
	2470 => array ('chapter_id' => 12, 'dict_value' => 'HP EI%GPASS%5130-48G-PoE+-2SFP+-2XGT (JG941A)'),
	2471 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1072-1G-8S+'),
	2472 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%3011UiAS-RM'),
	2473 => array ('chapter_id' => 16, 'dict_value' => 'OpenWrt 14.07'),
	2474 => array ('chapter_id' => 16, 'dict_value' => 'OpenWrt 15.05'),
	2475 => array ('chapter_id' => 16, 'dict_value' => 'RouterOS 6'),
	2476 => array ('chapter_id' => 37, 'dict_value' => 'OpenWrt 14.07'),
	2477 => array ('chapter_id' => 37, 'dict_value' => 'OpenWrt 15.05'),
	2478 => array ('chapter_id' => 37, 'dict_value' => 'RouterOS 6'),
	2479 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 7702 | http://www.cisco.com/c/en/us/products/switches/nexus-7700-2-slot-switch/index.html]]'),
	2480 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 7706 | http://www.cisco.com/c/en/us/products/switches/nexus-7700-6-slot-switch/index.html]]'),
	2481 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 7710 | http://www.cisco.com/c/en/us/products/switches/nexus-7700-10-slot-switch/index.html]]'),
	2482 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 7718 | http://www.cisco.com/c/en/us/products/switches/nexus-7700-18-slot-switch/index.html]]'),
	2483 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28X-LI-24S'),
	2484 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28P-LI-24S'),
	2485 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-52X-LI-48CS'),
	2486 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5701-28X-LI'),
	2487 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5701-28X-LI-24S'),
	2488 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28TP-LI'),
	2489 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5700-28TP-PWR-LI'),
	2490 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5701-28TP-PWR-LI'),
	2491 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-32P-EI'),
	2492 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-32X-EI'),
	2493 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-32X-EI-24S'),
	2494 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-36C-EI'),
	2495 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-36C-PWR-EI'),
	2496 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-36PC-EI'),
	2497 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-36C-EI-28S'),
	2498 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-50X-EI'),
	2499 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-50X-EI-46S'),
	2500 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-52X-EI'),
	2501 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-52P-EI'),
	2502 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-56C-EI-48S'),
	2503 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-56C-EI'),
	2504 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-56PC-EI'),
	2505 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S5720-56C-PWR-EI'),
	2506 => array ('chapter_id' => 12, 'dict_value' => 'Juniper%GPASS%QFX10000'),
	2507 => array ('chapter_id' => 12, 'dict_value' => 'Juniper%GPASS%QFX5200'),
	2508 => array ('chapter_id' => 12, 'dict_value' => 'Juniper%GPASS%QFX5100'),
	2509 => array ('chapter_id' => 12, 'dict_value' => 'Juniper%GPASS%QFX3600'),
	2510 => array ('chapter_id' => 12, 'dict_value' => 'Juniper%GPASS%QFX3500'),
	2511 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SB7800'),
	2512 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SB7890'),
	2513 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SB7700'),
	2514 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SB7790'),
	2515 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%CS7500'),
	2516 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%CS7510'),
	2517 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%CS7520'),
	2518 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6710'),
	2519 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6005'),
	2520 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6012'),
	2521 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6015'),
	2522 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6018'),
	2523 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6025'),
	2524 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6036'),
	2525 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6506'),
	2526 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6512'),
	2527 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6518'),
	2528 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX6536'),
	2529 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1710'),
	2530 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1410'),
	2531 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SX1024(52)'),
	2532 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SN2700'),
	2533 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SN2410'),
	2534 => array ('chapter_id' => 12, 'dict_value' => 'Mellanox%GPASS%SN2100'),
	2535 => array ('chapter_id' => 30, 'dict_value' => 'Ixia%GPASS%XM2'),
	2536 => array ('chapter_id' => 30, 'dict_value' => 'Ixia%GPASS%XM12'),
	2537 => array ('chapter_id' => 30, 'dict_value' => 'Ixia%GPASS%XG12'),
	2538 => array ('chapter_id' => 30, 'dict_value' => 'Ixia%GPASS%XGS2'),
	2539 => array ('chapter_id' => 30, 'dict_value' => 'Ixia%GPASS%XGS12-HS'),
	2540 => array ('chapter_id' => 30, 'dict_value' => 'Ixia%GPASS%XGS12-SD'),
	2541 => array ('chapter_id' => 30, 'dict_value' => 'Ixia%GPASS%400Tv2'),
	2542 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T1048-P02'),
	2543 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T1048-P02S'),
	2544 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T1048-LB9A'),
	2545 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T1048-LB9'),
	2546 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T1048-LY4A'),
	2547 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T1048-LY4B'),
	2548 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T1048-LY4C'),
	2549 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T3048-LY9'),
	2550 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T3048-LY8'),
	2551 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T3040-LY3'),
	2552 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T3048-LY2R'),
	2553 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T3048-LY2'),
	2554 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T3064-LY1R'),
	2555 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T5032-LY6'),
	2556 => array ('chapter_id' => 12, 'dict_value' => 'Quanta%GPASS%T5016-LB8D'),
	2557 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE8860-4C-EI'),
	2558 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B420 M4'),
	2559 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS B440 M1'),
	2560 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 15.10'),
	2561 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 16.04 LTS'),
	2562 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C22  M3(LFF)'),
	2563 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C220 M3(LFF)'),
	2564 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C24  M3(LFF)'),
	2565 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C240 M3(16 drive)'),
	2566 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C220 M4'),
	2567 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C220 M4(LFF)'),
	2568 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C240 M4'),
	2569 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C240 M4(16 drive)'),
	2570 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C240 M4(LFF)'),
	2571 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C240 M4S'),
	2572 => array ('chapter_id' => 13, 'dict_value' => 'Solaris%GSKIP%Solaris 11'),
	2573 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C420 M3'),
	2574 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%ASR 920'),
	2575 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C460 M4'),
	2576 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%UCS-Mini 6324 Fabric Interconnect'),
	2577 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%UCS 6332 Fabric Interconnect'),
	2578 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%UCS 6332-16UP Fabric Interconnect'),
	2579 => array ('chapter_id' => 11, 'dict_value' => 'Cisco%GPASS%UCS C240 M3(LFF)'),
	2580 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%1905 | http://www.cisco.com/c/en/us/products/routers/1905-serial-integrated-services-router-isr/index.html]]'),
	2581 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%1921 | http://www.cisco.com/c/en/us/products/routers/1921-integrated-services-router-isr/index.html]]'),
	2582 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%1941 | http://www.cisco.com/c/en/us/products/routers/1941-integrated-services-router-isr/index.html]]'),
	2583 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%1941W | http://www.cisco.com/c/en/us/products/routers/1941w-integrated-services-router-isr/index.html]]'),
	2584 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%3925 | http://www.cisco.com/c/en/us/products/routers/3925-integrated-services-router-isr/index.html]]'),
	2585 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%3925E | http://www.cisco.com/c/en/us/products/routers/3925e-integrated-services-router-isr/index.html]]'),
	2586 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%3945 | http://www.cisco.com/c/en/us/products/routers/3945-integrated-services-router-isr/index.html]]'),
	2587 => array ('chapter_id' => 17, 'dict_value' => '[[Cisco%GPASS%3945E | http://www.cisco.com/c/en/us/products/routers/3945e-integrated-services-router-isr/index.html]]'),
	2588 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE7855-32Q-EI'),
	2589 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6870-48S6CQ-EI'),
	2590 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6870-24S6CQ-EI'),
	2591 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6860-48S8CQ-EI'),
	2592 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6850U-24S2Q-HI'),
	2593 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6850U-48S6Q-HI'),
	2594 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6851-48S6Q-HI'),
	2595 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6855-48T6Q-HI'),
	2596 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6850-48T6Q-HI'),
	2597 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6855-48S6Q-HI'),
	2598 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6850-48S6Q-HI'),
	2599 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6810-48S4Q-EI'),
	2600 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6810-48S4Q-LI'),
	2601 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6810-48S-LI'),
	2602 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6810-24S2Q-LI'),
	2603 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6810-32T16S4Q-LI'),
	2604 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE5855-24T4S2Q-EI'),
	2605 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE5855-48T4S2Q-EI'),
	2606 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S6720-30C-EI-24S'),
	2607 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S6720-54C-EI-48S'),
	2608 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%S6720S-26Q-EI-24S'),
	2609 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.5 | http://www.openbsd.org/55.html]]'),
	2610 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.6 | http://www.openbsd.org/56.html]]'),
	2611 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.7 | http://www.openbsd.org/57.html]]'),
	2612 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.8 | http://www.openbsd.org/58.html]]'),
	2613 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 5.9 | http://www.openbsd.org/59.html]]'),
	2614 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 6.0 | http://www.openbsd.org/60.html]]'),
	2615 => array ('chapter_id' => 11, 'dict_value' => 'HP ProLiant%GPASS%DL980'),
	2616 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 11.x'),
	2617 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 5.6'),
	2618 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 6.0'),
	2619 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 6.1'),
	2620 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 6.2'),
	2621 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 6.5'),
	2622 => array ('chapter_id' => 13, 'dict_value' => 'Xen Hypervisor%GSKIP%XenServer 7.0'),
	2623 => array ('chapter_id' => 18, 'dict_value' => 'HP StorageWorks P6300'),
	2624 => array ('chapter_id' => 12, 'dict_value' => 'Ubiquiti EdgeSwitch ES-48-LITE'),
	2625 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%4221'),
	2626 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%4321'),
	2627 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%4331'),
	2628 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%4351'),
	2629 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%4431'),
	2630 => array ('chapter_id' => 17, 'dict_value' => 'Cisco%GPASS%4451'),
	2631 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-24TS'),
	2632 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48TS'),
	2633 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-24PS'),
	2634 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48PS'),
	2635 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48FS'),
	2636 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-24TD'),
	2637 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48TD'),
	2638 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-24PD'),
	2639 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-24PDM'),
	2640 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48PD'),
	2641 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48FD'),
	2642 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48TQ'),
	2643 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48PQ'),
	2644 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48FQ'),
	2645 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-48FQM'),
	2646 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-8X24UQ'),
	2647 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-12x48UQ'),
	2648 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-12X48UR'),
	2649 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3650-12X48UZ'),
	2650 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-24P'),
	2651 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-48P'),
	2652 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-48F'),
	2653 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-24U'),
	2654 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-48U'),
	2655 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-24XU'),
	2656 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-12X48U'),
	2657 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-12S'),
	2658 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-24S'),
	2659 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-12XS'),
	2660 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-24XS'),
	2661 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 3850-48XS'),
	2662 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Catalyst 4510R+E'),
	2663 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3600X-24TS-M'),
	2664 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME 3600X-24FS-M'),
	2665 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%VDX 6740 | http://www.brocade.com/en/products-services/switches/data-center-switches/vdx-6740-switches.html]]'),
	2666 => array ('chapter_id' => 12, 'dict_value' => '[[Brocade%GPASS%ICX-7250-48 | http://www.brocade.com/en/products-services/switches/campus-network-switches/icx-7250-switch.html]]'),
	2667 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 15.0'),
	2668 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 15.2'),
	2669 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 15.3'),
	2670 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 15.4'),
	2671 => array ('chapter_id' => 16, 'dict_value' => 'Cisco IOS 15.5'),
	2672 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX-72Q'),
	2673 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX2-72Q'),
	2674 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX2-128'),
	2675 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050TX-72Q'),
	2676 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050TX2-128'),
	2677 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050QX2-32S'),
	2678 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7060CX2-32S'),
	2679 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7060CX-32S'),
	2680 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7060QX-64'),
	2681 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7060CX-64'),
	2682 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280CR-48'),
	2683 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280QR-C72'),
	2684 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280QR-C36'),
	2685 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SR-48C6'),
	2686 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280TR-48C6'),
	2687 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7324X'),
	2688 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7328X'),
	2689 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7504R'),
	2690 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7508R'),
	2691 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7512R'),
	2692 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 21'),
	2693 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 22'),
	2694 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 23'),
	2695 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 24'),
	2696 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 25'),
	2697 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 16.10'),
	2698 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 5.1'),
	2699 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 5.2'),
	2700 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 6.0'),
	2701 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 6.1'),
	2702 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 7.0'),
	2703 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE Leap 42.x'),
	2704 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1009-7G-1C-1S+'),
	2705 => array ('chapter_id' => 13, 'dict_value' => '[[OpenBSD%GSKIP%OpenBSD 6.1 | http://www.openbsd.org/61.html]]'),
	2706 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE8850-32CQ-EI'),
	2707 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows Server 2016'),
	2708 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro%GPASS%6028R-E1CR12L | https://www.supermicro.com/products/system/2u/6028/ssg-6028r-e1cr12l.cfm]]'),
	2709 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC0R-SIOM | https://www.supermicro.com/products/system/2U/2028/SYS-2028TP-HC0R-SIOM.cfm]]'),
	2710 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026GT-TRF-FM375 | http://www.supermicro.com/products/system/1U/1026/SYS-1026GT-TRF-FM375.cfm]]'),
	2711 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-73HDP+ | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-73HDP_.cfm]]'),
	2712 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045W-NTR+ | http://www.supermicro.com/products/system/4U/7045/SYS-7045W-NTR_.cfm]]'),
	2713 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RC1PT+ | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-RC1PT_.cfm]]'),
	2714 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DC0R | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DC0R.cfm]]'),
	2715 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-N3RF | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-N3RF.cfm]]'),
	2716 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-E1CNRT+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-E1CNRT_.cfm]]'),
	2717 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025B-8R+ | http://www.supermicro.com/products/system/2U/6025/SYS-6025B-8R_.cfm]]'),
	2718 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-U | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-U.cfm]]'),
	2719 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-T | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-T.cfm]]'),
	2720 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DTR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DTR.cfm]]'),
	2721 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5013C-MT | http://www.supermicro.com/products/system/1U/5013/SYS-5013C-MT.cfm]]'),
	2722 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6048R-E1CR24H | http://www.supermicro.com/products/system/4u/6048/SSG-6048R-E1CR24H.cfm]]'),
	2723 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017TR-TFF | http://www.supermicro.com/products/system/1u/6017/SYS-6017TR-TFF.cfm]]'),
	2724 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-FTPT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R3-FTPT_.cfm]]'),
	2725 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018A-FTN4 | http://www.supermicro.com/products/system/1u/5018/SYS-5018A-FTN4.cfm]]'),
	2726 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017C-URF | http://www.supermicro.com/products/system/1u/5017/SYS-5017C-URF.cfm]]'),
	2727 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H72RF+ | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H72RF_.cfm]]'),
	2728 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRT2 | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TRT2.cfm]]'),
	2729 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017R-WRF | http://www.supermicro.com/products/system/1u/5017/SYS-5017R-WRF.cfm]]'),
	2730 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018TR-TF | http://www.supermicro.com/products/system/1u/6018/SYS-6018TR-TF.cfm]]'),
	2731 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044H-32R | http://www.supermicro.com/products/system/4U/7044/SYS-7044H-32R.cfm]]'),
	2732 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5026TI-HTRF | http://www.supermicro.com/products/system/2u/5026/sys-5026ti-htrf.cfm]]'),
	2733 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-BIBQRF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-BIBQRF.cfm]]'),
	2734 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038A-I | http://www.supermicro.com/products/system/tower/5038/sys-5038a-i.cfm]]'),
	2735 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DNCTR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DNCTR.cfm]]'),
	2736 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-MR | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-MR.cfm]]'),
	2737 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1025C-3 | http://www.supermicro.com/products/system/1U/1025/SYS-1025C-3.cfm]]'),
	2738 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018A-TN4 | http://www.supermicro.com/products/system/1u/5018/SYS-5018A-TN4.cfm]]'),
	2739 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-TF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-TF.cfm]]'),
	2740 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H72RF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H72RF.cfm]]'),
	2741 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-N3RF4+ | http://www.supermicro.com/products/system/2u/2027/SYS-2027R-N3RF4_.cfm]]'),
	2742 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6036T-6RF | http://www.supermicro.com/products/system/3U/6036/SYS-6036T-6RF.cfm]]'),
	2743 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WTRFTP | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-WTRFTP.cfm]]'),
	2744 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1016T-M3F | http://www.supermicro.com/products/system/1u/1016/sys-1016t-m3f.cfm]]'),
	2745 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-TRT+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-TRT_.cfm]]'),
	2746 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6034H-X8R | http://www.supermicro.com/products/system/3U/6034/SYS-6034H-X8R.cfm]]'),
	2747 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-73DARF | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-73DARF.cfm]]'),
	2748 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-MR+ | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-MR_.cfm]]'),
	2749 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028GQ-TR | http://www.supermicro.com/products/system/1u/1028/sys-1028gq-tr.cfm]]'),
	2750 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5013C-M8 | http://www.supermicro.com/products/system/1U/5013/SYS-5013C-M8.cfm]]'),
	2751 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025W-NTR+ | http://www.supermicro.com/products/system/2U/6025/SYS-6025W-NTR_.cfm]]'),
	2752 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017R-MTF | http://www.supermicro.com/products/system/1u/5017/SYS-5017R-MTF.cfm]]'),
	2753 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-NTF | http://www.supermicro.com/products/system/1u/6016/SYS-6016T-NTF.cfm]]'),
	2754 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-R72B+ | http://www.supermicro.com/products/system/4u/F628/SYS-F628R3-R72B_.cfm]]'),
	2755 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HC0TR | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HC0TR.cfm]]'),
	2756 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015V-MR | http://www.supermicro.com/products/system/1U/6015/SYS-6015V-MR.cfm]]'),
	2757 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015V-MT | http://www.supermicro.com/products/system/1U/6015/SYS-6015V-MT.cfm]]'),
	2758 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017B-URF | http://www.supermicro.com/products/system/1u/6017/SYS-6017B-URF.cfm]]'),
	2759 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%7037A-I | http://www.supermicro.com/products/system/tower/7037/SYS-7037A-i.cfm]]'),
	2760 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016T-T | http://www.supermicro.com/products/system/1U/5016/SYS-5016T-T.cfm]]'),
	2761 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-H6RF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-H6RF.cfm]]'),
	2762 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-NTRF | http://www.supermicro.com/products/system/1u/6016/SYS-6016T-NTRF.cfm]]'),
	2763 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HTR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-HTR.cfm]]'),
	2764 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5028L-TN2 | http://www.supermicro.com/products/system/midtower/5028/SYS-5028L-TN2.cfm]]'),
	2765 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro Tower%GPASS%8047R-7RFT+ | http://www.supermicro.com/products/system/4u/8047/SYS-8047R-7RFT_.cfm]]'),
	2766 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044A-82 | http://www.supermicro.com/products/system/4U/7044/SYS-7044A-82.cfm]]'),
	2767 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016T-MR | http://www.supermicro.com/products/system/1u/5016/sys-5016t-mr.cfm]]'),
	2768 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028GQ-TXR | http://www.supermicro.com/products/system/1U/1028/SYS-1028GQ-TXR.cfm]]'),
	2769 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8028B-TR4F | http://www.supermicro.com/products/system/2U/8028/SYS-8028B-TR4F.cfm]]'),
	2770 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038MD-H8TRF | http://www.supermicro.com/products/system/3U/5038/SYS-5038MD-H8TRF.cfm]]'),
	2771 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015TW-INF | http://www.supermicro.com/products/system/1U/6015/SYS-6015TW-INF.cfm]]'),
	2772 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026T-TF | http://www.supermicro.com/products/system/2U/6026/SYS-6026T-TF.cfm]]'),
	2773 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R2-R72+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617R2-R72_.cfm]]'),
	2774 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047R-TXRF | http://www.supermicro.com/products/system/4u/7047/SYS-7047R-TXRF.cfm]]'),
	2775 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6022L-6 | http://www.supermicro.com/products/system/2U/6022/SYS-6022L-6.cfm]]'),
	2776 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015V-M3 | http://www.supermicro.com/products/system/1U/6015/SYS-6015V-M3.cfm]]'),
	2777 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1018GR-T | http://www.supermicro.com/products/system/1u/1018/SYS-1018GR-T.cfm]]'),
	2778 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-D70QRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-D70QRF.cfm]]'),
	2779 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026TT-DLRF | http://www.supermicro.com/products/system/2U/2026/SYS-2026TT-DLRF.cfm]]'),
	2780 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047AX-72RF | http://www.supermicro.com/products/system/4u/7047/SYS-7047AX-72RF.cfm]]'),
	2781 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6033P-8R | http://www.supermicro.com/products/system/3U/6033/SYS-6033P-8R.cfm]]'),
	2782 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6047R-TXRF | http://www.supermicro.com/products/system/4u/6047/SYS-6047R-TXRF.cfm]]'),
	2783 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017C-LF | http://www.supermicro.com/products/system/1u/5017/sys-5017c-lf.cfm]]'),
	2784 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017R-IHDP | http://www.supermicro.com/products/system/1U/5017/SSG-5017R-iHDP.cfm]]'),
	2785 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%7038A-I | http://www.supermicro.com/products/system/tower/7038/SYS-7038A-i.cfm]]'),
	2786 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017P-TF | http://www.supermicro.com/products/system/1u/5017/SYS-5017P-TF.cfm]]'),
	2787 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048R-TRT | http://www.supermicro.com/products/system/4U/7048/SYS-7048R-TRT.cfm]]'),
	2788 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028UT-BC1NRT | http://www.supermicro.com/products/system/2u/2028/SYS-2028UT-BC1NRT.cfm]]'),
	2789 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WC1NRT | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-WC1NRT.cfm]]'),
	2790 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028UX-TR4 | http://www.supermicro.com/products/system/2u/6028/SYS-6028UX-TR4.cfm]]'),
	2791 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D71QRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D71QRF.cfm]]'),
	2792 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-HTQRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-HTQRF.cfm]]'),
	2793 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046T-6F | http://www.supermicro.com/products/system/4U/7046/SYS-7046T-6F.cfm]]'),
	2794 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6048R-TXR | http://www.supermicro.com/products/system/4U/6048/SYS-6048R-TXR.cfm]]'),
	2795 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H71RF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-H71RF.cfm]]'),
	2796 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044A-32 | http://www.supermicro.com/products/system/4U/7044/SYS-7044A-32.cfm]]'),
	2797 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-TR | http://www.supermicro.com/products/system/2u/6028/SYS-6028R-TR.cfm]]'),
	2798 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RC0PT+ | http://www.supermicro.com/products/system/4u/f618/SYS-F618R2-RC0PT_.cfm]]'),
	2799 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5025M-I+ | http://www.supermicro.com/products/system/2U/5025/SYS-5025M-i_.cfm]]'),
	2800 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-TT | http://www.supermicro.com/products/system/2u/6028/SYS-6028R-TT.cfm]]'),
	2801 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-T+ | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-T_.cfm]]'),
	2802 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC0R-SIOM | http://www.supermicro.com/products/system/2U/6028/SYS-6028TP-HC0R-SIOM.cfm]]'),
	2803 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-E1CR12 | http://www.supermicro.com/products/system/2U/6027/SSG-6027R-E1CR12N.cfm]]'),
	2804 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617H6-FT+ | http://www.supermicro.com/products/system/4u/f617/SYS-F617H6-FT_.cfm]]'),
	2805 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5012B-6 | http://www.supermicro.com/products/system/1U/5012/SYS-5012B-6.cfm]]'),
	2806 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H70RF | http://www.supermicro.com/products/system/2u/6027/SYS-6027TR-H70RF.cfm]]'),
	2807 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-E1CR12 | http://www.supermicro.com/products/system/2u/6028/SSG-6028R-E1CR12N.cfm]]'),
	2808 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R2-FT | http://www.supermicro.com/products/system/4U/F617/SYS-F617R2-FT.cfm]]'),
	2809 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-TRT | http://www.supermicro.com/products/system/2u/6028/SYS-6028R-TRT.cfm]]'),
	2810 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6023P-I | http://www.supermicro.com/products/system/2U/6023/SYS-6023P-i.cfm]]'),
	2811 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G3-F73+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627G3-F73_.cfm]]'),
	2812 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026T-UF | http://www.supermicro.com/products/system/1u/1026/SYS-1026T-UF.cfm]]'),
	2813 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015TB-T | http://www.supermicro.com/products/system/1U/5015/SYS-5015TB-T.cfm]]'),
	2814 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-HTRF+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-HTRF_.cfm]]'),
	2815 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026TT-HIBQRF | http://www.supermicro.com/products/system/2U/2026/SYS-2026TT-HIBQRF.cfm]]'),
	2816 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-M7UF | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-M7UF.cfm]]'),
	2817 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017A-EF | http://www.supermicro.com/products/system/1u/5017/SYS-5017A-EF.cfm]]'),
	2818 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H70FRF | http://www.supermicro.com/products/system/2u/6027/SYS-6027TR-H70FRF.cfm]]'),
	2819 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H70RF+ | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H70RF_.cfm]]'),
	2820 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R2-F73 | http://www.supermicro.com/products/system/4U/F627/SYS-F627R2-F73.cfm]]'),
	2821 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1018D-73MTF | http://www.supermicro.com/products/system/1u/1018/SYS-1018D-73MTF.cfm]]'),
	2822 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DTTR | http://www.supermicro.com/products/system/2u/2027/SYS-2027PR-DTTR.cfm]]'),
	2823 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4028GR-TRT | http://www.supermicro.com/products/system/4u/4028/sys-4028gr-trt.cfm]]'),
	2824 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017A-EP | http://www.supermicro.com/products/system/1u/5017/SYS-5017A-EP.cfm]]'),
	2825 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DC0TR | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DC0TR.cfm]]'),
	2826 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044A-I2 | http://www.supermicro.com/products/system/4U/7044/SYS-7044A-i2.cfm]]'),
	2827 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-32 | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-32.cfm]]'),
	2828 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018A-MHN4 | http://www.supermicro.com/products/system/1u/5018/SYS-5018A-MHN4.cfm]]'),
	2829 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047AX-TRF | http://www.supermicro.com/products/system/4u/7047/SYS-7047AX-TRF.cfm]]'),
	2830 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038AD-T | http://www.supermicro.com/products/system/tower/5038/SYS-5038AD-T.cfm]]'),
	2831 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1018L-MP | http://www.supermicro.com/products/system/mini-itx/1018/SYS-1018L-MP.cfm]]'),
	2832 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017GR-TF-FM275 | http://www.supermicro.com/products/system/1U/5017/SYS-5017GR-TF-FM275.cfm]]'),
	2833 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-E1CR16T | http://www.supermicro.com/products/system/2U/6028/SSG-6028R-E1CR16T.cfm]]'),
	2834 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018R-WR | http://www.supermicro.com/products/system/1u/5018/SYS-5018R-WR.cfm]]'),
	2835 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6046T-TUF | http://www.supermicro.com/products/system/4U/6046/SYS-6046T-TUF.cfm]]'),
	2836 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-DTQRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-DTQRF.cfm]]'),
	2837 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045A-T | http://www.supermicro.com/products/system/4U/7045/SYS-7045A-T.cfm]]'),
	2838 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-73DBRF | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-73DBRF.cfm]]'),
	2839 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-TDF+ | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-TDF_.cfm]]'),
	2840 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-C1RT | http://www.supermicro.com/products/system/2u/2028/SYS-2028R-C1RT.cfm]]'),
	2841 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RC1+ | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-RC1_.cfm]]'),
	2842 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRFHT | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TRFHT.cfm]]'),
	2843 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%7036A-T | http://www.supermicro.com/products/system/tower/7036/SYS-7036A-T.cfm]]'),
	2844 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-NT | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-NT.cfm]]'),
	2845 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026GT-TRF-FM309 | http://www.supermicro.com/products/system/1U/1026/SYS-1026GT-TRF-FM309.cfm]]'),
	2846 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DC1FR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-DC1FR.cfm]]'),
	2847 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-TDT+ | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-TDT_.cfm]]'),
	2848 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HTR | http://www.supermicro.com/products/system/2u/2027/SYS-2027PR-HTR.cfm]]'),
	2849 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046GT-TRF | http://www.supermicro.com/products/system/4U/7046/SYS-7046GT-TRF.cfm]]'),
	2850 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-I2 | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-i2.cfm]]'),
	2851 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026GT-TRF-FM307 | http://www.supermicro.com/products/system/1U/1026/SYS-1026GT-TRF-FM307.cfm]]'),
	2852 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5039D-I | http://www.supermicro.com/products/system/tower/5039/SYS-5039D-I.cfm]]'),
	2853 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-E1R12 | http://www.supermicro.com/products/system/2u/6027/SSG-6027R-E1R12N.cfm]]'),
	2854 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4028GR-TR2 | http://www.supermicro.com/products/system/4u/4028/SYS-4028GR-TR2.cfm]]'),
	2855 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-N3RF | http://www.supermicro.com/products/system/2u/6027/SYS-6027R-N3RF.cfm]]'),
	2856 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-FT | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-FT.cfm]]'),
	2857 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016TT-IBXF | http://www.supermicro.com/products/system/1U/6016/SYS-6016TT-IBXF.cfm]]'),
	2858 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6037R-TXRF | http://www.supermicro.com/products/system/3u/6037/SYS-6037R-TXRF.cfm]]'),
	2859 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028UT-BTNRT | http://www.supermicro.com/products/system/2u/2028/SYS-2028UT-BTNRT.cfm]]'),
	2860 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R2-FT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R2-FT_.cfm]]'),
	2861 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028TR-H72FR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TR-H72FR.cfm]]'),
	2862 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-F72+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R3-F72_.cfm]]'),
	2863 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-HTFRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-HTFRF.cfm]]'),
	2864 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-82R+ | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-82R_.cfm]]'),
	2865 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048R-C1R | http://www.supermicro.com/products/system/4U/7048/SYS-7048R-C1R.cfm]]'),
	2866 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DTFR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-DTFR.cfm]]'),
	2867 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-TR4T+ | http://www.supermicro.com/products/system/1u/1028/SYS-1028U-TR4T_.cfm]]'),
	2868 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H70FRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H70FRF.cfm]]'),
	2869 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-NTF | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-NTF.cfm]]'),
	2870 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045A-WT | http://www.supermicro.com/products/system/4U/7045/SYS-7045A-WT.cfm]]'),
	2871 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-TD8 | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-TD8.cfm]]'),
	2872 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-FT+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618R2-FT_.cfm]]'),
	2873 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6036T-TF | http://www.supermicro.com/products/system/3U/6036/SYS-6036T-TF.cfm]]'),
	2874 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017B-NTF | http://www.supermicro.com/products/system/1u/6017/SYS-6017B-NTF.cfm]]'),
	2875 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4028GR-TR | http://www.supermicro.com/products/system/4u/4028/sys-4028gr-tr.cfm]]'),
	2876 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5029A-2TN4 | http://www.supermicro.com/products/system/midtower/5029/SYS-5029A-2TN4.cfm]]'),
	2877 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038AD-I | http://www.supermicro.com/products/system/tower/5038/SYS-5038AD-I.cfm]]'),
	2878 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DC1TR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-DC1TR.cfm]]'),
	2879 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026T-URF | http://www.supermicro.com/products/system/2u/6026/SYS-6026T-URF.cfm]]'),
	2880 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5013G-6 | http://www.supermicro.com/products/system/1U/5013/SYS-5013G-6.cfm]]'),
	2881 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WTNRT | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WTNRT.cfm]]'),
	2882 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DC0FR | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DC0FR.cfm]]'),
	2883 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TSF | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TSF.cfm]]'),
	2884 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HTTR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HTTR.cfm]]'),
	2885 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015C-NTR | http://www.supermicro.com/products/system/1U/6015/SYS-6015C-NTR.cfm]]'),
	2886 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028TR-D72R | http://www.supermicro.com/products/system/2u/6028/SYS-6028TR-D72R.cfm]]'),
	2887 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045A-8 | http://www.supermicro.com/products/system/4U/7045/SYS-7045A-8.cfm]]'),
	2888 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044H-X8R | http://www.supermicro.com/products/system/4U/7044/SYS-7044H-X8R.cfm]]'),
	2889 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017B-MTRF | http://www.supermicro.com/products/system/1u/6017/SYS-6017B-MTRF.cfm]]'),
	2890 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618H6-FTPTL+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618H6-FTPTL_.cfm]]'),
	2891 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-TNR4T+ | http://www.supermicro.com/products/system/1u/1028/SYS-1028U-TNR4T_.cfm]]'),
	2892 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045A-3 | http://www.supermicro.com/products/system/4U/7045/SYS-7045A-3.cfm]]'),
	2893 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HTTR | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HTTR.cfm]]'),
	2894 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017P-TLN4F | http://www.supermicro.com/products/system/1u/5017/sys-5017p-tln4f.cfm]]'),
	2895 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-TDLRF | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-TDLRF.cfm]]'),
	2896 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047GR-TPRF-FM409 | http://www.supermicro.com/products/system/4U/7047/SYS-7047GR-TPRF-FM409.cfm]]'),
	2897 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R3-FT | http://www.supermicro.com/products/system/4u/F618/SYS-F618R3-FT.cfm]]'),
	2898 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HTFR | http://www.supermicro.com/products/system/2U/6028/sys-6028tp-htfr.cfm]]'),
	2899 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5035L-I | http://www.supermicro.com/products/system/tower/5035/SYS-5035L-I.cfm]]'),
	2900 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025B-3R | http://www.supermicro.com/products/system/2U/6025/SYS-6025B-3R.cfm]]'),
	2901 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016T-MTF | http://www.supermicro.com/products/system/1u/5016/sys-5016t-mtf.cfm]]'),
	2902 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WC1RT | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-WC1RT.cfm]]'),
	2903 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017GR-TF | http://www.supermicro.com/products/system/1u/1017/SYS-1017GR-TF.cfm]]'),
	2904 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045B-T | http://www.supermicro.com/products/system/4U/7045/SYS-7045B-T.cfm]]'),
	2905 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6037R-72RFT | http://www.supermicro.com/products/system/3U/6037/SYS-6037R-72RFT.cfm]]'),
	2906 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017GR-TF-FM209 | http://www.supermicro.com/products/system/1U/5017/SYS-5017GR-TF-FM209.cfm]]'),
	2907 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5013G-M | http://www.supermicro.com/products/system/1U/5013/SYS-5013G-M.cfm]]'),
	2908 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6013L-8 | http://www.supermicro.com/products/system/1U/6013/SYS-6013L-8.cfm]]'),
	2909 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5013G-I | http://www.supermicro.com/products/system/1U/5013/SYS-5013G-i.cfm]]'),
	2910 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046T-3R | http://www.supermicro.com/products/system/4U/7046/SYS-7046T-3R.cfm]]'),
	2911 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8028B-C0R4FT | http://www.supermicro.com/products/system/2U/8028/SYS-8028B-C0R4FT.cfm]]'),
	2912 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DTR | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DTR.cfm]]'),
	2913 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027TR-TFF | http://www.supermicro.com/products/system/1u/1027/SYS-1027TR-TFF.cfm]]'),
	2914 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6013P-8+ | http://www.supermicro.com/products/system/1U/6013/SYS-6013P-8_.cfm]]'),
	2915 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-E1CR24 | http://www.supermicro.com/products/system/2U/2027/SSG-2027R-E1CR24N.cfm]]'),
	2916 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017B-MTLF | http://www.supermicro.com/products/system/1u/6017/SYS-6017B-MTLF.cfm]]'),
	2917 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048GR-TR | http://www.supermicro.com/products/system/4u/7048/SYS-7048GR-TR.cfm]]'),
	2918 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5025M-UR | http://www.supermicro.com/products/system/2U/5025/SYS-5025M-UR.cfm]]'),
	2919 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017C-TF | http://www.supermicro.com/products/system/1u/1017/SYS-1017C-TF.cfm]]'),
	2920 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-MT | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-MT.cfm]]'),
	2921 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015C-MT | http://www.supermicro.com/products/system/1U/6015/SYS-6015C-MT.cfm]]'),
	2922 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026TT-H6IBXRF | http://www.supermicro.com/products/system/2U/2026/SYS-2026TT-H6IBXRF.cfm]]'),
	2923 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-TR | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-TR.cfm]]'),
	2924 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038MR-H8TRF | http://www.supermicro.com/products/system/3U/5038/SYS-5038MR-H8TRF.cfm]]'),
	2925 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-TNRT+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-TNRT_.cfm]]'),
	2926 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018A-LTN4 | http://www.supermicro.com/products/system/1U/5018/SYS-5018A-LTN4.cfm]]'),
	2927 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RTBPTN+ | http://www.supermicro.com/products/system/4u/F628/SYS-F628R3-RTBPTN_.cfm]]'),
	2928 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-LN4T | http://www.supermicro.com/products/system/1u/5018/SYS-5018D-LN4T.cfm]]'),
	2929 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-TDAF | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-TDAF.cfm]]'),
	2930 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R3-FT | http://www.supermicro.com/products/system/4u/f617/SYS-F617R3-FT.cfm]]'),
	2931 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-TNRT+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-TNRT_.cfm]]'),
	2932 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045B-3 | http://www.supermicro.com/products/system/4U/7045/SYS-7045B-3.cfm]]'),
	2933 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC1TR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HC1TR.cfm]]'),
	2934 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015W-NTR | http://www.supermicro.com/products/system/1U/6015/SYS-6015W-NTR.cfm]]'),
	2935 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015V-TLP | http://www.supermicro.com/products/system/1U/6015/SYS-6015V-TLP.cfm]]'),
	2936 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044H-82R | http://www.supermicro.com/products/system/4U/7044/SYS-7044H-82R.cfm]]'),
	2937 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DNCR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DNCR.cfm]]'),
	2938 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-E1CR12H | http://www.supermicro.com/products/system/2u/6028/SSG-6028R-E1CR12H.cfm]]'),
	2939 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-E1CR24 | http://www.supermicro.com/products/system/2U/6028/SSG-6028R-E1CR24N.cfm]]'),
	2940 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-E1R24 | http://www.supermicro.com/products/system/2U/2027/SSG-2027R-E1R24N.cfm]]'),
	2941 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HTTR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-HTTR.cfm]]'),
	2942 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-WTRT | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-WTRT.cfm]]'),
	2943 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-WTR | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-WTR.cfm]]'),
	2944 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-D70RF+ | http://www.supermicro.com/products/system/2u/2027/SYS-2027TR-D70RF_.cfm]]'),
	2945 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DECR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DECR.cfm]]'),
	2946 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-TRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-TRF.cfm]]'),
	2947 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-HTRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-HTRF.cfm]]'),
	2948 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018GR-T | http://www.supermicro.com/products/system/1u/5018/SYS-5018GR-T.cfm]]'),
	2949 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026T-M3 | http://www.supermicro.com/products/system/1U/1026/SYS-1026T-M3.cfm]]'),
	2950 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038K-I | http://www.supermicro.com/products/system/tower/5038/SYS-5038K-I.cfm]]'),
	2951 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HTFR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-HTFR.cfm]]'),
	2952 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-NTR | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-NTR.cfm]]'),
	2953 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015W-NI | http://www.supermicro.com/products/system/1U/6015/SYS-6015W-Ni.cfm]]'),
	2954 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618H6-FT+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618H6-FT_.cfm]]'),
	2955 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8028B-TR3F | http://www.supermicro.com/products/system/2u/8028/SYS-8028B-TR3F.cfm]]'),
	2956 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-FC0PT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R3-FC0PT_.cfm]]'),
	2957 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037MC-H12TRF | http://www.supermicro.com/products/system/3U/5037/SYS-5037MC-H12TRF.cfm]]'),
	2958 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRFH-FM609 | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TRFH-FM609.cfm]]'),
	2959 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6037R-E1R16 | http://www.supermicro.com/products/system/3U/6037/SSG-6037R-E1R16N.cfm]]'),
	2960 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-TN24R4T+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-TN24R4T_.cfm]]'),
	2961 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H70QRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H70QRF.cfm]]'),
	2962 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R2-RT+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617R2-RT_.cfm]]'),
	2963 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-C1RT4+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028R-C1RT4_.cfm]]'),
	2964 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-URF4+ | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-URF4_.cfm]]'),
	2965 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046A-HR+ | http://www.supermicro.com/products/system/4U/7046/SYS-7046A-HR_.cfm]]'),
	2966 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037MR-H8TRF | http://www.supermicro.com/products/system/3u/5037/SYS-5037MR-H8TRF.cfm]]'),
	2967 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015X-T | http://www.supermicro.com/products/system/1U/6015/SYS-6015X-T.cfm]]'),
	2968 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-WTRT | http://www.supermicro.com/products/system/2u/6028/SYS-6028R-WTRT.cfm]]'),
	2969 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-72RFTP | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-72RFTP.cfm]]'),
	2970 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037MC-H86RF | http://www.supermicro.com/products/system/3U/5037/SYS-5037MC-H86RF.cfm]]'),
	2971 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017GR-TF-FM175 | http://www.supermicro.com/products/system/1U/1017/SYS-1017GR-TF-FM175.cfm]]'),
	2972 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014L-M | http://www.supermicro.com/products/system/1U/6014/SYS-6014L-M.cfm]]'),
	2973 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-AR24 | http://www.supermicro.com/products/system/2U/2027/SSG-2027R-AR24.cfm]]'),
	2974 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-D70RF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-D70RF.cfm]]'),
	2975 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D71RF+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D71RF_.cfm]]'),
	2976 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRFH-FM675 | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TRFH-FM675.cfm]]'),
	2977 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R3-FT+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618R3-FT_.cfm]]'),
	2978 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-TR4T+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-TR4T_.cfm]]'),
	2979 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028GR-TR | http://www.supermicro.com/products/system/2u/2028/SYS-2028GR-TR.cfm]]'),
	2980 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-TDTP | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-TDTP.cfm]]'),
	2981 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-WTR | http://www.supermicro.com/products/system/2u/6028/SYS-6028R-WTR.cfm]]'),
	2982 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC0FR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-HC0FR.cfm]]'),
	2983 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC1FR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HC1FR.cfm]]'),
	2984 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027B-URF | http://www.supermicro.com/products/system/2u/6027/SYS-6027B-URF.cfm]]'),
	2985 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R2-RTB+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R2-RTB_.cfm]]'),
	2986 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TRF-FM375 | http://www.supermicro.com/products/system/1U/1027/SYS-1027GR-TRF-FM375.cfm]]'),
	2987 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046T-H6R | http://www.supermicro.com/products/system/4U/7046/SYS-7046T-H6R.cfm]]'),
	2988 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045A-C3 | http://www.supermicro.com/products/system/4U/7045/SYS-7045A-C3.cfm]]'),
	2989 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RC1B+ | http://www.supermicro.com/products/system/4u/F628/SYS-F628R3-RC1B_.cfm]]'),
	2990 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048R-C1RT4+ | http://www.supermicro.com/products/system/4u/7048/SYS-7048R-C1RT4_.cfm]]'),
	2991 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-UR | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-UR.cfm]]'),
	2992 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015TC-LFT | http://www.supermicro.com/products/system/1U/6015/SYS-6015TC-LFT.cfm]]'),
	2993 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-FC0 | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-FC0.cfm]]'),
	2994 => array ('chapter_id' => 18, 'dict_value' => '[[SuperMicro%GPASS%937R-E2CJB | http://www.supermicro.com/products/system/3U/937/SSG-937R-E2CJB.cfm]]'),
	2995 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015C-NI | http://www.supermicro.com/products/system/1U/6015/SYS-6015C-Ni.cfm]]'),
	2996 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6038R-E1CR16 | http://www.supermicro.com/products/system/3u/6038/SSG-6038R-E1CR16N.cfm]]'),
	2997 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-TDT+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-TDT_.cfm]]'),
	2998 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-E1CNR4T+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-E1CNR4T_.cfm]]'),
	2999 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017B-MTF | http://www.supermicro.com/products/system/1u/6017/SYS-6017B-MTF.cfm]]'),
	3000 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%8048B-TRFT | http://www.supermicro.com/products/system/4u/8048/SYS-8048B-TRFT.cfm]]'),
	3001 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-WTRFTP | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-WTRFTP.cfm]]'),
	3002 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014L-M4 | http://www.supermicro.com/products/system/1U/6014/SYS-6014L-M4.cfm]]'),
	3003 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DC1R | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DC1R.cfm]]'),
	3004 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028BT-HNC0R+ | http://www.supermicro.com/products/system/2U/2028/SYS-2028BT-HNC0R_.cfm]]'),
	3005 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5013C-M | http://www.supermicro.com/products/system/1U/5013/SYS-5013C-M.cfm]]'),
	3006 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028TR-HTFR | http://www.supermicro.com/products/system/2U/2028/SYS-2028TR-HTFR.cfm]]'),
	3007 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DNCFR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DNCFR.cfm]]'),
	3008 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038D-I | http://www.supermicro.com/products/system/tower/5038/sys-5038d-i.cfm]]'),
	3009 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5039MS-H8TRF | http://www.supermicro.com/products/system/3U/5039/SYS-5039MS-H8TRF.cfm]]'),
	3010 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-MHR7N4P | http://www.supermicro.com/products/system/1U/5018/SYS-5018D-MHR7N4P.cfm]]'),
	3011 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5026T-T | http://www.supermicro.com/products/system/2U/5026/SYS-5026T-T.cfm]]'),
	3012 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DECTR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DECTR.cfm]]'),
	3013 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HC1FR | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HC1FR.cfm]]'),
	3014 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DC0R | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-DC0R.cfm]]'),
	3015 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026T-6RF+ | http://www.supermicro.com/products/system/1U/1026/SYS-1026T-6RF_.cfm]]'),
	3016 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-TNRT+ | http://www.supermicro.com/products/system/1u/1028/SYS-1028U-TNRT_.cfm]]'),
	3017 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H70RF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H70RF.cfm]]'),
	3018 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-E1CR48 | http://www.supermicro.com/products/system/2U/2028/SSG-2028R-E1CR48N.cfm]]'),
	3019 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046A-3 | http://www.supermicro.com/products/system/4U/7046/SYS-7046A-3.cfm]]'),
	3020 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-DTRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-DTRF.cfm]]'),
	3021 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028TR-HTR | http://www.supermicro.com/products/system/2U/2028/SYS-2028TR-HTR.cfm]]'),
	3022 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DC1R | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-DC1R.cfm]]'),
	3023 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-32 | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-32.cfm]]'),
	3024 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R2-FC0 | http://www.supermicro.com/products/system/4u/F628/SYS-F628R2-FC0.cfm]]'),
	3025 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-TXR | http://www.supermicro.com/products/system/2U/2028/SYS-2028R-TXR.cfm]]'),
	3026 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HTR-SIOM | http://www.supermicro.com/products/system/2U/6028/SYS-6028TP-HTR-SIOM.cfm]]'),
	3027 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045A-CT | http://www.supermicro.com/products/system/4U/7045/SYS-7045A-CT.cfm]]'),
	3028 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-NTR | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-NTR.cfm]]'),
	3029 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-URF | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-URF.cfm]]'),
	3030 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1016I-M6F | http://www.supermicro.com/products/system/1u/1016/sys-1016i-m6f.cfm]]'),
	3031 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-DTFRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-DTFRF.cfm]]'),
	3032 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6023P-8 | http://www.supermicro.com/products/system/2U/6023/SYS-6023P-8.cfm]]'),
	3033 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RTB+ | http://www.supermicro.com/products/system/4u/f628/SYS-F628R3-RTB_.cfm]]'),
	3034 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-DNCFR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-DNCFR.cfm]]'),
	3035 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DECFR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DECFR.cfm]]'),
	3036 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-XI | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-Xi.cfm]]'),
	3037 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DTFR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DTFR.cfm]]'),
	3038 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018U-TR4T+ | http://www.supermicro.com/products/system/1U/6018/SYS-6018U-TR4T_.cfm]]'),
	3039 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1025TC-T | http://www.supermicro.com/products/system/1U/1025/SYS-1025TC-T.cfm]]'),
	3040 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028UX-CR-LL2 | http://www.supermicro.com/products/system/1u/1028/SYS-1028UX-CR-LL2.cfm]]'),
	3041 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R3-FT+ | http://www.supermicro.com/products/system/4u/f617/SYS-F617R3-FT_.cfm]]'),
	3042 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TQFT | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TQFT.cfm]]'),
	3043 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028UX-CR-LL1 | http://www.supermicro.com/products/system/1u/1028/SYS-1028UX-CR-LL1.cfm]]'),
	3044 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047R-3RF4+ | http://www.supermicro.com/products/system/4U/7047/SYS-7047R-3RF4_.cfm]]'),
	3045 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F647G2-F73PT+ | http://www.supermicro.com/products/system/4U/F647/SYS-F647G2-F73PT_.cfm]]'),
	3046 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-WRF | http://www.supermicro.com/products/system/2u/6027/SYS-6027R-WRF.cfm]]'),
	3047 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-32R | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-32R.cfm]]'),
	3048 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-FT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R3-FT_.cfm]]'),
	3049 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-FN4T | http://www.supermicro.com/products/system/1u/5018/SYS-5018D-FN4T.cfm]]'),
	3050 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5019S-TN4 | http://www.supermicro.com/products/system/1U/5019/SYS-5019S-TN4.cfm]]'),
	3051 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R2-FTPT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R2-FTPT_.cfm]]'),
	3052 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-MTR | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-MTR.cfm]]'),
	3053 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-E1R12T | http://www.supermicro.com/products/system/2U/6027/SSG-6027R-E1R12T.cfm]]'),
	3054 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015TB-10G | http://www.supermicro.com/products/system/1U/5015/SYS-5015TB-10G.cfm]]'),
	3055 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-3RF4+ | http://www.supermicro.com/products/system/2u/6027/SYS-6027R-3RF4_.cfm]]'),
	3056 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017R-MTRF | http://www.supermicro.com/products/system/1u/5017/SYS-5017R-MTRF.cfm]]'),
	3057 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044A-82R | http://www.supermicro.com/products/system/4U/7044/SYS-7044A-82R.cfm]]'),
	3058 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F648G2-FT+ | http://www.supermicro.com/products/system/4U/F648/SYS-F648G2-FT_.cfm]]'),
	3059 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D71FRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D71FRF.cfm]]'),
	3060 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026TT-H6RF | http://www.supermicro.com/products/system/2U/2026/SYS-2026TT-H6RF.cfm]]'),
	3061 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016TT-TF | http://www.supermicro.com/products/system/1U/6016/SYS-6016TT-TF.cfm]]'),
	3062 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DC1R | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DC1R.cfm]]'),
	3063 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-TDW | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-TDW.cfm]]'),
	3064 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G2-FC0PT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G2-FC0PT_.cfm]]'),
	3065 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1016I-U | http://www.supermicro.com/products/system/1u/1016/sys-1016i-u.cfm]]'),
	3066 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8027R-TRF+ | http://www.supermicro.com/products/system/2u/8027/SYS-8027R-TRF_.cfm]]'),
	3067 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5036T-T | http://www.supermicro.com/products/system/midtower/5036/SYS-5036T-T.cfm]]'),
	3068 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044H-82 | http://www.supermicro.com/products/system/4U/7044/SYS-7044H-82.cfm]]'),
	3069 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC0R | http://www.supermicro.com/products/system/2U/2028/SYS-2028TP-HC0R.cfm]]'),
	3070 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-MTLF | http://www.supermicro.com/products/system/1u/6016/SYS-6016T-MTLF.cfm]]'),
	3071 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046T-NTR+ | http://www.supermicro.com/products/system/4U/7046/SYS-7046T-NTR_.cfm]]'),
	3072 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047R-72RFT | http://www.supermicro.com/products/system/4U/7047/SYS-7047R-72RFT.cfm]]'),
	3073 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1015B-3 | http://www.supermicro.com/products/system/1U/1015/SYS-1015B-3.cfm]]'),
	3074 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-X8 | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-X8.cfm]]'),
	3075 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017R-MF | http://www.supermicro.com/products/system/1u/5017/SYS-5017R-MF.cfm]]'),
	3076 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8026B-6R | http://www.supermicro.com/products/system/2U/8026/SYS-8026B-6R.cfm]]'),
	3077 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G3-FC0PT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G3-FC0PT_.cfm]]'),
	3078 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018U-TRTP+ | http://www.supermicro.com/products/system/1u/6018/SYS-6018U-TRTP_.cfm]]'),
	3079 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026TT-IBXF | http://www.supermicro.com/products/system/1U/1026/SYS-1026TT-IBXF.cfm]]'),
	3080 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028TR-HTR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TR-HTR.cfm]]'),
	3081 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017C-MTRF | http://www.supermicro.com/products/system/1u/5017/sys-5017c-mtrf.cfm]]'),
	3082 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-BTF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-BTF.cfm]]'),
	3083 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro Tower%GPASS%8047R-7JRFT | http://www.supermicro.com/products/system/4u/8047/SYS-8047R-7JRFT.cfm]]'),
	3084 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015C-M3 | http://www.supermicro.com/products/system/1U/6015/SYS-6015C-M3.cfm]]'),
	3085 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-72RFT+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-72RFT_.cfm]]'),
	3086 => array ('chapter_id' => 18, 'dict_value' => '[[SuperMicro%GPASS%947R-E2CJB | http://www.supermicro.com/products/system/4U/947/SSG-947R-E2CJB.cfm]]'),
	3087 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-MTLN4F | http://www.supermicro.com/products/system/1u/5018/SYS-5018D-MTLN4F.cfm]]'),
	3088 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TQF-FM409 | http://www.supermicro.com/products/system/1U/1027/SYS-1027GR-TQF-FM409.cfm]]'),
	3089 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6048R-E1CR24 | http://www.supermicro.com/products/system/4u/6048/SSG-6048R-E1CR24N.cfm]]'),
	3090 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-TDWNR | http://www.supermicro.com/products/system/2u/6028/SYS-6028R-TDWNR.cfm]]'),
	3091 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RC0+ | http://www.supermicro.com/products/system/4u/f618/SYS-F618R2-RC0_.cfm]]'),
	3092 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015T-INF | http://www.supermicro.com/products/system/1U/6015/SYS-6015T-INF.cfm]]'),
	3093 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC1R-SIOM | http://www.supermicro.com/products/system/2U/2028/SYS-2028TP-HC1R-SIOM.cfm]]'),
	3094 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6035B-8R | http://www.supermicro.com/products/system/3U/6035/SYS-6035B-8R.cfm]]'),
	3095 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5028D-TN4T | http://www.supermicro.com/products/system/midtower/5028/sys-5028d-tn4t.cfm]]'),
	3096 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WC1R | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WC1R.cfm]]'),
	3097 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5019S-M | http://www.supermicro.com/products/system/1U/5019/SYS-5019S-M.cfm]]'),
	3098 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028UX-LL3-B8 | http://www.supermicro.com/products/system/1U/1028/SYS-1028UX-LL3-B8.cfm]]'),
	3099 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-TNR4T+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-TNR4T_.cfm]]'),
	3100 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G2-FT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G2-FT_.cfm]]'),
	3101 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-FN8T | http://www.supermicro.com/products/system/1u/5018/SYS-5018D-FN8T.cfm]]'),
	3102 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1019S-MP | http://www.supermicro.com/products/system/Mini-ITX/1019/SYS-1019S-MP.cfm]]'),
	3103 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617H6-FTL+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617H6-FTL_.cfm]]'),
	3104 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028TR-H72R | http://www.supermicro.com/products/system/2u/2028/SYS-2028TR-H72R.cfm]]'),
	3105 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRF-FM409 | http://www.supermicro.com/products/system/2U/2027/SYS-2027GR-TRF-FM409.cfm]]'),
	3106 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H71QRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-H71QRF.cfm]]'),
	3107 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H71QRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H71QRF.cfm]]'),
	3108 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DTFR | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DTFR.cfm]]'),
	3109 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015TC-10G | http://www.supermicro.com/products/system/1U/6015/SYS-6015TC-10G.cfm]]'),
	3110 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038MA-H24TRF | http://www.supermicro.com/products/system/3U/5038/SYS-5038MA-H24TRF.cfm]]'),
	3111 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5025M-I | http://www.supermicro.com/products/system/2U/5025/SYS-5025M-i.cfm]]'),
	3112 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HC0TR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HC0TR.cfm]]'),
	3113 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D70RF+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D70RF_.cfm]]'),
	3114 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014V-T2 | http://www.supermicro.com/products/system/1U/6014/SYS-6014V-T2.cfm]]'),
	3115 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-E1CR4+ | http://www.supermicro.com/products/system/1U/1028/SYS-1028U-E1CR4_.cfm]]'),
	3116 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5025M-4 | http://www.supermicro.com/products/system/2U/5025/SYS-5025M-4.cfm]]'),
	3117 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-R72PT+ | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-R72PT_.cfm]]'),
	3118 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-NT | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-NT.cfm]]'),
	3119 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044H-82R+ | http://www.supermicro.com/products/system/4U/7044/SYS-7044H-82R_.cfm]]'),
	3120 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-NI | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-Ni.cfm]]'),
	3121 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6037R-72RF | http://www.supermicro.com/products/system/3U/6037/SYS-6037R-72RF.cfm]]'),
	3122 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026GT-TRF | http://www.supermicro.com/products/system/2U/2026/SYS-2026GT-TRF.cfm]]'),
	3123 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5019S-WR | http://www.supermicro.com/products/system/1U/5019/SYS-5019S-WR.cfm]]'),
	3124 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-F73 | http://www.supermicro.com/products/system/4U/F627/SYS-F627R3-F73.cfm]]'),
	3125 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017GR-TF-FM175 | http://www.supermicro.com/products/system/1U/5017/SYS-5017GR-TF-FM175.cfm]]'),
	3126 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WRFT+ | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-WRFT_.cfm]]'),
	3127 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038MD-H24TRF | http://www.supermicro.com/products/system/3U/5038/SYS-5038MD-H24TRF.cfm]]'),
	3128 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RC1BPT+ | http://www.supermicro.com/products/system/4u/F628/SYS-F628R3-RC1BPT_.cfm]]'),
	3129 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RTBPT+ | http://www.supermicro.com/products/system/4u/F628/SYS-F628R3-RTBPT_.cfm]]'),
	3130 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TRT2 | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TRT2.cfm]]'),
	3131 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RT+ | http://www.supermicro.com/products/system/4u/f618/SYS-F618R2-RT_.cfm]]'),
	3132 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026GT-TF | http://www.supermicro.com/products/system/1U/1026/SYS-1026GT-TF.cfm]]'),
	3133 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017K-N6 | http://www.supermicro.com/products/system/1U/5017/SYS-5017K-N6.cfm]]'),
	3134 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-DNCTR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-DNCTR.cfm]]'),
	3135 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-T+ | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-T_.cfm]]'),
	3136 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TQF-FM475 | http://www.supermicro.com/products/system/1U/1027/SYS-1027GR-TQF-FM475.cfm]]'),
	3137 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC0FR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HC0FR.cfm]]'),
	3138 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016I-MTHF | http://www.supermicro.com/products/system/1u/5016/sys-5016i-mthf.cfm]]'),
	3139 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-HTFRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-HTFRF.cfm]]'),
	3140 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-DNCR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-DNCR.cfm]]'),
	3141 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TRF+ | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TRF_.cfm]]'),
	3142 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%8016B-6 | http://www.supermicro.com/products/system/1U/8016/SYS-8016B-6.cfm]]'),
	3143 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC0TR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HC0TR.cfm]]'),
	3144 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-I | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-i.cfm]]'),
	3145 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-T | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-T.cfm]]'),
	3146 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-TRTP+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-TRTP_.cfm]]'),
	3147 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-R72+ | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-R72_.cfm]]'),
	3148 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017GR-TF-FM209 | http://www.supermicro.com/products/system/1U/1017/SYS-1017GR-TF-FM209.cfm]]'),
	3149 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8027R-7RFT+ | http://www.supermicro.com/products/system/2U/8027/SYS-8027R-7RFT_.cfm]]'),
	3150 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-T | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-T.cfm]]'),
	3151 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016TT-IBQF | http://www.supermicro.com/products/system/1U/6016/SYS-6016TT-IBQF.cfm]]'),
	3152 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014A-8 | http://www.supermicro.com/products/system/1U/6014/SYS-6014A-8.cfm]]'),
	3153 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-HTQRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-HTQRF.cfm]]'),
	3154 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5039MS-H12TRF | http://www.supermicro.com/products/system/3U/5039/SYS-5039MS-H12TRF.cfm]]'),
	3155 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-MTF | http://www.supermicro.com/products/system/1U/5018/SYS-5018D-MTF.cfm]]'),
	3156 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014V-M4 | http://www.supermicro.com/products/system/1U/6014/SYS-6014V-M4.cfm]]'),
	3157 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015X-8 | http://www.supermicro.com/products/system/1U/6015/SYS-6015X-8.cfm]]'),
	3158 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026T-6RF+ | http://www.supermicro.com/products/system/2U/2026/SYS-2026T-6RF_.cfm]]'),
	3159 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-T | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-T.cfm]]'),
	3160 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-U | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-U.cfm]]'),
	3161 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4028GR-TRT2 | http://www.supermicro.com/products/system/4u/4028/SYS-4028GR-TRT2.cfm]]'),
	3162 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F648G2-FTPT+ | http://www.supermicro.com/products/system/4U/F648/SYS-F648G2-FTPT_.cfm]]'),
	3163 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015TW-T | http://www.supermicro.com/products/system/1U/6015/SYS-6015TW-T.cfm]]'),
	3164 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-FTPT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R3-FTPT_.cfm]]'),
	3165 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015X-3 | http://www.supermicro.com/products/system/1U/6015/SYS-6015X-3.cfm]]'),
	3166 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-TR4T+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-TR4T_.cfm]]'),
	3167 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5029S-TN2 | http://www.supermicro.com/products/system/midtower/5029/SYS-5029S-TN2.cfm]]'),
	3168 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G3-FC0+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G3-FC0_.cfm]]'),
	3169 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WC1RT | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WC1RT.cfm]]'),
	3170 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4048B-TRFT | http://www.supermicro.com/products/system/4u/4048/SYS-4048B-TRFT.cfm]]'),
	3171 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G3-F73PT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627G3-F73PT_.cfm]]'),
	3172 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-TR4+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-TR4_.cfm]]'),
	3173 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-82 | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-82.cfm]]'),
	3174 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037C-T | http://www.supermicro.com/products/system/tower/5037/sys-5037c-t.cfm]]'),
	3175 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-TNR4T+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-TNR4T_.cfm]]'),
	3176 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%5046A-X | http://www.supermicro.com/products/system/tower/5046/SYS-5046A-X.cfm]]'),
	3177 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-FT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R3-FT_.cfm]]'),
	3178 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027TR-TQF | http://www.supermicro.com/products/system/1u/1027/SYS-1027TR-TQF.cfm]]'),
	3179 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-3 | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-3.cfm]]'),
	3180 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H71RF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H71RF.cfm]]'),
	3181 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HC1TR | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HC1TR.cfm]]'),
	3182 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R2-FC0+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R2-FC0_.cfm]]'),
	3183 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DTTR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DTTR.cfm]]'),
	3184 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4048B-TR4FT | http://www.supermicro.com/products/system/4u/4048/SYS-4048B-TR4FT.cfm]]'),
	3185 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-8 | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-8.cfm]]'),
	3186 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026T-URF4+ | http://www.supermicro.com/products/system/1U/1026/SYS-1026T-URF4_.cfm]]'),
	3187 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-8 | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-8.cfm]]'),
	3188 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F517H6-FT | http://www.supermicro.com/products/system/4u/f517/SYS-F517H6-FT.cfm]]'),
	3189 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DTTR | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DTTR.cfm]]'),
	3190 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8025C-3R | http://www.supermicro.com/products/system/2U/8025/SYS-8025C-3R.cfm]]'),
	3191 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-GIBQF | http://www.supermicro.com/products/system/1u/6016/SYS-6016T-GIBQF.cfm]]'),
	3192 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618H6-FTL+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618H6-FTL_.cfm]]'),
	3193 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017GR-TF-FM109 | http://www.supermicro.com/products/system/1U/5017/SYS-5017GR-TF-FM109.cfm]]'),
	3194 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DC0FR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DC0FR.cfm]]'),
	3195 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G2-FTPT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627G2-FTPT_.cfm]]'),
	3196 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%8016B-TLF | http://www.supermicro.com/products/system/1U/8016/SYS-8016B-TLF.cfm]]'),
	3197 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014L-T | http://www.supermicro.com/products/system/1U/6014/SYS-6014L-T.cfm]]'),
	3198 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-TRT+ | http://www.supermicro.com/products/system/1u/1028/SYS-1028U-TRT_.cfm]]'),
	3199 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RTN+ | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-RTN_.cfm]]'),
	3200 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TRFT | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TRFT.cfm]]'),
	3201 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014H-8 | http://www.supermicro.com/products/system/1U/6014/SYS-6014H-8.cfm]]'),
	3202 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2015TA-HTRF | http://www.supermicro.com/products/system/2U/2015/SYS-2015TA-HTRF.cfm]]'),
	3203 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HTR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HTR.cfm]]'),
	3204 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028GR-TRT | http://www.supermicro.com/products/system/2u/2028/SYS-2028GR-TRT.cfm]]'),
	3205 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-82 | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-82.cfm]]'),
	3206 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-MF | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-MF.cfm]]'),
	3207 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015P-8 | http://www.supermicro.com/products/system/1U/6015/SYS-6015P-8.cfm]]'),
	3208 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-ACR24H | http://www.supermicro.com/products/system/2U/2028/SSG-2028R-ACR24H.cfm]]'),
	3209 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037MC-H8TRF | http://www.supermicro.com/products/system/tower/5037/SYS-5037MC-H8TRF.cfm]]'),
	3210 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R3-FTPT+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617R3-FTPT_.cfm]]'),
	3211 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6047R-E1CR36 | http://www.supermicro.com/products/system/4U/6047/SSG-6047R-E1CR36N.cfm]]'),
	3212 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-MR | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-MR.cfm]]'),
	3213 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D70RF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D70RF.cfm]]'),
	3214 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-MT | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-MT.cfm]]'),
	3215 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HTFR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HTFR.cfm]]'),
	3216 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028GR-TRH | http://www.supermicro.com/products/system/2u/2028/SYS-2028GR-TRH.cfm]]'),
	3217 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016GT-TF | http://www.supermicro.com/products/system/1U/6016/SYS-6016GT-TF.cfm]]'),
	3218 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TSF | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TSF.cfm]]'),
	3219 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRF-FM475 | http://www.supermicro.com/products/system/2U/2027/SYS-2027GR-TRF-FM475.cfm]]'),
	3220 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%8015C-T | http://www.supermicro.com/products/system/1U/8015/SYS-8015C-T.cfm]]'),
	3221 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-32 | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-32.cfm]]'),
	3222 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G3-FTPT+ | http://www.supermicro.com/products/system/4u/f627/SYS-F627G3-FTPT_.cfm]]'),
	3223 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-72BRFTP | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-72BRFTP.cfm]]'),
	3224 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%8048B-C0R3FT | http://www.supermicro.com/products/system/4U/8048/SYS-8048B-C0R3FT.cfm]]'),
	3225 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6037R-72RFT+ | http://www.supermicro.com/products/system/3u/6037/SYS-6037R-72RFT_.cfm]]'),
	3226 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5025B-T | http://www.supermicro.com/products/system/2U/5025/SYS-5025B-T.cfm]]'),
	3227 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026T-3RF | http://www.supermicro.com/products/system/2U/6026/SYS-6026T-3RF.cfm]]'),
	3228 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015T-T | http://www.supermicro.com/products/system/1U/6015/SYS-6015T-T.cfm]]'),
	3229 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026T-URF | http://www.supermicro.com/products/system/1u/1026/SYS-1026T-URF.cfm]]'),
	3230 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HC0FR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HC0FR.cfm]]'),
	3231 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-T | http://www.supermicro.com/products/system/2u/6028/SYS-6028R-T.cfm]]'),
	3232 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5029AP-TN2 | http://www.supermicro.com/products/system/midtower/5029/SYS-5029AP-TN2.cfm]]'),
	3233 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HTR-SIOM | http://www.supermicro.com/products/system/2U/2028/SYS-2028TP-HTR-SIOM.cfm]]'),
	3234 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R2-F72+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R2-F72_.cfm]]'),
	3235 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-TDW | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-TDW.cfm]]'),
	3236 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G3-FTPT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G3-FTPT_.cfm]]'),
	3237 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DC0TR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DC0TR.cfm]]'),
	3238 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-72RFT | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-72RFT.cfm]]'),
	3239 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617H6-FTPTL+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617H6-FTPTL_.cfm]]'),
	3240 => array ('chapter_id' => 18, 'dict_value' => '[[SuperMicro%GPASS%937R-E2JB | http://www.supermicro.com/products/system/3U/937/SYS-937R-E2JB.cfm]]'),
	3241 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H71FRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H71FRF.cfm]]'),
	3242 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HTR | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HTR.cfm]]'),
	3243 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015P-T | http://www.supermicro.com/products/system/1U/6015/SYS-6015P-T.cfm]]'),
	3244 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018R-M | http://www.supermicro.com/products/system/1u/5018/SYS-5018R-M.cfm]]'),
	3245 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-TD | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-TD.cfm]]'),
	3246 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-TRT+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-TRT_.cfm]]'),
	3247 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R2-F72+ | http://www.supermicro.com/products/system/4u/f617/SYS-F617R2-F72_.cfm]]'),
	3248 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6038R-TXR | http://www.supermicro.com/products/system/3U/6038/SYS-6038R-TXR.cfm]]'),
	3249 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-72RFTP | http://www.supermicro.com/products/system/1U/1027/SYS-1027R-72RFTP.cfm]]'),
	3250 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-TRTP+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-TRTP_.cfm]]'),
	3251 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-MT+ | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-MT_.cfm]]'),
	3252 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016I-MT | http://www.supermicro.com/products/system/1u/5016/sys-5016i-mt.cfm]]'),
	3253 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016I-MR | http://www.supermicro.com/products/system/1u/5016/sys-5016i-mr.cfm]]'),
	3254 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017R-WR | http://www.supermicro.com/products/system/1u/1017/SYS-1017R-WR.cfm]]'),
	3255 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-DTRF+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-DTRF_.cfm]]'),
	3256 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028TR-TF | http://www.supermicro.com/products/system/1u/1028/SYS-1028TR-TF.cfm]]'),
	3257 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017GR-TF-FM109 | http://www.supermicro.com/products/system/1U/1017/SYS-1017GR-TF-FM109.cfm]]'),
	3258 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-MF+ | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-MF_.cfm]]'),
	3259 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028R-E1CR12T | http://www.supermicro.com/products/system/2U/6028/SSG-6028R-E1CR12T.cfm]]'),
	3260 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018A-MLHN4 | http://www.supermicro.com/products/system/1U/5018/SYS-5018A-MLHN4.cfm]]'),
	3261 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DC0R | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DC0R.cfm]]'),
	3262 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037A-T | http://www.supermicro.com/products/system/tower/5037/SYS-5037A-T.cfm]]'),
	3263 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 5U%GPASS%5086B-TRF | http://www.supermicro.com/products/system/5U/5086/SYS-5086B-TRF.cfm]]'),
	3264 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037A-I | http://www.supermicro.com/products/system/tower/5037/SYS-5037A-i.cfm]]'),
	3265 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-WRF | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-WRF.cfm]]'),
	3266 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015P-8R | http://www.supermicro.com/products/system/1U/5015/SYS-5015P-8R.cfm]]'),
	3267 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-IBXF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-IBXF.cfm]]'),
	3268 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028UX-LL1-B8 | http://www.supermicro.com/products/system/1U/1028/SYS-1028UX-LL1-B8.cfm]]'),
	3269 => array ('chapter_id' => 18, 'dict_value' => '[[SuperMicro%GPASS%927R-E2CJB | http://www.supermicro.com/products/system/2U/927/SSG-927R-E2CJB.cfm]]'),
	3270 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC1R | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-HC1R.cfm]]'),
	3271 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5025B-4 | http://www.supermicro.com/products/system/2U/5025/SYS-5025B-4.cfm]]'),
	3272 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017GR-TF | http://www.supermicro.com/products/system/1u/5017/SYS-5017GR-TF.cfm]]'),
	3273 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%7034L-I | http://www.supermicro.com/products/system/tower/7034/SYS-7034L-i.cfm]]'),
	3274 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5036I-I | http://www.supermicro.com/products/system/midtower/5036/SYS-5036I-I.cfm]]'),
	3275 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-I | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-i.cfm]]'),
	3276 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-3R | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-3R.cfm]]'),
	3277 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC1TR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-HC1TR.cfm]]'),
	3278 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HC1FR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HC1FR.cfm]]'),
	3279 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015A-NT | http://www.supermicro.com/products/system/1U/6015/SYS-6015A-NT.cfm]]'),
	3280 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-R72BPT+ | http://www.supermicro.com/products/system/4u/F628/SYS-F628R3-R72BPT_.cfm]]'),
	3281 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015P-TR | http://www.supermicro.com/products/system/1U/6015/SYS-6015P-TR.cfm]]'),
	3282 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-T | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-T.cfm]]'),
	3283 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5039AD-T | http://www.supermicro.com/products/system/tower/5039/SYS-5039AD-T.cfm]]'),
	3284 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-HDTRF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-HDTRF.cfm]]'),
	3285 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-WTRFTP+ | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-WTRFTP_.cfm]]'),
	3286 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HC0FR | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HC0FR.cfm]]'),
	3287 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-MTLF | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-MTLF.cfm]]'),
	3288 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028TR-HTFR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TR-HTFR.cfm]]'),
	3289 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-MCTR | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-MCTR.cfm]]'),
	3290 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-TRF-HFT1 | http://www.supermicro.com/products/system/2u/6027/SYS-6027AX-TRF-HFT1.cfm]]'),
	3291 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-TRF-HFT3 | http://www.supermicro.com/products/system/2u/6027/SYS-6027AX-TRF-HFT3.cfm]]'),
	3292 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-TRF-HFT2 | http://www.supermicro.com/products/system/2u/6027/SYS-6027AX-TRF-HFT2.cfm]]'),
	3293 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-MTRF | http://www.supermicro.com/products/system/1u/5018/SYS-5018D-MTRF.cfm]]'),
	3294 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-TDTPR | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-TDTPR.cfm]]'),
	3295 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-72RFTP+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-72RFTP_.cfm]]'),
	3296 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047GR-TRF | http://www.supermicro.com/products/system/4u/7047/sys-7047gr-trf.cfm]]'),
	3297 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-TR4+ | http://www.supermicro.com/products/system/1u/1028/SYS-1028U-TR4_.cfm]]'),
	3298 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DTR | http://www.supermicro.com/products/system/2u/2027/SYS-2027PR-DTR.cfm]]'),
	3299 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F647G2-FT+ | http://www.supermicro.com/products/system/4u/f647/SYS-F647G2-FT_.cfm]]'),
	3300 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046A-T | http://www.supermicro.com/products/system/4U/7046/SYS-7046A-T.cfm]]'),
	3301 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5130AD-T | http://www.supermicro.com/products/system/tower/5130/SYS-5130AD-T.cfm]]'),
	3302 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRFT | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TRFT.cfm]]'),
	3303 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-8+ | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-8_.cfm]]'),
	3304 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-FC0 | http://www.supermicro.com/products/system/4u/F628/SYS-F628R3-FC0.cfm]]'),
	3305 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-MT | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-MT.cfm]]'),
	3306 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5037C-I | http://www.supermicro.com/products/system/tower/5037/sys-5037c-i.cfm]]'),
	3307 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028BT-HNR+ | http://www.supermicro.com/products/system/2U/2028/SYS-2028BT-HNR_.cfm]]'),
	3308 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R2-FT+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617R2-FT_.cfm]]'),
	3309 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%8014T-T | http://www.supermicro.com/products/system/1U/8014/SYS-8014T-T.cfm]]'),
	3310 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5014C-M8 | http://www.supermicro.com/products/system/1U/5014/SYS-5014C-M8.cfm]]'),
	3311 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-N3RFT+ | http://www.supermicro.com/products/system/2u/6027/SYS-6027R-N3RFT_.cfm]]'),
	3312 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4047R-7JRFT | http://www.supermicro.com/products/system/4U/4047/SYS-4047R-7JRFT.cfm]]'),
	3313 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047GR-TPRF-FM475 | http://www.supermicro.com/products/system/4U/7047/SYS-7047GR-TPRF-FM475.cfm]]'),
	3314 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-D6IBQRF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-D6IBQRF.cfm]]'),
	3315 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-MD | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-MD.cfm]]'),
	3316 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015V-T | http://www.supermicro.com/products/system/1U/6015/SYS-6015V-T.cfm]]'),
	3317 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-73DAF | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-73DAF.cfm]]'),
	3318 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-82R | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-82R.cfm]]'),
	3319 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-8R | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-8R.cfm]]'),
	3320 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018U-TR4+ | http://www.supermicro.com/products/system/1u/6018/SYS-6018U-TR4_.cfm]]'),
	3321 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017A-MP | http://www.supermicro.com/products/system/mini-itx/1017/SYS-1017A-MP.cfm]]'),
	3322 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045B-8R+ | http://www.supermicro.com/products/system/4U/7045/SYS-7045B-8R_.cfm]]'),
	3323 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-73THDP+ | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-73THDP_.cfm]]'),
	3324 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC1FR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-HC1FR.cfm]]'),
	3325 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HC1TR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HC1TR.cfm]]'),
	3326 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-D6IBXRF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-D6IBXRF.cfm]]'),
	3327 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G2-F73+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627G2-F73_.cfm]]'),
	3328 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-72RF | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-72RF.cfm]]'),
	3329 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6047R-E1R24 | http://www.supermicro.com/products/system/4U/6047/SSG-6047R-E1R24N.cfm]]'),
	3330 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017GR-TF-FM275 | http://www.supermicro.com/products/system/1U/1017/SYS-1017GR-TF-FM275.cfm]]'),
	3331 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015P-TR | http://www.supermicro.com/products/system/1U/5015/SYS-5015P-TR.cfm]]'),
	3332 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-C1R4+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028R-C1R4_.cfm]]'),
	3333 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048R-TR | http://www.supermicro.com/products/system/4U/7048/SYS-7048R-TR.cfm]]'),
	3334 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F647G2-FTPT+ | http://www.supermicro.com/products/system/4U/F647/SYS-F647G2-FTPT_.cfm]]'),
	3335 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1019S-MC0T | http://www.supermicro.com/products/system/1U/1019/SYS-1019S-MC0T.cfm]]'),
	3336 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6013P-I | http://www.supermicro.com/products/system/1U/6013/SYS-6013P-i.cfm]]'),
	3337 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC1R | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HC1R.cfm]]'),
	3338 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-32R | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-32R.cfm]]'),
	3339 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6013P-T | http://www.supermicro.com/products/system/1U/6013/SYS-6013P-T.cfm]]'),
	3340 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047GR-TPRF | http://www.supermicro.com/products/system/4u/7047/SYS-7047GR-TPRF.cfm]]'),
	3341 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6012P-6 | http://www.supermicro.com/products/system/1U/6012/SYS-6012P-6.cfm]]'),
	3342 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-E1CR24 | http://www.supermicro.com/products/system/2u/2028/SSG-2028R-E1CR24N.cfm]]'),
	3343 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6012P-8 | http://www.supermicro.com/products/system/1U/6012/SYS-6012P-8.cfm]]'),
	3344 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-TDF | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-TDF.cfm]]'),
	3345 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7045B-TR+ | http://www.supermicro.com/products/system/4U/7045/SYS-7045B-TR_.cfm]]'),
	3346 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G2-FTPT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G2-FTPT_.cfm]]'),
	3347 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6048R-E1CR60 | http://www.supermicro.com/products/system/4U/6048/SSG-6048R-E1CR60N.cfm]]'),
	3348 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015A-EHF-D525 | http://www.supermicro.com/products/system/1U/5015/SYS-5015A-EHF-D525.cfm]]'),
	3349 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-IBQF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-IBQF.cfm]]'),
	3350 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028U-E1CNRT+ | http://www.supermicro.com/products/system/2u/6028/SYS-6028U-E1CNRT_.cfm]]'),
	3351 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047GR-TRF-FC475 | http://www.supermicro.com/products/system/4u/7047/SYS-7047GR-TRF-FC475.cfm]]'),
	3352 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015TC-T | http://www.supermicro.com/products/system/1U/6015/SYS-6015TC-T.cfm]]'),
	3353 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G2-FC0+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G2-FC0_.cfm]]'),
	3354 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HC1R | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HC1R.cfm]]'),
	3355 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F648G2-FC0+ | http://www.supermicro.com/products/system/4U/F648/SYS-F648G2-FC0_.cfm]]'),
	3356 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618H6-FTPT+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618H6-FTPT_.cfm]]'),
	3357 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-TNRTP+ | http://www.supermicro.com/products/system/1u/1028/SYS-1028U-TNRTP_.cfm]]'),
	3358 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 7U%GPASS%7088B-TR4FT | http://www.supermicro.com/products/system/7U/7088/SYS-7088B-TR4FT.cfm]]'),
	3359 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6013E-I | http://www.supermicro.com/products/system/1U/6013/SYS-6013E-i.cfm]]'),
	3360 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-MCT | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-MCT.cfm]]'),
	3361 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HTTR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HTTR.cfm]]'),
	3362 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-T | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-T.cfm]]'),
	3363 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6013P-8 | http://www.supermicro.com/products/system/1U/6013/SYS-6013P-8.cfm]]'),
	3364 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-E1CRTP+ | http://www.supermicro.com/products/system/1u/1028/SYS-1028U-E1CRTP_.cfm]]'),
	3365 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-82 | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-82.cfm]]'),
	3366 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017TR-TF | http://www.supermicro.com/products/system/1u/6017/SYS-6017TR-TF.cfm]]'),
	3367 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H70RF+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-H70RF_.cfm]]'),
	3368 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6047R-E1R72L2K | http://www.supermicro.com/products/system/4U/6047/SSG-6047R-E1R72L2K.cfm]]'),
	3369 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TRFT+ | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TRFT_.cfm]]'),
	3370 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-WRF | http://www.supermicro.com/products/system/2u/2027/SYS-2027R-WRF.cfm]]'),
	3371 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5035B-T | http://www.supermicro.com/products/system/tower/5035/SYS-5035B-T.cfm]]'),
	3372 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1017R-MTF | http://www.supermicro.com/products/system/1u/1017/SYS-1017R-MTF.cfm]]'),
	3373 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DC1TR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DC1TR.cfm]]'),
	3374 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRFH | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TRFH.cfm]]'),
	3375 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1025C-M3 | http://www.supermicro.com/products/system/1U/1025/SYS-1025C-M3.cfm]]'),
	3376 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5019S-MR | http://www.supermicro.com/products/system/1U/5019/SYS-5019S-MR.cfm]]'),
	3377 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-DC1FR | http://www.supermicro.com/products/system/2u/2028/SYS-2028TP-DC1FR.cfm]]'),
	3378 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5019S-MT | http://www.supermicro.com/products/system/1U/5019/SYS-5019S-MT.cfm]]'),
	3379 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WRF | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-WRF.cfm]]'),
	3380 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-NI | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-Ni.cfm]]'),
	3381 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048R-C1RT | http://www.supermicro.com/products/system/4U/7048/SYS-7048R-C1RT.cfm]]'),
	3382 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025B-T | http://www.supermicro.com/products/system/2U/6025/SYS-6025B-T.cfm]]'),
	3383 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7046A-6 | http://www.supermicro.com/products/system/4U/7046/SYS-7046A-6.cfm]]'),
	3384 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027TR-TF | http://www.supermicro.com/products/system/1u/1027/SYS-1027TR-TF.cfm]]'),
	3385 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H71FRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-H71FRF.cfm]]'),
	3386 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048R-C1R4+ | http://www.supermicro.com/products/system/4U/7048/SYS-7048R-C1R4_.cfm]]'),
	3387 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HTFR | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HTFR.cfm]]'),
	3388 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-HIBQRF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-HIBQRF.cfm]]'),
	3389 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-UF | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-UF.cfm]]'),
	3390 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6048R-E1CR36H | http://www.supermicro.com/products/system/4u/6048/SSG-6048R-E1CR36H.cfm]]'),
	3391 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TQF | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TQF.cfm]]'),
	3392 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-AR24NV | http://www.supermicro.com/products/system/2u/2027/SSG-2027R-AR24NV.cfm]]'),
	3393 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5026TI-BTRF | http://www.supermicro.com/products/system/2U/5026/SYS-5026TI-BTRF.cfm]]'),
	3394 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-N3RF4+ | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-N3RF4_.cfm]]'),
	3395 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018U-TRT+ | http://www.supermicro.com/products/system/1u/6018/SYS-6018U-TRT_.cfm]]'),
	3396 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1015B-M3 | http://www.supermicro.com/products/system/1U/1015/SYS-1015B-M3.cfm]]'),
	3397 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G3-FT+ | http://www.supermicro.com/products/system/4u/f627/SYS-F627G3-FT_.cfm]]'),
	3398 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018D-MF | http://www.supermicro.com/products/system/1u/5018/SYS-5018D-MF.cfm]]'),
	3399 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5028R-WR | http://www.supermicro.com/products/system/2u/5028/SYS-5028R-WR.cfm]]'),
	3400 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027B-TLF | http://www.supermicro.com/products/system/2u/6027/SYS-6027B-TLF.cfm]]'),
	3401 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028BT-HTR+ | http://www.supermicro.com/products/system/2U/2028/SYS-2028BT-HTR_.cfm]]'),
	3402 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WC1NR | http://www.supermicro.com/products/system/1U/1027/SYS-1027R-WC1NR.cfm]]'),
	3403 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6048R-E1CR36 | http://www.supermicro.com/products/system/4u/6048/SSG-6048R-E1CR36N.cfm]]'),
	3404 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H71RF+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-H71RF_.cfm]]'),
	3405 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047A-T | http://www.supermicro.com/products/system/4u/7047/SYS-7047A-T.cfm]]'),
	3406 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028GR-TR | http://www.supermicro.com/products/system/1u/1028/SYS-1028GR-TR.cfm]]'),
	3407 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-TR | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-TR.cfm]]'),
	3408 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-UR | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-UR.cfm]]'),
	3409 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F647G2-F73+ | http://www.supermicro.com/products/system/4U/F647/SYS-F647G2-F73_.cfm]]'),
	3410 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-RTB+ | http://www.supermicro.com/products/system/4u/f627/SYS-F627R3-RTB_.cfm]]'),
	3411 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2048U-RTR4 | http://www.supermicro.com/products/system/2u/2048/SYS-2048U-RTR4.cfm]]'),
	3412 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6013A-T | http://www.supermicro.com/products/system/1U/6013/SYS-6013A-T.cfm]]'),
	3413 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R2-FC0PT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R2-FC0PT_.cfm]]'),
	3414 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-E1CR24H | http://www.supermicro.com/products/system/2u/2028/SSG-2028R-E1CR24H.cfm]]'),
	3415 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018A-MLTN4 | http://www.supermicro.com/products/system/1u/5018/SYS-5018A-MLTN4.cfm]]'),
	3416 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5019S-M2 | http://www.supermicro.com/products/system/1U/5019/SYS-5019S-M2.cfm]]'),
	3417 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RTBN+ | http://www.supermicro.com/products/system/4u/f628/SYS-F628R3-RTBN_.cfm]]'),
	3418 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6012L-6 | http://www.supermicro.com/products/system/1U/6012/SYS-6012L-6.cfm]]'),
	3419 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WRF4+ | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-WRF4_.cfm]]'),
	3420 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1025TC-3F | http://www.supermicro.com/products/system/1U/1025/SYS-1025TC-3F.cfm]]'),
	3421 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4028GR-TXRT | http://www.supermicro.com/products/system/4U/4028/SYS-4028GR-TXRT.cfm]]'),
	3422 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-E1CNR4T+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-E1CNR4T_.cfm]]'),
	3423 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R2-RT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R2-RT_.cfm]]'),
	3424 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6014P-8R | http://www.supermicro.com/products/system/1U/6014/SYS-6014P-8R.cfm]]'),
	3425 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028GR-TRHT | http://www.supermicro.com/products/system/2u/2028/SYS-2028GR-TRHT.cfm]]'),
	3426 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025B-8 | http://www.supermicro.com/products/system/2U/6025/SYS-6025B-8.cfm]]'),
	3427 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5036A-T | http://www.supermicro.com/products/system/midtower/5036/SYS-5036A-T.cfm]]'),
	3428 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1025TC-10G | http://www.supermicro.com/products/system/1U/1025/SYS-1025TC-10G.cfm]]'),
	3429 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-FTPT+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618R2-FTPT_.cfm]]'),
	3430 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-C1R | http://www.supermicro.com/products/system/2u/2028/SYS-2028R-C1R.cfm]]'),
	3431 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-H6IBXRF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-H6IBXRF.cfm]]'),
	3432 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016I-MRHF | http://www.supermicro.com/products/system/1u/5016/sys-5016i-mrhf.cfm]]'),
	3433 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026TT-IBQF | http://www.supermicro.com/products/system/1U/1026/SYS-1026TT-IBQF.cfm]]'),
	3434 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4027GR-TR | http://www.supermicro.com/products/system/4u/4027/SYS-4027GR-TR.cfm]]'),
	3435 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028U-TN10RT+ | http://www.supermicro.com/products/system/1u/1028/sys-1028u-tn10rt_.cfm]]'),
	3436 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017TR-TQF | http://www.supermicro.com/products/system/1u/6017/SYS-6017TR-TQF.cfm]]'),
	3437 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4027GR-TRT | http://www.supermicro.com/products/system/4u/4027/SYS-4027GR-TRT.cfm]]'),
	3438 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WMRT | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WMRT.cfm]]'),
	3439 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018A-TN7B | http://www.supermicro.com/products/system/1u/5018/SYS-5018A-TN7B.cfm]]'),
	3440 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026T-URF4+ | http://www.supermicro.com/products/system/2U/2026/SYS-2026T-URF4_.cfm]]'),
	3441 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%8048B-C0R4FT | http://www.supermicro.com/products/system/4U/8048/SYS-8048B-C0R4FT.cfm]]'),
	3442 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6023L-8R | http://www.supermicro.com/products/system/2U/6023/SYS-6023L-8R.cfm]]'),
	3443 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038ML-H24TRF | http://www.supermicro.com/products/system/3U/5038/SYS-5038ML-H24TRF.cfm]]'),
	3444 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5027R-WRF | http://www.supermicro.com/products/system/2u/5027/SYS-5027R-WRF.cfm]]'),
	3445 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DC1FR | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DC1FR.cfm]]'),
	3446 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RTPT+ | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-RTPT_.cfm]]'),
	3447 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017C-TF | http://www.supermicro.com/products/system/1u/5017/sys-5017c-tf.cfm]]'),
	3448 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028GQ-TRT | http://www.supermicro.com/products/system/1u/1028/sys-1028gq-trt.cfm]]'),
	3449 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015P-T | http://www.supermicro.com/products/system/1U/5015/SYS-5015P-T.cfm]]'),
	3450 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6022P-6 | http://www.supermicro.com/products/system/2U/6022/SYS-6022P-6.cfm]]'),
	3451 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R2-F73 | http://www.supermicro.com/products/system/4U/F617/SYS-F617R2-F73.cfm]]'),
	3452 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WTNR | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WTNR.cfm]]'),
	3453 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026T-URF4+ | http://www.supermicro.com/products/system/2U/6026/SYS-6026T-URF4_.cfm]]'),
	3454 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026GT-TRF | http://www.supermicro.com/products/system/1U/1026/SYS-1026GT-TRF.cfm]]'),
	3455 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028UX-LL2-B8 | http://www.supermicro.com/products/system/1U/1028/SYS-1028UX-LL2-B8.cfm]]'),
	3456 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044H-TR | http://www.supermicro.com/products/system/4U/7044/SYS-7044H-TR.cfm]]'),
	3457 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025W-UR | http://www.supermicro.com/products/system/2U/6025/SYS-6025W-UR.cfm]]'),
	3458 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015P-8 | http://www.supermicro.com/products/system/1U/5015/SYS-5015P-8.cfm]]'),
	3459 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025B-3 | http://www.supermicro.com/products/system/2U/6025/SYS-6025B-3.cfm]]'),
	3460 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-R72B+ | http://www.supermicro.com/products/system/4u/f627/SYS-F627R3-R72B_.cfm]]'),
	3461 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-FC0+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R3-FC0_.cfm]]'),
	3462 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-FT | http://www.supermicro.com/products/system/4U/F627/SYS-F627R3-FT.cfm]]'),
	3463 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028TR-T | http://www.supermicro.com/products/system/1u/1028/SYS-1028TR-T.cfm]]'),
	3464 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-HTRF | http://www.supermicro.com/products/system/2u/6027/SYS-6027TR-HTRF.cfm]]'),
	3465 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-72RT2 | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-72RT2.cfm]]'),
	3466 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-NTR | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-NTR.cfm]]'),
	3467 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018TR-T | http://www.supermicro.com/products/system/1u/6018/SYS-6018TR-T.cfm]]'),
	3468 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026TT-DLIBQRF | http://www.supermicro.com/products/system/2U/2026/SYS-2026TT-DLIBQRF.cfm]]'),
	3469 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-UR | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-UR.cfm]]'),
	3470 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047R-TRF | http://www.supermicro.com/products/system/4U/7047/SYS-7047R-TRF.cfm]]'),
	3471 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047R-72RF | http://www.supermicro.com/products/system/4U/7047/SYS-7047R-72RF.cfm]]'),
	3472 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027B-MTF | http://www.supermicro.com/products/system/1u/1027/SYS-1027B-MTF.cfm]]'),
	3473 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-N3RF4+ | http://www.supermicro.com/products/system/2u/6027/SYS-6027R-N3RF4_.cfm]]'),
	3474 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016I-UR | http://www.supermicro.com/products/system/1u/5016/sys-5016i-ur.cfm]]'),
	3475 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R2-F72PT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R2-F72PT_.cfm]]'),
	3476 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-H70QRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-H70QRF.cfm]]'),
	3477 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-H6IBQRF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-H6IBQRF.cfm]]'),
	3478 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038ML-H12TRF | http://www.supermicro.com/products/system/3u/5038/SYS-5038ML-H12TRF.cfm]]'),
	3479 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DC0TR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-DC0TR.cfm]]'),
	3480 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro Tower%GPASS%8047R-TRF+ | http://www.supermicro.com/products/system/4U/8047/SYS-8047R-TRF_.cfm]]'),
	3481 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017C-MF | http://www.supermicro.com/products/system/1u/5017/sys-5017c-mf.cfm]]'),
	3482 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%8017R-7FT+ | http://www.supermicro.com/products/system/1U/8017/SYS-8017R-7FT_.cfm]]'),
	3483 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5017C-MTF | http://www.supermicro.com/products/system/1u/5017/sys-5017c-mtf.cfm]]'),
	3484 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5018R-MR | http://www.supermicro.com/products/system/1u/5018/SYS-5018R-MR.cfm]]'),
	3485 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro%GPASS%8044T-8R | http://www.supermicro.com/products/system/4U/8044/SYS-8044T-8R.cfm]]'),
	3486 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%8048B-TR4FT | http://www.supermicro.com/products/system/4u/8048/SYS-8048B-TR4FT.cfm]]'),
	3487 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5019S-MN4 | http://www.supermicro.com/products/system/1U/5019/SYS-5019S-MN4.cfm]]'),
	3488 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6035B-8 | http://www.supermicro.com/products/system/3U/6035/SYS-6035B-8.cfm]]'),
	3489 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-82R | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-82R.cfm]]'),
	3490 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-N3RFT+ | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-N3RFT_.cfm]]'),
	3491 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026TT-DLIBXRF | http://www.supermicro.com/products/system/2U/2026/SYS-2026TT-DLIBXRF.cfm]]'),
	3492 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-BIBXF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-BIBXF.cfm]]'),
	3493 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%1028TP-DC1TR | http://www.supermicro.com/products/system/1U/1028/SYS-1028TP-DC1TR.cfm]]'),
	3494 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D70QRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D70QRF.cfm]]'),
	3495 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F648G2-FC0PT+ | http://www.supermicro.com/products/system/4U/F648/SYS-F648G2-FC0PT_.cfm]]'),
	3496 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026T-NTR+ | http://www.supermicro.com/products/system/2U/6026/SYS-6026T-NTR_.cfm]]'),
	3497 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015V-MRLP | http://www.supermicro.com/products/system/1U/6015/SYS-6015V-MRLP.cfm]]'),
	3498 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-6RF+ | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-6RF_.cfm]]'),
	3499 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WTR | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WTR.cfm]]'),
	3500 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%5038ML-H8TRF | http://www.supermicro.com/products/system/3u/5038/SYS-5038ML-H8TRF.cfm]]'),
	3501 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617R2-FTPT+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617R2-FTPT_.cfm]]'),
	3502 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1019S-M2 | http://www.supermicro.com/products/system/1U/1019/SYS-1019S-M2.cfm]]'),
	3503 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro Tower%GPASS%8046B-6R | http://www.supermicro.com/products/system/4U/8046/SYS-8046B-6R.cfm]]'),
	3504 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%2028TP-HC0TR | http://www.supermicro.com/products/system/2U/2028/SYS-2028TP-HC0TR.cfm]]'),
	3505 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5026T-3F | http://www.supermicro.com/products/system/2U/5026/SYS-5026T-3F.cfm]]'),
	3506 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TR2 | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TR2.cfm]]'),
	3507 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D71RF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D71RF.cfm]]'),
	3508 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6024H-I2 | http://www.supermicro.com/products/system/2U/6024/SYS-6024H-i2.cfm]]'),
	3509 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2026TT-H6IBQRF | http://www.supermicro.com/products/system/2U/2026/SYS-2026TT-H6IBQRF.cfm]]'),
	3510 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-M7RF | http://www.supermicro.com/products/system/1U/6017/SYS-6017R-M7RF.cfm]]'),
	3511 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-N3RFT+ | http://www.supermicro.com/products/system/2u/2027/SYS-2027R-N3RFT_.cfm]]'),
	3512 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015P-8R | http://www.supermicro.com/products/system/1U/6015/SYS-6015P-8R.cfm]]'),
	3513 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R3-FTPT+ | http://www.supermicro.com/products/system/4U/F618/SYS-F618R3-FTPT_.cfm]]'),
	3514 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H72FRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H72FRF.cfm]]'),
	3515 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015A-PHF | http://www.supermicro.com/products/system/1U/5015/SYS-5015A-PHF.cfm]]'),
	3516 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5014C-T | http://www.supermicro.com/products/system/1U/5014/SYS-5014C-T.cfm]]'),
	3517 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026T-6RF+ | http://www.supermicro.com/products/system/2U/6026/SYS-6026T-6RF_.cfm]]'),
	3518 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%8048B-TR4F | http://www.supermicro.com/products/system/4U/8048/SYS-8048B-TR4F.cfm]]'),
	3519 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F618R2-RTPTN+ | http://www.supermicro.com/products/system/4u/F618/SYS-F618R2-RTPTN_.cfm]]'),
	3520 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027R-72RFTP+ | http://www.supermicro.com/products/system/2U/2027/SYS-2027R-72RFTP_.cfm]]'),
	3521 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028GR-TRT | http://www.supermicro.com/products/system/1u/1028/SYS-1028GR-TRT.cfm]]'),
	3522 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro%GPASS%8045C-3R | http://www.supermicro.com/products/system/4U/8045/SYS-8045C-3R.cfm]]'),
	3523 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H72QRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H72QRF.cfm]]'),
	3524 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6023P-8R | http://www.supermicro.com/products/system/2U/6023/SYS-6023P-8R.cfm]]'),
	3525 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TRF-FM309 | http://www.supermicro.com/products/system/1U/1027/SYS-1027GR-TRF-FM309.cfm]]'),
	3526 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro Tower%GPASS%8046B-TRLF | http://www.supermicro.com/products/system/4U/8046/SYS-8046B-TRLF.cfm]]'),
	3527 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WMR | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WMR.cfm]]'),
	3528 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-DC0FR | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-DC0FR.cfm]]'),
	3529 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027GR-TRF | http://www.supermicro.com/products/system/2u/2027/SYS-2027GR-TRF.cfm]]'),
	3530 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-73DARF | http://www.supermicro.com/products/system/2u/6027/sys-6027r-73darf.cfm]]'),
	3531 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-72R2 | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-72R2.cfm]]'),
	3532 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R2-FT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R2-FT_.cfm]]'),
	3533 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6016T-6F | http://www.supermicro.com/products/system/1U/6016/SYS-6016T-6F.cfm]]'),
	3534 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-72RF | http://www.supermicro.com/products/system/2u/6027/SYS-6027AX-72RF.cfm]]'),
	3535 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6025B-TR+ | http://www.supermicro.com/products/system/2U/6025/SYS-6025B-TR_.cfm]]'),
	3536 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%8048B-TR3F | http://www.supermicro.com/products/system/4U/8048/SYS-8048B-TR3F.cfm]]'),
	3537 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027TR-D70FRF | http://www.supermicro.com/products/system/2U/6027/SYS-6027TR-D70FRF.cfm]]'),
	3538 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-BIBQF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-BIBQF.cfm]]'),
	3539 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6015B-NT | http://www.supermicro.com/products/system/1U/6015/SYS-6015B-NT.cfm]]'),
	3540 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-TDARF | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-TDARF.cfm]]'),
	3541 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7047A-73 | http://www.supermicro.com/products/system/4u/7047/SYS-7047A-73.cfm]]'),
	3542 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5028TK-HTR | http://www.supermicro.com/products/system/2U/5028/SYS-5028TK-HTR.cfm]]'),
	3543 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027PR-HC0R | http://www.supermicro.com/products/system/2U/2027/SYS-2027PR-HC0R.cfm]]'),
	3544 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1018R-WC0R | http://www.supermicro.com/products/system/1u/1018/SYS-1018R-WC0R.cfm]]'),
	3545 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-T | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-T.cfm]]'),
	3546 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015M-U | http://www.supermicro.com/products/system/1U/5015/SYS-5015M-U.cfm]]'),
	3547 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-MT | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-MT.cfm]]'),
	3548 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7048A-T | http://www.supermicro.com/products/system/4U/7048/SYS-7048A-T.cfm]]'),
	3549 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC1R-SIOM | http://www.supermicro.com/products/system/2U/6028/SYS-6028TP-HC1R-SIOM.cfm]]'),
	3550 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-MR | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-MR.cfm]]'),
	3551 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027B-URF | http://www.supermicro.com/products/system/1u/1027/SYS-1027B-URF.cfm]]'),
	3552 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628G3-FT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628G3-FT_.cfm]]'),
	3553 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%8017R-TF+ | http://www.supermicro.com/products/system/1u/8017/SYS-8017R-TF_.cfm]]'),
	3554 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028GQ-TXRT | http://www.supermicro.com/products/system/1U/1028/SYS-1028GQ-TXRT.cfm]]'),
	3555 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-HTRF+ | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-HTRF_.cfm]]'),
	3556 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-MF | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-MF.cfm]]'),
	3557 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TR2 | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TR2.cfm]]'),
	3558 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5028A-TN4 | http://www.supermicro.com/products/system/midtower/5028/sys-5028a-tn4.cfm]]'),
	3559 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016TI-TF | http://www.supermicro.com/products/system/1U/5016/SYS-5016TI-TF.cfm]]'),
	3560 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6038R-E1CR16H | http://www.supermicro.com/products/system/3U/6038/SSG-6038R-E1CR16H.cfm]]'),
	3561 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1028R-WTRT | http://www.supermicro.com/products/system/1u/1028/SYS-1028R-WTRT.cfm]]'),
	3562 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-H71RF+ | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-H71RF_.cfm]]'),
	3563 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2027TR-D70FRF | http://www.supermicro.com/products/system/2U/2027/SYS-2027TR-D70FRF.cfm]]'),
	3564 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1026TT-TF | http://www.supermicro.com/products/system/1U/1026/SYS-1026TT-TF.cfm]]'),
	3565 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RC0BPT+ | http://www.supermicro.com/products/system/4u/f628/SYS-F628R3-RC0BPT_.cfm]]'),
	3566 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6026TT-D6RF | http://www.supermicro.com/products/system/2U/6026/SYS-6026TT-D6RF.cfm]]'),
	3567 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-FT | http://www.supermicro.com/products/system/4u/f628/SYS-F628R3-FT.cfm]]'),
	3568 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G2-FT+ | http://www.supermicro.com/products/system/4u/f627/SYS-F627G2-FT_.cfm]]'),
	3569 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 3U%GPASS%6035B-8R+ | http://www.supermicro.com/products/system/3U/6035/SYS-6035B-8R_.cfm]]'),
	3570 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%4028GR-TXR | http://www.supermicro.com/products/system/4U/4028/SYS-4028GR-TXR.cfm]]'),
	3571 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R2-FTPT+ | http://www.supermicro.com/products/system/4U/F628/SYS-F628R2-FTPT_.cfm]]'),
	3572 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627G2-F73PT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627G2-F73PT_.cfm]]'),
	3573 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027R-WC1R | http://www.supermicro.com/products/system/1u/1027/SYS-1027R-WC1R.cfm]]'),
	3574 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028U-TR4+ | http://www.supermicro.com/products/system/2u/2028/SYS-2028U-TR4_.cfm]]'),
	3575 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%7044H-T | http://www.supermicro.com/products/system/4U/7044/SYS-7044H-T.cfm]]'),
	3576 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6018R-MDR | http://www.supermicro.com/products/system/1u/6018/SYS-6018R-MDR.cfm]]'),
	3577 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-72RF-HFT1 | http://www.supermicro.com/products/system/2u/6027/SYS-6027AX-72RF-HFT1.cfm]]'),
	3578 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-72RF-HFT3 | http://www.supermicro.com/products/system/2u/6027/sys-6027ax-72rf-hft3.cfm]]'),
	3579 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-72RF-HFT2 | http://www.supermicro.com/products/system/2u/6027/SYS-6027AX-72RF-HFT2.cfm]]'),
	3580 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F627R3-F72PT+ | http://www.supermicro.com/products/system/4U/F627/SYS-F627R3-F72PT_.cfm]]'),
	3581 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2028R-NR48 | http://www.supermicro.com/products/system/2U/2028/SSG-2028R-NR48N.cfm]]'),
	3582 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5016I-T | http://www.supermicro.com/products/system/1u/5016/sys-5016i-t.cfm]]'),
	3583 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F617H6-FTPT+ | http://www.supermicro.com/products/system/4U/F617/SYS-F617H6-FTPT_.cfm]]'),
	3584 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027AX-TRF | http://www.supermicro.com/products/system/2u/6027/SYS-6027AX-TRF.cfm]]'),
	3585 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1018D-FRN8T | http://www.supermicro.com/products/system/1u/1018/SYS-1018D-FRN8T.cfm]]'),
	3586 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%8028B-C0R3FT | http://www.supermicro.com/products/system/2u/8028/SYS-8028B-C0R3FT.cfm]]'),
	3587 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1018R-WR | http://www.supermicro.com/products/system/1U/1018/SYS-1018R-WR.cfm]]'),
	3588 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1019S-WR | http://www.supermicro.com/products/system/1U/1019/SYS-1019S-WR.cfm]]'),
	3589 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%2016TI-HTRF | http://www.supermicro.com/products/system/2u/2016/sys-2016ti-htrf.cfm]]'),
	3590 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 4U%GPASS%6047R-E1R36 | http://www.supermicro.com/products/system/4u/6047/SSG-6047R-E1R36N.cfm]]'),
	3591 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%5025M-4+ | http://www.supermicro.com/products/system/2U/5025/SYS-5025M-4_.cfm]]'),
	3592 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6028TR-DTR | http://www.supermicro.com/products/system/2u/6028/SYS-6028TR-DTR.cfm]]'),
	3593 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%1027GR-TRF | http://www.supermicro.com/products/system/1u/1027/SYS-1027GR-TRF.cfm]]'),
	3594 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HC1R | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HC1R.cfm]]'),
	3595 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%F628R3-RC0B+ | http://www.supermicro.com/products/system/4u/f628/SYS-F628R3-RC0B_.cfm]]'),
	3596 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5014C-MF | http://www.supermicro.com/products/system/1U/5014/SYS-5014C-MF.cfm]]'),
	3597 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5015B-M3 | http://www.supermicro.com/products/system/1U/5015/SYS-5015B-M3.cfm]]'),
	3598 => array ('chapter_id' => 18, 'dict_value' => '[[SuperMicro%GPASS%K1048-RT | http://www.supermicro.com/products/system/1u/k1048/ssg-k1048-rt.cfm]]'),
	3599 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%6017R-TDLF | http://www.supermicro.com/products/system/1u/6017/SYS-6017R-TDLF.cfm]]'),
	3600 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027PR-HC0R | http://www.supermicro.com/products/system/2U/6027/SYS-6027PR-HC0R.cfm]]'),
	3601 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5014C-MR | http://www.supermicro.com/products/system/1U/5014/SYS-5014C-MR.cfm]]'),
	3602 => array ('chapter_id' => 31, 'dict_value' => '[[SuperMicro%GPASS%6028TP-HC0R | http://www.supermicro.com/products/system/2u/6028/SYS-6028TP-HC0R.cfm]]'),
	3603 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 2U%GPASS%6027R-CDNRT+ | http://www.supermicro.com/products/system/2U/6027/SYS-6027R-CDNRT_.cfm]]'),
	3604 => array ('chapter_id' => 11, 'dict_value' => '[[SuperMicro 1U%GPASS%5014C-MT | http://www.supermicro.com/products/system/1U/5014/SYS-5014C-MT.cfm]]'),
	3605 => array ('chapter_id' => 26, 'dict_value' => '[[Cisco%GPASS%MDS 9148 | http://www.cisco.com/c/en/us/products/storage-networking/mds-9148-multilayer-fabric-switch/index.html]]'),
	3606 => array ('chapter_id' => 26, 'dict_value' => '[[Cisco%GPASS%MDS 9148S | http://www.cisco.com/c/en/us/products/storage-networking/mds-9148s-16g-multilayer-fabric-switch/index.html]]'),
	3607 => array ('chapter_id' => 26, 'dict_value' => '[[Cisco%GPASS%MDS 9396S | http://www.cisco.com/c/en/us/products/storage-networking/mds-9396s-16g-multilayer-fabric-switch/index.html]]'),
	3608 => array ('chapter_id' => 26, 'dict_value' => '[[Cisco%GPASS%MDS 9706 | http://www.cisco.com/c/en/us/products/storage-networking/mds-9706-multilayer-director/index.html]]'),
	3609 => array ('chapter_id' => 26, 'dict_value' => '[[Cisco%GPASS%MDS 9710 | http://www.cisco.com/c/en/us/products/storage-networking/mds-9710-multilayer-director/index.html]]'),
	3610 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 300'),
	3611 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 400'),
	3612 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 500'),
	3613 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 600'),
	3614 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 350F'),
	3615 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 450F'),
	3616 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 550F'),
	3617 => array ('chapter_id' => 18, 'dict_value' => 'EMC Unity 650F'),
	3618 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 10K'),
	3619 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 20K'),
	3620 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 40K'),
	3621 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 100K'),
	3622 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 200K'),
	3623 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 250F'),
	3624 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 400K'),
	3625 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 450F'),
	3626 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 850F'),
	3627 => array ('chapter_id' => 18, 'dict_value' => 'EMC VMAX 950F'),
	3628 => array ('chapter_id' => 18, 'dict_value' => 'EMC XtremIO'),
	3629 => array ('chapter_id' => 18, 'dict_value' => 'EMC XtremIO X2'),
	3630 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 5300'),
	3631 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 5500'),
	3632 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 5700'),
	3633 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 7500'),
	3634 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 5200'),
	3635 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 5400'),
	3636 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 5600'),
	3637 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 5800'),
	3638 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 7600'),
	3639 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNX 8000'),
	3640 => array ('chapter_id' => 18, 'dict_value' => 'EMC Isilon NL410'),
	3641 => array ('chapter_id' => 18, 'dict_value' => 'EMC Isilon S210'),
	3642 => array ('chapter_id' => 18, 'dict_value' => 'EMC Isilon X410'),
	3643 => array ('chapter_id' => 18, 'dict_value' => 'EMC RecoverPoint G5'),
	3644 => array ('chapter_id' => 18, 'dict_value' => 'EMC RecoverPoint G6'),
	3645 => array ('chapter_id' => 18, 'dict_value' => 'EMC VPLEX'),
	3646 => array ('chapter_id' => 18, 'dict_value' => 'EMC VPLEX V6'),
	3647 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNXe 3150'),
	3648 => array ('chapter_id' => 18, 'dict_value' => 'EMC VNXe 3300'),
	3649 => array ('chapter_id' => 12, 'dict_value' => '[[Cisco%GPASS%Nexus 5596UP | http://www.cisco.com/c/en/us/products/switches/nexus-5596up-switch/index.html]]'),
	3650 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Nexus 93180YC-EX'),
	3651 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CSS326-24G-2S+RM'),
	3652 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS326-24G-2S+RM'),
	3653 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS317-1G-16S+RM'),
	3654 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 26'),
	3655 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 17.04'),
	3656 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 7.1'),
	3657 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7020TR-48'),
	3658 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7020TRA-48'),
	3659 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7160-32CQ'),
	3660 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7160-48YC6'),
	3661 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7160-48TC6'),
	3662 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7516R'),
	3663 => array ('chapter_id' => 31, 'dict_value' => 'HPE%GPASS%BladeSystem c3000%L4,2H%'),
	3664 => array ('chapter_id' => 31, 'dict_value' => 'HPE%GPASS%BladeSystem c7000%L2,8V%'),
	3665 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%RB1100AHx4'),
	3666 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5459-48XT-4Q'),
	3667 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5468-32QP'),
	3668 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5340-48XP-6Q'),
	3669 => array ('chapter_id' => 12, 'dict_value' => 'NEC%GPASS%PF5340-32QP'),
	3670 => array ('chapter_id' => 12, 'dict_value' => '[[TP-Link%GPASS%T1700G-28TQ | http://www.tp-link.com/en/products/details/cat-40_T1700G-28TQ.html]]'),
	3671 => array ('chapter_id' => 12, 'dict_value' => '[[TP-Link%GPASS%TL-SG2216 | http://www.tp-link.com/en/products/details/cat-5070_TL-SG2216.html]]'),
	3672 => array ('chapter_id' => 12, 'dict_value' => '[[TP-Link%GPASS%TL-SG3424 | http://www.tp-link.com/en/products/details/cat-39_TL-SG3424.html]]'),
	3673 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1009-7G-1C-PC'),
	3674 => array ('chapter_id' => 17, 'dict_value' => 'MikroTik%GPASS%CCR1009-7G-1C-1S+PC'),
	3675 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280QRA-C36S'),
	3676 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SRA-48C6'),
	3677 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280TRA-48C6'),
	3678 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SRAM-48C6'),
	3679 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280CR2-60'),
	3680 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280CR2A-60'),
	3681 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280CR2K-60'),
	3682 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280CR2K-30'),
	3683 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SR2-48YC6'),
	3684 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7280SR2A-48YC6'),
	3685 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS112-8P-4S-IN'),
	3686 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 17.10'),
	3687 => array ('chapter_id' => 13, 'dict_value' => 'SUSE Enterprise%GSKIP%SLES15'),
	3688 => array ('chapter_id' => 13, 'dict_value' => 'OpenSUSE%GSKIP%openSUSE Leap 15.x'),
	3689 => array ('chapter_id' => 39, 'dict_value' => '[[APC%GPASS%SMT1500RMI2U | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=SMT1500RMI2U]]'),
	3690 => array ('chapter_id' => 39, 'dict_value' => '[[APC%GPASS%SMT1500RMI2UNC | http://www.apc.com/products/resource/include/techspec_index.cfm?base_sku=SMT1500RMI2UNC]]'),
	3691 => array ('chapter_id' => 12, 'dict_value' => '[[NETGEAR%GPASS%GS108 | https://www.netgear.com/business/products/switches/unmanaged/GS108.aspx]]'),
	3692 => array ('chapter_id' => 12, 'dict_value' => '[[NETGEAR%GPASS%GS105 | https://www.netgear.com/business/products/switches/unmanaged/GS105.aspx]]'),
	3693 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 3.4 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_3.4]]'),
	3694 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 4.0 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_4.0]]'),
	3695 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 4.1 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_4.1]]'),
	3696 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 4.2 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_4.2]]'),
	3697 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 4.3 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_4.3]]'),
	3698 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 4.4 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_4.4]]'),
	3699 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 5.0 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_5.0]]'),
	3700 => array ('chapter_id' => 13, 'dict_value' => '[[PROXMOX%GSKIP%Proxmox VE 5.1 | http://pve.proxmox.com/wiki/Roadmap#Proxmox_VE_5.1]]'),
	3701 => array ('chapter_id' => 12, 'dict_value' => '[[TP-Link%GPASS%T1600G-18TS | https://www.tp-link.com/en/products/details/cat-40_T1600G-18TS.html]]'),
	3702 => array ('chapter_id' => 27, 'dict_value' => '[[Raritan%GPASS%PX3-5514U | http://cdn.raritan.com/product-selector/pdus/PX3-5514U/MPX3-5514U.pdf]]'),
	3703 => array ('chapter_id' => 12, 'dict_value' => '[[HP Aruba%GPASS%3810M 16SFP+ 2-slot (JL075A) | http://duckduckgo.com/?q=JL075A+manual ]]'),
	3704 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 6.5'),
	3705 => array ('chapter_id' => 17, 'dict_value' => '[[ Fortinet%GPASS%Fortigate 600D | http://www.fortinet.com/content/dam/fortinet/assets/data-sheets/FortiGate_600D.pdf ]]'),
	3706 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 18.04 LTS'),
	3707 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS328-4C-20S-4S+RM'),
	3708 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS328-24P-4S+RM'),
	3709 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 9 (Stretch) | http://debian.org/releases/stretch/]]'),
	3710 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7170-32C'),
	3711 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7170-64C'),
	3712 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7060SX2-48YC6'),
	3713 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7260CX3-64'),
	3714 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050CX3-32S'),
	3715 => array ('chapter_id' => 12, 'dict_value' => 'Arista%GPASS%7050SX3-48YC12'),
	3716 => array ('chapter_id' => 12, 'dict_value' => 'Huawei%GPASS%CE6865-48S8CQ-EI'),
	3717 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R320'),
	3718 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R330'),
	3719 => array ('chapter_id' => 11, 'dict_value' => 'Dell PowerEdge%GPASS%R740xd'),
	3720 => array ('chapter_id' => 14, 'dict_value' => 'HP Procurve OS N.11.78'),
	3721 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP Aruba 2530 48 PoE+ Switch | https://www.hpe.com/us/en/product-catalog/networking/networking-switches/pip.specifications.aruba-2530-48-poeplus-switch.5384996.html]]'),
	3722 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP Aruba 2530 24 PoE+ Switch | https://www.hpe.com/uk/en/product-catalog/networking/networking-switches/pip.specifications.aruba-2530-24-poeplus-switch.5384999.html]]'),
	3723 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP 1950 48G 2SFP+ 2XGT Switch | https://www.hpe.com/us/en/product-catalog/networking/networking-switches/pip.specifications.hpe-officeconnect-1950-48g-2sfpplus-2xgt-poeplus-switch.6887601.html]]'),
	3724 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP FlexFabric 5900AF 48XG 4QSFP+ Switch | https://www.hpe.com/us/en/product-catalog/networking/networking-switches/pip.specifications.hpe-flexfabric-5900af-48xg-4qsfpplus-switch.5223200.html]]'),
	3725 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HPE 5500-24G-4SFP | https://h20195.www2.hpe.com/v2/default.aspx?cc=az&lc=az&oid=5195377]]'),
	3726 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP A5800AF-48G Switch with 2 Processors (JG225A) | https://www.hpe.com/us/en/product-catalog/networking/networking-switches/pip.specifications.hpe-flexfabric-5800af-48g-switch.7482188.html]]'),
	3727 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP 1810-8G v2 (J9802A)]]'),
	3728 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP ProCurve 5406zl (J8697A) | http://www.hp.com/hpinfo/newsroom/press_kits/2010/HPOptimizesAppDelivery/E5400zl_Switch_Series_Data_Sheet.pdf]]'),
	3729 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP 1810-8G v2 (J9449A) | https://h10057.www1.hp.com/ecomcat/hpcatalog/specs/provisioner/99/J9449A.htm]]'),
	3730 => array ('chapter_id' => 12, 'dict_value' => 'HP ProCurve%GPASS%1810G-24 (J9803A)'),
	3731 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%Cisco 871'),
	3732 => array ('chapter_id' => 12, 'dict_value' => '[[HP%GPASS%HP A5120-24G EI (JE068A) | [[https://h20195.www2.hpe.com/v2/GetDocument.aspx?docname=c04111657&doctype=quickspecs&doclang=EN_US&searchquery=&cc=za&lc=en]]'),
	3733 => array ('chapter_id' => 16, 'dict_value' => 'OpenWrt 17.01'),
	3734 => array ('chapter_id' => 16, 'dict_value' => 'OpenWrt 18.06'),
	3735 => array ('chapter_id' => 37, 'dict_value' => 'OpenWrt 17.01'),
	3736 => array ('chapter_id' => 37, 'dict_value' => 'OpenWrt 18.06'),
	3737 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 27'),
	3738 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 28'),
	3739 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 29'),
	3740 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 30'),
	3741 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 31'),
	3742 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 18.10'),
	3743 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 19.04'),
	3744 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 19.10'),
	3745 => array ('chapter_id' => 13, 'dict_value' => '[[Debian%GSKIP%Debian 10 | https://www.debian.org/releases/buster/]]'),
	3746 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 20.04 LTS'),
	3747 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 32'),
	3748 => array ('chapter_id' => 12, 'dict_value' => '[[HP EI%GPASS%A5120-48G-PoE+ (JG237A) | [[https://h20195.www2.hpe.com/v2/GetDocument.aspx?docname=c04111657&doctype=quickspecs&doclang=EN_US&searchquery=&cc=za&lc=en]]'),
	3749 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS728TPv2'),
	3750 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS728TPPv2'),
	3751 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS752TPv2'),
	3752 => array ('chapter_id' => 12, 'dict_value' => 'NETGEAR%GPASS%GS752TPP'),
	3753 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 7.2'),
	3754 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 8.0'),
	3755 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 8.1'),
	3756 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 8.2'),
	3757 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 9.0'),
	3758 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 9.1'),
	3759 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.2'),
	3760 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.3'),
	3761 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.4'),
	3762 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.5'),
	3763 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.6'),
	3764 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.7'),
	3765 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.8'),
	3766 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 6.9'),
	3767 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 12.0'),
	3768 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 12.1'),
	3769 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 12.2'),
	3770 => array ('chapter_id' => 13, 'dict_value' => 'FreeBSD%GSKIP%FreeBSD 13.0'),
	3771 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 33'),
	3772 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 34'),
	3773 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 20.10'),
	3774 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 21.04'),
	3775 => array ('chapter_id' => 13, 'dict_value' => 'Ubuntu%GSKIP%Ubuntu 21.10'),
	3776 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 6.7'),
	3777 => array ('chapter_id' => 13, 'dict_value' => 'VMWare Hypervisor%GSKIP%VMware ESXi 7.0'),
	3778 => array ('chapter_id' => 13, 'dict_value' => '[[CentOS%GSKIP%CentOS V8 | http://www.centos.org/]]'),
	3779 => array ('chapter_id' => 13, 'dict_value' => 'Red Hat Enterprise%GSKIP%RHEL V8'),
	3780 => array ('chapter_id' => 13, 'dict_value' => '[[AlmaLinux%GSKIP%AlmaLinux V8 | https://almalinux.org/]]'),
	3781 => array ('chapter_id' => 13, 'dict_value' => '[[Oracle Linux%GSKIP%Oracle Linux V7 | https://www.oracle.com/linux/]]'),
	3782 => array ('chapter_id' => 13, 'dict_value' => '[[Oracle Linux%GSKIP%Oracle Linux V8 | https://www.oracle.com/linux/]]'),
	3783 => array ('chapter_id' => 13, 'dict_value' => 'XCP-ng 8.0'),
	3784 => array ('chapter_id' => 13, 'dict_value' => 'XCP-ng 8.1'),
	3785 => array ('chapter_id' => 13, 'dict_value' => 'XCP-ng 8.2'),
	3786 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows Server 2019'),
	3787 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows 7'),
	3788 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows 8'),
	3789 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows 8.1'),
	3790 => array ('chapter_id' => 13, 'dict_value' => 'MicroSoft%GSKIP%Windows 10'),
	3791 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS354-48G-4S+2Q+RM'),
	3792 => array ('chapter_id' => 12, 'dict_value' => 'MikroTik%GPASS%CRS354-48P-4S+2Q+RM'),
	3793 => array ('chapter_id' => 12, 'dict_value' => 'Cisco%GPASS%ME-C3750-24TE'),
	3794 => array ('chapter_id' => 12, 'dict_value' => 'Nortel%GPASS%Avaya 4548GT-PWR'),
	3795 => array ('chapter_id' => 24, 'dict_value' => 'Juniper%GPASS%SRX100'),
	3796 => array ('chapter_id' => 13, 'dict_value' => 'RH Fedora%GSKIP%Fedora 35'),
	3797 => array ('chapter_id' => 13, 'dict_value' => 'NetBSD%GSKIP%NetBSD 9.2'),
	3798 => array ('chapter_id' => 13, 'dict_value' => 'OpenBSD%GSKIP%OpenBSD 7.0'),
	3799 => array ('chapter_id' => 13, 'dict_value' => 'Debian%GSKIP%Debian 11'),

# Any new "default" dictionary records must go above this line (i.e., with
# dict_key code less than 50000). This is necessary to keep AUTO_INCREMENT
# and dictionary updates working properly.
	49999 => array ('chapter_id' => 13, 'dict_value' => '[[RH Fedora%GSKIP%Fedora 15 | http://docs.fedoraproject.org/release-notes/f15/en-US/html/]]'),
);
