"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssistantPlugin = void 0;
var _operators = require("rxjs/operators");
var _basic_input_output_parser = require("./parsers/basic_input_output_parser");
var _visualization_card_parser = require("./parsers/visualization_card_parser");
var _chat_routes = require("./routes/chat_routes");
var _text2viz_routes = require("./routes/text2viz_routes");
var _assistant_service = require("./services/assistant_service");
var _agent_routes = require("./routes/agent_routes");
var _summary_routes = require("./routes/summary_routes");
var _capabilities_provider = require("./vis_type_nlq/capabilities_provider");
var _saved_object_type = require("./vis_type_nlq/saved_object_type");
var _capabilities = require("./capabilities");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class AssistantPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "messageParsers", []);
    _defineProperty(this, "assistantService", new _assistant_service.AssistantService());
    this.logger = initializerContext.logger.get();
  }
  async setup(core) {
    this.logger.debug('Assistant: Setup');
    const config = await this.initializerContext.config.create().pipe((0, _operators.first)()).toPromise();
    const assistantServiceSetup = this.assistantService.setup();
    const router = core.http.createRouter();
    core.http.registerRouteHandlerContext('assistant_plugin', () => {
      return {
        config,
        logger: this.logger
      };
    });
    (0, _agent_routes.registerAgentRoutes)(router, assistantServiceSetup);

    // Register server side APIs
    (0, _chat_routes.registerChatRoutes)(router, {
      messageParsers: this.messageParsers,
      auth: core.http.auth
    });

    // Register router for text to visualization
    if (config.text2viz.enabled) {
      (0, _text2viz_routes.registerText2VizRoutes)(router, assistantServiceSetup);
      core.capabilities.registerProvider(_capabilities_provider.capabilitiesProvider);
      core.savedObjects.registerType(_saved_object_type.visNLQSavedObjectType);
    }

    // Register router for alert insight
    if (config.alertInsight.enabled) {
      (0, _summary_routes.registerSummaryAssistantRoutes)(router, assistantServiceSetup);
    }
    core.capabilities.registerProvider(_capabilities.capabilitiesProvider);

    // Register router for discovery summary
    (0, _summary_routes.registerData2SummaryRoutes)(router, assistantServiceSetup);
    const registerMessageParser = messageParser => {
      const findItem = this.messageParsers.find(item => item.id === messageParser.id);
      if (findItem) {
        throw new Error(`There is already a messageParser whose id is ${messageParser.id}`);
      }
      this.messageParsers.push(messageParser);
    };
    registerMessageParser(_basic_input_output_parser.BasicInputOutputParser);
    registerMessageParser(_visualization_card_parser.VisualizationCardParser);
    return {
      assistantService: assistantServiceSetup,
      registerMessageParser,
      removeMessageParser: parserId => {
        const findIndex = this.messageParsers.findIndex(item => item.id === parserId);
        if (findIndex < 0) {
          this.logger.error(`There is not a messageParser whose id is ${parserId}`);
        }
        this.messageParsers.splice(findIndex, 1);
      }
    };
  }
  start(core) {
    this.logger.debug('Assistant: Started');
    this.assistantService.start();
    return {};
  }
  stop() {
    this.assistantService.stop();
  }
}
exports.AssistantPlugin = AssistantPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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