# This file is generated by gendocstrings - edit that
import sys

from typing import Optional, Callable, Any, Iterator, Iterable, Sequence, Literal, Protocol, TypeAlias, final
from collections.abc import Mapping
import array
import types

SQLiteValue = None | int | float | bytes | str
"""SQLite supports 5 types - None (NULL), 64 bit signed int, 64 bit
float, bytes, and str (unicode text)"""

SQLiteValues = tuple[SQLiteValue, ...]
"A sequence of zero or more SQLiteValue"

Bindings = Sequence[SQLiteValue | zeroblob] | Mapping[str, SQLiteValue | zeroblob]
"""Query bindings are either a sequence of SQLiteValue, or a dict mapping names
to SQLiteValues.  You can also provide zeroblob in Bindings. You can use
dict subclasses or any type registered with :class:`collections.abc.Mapping`
for named bindings"""


class AggregateClass(Protocol):
    "Represents a running aggregate function"

    def step(self, *values: SQLiteValue) -> None:
        "Called with value(s) from a matching row"
        ...

    def final(self) -> SQLiteValue:
        "Called after all matching rows have been processed to get the final value"
        ...


# Neither TypeVar nor ParamSpec work, when either should
AggregateT = Any
"An object provided as first parameter of step and final aggregate functions"

AggregateStep = Callable[[AggregateT, *SQLiteValues], None]

"AggregateStep is called on each matching row with the relevant number of SQLiteValue"

AggregateFinal = Callable[[AggregateT], SQLiteValue]
"Final is called after all matching rows have been processed by step, and returns a SQLiteValue"

AggregateFactory = Callable[[], AggregateClass | tuple[AggregateT, AggregateStep, AggregateFinal]]
"""Called each time for the start of a new calculation using an aggregate function,
returning an object, a step function and a final function"""

ScalarProtocol = Callable[[*SQLiteValues], SQLiteValue]

"""Scalar callbacks take zero or more SQLiteValues, and return a SQLiteValue"""


class WindowClass(Protocol):
    "Represents a running window function"

    def step(self, param: SQLiteValue) -> None:
        "Adds the param(s) to the window"
        ...

    def final(self) -> SQLiteValue:
        "Finishes the function and returns final value"
        ...

    def value(self) -> SQLiteValue:
        "Returns the current value"
        ...

    def inverse(self, param: SQLiteValue) -> None:
        "Removes the param(s) from the window"
        ...


WindowT = Any
"An object provided as first parameter of the 4 window functions, if not using class based callbacks"

WindowStep = Callable[[WindowT, *SQLiteValues], None]
"""Window function step takes zero or more SQLiteValues"""

WindowFinal =    Callable[[WindowT, *SQLiteValues], SQLiteValue]
"""Window function final takes zero or more SQLiteValues, and returns a SQLiteValue"""

WindowValue = Callable[[WindowT], SQLiteValue]
"""Window function value returns the current  SQLiteValue"""

WindowInverse =  Callable[[WindowT, *SQLiteValues], None]
"""Window function inverse takes zero or more SQLiteValues"""

WindowFactory = Callable[[], WindowClass | tuple[WindowT, WindowStep, WindowFinal, WindowValue, WindowInverse]]
"""Called each time at the start of a new window function execution.  It should return either an object
with relevant methods or an object used as the first parameter and the 4 methods"""

RowTracer = Callable[[Cursor, SQLiteValues], Any]
"""Row tracers are called with the Cursor, and the row that would
be returned.  If you return None, then no row is returned, otherwise
whatever is returned is returned as a result row for the query"""

ExecTracer = Callable[[Cursor, str, Optional[Bindings]], bool]
"""Execution tracers are called with the cursor, sql query text, and the bindings
used.  Return False/None to abort execution, or True to continue"""

Authorizer = Callable[[int, Optional[str], Optional[str], Optional[str], Optional[str]], int]
"""Authorizers are called with an operation code and 4 strings (which could be None) depending
on the operatation.  Return SQLITE_OK, SQLITE_DENY, or SQLITE_IGNORE"""

CommitHook = Callable[[], bool]
"""Commit hook is called with no arguments and should return True to abort the commit and False
to let it continue"""

TokenizerResult = Iterable[str | tuple[str, ...] | tuple[int, int, *tuple[str, ...]]]
"""The return from a tokenizer is based on the include_offsets and
include_colocated parameters you provided, both defaulting to
``True``.  See :meth:`FTS5Tokenizer.__call__` for examples.
"""

Tokenizer = Callable[[bytes, int, str | None], TokenizerResult]
"""The tokenizer is called with UTF8 encoded bytes, int flags, and a locale."""

FTS5TokenizerFactory = Callable[[Connection, list[str]], Tokenizer]
"""The factory is called with a list of strings as an argument and should
return a suitably configured Tokenizer"""

FTS5Function = Callable[[FTS5ExtensionApi, *SQLiteValues], SQLiteValue]
"""The first argument is the extension API while the rest are the function parameters
from the SQL"""

FTS5QueryPhrase = Callable[[FTS5ExtensionApi, Any], None]
"""Callback from :meth:`FTS5ExtensionApi.query_phrase`"""

SQLITE_VERSION_NUMBER: int
"""The integer version number of SQLite that APSW was compiled
against.  For example SQLite 3.44.1 will have the value *3440100*.
This number may be different than the actual library in use if the
library is shared and has been updated.  Call
:meth:`sqlite_lib_version` to get the actual library version."""

def allow_missing_dict_bindings(value: bool) -> bool:
    """Changes how missing bindings are handled when using a :class:`dict`.
    Historically missing bindings were treated as *None*.  It was
    anticipated that dict bindings would be used when there were lots
    of columns, so having missing ones defaulting to *None* was
    convenient.

    Unfortunately this also has the side effect of not catching typos
    and similar issues.

    APSW 3.41.0.0 changed the default so that missing dict entries
    will result in an exception.  Call this with *True* to restore
    the earlier behaviour, and *False* to have an exception.

    The previous value is returned."""
    ...

def apsw_version() -> str:
    """Returns the APSW version."""
    ...

apswversion = apsw_version ## OLD-NAME

compile_options: tuple[str, ...]
"""A tuple of the options used to compile SQLite.  For example it
will be something like this, but with around 50 entries::

    ('ENABLE_LOCKING_STYLE=0', 'TEMP_STORE=1', 'THREADSAFE=1', 'ENABLE_FTS5',
     'OMIT_SHARED_CACHE', 'SYSTEM_MALLOC')

Calls: `sqlite3_compileoption_get <https://sqlite.org/c3ref/compileoption_get.html>`__"""

def complete(statement: str) -> bool:
    """Returns True if the input string comprises one or more complete SQL
    statements by looking for an unquoted trailing semi-colon.  It does
    not consider comments or blank lines to be complete.

    An example use would be if you were prompting the user for SQL
    statements and needed to know if you had a whole statement, or
    needed to ask for another line::

      statement = input("SQL> ")
      while not apsw.complete(statement):
         more = input("  .. ")
         statement = statement + "\\n" + more

    Calls: `sqlite3_complete <https://sqlite.org/c3ref/complete.html>`__"""
    ...

def config(op: int, *args: Any) -> None:
    """:param op: A `configuration operation <https://sqlite.org/c3ref/c_config_chunkalloc.html>`_
    :param args: Zero or more arguments as appropriate for *op*

    Some operations don't make sense from a Python program.  All the
    remaining are supported.

    Calls: `sqlite3_config <https://sqlite.org/c3ref/config.html>`__"""
    ...

connection_hooks: list[Callable[[Connection], None]]
"""The purpose of the hooks is to allow the easy registration of
:meth:`functions <Connection.create_scalar_function>`,
:ref:`virtual tables <virtualtables>` or similar items with
each :class:`Connection` as it is created. The default value is an empty
list. Whenever a Connection is created, each item in
apsw.connection_hooks is invoked with a single parameter being
the new Connection object. If the hook raises an exception then
the creation of the Connection fails."""

def connections() -> list[Connection]:
    """Returns a list of the connections"""
    ...

def enable_shared_cache(enable: bool) -> None:
    """`Discouraged
    <https://sqlite.org/sharedcache.html#use_of_shared_cache_is_discouraged>`__.

    Calls: `sqlite3_enable_shared_cache <https://sqlite.org/c3ref/enable_shared_cache.html>`__"""
    ...

enablesharedcache = enable_shared_cache ## OLD-NAME

def exception_for(code: int) -> Exception:
    """If you would like to raise an exception that corresponds to a
    particular SQLite `error code
    <https://sqlite.org/c3ref/c_abort.html>`_ then call this function.
    It also understands `extended error codes
    <https://sqlite.org/c3ref/c_ioerr_access.html>`_.

    For example to raise `SQLITE_IOERR_ACCESS <https://sqlite.org/c3ref/c_ioerr_access.html>`_::

      raise apsw.exception_for(apsw.SQLITE_IOERR_ACCESS)"""
    ...

exceptionfor = exception_for ## OLD-NAME

def fork_checker() -> None:
    """**Note** This method is not available on Windows as it does not
    support the fork system call.

    SQLite does not allow the use of database connections across `forked
    <https://en.wikipedia.org/wiki/Fork_(operating_system)>`__ processes
    (see the `SQLite FAQ Q6 <https://sqlite.org/faq.html#q6>`__).
    (Forking creates a child process that is a duplicate of the parent
    including the state of all data structures in the program.  If you
    do this to SQLite then parent and child would both consider
    themselves owners of open databases and silently corrupt each
    other's work and interfere with each other's locks.)

    One example of how you may end up using fork is if you use the
    :mod:`multiprocessing module <multiprocessing>` which can use
    fork to make child processes.

    If you do use fork or multiprocessing on a platform that supports fork
    then you **must** ensure database connections and their objects
    (cursors, backup, blobs etc) are not used in the parent process, or
    are all closed before calling fork or starting a `Process
    <https://docs.python.org/3/library/multiprocessing.html#process-and-exceptions>`__.
    (Note you must call close to ensure the underlying SQLite objects are
    closed.  It is also a good idea to call :func:`gc.collect(2)
    <gc.collect>` to ensure anything you may have missed is also
    deallocated.)

    Once you run this method, extra checking code is inserted into
    SQLite's mutex operations (at a very small performance penalty) that
    verifies objects are not used across processes.  You will get a
    :exc:`ForkingViolationError` if you do so.  Note that due to the way
    Python's internals work, the exception will be delivered to
    :func:`sys.excepthook` in addition to the normal exception mechanisms and
    may be reported by Python after the line where the issue actually
    arose.  (Destructors of objects you didn't close also run between
    lines.)

    You should only call this method as the first line after importing
    APSW, as it has to shutdown and re-initialize SQLite.  If you have
    any SQLite objects already allocated when calling the method then
    the program will later crash.  The recommended use is to use the fork
    checking as part of your test suite."""
    ...

def format_sql_value(value: SQLiteValue) -> str:
    """Returns a Python string representing the supplied value in SQLite
    syntax.

    Note that SQLite represents floating point `Nan
    <https://en.wikipedia.org/wiki/NaN>`__ as :code:`NULL`, infinity as
    :code:`1e999` and loses the sign on `negative zero
    <https://en.wikipedia.org/wiki/Signed_zero>`__."""
    ...

def hard_heap_limit(limit: int) -> int:
    """Enforces SQLite keeping memory usage below *limit* bytes and
    returns the previous limit.

    .. seealso::

        :meth:`soft_heap_limit`

    Calls: `sqlite3_hard_heap_limit64 <https://sqlite.org/c3ref/hard_heap_limit64.html>`__"""
    ...

def initialize() -> None:
    """It is unlikely you will want to call this method as SQLite automatically initializes.

    Calls: `sqlite3_initialize <https://sqlite.org/c3ref/initialize.html>`__"""
    ...

keywords: set[str]
"""A set containing every SQLite keyword

Calls:
  * `sqlite3_keyword_count <https://sqlite.org/c3ref/keyword_check.html>`__
  * `sqlite3_keyword_name <https://sqlite.org/c3ref/keyword_check.html>`__"""

def log(errorcode: int, message: str) -> None:
    """Calls the SQLite logging interface.  You must format the
    message before passing it to this method::

        apsw.log(apsw.SQLITE_NOMEM, f"Need { needed } bytes of memory")

    Calls: `sqlite3_log <https://sqlite.org/c3ref/log.html>`__"""
    ...

def memory_high_water(reset: bool = False) -> int:
    """Returns the maximum amount of memory SQLite has used.  If *reset* is
    True then the high water mark is reset to the current value.

    .. seealso::

      :meth:`status`

    Calls: `sqlite3_memory_highwater <https://sqlite.org/c3ref/memory_highwater.html>`__"""
    ...

memoryhighwater = memory_high_water ## OLD-NAME

def memory_used() -> int:
    """Returns the amount of memory SQLite is currently using.

    .. seealso::
      :meth:`status`


    Calls: `sqlite3_memory_used <https://sqlite.org/c3ref/memory_highwater.html>`__"""
    ...

memoryused = memory_used ## OLD-NAME

no_change: object
"""A sentinel value used to indicate no change in a value when
used with :meth:`VTCursor.ColumnNoChange` and
:meth:`VTTable.UpdateChangeRow`"""

def pyobject(object: Any):
    """Indicates a Python object is being provided as a
    :ref:`runtime value <pyobject>`."""
    ...

def randomness(amount: int)  -> bytes:
    """Gets random data from SQLite's random number generator.

    :param amount: How many bytes to return

    Calls: `sqlite3_randomness <https://sqlite.org/c3ref/randomness.html>`__"""
    ...

def release_memory(amount: int) -> int:
    """Requests SQLite try to free *amount* bytes of memory.  Returns how
    many bytes were freed.

    Calls: `sqlite3_release_memory <https://sqlite.org/c3ref/release_memory.html>`__"""
    ...

releasememory = release_memory ## OLD-NAME

def set_default_vfs(name: str) -> None:
    """Sets the default vfs to *name* which must be an existing vfs.
    See :meth:`vfs_names`.

    Calls:
      * `sqlite3_vfs_register <https://sqlite.org/c3ref/vfs_find.html>`__
      * `sqlite3_vfs_find <https://sqlite.org/c3ref/vfs_find.html>`__"""
    ...

def shutdown() -> None:
    """It is unlikely you will want to call this method and there is no
    need to do so.  It is a **really** bad idea to call it unless you
    are absolutely sure all :class:`connections <Connection>`,
    :class:`blobs <Blob>`, :class:`cursors <Cursor>`, :class:`vfs <VFS>`
    etc have been closed, deleted and garbage collected.

    Calls: `sqlite3_shutdown <https://sqlite.org/c3ref/initialize.html>`__"""
    ...

def sleep(milliseconds: int) -> int:
    """Sleep for at least the number of `milliseconds`, returning how many
     milliseconds were requested from the operating system.

    Calls: `sqlite3_sleep <https://sqlite.org/c3ref/sleep.html>`__"""
    ...

def soft_heap_limit(limit: int) -> int:
    """Requests SQLite try to keep memory usage below *limit* bytes and
    returns the previous limit.

    .. seealso::

        :meth:`hard_heap_limit`

    Calls: `sqlite3_soft_heap_limit64 <https://sqlite.org/c3ref/hard_heap_limit64.html>`__"""
    ...

softheaplimit = soft_heap_limit ## OLD-NAME

def sqlite3_sourceid() -> str:
    """Returns the exact checkin information for the SQLite 3 source
    being used.

    Calls: `sqlite3_sourceid <https://sqlite.org/c3ref/libversion.html>`__"""
    ...

def sqlite_lib_version() -> str:
    """Returns the version of the SQLite library.  This value is queried at
    run time from the library so if you use shared libraries it will be
    the version in the shared library.

    Calls: `sqlite3_libversion <https://sqlite.org/c3ref/libversion.html>`__"""
    ...

sqlitelibversion = sqlite_lib_version ## OLD-NAME

def status(op: int, reset: bool = False) -> tuple[int, int]:
    """Returns current and highwater measurements.

    :param op: A `status parameter <https://sqlite.org/c3ref/c_status_malloc_size.html>`_
    :param reset: If *True* then the highwater is set to the current value
    :returns: A tuple of current value and highwater value

    .. seealso::

      * :meth:`Connection.status` for statistics about a :class:`Connection`
      * :ref:`Status example <example_status>`

    Calls: `sqlite3_status64 <https://sqlite.org/c3ref/status.html>`__"""
    ...

def strglob(glob: str, string: str) -> int:
    """Does string GLOB matching.  Zero is returned on a match.

    Calls: `sqlite3_strglob <https://sqlite.org/c3ref/strglob.html>`__"""
    ...

def stricmp(string1: str, string2: str) -> int:
    """Does string case-insensitive comparison.  Zero is returned
    on a match.

    Calls: `sqlite3_stricmp <https://sqlite.org/c3ref/stricmp.html>`__"""
    ...

def strlike(glob: str, string: str, escape: int = 0) -> int:
    """Does string LIKE matching.  Zero is returned on a match.

    Calls: `sqlite3_strlike <https://sqlite.org/c3ref/strlike.html>`__"""
    ...

def strnicmp(string1: str, string2: str, count: int) -> int:
    """Does string case-insensitive comparison.  Zero is returned
    on a match.

    Calls: `sqlite3_strnicmp <https://sqlite.org/c3ref/stricmp.html>`__"""
    ...

def unregister_vfs(name: str) -> None:
    """Unregisters the named vfs.  See :meth:`vfs_names`.

    Calls:
      * `sqlite3_vfs_unregister <https://sqlite.org/c3ref/vfs_find.html>`__
      * `sqlite3_vfs_find <https://sqlite.org/c3ref/vfs_find.html>`__"""
    ...

using_amalgamation: bool
"""If True then `SQLite amalgamation
<https://www.sqlite.org/amalgamation.html>`__ is in
use (statically compiled into APSW).  Using the amalgamation means
that SQLite shared libraries are not used and will not affect your
code."""

def vfs_details() -> list[dict[str, int | str]]:
    """Returns a list with details of each :ref:`vfs <vfs>`.  The detail is a
    dictionary with the keys being the names of the `sqlite3_vfs
    <https://sqlite.org/c3ref/vfs.html>`__ data structure, and their
    corresponding values.

    Pointers are converted using :c:func:`PyLong_FromVoidPtr`.

    Calls: `sqlite3_vfs_find <https://sqlite.org/c3ref/vfs_find.html>`__"""
    ...

def vfs_names() -> list[str]:
    """Returns a list of the currently installed :ref:`vfs <vfs>`.  The first
    item in the list is the default vfs.

    Calls: `sqlite3_vfs_find <https://sqlite.org/c3ref/vfs_find.html>`__"""
    ...

vfsnames = vfs_names ## OLD-NAME

@final
class Backup:
    """You create a backup instance by calling :meth:`Connection.backup`."""
    def close(self, force: bool = False) -> None:
        """Does the same thing as :meth:`~Backup.finish`.  This extra api is
        provided to give the same api as other APSW objects and files.
        It is safe to call this method multiple  times.

        :param force: If true then any exceptions are ignored."""
        ...

    done: bool
    """A boolean that is True if the copy completed in the last call to :meth:`~Backup.step`."""

    def __enter__(self) -> Backup:
        """You can use the backup object as a `context manager
        <https://docs.python.org/3/reference/datamodel.html#with-statement-context-managers>`_
        as defined in :pep:`0343`.  The :meth:`~Backup.__exit__` method ensures that backup
        is :meth:`finished <Backup.finish>`."""
        ...

    def __exit__(self, etype: Optional[type[BaseException]], evalue: Optional[BaseException], etraceback: Optional[types.TracebackType]) -> Optional[bool]:
        """Implements context manager in conjunction with :meth:`~Backup.__enter__` ensuring
        that the copy is :meth:`finished <Backup.finish>`."""
        ...

    def finish(self) -> None:
        """Completes the copy process.  If all pages have been copied then the
        transaction is committed on the destination database, otherwise it
        is rolled back.  This method must be called for your backup to take
        effect.  The backup object will always be finished even if there is
        an exception.  It is safe to call this method multiple times.

        Calls: `sqlite3_backup_finish <https://sqlite.org/c3ref/backup_finish.html#sqlite3backupfinish>`__"""
        ...

    page_count: int
    """Read only. How many pages were in the source database after the last
    step.  If you haven't called :meth:`~Backup.step` or the backup
    object has been :meth:`finished <Backup.finish>` then zero is
    returned.

    Calls: `sqlite3_backup_pagecount <https://sqlite.org/c3ref/backup_finish.html#sqlite3backuppagecount>`__"""

    pagecount = page_count ## OLD-NAME

    remaining: int
    """Read only. How many pages were remaining to be copied after the last
    step.  If you haven't called :meth:`~Backup.step` or the backup
    object has been :meth:`finished <Backup.finish>` then zero is
    returned.

    Calls: `sqlite3_backup_remaining <https://sqlite.org/c3ref/backup_finish.html#sqlite3backupremaining>`__"""

    def step(self, npages: int = -1) -> bool:
        """Copies *npages* pages from the source to destination database.  The source database is locked during the copy so
        using smaller values allows other access to the source database.  The destination database is always locked until the
        backup object is :meth:`finished <Backup.finish>`.

        :param npages: How many pages to copy. If the parameter is omitted
           or negative then all remaining pages are copied.

        This method may throw a :exc:`BusyError` or :exc:`LockedError` if
        unable to lock the source database.  You can catch those and try
        again.

        :returns: True if this copied the last remaining outstanding pages, else False.  This is the same value as :attr:`~Backup.done`

        Calls: `sqlite3_backup_step <https://sqlite.org/c3ref/backup_finish.html#sqlite3backupstep>`__"""
        ...

@final
class Blob:
    """This object is created by :meth:`Connection.blob_open` and provides
    access to a blob in the database.  It behaves like a Python file.
    It wraps a `sqlite3_blob
    <https://sqlite.org/c3ref/blob.html>`_.

    .. note::

      You cannot change the size of a blob using this object. You should
      create it with the correct size in advance either by using
      :class:`zeroblob` or the `zeroblob()
      <https://sqlite.org/lang_corefunc.html>`_ function.

    See the :ref:`example <example_blob_io>`."""
    def close(self, force: bool = False) -> None:
        """Closes the blob.  Note that even if an error occurs the blob is
        still closed.

        .. note::

           In some cases errors that technically occurred in the
           :meth:`~Blob.read` and :meth:`~Blob.write` routines may not be
           reported until close is called.  Similarly errors that occurred
           in those methods (eg calling :meth:`~Blob.write` on a read-only
           blob) may also be re-reported in :meth:`~Blob.close`.  (This
           behaviour is what the underlying SQLite APIs do - it is not APSW
           doing it.)

        It is okay to call :meth:`~Blob.close` multiple times.

        :param force: Ignores any errors during close.

        Calls: `sqlite3_blob_close <https://sqlite.org/c3ref/blob_close.html>`__"""
        ...

    def __enter__(self) -> Blob:
        """You can use a blob as a `context manager
        <https://docs.python.org/3/reference/datamodel.html#with-statement-context-managers>`_
        as defined in :pep:`0343`.  When you use *with* statement,
        the blob is always :meth:`closed <Blob.close>` on exit from the block, even if an
        exception occurred in the block.

        For example::

          with connection.blob_open() as blob:
              blob.write("...")
              res=blob.read(1024)"""
        ...

    def __exit__(self, etype: Optional[type[BaseException]], evalue: Optional[BaseException], etraceback: Optional[types.TracebackType]) -> Optional[bool]:
        """Implements context manager in conjunction with
        :meth:`~Blob.__enter__`.  Any exception that happened in the
        *with* block is raised after closing the blob."""
        ...

    def length(self) -> int:
        """Returns the size of the blob in bytes.

        Calls: `sqlite3_blob_bytes <https://sqlite.org/c3ref/blob_bytes.html>`__"""
        ...

    def read(self, length: int = -1) -> bytes:
        """Reads amount of data requested, or till end of file, whichever is
        earlier. Attempting to read beyond the end of the blob returns an
        empty bytes in the same manner as end of file on normal file
        objects.  Negative numbers read all remaining data.

        Calls: `sqlite3_blob_read <https://sqlite.org/c3ref/blob_read.html>`__"""
        ...

    def read_into(self, buffer: bytearray |  array.array[Any] | memoryview, offset: int = 0, length: int = -1) -> None:
        """Reads from the blob into a buffer you have supplied.  This method is
        useful if you already have a buffer like object that data is being
        assembled in, and avoids allocating results in :meth:`Blob.read` and
        then copying into buffer.

        :param buffer: A writable buffer like object.
                       There is a :class:`bytearray` type that is very useful.
                       :mod:`Arrays <array>` also work.

        :param offset: The position to start writing into the buffer
                       defaulting to the beginning.

        :param length: How much of the blob to read.  The default is the
                       remaining space left in the buffer.  Note that if
                       there is more space available than blob left then you
                       will get a *ValueError* exception.

        Calls: `sqlite3_blob_read <https://sqlite.org/c3ref/blob_read.html>`__"""
        ...

    readinto = read_into ## OLD-NAME

    def reopen(self, rowid: int) -> None:
        """Change this blob object to point to a different row.  It can be
        faster than closing an existing blob an opening a new one.

        Calls: `sqlite3_blob_reopen <https://sqlite.org/c3ref/blob_reopen.html>`__"""
        ...

    def seek(self, offset: int, whence: int = 0) -> None:
        """Changes current position to *offset* biased by *whence*.

        :param offset: New position to seek to.  Can be positive or negative number.
        :param whence: Use 0 if *offset* is relative to the beginning of the blob,
                       1 if *offset* is relative to the current position,
                       and 2 if *offset* is relative to the end of the blob.
        :raises ValueError: If the resulting offset is before the beginning (less than zero) or beyond the end of the blob."""
        ...

    def tell(self) -> int:
        """Returns the current offset."""
        ...

    def write(self, data: bytes) -> None:
        """Writes the data to the blob.

        :param data: bytes to write

        :raises TypeError: Wrong data type

        :raises ValueError: If the data would go beyond the end of the blob.
            You cannot increase the size of a blob by writing beyond the end.
            You need to use :class:`zeroblob` to set the desired size first when
            inserting the blob.

        Calls: `sqlite3_blob_write <https://sqlite.org/c3ref/blob_write.html>`__"""
        ...

class Connection:
    """This object wraps a `sqlite3 pointer
    <https://sqlite.org/c3ref/sqlite3.html>`_."""
    authorizer: Optional[Authorizer]
    """While `preparing <https://sqlite.org/c3ref/prepare.html>`_
    statements, SQLite will call any defined authorizer to see if a
    particular action is ok to be part of the statement.

    Typical usage would be if you are running user supplied SQL and want
    to prevent harmful operations.  You should also
    set the :class:`statementcachesize <Connection>` to zero.

    The authorizer callback has 5 parameters:

      * An `operation code <https://sqlite.org/c3ref/c_alter_table.html>`_
      * A string (or None) dependent on the operation `(listed as 3rd) <https://sqlite.org/c3ref/c_alter_table.html>`_
      * A string (or None) dependent on the operation `(listed as 4th) <https://sqlite.org/c3ref/c_alter_table.html>`_
      * A string name of the database (or None)
      * Name of the innermost trigger or view doing the access (or None)

    The authorizer callback should return one of *SQLITE_OK*,
    *SQLITE_DENY* or *SQLITE_IGNORE*.
    (*SQLITE_DENY* is returned if there is an error in your
    Python code).

    .. seealso::

      * :ref:`Example <example_authorizer>`
      * :ref:`statementcache`

    Calls: `sqlite3_set_authorizer <https://sqlite.org/c3ref/set_authorizer.html>`__"""

    def autovacuum_pages(self, callable: Optional[Callable[[str, int, int, int], int]]) -> None:
        """Calls `callable` to find out how many pages to autovacuum.  The callback has 4 parameters:

        * Database name: str. `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__
        * Database pages: int (how many pages make up the database now)
        * Free pages: int (how many pages could be freed)
        * Page size: int (page size in bytes)

        Return how many pages should be freed.  Values less than zero or more than the free pages are
        treated as zero or free page count.  On error zero is returned.

        .. warning:: READ THE NOTE IN THE SQLITE DOCUMENTATION.

          Calling back into SQLite can result in crashes, corrupt
          databases, or worse.

        Calls: `sqlite3_autovacuum_pages <https://sqlite.org/c3ref/autovacuum_pages.html>`__"""
        ...

    def backup(self, databasename: str, sourceconnection: Connection, sourcedatabasename: str)  -> Backup:
        """Opens a :ref:`backup object <Backup>`.  All data will be copied from source
        database to this database.

        :param databasename: Name of the database. `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__
        :param sourceconnection: The :class:`Connection` to copy a database from.
        :param sourcedatabasename: Name of the database in the source (eg ``main``).

        :rtype: :class:`Backup`

        .. seealso::

          * :doc:`Backup reference <backup>`
          * :ref:`Backup example <example_backup>`

        Calls: `sqlite3_backup_init <https://sqlite.org/c3ref/backup_finish.html#sqlite3backupinit>`__"""
        ...

    def blob_open(self, database: str, table: str, column: str, rowid: int, writeable: bool)  -> Blob:
        """Opens a blob for :ref:`incremental I/O <blobio>`.

        :param database: Name of the database.  `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__.
        :param table: The name of the table
        :param column: The name of the column
        :param rowid: The id that uniquely identifies the row.
        :param writeable: If True then you can read and write the blob.  If False then you can only read it.

        :rtype: :class:`Blob`

        .. seealso::

          * :ref:`Blob I/O example <example_blob_io>`
          * `SQLite row ids <https://sqlite.org/autoinc.html>`_

        Calls: `sqlite3_blob_open <https://sqlite.org/c3ref/blob_open.html>`__"""
        ...

    blobopen = blob_open ## OLD-NAME

    def cache_flush(self) -> None:
        """Flushes caches to disk mid-transaction.

        Calls: `sqlite3_db_cacheflush <https://sqlite.org/c3ref/db_cacheflush.html>`__"""
        ...

    cacheflush = cache_flush ## OLD-NAME

    def cache_stats(self, include_entries: bool = False) -> dict[str, int]:
        """Returns information about the statement cache as dict.

        .. note::

          Calling execute with "select a; select b; insert into c ..." will
          result in 3 cache entries corresponding to each of the 3 queries
          present.

        The returned dictionary has the following information.

        .. list-table::
          :header-rows: 1
          :widths: auto

          * - Key
            - Explanation
          * - size
            - Maximum number of entries in the cache
          * - evictions
            - How many entries were removed (expired) to make space for a newer
              entry
          * - no_cache
            - Queries that had can_cache parameter set to False
          * - hits
            - A match was found in the cache
          * - misses
            - No match was found in the cache, or the cache couldn't be used
          * - no_vdbe
            - The statement was empty (eg a comment) or SQLite took action
              during parsing (eg some pragmas).  These are not cached and also
              included in the misses count
          * - too_big
            - UTF8 query size was larger than considered for caching.  These are also included
              in the misses count.
          * - max_cacheable_bytes
            - Maximum size of query (in bytes of utf8) that will be considered for caching
          * - entries
            - (Only present if `include_entries` is True) A list of the cache entries

        If `entries` is present, then each list entry is a dict with the following information.

        .. list-table::
          :header-rows: 1
          :widths: auto

          * - Key
            - Explanation
          * - query
            - Text of the query itself (first statement only)
          * - prepare_flags
            - Flags passed to `sqlite3_prepare_v3 <https://sqlite.org/c3ref/prepare.html>`__
              for this query
          * - explain
            - The value passed to `sqlite3_stmt_explain <https://sqlite.org/c3ref/stmt_explain.html>`__
              if >= 0
          * - uses
            - How many times this entry has been (re)used
          * - has_more
            - Boolean indicating if there was more query text than
              the first statement"""
        ...

    def changes(self) -> int:
        """Returns the number of database rows that were changed (or inserted
        or deleted) by the most recently completed INSERT, UPDATE, or DELETE
        statement.

        Calls: `sqlite3_changes64 <https://sqlite.org/c3ref/changes.html>`__"""
        ...

    def close(self, force: bool = False) -> None:
        """Closes the database.  If there are any outstanding :class:`cursors
        <Cursor>`, :class:`blobs <Blob>` or :class:`backups <Backup>` then
        they are closed too.  It is normally not necessary to call this
        method as the database is automatically closed when there are no
        more references.  It is ok to call the method multiple times.

        If your user defined functions or collations have direct or indirect
        references to the Connection then it won't be automatically garbage
        collected because of circular referencing that can't be
        automatically broken.  Calling *close* will free all those objects
        and what they reference.

        SQLite is designed to survive power failures at even the most
        awkward moments.  Consequently it doesn't matter if it is closed
        when the process is exited, or even if the exit is graceful or
        abrupt.  In the worst case of having a transaction in progress, that
        transaction will be rolled back by the next program to open the
        database, reverting the database to a know good state.

        If *force* is *True* then any exceptions are ignored.

        Calls: `sqlite3_close <https://sqlite.org/c3ref/close.html>`__"""
        ...

    def collation_needed(self, callable: Optional[Callable[[Connection, str], None]]) -> None:
        """*callable* will be called if a statement requires a `collation
        <https://en.wikipedia.org/wiki/Collation>`_ that hasn't been
        registered. Your callable will be passed two parameters. The first
        is the connection object. The second is the name of the
        collation. If you have the collation code available then call
        :meth:`Connection.create_collation`.

        This is useful for creating collations on demand.  For example you
        may include the `locale <https://en.wikipedia.org/wiki/Locale>`_ in
        the collation name, but since there are thousands of locales in
        popular use it would not be useful to :meth:`prereigster
        <Connection.create_collation>` them all.  Using
        :meth:`~Connection.collation_needed` tells you when you need to
        register them.

        .. seealso::

          * :meth:`~Connection.create_collation`

        Calls: `sqlite3_collation_needed <https://sqlite.org/c3ref/collation_needed.html>`__"""
        ...

    collationneeded = collation_needed ## OLD-NAME

    def column_metadata(self, dbname: Optional[str], table_name: str, column_name: str) -> tuple[str, str, bool, bool, bool]:
        """`dbname` is `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__, or None to search
        all databases.

        The returned :class:`tuple` has these fields:

        0: str - declared data type

        1: str - name of default collation sequence

        2: bool - True if not null constraint

        3: bool - True if part of primary key

        4: bool - True if column is `autoincrement <https://www.sqlite.org/autoinc.html>`__

        Calls: `sqlite3_table_column_metadata <https://sqlite.org/c3ref/table_column_metadata.html>`__"""
        ...

    def config(self, op: int, *args: int) -> int:
        """:param op: A `configuration operation
          <https://sqlite.org/c3ref/c_dbconfig_enable_fkey.html>`__
        :param args: Zero or more arguments as appropriate for *op*

        This is how to get the fkey setting::

          val = db.config(apsw.SQLITE_DBCONFIG_ENABLE_FKEY, -1)

        A parameter of zero would turn it off, 1 turns on, and negative
        leaves unaltered.  The effective value is always returned.

        Calls: `sqlite3_db_config <https://sqlite.org/c3ref/db_config.html>`__"""
        ...

    def create_aggregate_function(self, name: str, factory: Optional[AggregateFactory], numargs: int = -1, *, flags: int = 0) -> None:
        """Registers an aggregate function.  Aggregate functions operate on all
        the relevant rows such as counting how many there are.

        :param name: The string name of the function.  It should be less than 255 characters
        :param factory: The function that will be called.  Use None to delete the function.
        :param numargs: How many arguments the function takes, with -1 meaning any number
        :param flags: `Function flags <https://www.sqlite.org/c3ref/c_deterministic.html>`__

        When a query starts, the *factory* will be called.  It can return an object
        with a *step* function called for each matching row, and a *final* function
        to provide the final value.

        Alternatively a non-class approach can return a tuple of 3 items:

          a context object
             This can be of any type

          a step function
             This function is called once for each row.  The first parameter
             will be the context object and the remaining parameters will be
             from the SQL statement.  Any value returned will be ignored.

          a final function
             This function is called at the very end with the context object
             as a parameter.  The value returned is set as the return for
             the function. The final function is always called even if an
             exception was raised by the step function. This allows you to
             ensure any resources are cleaned up.

        .. note::

          You can register the same named function but with different
          callables and *numargs*.  See
          :meth:`~Connection.create_scalar_function` for an example.

        .. seealso::

           * :ref:`Example <example_aggregate>`
           * :meth:`~Connection.create_scalar_function`
           * :meth:`~Connection.create_window_function`

        Calls: `sqlite3_create_function_v2 <https://sqlite.org/c3ref/create_function.html>`__"""
        ...

    createaggregatefunction = create_aggregate_function ## OLD-NAME

    def create_collation(self, name: str, callback: Optional[Callable[[str, str], int]]) -> None:
        """You can control how SQLite sorts (termed `collation
        <https://en.wikipedia.org/wiki/Collation>`_) when giving the
        ``COLLATE`` term to a `SELECT
        <https://sqlite.org/lang_select.html>`_.  For example your
        collation could take into account locale or do numeric sorting.

        The *callback* will be called with two items.  It should return -1
        if the first is less then the second, 0 if they are equal, and 1 if
        first is greater::

           def mycollation(first: str, two: str) -> int:
               if first < second:
                   return -1
               if first == second:
                   return 0
               if first > second:
                   return 1

        Passing None as the callback will unregister the collation.

        .. seealso::

          * :ref:`Example <example_collation>`
          * :meth:`Connection.collation_needed`

        Calls: `sqlite3_create_collation_v2 <https://sqlite.org/c3ref/create_collation.html>`__"""
        ...

    createcollation = create_collation ## OLD-NAME

    def create_module(self, name: str, datasource: Optional[VTModule], *, use_bestindex_object: bool = False, use_no_change: bool = False, iVersion: int = 1, eponymous: bool=False, eponymous_only: bool = False, read_only: bool = False) -> None:
        """Registers a virtual table, or drops it if *datasource* is *None*.
        See :ref:`virtualtables` for details.

        :param name: Module name (CREATE VIRTUAL TABLE table_name USING module_name...)
        :param datasource: Provides :class:`VTModule` methods
        :param use_bestindex_object: If True then BestIndexObject is used, else BestIndex
        :param use_no_change: Turn on understanding :meth:`VTCursor.ColumnNoChange` and using :attr:`apsw.no_change` to reduce :meth:`VTTable.UpdateChangeRow` work
        :param iVersion: iVersion field in `sqlite3_module <https://www.sqlite.org/c3ref/module.html>`__
        :param eponymous: Configures module to be `eponymous <https://www.sqlite.org/vtab.html#eponymous_virtual_tables>`__
        :param eponymous_only: Configures module to be `eponymous only <https://www.sqlite.org/vtab.html#eponymous_only_virtual_tables>`__
        :param read_only: Leaves `sqlite3_module <https://www.sqlite.org/c3ref/module.html>`__ methods that involve writing and transactions as NULL

        .. seealso::

           * :ref:`Example <example_virtual_tables>`

        Calls: `sqlite3_create_module_v2 <https://sqlite.org/c3ref/create_module.html>`__"""
        ...

    createmodule = create_module ## OLD-NAME

    def create_scalar_function(self, name: str, callable: Optional[ScalarProtocol], numargs: int = -1, *, deterministic: bool = False, flags: int = 0) -> None:
        """Registers a scalar function.  Scalar functions operate on one set of parameters once.

        :param name: The string name of the function.  It should be less than 255 characters
        :param callable: The function that will be called.  Use None to unregister.
        :param numargs: How many arguments the function takes, with -1 meaning any number
        :param deterministic: When True this means the function always
                 returns the same result for the same input arguments.
                 SQLite's query planner can perform additional optimisations
                 for deterministic functions.  For example a random()
                 function is not deterministic while one that returns the
                 length of a string is.
        :param flags: Additional `function flags <https://www.sqlite.org/c3ref/c_deterministic.html>`__

        .. note::

          You can register the same named function but with different
          *callable* and *numargs*.  For example::

            connection.create_scalar_function("toip", ipv4convert, 4)
            connection.create_scalar_function("toip", ipv6convert, 16)
            connection.create_scalar_function("toip", strconvert, -1)

          The one with the correct *numargs* will be called and only if that
          doesn't exist then the one with negative *numargs* will be called.

        .. seealso::

           * :ref:`Example <example_scalar>`
           * :meth:`~Connection.create_aggregate_function`
           * :meth:`~Connection.create_window_function`

        Calls: `sqlite3_create_function_v2 <https://sqlite.org/c3ref/create_function.html>`__"""
        ...

    createscalarfunction = create_scalar_function ## OLD-NAME

    def create_window_function(self, name:str, factory: Optional[WindowFactory], numargs: int =-1, *, flags: int = 0) -> None:
        """Registers a `window function
        <https://sqlite.org/windowfunctions.html#user_defined_aggregate_window_functions>`__

        :param name: The string name of the function.  It should be less than 255 characters
        :param factory: Called to start a new window.  Use None to delete the function.
        :param numargs: How many arguments the function takes, with -1 meaning any number
        :param flags: `Function flags <https://www.sqlite.org/c3ref/c_deterministic.html>`__

        You need to provide callbacks for the ``step``, ``final``, ``value``
        and ``inverse`` methods.  This can be done by having `factory` as a
        class, returning an instance with the corresponding method names, or by having `factory`
        return a sequence of a first parameter, and then each of the 4
        functions.

        **Debugging note** SQlite always calls the ``final`` method to allow
        for cleanup.  If you have an exception in one of the other methods, then
        ``final`` will also be called, and you may see both methods in
        tracebacks.

        .. seealso::

         * :ref:`Example <example_window>`
         * :meth:`~Connection.create_scalar_function`
         * :meth:`~Connection.create_aggregate_function`

        Calls: `sqlite3_create_window_function <https://sqlite.org/c3ref/create_function.html>`__"""
        ...

    def cursor(self) -> Cursor:
        """Creates a new :class:`Cursor` object on this database.

        :rtype: :class:`Cursor`"""
        ...

    cursor_factory: Callable[[Connection], Any]
    """Defaults to :class:`Cursor`

    Called with a :class:`Connection` as the only parameter when a cursor
    is needed such as by the :meth:`cursor` method, or
    :meth:`Connection.execute`.

    Note that whatever is returned doesn't have to be an actual
    :class:`Cursor` instance, and just needs to have the methods present
    that are actually called.  These are likely to be `execute`,
    `executemany`, `close` etc."""

    def data_version(self, schema: Optional[str] = None) -> int:
        """Unlike `pragma data_version
        <https://sqlite.org/pragma.html#pragma_data_version>`__ this value
        updates when changes are made by other connections, **AND** this one.

        :param schema: `schema` is `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__,
            defaulting to `main` if not supplied.

        Calls: `sqlite3_file_control <https://sqlite.org/c3ref/file_control.html>`__"""
        ...

    def db_filename(self, name: str) -> str:
        """Returns the full filename of the named (attached) database.  The
        main is `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__

        Calls: `sqlite3_db_filename <https://sqlite.org/c3ref/db_filename.html>`__"""
        ...

    def db_names(self) -> list[str]:
        """Returns the list of database names.  For example the first database
        is named 'main', the next 'temp', and the rest with the name provided
        in `ATTACH <https://www.sqlite.org/lang_attach.html>`__

        Calls: `sqlite3_db_name <https://sqlite.org/c3ref/db_name.html>`__"""
        ...

    def deserialize(self, name: str, contents: bytes) -> None:
        """Replaces the named database with an in-memory copy of *contents*.
        *name* is `main`, `temp`, the name in `ATTACH
        <https://sqlite.org/lang_attach.html>`__

        The resulting database is in-memory, read-write, and the memory is
        owned, resized, and freed by SQLite.

        .. seealso::

          * :meth:`Connection.serialize`

        Calls: `sqlite3_deserialize <https://sqlite.org/c3ref/deserialize.html>`__"""
        ...

    def drop_modules(self, keep: Optional[Iterable[str]]) -> None:
        """If *keep* is *None* then all registered virtual tables are dropped.

        Otherwise *keep* is a sequence of strings, naming the virtual tables that
        are kept, dropping all others."""
        ...

    def enable_load_extension(self, enable: bool) -> None:
        """Enables/disables `extension loading
        <https://www.sqlite.org/loadext.html>`_
        which is disabled by default.

        :param enable: If True then extension loading is enabled, else it is disabled.

        Calls: `sqlite3_enable_load_extension <https://sqlite.org/c3ref/enable_load_extension.html>`__

        .. seealso::

          * :meth:`~Connection.load_extension`"""
        ...

    enableloadextension = enable_load_extension ## OLD-NAME

    def __enter__(self) -> Connection:
        """You can use the database as a `context manager
        <https://docs.python.org/3/reference/datamodel.html#with-statement-context-managers>`_
        as defined in :pep:`0343`.  When you use *with* a transaction is
        started.  If the block finishes with an exception then the
        transaction is rolled back, otherwise it is committed.  For example::

          with connection:
              connection.execute("....")
              with connection:
                  # nested is supported
                  call_function(connection)
                  connection.execute("...")
                  with connection as db:
                      # You can also use 'as'
                      call_function2(db)
                      db.execute("...")

        Behind the scenes `savepoints <https://sqlite.org/lang_savepoint.html>`__
         are used to provide nested transactions."""
        ...

    exec_trace: Optional[ExecTracer]
    """Called with the cursor, statement and bindings for
    each :meth:`~Cursor.execute` or :meth:`~Cursor.executemany` on this
    Connection, unless the :class:`Cursor` installed its own
    tracer. Your execution tracer can also abort execution of a
    statement.

    If *callable* is *None* then any existing execution tracer is
    removed.

    .. seealso::

      * :ref:`tracing`
      * :ref:`rowtracer`
      * :attr:`Cursor.exec_trace`"""

    exectrace = exec_trace ## OLD-NAME

    def execute(self, statements: str, bindings: Optional[Bindings] = None, *, can_cache: bool = True, prepare_flags: int = 0, explain: int = -1) -> Cursor:
        """Executes the statements using the supplied bindings.  Execution
        returns when the first row is available or all statements have
        completed.  (A cursor is automatically obtained).

        For pragmas you should use :meth:`pragma` which handles quoting and
        caching correctly.

        See :meth:`Cursor.execute` for more details, and the :ref:`example <example_executing_sql>`."""
        ...

    def executemany(self, statements: str, sequenceofbindings:Iterable[Bindings], *, can_cache: bool = True, prepare_flags: int = 0, explain: int = -1) -> Cursor:
        """This method is for when you want to execute the same statements over a
        sequence of bindings, such as inserting into a database.  (A cursor is
        automatically obtained).

        See :meth:`Cursor.executemany` for more details, and the :ref:`example <example_executemany>`."""
        ...

    def __exit__(self, etype: Optional[type[BaseException]], evalue: Optional[BaseException], etraceback: Optional[types.TracebackType]) -> Optional[bool]:
        """Implements context manager in conjunction with
        :meth:`~Connection.__enter__`.  If no exception happened then
        the pending transaction is committed, while an exception results in a
        rollback."""
        ...

    def file_control(self, dbname: str, op: int, pointer: int) -> bool:
        """Calls the :meth:`~VFSFile.xFileControl` method on the :ref:`VFS`
        implementing :class:`file access <VFSFile>` for the database.

        :param dbname: The name of the database to affect.  `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__
        :param op: A `numeric code
          <https://sqlite.org/c3ref/c_fcntl_lockstate.html>`_ with values less
          than 100 reserved for SQLite internal use.
        :param pointer: A number which is treated as a ``void pointer`` at the C level.

        :returns: True or False indicating if the VFS understood the op.

        The :ref:`example <example_filecontrol>` shows getting
        `SQLITE_FCNTL_DATA_VERSION
        <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html#sqlitefcntldataversion>`__.

        If you want data returned back then the *pointer* needs to point to
        something mutable.  Here is an example using :mod:`ctypes` of
        passing a Python dictionary to :meth:`~VFSFile.xFileControl` which
        can then modify the dictionary to set return values::

          obj={"foo": 1, 2: 3}                 # object we want to pass
          objwrap=ctypes.py_object(obj)        # objwrap must live before and after the call else
                                               # it gets garbage collected
          connection.file_control(
                   "main",                     # which db
                   123,                        # our op code
                   ctypes.addressof(objwrap))  # get pointer

        The :meth:`~VFSFile.xFileControl` method then looks like this::

          def xFileControl(self, op, pointer):
              if op==123:                      # our op code
                  obj=ctypes.py_object.from_address(pointer).value
                  # play with obj - you can use id() to verify it is the same
                  print(obj["foo"])
                  obj["result"]="it worked"
                  return True
              else:
                  # pass to parent/superclass
                  return super().xFileControl(op, pointer)

        This is how you set the chunk size by which the database grows.  Do
        not combine it into one line as the c_int would be garbage collected
        before the file control call is made::

           chunksize=ctypes.c_int(32768)
           connection.file_control("main", apsw.SQLITE_FCNTL_CHUNK_SIZE, ctypes.addressof(chunksize))

        Calls: `sqlite3_file_control <https://sqlite.org/c3ref/file_control.html>`__"""
        ...

    filecontrol = file_control ## OLD-NAME

    filename: str
    """The filename of the database.

    Calls: `sqlite3_db_filename <https://sqlite.org/c3ref/db_filename.html>`__"""

    filename_journal: str
    """The journal filename of the database,

    Calls: `sqlite3_filename_journal <https://sqlite.org/c3ref/filename_database.html>`__"""

    filename_wal: str
    """The WAL filename of the database,

    Calls: `sqlite3_filename_wal <https://sqlite.org/c3ref/filename_database.html>`__"""

    def fts5_tokenizer(self, name: str, args: list[str] | None = None) -> FTS5Tokenizer:
        """Returns the named tokenizer initialized with ``args``.  Names are case insensitive.

        .. seealso::

            * :meth:`register_fts5_tokenizer`
            * :doc:`textsearch`"""
        ...

    def fts5_tokenizer_available(self, name: str) -> bool:
        """Checks if the named tokenizer is registered.

        .. seealso::

            * :meth:`fts5_tokenizer`
            * :doc:`textsearch`
            * `FTS5 documentation <https://www.sqlite.org/fts5.html#custom_tokenizers>`__"""
        ...

    def get_autocommit(self) -> bool:
        """Returns if the Connection is in auto commit mode (ie not in a transaction).

        Calls: `sqlite3_get_autocommit <https://sqlite.org/c3ref/get_autocommit.html>`__"""
        ...

    getautocommit = get_autocommit ## OLD-NAME

    def get_exec_trace(self) -> Optional[ExecTracer]:
        """Returns the currently installed :attr:`execution tracer
        <Connection.exec_trace>`"""
        ...

    getexectrace = get_exec_trace ## OLD-NAME

    def get_row_trace(self) -> Optional[RowTracer]:
        """Returns the currently installed :attr:`row tracer
        <Connection.row_trace>`"""
        ...

    getrowtrace = get_row_trace ## OLD-NAME

    in_transaction: bool
    """True if currently in a transaction, else False

    Calls: `sqlite3_get_autocommit <https://sqlite.org/c3ref/get_autocommit.html>`__"""

    def __init__(self, filename: str, flags: int = SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE, vfs: Optional[str] = None, statementcachesize: int = 100):
        """Opens the named database.  You can use ``:memory:`` to get a private temporary
        in-memory database that is not shared with any other connections.

        :param flags: One or more of the `open flags <https://sqlite.org/c3ref/c_open_autoproxy.html>`_ orred together
        :param vfs: The name of the `vfs <https://sqlite.org/c3ref/vfs.html>`_ to use.  If *None* then the default
           vfs will be used.

        :param statementcachesize: Use zero to disable the statement cache,
          or a number larger than the total distinct SQL statements you
          execute frequently.

        Calls: `sqlite3_open_v2 <https://sqlite.org/c3ref/open.html>`__

        .. seealso::

          * :attr:`apsw.connection_hooks`
          * :ref:`statementcache`
          * :ref:`vfs`"""
        ...

    def interrupt(self) -> None:
        """Causes all pending operations on the database to abort at the
        earliest opportunity. You can call this from any thread.  For
        example you may have a long running query when the user presses the
        stop button in your user interface.  :exc:`InterruptError`
        will be raised in the queries that got interrupted.

        Calls: `sqlite3_interrupt <https://sqlite.org/c3ref/interrupt.html>`__"""
        ...

    is_interrupted: bool
    """Indicates if this connection has been interrupted.

    Calls: `sqlite3_is_interrupted <https://sqlite.org/c3ref/interrupt.html>`__"""

    def last_insert_rowid(self) -> int:
        """Returns the integer key of the most recent insert in the database.

        Calls: `sqlite3_last_insert_rowid <https://sqlite.org/c3ref/last_insert_rowid.html>`__"""
        ...

    def limit(self, id: int, newval: int = -1) -> int:
        """If called with one parameter then the current limit for that *id* is
        returned.  If called with two then the limit is set to *newval*.


        :param id: One of the `runtime limit ids <https://sqlite.org/c3ref/c_limit_attached.html>`_
        :param newval: The new limit.  This is a 32 bit signed integer even on 64 bit platforms.

        :returns: The limit in place on entry to the call.

        Calls: `sqlite3_limit <https://sqlite.org/c3ref/limit.html>`__

        .. seealso::

          * :ref:`Example <example_limits>`"""
        ...

    def load_extension(self, filename: str, entrypoint: Optional[str] = None) -> None:
        """Loads *filename* as an `extension <https://www.sqlite.org/loadext.html>`_

        :param filename: The file to load.

        :param entrypoint: The initialization method to call.  If this
          parameter is not supplied then the SQLite default of
          ``sqlite3_extension_init`` is used.

        :raises ExtensionLoadingError: If the extension could not be
          loaded.  The exception string includes more details.

        Calls: `sqlite3_load_extension <https://sqlite.org/c3ref/load_extension.html>`__

        .. seealso::

          * :meth:`~Connection.enable_load_extension`"""
        ...

    loadextension = load_extension ## OLD-NAME

    open_flags: int
    """The combination of :attr:`flags <apsw.mapping_open_flags>` used to open the database."""

    open_vfs: str
    """The string name of the vfs used to open the database."""

    def overload_function(self, name: str, nargs: int) -> None:
        """Registers a placeholder function so that a virtual table can provide an implementation via
        :meth:`VTTable.FindFunction`.

        :param name: Function name
        :param nargs: How many arguments the function takes

          Calls: `sqlite3_overload_function <https://sqlite.org/c3ref/overload_function.html>`__"""
        ...

    overloadfunction = overload_function ## OLD-NAME

    def pragma(self, name: str, value: Optional[SQLiteValue] = None, *, schema: Optional[str] = None) -> Any:
        """Issues the pragma (with the value if supplied) and returns the result with
        :attr:`the least amount of structure <Cursor.get>`.  For example
        :code:`pragma("user_version")` will return just the number, while
        :code:`pragma("journal_mode", "WAL")` will return the journal mode
        now in effect.

        Pragmas do not support bindings, so this method is a convenient
        alternative to composing SQL text.  Pragmas are often executed
        while being prepared, instead of when run like regular SQL.  They
        may also contain encryption keys.  This method ensures they are
        not cached to avoid problems.

        Use the `schema` parameter to run the pragma against a different
        attached database (eg ``temp``).

        * :ref:`Example <example_pragma>`"""
        ...

    def read(self, schema: str, which: int, offset: int, amount: int) -> tuple[bool, bytes]:
        """Invokes the underlying VFS method to read data from the database.  It
        is strongly recommended to read aligned complete pages, since that is
        only what SQLite does.

        `schema` is `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__

        `which` is 0 for the database file, 1 for the journal.

        The return value is a tuple of a boolean indicating a complete read if
        True, and the bytes read which will always be the amount requested
        in size.

        `SQLITE_IOERR_SHORT_READ` will give a `False` value for the boolean,
        and there is no way of knowing how much was read.

        Implemented using `SQLITE_FCNTL_FILE_POINTER` and `SQLITE_FCNTL_JOURNAL_POINTER`.
        Errors will usually be generic `SQLITE_ERROR` with no message.

        Calls: `sqlite3_file_control <https://sqlite.org/c3ref/file_control.html>`__"""
        ...

    def readonly(self, name: str) -> bool:
        """True or False if the named (attached) database was opened readonly or file
        permissions don't allow writing.  The name is `main`, `temp`, the
        name in `ATTACH <https://sqlite.org/lang_attach.html>`__

        An exception is raised if the database doesn't exist.

        Calls: `sqlite3_db_readonly <https://sqlite.org/c3ref/db_readonly.html>`__"""
        ...

    def register_fts5_function(self, name: str, function: FTS5Function) -> None:
        """Registers the (case insensitive) named function used as an `auxiliary
        function  <https://www.sqlite.org/fts5.html#custom_auxiliary_functions>`__.

        The first parameter to the function will be :class:`FTS5ExtensionApi`
        and the rest will be the function arguments at the SQL level."""
        ...

    def register_fts5_tokenizer(self, name: str, tokenizer_factory: FTS5TokenizerFactory) -> None:
        """Registers a tokenizer factory.  Names are case insensitive.  It is not possible to
        unregister a tokenizer.

        .. seealso::

            * :meth:`fts5_tokenizer`
            * :doc:`textsearch`
            * `FTS5 documentation <https://www.sqlite.org/fts5.html#custom_tokenizers>`__"""
        ...

    def release_memory(self) -> None:
        """Attempts to free as much heap memory as possible used by this connection.

        Calls: `sqlite3_db_release_memory <https://sqlite.org/c3ref/db_release_memory.html>`__"""
        ...

    row_trace: Optional[RowTracer]
    """Called with the cursor and row being returned for
    :class:`cursors <Cursor>` associated with this Connection, unless
    the Cursor installed its own tracer.  You can change the data that
    is returned or cause the row to be skipped altogether.

    If *callable* is *None* then any existing row tracer is
    removed.

    .. seealso::

      * :ref:`tracing`
      * :ref:`rowtracer`
      * :attr:`Cursor.exec_trace`"""

    rowtrace = row_trace ## OLD-NAME

    def serialize(self, name: str) -> bytes:
        """Returns a memory copy of the database. *name* is `main`, `temp`, the name
        in `ATTACH <https://sqlite.org/lang_attach.html>`__

        The memory copy is the same as if the database was backed up to
        disk.

        If the database name doesn't exist, then None is returned, not an
        exception (this is SQLite's behaviour).  One exception is than an
        empty temp will result in a None return.

         .. seealso::

           * :meth:`Connection.deserialize`

         Calls: `sqlite3_serialize <https://sqlite.org/c3ref/serialize.html>`__"""
        ...

    def set_authorizer(self, callable: Optional[Authorizer]) -> None:
        """Sets the :attr:`authorizer`"""
        ...

    setauthorizer = set_authorizer ## OLD-NAME

    def set_busy_handler(self, callable: Optional[Callable[[int], bool]]) -> None:
        """Sets the busy handler to callable. callable will be called with one
        integer argument which is the number of prior calls to the busy
        callback for the same lock. If the busy callback returns False,
        then SQLite returns *SQLITE_BUSY* to the calling code. If
        the callback returns True, then SQLite tries to open the table
        again and the cycle repeats.

        If you previously called :meth:`~Connection.set_busy_timeout` then
        calling this overrides that.

        Passing None unregisters the existing handler.

        .. seealso::

          * :meth:`Connection.set_busy_timeout`
          * :ref:`Busy handling <busyhandling>`

        Calls: `sqlite3_busy_handler <https://sqlite.org/c3ref/busy_handler.html>`__"""
        ...

    setbusyhandler = set_busy_handler ## OLD-NAME

    def set_busy_timeout(self, milliseconds: int) -> None:
        """If the database is locked such as when another connection is making
        changes, SQLite will keep retrying.  This sets the maximum amount of
        time SQLite will keep retrying before giving up.  If the database is
        still busy then :class:`apsw.BusyError` will be returned.

        :param milliseconds: Maximum thousandths of a second to wait.

        If you previously called :meth:`~Connection.set_busy_handler` then
        calling this overrides that.

        .. seealso::

           * :meth:`Connection.set_busy_handler`
           * :ref:`Busy handling <busyhandling>`

        Calls: `sqlite3_busy_timeout <https://sqlite.org/c3ref/busy_timeout.html>`__"""
        ...

    setbusytimeout = set_busy_timeout ## OLD-NAME

    def set_commit_hook(self, callable: Optional[CommitHook]) -> None:
        """*callable* will be called just before a commit.  It should return
        False for the commit to go ahead and True for it to be turned
        into a rollback. In the case of an exception in your callable, a
        True (rollback) value is returned.  Pass None to unregister
        the existing hook.

        .. seealso::

          * :ref:`Example <example_commit_hook>`

        Calls: `sqlite3_commit_hook <https://sqlite.org/c3ref/commit_hook.html>`__"""
        ...

    setcommithook = set_commit_hook ## OLD-NAME

    def set_exec_trace(self, callable: Optional[ExecTracer]) -> None:
        """Method to set :attr:`Connection.exec_trace`"""
        ...

    setexectrace = set_exec_trace ## OLD-NAME

    def set_last_insert_rowid(self, rowid: int) -> None:
        """Sets the value calls to :meth:`last_insert_rowid` will return.

        Calls: `sqlite3_set_last_insert_rowid <https://sqlite.org/c3ref/set_last_insert_rowid.html>`__"""
        ...

    def set_profile(self, callable: Optional[Callable[[str, int], None]]) -> None:
        """Sets a callable which is invoked at the end of execution of each
        statement and passed the statement string and how long it took to
        execute. (The execution time is in nanoseconds.) Note that it is
        called only on completion. If for example you do a ``SELECT`` and only
        read the first result, then you won't reach the end of the statement.

        Calls: `sqlite3_trace_v2 <https://sqlite.org/c3ref/trace_v2.html>`__"""
        ...

    setprofile = set_profile ## OLD-NAME

    def set_progress_handler(self, callable: Optional[Callable[[], bool]], nsteps: int = 100, *, id: Optional[Any] = None) -> None:
        """Sets a callable which is invoked every *nsteps* SQLite inststructions.
        The callable should return True to abort or False to continue. (If
        there is an error in your Python *callable* then True/abort will be
        returned).  SQLite raises :exc:`InterruptError` for aborts.

        Use :class:`None` to cancel the progress handler.  Multiple handlers
        can be present at once (implemented by APSW). Registered callbacks are
        distinguished by their ``id`` - an equality test is done to match ids.

        You can use :class:`apsw.ext.Trace` to see how many steps are used for
        a representative statement, or :class:`apsw.ext.ShowResourceUsage` to
        see how many are used in a block.  It will generally be several million
        per second.

        .. seealso::

           * :ref:`Example <example_progress_handler>`

        Calls: `sqlite3_progress_handler <https://sqlite.org/c3ref/progress_handler.html>`__"""
        ...

    setprogresshandler = set_progress_handler ## OLD-NAME

    def set_rollback_hook(self, callable: Optional[Callable[[], None]]) -> None:
        """Sets a callable which is invoked during a rollback.  If *callable*
        is *None* then any existing rollback hook is unregistered.

        The *callable* is called with no parameters and the return value is ignored.

        Calls: `sqlite3_rollback_hook <https://sqlite.org/c3ref/commit_hook.html>`__"""
        ...

    setrollbackhook = set_rollback_hook ## OLD-NAME

    def set_row_trace(self, callable: Optional[RowTracer]) -> None:
        """Method to set :attr:`Connection.row_trace`"""
        ...

    setrowtrace = set_row_trace ## OLD-NAME

    def set_update_hook(self, callable: Optional[Callable[[int, str, str, int], None]]) -> None:
        """Calls *callable* whenever a row is updated, deleted or inserted.  If
        *callable* is *None* then any existing update hook is
        unregistered.  The update hook cannot make changes to the database while
        the query is still executing, but can record them for later use or
        apply them in a different connection.

        The update hook is called with 4 parameters:

          type (int)
            *SQLITE_INSERT*, *SQLITE_DELETE* or *SQLITE_UPDATE*
          database name (str)
            `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__
          table name (str)
            The table on which the update happened
          rowid (int)
            The affected row

        .. seealso::

            * :ref:`Example <example_update_hook>`

        Calls: `sqlite3_update_hook <https://sqlite.org/c3ref/update_hook.html>`__"""
        ...

    setupdatehook = set_update_hook ## OLD-NAME

    def set_wal_hook(self, callable: Optional[Callable[[Connection, str, int], int]]) -> None:
        """*callable* will be called just after data is committed in :ref:`wal`
        mode.  It should return *SQLITE_OK* or an error code.  The
        callback is called with 3 parameters:

          * The Connection
          * The database name.  `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__
          * The number of pages in the wal log

        You can pass in None in order to unregister an existing hook.

        Calls: `sqlite3_wal_hook <https://sqlite.org/c3ref/wal_hook.html>`__"""
        ...

    setwalhook = set_wal_hook ## OLD-NAME

    def sqlite3_pointer(self) -> int:
        """Returns the underlying `sqlite3 *
        <https://sqlite.org/c3ref/sqlite3.html>`_ for the connection. This
        method is useful if there are other C level libraries in the same
        process and you want them to use the APSW connection handle. The value
        is returned as a number using `PyLong_FromVoidPtr
        <https://docs.python.org/3/c-api/long.html?highlight=pylong_fromvoidptr#c.PyLong_FromVoidPtr>`__
        under the hood. You should also ensure that you increment the
        reference count on the :class:`Connection` for as long as the other
        libraries are using the pointer.  It is also a very good idea to call
        :meth:`sqlite_lib_version` and ensure it is the same as the other
        libraries."""
        ...

    sqlite3pointer = sqlite3_pointer ## OLD-NAME

    def status(self, op: int, reset: bool = False) -> tuple[int, int]:
        """Returns current and highwater measurements for the database.

        :param op: A `status parameter <https://sqlite.org/c3ref/c_dbstatus_options.html>`_
        :param reset: If *True* then the highwater is set to the current value
        :returns: A tuple of current value and highwater value

        .. seealso::

          * :func:`apsw.status` which does the same for SQLite as a whole
          * :ref:`Example <example_status>`

        Calls: `sqlite3_db_status <https://sqlite.org/c3ref/db_status.html>`__"""
        ...

    system_errno: int
    """The underlying system error code for the most recent I/O error.

    Calls: `sqlite3_system_errno <https://sqlite.org/c3ref/system_errno.html>`__"""

    def table_exists(self, dbname: Optional[str], table_name: str) -> bool:
        """Returns True if the named table exists, else False.

        ``dbname`` is ``main``, ``temp``, the name in `ATTACH
        <https://sqlite.org/lang_attach.html>`__, or None to search  all
        databases

        Calls: `sqlite3_table_column_metadata <https://sqlite.org/c3ref/table_column_metadata.html>`__"""
        ...

    def total_changes(self) -> int:
        """Returns the total number of database rows that have be modified,
        inserted, or deleted since the database connection was opened.

        Calls: `sqlite3_total_changes64 <https://sqlite.org/c3ref/total_changes.html>`__"""
        ...

    totalchanges = total_changes ## OLD-NAME

    def trace_v2(self, mask: int, callback: Optional[Callable[[dict], None]] = None, *, id: Optional[Any] = None) -> None:
        """Registers a trace callback.  Multiple traces can be active at once
        (implemented by APSW).  A callback of :class:`None` unregisters a
        trace.  Registered callbacks are distinguished by their ``id`` - an
        equality test is done to match ids.

        The callback is called with a dict of relevant values based on the
        code.

        .. list-table::
          :header-rows: 1
          :widths: auto

          * - Key
            - Type
            - Explanation
          * - code
            - :class:`int`
            - One of the `trace event codes <https://www.sqlite.org/c3ref/c_trace.html>`__
          * - connection
            - :class:`Connection`
            - Connection this trace event belongs to
          * - sql
            - :class:`str`
            - SQL text (except SQLITE_TRACE_ROW and SQLITE_TRACE_CLOSE).
          * - id
            - :class:`int`
            - An opaque key to correlate events on the same statement.  The
              id can be reused after SQLITE_TRACE_PROFILE.
          * - trigger
            - :class:`bool`
            - If `trigger <https://www.sqlite.org/lang_createtrigger.html>`__
              SQL is executing then this is ``True`` and the SQL is of the trigger.
              Virtual table nested queries also come through as trigger activity.
          * - total_changes
            - :class:`int`
            - Value of :meth:`total_changes`  (SQLITE_TRACE_STMT and SQLITE_TRACE_PROFILE only)
          * - nanoseconds
            - :class:`int`
            - nanoseconds SQL took to execute (SQLITE_TRACE_PROFILE only)
          * - stmt_status
            - :class:`dict`
            - SQLITE_TRACE_PROFILE only: Keys are names from `status parameters
              <https://www.sqlite.org/c3ref/c_stmtstatus_counter.html>`__ - eg
              *"SQLITE_STMTSTATUS_VM_STEP"* and corresponding integer values.
              The counters are reset each time a statement
              starts execution.  This includes any changes made by triggers.

        Note that SQLite ignores any errors from the trace callbacks, so
        whatever was being traced will still proceed.  Exceptions will be
        delivered when your Python code resumes.

        If you register for all trace types, the following sequence will happen.

        * SQLITE_TRACE_STMT with `trigger` `False` and an `id` and `sql` of
          the statement.
        * Multiple times: SQLITE_TRACE_STMT with the same `id` and `trigger`
          `True` if a trigger is executed.  The first time the `sql` will be
          ``TRIGGER name`` and then subsequent calls will be lines of the
          trigger.  This also happens for virtual tables that make queries.
        * Multiple times: SQLITE_TRACE_ROW with the same `id` for each time
          execution stopped at a row. (Rows visited by triggers do not cause
          thie event)
        * SQLITE_TRACE_PROFILE with the same `id` for any virtual table
          queries - the ``sql`` will be of those queries
        * SQLITE_TRACE_PROFILE with the same `id` for the initial SQL.

        .. seealso::

          * :ref:`Example <example_trace_v2>`
          * :class:`apsw.ext.Trace`

        Calls:
          * `sqlite3_trace_v2 <https://sqlite.org/c3ref/trace_v2.html>`__
          * `sqlite3_stmt_status <https://sqlite.org/c3ref/stmt_status.html>`__"""
        ...

    def txn_state(self, schema: Optional[str] = None) -> int:
        """Returns the current transaction state of the database, or a specific schema
        if provided.  :attr:`apsw.mapping_txn_state` contains the values returned.

        Calls: `sqlite3_txn_state <https://sqlite.org/c3ref/txn_state.html>`__"""
        ...

    def vfsname(self, dbname: str) -> str | None:
        """Issues the `SQLITE_FCNTL_VFSNAME
        <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html#sqlitefcntlvfsname>`__
        file control against the named database (`main`, `temp`, attached
        name).

        This is useful to see which VFS is in use, and if inheritance is used
        then ``/`` will separate the names.  If you have a :class:`VFSFile` in
        use then its fully qualified class name will also be included.

        If ``SQLITE_FCNTL_VFSNAME`` is not implemented, ``dbname`` is not a
        database name, or an error occurred then ``None`` is returned."""
        ...

    def vtab_config(self, op: int, val: int = 0) -> None:
        """Callable during virtual table :meth:`~VTModule.Connect`/:meth:`~VTModule.Create`.

        Calls: `sqlite3_vtab_config <https://sqlite.org/c3ref/vtab_config.html>`__"""
        ...

    def vtab_on_conflict(self) -> int:
        """Callable during virtual table :meth:`insert <VTTable.UpdateInsertRow>` or
        :meth:`update <VTTable.UpdateChangeRow>`

        Calls: `sqlite3_vtab_on_conflict <https://sqlite.org/c3ref/vtab_on_conflict.html>`__"""
        ...

    def wal_autocheckpoint(self, n: int) -> None:
        """Sets how often the :ref:`wal` checkpointing is run.

         :param n: A number representing the checkpointing interval or
           zero/negative to disable auto checkpointing.

        Calls: `sqlite3_wal_autocheckpoint <https://sqlite.org/c3ref/wal_autocheckpoint.html>`__"""
        ...

    def wal_checkpoint(self, dbname: Optional[str] = None, mode: int = SQLITE_CHECKPOINT_PASSIVE) -> tuple[int, int]:
        """Does a WAL checkpoint.  Has no effect if the database(s) are not in WAL mode.

          :param dbname:  The name of the database or all databases if None

          :param mode: One of the `checkpoint modes <https://sqlite.org/c3ref/wal_checkpoint_v2.html>`__.

          :return: A tuple of the size of the WAL log in frames and the
             number of frames checkpointed as described in the
             `documentation
             <https://sqlite.org/c3ref/wal_checkpoint_v2.html>`__.

        Calls: `sqlite3_wal_checkpoint_v2 <https://sqlite.org/c3ref/wal_checkpoint_v2.html>`__"""
        ...

class Cursor:
    """"""
    bindings_count: int
    """How many bindings are in the statement.  The ``?`` form
    results in the largest number.  For example you could do
    ``SELECT ?123``` in which case the count will be ``123``.

    Calls: `sqlite3_bind_parameter_count <https://sqlite.org/c3ref/bind_parameter_count.html>`__"""

    bindings_names: tuple[str | None]
    """A tuple of the name of each bind parameter, or None for no name.  The
    leading marker (``?:@$``) is omitted

    .. note::

      SQLite parameter numbering starts at ``1``, while Python
      indexing starts at ``0``.

    Calls: `sqlite3_bind_parameter_name <https://sqlite.org/c3ref/bind_parameter_name.html>`__"""

    def close(self, force: bool = False) -> None:
        """It is very unlikely you will need to call this method.
        Cursors are automatically garbage collected and when there
        are none left will allow the connection to be garbage collected if
        it has no other references.

        A cursor is open if there are remaining statements to execute (if
        your query included multiple statements), or if you called
        :meth:`~Cursor.executemany` and not all of the sequence of bindings
        have been used yet.

        :param force: If False then you will get exceptions if there is
         remaining work to do be in the Cursor such as more statements to
         execute, more data from the executemany binding sequence etc. If
         force is True then all remaining work and state information will be
         silently discarded."""
        ...

    connection: Connection
    """:class:`Connection` this cursor is using"""

    description: tuple[tuple[str, str, None, None, None, None, None], ...]
    """Based on the `DB-API cursor property
    <https://www.python.org/dev/peps/pep-0249/>`__, this returns the
    same as :meth:`get_description` but with 5 Nones appended because
    SQLite does not have the information."""

    description_full: tuple[tuple[str, str, str, str, str], ...]
    """Only present if SQLITE_ENABLE_COLUMN_METADATA was defined at
    compile time.

    Returns all information about the query result columns. In
    addition to the name and declared type, you also get the database
    name, table name, and origin name.

    Calls:
      * `sqlite3_column_name <https://sqlite.org/c3ref/column_name.html>`__
      * `sqlite3_column_decltype <https://sqlite.org/c3ref/column_decltype.html>`__
      * `sqlite3_column_database_name <https://sqlite.org/c3ref/column_database_name.html>`__
      * `sqlite3_column_table_name <https://sqlite.org/c3ref/column_database_name.html>`__
      * `sqlite3_column_origin_name <https://sqlite.org/c3ref/column_database_name.html>`__"""

    exec_trace: Optional[ExecTracer]
    """Called with the cursor, statement and bindings for
    each :meth:`~Cursor.execute` or :meth:`~Cursor.executemany` on this
    cursor.

    If *callable* is *None* then any existing execution tracer is
    unregistered.

    .. seealso::

      * :ref:`tracing`
      * :ref:`executiontracer`
      * :attr:`Connection.exec_trace`"""

    exectrace = exec_trace ## OLD-NAME

    def execute(self, statements: str, bindings: Optional[Bindings] = None, *, can_cache: bool = True, prepare_flags: int = 0, explain: int = -1) -> Cursor:
        """Executes the statements using the supplied bindings.  Execution
        returns when the first row is available or all statements have
        completed.

        :param statements: One or more SQL statements such as ``select *
          from books`` or ``begin; insert into books ...; select
          last_insert_rowid(); end``.
        :param bindings: If supplied should either be a sequence or a dictionary.  Each item must be one of the :ref:`supported types <types>`,
          :class:`zeroblob`, or a wrapped :ref:`Python object <pyobject>`
        :param can_cache: If False then the statement cache will not be used to find an already prepared query, nor will it be
          placed in the cache after execution
        :param prepare_flags: `flags <https://sqlite.org/c3ref/c_prepare_normalize.html>`__ passed to
          `sqlite_prepare_v3 <https://sqlite.org/c3ref/prepare.html>`__
        :param explain: If 0 or greater then the statement is passed to `sqlite3_stmt_explain <https://sqlite.org/c3ref/stmt_explain.html>`__
           where you can force it to not be an explain, or force explain or explain query plan.

        :raises TypeError: The bindings supplied were neither a dict nor a sequence
        :raises BindingsError: You supplied too many or too few bindings for the statements
        :raises IncompleteExecutionError: There are remaining unexecuted queries from your last execute

        Calls:
          * `sqlite3_prepare_v3 <https://sqlite.org/c3ref/prepare.html>`__
          * `sqlite3_step <https://sqlite.org/c3ref/step.html>`__
          * `sqlite3_bind_int64 <https://sqlite.org/c3ref/bind_blob.html>`__
          * `sqlite3_bind_null <https://sqlite.org/c3ref/bind_blob.html>`__
          * `sqlite3_bind_text64 <https://sqlite.org/c3ref/bind_blob.html>`__
          * `sqlite3_bind_double <https://sqlite.org/c3ref/bind_blob.html>`__
          * `sqlite3_bind_blob64 <https://sqlite.org/c3ref/bind_blob.html>`__
          * `sqlite3_bind_zeroblob64 <https://sqlite.org/c3ref/bind_blob.html>`__
          * `sqlite3_stmt_explain <https://sqlite.org/c3ref/stmt_explain.html>`__"""
        ...

    def executemany(self, statements: str, sequenceofbindings: Iterable[Bindings], *, can_cache: bool = True, prepare_flags: int = 0, explain: int = -1) -> Cursor:
        """This method is for when you want to execute the same statements over
        a sequence of bindings.  Conceptually it does this::

          for binding in sequenceofbindings:
              cursor.execute(statements, binding)

        The return is the cursor itself which acts as an iterator.  Your
        statements can return data.  See :meth:`~Cursor.execute` for more
        information, and the :ref:`example <example_executemany>`."""
        ...

    expanded_sql: str
    """The SQL text with bound parameters expanded.  For example::

       execute("select ?, ?", (3, "three"))

    would return::

       select 3, 'three'

    Note that while SQLite supports nulls in strings, their implementation
    of sqlite3_expanded_sql stops at the first null.

    You will get :exc:`MemoryError` if SQLite ran out of memory, or if
    the expanded string would exceed `SQLITE_LIMIT_LENGTH
    <https://www.sqlite.org/c3ref/c_limit_attached.html>`__.

    Calls: `sqlite3_expanded_sql <https://sqlite.org/c3ref/expanded_sql.html>`__"""

    def fetchall(self) -> list[tuple[SQLiteValue, ...]]:
        """Returns all remaining result rows as a list.  This method is defined
        in DBAPI.  See :meth:`get` which does the same thing, but with the least
        amount of structure to unpack."""
        ...

    def fetchone(self) -> Optional[Any]:
        """Returns the next row of data or None if there are no more rows."""
        ...

    get: Any
    """Like :meth:`fetchall` but returns the data with the least amount of structure
    possible.

    .. list-table:: Some examples
       :header-rows: 1
       :widths: auto

       * - Query
         - Result
       * - select 3
         - 3
       * - select 3,4
         - (3, 4)
       * - select 3; select 4
         - [3, 4]
       * - select 3,4; select 4,5
         - [(3, 4), (4, 5)]
       * - select 3,4; select 5
         - [(3, 4), 5]

    Row tracers are not called when using this method."""

    def get_connection(self) -> Connection:
        """Returns the :attr:`connection` this cursor is part of"""
        ...

    getconnection = get_connection ## OLD-NAME

    def get_description(self) -> tuple[tuple[str, str], ...]:
        """If you are trying to get information about a table or view,
        then `pragma table_info <https://sqlite.org/pragma.html#pragma_table_info>`__
        is better.  If you want to know up front what columns and other
        details a query does then :func:`apsw.ext.query_info` is useful.

        Returns a tuple describing each column in the result row.  The
        return is identical for every row of the results.

        The information about each column is a tuple of ``(column_name,
        declared_column_type)``.  The type is what was declared in the
        ``CREATE TABLE`` statement - the value returned in the row will be
        whatever type you put in for that row and column.

        See the :ref:`query_info example <example_query_details>`.

        Calls:
          * `sqlite3_column_name <https://sqlite.org/c3ref/column_name.html>`__
          * `sqlite3_column_decltype <https://sqlite.org/c3ref/column_decltype.html>`__"""
        ...

    getdescription = get_description ## OLD-NAME

    def get_exec_trace(self) -> Optional[ExecTracer]:
        """Returns the currently installed :attr:`execution tracer
        <Cursor.exec_trace>`

        .. seealso::

          * :ref:`tracing`"""
        ...

    getexectrace = get_exec_trace ## OLD-NAME

    def get_row_trace(self) -> Optional[RowTracer]:
        """Returns the currently installed (via :meth:`~Cursor.set_row_trace`)
        row tracer.

        .. seealso::

          * :ref:`tracing`"""
        ...

    getrowtrace = get_row_trace ## OLD-NAME

    has_vdbe: bool
    """``True`` if the SQL does anything.  Comments have nothing to
    evaluate, and so are ``False``."""

    def __init__(self, connection: Connection):
        """Use :meth:`Connection.cursor` to make a new cursor."""
        ...

    is_explain: int
    """Returns 0 if executing a normal query, 1 if it is an EXPLAIN query,
    and 2 if an EXPLAIN QUERY PLAN query.

    Calls: `sqlite3_stmt_isexplain <https://sqlite.org/c3ref/stmt_isexplain.html>`__"""

    is_readonly: bool
    """Returns True if the current query does not change the database.

    Note that called functions, virtual tables etc could make changes though.

    Calls: `sqlite3_stmt_readonly <https://sqlite.org/c3ref/stmt_readonly.html>`__"""

    def __iter__(self: Cursor) -> Cursor:
        """Cursors are iterators"""
        ...

    def __next__(self: Cursor) -> Any:
        """Cursors are iterators"""
        ...

    row_trace: Optional[RowTracer]
    """Called with cursor and row being returned.  You can
    change the data that is returned or cause the row to be skipped
    altogether.

    If *callable* is *None* then any existing row tracer is
    unregistered.

    .. seealso::

      * :ref:`tracing`
      * :ref:`rowtracer`
      * :attr:`Connection.row_trace`"""

    rowtrace = row_trace ## OLD-NAME

    def set_exec_trace(self, callable: Optional[ExecTracer]) -> None:
        """Sets the :attr:`execution tracer <Cursor.exec_trace>`"""
        ...

    setexectrace = set_exec_trace ## OLD-NAME

    def set_row_trace(self, callable: Optional[RowTracer]) -> None:
        """Sets the :attr:`row tracer <Cursor.row_trace>`"""
        ...

    setrowtrace = set_row_trace ## OLD-NAME

@final
class FTS5ExtensionApi:
    """`Auxiliary functions
    <https://www.sqlite.org/fts5.html#_auxiliary_functions_>`__  run in
    the context of a FTS5 search, and can be used for ranking,
    highlighting, and similar operations.  Auxiliary functions are
    registered via :meth:`Connection.register_fts5_function`.  This wraps
    the `auxiliary functions API
    <https://www.sqlite.org/fts5.html#custom_auxiliary_functions>`__
    passed as the first parameter to auxiliary functions.

    See :ref:`the example <example_fts5_auxfunc>`."""
    aux_data: Any
    """You can store an object as `auxiliary data <https://www.sqlite.org/fts5.html#xSetAuxdata>`__
    which is available across matching rows.  It starts out as :class:`None`.

    An example use is to do up front calculations once, rather than on
    every matched row, such as
    :func:`fts5aux.inverse_document_frequency`."""

    column_count: int
    """Returns the `number of columns in the table
    <https://www.sqlite.org/fts5.html#xColumnCount>`__"""

    def column_locale(self, column: int) -> str | None:
        """`Retrieves the locale for a column  <https://www.sqlite.org/fts5.html#xColumnLocale>`__ on
        this row."""
        ...

    def column_size(self, col: int = -1) -> int:
        """Returns the `total number of tokens in the current row
        <https://www.sqlite.org/fts5.html#xColumnSize>`__ for a specific
        column, or if ``col`` is negative then for all columns."""
        ...

    def column_text(self, col: int) -> bytes:
        """Returns the `utf8 bytes for the column of the current row <https://www.sqlite.org/fts5.html#xColumnText>`__."""
        ...

    def column_total_size(self, col: int = -1) -> int:
        """Returns the `total number of tokens in the table
        <https://www.sqlite.org/fts5.html#xColumnTotalSize>`__ for a specific
        column, or if ``col`` is negative then for all columns."""
        ...

    inst_count: int
    """Returns the `number of hits in the current row
    <https://www.sqlite.org/fts5.html#xInstCount>`__"""

    def inst_tokens(self, inst: int) -> tuple[str, ...] | None:
        """`Access tokens of hit inst in current row <https://www.sqlite.org/fts5.html#xInstToken>`__.
        None is returned if the call is not supported."""
        ...

    def phrase_column_offsets(self, phrase: int, column: int) -> list[int]:
        """Returns `token offsets the phrase number occurs in
        <https://www.sqlite.org/fts5.html#xPhraseFirst>`__  in the specified
        column."""
        ...

    def phrase_columns(self, phrase: int) -> tuple[int]:
        """Returns `which columns the phrase number occurs in <https://www.sqlite.org/fts5.html#xPhraseFirstColumn>`__"""
        ...

    phrase_count: int
    """Returns the `number of phrases in the query
    <https://www.sqlite.org/fts5.html#xPhraseCount>`__"""

    def phrase_locations(self, phrase: int) -> list[list[int]]:
        """Returns `which columns and token offsets  the phrase number occurs in
        <https://www.sqlite.org/fts5.html#xPhraseFirst>`__.

        The returned list is the same length as the number of columns.  Each
        member is a list of token offsets in that column, and will be empty
        if the phrase is not in that column."""
        ...

    phrases: tuple[tuple[str | None, ...], ...]
    """A tuple where each member is a phrase from the query.  Each phrase is a tuple
    of str (or None when not available) per token of the phrase.

    This combines the results of `xPhraseCount <https://www.sqlite.org/fts5.html#xPhraseCount>`__,
    `xPhraseSize <https://www.sqlite.org/fts5.html#xPhraseSize>`__ and
    `xQueryToken <https://www.sqlite.org/fts5.html#xQueryToken>`__"""

    def query_phrase(self, phrase: int, callback: FTS5QueryPhrase, closure: Any) -> None:
        """Searches the table for the `numbered query <https://www.sqlite.org/fts5.html#xQueryPhrase>`__.
        The callback takes two parameters - a different :class:`apsw.FTS5ExtensionApi` and closure.

        An example usage for this method is to see how often the phrases occur in the table.  Setup a
        tracking counter here, and then in the callback you can update it on each visited row.  This
        is shown in :ref:`the example <example_fts5_auxfunc>`."""
        ...

    row_count: int
    """Returns the `number of rows in the table
    <https://www.sqlite.org/fts5.html#xRowCount>`__"""

    rowid: int
    """Rowid of the `current row <https://www.sqlite.org/fts5.html#xGetAuxdata>`__"""

    def tokenize(self, utf8: bytes, locale: Optional[str], *, include_offsets: bool = True, include_colocated: bool = True) -> list:
        """`Tokenizes the utf8 <https://www.sqlite.org/fts5.html#xTokenize_v2>`__.  FTS5 sets the reason to ``FTS5_TOKENIZE_AUX``.
        See :meth:`apsw.FTS5Tokenizer.__call__` for details."""
        ...

@final
class FTS5Tokenizer:
    """Wraps a registered tokenizer.  Returned by :meth:`Connection.fts5_tokenizer`."""
    args: tuple[str]
    """The arguments the tokenizer was created with."""

    def __call__(self, utf8: bytes, flags: int,  locale: Optional[str], *, include_offsets: bool = True, include_colocated: bool = True) -> TokenizerResult:
        """Does a tokenization, returning a list of the results.  If you have no
        interest in token offsets or colocated tokens then they can be omitted from
        the results.

        :param utf8: Input bytes
        :param reason: :data:`Reason <apsw.mapping_fts5_tokenize_reason>` flag
        :param include_offsets: Returned list includes offsets into utf8 for each token
        :param include_colocated: Returned list can include colocated tokens

        Example outputs
        ---------------

        Tokenizing ``b"first place"`` where ``1st`` has been provided as a
        colocated token for ``first``.

        (**Default**) include_offsets **True**, include_colocated **True**

          .. code-block:: python

                [
                  (0, 5, "first", "1st"),
                  (6, 11, "place"),
                ]

        include_offsets **False**, include_colocated **True**

          .. code-block:: python

                [
                  ("first", "1st"),
                  ("place", ),
                ]

        include_offsets **True**, include_colocated **False**

          .. code-block:: python

                [
                  (0, 5, "first"),
                  (6, 11, "place"),
                ]

        include_offsets **False**, include_colocated **False**

          .. code-block:: python

                [
                  "first",
                  "place",
                ]"""
        ...

    connection: Connection
    """The :class:`Connection` this tokenizer is registered with."""

    name: str
    """Tokenizer name"""

@final
class IndexInfo:
    """IndexInfo represents the `sqlite3_index_info
    <https://www.sqlite.org/c3ref/index_info.html>`__ and associated
    methods used in the :meth:`VTTable.BestIndexObject` method.

    Naming is identical to the C structure rather than Pythonic.  You can
    access members directly while needing to use get/set methods for array
    members.

    You will get :exc:`InvalidContextError` if you use the object outside of an
    BestIndex method.

    :meth:`apsw.ext.index_info_to_dict` provides a convenient
    representation of this object as a :class:`dict`."""
    colUsed: set[int]
    """(Read-only) Columns used by the statement.  Note that a set is returned, not
    the underlying integer."""

    distinct: int
    """(Read-only) How the query planner would like output ordered
    if the query is using group by or distinct.

    Calls: `sqlite3_vtab_distinct <https://sqlite.org/c3ref/vtab_distinct.html>`__"""

    estimatedCost: float
    """Estimated cost of using this index"""

    estimatedRows: int
    """Estimated number of rows returned"""

    def get_aConstraintUsage_argvIndex(self, which: int) -> int:
        """Returns *argvIndex* for *aConstraintUsage[which]*"""
        ...

    def get_aConstraintUsage_in(self, which: int) -> bool:
        """Returns True if the constraint is *in* - eg column in (3, 7, 9)

        Calls: `sqlite3_vtab_in <https://sqlite.org/c3ref/vtab_in.html>`__"""
        ...

    def get_aConstraintUsage_omit(self, which: int) -> bool:
        """Returns *omit* for *aConstraintUsage[which]*"""
        ...

    def get_aConstraint_collation(self, which: int) -> str:
        """Returns collation name for *aConstraint[which]*

        Calls: `sqlite3_vtab_collation <https://sqlite.org/c3ref/vtab_collation.html>`__"""
        ...

    def get_aConstraint_iColumn(self, which: int) -> int:
        """Returns *iColumn* for *aConstraint[which]*"""
        ...

    def get_aConstraint_op(self, which: int) -> int:
        """Returns *op* for *aConstraint[which]*"""
        ...

    def get_aConstraint_rhs(self, which: int) -> SQLiteValue:
        """Returns right hand side value if known, else None.

        Calls: `sqlite3_vtab_rhs_value <https://sqlite.org/c3ref/vtab_rhs_value.html>`__"""
        ...

    def get_aConstraint_usable(self, which: int) -> bool:
        """Returns *usable* for *aConstraint[which]*"""
        ...

    def get_aOrderBy_desc(self, which: int) -> bool:
        """Returns *desc* for *aOrderBy[which]*"""
        ...

    def get_aOrderBy_iColumn(self, which: int) -> int:
        """Returns *iColumn* for *aOrderBy[which]*"""
        ...

    idxFlags: int
    """Mask of :attr:`SQLITE_INDEX_SCAN flags <apsw.mapping_virtual_table_scan_flags>`"""

    idxNum: int
    """Number used to identify the index"""

    idxStr: Optional[str]
    """Name used to identify the index"""

    nConstraint: int
    """(Read-only) Number of constraint entries"""

    nOrderBy: int
    """(Read-only) Number of order by  entries"""

    orderByConsumed: bool
    """True if index output is already ordered"""

    def set_aConstraintUsage_argvIndex(self, which: int, argvIndex: int) -> None:
        """Sets *argvIndex* for *aConstraintUsage[which]*"""
        ...

    def set_aConstraintUsage_in(self, which: int, filter_all: bool) -> None:
        """If *which* is an *in* constraint, and *filter_all* is True then your :meth:`VTCursor.Filter`
        method will have all of the values at once.

        Calls: `sqlite3_vtab_in <https://sqlite.org/c3ref/vtab_in.html>`__"""
        ...

    def set_aConstraintUsage_omit(self, which: int, omit: bool) -> None:
        """Sets *omit* for *aConstraintUsage[which]*"""
        ...

@final
class URIFilename:
    """SQLite packs `uri parameters
    <https://sqlite.org/uri.html>`__ and the filename together   This class
    encapsulates that packing.  The :ref:`example <example_vfs>` shows
    usage of this class.

    Your :meth:`VFS.xOpen` method will generally be passed one of
    these instead of a string as the filename if the URI flag was used
    or the main database flag is set.

    You can safely pass it on to the :class:`VFSFile` constructor
    which knows how to get the name back out.  The URIFilename is
    only valid for the duration of the xOpen call.  If you save
    and use the object later you will get an exception."""
    def filename(self) -> str:
        """Returns the filename."""
        ...

    parameters: tuple[str, ...]
    """A tuple of the parameter names present.

    Calls: `sqlite3_uri_key <https://sqlite.org/c3ref/uri_boolean.html>`__"""

    def uri_boolean(self, name: str, default: bool) -> bool:
        """Returns the boolean value for parameter `name` or `default` if not
        present.

        Calls: `sqlite3_uri_boolean <https://sqlite.org/c3ref/uri_boolean.html>`__"""
        ...

    def uri_int(self, name: str, default: int) -> int:
        """Returns the integer value for parameter `name` or `default` if not
        present.

        Calls: `sqlite3_uri_int64 <https://sqlite.org/c3ref/uri_boolean.html>`__"""
        ...

    def uri_parameter(self, name: str) -> Optional[str]:
        """Returns the value of parameter `name` or None.

        Calls: `sqlite3_uri_parameter <https://sqlite.org/c3ref/uri_boolean.html>`__"""
        ...

@final
class VFSFcntlPragma:
    """A helper class to work with `SQLITE_FCNTL_PRAGMA
    <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html#sqlitefcntlpragma>`__
    in :meth:`VFSFile.xFileControl`. The :ref:`example <example_vfs>`
    shows usage of this class.

    It is only valid while in :meth:`VFSFile.xFileControl`, and using
    outside of that will result in memory corruption and crashes."""
    def __init__(self, pointer: int):
        """The pointer must be what your xFileControl method received."""
        ...

    name: str
    """The name of the pragma"""

    result: str | None
    """The first element which becomes the result or error message"""

    value: str | None
    """The value for the pragma, if provided else None,"""

class VFSFile:
    """Wraps access to a file.  You only need to derive from this class
    if you want the file object returned from :meth:`VFS.xOpen` to
    inherit from an existing VFS implementation."""
    def excepthook(self, etype: type[BaseException], evalue: BaseException, etraceback: Optional[types.TracebackType]) ->None:
        """Called when there has been an exception in a :class:`VFSFile`
        routine, and it can't be reported to the caller as usual.

        The default implementation passes the exception information
        to sqlite3_log, and the first non-error of
        :func:`sys.unraisablehook` and :func:`sys.excepthook`, falling back to
        `PyErr_Display`."""
        ...

    def __init__(self, vfs: str, filename: str | URIFilename | None, flags: list[int]):
        """:param vfs: The vfs you want to inherit behaviour from.  You can
           use an empty string ``""`` to inherit from the default vfs.
        :param name: The name of the file being opened.  May be an instance of :class:`URIFilename`.
        :param flags: A two item list ``[inflags, outflags]`` as detailed in :meth:`VFS.xOpen`.

        :raises ValueError: If the named VFS is not registered.

        .. note::

          If the VFS that you inherit from supports :ref:`write ahead
          logging <wal>` then your :class:`VFSFile` will also support the
          xShm methods necessary to implement wal.

        .. seealso::

          :meth:`VFS.xOpen`"""
        ...

    def xCheckReservedLock(self) -> bool:
        """Returns True if any database connection (in this or another process)
        has a lock other than `SQLITE_LOCK_NONE or SQLITE_LOCK_SHARED
        <https://sqlite.org/c3ref/c_lock_exclusive.html>`_."""
        ...

    def xClose(self) -> None:
        """Close the database. Note that even if you return an error you should
        still close the file.  It is safe to call this method multiple
        times."""
        ...

    def xDeviceCharacteristics(self) -> int:
        """Return `I/O capabilities
        <https://sqlite.org/c3ref/c_iocap_atomic.html>`_ (bitwise or of
        appropriate values). If you do not implement the function or have an
        error then 0 (the SQLite default) is returned."""
        ...

    def xFileControl(self, op: int, ptr: int) -> bool:
        """Receives `file control
         <https://sqlite.org/c3ref/file_control.html>`_ request typically
         issued by :meth:`Connection.file_control`.  See
         :meth:`Connection.file_control` for an example of how to pass a
         Python object to this routine.

         :param op: A numeric code.  Codes below 100 are reserved for SQLite
           internal use.
         :param ptr: An integer corresponding to a pointer at the C level.

         :returns: A boolean indicating if the op was understood

         Ensure you pass any unrecognised codes through to your super class.
         For example::

             def xFileControl(self, op: int, ptr: int) -> bool:
                 if op == 1027:
                     process_quick(ptr)
                 elif op == 1028:
                     obj=ctypes.py_object.from_address(ptr).value
                 else:
                     # this ensures superclass implementation is called
                     return super().xFileControl(op, ptr)
                # we understood the op
                return True

        .. note::

          `SQLITE_FCNTL_VFSNAME
          <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html#sqlitefcntlvfsname>`__
          is automatically handled for you dealing with the necessary memory allocation
          and listing all the VFS if you are inheriting.  It includes the fully qualified
          class name for this object."""
        ...

    def xFileSize(self) -> int:
        """Return the size of the file in bytes."""
        ...

    def xLock(self, level: int) -> None:
        """Increase the lock to the level specified which is one of the
        `SQLITE_LOCK <https://sqlite.org/c3ref/c_lock_exclusive.html>`_
        family of constants. If you can't increase the lock level because
        someone else has locked it, then raise :exc:`BusyError`."""
        ...

    def xRead(self, amount: int, offset: int) -> bytes:
        """Read the specified *amount* of data starting at *offset*. You
        should make every effort to read all the data requested, or return
        an error. If you have the file open for non-blocking I/O or if
        signals happen then it is possible for the underlying operating
        system to do a partial read. You will need to request the
        remaining data. Except for empty files SQLite considers short
        reads to be a fatal error.

        :param amount: Number of bytes to read
        :param offset: Where to start reading."""
        ...

    def xSectorSize(self) -> int:
        """Return the native underlying sector size. SQLite uses the value
        returned in determining the default database page size. If you do
        not implement the function or have an error then 4096 (the SQLite
        default) is returned."""
        ...

    def xSync(self, flags: int) -> None:
        """Ensure data is on the disk platters (ie could survive a power
        failure immediately after the call returns) with the `sync flags
        <https://sqlite.org/c3ref/c_sync_dataonly.html>`_ detailing what
        needs to be synced."""
        ...

    def xTruncate(self, newsize: int) -> None:
        """Set the file length to *newsize* (which may be more or less than the
        current length)."""
        ...

    def xUnlock(self, level: int) -> None:
        """Decrease the lock to the level specified which is one of the
        `SQLITE_LOCK <https://sqlite.org/c3ref/c_lock_exclusive.html>`_
        family of constants."""
        ...

    def xWrite(self, data: bytes, offset: int) -> None:
        """Write the *data* starting at absolute *offset*. You must write all the data
        requested, or return an error. If you have the file open for
        non-blocking I/O or if signals happen then it is possible for the
        underlying operating system to do a partial write. You will need to
        write the remaining data.

        :param offset: Where to start writing."""
        ...

class VFS:
    """Provides operating system access.  You can get an overview in the
    `SQLite documentation <https://sqlite.org/c3ref/vfs.html>`_.  To
    create a VFS your Python class must inherit from :class:`VFS`."""
    def excepthook(self, etype: type[BaseException], evalue: BaseException, etraceback: Optional[types.TracebackType]) -> Any:
        """Called when there has been an exception in a :class:`VFS` routine,
        and it can't be reported to the caller as usual.

        The default implementation passes the exception information
        to sqlite3_log, and the first non-error of
        :func:`sys.unraisablehook` and :func:`sys.excepthook`, falling back to
        `PyErr_Display`."""
        ...

    def __init__(self, name: str, base: Optional[str] = None, makedefault: bool = False, maxpathname: int = 1024, *, iVersion: int = 3, exclude: Optional[set[str]] = None):
        """:param name: The name to register this vfs under.  If the name
            already exists then this vfs will replace the prior one of the
            same name.  Use :meth:`apsw.vfs_names` to get a list of
            registered vfs names.

        :param base: If you would like to inherit behaviour from an already registered vfs then give
            their name.  To inherit from the default vfs, use a zero
            length string ``""`` as the name.

        :param makedefault: If true then this vfs will be registered as the default, and will be
            used by any opens that don't specify a vfs.

        :param maxpathname: The maximum length of database name in bytes when
            represented in UTF-8.  If a pathname is passed in longer than
            this value then SQLite will not`be able to open it.  If you are
            using a base, then a value of zero will use the value from base.

        :param iVersion: Version number for the `sqlite3_vfs <https://sqlite.org/c3ref/vfs.html>`__
            structure.

        :param exclude: A set of strings, naming the methods that will be filled in with ``NULL`` in the `sqlite3_vfs
            <https://sqlite.org/c3ref/vfs.html>`__  structure to indicate to SQLite that they are
            not supported.

        Calls:
          * `sqlite3_vfs_register <https://sqlite.org/c3ref/vfs_find.html>`__
          * `sqlite3_vfs_find <https://sqlite.org/c3ref/vfs_find.html>`__"""
        ...

    def unregister(self) -> None:
        """Unregisters the VFS making it unavailable to future database
        opens. You do not need to call this as the VFS is automatically
        unregistered by when the VFS has no more references or open
        databases using it. It is however useful to call if you have made
        your VFS be the default and wish to immediately make it be
        unavailable. It is safe to call this routine multiple times.

        Calls: `sqlite3_vfs_unregister <https://sqlite.org/c3ref/vfs_find.html>`__"""
        ...

    def xAccess(self, pathname: str, flags: int) -> bool:
        """SQLite wants to check access permissions.  Return True or False
        accordingly.

        :param pathname: File or directory to check
        :param flags: One of the `access flags <https://sqlite.org/c3ref/c_access_exists.html>`_"""
        ...

    def xCurrentTime(self)  -> float:
        """Return the `Julian Day Number
        <https://en.wikipedia.org/wiki/Julian_day>`_ as a floating point
        number where the integer portion is the day and the fractional part
        is the time."""
        ...

    def xCurrentTimeInt64(self)  -> int:
        """Returns as an integer the `Julian Day Number
        <https://en.wikipedia.org/wiki/Julian_day>`__ multiplied by 86400000
        (the number of milliseconds in a 24-hour day)."""
        ...

    def xDelete(self, filename: str, syncdir: bool) -> None:
        """Delete the named file. If the file is missing then raise an
        :exc:`IOError` exception with extendedresult
        *SQLITE_IOERR_DELETE_NOENT*

        :param filename: File to delete

        :param syncdir: If True then the directory should be synced
          ensuring that the file deletion has been recorded on the disk
          platters.  ie if there was an immediate power failure after this
          call returns, on a reboot the file would still be deleted."""
        ...

    def xDlClose(self, handle: int) -> None:
        """Close and unload the library corresponding to the handle you
        returned from :meth:`~VFS.xDlOpen`.  You can use ctypes to do
        this::

          def xDlClose(handle: int):
             # Note leading underscore in _ctypes
             _ctypes.dlclose(handle)       # Linux/Mac/Unix
             _ctypes.FreeLibrary(handle)   # Windows"""
        ...

    def xDlError(self) -> str:
        """Return an error string describing the last error of
        :meth:`~VFS.xDlOpen` or :meth:`~VFS.xDlSym` (ie they returned
        zero/NULL). If you do not supply this routine then SQLite provides
        a generic message. To implement this method, catch exceptions in
        :meth:`~VFS.xDlOpen` or :meth:`~VFS.xDlSym`, turn them into
        strings, save them, and return them in this routine.  If you have
        an error in this routine or return None then SQLite's generic
        message will be used."""
        ...

    def xDlOpen(self, filename: str) -> int:
        """Load the shared library. You should return a number which will be
        treated as a void pointer at the C level. On error you should
        return 0 (NULL). The number is passed as is to
        :meth:`~VFS.xDlSym`/:meth:`~VFS.xDlClose` so it can represent
        anything that is convenient for you (eg an index into an
        array). You can use ctypes to load a library::

          def xDlOpen(name: str):
             return ctypes.cdll.LoadLibrary(name)._handle"""
        ...

    def xDlSym(self, handle: int, symbol: str) -> int:
        """Returns the address of the named symbol which will be called by
        SQLite. On error you should return 0 (NULL). You can use ctypes::

          def xDlSym(ptr: int, name: str):
             return _ctypes.dlsym (ptr, name)  # Linux/Unix/Mac etc (note leading underscore)
             return ctypes.win32.kernel32.GetProcAddress (ptr, name)  # Windows

        :param handle: The value returned from an earlier :meth:`~VFS.xDlOpen` call
        :param symbol: A string"""
        ...

    def xFullPathname(self, name: str) -> str:
        """Return the absolute pathname for name.  You can use ``os.path.abspath`` to do this."""
        ...

    def xGetLastError(self) -> tuple[int, str]:
        """Return an integer error code and (optional) text describing
        the last error code and message that happened in this thread."""
        ...

    def xGetSystemCall(self, name: str) -> Optional[int]:
        """Returns a pointer for the current method implementing the named
        system call.  Return None if the call does not exist."""
        ...

    def xNextSystemCall(self, name: Optional[str]) -> Optional[str]:
        """This method is repeatedly called to iterate over all of the system
        calls in the vfs.  When called with None you should return the
        name of the first system call.  In subsequent calls return the
        name after the one passed in.  If name is the last system call
        then return None."""
        ...

    def xOpen(self, name: Optional[str | URIFilename], flags: list[int]) -> VFSFile:
        """This method should return a new file object based on name.  You
        can return a :class:`VFSFile` from a completely different VFS.

        :param name: File to open.  Note that *name* may be *None* in which
            case you should open a temporary file with a name of your
            choosing.  May be an instance of :class:`URIFilename`.

        :param flags: A list of two integers ``[inputflags,
          outputflags]``.  Each integer is one or more of the `open flags
          <https://sqlite.org/c3ref/c_open_autoproxy.html>`_ binary orred
          together.  The ``inputflags`` tells you what SQLite wants.  For
          example *SQLITE_OPEN_DELETEONCLOSE* means the file should
          be automatically deleted when closed.  The ``outputflags``
          describes how you actually did open the file.  For example if you
          opened it read only then *SQLITE_OPEN_READONLY* should be
          set."""
        ...

    def xRandomness(self, numbytes: int) -> bytes:
        """This method is called once on the default VFS when SQLite needs to
        seed the random number generator.  You can return less than the
        number of bytes requested including None. If you return more then
        the surplus is ignored."""
        ...

    def xSetSystemCall(self, name: Optional[str], pointer: int) -> bool:
        """Change a system call used by the VFS.  This is useful for testing
        and some other scenarios such as sandboxing.

        :param name: The string name of the system call

        :param pointer: A pointer provided as an int.  There is no
          reference counting or other memory tracking of the pointer.  If
          you provide one you need to ensure it is around for the lifetime
          of this and any other related VFS.

        Raise an exception to return an error.  If the system call does
        not exist then raise :exc:`NotFoundError`.

        If `name` is None, then all systemcalls are reset to their defaults.

        :returns: True if the system call was set.  False if the system
          call is not known."""
        ...

    def xSleep(self, microseconds: int) -> int:
        """Pause execution of the thread for at least the specified number of
        microseconds (millionths of a second).  This routine is typically called from the busy handler.

        :returns: How many microseconds you actually requested the
          operating system to sleep for. For example if your operating
          system sleep call only takes seconds then you would have to have
          rounded the microseconds number up to the nearest second and
          should return that rounded up value."""
        ...

class VTCursor(Protocol):
    """.. note::

      There is no actual *VTCursor* class - it is shown this way for
      documentation convenience and is present as a :class:`typing protocol
      <typing.Protocol>`.


    The :class:`VTCursor` object is used for iterating over a table.
    There may be many cursors simultaneously so each one needs to keep
    track of where in the table it is."""
    def Close(self) -> None:
        """This is the destructor for the cursor. Note that you must
        cleanup. The method will not be called again if you raise an
        exception."""
        ...

    def Column(self, number: int) -> SQLiteValue:
        """Requests the value of the specified column *number* of the current
        row.  If *number* is -1 then return the rowid.

        :returns: Must be one one of the :ref:`5
          supported types <types>`"""
        ...

    def ColumnNoChange(self, number: int) -> SQLiteValue:
        """:meth:`VTTable.UpdateChangeRow` is going to be called which includes
        values for all columns.  However this column is not going to be changed
        in that update.

        If you return :attr:`apsw.no_change` then :meth:`VTTable.UpdateChangeRow`
        will have :attr:`apsw.no_change` for this column.  If you return
        anything else then it will have that value - as though :meth:`VTCursor.Column`
        had been called.

        This method will only be called if *use_no_change* was *True* in the
        call to :meth:`Connection.create_module`.

        Calls: `sqlite3_vtab_nochange <https://sqlite.org/c3ref/vtab_nochange.html>`__"""
        ...

    def Eof(self) -> bool:
        """Called to ask if we are at the end of the table. It is called after each call to Filter and Next.

        :returns: False if the cursor is at a valid row of data, else True

        .. note::

          This method can only return True or False to SQLite.  If you have
          an exception in the method or provide a non-boolean return then
          True (no more data) will be returned to SQLite."""
        ...

    def Filter(self, indexnum: int, indexname: str, constraintargs: Optional[tuple]) -> None:
        """This method is always called first to initialize an iteration to the
        first row of the table. The arguments come from the
        :meth:`~VTTable.BestIndex` or :meth:`~VTTable.BestIndexObject`
        with constraintargs being a tuple of the constraints you
        requested. If you always return None in BestIndex then indexnum will
        be zero, indexstring will be None and constraintargs will be empty).

        If you had an *in* constraint and set :meth:`IndexInfo.set_aConstraintUsage_in`
        then that value will be a :class:`set`.

        Calls:
          * `sqlite3_vtab_in_first <https://sqlite.org/c3ref/vtab_in_first.html>`__
          * `sqlite3_vtab_in_next <https://sqlite.org/c3ref/vtab_in_first.html>`__"""
        ...

    def Next(self) -> None:
        """Move the cursor to the next row.  Do not have an exception if there
        is no next row.  Instead return False when :meth:`~VTCursor.Eof` is
        subsequently called.

        If you said you had indices in your :meth:`VTTable.BestIndex`
        return, and they were selected for use as provided in the parameters
        to :meth:`~VTCursor.Filter` then you should move to the next
        appropriate indexed and constrained row."""
        ...

    def Rowid(self) -> int:
        """Return the current rowid."""
        ...

class VTModule(Protocol):
    """.. note::

      There is no actual *VTModule* class - it is shown this way for
      documentation convenience and is present as a :class:`typing protocol
      <typing.Protocol>`.

    A module instance is used to create the virtual tables.  Once you have
    a module object, you register it with a connection by calling
    :meth:`Connection.create_module`::

      # make an instance
      mymod=MyModuleClass()

      # register the vtable on connection con
      con.create_module("modulename", mymod)

      # tell SQLite about the table
      con.execute("create VIRTUAL table tablename USING modulename('arg1', 2)")

    The create step is to tell SQLite about the existence of the table.
    Any number of tables referring to the same module can be made this
    way."""
    def Connect(self, connection: Connection, modulename: str, databasename: str, tablename: str, *args: tuple[SQLiteValue, ...])  -> tuple[str, VTTable]:
        """The parameters and return are identical to
        :meth:`~VTModule.Create`.  This method is called
        when there are additional references to the table.  :meth:`~VTModule.Create` will be called the first time and
        :meth:`~VTModule.Connect` after that.

        The advise is to create caches, generated data and other
        heavyweight processing on :meth:`~VTModule.Create` calls and then
        find and reuse that on the subsequent :meth:`~VTModule.Connect`
        calls.

        The corresponding call is :meth:`VTTable.Disconnect`.  If you have a simple virtual table implementation, then just
        set :meth:`~VTModule.Connect` to be the same as :meth:`~VTModule.Create`::

          class MyModule:

               def Create(self, connection, modulename, databasename, tablename, *args):
                   # do lots of hard work

               Connect=Create"""
        ...

    def Create(self, connection: Connection, modulename: str, databasename: str, tablename: str, *args: tuple[SQLiteValue, ...])  -> tuple[str, VTTable]:
        """Called when a table is first created on a :class:`connection
        <Connection>`.

        :param connection: An instance of :class:`Connection`
        :param modulename: The string name under which the module was :meth:`registered <Connection.create_module>`
        :param databasename: The name of the database.  `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__
        :param tablename: Name of the table the user wants to create.
        :param args: Any arguments that were specified in the `create virtual table <https://sqlite.org/lang_createvtab.html>`_ statement.

        :returns: A list of two items.  The first is a SQL `create table <https://sqlite.org/lang_createtable.html>`_ statement.  The
             columns are parsed so that SQLite knows what columns and declared types exist for the table.  The second item
             is an object that implements the :class:`table <VTTable>` methods.

        The corresponding call is :meth:`VTTable.Destroy`."""
        ...

    def ShadowName(self, table_suffix: str) -> bool:
        """This method is called to check if
        *table_suffix* is a `shadow name
        <https://www.sqlite.org/vtab.html#the_xshadowname_method>`__

        The default implementation always returns *False*.

        If a virtual table is created using this module
        named :code:`example` and then a  real table is created
        named :code:`example_content`, this would be called with
        a *table_suffix* of :code:`content`"""
        ...

class VTTable(Protocol):
    """.. note::

      There is no actual *VTTable* class - it is shown this way for
      documentation convenience and is present as a :class:`typing protocol
      <typing.Protocol>`.

    The :class:`VTTable` object contains knowledge of the indices, makes
    cursors and can perform transactions.

    A virtual table is structured as a series of rows, each of which has
    the same number of columns.  The value in a column must be one of the `5
    supported types <https://sqlite.org/datatype3.html>`_, but the
    type can be different between rows for the same column.  The virtual
    table routines identify the columns by number, starting at zero.

    Each row has a **unique** 64 bit integer `rowid
    <https://sqlite.org/autoinc.html>`_ with the :class:`Cursor
    <VTCursor>` routines operating on this number, as well as some of
    the :class:`Table <VTTable>` routines such as :meth:`UpdateChangeRow
    <VTTable.UpdateChangeRow>`.

    It is possible to `not have a rowid
    <https://www.sqlite.org/vtab.html#_without_rowid_virtual_tables_>`__"""
    def Begin(self) -> None:
        """This function is used as part of transactions.  You do not have to
        provide the method."""
        ...

    def BestIndex(self, constraints: Sequence[tuple[int, int]], orderbys: Sequence[tuple[int, int]]) -> Any:
        """This is a complex method. To get going initially, just return
        *None* and you will be fine. You should also consider using
        :meth:`BestIndexObject` instead.

        Implementing this method reduces the number of rows scanned
        in your table to satisfy queries, but only if you have an
        index or index like mechanism available.

        .. note::

          The implementation of this method differs slightly from the
          `SQLite documentation
          <https://sqlite.org/vtab.html>`__
          for the C API. You are not passed "unusable" constraints. The
          argv/constraintarg positions are not off by one. In the C api, you
          have to return position 1 to get something passed to
          :meth:`VTCursor.Filter` in position 0. With the APSW
          implementation, you return position 0 to get Filter arg 0,
          position 1 to get Filter arg 1 etc.

        The purpose of this method is to ask if you have the ability to
        determine if a row meets certain constraints that doesn't involve
        visiting every row. An example constraint is ``price > 74.99``. In a
        traditional SQL database, queries with constraints can be speeded up
        `with indices <https://sqlite.org/lang_createindex.html>`_. If
        you return None, then SQLite will visit every row in your table and
        evaluate the constraints itself. Your index choice returned from
        BestIndex will also be passed to the :meth:`~VTCursor.Filter` method on your cursor
        object. Note that SQLite may call this method multiple times trying
        to find the most efficient way of answering a complex query.

        **constraints**

        You will be passed the constraints as a sequence of tuples containing two
        items. The first item is the column number and the second item is
        the operation.

           Example query: ``select * from foo where price > 74.99 and
           quantity<=10 and customer='Acme Widgets'``

           If customer is column 0, price column 2 and quantity column 5
           then the constraints will be::

             (2, apsw.SQLITE_INDEX_CONSTRAINT_GT),
             (5, apsw.SQLITE_INDEX_CONSTRAINT_LE),
             (0, apsw.SQLITE_INDEX_CONSTRAINT_EQ)

           Note that you do not get the value of the constraint (ie "Acme
           Widgets", 74.99 and 10 in this example).

        If you do have any suitable indices then you return a sequence the
        same length as constraints with the members mapping to the
        constraints in order. Each can be one of None, an integer or a tuple
        of an integer and a boolean.  Conceptually SQLite is giving you a
        list of constraints and you are returning a list of the same length
        describing how you could satisfy each one.

        Each list item returned corresponding to a constraint is one of:

           None
             This means you have no index for that constraint. SQLite
             will have to iterate over every row for it.

           integer
             This is the argument number for the constraintargs being passed
             into the :meth:`~VTCursor.Filter` function of your
             :class:`cursor <VTCursor>` (the values "Acme Widgets", 74.99
             and 10 in the example).

           (integer, boolean)
             By default SQLite will check what you return. For example if
             you said that you had an index on price and so would only
             return rows greater than 74.99, then SQLite will still
             check that each row you returned is greater than 74.99.
             If the boolean is True then SQLite will not double
             check, while False retains the default double checking.

        Example query: ``select * from foo where price > 74.99 and
        quantity<=10 and customer=='Acme Widgets'``.  customer is column 0,
        price column 2 and quantity column 5.  You can index on customer
        equality and price.

        +----------------------------------------+--------------------------------+
        | Constraints (in)                       | Constraints used (out)         |
        +========================================+================================+
        | ::                                     | ::                             |
        |                                        |                                |
        |  (2, apsw.SQLITE_INDEX_CONSTRAINT_GT), |     1,                         |
        |  (5, apsw.SQLITE_INDEX_CONSTRAINT_LE), |     None,                      |
        |  (0, apsw.SQLITE_INDEX_CONSTRAINT_EQ)  |     0                          |
        |                                        |                                |
        +----------------------------------------+--------------------------------+

        When your :class:`~VTCursor.Filter` method in the cursor is called,
        constraintarg[0] will be "Acme Widgets" (customer constraint value)
        and constraintarg[1] will be 74.99 (price constraint value). You can
        also return an index number (integer) and index string to use
        (SQLite attaches no significance to these values - they are passed
        as is to your :meth:`VTCursor.Filter` method as a way for the
        BestIndex method to let the :meth:`~VTCursor.Filter` method know
        which of your indices or similar mechanism to use.

        **orderbys**


        The second argument to BestIndex is a sequence of orderbys because
        the query requested the results in a certain order. If your data is
        already in that order then SQLite can give the results back as
        is. If not, then SQLite will have to sort the results first.

          Example query: ``select * from foo order by price desc, quantity asc``

          Price is column 2, quantity column 5 so orderbys will be::

            (2, True),  # True means descending, False is ascending
            (5, False)

        **Return**

        You should return up to 5 items. Items not present in the return have a default value.

        0: constraints used (default None)
          This must either be None or a sequence the same length as
          constraints passed in. Each item should be as specified above
          saying if that constraint is used, and if so which constraintarg
          to make the value be in your :meth:`VTCursor.Filter` function.

        1: index number (default zero)
          This value is passed as is to :meth:`VTCursor.Filter`

        2: index string (default None)
          This value is passed as is to :meth:`VTCursor.Filter`

        3: orderby consumed (default False)
          Return True if your output will be in exactly the same order as the orderbys passed in

        4: estimated cost (default a huge number)
          Approximately how many disk operations are needed to provide the
          results. SQLite uses the cost to optimise queries. For example if
          the query includes *A or B* and A has 2,000 operations and B has 100
          then it is best to evaluate B before A.

        **A complete example**

        Query is ``select * from foo where price>74.99 and quantity<=10 and
        customer=="Acme Widgets" order by price desc, quantity asc``.
        Customer is column 0, price column 2 and quantity column 5. You can
        index on customer equality and price.

        ::

          BestIndex(constraints, orderbys)

          constraints= ( (2, apsw.SQLITE_INDEX_CONSTRAINT_GT),
                         (5, apsw.SQLITE_INDEX_CONSTRAINT_LE),
                         (0, apsw.SQLITE_INDEX_CONSTRAINT_EQ)  )

          orderbys= ( (2, True), (5, False) )


          # You return

          ( (1, None, 0),   # constraints used
            27,             # index number
            "idx_pr_cust",  # index name
            False,          # results are not in orderbys order
            1000            # about 1000 disk operations to access index
          )


          # Your Cursor.Filter method will be called with:

          27,              # index number you returned
          "idx_pr_cust",   # index name you returned
          "Acme Widgets",  # constraintarg[0] - customer
          74.99            # constraintarg[1] - price"""
        ...

    def BestIndexObject(self, index_info: IndexInfo) -> bool:
        """This method is called instead of :meth:`BestIndex` if
        *use_bestindex_object* was *True* in the call to
        :meth:`Connection.create_module`.

        Use the :class:`IndexInfo` to tell SQLite about your indexes, and
        extract other information.

        Return *True* to indicate all is well.  If you return *False* or there is an error,
        then `SQLITE_CONSTRAINT
        <https://www.sqlite.org/vtab.html#return_value>`__ is returned to
        SQLite."""
        ...

    def Commit(self) -> None:
        """This function is used as part of transactions.  You do not have to
        provide the method."""
        ...

    def Destroy(self) -> None:
        """The opposite of :meth:`VTModule.Create`.  This method is called when
        the table is no longer used.  Note that you must always release
        resources even if you intend to return an error, as it will not be
        called again on error."""
        ...

    def Disconnect(self) -> None:
        """The opposite of :meth:`VTModule.Connect`.  This method is called when
        a reference to a virtual table is no longer used, but :meth:`VTTable.Destroy` will
        be called when the table is no longer used."""
        ...

    def FindFunction(self, name: str, nargs: int) -> None |  Callable | tuple[int, Callable]:
        """Called to find if the virtual table has its own implementation of a
        particular scalar function. You do not have to provide this method.

        :param name: The function name
        :param nargs: How many arguments the function takes

        Return *None* if you don't have the function.  Zero is then returned to SQLite.

        Return a callable if you have one.  One is then returned to SQLite with the function.

        Return a sequence of int, callable.  The int is returned to SQLite with the function.
        This is useful for *SQLITE_INDEX_CONSTRAINT_FUNCTION* returns.

        It isn't possible to tell SQLite about exceptions in this function, so an
        :ref:`unraisable exception <unraisable>` is used.

        .. seealso::

          * :meth:`Connection.overload_function`"""
        ...

    def Integrity(self, schema: str, name: str, is_quick: int) -> str | None:
        """If present, check the integrity of the virtual table.

        :param schema: Database name `main`, `temp`, the name in `ATTACH <https://sqlite.org/lang_attach.html>`__
        :param name: Name of the table
        :param is_quick: 0 if `pragma integrity_check <https://sqlite.org/pragma.html#pragma_integrity_check>`__ was used,
           1 if `pragma quick_check <https://sqlite.org/pragma.html#pragma_quick_check>`__ was used, and may contain
           other values in the future.

        :returns: None if there are no problems, else a string to be used as an error message.  The string is returned to the
          pragma as is, so it is recommended that you include the database and table name to clarify what database and
          table the message is referring to."""
        ...

    def Open(self) -> VTCursor:
        """Returns a :class:`cursor <VTCursor>` object."""
        ...

    def Release(self, level: int) -> None:
        """Release nested transactions back to *level*.

        If you do not provide this method then the call succeeds (matching
        SQLite behaviour when no callback is provided)."""
        ...

    def Rename(self, newname: str) -> None:
        """Notification that the table will be given a new name. If you return
        without raising an exception, then SQLite renames the table (you
        don't have to do anything). If you raise an exception then the
        renaming is prevented.  You do not have to provide this method."""
        ...

    def Rollback(self) -> None:
        """This function is used as part of transactions.  You do not have to
        provide the method."""
        ...

    def Savepoint(self, level: int) -> None:
        """Set nested transaction to *level*.

        If you do not provide this method then the call succeeds (matching
        SQLite behaviour when no callback is provided)."""
        ...

    def Sync(self) -> None:
        """This function is used as part of transactions.  You do not have to
        provide the method."""
        ...

    def UpdateChangeRow(self, row: int, newrowid: int, fields: tuple[SQLiteValue, ...]) -> None:
        """Change an existing row.  You may also need to change the rowid - for example if the query was
        ``UPDATE table SET rowid=rowid+100 WHERE ...``

        :param row: The existing 64 bit integer rowid
        :param newrowid: If not the same as *row* then also change the rowid to this.
        :param fields: A tuple of values the same length and order as columns in your table"""
        ...

    def UpdateDeleteRow(self, rowid: int) -> None:
        """Delete the row with the specified *rowid*.

        :param rowid: 64 bit integer"""
        ...

    def UpdateInsertRow(self, rowid: Optional[int], fields: tuple[SQLiteValue, ...])  -> Optional[int]:
        """Insert a row with the specified *rowid*.

        :param rowid: *None* if you should choose the rowid yourself, else a 64 bit integer
        :param fields: A tuple of values the same length and order as columns in your table

        :returns: If *rowid* was *None* then return the id you assigned
          to the row.  If *rowid* was not *None* then the return value
          is ignored."""
        ...

class zeroblob:
    """If you want to insert a blob into a row, you need to
    supply the entire blob in one go.  Using this class or
    `function <https://www.sqlite.org/lang_corefunc.html#zeroblob>`__
    allocates the space in the database filling it with zeroes.

    You can then overwrite parts in smaller chunks, without having
    to do it all at once.  The :ref:`example <example_blob_io>` shows
    how to use it."""
    def __init__(self, size: int):
        """:param size: Number of zeroed bytes to create"""
        ...

    def length(self) -> int:
        """Size of zero blob in bytes."""
        ...



FTS5_TOKENIZE_AUX: int = 8
"""For `FTS5 Tokenize Reason <https://sqlite.org/fts5.html>'__"""
FTS5_TOKENIZE_DOCUMENT: int = 4
"""For `FTS5 Tokenize Reason <https://sqlite.org/fts5.html>'__"""
FTS5_TOKENIZE_PREFIX: int = 2
"""For `FTS5 Tokenize Reason <https://sqlite.org/fts5.html>'__"""
FTS5_TOKENIZE_QUERY: int = 1
"""For `FTS5 Tokenize Reason <https://sqlite.org/fts5.html>'__"""
SQLITE_ABORT: int = 4
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_ABORT_ROLLBACK: int = 516
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_ACCESS_EXISTS: int = 0
"""For `Flags for the xAccess VFS method <https://sqlite.org/c3ref/c_access_exists.html>'__"""
SQLITE_ACCESS_READ: int = 2
"""For `Flags for the xAccess VFS method <https://sqlite.org/c3ref/c_access_exists.html>'__"""
SQLITE_ACCESS_READWRITE: int = 1
"""For `Flags for the xAccess VFS method <https://sqlite.org/c3ref/c_access_exists.html>'__"""
SQLITE_ALTER_TABLE: int = 26
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_ANALYZE: int = 28
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_ATTACH: int = 24
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_AUTH: int = 23
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_AUTH_USER: int = 279
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_BUSY: int = 5
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_BUSY_RECOVERY: int = 261
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_BUSY_SNAPSHOT: int = 517
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_BUSY_TIMEOUT: int = 773
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CANTOPEN: int = 14
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CANTOPEN_CONVPATH: int = 1038
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CANTOPEN_DIRTYWAL: int = 1294
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CANTOPEN_FULLPATH: int = 782
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CANTOPEN_ISDIR: int = 526
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CANTOPEN_NOTEMPDIR: int = 270
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CANTOPEN_SYMLINK: int = 1550
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CHECKPOINT_FULL: int = 1
"""For `Checkpoint Mode Values <https://sqlite.org/c3ref/c_checkpoint_full.html>'__"""
SQLITE_CHECKPOINT_PASSIVE: int = 0
"""For `Checkpoint Mode Values <https://sqlite.org/c3ref/c_checkpoint_full.html>'__"""
SQLITE_CHECKPOINT_RESTART: int = 2
"""For `Checkpoint Mode Values <https://sqlite.org/c3ref/c_checkpoint_full.html>'__"""
SQLITE_CHECKPOINT_TRUNCATE: int = 3
"""For `Checkpoint Mode Values <https://sqlite.org/c3ref/c_checkpoint_full.html>'__"""
SQLITE_CONFIG_COVERING_INDEX_SCAN: int = 20
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_GETMALLOC: int = 5
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_GETMUTEX: int = 11
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_GETPCACHE: int = 15
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_GETPCACHE2: int = 19
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_HEAP: int = 8
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_LOG: int = 16
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_LOOKASIDE: int = 13
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_MALLOC: int = 4
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_MEMDB_MAXSIZE: int = 29
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_MEMSTATUS: int = 9
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_MMAP_SIZE: int = 22
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_MULTITHREAD: int = 2
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_MUTEX: int = 10
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_PAGECACHE: int = 7
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_PCACHE: int = 14
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_PCACHE2: int = 18
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_PCACHE_HDRSZ: int = 24
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_PMASZ: int = 25
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_SCRATCH: int = 6
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_SERIALIZED: int = 3
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_SINGLETHREAD: int = 1
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_SMALL_MALLOC: int = 27
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_SORTERREF_SIZE: int = 28
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_SQLLOG: int = 21
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_STMTJRNL_SPILL: int = 26
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_URI: int = 17
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONFIG_WIN32_HEAPSIZE: int = 23
"""For `Configuration Options <https://sqlite.org/c3ref/c_config_covering_index_scan.html>'__"""
SQLITE_CONSTRAINT: int = 19
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_CHECK: int = 275
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_COMMITHOOK: int = 531
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_DATATYPE: int = 3091
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_FOREIGNKEY: int = 787
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_FUNCTION: int = 1043
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_NOTNULL: int = 1299
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_PINNED: int = 2835
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_PRIMARYKEY: int = 1555
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_ROWID: int = 2579
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_TRIGGER: int = 1811
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_UNIQUE: int = 2067
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CONSTRAINT_VTAB: int = 2323
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_COPY: int = 0
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CORRUPT: int = 11
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CORRUPT_INDEX: int = 779
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CORRUPT_SEQUENCE: int = 523
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CORRUPT_VTAB: int = 267
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_CREATE_INDEX: int = 1
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_TABLE: int = 2
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_TEMP_INDEX: int = 3
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_TEMP_TABLE: int = 4
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_TEMP_TRIGGER: int = 5
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_TEMP_VIEW: int = 6
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_TRIGGER: int = 7
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_VIEW: int = 8
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_CREATE_VTABLE: int = 29
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DBCONFIG_DEFENSIVE: int = 1010
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_DQS_DDL: int = 1014
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_DQS_DML: int = 1013
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_ATTACH_CREATE: int = 1020
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_ATTACH_WRITE: int = 1021
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_COMMENTS: int = 1022
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_FKEY: int = 1002
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_FTS3_TOKENIZER: int = 1004
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_LOAD_EXTENSION: int = 1005
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_QPSG: int = 1007
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_TRIGGER: int = 1003
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_ENABLE_VIEW: int = 1015
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_LEGACY_ALTER_TABLE: int = 1012
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_LEGACY_FILE_FORMAT: int = 1016
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_LOOKASIDE: int = 1001
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_MAINDBNAME: int = 1000
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_MAX: int = 1022
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_NO_CKPT_ON_CLOSE: int = 1006
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_RESET_DATABASE: int = 1009
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_REVERSE_SCANORDER: int = 1019
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_STMT_SCANSTATUS: int = 1018
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_TRIGGER_EQP: int = 1008
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_TRUSTED_SCHEMA: int = 1017
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBCONFIG_WRITABLE_SCHEMA: int = 1011
"""For `Database Connection Configuration Options <https://sqlite.org/c3ref/c_dbconfig_defensive.html>'__"""
SQLITE_DBSTATUS_CACHE_HIT: int = 7
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_CACHE_MISS: int = 8
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_CACHE_SPILL: int = 12
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_CACHE_USED: int = 1
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_CACHE_USED_SHARED: int = 11
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_CACHE_WRITE: int = 9
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_DEFERRED_FKS: int = 10
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_LOOKASIDE_HIT: int = 4
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_LOOKASIDE_MISS_FULL: int = 6
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_LOOKASIDE_MISS_SIZE: int = 5
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_LOOKASIDE_USED: int = 0
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_MAX: int = 12
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_SCHEMA_USED: int = 2
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DBSTATUS_STMT_USED: int = 3
"""For `Status Parameters for database connections <https://sqlite.org/c3ref/c_dbstatus_options.html>'__"""
SQLITE_DELETE: int = 9
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DENY: int = 1
"""For `Authorizer Return Codes <https://sqlite.org/c3ref/c_deny.html>'__"""
SQLITE_DETACH: int = 25
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DETERMINISTIC: int = 2048
"""For `Function Flags <https://sqlite.org/c3ref/c_deterministic.html>'__"""
SQLITE_DIRECTONLY: int = 524288
"""For `Function Flags <https://sqlite.org/c3ref/c_deterministic.html>'__"""
SQLITE_DONE: int = 101
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_DROP_INDEX: int = 10
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_TABLE: int = 11
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_TEMP_INDEX: int = 12
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_TEMP_TABLE: int = 13
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_TEMP_TRIGGER: int = 14
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_TEMP_VIEW: int = 15
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_TRIGGER: int = 16
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_VIEW: int = 17
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_DROP_VTABLE: int = 30
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_EMPTY: int = 16
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_ERROR: int = 1
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_ERROR_MISSING_COLLSEQ: int = 257
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_ERROR_RETRY: int = 513
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_ERROR_SNAPSHOT: int = 769
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_FAIL: int = 3
"""For `Conflict resolution modes <https://sqlite.org/c3ref/c_fail.html>'__"""
SQLITE_FCNTL_BEGIN_ATOMIC_WRITE: int = 31
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_BUSYHANDLER: int = 15
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_CHUNK_SIZE: int = 6
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_CKPT_DONE: int = 37
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_CKPT_START: int = 39
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_CKSM_FILE: int = 41
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_COMMIT_ATOMIC_WRITE: int = 32
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_COMMIT_PHASETWO: int = 22
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_DATA_VERSION: int = 35
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_EXTERNAL_READER: int = 40
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_FILE_POINTER: int = 7
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_GET_LOCKPROXYFILE: int = 2
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_HAS_MOVED: int = 20
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_JOURNAL_POINTER: int = 28
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_LAST_ERRNO: int = 4
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_LOCKSTATE: int = 1
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_LOCK_TIMEOUT: int = 34
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_MMAP_SIZE: int = 18
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_NULL_IO: int = 43
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_OVERWRITE: int = 11
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_PDB: int = 30
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_PERSIST_WAL: int = 10
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_POWERSAFE_OVERWRITE: int = 13
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_PRAGMA: int = 14
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_RBU: int = 26
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_RESERVE_BYTES: int = 38
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_RESET_CACHE: int = 42
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_ROLLBACK_ATOMIC_WRITE: int = 33
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_SET_LOCKPROXYFILE: int = 3
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_SIZE_HINT: int = 5
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_SIZE_LIMIT: int = 36
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_SYNC: int = 21
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_SYNC_OMITTED: int = 8
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_TEMPFILENAME: int = 16
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_TRACE: int = 19
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_VFSNAME: int = 12
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_VFS_POINTER: int = 27
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_WAL_BLOCK: int = 24
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_WIN32_AV_RETRY: int = 9
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_WIN32_GET_HANDLE: int = 29
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_WIN32_SET_HANDLE: int = 23
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FCNTL_ZIPVFS: int = 25
"""For `Standard File Control Opcodes <https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html>'__"""
SQLITE_FORMAT: int = 24
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_FULL: int = 13
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_FUNCTION: int = 31
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_IGNORE: int = 2
"""For `Authorizer Return Codes <https://sqlite.org/c3ref/c_deny.html>'__"""
SQLITE_INDEX_CONSTRAINT_EQ: int = 2
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_FUNCTION: int = 150
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_GE: int = 32
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_GLOB: int = 66
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_GT: int = 4
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_IS: int = 72
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_ISNOT: int = 69
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_ISNOTNULL: int = 70
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_ISNULL: int = 71
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_LE: int = 8
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_LIKE: int = 65
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_LIMIT: int = 73
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_LT: int = 16
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_MATCH: int = 64
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_NE: int = 68
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_OFFSET: int = 74
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_CONSTRAINT_REGEXP: int = 67
"""For `Virtual Table Constraint Operator Codes <https://sqlite.org/c3ref/c_index_constraint_eq.html>'__"""
SQLITE_INDEX_SCAN_HEX: int = 2
"""For `Virtual Table Scan Flags <https://sqlite.org/c3ref/c_index_scan_hex.html>'__"""
SQLITE_INDEX_SCAN_UNIQUE: int = 1
"""For `Virtual Table Scan Flags <https://sqlite.org/c3ref/c_index_scan_hex.html>'__"""
SQLITE_INNOCUOUS: int = 2097152
"""For `Function Flags <https://sqlite.org/c3ref/c_deterministic.html>'__"""
SQLITE_INSERT: int = 18
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_INTERNAL: int = 2
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_INTERRUPT: int = 9
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOCAP_ATOMIC: int = 1
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC16K: int = 64
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC1K: int = 4
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC2K: int = 8
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC32K: int = 128
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC4K: int = 16
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC512: int = 2
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC64K: int = 256
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_ATOMIC8K: int = 32
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_BATCH_ATOMIC: int = 16384
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_IMMUTABLE: int = 8192
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_POWERSAFE_OVERWRITE: int = 4096
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_SAFE_APPEND: int = 512
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_SEQUENTIAL: int = 1024
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_SUBPAGE_READ: int = 32768
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOCAP_UNDELETABLE_WHEN_OPEN: int = 2048
"""For `Device Characteristics <https://sqlite.org/c3ref/c_iocap_atomic.html>'__"""
SQLITE_IOERR: int = 10
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_ACCESS: int = 3338
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_AUTH: int = 7178
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_BEGIN_ATOMIC: int = 7434
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_BLOCKED: int = 2826
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_CHECKRESERVEDLOCK: int = 3594
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_CLOSE: int = 4106
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_COMMIT_ATOMIC: int = 7690
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_CONVPATH: int = 6666
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_CORRUPTFS: int = 8458
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_DATA: int = 8202
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_DELETE: int = 2570
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_DELETE_NOENT: int = 5898
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_DIR_CLOSE: int = 4362
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_DIR_FSYNC: int = 1290
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_FSTAT: int = 1802
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_FSYNC: int = 1034
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_GETTEMPPATH: int = 6410
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_IN_PAGE: int = 8714
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_LOCK: int = 3850
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_MMAP: int = 6154
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_NOMEM: int = 3082
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_RDLOCK: int = 2314
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_READ: int = 266
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_ROLLBACK_ATOMIC: int = 7946
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_SEEK: int = 5642
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_SHMLOCK: int = 5130
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_SHMMAP: int = 5386
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_SHMOPEN: int = 4618
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_SHMSIZE: int = 4874
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_SHORT_READ: int = 522
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_TRUNCATE: int = 1546
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_UNLOCK: int = 2058
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_VNODE: int = 6922
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_IOERR_WRITE: int = 778
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_LIMIT_ATTACHED: int = 7
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_COLUMN: int = 2
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_COMPOUND_SELECT: int = 4
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_EXPR_DEPTH: int = 3
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_FUNCTION_ARG: int = 6
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_LENGTH: int = 0
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_LIKE_PATTERN_LENGTH: int = 8
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_SQL_LENGTH: int = 1
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_TRIGGER_DEPTH: int = 10
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_VARIABLE_NUMBER: int = 9
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_VDBE_OP: int = 5
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LIMIT_WORKER_THREADS: int = 11
"""For `Run-Time Limit Categories <https://sqlite.org/c3ref/c_limit_attached.html>'__"""
SQLITE_LOCKED: int = 6
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_LOCKED_SHAREDCACHE: int = 262
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_LOCKED_VTAB: int = 518
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_LOCK_EXCLUSIVE: int = 4
"""For `File Locking Levels <https://sqlite.org/c3ref/c_lock_exclusive.html>'__"""
SQLITE_LOCK_NONE: int = 0
"""For `File Locking Levels <https://sqlite.org/c3ref/c_lock_exclusive.html>'__"""
SQLITE_LOCK_PENDING: int = 3
"""For `File Locking Levels <https://sqlite.org/c3ref/c_lock_exclusive.html>'__"""
SQLITE_LOCK_RESERVED: int = 2
"""For `File Locking Levels <https://sqlite.org/c3ref/c_lock_exclusive.html>'__"""
SQLITE_LOCK_SHARED: int = 1
"""For `File Locking Levels <https://sqlite.org/c3ref/c_lock_exclusive.html>'__"""
SQLITE_MISMATCH: int = 20
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_MISUSE: int = 21
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOLFS: int = 22
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOMEM: int = 7
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOTADB: int = 26
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOTFOUND: int = 12
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOTICE: int = 27
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOTICE_RBU: int = 795
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOTICE_RECOVER_ROLLBACK: int = 539
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_NOTICE_RECOVER_WAL: int = 283
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_OK: int = 0
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_OK_LOAD_PERMANENTLY: int = 256
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_OK_SYMLINK: int = 512
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_OPEN_AUTOPROXY: int = 32
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_CREATE: int = 4
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_DELETEONCLOSE: int = 8
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_EXCLUSIVE: int = 16
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_EXRESCODE: int = 33554432
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_FULLMUTEX: int = 65536
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_MAIN_DB: int = 256
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_MAIN_JOURNAL: int = 2048
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_MEMORY: int = 128
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_NOFOLLOW: int = 16777216
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_NOMUTEX: int = 32768
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_PRIVATECACHE: int = 262144
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_READONLY: int = 1
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_READWRITE: int = 2
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_SHAREDCACHE: int = 131072
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_SUBJOURNAL: int = 8192
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_SUPER_JOURNAL: int = 16384
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_TEMP_DB: int = 512
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_TEMP_JOURNAL: int = 4096
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_TRANSIENT_DB: int = 1024
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_URI: int = 64
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_OPEN_WAL: int = 524288
"""For `Flags For File Open Operations <https://sqlite.org/c3ref/c_open_autoproxy.html>'__"""
SQLITE_PERM: int = 3
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_PRAGMA: int = 19
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_PREPARE_DONT_LOG: int = 16
"""For `Prepare Flags <https://sqlite.org/c3ref/c_prepare_dont_log.html>'__"""
SQLITE_PREPARE_NORMALIZE: int = 2
"""For `Prepare Flags <https://sqlite.org/c3ref/c_prepare_dont_log.html>'__"""
SQLITE_PREPARE_NO_VTAB: int = 4
"""For `Prepare Flags <https://sqlite.org/c3ref/c_prepare_dont_log.html>'__"""
SQLITE_PREPARE_PERSISTENT: int = 1
"""For `Prepare Flags <https://sqlite.org/c3ref/c_prepare_dont_log.html>'__"""
SQLITE_PROTOCOL: int = 15
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_RANGE: int = 25
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_READ: int = 20
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_READONLY: int = 8
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_READONLY_CANTINIT: int = 1288
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_READONLY_CANTLOCK: int = 520
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_READONLY_DBMOVED: int = 1032
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_READONLY_DIRECTORY: int = 1544
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_READONLY_RECOVERY: int = 264
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_READONLY_ROLLBACK: int = 776
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_RECURSIVE: int = 33
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_REINDEX: int = 27
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_REPLACE: int = 5
"""For `Conflict resolution modes <https://sqlite.org/c3ref/c_fail.html>'__"""
SQLITE_RESULT_SUBTYPE: int = 16777216
"""For `Function Flags <https://sqlite.org/c3ref/c_deterministic.html>'__"""
SQLITE_ROLLBACK: int = 1
"""For `Conflict resolution modes <https://sqlite.org/c3ref/c_fail.html>'__"""
SQLITE_ROW: int = 100
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_SAVEPOINT: int = 32
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_SCHEMA: int = 17
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_SELECT: int = 21
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_SELFORDER1: int = 33554432
"""For `Function Flags <https://sqlite.org/c3ref/c_deterministic.html>'__"""
SQLITE_SHM_EXCLUSIVE: int = 8
"""For `Flags for the xShmLock VFS method <https://sqlite.org/c3ref/c_shm_exclusive.html>'__"""
SQLITE_SHM_LOCK: int = 2
"""For `Flags for the xShmLock VFS method <https://sqlite.org/c3ref/c_shm_exclusive.html>'__"""
SQLITE_SHM_SHARED: int = 4
"""For `Flags for the xShmLock VFS method <https://sqlite.org/c3ref/c_shm_exclusive.html>'__"""
SQLITE_SHM_UNLOCK: int = 1
"""For `Flags for the xShmLock VFS method <https://sqlite.org/c3ref/c_shm_exclusive.html>'__"""
SQLITE_STATUS_MALLOC_COUNT: int = 9
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_MALLOC_SIZE: int = 5
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_MEMORY_USED: int = 0
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_PAGECACHE_OVERFLOW: int = 2
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_PAGECACHE_SIZE: int = 7
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_PAGECACHE_USED: int = 1
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_PARSER_STACK: int = 6
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_SCRATCH_OVERFLOW: int = 4
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_SCRATCH_SIZE: int = 8
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STATUS_SCRATCH_USED: int = 3
"""For `Status Parameters <https://sqlite.org/c3ref/c_status_malloc_count.html>'__"""
SQLITE_STMTSTATUS_AUTOINDEX: int = 3
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_FILTER_HIT: int = 8
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_FILTER_MISS: int = 7
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_FULLSCAN_STEP: int = 1
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_MEMUSED: int = 99
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_REPREPARE: int = 5
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_RUN: int = 6
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_SORT: int = 2
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_STMTSTATUS_VM_STEP: int = 4
"""For `Status Parameters for prepared statements <https://sqlite.org/c3ref/c_stmtstatus_counter.html>'__"""
SQLITE_SUBTYPE: int = 1048576
"""For `Function Flags <https://sqlite.org/c3ref/c_deterministic.html>'__"""
SQLITE_SYNC_DATAONLY: int = 16
"""For `Synchronization Type Flags <https://sqlite.org/c3ref/c_sync_dataonly.html>'__"""
SQLITE_SYNC_FULL: int = 3
"""For `Synchronization Type Flags <https://sqlite.org/c3ref/c_sync_dataonly.html>'__"""
SQLITE_SYNC_NORMAL: int = 2
"""For `Synchronization Type Flags <https://sqlite.org/c3ref/c_sync_dataonly.html>'__"""
SQLITE_TOOBIG: int = 18
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_TRACE_CLOSE: int = 8
"""For `SQL Trace Event Codes <https://sqlite.org/c3ref/c_trace.html>'__"""
SQLITE_TRACE_PROFILE: int = 2
"""For `SQL Trace Event Codes <https://sqlite.org/c3ref/c_trace.html>'__"""
SQLITE_TRACE_ROW: int = 4
"""For `SQL Trace Event Codes <https://sqlite.org/c3ref/c_trace.html>'__"""
SQLITE_TRACE_STMT: int = 1
"""For `SQL Trace Event Codes <https://sqlite.org/c3ref/c_trace.html>'__"""
SQLITE_TRANSACTION: int = 22
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_TXN_NONE: int = 0
"""For `Allowed return values from sqlite3_txn_state() <https://sqlite.org/c3ref/c_txn_none.html>'__"""
SQLITE_TXN_READ: int = 1
"""For `Allowed return values from sqlite3_txn_state() <https://sqlite.org/c3ref/c_txn_none.html>'__"""
SQLITE_TXN_WRITE: int = 2
"""For `Allowed return values from sqlite3_txn_state() <https://sqlite.org/c3ref/c_txn_none.html>'__"""
SQLITE_UPDATE: int = 23
"""For `Authorizer Action Codes <https://sqlite.org/c3ref/c_alter_table.html>'__"""
SQLITE_VTAB_CONSTRAINT_SUPPORT: int = 1
"""For `Virtual Table Configuration Options <https://sqlite.org/c3ref/c_vtab_constraint_support.html>'__"""
SQLITE_VTAB_DIRECTONLY: int = 3
"""For `Virtual Table Configuration Options <https://sqlite.org/c3ref/c_vtab_constraint_support.html>'__"""
SQLITE_VTAB_INNOCUOUS: int = 2
"""For `Virtual Table Configuration Options <https://sqlite.org/c3ref/c_vtab_constraint_support.html>'__"""
SQLITE_VTAB_USES_ALL_SCHEMAS: int = 4
"""For `Virtual Table Configuration Options <https://sqlite.org/c3ref/c_vtab_constraint_support.html>'__"""
SQLITE_WARNING: int = 28
"""For `Result Codes <https://sqlite.org/rescode.html>'__"""
SQLITE_WARNING_AUTOINDEX: int = 284
"""For `Extended Result Codes <https://sqlite.org/rescode.html>'__"""


mapping_access: dict[str | int, int | str]
"""Flags for the xAccess VFS method mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_access_exists.html

SQLITE_ACCESS_EXISTS SQLITE_ACCESS_READ SQLITE_ACCESS_READWRITE"""

mapping_authorizer_function: dict[str | int, int | str]
"""Authorizer Action Codes mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_alter_table.html

SQLITE_ALTER_TABLE SQLITE_ANALYZE SQLITE_ATTACH SQLITE_COPY
SQLITE_CREATE_INDEX SQLITE_CREATE_TABLE SQLITE_CREATE_TEMP_INDEX
SQLITE_CREATE_TEMP_TABLE SQLITE_CREATE_TEMP_TRIGGER
SQLITE_CREATE_TEMP_VIEW SQLITE_CREATE_TRIGGER SQLITE_CREATE_VIEW
SQLITE_CREATE_VTABLE SQLITE_DELETE SQLITE_DETACH SQLITE_DROP_INDEX
SQLITE_DROP_TABLE SQLITE_DROP_TEMP_INDEX SQLITE_DROP_TEMP_TABLE
SQLITE_DROP_TEMP_TRIGGER SQLITE_DROP_TEMP_VIEW SQLITE_DROP_TRIGGER
SQLITE_DROP_VIEW SQLITE_DROP_VTABLE SQLITE_FUNCTION SQLITE_INSERT
SQLITE_PRAGMA SQLITE_READ SQLITE_RECURSIVE SQLITE_REINDEX
SQLITE_SAVEPOINT SQLITE_SELECT SQLITE_TRANSACTION SQLITE_UPDATE"""

mapping_authorizer_return_codes: dict[str | int, int | str]
"""Authorizer Return Codes mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_deny.html

SQLITE_DENY SQLITE_IGNORE SQLITE_OK"""

mapping_bestindex_constraints: dict[str | int, int | str]
"""Virtual Table Constraint Operator Codes mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_index_constraint_eq.html

SQLITE_INDEX_CONSTRAINT_EQ SQLITE_INDEX_CONSTRAINT_FUNCTION
SQLITE_INDEX_CONSTRAINT_GE SQLITE_INDEX_CONSTRAINT_GLOB
SQLITE_INDEX_CONSTRAINT_GT SQLITE_INDEX_CONSTRAINT_IS
SQLITE_INDEX_CONSTRAINT_ISNOT SQLITE_INDEX_CONSTRAINT_ISNOTNULL
SQLITE_INDEX_CONSTRAINT_ISNULL SQLITE_INDEX_CONSTRAINT_LE
SQLITE_INDEX_CONSTRAINT_LIKE SQLITE_INDEX_CONSTRAINT_LIMIT
SQLITE_INDEX_CONSTRAINT_LT SQLITE_INDEX_CONSTRAINT_MATCH
SQLITE_INDEX_CONSTRAINT_NE SQLITE_INDEX_CONSTRAINT_OFFSET
SQLITE_INDEX_CONSTRAINT_REGEXP"""

mapping_config: dict[str | int, int | str]
"""Configuration Options mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_config_covering_index_scan.html

SQLITE_CONFIG_COVERING_INDEX_SCAN SQLITE_CONFIG_GETMALLOC
SQLITE_CONFIG_GETMUTEX SQLITE_CONFIG_GETPCACHE
SQLITE_CONFIG_GETPCACHE2 SQLITE_CONFIG_HEAP SQLITE_CONFIG_LOG
SQLITE_CONFIG_LOOKASIDE SQLITE_CONFIG_MALLOC
SQLITE_CONFIG_MEMDB_MAXSIZE SQLITE_CONFIG_MEMSTATUS
SQLITE_CONFIG_MMAP_SIZE SQLITE_CONFIG_MULTITHREAD SQLITE_CONFIG_MUTEX
SQLITE_CONFIG_PAGECACHE SQLITE_CONFIG_PCACHE SQLITE_CONFIG_PCACHE2
SQLITE_CONFIG_PCACHE_HDRSZ SQLITE_CONFIG_PMASZ SQLITE_CONFIG_SCRATCH
SQLITE_CONFIG_SERIALIZED SQLITE_CONFIG_SINGLETHREAD
SQLITE_CONFIG_SMALL_MALLOC SQLITE_CONFIG_SORTERREF_SIZE
SQLITE_CONFIG_SQLLOG SQLITE_CONFIG_STMTJRNL_SPILL SQLITE_CONFIG_URI
SQLITE_CONFIG_WIN32_HEAPSIZE"""

mapping_conflict_resolution_modes: dict[str | int, int | str]
"""Conflict resolution modes mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_fail.html

SQLITE_ABORT SQLITE_FAIL SQLITE_IGNORE SQLITE_REPLACE SQLITE_ROLLBACK"""

mapping_db_config: dict[str | int, int | str]
"""Database Connection Configuration Options mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_dbconfig_defensive.html

SQLITE_DBCONFIG_DEFENSIVE SQLITE_DBCONFIG_DQS_DDL
SQLITE_DBCONFIG_DQS_DML SQLITE_DBCONFIG_ENABLE_ATTACH_CREATE
SQLITE_DBCONFIG_ENABLE_ATTACH_WRITE SQLITE_DBCONFIG_ENABLE_COMMENTS
SQLITE_DBCONFIG_ENABLE_FKEY SQLITE_DBCONFIG_ENABLE_FTS3_TOKENIZER
SQLITE_DBCONFIG_ENABLE_LOAD_EXTENSION SQLITE_DBCONFIG_ENABLE_QPSG
SQLITE_DBCONFIG_ENABLE_TRIGGER SQLITE_DBCONFIG_ENABLE_VIEW
SQLITE_DBCONFIG_LEGACY_ALTER_TABLE SQLITE_DBCONFIG_LEGACY_FILE_FORMAT
SQLITE_DBCONFIG_LOOKASIDE SQLITE_DBCONFIG_MAINDBNAME
SQLITE_DBCONFIG_MAX SQLITE_DBCONFIG_NO_CKPT_ON_CLOSE
SQLITE_DBCONFIG_RESET_DATABASE SQLITE_DBCONFIG_REVERSE_SCANORDER
SQLITE_DBCONFIG_STMT_SCANSTATUS SQLITE_DBCONFIG_TRIGGER_EQP
SQLITE_DBCONFIG_TRUSTED_SCHEMA SQLITE_DBCONFIG_WRITABLE_SCHEMA"""

mapping_db_status: dict[str | int, int | str]
"""Status Parameters for database connections mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_dbstatus_options.html

SQLITE_DBSTATUS_CACHE_HIT SQLITE_DBSTATUS_CACHE_MISS
SQLITE_DBSTATUS_CACHE_SPILL SQLITE_DBSTATUS_CACHE_USED
SQLITE_DBSTATUS_CACHE_USED_SHARED SQLITE_DBSTATUS_CACHE_WRITE
SQLITE_DBSTATUS_DEFERRED_FKS SQLITE_DBSTATUS_LOOKASIDE_HIT
SQLITE_DBSTATUS_LOOKASIDE_MISS_FULL
SQLITE_DBSTATUS_LOOKASIDE_MISS_SIZE SQLITE_DBSTATUS_LOOKASIDE_USED
SQLITE_DBSTATUS_MAX SQLITE_DBSTATUS_SCHEMA_USED
SQLITE_DBSTATUS_STMT_USED"""

mapping_device_characteristics: dict[str | int, int | str]
"""Device Characteristics mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_iocap_atomic.html

SQLITE_IOCAP_ATOMIC SQLITE_IOCAP_ATOMIC16K SQLITE_IOCAP_ATOMIC1K
SQLITE_IOCAP_ATOMIC2K SQLITE_IOCAP_ATOMIC32K SQLITE_IOCAP_ATOMIC4K
SQLITE_IOCAP_ATOMIC512 SQLITE_IOCAP_ATOMIC64K SQLITE_IOCAP_ATOMIC8K
SQLITE_IOCAP_BATCH_ATOMIC SQLITE_IOCAP_IMMUTABLE
SQLITE_IOCAP_POWERSAFE_OVERWRITE SQLITE_IOCAP_SAFE_APPEND
SQLITE_IOCAP_SEQUENTIAL SQLITE_IOCAP_SUBPAGE_READ
SQLITE_IOCAP_UNDELETABLE_WHEN_OPEN"""

mapping_extended_result_codes: dict[str | int, int | str]
"""Extended Result Codes mapping names to int and int to names.
Doc at https://sqlite.org/rescode.html

SQLITE_ABORT_ROLLBACK SQLITE_AUTH_USER SQLITE_BUSY_RECOVERY
SQLITE_BUSY_SNAPSHOT SQLITE_BUSY_TIMEOUT SQLITE_CANTOPEN_CONVPATH
SQLITE_CANTOPEN_DIRTYWAL SQLITE_CANTOPEN_FULLPATH
SQLITE_CANTOPEN_ISDIR SQLITE_CANTOPEN_NOTEMPDIR
SQLITE_CANTOPEN_SYMLINK SQLITE_CONSTRAINT_CHECK
SQLITE_CONSTRAINT_COMMITHOOK SQLITE_CONSTRAINT_DATATYPE
SQLITE_CONSTRAINT_FOREIGNKEY SQLITE_CONSTRAINT_FUNCTION
SQLITE_CONSTRAINT_NOTNULL SQLITE_CONSTRAINT_PINNED
SQLITE_CONSTRAINT_PRIMARYKEY SQLITE_CONSTRAINT_ROWID
SQLITE_CONSTRAINT_TRIGGER SQLITE_CONSTRAINT_UNIQUE
SQLITE_CONSTRAINT_VTAB SQLITE_CORRUPT_INDEX SQLITE_CORRUPT_SEQUENCE
SQLITE_CORRUPT_VTAB SQLITE_ERROR_MISSING_COLLSEQ SQLITE_ERROR_RETRY
SQLITE_ERROR_SNAPSHOT SQLITE_IOERR_ACCESS SQLITE_IOERR_AUTH
SQLITE_IOERR_BEGIN_ATOMIC SQLITE_IOERR_BLOCKED
SQLITE_IOERR_CHECKRESERVEDLOCK SQLITE_IOERR_CLOSE
SQLITE_IOERR_COMMIT_ATOMIC SQLITE_IOERR_CONVPATH
SQLITE_IOERR_CORRUPTFS SQLITE_IOERR_DATA SQLITE_IOERR_DELETE
SQLITE_IOERR_DELETE_NOENT SQLITE_IOERR_DIR_CLOSE
SQLITE_IOERR_DIR_FSYNC SQLITE_IOERR_FSTAT SQLITE_IOERR_FSYNC
SQLITE_IOERR_GETTEMPPATH SQLITE_IOERR_IN_PAGE SQLITE_IOERR_LOCK
SQLITE_IOERR_MMAP SQLITE_IOERR_NOMEM SQLITE_IOERR_RDLOCK
SQLITE_IOERR_READ SQLITE_IOERR_ROLLBACK_ATOMIC SQLITE_IOERR_SEEK
SQLITE_IOERR_SHMLOCK SQLITE_IOERR_SHMMAP SQLITE_IOERR_SHMOPEN
SQLITE_IOERR_SHMSIZE SQLITE_IOERR_SHORT_READ SQLITE_IOERR_TRUNCATE
SQLITE_IOERR_UNLOCK SQLITE_IOERR_VNODE SQLITE_IOERR_WRITE
SQLITE_LOCKED_SHAREDCACHE SQLITE_LOCKED_VTAB SQLITE_NOTICE_RBU
SQLITE_NOTICE_RECOVER_ROLLBACK SQLITE_NOTICE_RECOVER_WAL
SQLITE_OK_LOAD_PERMANENTLY SQLITE_OK_SYMLINK SQLITE_READONLY_CANTINIT
SQLITE_READONLY_CANTLOCK SQLITE_READONLY_DBMOVED
SQLITE_READONLY_DIRECTORY SQLITE_READONLY_RECOVERY
SQLITE_READONLY_ROLLBACK SQLITE_WARNING_AUTOINDEX"""

mapping_file_control: dict[str | int, int | str]
"""Standard File Control Opcodes mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_fcntl_begin_atomic_write.html

SQLITE_FCNTL_BEGIN_ATOMIC_WRITE SQLITE_FCNTL_BUSYHANDLER
SQLITE_FCNTL_CHUNK_SIZE SQLITE_FCNTL_CKPT_DONE SQLITE_FCNTL_CKPT_START
SQLITE_FCNTL_CKSM_FILE SQLITE_FCNTL_COMMIT_ATOMIC_WRITE
SQLITE_FCNTL_COMMIT_PHASETWO SQLITE_FCNTL_DATA_VERSION
SQLITE_FCNTL_EXTERNAL_READER SQLITE_FCNTL_FILE_POINTER
SQLITE_FCNTL_GET_LOCKPROXYFILE SQLITE_FCNTL_HAS_MOVED
SQLITE_FCNTL_JOURNAL_POINTER SQLITE_FCNTL_LAST_ERRNO
SQLITE_FCNTL_LOCKSTATE SQLITE_FCNTL_LOCK_TIMEOUT
SQLITE_FCNTL_MMAP_SIZE SQLITE_FCNTL_NULL_IO SQLITE_FCNTL_OVERWRITE
SQLITE_FCNTL_PDB SQLITE_FCNTL_PERSIST_WAL
SQLITE_FCNTL_POWERSAFE_OVERWRITE SQLITE_FCNTL_PRAGMA SQLITE_FCNTL_RBU
SQLITE_FCNTL_RESERVE_BYTES SQLITE_FCNTL_RESET_CACHE
SQLITE_FCNTL_ROLLBACK_ATOMIC_WRITE SQLITE_FCNTL_SET_LOCKPROXYFILE
SQLITE_FCNTL_SIZE_HINT SQLITE_FCNTL_SIZE_LIMIT SQLITE_FCNTL_SYNC
SQLITE_FCNTL_SYNC_OMITTED SQLITE_FCNTL_TEMPFILENAME SQLITE_FCNTL_TRACE
SQLITE_FCNTL_VFSNAME SQLITE_FCNTL_VFS_POINTER SQLITE_FCNTL_WAL_BLOCK
SQLITE_FCNTL_WIN32_AV_RETRY SQLITE_FCNTL_WIN32_GET_HANDLE
SQLITE_FCNTL_WIN32_SET_HANDLE SQLITE_FCNTL_ZIPVFS"""

mapping_fts5_token_flags: dict[str | int, int | str]
"""FTS5 Token Flag mapping names to int and int to names.
Doc at https://sqlite.org/fts5.html

FTS5_TOKEN_COLOCATED"""

mapping_fts5_tokenize_reason: dict[str | int, int | str]
"""FTS5 Tokenize Reason mapping names to int and int to names.
Doc at https://sqlite.org/fts5.html

FTS5_TOKENIZE_AUX FTS5_TOKENIZE_DOCUMENT FTS5_TOKENIZE_PREFIX
FTS5_TOKENIZE_QUERY"""

mapping_function_flags: dict[str | int, int | str]
"""Function Flags mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_deterministic.html

SQLITE_DETERMINISTIC SQLITE_DIRECTONLY SQLITE_INNOCUOUS
SQLITE_RESULT_SUBTYPE SQLITE_SELFORDER1 SQLITE_SUBTYPE"""

mapping_limits: dict[str | int, int | str]
"""Run-Time Limit Categories mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_limit_attached.html

SQLITE_LIMIT_ATTACHED SQLITE_LIMIT_COLUMN SQLITE_LIMIT_COMPOUND_SELECT
SQLITE_LIMIT_EXPR_DEPTH SQLITE_LIMIT_FUNCTION_ARG SQLITE_LIMIT_LENGTH
SQLITE_LIMIT_LIKE_PATTERN_LENGTH SQLITE_LIMIT_SQL_LENGTH
SQLITE_LIMIT_TRIGGER_DEPTH SQLITE_LIMIT_VARIABLE_NUMBER
SQLITE_LIMIT_VDBE_OP SQLITE_LIMIT_WORKER_THREADS"""

mapping_locking_level: dict[str | int, int | str]
"""File Locking Levels mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_lock_exclusive.html

SQLITE_LOCK_EXCLUSIVE SQLITE_LOCK_NONE SQLITE_LOCK_PENDING
SQLITE_LOCK_RESERVED SQLITE_LOCK_SHARED"""

mapping_open_flags: dict[str | int, int | str]
"""Flags For File Open Operations mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_open_autoproxy.html

SQLITE_OPEN_AUTOPROXY SQLITE_OPEN_CREATE SQLITE_OPEN_DELETEONCLOSE
SQLITE_OPEN_EXCLUSIVE SQLITE_OPEN_EXRESCODE SQLITE_OPEN_FULLMUTEX
SQLITE_OPEN_MAIN_DB SQLITE_OPEN_MAIN_JOURNAL SQLITE_OPEN_MEMORY
SQLITE_OPEN_NOFOLLOW SQLITE_OPEN_NOMUTEX SQLITE_OPEN_PRIVATECACHE
SQLITE_OPEN_READONLY SQLITE_OPEN_READWRITE SQLITE_OPEN_SHAREDCACHE
SQLITE_OPEN_SUBJOURNAL SQLITE_OPEN_SUPER_JOURNAL SQLITE_OPEN_TEMP_DB
SQLITE_OPEN_TEMP_JOURNAL SQLITE_OPEN_TRANSIENT_DB SQLITE_OPEN_URI
SQLITE_OPEN_WAL"""

mapping_prepare_flags: dict[str | int, int | str]
"""Prepare Flags mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_prepare_dont_log.html

SQLITE_PREPARE_DONT_LOG SQLITE_PREPARE_NORMALIZE
SQLITE_PREPARE_NO_VTAB SQLITE_PREPARE_PERSISTENT"""

mapping_result_codes: dict[str | int, int | str]
"""Result Codes mapping names to int and int to names.
Doc at https://sqlite.org/rescode.html

SQLITE_ABORT SQLITE_AUTH SQLITE_BUSY SQLITE_CANTOPEN SQLITE_CONSTRAINT
SQLITE_CORRUPT SQLITE_DONE SQLITE_EMPTY SQLITE_ERROR SQLITE_FORMAT
SQLITE_FULL SQLITE_INTERNAL SQLITE_INTERRUPT SQLITE_IOERR
SQLITE_LOCKED SQLITE_MISMATCH SQLITE_MISUSE SQLITE_NOLFS SQLITE_NOMEM
SQLITE_NOTADB SQLITE_NOTFOUND SQLITE_NOTICE SQLITE_OK SQLITE_PERM
SQLITE_PROTOCOL SQLITE_RANGE SQLITE_READONLY SQLITE_ROW SQLITE_SCHEMA
SQLITE_TOOBIG SQLITE_WARNING"""

mapping_statement_status: dict[str | int, int | str]
"""Status Parameters for prepared statements mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_stmtstatus_counter.html

SQLITE_STMTSTATUS_AUTOINDEX SQLITE_STMTSTATUS_FILTER_HIT
SQLITE_STMTSTATUS_FILTER_MISS SQLITE_STMTSTATUS_FULLSCAN_STEP
SQLITE_STMTSTATUS_MEMUSED SQLITE_STMTSTATUS_REPREPARE
SQLITE_STMTSTATUS_RUN SQLITE_STMTSTATUS_SORT SQLITE_STMTSTATUS_VM_STEP"""

mapping_status: dict[str | int, int | str]
"""Status Parameters mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_status_malloc_count.html

SQLITE_STATUS_MALLOC_COUNT SQLITE_STATUS_MALLOC_SIZE
SQLITE_STATUS_MEMORY_USED SQLITE_STATUS_PAGECACHE_OVERFLOW
SQLITE_STATUS_PAGECACHE_SIZE SQLITE_STATUS_PAGECACHE_USED
SQLITE_STATUS_PARSER_STACK SQLITE_STATUS_SCRATCH_OVERFLOW
SQLITE_STATUS_SCRATCH_SIZE SQLITE_STATUS_SCRATCH_USED"""

mapping_sync: dict[str | int, int | str]
"""Synchronization Type Flags mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_sync_dataonly.html

SQLITE_SYNC_DATAONLY SQLITE_SYNC_FULL SQLITE_SYNC_NORMAL"""

mapping_trace_codes: dict[str | int, int | str]
"""SQL Trace Event Codes mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_trace.html

SQLITE_TRACE_CLOSE SQLITE_TRACE_PROFILE SQLITE_TRACE_ROW
SQLITE_TRACE_STMT"""

mapping_txn_state: dict[str | int, int | str]
"""Allowed return values from sqlite3_txn_state() mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_txn_none.html

SQLITE_TXN_NONE SQLITE_TXN_READ SQLITE_TXN_WRITE"""

mapping_virtual_table_configuration_options: dict[str | int, int | str]
"""Virtual Table Configuration Options mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_vtab_constraint_support.html

SQLITE_VTAB_CONSTRAINT_SUPPORT SQLITE_VTAB_DIRECTONLY
SQLITE_VTAB_INNOCUOUS SQLITE_VTAB_USES_ALL_SCHEMAS"""

mapping_virtual_table_scan_flags: dict[str | int, int | str]
"""Virtual Table Scan Flags mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_index_scan_hex.html

SQLITE_INDEX_SCAN_HEX SQLITE_INDEX_SCAN_UNIQUE"""

mapping_wal_checkpoint: dict[str | int, int | str]
"""Checkpoint Mode Values mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_checkpoint_full.html

SQLITE_CHECKPOINT_FULL SQLITE_CHECKPOINT_PASSIVE
SQLITE_CHECKPOINT_RESTART SQLITE_CHECKPOINT_TRUNCATE"""

mapping_xshmlock_flags: dict[str | int, int | str]
"""Flags for the xShmLock VFS method mapping names to int and int to names.
Doc at https://sqlite.org/c3ref/c_shm_exclusive.html

SQLITE_SHM_EXCLUSIVE SQLITE_SHM_LOCK SQLITE_SHM_SHARED
SQLITE_SHM_UNLOCK"""



class Error(Exception):
    """  This is the base for APSW exceptions.

    .. attribute:: Error.result

             For exceptions corresponding to `SQLite error codes
             <https://sqlite.org/c3ref/c_abort.html>`_ codes this attribute
             is the numeric error code.

    .. attribute:: Error.extendedresult

             APSW runs with `extended result codes
             <https://sqlite.org/rescode.html>`_ turned on.
             This attribute includes the detailed code.

             As an example, if SQLite issued a read request and the system
             returned less data than expected then :attr:`~Error.result`
             would have the value *SQLITE_IOERR* while
             :attr:`~Error.extendedresult` would have the value
             *SQLITE_IOERR_SHORT_READ*.

    .. attribute:: Error.error_offset

            The location of the error in the SQL when encoded in UTF-8.
            The value is from `sqlite3_error_offset
            <https://www.sqlite.org/c3ref/errcode.html>`__, and will be
            `-1` when a specific token in the input is not the cause."""

class AbortError(Error):
    """`SQLITE_ABORT <https://sqlite.org/rescode.html#abort>`__. Callback
    routine requested an abort."""

class AuthError(Error):
    """`SQLITE_AUTH <https://sqlite.org/rescode.html#auth>`__.
    :attr:`Authorization <Connection.authorizer>` denied."""

class BindingsError(Error):
    """There are several causes for this exception.  When using tuples, an incorrect number of bindings where supplied::

       cursor.execute("select ?,?,?", (1,2))     # too few bindings
       cursor.execute("select ?,?,?", (1,2,3,4)) # too many bindings

    You are using named bindings, but not all bindings are named.  You should either use entirely the
    named style or entirely numeric (unnamed) style::

       cursor.execute("select * from foo where x=:name and y=?")"""

class BusyError(Error):
    """`SQLITE_BUSY <https://sqlite.org/rescode.html#busy>`__.  The
    database file is locked.  Use  :meth:`Connection.set_busy_timeout`
    to change how long SQLite waits for the database to be unlocked or
    :meth:`Connection.set_busy_handler` to use your own handler."""

class CantOpenError(Error):
    """`SQLITE_CANTOPEN <https://sqlite.org/rescode.html#cantopen>`__.
    Unable to open the database file."""

class ConnectionClosedError(Error):
    """You have called :meth:`Connection.close` and then continued to use
    the :class:`Connection` or associated :class:`cursors <Cursor>`."""

class ConnectionNotClosedError(Error):
    """This exception is no longer generated.  It was required in earlier
    releases due to constraints in threading usage with SQLite."""

class ConstraintError(Error):
    """`SQLITE_CONSTRAINT <https://sqlite.org/rescode.html#constraint>`__.
    Abort due to `constraint
    <https://sqlite.org/lang_createtable.html>`_ violation."""

class CorruptError(Error):
    """`SQLITE_CORRUPT <https://sqlite.org/rescode.html#corrupt>`__.  The
    database disk image appears to be a SQLite database but the values
    inside are inconsistent."""

class CursorClosedError(Error):
    """You have called :meth:`Cursor.close` and then tried to use the cursor."""

class EmptyError(Error):
    """`SQLITE_EMPTY <https://sqlite.org/rescode.html#empty>`__. Not
    currently used."""

class ExecTraceAbort(Error):
    """The :ref:`execution tracer <executiontracer>` returned False so
    execution was aborted."""

class ExecutionCompleteError(Error):
    """Execution of the statements is complete and cannot be run further."""

class ExtensionLoadingError(Error):
    """An error happened loading an `extension
    <https://sqlite.org/loadext.html>`_."""

class ForkingViolationError(Error):
    """See :meth:`apsw.fork_checker`."""

class FormatError(Error):
    """`SQLITE_FORMAT <https://sqlite.org/rescode.html#format>`__. (No
    longer used) `Auxiliary database
    <https://sqlite.org/lang_attach.html>`_ format error."""

class FullError(Error):
    """`SQLITE_FULL <https://sqlite.org/rescode.html#full>`__.  The disk
    appears to be full."""

class IOError(Error):
    """`SQLITE_IOERR <https://sqlite.org/rescode.html#ioerr>`__.  A disk
    I/O error occurred.  The :ref:`extended error code <exceptions>`
    will give more detail."""

class IncompleteExecutionError(Error):
    """You have tried to start a new SQL execute call before executing all
    the previous ones. See the :ref:`execution model <executionmodel>`
    for more details."""

class InternalError(Error):
    """`SQLITE_INTERNAL <https://sqlite.org/rescode.html#internal>`__. (No
    longer used) Internal logic error in SQLite."""

class InterruptError(Error):
    """`SQLITE_INTERRUPT <https://sqlite.org/rescode.html#interrupt>`__.
    Operation terminated by `sqlite3_interrupt
    <https://sqlite.org/c3ref/interrupt.html>`_ - use
    :meth:`Connection.interrupt`."""

class InvalidContextError(Error):
    """Context is no longer valid.  Examples include using an
    :class:`IndexInfo` outside of the :meth:`VTTable.BestIndexObject`
    method, a registered :class:`FTS5Tokenizer` when the underlying
    tokenizer has been deleted/replaced, or :meth:`Connection.vtab_config`
    when not inside :meth:`VTModule.Create`."""

class LockedError(Error):
    """`SQLITE_LOCKED <https://sqlite.org/rescode.html#locked>`__.  Shared
    cache lock."""

class MismatchError(Error):
    """`SQLITE_MISMATCH <https://sqlite.org/rescode.html#mismatch>`__. Data
    type mismatch.  For example a rowid or integer primary key must be
    an integer."""

class MisuseError(Error):
    """`SQLITE_MISUSE <https://sqlite.org/rescode.html#misuse>`__.  SQLite
    library used incorrectly - typically similar to *ValueError* in
    Python.  Examples include not having enough flags when opening a
    connection (eg not including a READ or WRITE flag), or out of spec
    such as registering a function with more than 127 parameters."""

class NoFTS5Error(Error):
    """The FTS5 extension is not present in SQLite."""

class NoLFSError(Error):
    """`SQLITE_NOLFS <https://sqlite.org/rescode.html#nolfs>`__.  SQLite
    has attempted to use a feature not supported by the operating system
    such as `large file support
    <https://en.wikipedia.org/wiki/Large_file_support>`_."""

class NoMemError(Error):
    """`SQLITE_NOMEM <https://sqlite.org/rescode.html#nomem>`__.  A memory
     allocation failed."""

class NotADBError(Error):
    """`SQLITE_NOTADB <https://sqlite.org/rescode.html#notadb>`__.  File
    opened that is not a database file.  SQLite has a header on database
    files to verify they are indeed SQLite databases."""

class NotFoundError(Error):
    """`SQLITE_NOTFOUND <https://sqlite.org/rescode.html#notfound>`__.
    Returned when various internal items were not found such as requests
    for non-existent system calls or file controls."""

class PermissionsError(Error):
    """`SQLITE_PERM <https://sqlite.org/rescode.html#perm>`__. Access
    permission denied by the operating system."""

class ProtocolError(Error):
    """`SQLITE_PROTOCOL <https://sqlite.org/rescode.html#protocol>`__. (No
    longer used) Database lock protocol error."""

class RangeError(Error):
    """`SQLITE_RANGE <https://sqlite.org/rescode.html#range>`__.  (Cannot
    be generated using APSW).  2nd parameter to `sqlite3_bind
    <https://sqlite.org/c3ref/bind_blob.html>`_ out of range"""

class ReadOnlyError(Error):
    """`SQLITE_READONLY <https://sqlite.org/rescode.html#readonly>`__.
    Attempt to write to a readonly database."""

class SQLError(Error):
    """`SQLITE_ERROR <https://sqlite.org/rescode.html#error>`__.  The
    standard error code, unless a more specific one is  applicable."""

class SchemaChangeError(Error):
    """`SQLITE_SCHEMA <https://sqlite.org/rescode.html#schema>`__.  The
    database schema changed.  A  :meth:`prepared statement
    <Cursor.execute>` becomes invalid if the database schema was
    changed.  Behind the scenes SQLite reprepares the statement.
    Another or the same :class:`Connection` may change the schema again
    before the statement runs.  SQLite will retry before giving up and
    returning this error."""

class ThreadingViolationError(Error):
    """You have used an object concurrently in two threads. For example you
    may try to use the same cursor in two different threads at the same
    time, or tried to close the same connection in two threads at the
    same time.

    You can also get this exception by using a cursor while it is
    already inside execution such as executing a new query while inside
    a function called by that cursor.  It is recommended to use
    :meth:`Connection.execute` and :meth:`Connection.executemany`
    instead of using cursors directly."""

class TooBigError(Error):
    """`SQLITE_TOOBIG <https://sqlite.org/rescode.html#toobig>`__.  String
    or BLOB exceeds size limit.  You can  change the limits using
    :meth:`Connection.limit`."""

class VFSFileClosedError(Error):
    """The VFS file is closed so the operation cannot be performed."""

class VFSNotImplementedError(Error):
    """A call cannot be made to an inherited :ref:`VFS` method as the VFS
    does not implement the method."""

