/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.bin.format.elf;

import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.StructConverter;
import ghidra.app.util.bin.UnlimitedByteProviderWrapper;
import ghidra.app.util.bin.format.elf.ElfConstants;
import ghidra.app.util.bin.format.elf.ElfDynamic;
import ghidra.app.util.bin.format.elf.ElfDynamicTable;
import ghidra.app.util.bin.format.elf.ElfDynamicType;
import ghidra.app.util.bin.format.elf.ElfException;
import ghidra.app.util.bin.format.elf.ElfProgramHeader;
import ghidra.app.util.bin.format.elf.ElfProgramHeaderType;
import ghidra.app.util.bin.format.elf.ElfRelocationTable;
import ghidra.app.util.bin.format.elf.ElfSectionHeader;
import ghidra.app.util.bin.format.elf.ElfSectionHeaderType;
import ghidra.app.util.bin.format.elf.ElfStringTable;
import ghidra.app.util.bin.format.elf.ElfSymbolTable;
import ghidra.app.util.bin.format.elf.extend.ElfExtensionFactory;
import ghidra.app.util.bin.format.elf.extend.ElfLoadAdapter;
import ghidra.program.model.data.ArrayDataType;
import ghidra.program.model.data.CategoryPath;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.Structure;
import ghidra.program.model.data.StructureDataType;
import ghidra.util.Msg;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.NotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.function.Consumer;

public class ElfHeader
implements StructConverter {
    private static final int MAX_HEADERS_TO_CHECK_FOR_IMAGEBASE = 20;
    private static final int PAD_LENGTH = 7;
    private HashMap<Integer, ElfProgramHeaderType> programHeaderTypeMap;
    private HashMap<Integer, ElfSectionHeaderType> sectionHeaderTypeMap;
    private HashMap<Integer, ElfDynamicType> dynamicTypeMap;
    private final ByteProvider provider;
    private final BinaryReader reader;
    private final byte e_ident_magic_num;
    private final String e_ident_magic_str;
    private final byte e_ident_class;
    private final byte e_ident_data;
    private final byte e_ident_version;
    private final byte e_ident_osabi;
    private final byte e_ident_abiversion;
    private final byte[] e_ident_pad;
    private final short e_type;
    private final short e_machine;
    private final int e_version;
    private final long e_entry;
    private final long e_phoff;
    private final long e_shoff;
    private final int e_flags;
    private final short e_ehsize;
    private final short e_phentsize;
    private final int e_phnum;
    private final short e_shentsize;
    private final int e_shnum;
    private final int e_shstrndx;
    private Structure headerStructure;
    private boolean parsed = false;
    private boolean parsedSectionHeaders = false;
    private ElfLoadAdapter elfLoadAdapter = new ElfLoadAdapter();
    private ElfSectionHeader section0 = null;
    private ElfSectionHeader[] sectionHeaders = new ElfSectionHeader[0];
    private ElfProgramHeader[] programHeaders = new ElfProgramHeader[0];
    private ElfStringTable[] stringTables = new ElfStringTable[0];
    private ElfSymbolTable[] symbolTables = new ElfSymbolTable[0];
    private ElfRelocationTable[] relocationTables = new ElfRelocationTable[0];
    private ElfDynamicTable dynamicTable;
    private ElfStringTable dynamicStringTable;
    private ElfSymbolTable dynamicSymbolTable;
    private boolean hasExtendedSymbolSectionIndexTable;
    private String[] dynamicLibraryNames;
    private boolean hasLittleEndianHeaders;
    private Consumer<String> errorConsumer;
    private static int INITIAL_READ_LEN = 34;
    private Long elfImageBase;
    private Long preLinkImageBase = null;

    public ElfHeader(ByteProvider provider, Consumer<String> errorConsumer) throws ElfException {
        this.provider = provider;
        this.errorConsumer = errorConsumer != null ? errorConsumer : msg -> {};
        try {
            if (!Arrays.equals(ElfConstants.MAGIC_BYTES, provider.readBytes(0L, ElfConstants.MAGIC_BYTES.length))) {
                throw new ElfException("Not a valid ELF executable.");
            }
            this.e_ident_magic_num = (byte)127;
            this.e_ident_magic_str = "ELF";
            this.determineHeaderEndianness();
            this.reader = new BinaryReader(new UnlimitedByteProviderWrapper(provider), this.hasLittleEndianHeaders);
            this.reader.setPointerIndex(ElfConstants.MAGIC_BYTES.length);
            this.e_ident_class = this.reader.readNextByte();
            this.e_ident_data = this.reader.readNextByte();
            this.e_ident_version = this.reader.readNextByte();
            this.e_ident_osabi = this.reader.readNextByte();
            this.e_ident_abiversion = this.reader.readNextByte();
            this.e_ident_pad = this.reader.readNextByteArray(7);
            this.e_type = this.reader.readNextShort();
            this.e_machine = this.reader.readNextShort();
            this.e_version = this.reader.readNextInt();
            if (this.is32Bit()) {
                this.e_entry = this.reader.readNextUnsignedInt();
                this.e_phoff = this.reader.readNextUnsignedInt();
                this.e_shoff = this.reader.readNextUnsignedInt();
            } else if (this.is64Bit()) {
                this.e_entry = this.reader.readNextLong();
                this.e_phoff = this.reader.readNextLong();
                this.e_shoff = this.reader.readNextLong();
            } else {
                throw new ElfException("Only 32-bit and 64-bit ELF headers are supported (EI_CLASS=0x" + Integer.toHexString(this.e_ident_class) + ")");
            }
            this.e_flags = this.reader.readNextInt();
            this.e_ehsize = this.reader.readNextShort();
            this.e_phentsize = this.reader.readNextShort();
            int phnum = this.reader.readNextUnsignedShort();
            if (phnum == Short.toUnsignedInt((short)-1)) {
                phnum = this.readExtendedProgramHeaderCount();
            }
            this.e_phnum = phnum;
            this.e_shentsize = this.reader.readNextShort();
            int shnum = this.reader.readNextUnsignedShort();
            if (shnum == 0 || shnum >= Short.toUnsignedInt((short)-256)) {
                shnum = this.readExtendedSectionHeaderCount();
            }
            this.e_shnum = shnum;
            int shstrndx = this.reader.readNextUnsignedShort();
            if (this.e_shnum == 0) {
                shstrndx = 0;
            } else if (shstrndx == Short.toUnsignedInt((short)-1)) {
                shstrndx = this.readExtendedSectionHeaderStringTableIndex();
            }
            this.e_shstrndx = shstrndx;
        }
        catch (IOException e) {
            throw new ElfException(e);
        }
    }

    public BinaryReader getReader() {
        return this.reader;
    }

    public ByteProvider getByteProvider() {
        return this.provider;
    }

    void logError(String msg) {
        this.errorConsumer.accept(msg);
    }

    private ElfSectionHeader getSection0() throws IOException {
        if (this.section0 == null && this.e_shoff != 0L) {
            if (!this.providerContainsRegion(this.e_shoff, this.e_shentsize)) {
                return null;
            }
            this.section0 = new ElfSectionHeader(this.reader.clone(this.e_shoff), this);
        }
        return this.section0;
    }

    private int readExtendedProgramHeaderCount() throws IOException {
        ElfSectionHeader s = this.getSection0();
        if (s != null && s.getType() == 0) {
            int val = s.getInfo();
            return val < 0 ? 0 : val;
        }
        return 0;
    }

    private int readExtendedSectionHeaderCount() throws IOException {
        ElfSectionHeader s = this.getSection0();
        if (s != null && s.getType() == 0) {
            long val = s.getSize();
            return val < 0L || val > Integer.MAX_VALUE ? 0 : (int)val;
        }
        return 0;
    }

    private int readExtendedSectionHeaderStringTableIndex() throws IOException {
        ElfSectionHeader s = this.getSection0();
        if (s != null && s.getType() == 0) {
            int val = s.getLink();
            return val < 0 ? 0 : val;
        }
        return 0;
    }

    private void initElfLoadAdapter() {
        this.programHeaderTypeMap = new HashMap();
        ElfProgramHeaderType.addDefaultTypes(this.programHeaderTypeMap);
        this.sectionHeaderTypeMap = new HashMap();
        ElfSectionHeaderType.addDefaultTypes(this.sectionHeaderTypeMap);
        this.dynamicTypeMap = new HashMap();
        ElfDynamicType.addDefaultTypes(this.dynamicTypeMap);
        ElfLoadAdapter extensionAdapter = ElfExtensionFactory.getLoadAdapter(this);
        if (extensionAdapter != null) {
            extensionAdapter.addProgramHeaderTypes(this.programHeaderTypeMap);
            extensionAdapter.addSectionHeaderTypes(this.sectionHeaderTypeMap);
            extensionAdapter.addDynamicTypes(this.dynamicTypeMap);
            this.elfLoadAdapter = extensionAdapter;
        }
    }

    public void parse() throws IOException {
        if (this.reader == null) {
            throw new IOException("ELF binary reader is null!");
        }
        if (this.parsed) {
            return;
        }
        this.initElfLoadAdapter();
        this.parsed = true;
        this.parseProgramHeaders();
        this.parseSectionHeaders();
        this.parseDynamicTable();
        this.parseStringTables();
        this.parseDynamicLibraryNames();
        this.parseSymbolTables();
        this.parseRelocationTables();
        this.parseGNU_d();
        this.parseGNU_r();
    }

    public ElfLoadAdapter getLoadAdapter() {
        return this.elfLoadAdapter;
    }

    public long adjustAddressForPrelink(long address) {
        long base = this.getPreLinkImageBase();
        if (base == -1L) {
            base = 0L;
        }
        return base + address;
    }

    public long unadjustAddressForPrelink(long address) {
        long base = this.getPreLinkImageBase();
        if (base == -1L) {
            base = 0L;
        }
        return address - base;
    }

    protected HashMap<Integer, ElfProgramHeaderType> getProgramHeaderTypeMap() {
        return this.programHeaderTypeMap;
    }

    protected HashMap<Integer, ElfSectionHeaderType> getSectionHeaderTypeMap() {
        return this.sectionHeaderTypeMap;
    }

    public ElfProgramHeaderType getProgramHeaderType(int type) {
        if (this.programHeaderTypeMap != null) {
            return this.programHeaderTypeMap.get(type);
        }
        return null;
    }

    public ElfSectionHeaderType getSectionHeaderType(int type) {
        if (this.sectionHeaderTypeMap != null) {
            return this.sectionHeaderTypeMap.get(type);
        }
        return null;
    }

    protected HashMap<Integer, ElfDynamicType> getDynamicTypeMap() {
        return this.dynamicTypeMap;
    }

    public ElfDynamicType getDynamicType(int type) {
        if (this.dynamicTypeMap != null) {
            return this.dynamicTypeMap.get(type);
        }
        return null;
    }

    String getTypeSuffix() {
        if (this.elfLoadAdapter == null) {
            return null;
        }
        String typeSuffix = this.elfLoadAdapter.getDataTypeSuffix();
        if (typeSuffix != null && typeSuffix.length() == 0) {
            typeSuffix = null;
        }
        return typeSuffix;
    }

    private void parseGNU_d() {
        ElfSectionHeader[] sections = this.getSections(0x6FFFFFFD);
        if (sections.length == 0) {
            return;
        }
    }

    private void parseGNU_r() {
        ElfSectionHeader[] sections = this.getSections(0x6FFFFFFE);
        if (sections.length == 0) {
            return;
        }
    }

    private void parseRelocationTables() throws IOException {
        ArrayList<ElfRelocationTable> relocationTableList = new ArrayList<ElfRelocationTable>();
        this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_REL, ElfDynamicType.DT_RELENT, ElfDynamicType.DT_RELSZ, false);
        this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_RELA, ElfDynamicType.DT_RELAENT, ElfDynamicType.DT_RELASZ, true);
        if (this.dynamicTable != null && this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_PLTREL)) {
            try {
                boolean isRela = this.dynamicTable.getDynamicValue(ElfDynamicType.DT_PLTREL) == (long)ElfDynamicType.DT_RELA.value;
                this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_JMPREL, null, ElfDynamicType.DT_PLTRELSZ, isRela);
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
        }
        this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_ANDROID_REL, null, ElfDynamicType.DT_ANDROID_RELSZ, false);
        this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_ANDROID_RELA, null, ElfDynamicType.DT_ANDROID_RELASZ, true);
        this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_RELR, ElfDynamicType.DT_RELRENT, ElfDynamicType.DT_RELRSZ, false);
        this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_ANDROID_RELR, ElfDynamicType.DT_ANDROID_RELRENT, ElfDynamicType.DT_ANDROID_RELRSZ, false);
        this.parseJMPRelocTable(relocationTableList);
        for (ElfSectionHeader section : this.sectionHeaders) {
            this.parseSectionBasedRelocationTable(section, relocationTableList);
        }
        this.relocationTables = new ElfRelocationTable[relocationTableList.size()];
        relocationTableList.toArray(this.relocationTables);
    }

    private void parseSectionBasedRelocationTable(ElfSectionHeader section, ArrayList<ElfRelocationTable> relocationTableList) throws IOException {
        try {
            int sectionHeaderType = section.getType();
            if (sectionHeaderType == 9 || sectionHeaderType == 4 || sectionHeaderType == 19 || sectionHeaderType == 0x60000001 || sectionHeaderType == 0x60000002 || sectionHeaderType == 0x6FFFFF00) {
                for (ElfRelocationTable relocTable : relocationTableList) {
                    if (relocTable.getFileOffset() != section.getOffset()) continue;
                    return;
                }
                if (section.isInvalidOffset()) {
                    Msg.debug((Object)this, (Object)("Skipping Elf relocation table section with invalid offset " + section.getNameAsString()));
                    return;
                }
                int link = section.getLink();
                int info = section.getInfo();
                ElfSectionHeader sectionToBeRelocated = info != 0 ? this.getLinkedSection(info, new int[0]) : null;
                String relocaBaseName = sectionToBeRelocated != null ? sectionToBeRelocated.getNameAsString() : "PT_LOAD";
                ElfSectionHeader symbolTableSection = link == 0 ? this.getSection(".dynsym") : this.getLinkedSection(link, 11, 2);
                ElfSymbolTable symbolTable = this.getSymbolTable(symbolTableSection);
                boolean addendTypeReloc = sectionHeaderType == 4 || sectionHeaderType == 0x60000002;
                String details = "Elf relocation table section " + section.getNameAsString();
                if (symbolTableSection != null) {
                    details = details + " linked to symbol table section " + symbolTableSection.getNameAsString();
                }
                details = details + " affecting " + relocaBaseName;
                Msg.debug((Object)this, (Object)details);
                ElfRelocationTable.TableFormat format = ElfRelocationTable.TableFormat.DEFAULT;
                if (sectionHeaderType == 0x60000001 || sectionHeaderType == 0x60000002) {
                    format = ElfRelocationTable.TableFormat.ANDROID;
                } else if (sectionHeaderType == 19 || sectionHeaderType == 0x6FFFFF00) {
                    format = ElfRelocationTable.TableFormat.RELR;
                }
                ElfRelocationTable relocTable = new ElfRelocationTable(this.reader, this, section, section.getOffset(), section.getAddress(), section.getSize(), section.getEntrySize(), addendTypeReloc, symbolTable, sectionToBeRelocated, format);
                relocationTableList.add(relocTable);
            }
        }
        catch (NotFoundException e) {
            this.errorConsumer.accept("Failed to process relocation section " + section.getNameAsString() + ": " + e.getMessage());
        }
    }

    private void parseJMPRelocTable(ArrayList<ElfRelocationTable> relocationTableList) throws IOException {
        boolean addendTypeReloc;
        if (this.dynamicTable == null) {
            return;
        }
        try {
            long tableType = this.dynamicTable.getDynamicValue(ElfDynamicType.DT_PLTREL);
            addendTypeReloc = tableType == (long)ElfDynamicType.DT_RELA.value;
        }
        catch (NotFoundException e) {
            return;
        }
        this.parseDynamicRelocTable(relocationTableList, ElfDynamicType.DT_JMPREL, addendTypeReloc ? ElfDynamicType.DT_RELAENT : ElfDynamicType.DT_RELENT, ElfDynamicType.DT_PLTRELSZ, addendTypeReloc);
    }

    private void parseDynamicRelocTable(ArrayList<ElfRelocationTable> relocationTableList, ElfDynamicType relocTableAddrType, ElfDynamicType relocEntrySizeType, ElfDynamicType relocTableSizeType, boolean addendTypeReloc) throws IOException {
        if (this.dynamicTable == null) {
            return;
        }
        try {
            long relocTableAddr = this.adjustAddressForPrelink(this.dynamicTable.getDynamicValue(relocTableAddrType));
            ElfSectionHeader relocTableSectionHeader = this.getSectionLoadHeaderContaining(relocTableAddr);
            if (relocTableSectionHeader != null) {
                this.parseSectionBasedRelocationTable(relocTableSectionHeader, relocationTableList);
                return;
            }
            ElfProgramHeader relocTableLoadHeader = this.getProgramLoadHeaderContaining(relocTableAddr);
            if (relocTableLoadHeader == null) {
                this.errorConsumer.accept("Failed to locate " + relocTableAddrType.name + " in memory at 0x" + Long.toHexString(relocTableAddr));
                return;
            }
            if (relocTableLoadHeader.isInvalidOffset()) {
                return;
            }
            long relocTableOffset = relocTableLoadHeader.getOffset(relocTableAddr);
            for (ElfRelocationTable relocTable : relocationTableList) {
                if (relocTable.getFileOffset() != relocTableOffset) continue;
                return;
            }
            long tableEntrySize = relocEntrySizeType != null ? this.dynamicTable.getDynamicValue(relocEntrySizeType) : -1L;
            long tableSize = this.dynamicTable.getDynamicValue(relocTableSizeType);
            ElfRelocationTable.TableFormat format = ElfRelocationTable.TableFormat.DEFAULT;
            if (relocTableAddrType == ElfDynamicType.DT_ANDROID_REL || relocTableAddrType == ElfDynamicType.DT_ANDROID_RELA) {
                format = ElfRelocationTable.TableFormat.ANDROID;
            } else if (relocTableAddrType == ElfDynamicType.DT_RELR || relocTableAddrType == ElfDynamicType.DT_ANDROID_RELR) {
                format = ElfRelocationTable.TableFormat.RELR;
            }
            ElfRelocationTable relocTable = new ElfRelocationTable(this.reader, this, null, relocTableOffset, relocTableAddr, tableSize, tableEntrySize, addendTypeReloc, this.dynamicSymbolTable, null, format);
            relocationTableList.add(relocTable);
        }
        catch (NotFoundException notFoundException) {
            // empty catch block
        }
    }

    private ElfSectionHeader getLinkedSection(int sectionIndex, int ... expectedTypes) throws NotFoundException {
        if (sectionIndex < 0 || sectionIndex >= this.sectionHeaders.length) {
            throw new NotFoundException("invalid linked section index " + sectionIndex);
        }
        ElfSectionHeader section = this.sectionHeaders[sectionIndex];
        if (expectedTypes.length == 0) {
            return section;
        }
        for (int type : expectedTypes) {
            if (type != section.getType()) continue;
            return section;
        }
        throw new NotFoundException("unexpected section type for section index " + sectionIndex);
    }

    private void parseDynamicLibraryNames() {
        if (this.dynamicTable == null) {
            this.dynamicLibraryNames = new String[0];
            return;
        }
        ElfDynamic[] needed = this.dynamicTable.getDynamics(ElfDynamicType.DT_NEEDED);
        this.dynamicLibraryNames = new String[needed.length];
        for (int i = 0; i < needed.length; ++i) {
            if (this.dynamicStringTable != null) {
                try {
                    this.dynamicLibraryNames[i] = this.dynamicStringTable.readString(this.reader, needed[i].getValue());
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            if (this.dynamicLibraryNames[i] != null) continue;
            this.dynamicLibraryNames[i] = "UNK_LIB_NAME_" + i;
        }
    }

    private void parseDynamicTable() throws IOException {
        ElfProgramHeader loadHeader;
        ElfSectionHeader[] dynamicSections;
        ElfProgramHeader[] dynamicHeaders = this.getProgramHeaders(2);
        if (dynamicHeaders.length == 1) {
            long vaddr = dynamicHeaders[0].getVirtualAddress();
            if (vaddr == 0L || dynamicHeaders[0].getFileSize() == 0L) {
                Msg.warn((Object)this, (Object)"ELF Dynamic table appears to have been stripped from binary");
                return;
            }
            ElfProgramHeader loadHeader2 = this.getProgramLoadHeaderContaining(vaddr);
            if (loadHeader2 == null) {
                loadHeader2 = dynamicHeaders[0];
            }
            long dynamicTableOffset = loadHeader2.getOffset() + (dynamicHeaders[0].getVirtualAddress() - loadHeader2.getVirtualAddress());
            this.dynamicTable = new ElfDynamicTable(this.reader, this, dynamicTableOffset, dynamicHeaders[0].getVirtualAddress());
            return;
        }
        if (dynamicHeaders.length > 1) {
            this.errorConsumer.accept("Multiple ELF Dynamic table program headers found");
        }
        if ((dynamicSections = this.getSections(6)).length == 1 && (loadHeader = this.getProgramLoadHeaderContaining(dynamicSections[0].getAddress())) != null) {
            long dynamicTableOffset = loadHeader.getOffset() + (dynamicSections[0].getAddress() - loadHeader.getVirtualAddress());
            this.dynamicTable = new ElfDynamicTable(this.reader, this, dynamicTableOffset, dynamicSections[0].getAddress());
            return;
        }
    }

    private void parseStringTables() {
        long dynamicStringTableAddr = -1L;
        if (this.dynamicTable != null) {
            try {
                dynamicStringTableAddr = this.adjustAddressForPrelink(this.dynamicTable.getDynamicValue(ElfDynamicType.DT_STRTAB));
            }
            catch (NotFoundException e) {
                this.errorConsumer.accept("ELF does not contain a dynamic string table (DT_STRTAB)");
            }
        }
        ArrayList<ElfStringTable> stringTableList = new ArrayList<ElfStringTable>();
        for (ElfSectionHeader stringTableSectionHeader : this.sectionHeaders) {
            if (stringTableSectionHeader.getType() != 3) continue;
            ElfStringTable stringTable = new ElfStringTable(this, stringTableSectionHeader, stringTableSectionHeader.getOffset(), stringTableSectionHeader.getAddress(), stringTableSectionHeader.getSize());
            stringTableList.add(stringTable);
            if (stringTable.getAddressOffset() != dynamicStringTableAddr) continue;
            this.dynamicStringTable = stringTable;
        }
        if (this.dynamicStringTable == null && dynamicStringTableAddr != -1L) {
            this.dynamicStringTable = this.parseDynamicStringTable(dynamicStringTableAddr);
            if (this.dynamicStringTable != null) {
                stringTableList.add(this.dynamicStringTable);
            }
        }
        this.stringTables = new ElfStringTable[stringTableList.size()];
        stringTableList.toArray(this.stringTables);
    }

    private ElfStringTable parseDynamicStringTable(long dynamicStringTableAddr) {
        if (!this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_STRSZ)) {
            this.errorConsumer.accept("Failed to parse DT_STRTAB, missing dynamic dependency");
            return null;
        }
        try {
            long stringTableSize = this.dynamicTable.getDynamicValue(ElfDynamicType.DT_STRSZ);
            if (dynamicStringTableAddr == 0L) {
                this.errorConsumer.accept("ELF Dynamic String Table of size " + stringTableSize + " appears to have been stripped from binary");
                return null;
            }
            ElfProgramHeader stringTableLoadHeader = this.getProgramLoadHeaderContaining(dynamicStringTableAddr);
            if (stringTableLoadHeader == null) {
                this.errorConsumer.accept("Failed to locate DT_STRTAB in memory at 0x" + Long.toHexString(dynamicStringTableAddr));
                return null;
            }
            return new ElfStringTable(this, null, stringTableLoadHeader.getOffset(dynamicStringTableAddr), dynamicStringTableAddr, stringTableSize);
        }
        catch (NotFoundException e) {
            throw new AssertException((Throwable)e);
        }
    }

    private int[] getExtendedSymbolSectionIndexTable(ElfSectionHeader symbolTableSectionHeader) {
        if (!this.hasExtendedSymbolSectionIndexTable) {
            return null;
        }
        ElfSectionHeader symbolSectionIndexHeader = null;
        for (ElfSectionHeader section : this.sectionHeaders) {
            int linkIndex;
            if (section.getType() != 18 || (linkIndex = section.getLink()) <= 0 || linkIndex >= this.sectionHeaders.length || this.sectionHeaders[linkIndex] != symbolTableSectionHeader) continue;
            symbolSectionIndexHeader = section;
            break;
        }
        if (symbolSectionIndexHeader == null) {
            return null;
        }
        int count = (int)(symbolSectionIndexHeader.getSize() / 4L);
        int[] indexTable = new int[count];
        try {
            BinaryReader r = this.reader.clone(symbolSectionIndexHeader.getOffset());
            for (int i = 0; i < count; ++i) {
                indexTable[i] = r.readNextInt();
            }
        }
        catch (IOException e) {
            this.errorConsumer.accept("Failed to read symbol section index table at 0x" + Long.toHexString(symbolSectionIndexHeader.getOffset()) + ": " + symbolSectionIndexHeader.getNameAsString());
        }
        return indexTable;
    }

    private void parseSymbolTables() throws IOException {
        long dynamicSymbolTableAddr = -1L;
        if (this.dynamicTable != null) {
            try {
                dynamicSymbolTableAddr = this.adjustAddressForPrelink(this.dynamicTable.getDynamicValue(ElfDynamicType.DT_SYMTAB));
            }
            catch (NotFoundException e) {
                this.errorConsumer.accept("ELF does not contain a dynamic symbol table (DT_SYMTAB)");
            }
        }
        ArrayList<ElfSymbolTable> symbolTableList = new ArrayList<ElfSymbolTable>();
        for (ElfSectionHeader symbolTableSectionHeader : this.sectionHeaders) {
            if (symbolTableSectionHeader.getType() != 2 && symbolTableSectionHeader.getType() != 11 || symbolTableSectionHeader.isInvalidOffset()) continue;
            ElfSectionHeader stringTableSectionHeader = this.sectionHeaders[symbolTableSectionHeader.getLink()];
            ElfStringTable stringTable = this.getStringTable(stringTableSectionHeader);
            Msg.debug((Object)this, (Object)("Elf symbol table section " + symbolTableSectionHeader.getNameAsString() + " linked to string table section " + stringTableSectionHeader.getNameAsString()));
            boolean isDyanmic = ".dynsym".equals(symbolTableSectionHeader.getNameAsString());
            int[] symbolSectionIndexTable = this.getExtendedSymbolSectionIndexTable(symbolTableSectionHeader);
            ElfSymbolTable symbolTable = new ElfSymbolTable(this.reader, this, symbolTableSectionHeader, symbolTableSectionHeader.getOffset(), symbolTableSectionHeader.getAddress(), symbolTableSectionHeader.getSize(), symbolTableSectionHeader.getEntrySize(), stringTable, symbolSectionIndexTable, isDyanmic);
            symbolTableList.add(symbolTable);
            if (symbolTable.getAddressOffset() != dynamicSymbolTableAddr) continue;
            this.dynamicSymbolTable = symbolTable;
        }
        if (this.dynamicSymbolTable == null && dynamicSymbolTableAddr != -1L) {
            this.dynamicSymbolTable = this.parseDynamicSymbolTable();
            if (this.dynamicSymbolTable != null) {
                symbolTableList.add(this.dynamicSymbolTable);
            }
        }
        this.symbolTables = new ElfSymbolTable[symbolTableList.size()];
        symbolTableList.toArray(this.symbolTables);
    }

    private ElfDynamicType getDynamicHashTableType() {
        if (this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_HASH)) {
            return ElfDynamicType.DT_HASH;
        }
        if (this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_GNU_HASH)) {
            return ElfDynamicType.DT_GNU_HASH;
        }
        if (this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_GNU_XHASH)) {
            return ElfDynamicType.DT_GNU_XHASH;
        }
        return null;
    }

    private ElfSymbolTable parseDynamicSymbolTable() throws IOException {
        ElfDynamicType dynamicHashType = this.getDynamicHashTableType();
        if (!this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_SYMTAB) || !this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_SYMENT) || dynamicHashType == null) {
            if (this.dynamicStringTable != null) {
                Msg.warn((Object)this, (Object)"Failed to parse DT_SYMTAB, missing dynamic dependency");
            }
            return null;
        }
        try {
            long tableAddr = this.dynamicTable.getDynamicValue(ElfDynamicType.DT_SYMTAB);
            if (tableAddr == 0L) {
                this.errorConsumer.accept("ELF Dynamic String Table of size appears to have been stripped from binary");
            }
            if (this.dynamicStringTable == null) {
                this.errorConsumer.accept("Failed to process DT_SYMTAB, missing dynamic string table");
                return null;
            }
            if (tableAddr == 0L) {
                return null;
            }
            tableAddr = this.adjustAddressForPrelink(tableAddr);
            long tableEntrySize = this.dynamicTable.getDynamicValue(ElfDynamicType.DT_SYMENT);
            long hashTableAddr = this.dynamicTable.getDynamicValue(dynamicHashType);
            hashTableAddr = this.adjustAddressForPrelink(hashTableAddr);
            ElfProgramHeader symbolTableLoadHeader = this.getProgramLoadHeaderContaining(tableAddr);
            if (symbolTableLoadHeader == null) {
                this.errorConsumer.accept("Failed to locate DT_SYMTAB in memory at 0x" + Long.toHexString(tableAddr));
                return null;
            }
            ElfProgramHeader hashTableLoadHeader = this.getProgramLoadHeaderContaining(hashTableAddr);
            if (hashTableLoadHeader == null) {
                this.errorConsumer.accept("Failed to locate DT_HASH, DT_GNU_HASH, or DT_GNU_XHASH in memory at 0x" + Long.toHexString(hashTableAddr));
                return null;
            }
            long symbolTableOffset = symbolTableLoadHeader.getOffset(tableAddr);
            long symbolHashTableOffset = hashTableLoadHeader.getOffset(hashTableAddr);
            int symCount = dynamicHashType == ElfDynamicType.DT_GNU_HASH ? this.deriveGnuHashDynamicSymbolCount(symbolHashTableOffset) : (dynamicHashType == ElfDynamicType.DT_GNU_XHASH ? this.deriveGnuXHashDynamicSymbolCount(symbolHashTableOffset) : this.reader.readInt(symbolHashTableOffset + 4L));
            return new ElfSymbolTable(this.reader, this, null, symbolTableOffset, tableAddr, tableEntrySize * (long)symCount, tableEntrySize, this.dynamicStringTable, null, true);
        }
        catch (NotFoundException e) {
            throw new AssertException((Throwable)e);
        }
    }

    private int deriveGnuHashDynamicSymbolCount(long gnuHashTableOffset) throws IOException {
        int chainValue;
        long bucketsOffset;
        int numBuckets = this.reader.readInt(gnuHashTableOffset);
        int symbolBase = this.reader.readInt(gnuHashTableOffset + 4L);
        int bloomSize = this.reader.readInt(gnuHashTableOffset + 8L);
        int bloomWordSize = this.is64Bit() ? 8 : 4;
        long bucketOffset = bucketsOffset = gnuHashTableOffset + 16L + (long)(bloomWordSize * bloomSize);
        int maxSymbolIndex = 0;
        for (int i = 0; i < numBuckets; ++i) {
            int symbolIndex = this.reader.readInt(bucketOffset);
            if (symbolIndex > maxSymbolIndex) {
                maxSymbolIndex = symbolIndex;
            }
            bucketOffset += 4L;
        }
        int chainIndex = maxSymbolIndex - symbolBase;
        ++maxSymbolIndex;
        long chainOffset = bucketOffset + (long)(4 * chainIndex);
        while (((chainValue = this.reader.readInt(chainOffset)) & 1) == 0) {
            ++maxSymbolIndex;
            chainOffset += 4L;
        }
        return maxSymbolIndex;
    }

    private int deriveGnuXHashDynamicSymbolCount(long gnuHashTableOffset) throws IOException {
        int ngnusyms = this.reader.readInt(gnuHashTableOffset);
        int symndx = this.reader.readInt(gnuHashTableOffset + 8L);
        return symndx + ngnusyms;
    }

    private boolean providerContainsRegion(long offset, int length) {
        try {
            return offset >= 0L && offset + (long)length <= this.provider.length();
        }
        catch (IOException e) {
            return false;
        }
    }

    protected void parseSectionHeaders() throws IOException {
        int i;
        if (this.reader == null) {
            throw new IOException("ELF binary reader is null!");
        }
        if (this.parsedSectionHeaders) {
            return;
        }
        this.parsedSectionHeaders = true;
        boolean missing = false;
        this.sectionHeaders = new ElfSectionHeader[this.e_shnum];
        for (i = 0; i < this.e_shnum; ++i) {
            long index = this.e_shoff + (long)(i * this.e_shentsize);
            if (!missing && !this.providerContainsRegion(index, this.e_shentsize)) {
                int unreadCnt = this.e_shnum - i;
                this.errorConsumer.accept(unreadCnt + " of " + this.e_shnum + " section headers are truncated/missing from file");
                missing = true;
            }
            this.sectionHeaders[i] = new ElfSectionHeader(this.reader.clone(index), this);
            if (this.sectionHeaders[i].getType() != 18) continue;
            this.hasExtendedSymbolSectionIndexTable = true;
        }
        if (this.sectionHeaders.length != 0) {
            this.section0 = this.sectionHeaders[0];
        }
        for (i = 0; i < this.e_shnum; ++i) {
            this.sectionHeaders[i].updateName();
        }
    }

    private void parseProgramHeaders() throws IOException {
        boolean missing = false;
        this.programHeaders = new ElfProgramHeader[this.e_phnum];
        for (int i = 0; i < this.e_phnum; ++i) {
            long index = this.e_phoff + (long)(i * this.e_phentsize);
            if (!missing && !this.providerContainsRegion(index, this.e_phentsize)) {
                int unreadCnt = this.e_phnum - i;
                this.errorConsumer.accept(unreadCnt + " of " + this.e_phnum + " program headers are truncated/missing from file");
                missing = true;
            }
            this.programHeaders[i] = new ElfProgramHeader(this.reader.clone(index), this);
        }
        long size = 0L;
        for (ElfProgramHeader pheader : this.programHeaders) {
            size += pheader.getFileSize();
        }
        if (size == this.reader.length()) {
            long relOffset = 0L;
            for (ElfProgramHeader pheader : this.programHeaders) {
                pheader.setOffset(relOffset);
                relOffset += pheader.getFileSize();
            }
        }
    }

    public boolean isBigEndian() {
        return this.e_ident_data == 2;
    }

    public boolean isLittleEndian() {
        return this.e_ident_data == 1;
    }

    public boolean is32Bit() {
        return this.e_ident_class == 1;
    }

    public boolean is64Bit() {
        return this.e_ident_class == 2;
    }

    private long getMinBase(long addr, long minBase) {
        if (this.is32Bit()) {
            addr = Integer.toUnsignedLong((int)addr);
        }
        if (Long.compareUnsigned(addr, minBase) < 0) {
            minBase = addr;
        }
        return minBase;
    }

    public long findImageBase() {
        long index;
        long minBase = -1L;
        int n = Math.min(this.e_phnum, 20);
        for (int i = 0; i < n && this.providerContainsRegion(index = this.e_phoff + (long)(i * this.e_phentsize), this.e_phentsize); ++i) {
            try {
                int headerType = this.reader.readInt(index);
                if (headerType != 1) continue;
                ElfProgramHeader header = new ElfProgramHeader(this.reader.clone(index), this);
                minBase = this.getMinBase(header.getVirtualAddress(), minBase);
                continue;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return minBase == -1L ? 0L : minBase;
    }

    public long getImageBase() {
        if (this.elfImageBase != null) {
            return this.elfImageBase;
        }
        this.elfImageBase = 0L;
        long base = this.getPreLinkImageBase();
        if (base != -1L) {
            this.elfImageBase = base;
        } else {
            int n = Math.min(this.programHeaders.length, 20);
            long minBase = -1L;
            for (int i = 0; i < n; ++i) {
                ElfProgramHeader header = this.programHeaders[i];
                if (this.programHeaders[i].getType() != 1) continue;
                minBase = this.getMinBase(header.getVirtualAddress(), minBase);
            }
            this.elfImageBase = minBase == -1L ? 0L : minBase;
        }
        return this.elfImageBase;
    }

    public boolean isPreLinked() {
        if (this.getPreLinkImageBase() != -1L) {
            return true;
        }
        return this.dynamicTable != null && this.dynamicTable.containsDynamicValue(ElfDynamicType.DT_GNU_PRELINKED);
    }

    private long getPreLinkImageBase() {
        if (this.preLinkImageBase != null) {
            return this.preLinkImageBase;
        }
        this.preLinkImageBase = -1L;
        try {
            long fileLength = this.reader.getByteProvider().length();
            if (fileLength < 8L) {
                return -1L;
            }
            int preLinkImageBaseInt = this.reader.readInt(fileLength - 8L);
            String preLinkMagicString = this.reader.readAsciiString(fileLength - 4L, 4).trim();
            if (preLinkMagicString.equals("PRE")) {
                this.preLinkImageBase = Integer.toUnsignedLong(preLinkImageBaseInt);
            }
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Elf prelink read failure", (Throwable)e);
        }
        return this.preLinkImageBase;
    }

    public boolean isSectionLoaded(ElfSectionHeader section) {
        if (section.getType() == 0) {
            return false;
        }
        long sectionStart = section.getAddress();
        if (sectionStart == 0L) {
            return false;
        }
        long sectionEnd = section.getSize() - 1L + sectionStart;
        for (ElfProgramHeader segment : this.programHeaders) {
            if (segment.getType() != 1) continue;
            long segmentStart = segment.getVirtualAddress();
            long segmentEnd = segment.getMemorySize() - 1L + segmentStart;
            if (segmentStart > sectionStart || segmentEnd < sectionEnd) continue;
            return true;
        }
        return false;
    }

    private void determineHeaderEndianness() throws ElfException, IOException {
        if (this.provider.length() < (long)INITIAL_READ_LEN) {
            throw new ElfException("Not enough bytes to be a valid ELF executable.");
        }
        this.hasLittleEndianHeaders = true;
        byte[] bytes = this.provider.readBytes(0L, INITIAL_READ_LEN);
        if (bytes[5] == 2) {
            this.hasLittleEndianHeaders = false;
        } else if (bytes[5] != 1) {
            this.errorConsumer.accept("Invalid EI_DATA, assuming little-endian headers (EI_DATA=0x" + Integer.toHexString(bytes[5]) + ")");
        }
        if (!this.hasLittleEndianHeaders && bytes[16] != 0 && bytes[20] == 1) {
            this.hasLittleEndianHeaders = true;
        }
    }

    public short e_ehsize() {
        return this.e_ehsize;
    }

    public long e_entry() {
        if (this.e_entry == 0L) {
            return 0L;
        }
        return this.adjustAddressForPrelink(this.e_entry);
    }

    public int e_flags() {
        return this.e_flags;
    }

    public short e_machine() {
        return this.e_machine;
    }

    public byte e_ident_osabi() {
        return this.e_ident_osabi;
    }

    public byte e_ident_abiversion() {
        return this.e_ident_abiversion;
    }

    public short e_phentsize() {
        return this.e_phentsize;
    }

    public int getProgramHeaderCount() {
        return this.e_phnum;
    }

    public long e_phoff() {
        return this.e_phoff;
    }

    public short e_shentsize() {
        return this.e_shentsize;
    }

    public int getSectionHeaderCount() {
        return this.e_shnum;
    }

    public long e_shoff() {
        return this.e_shoff;
    }

    public int e_shstrndx() {
        return this.e_shstrndx;
    }

    public short e_type() {
        return this.e_type;
    }

    public boolean isRelocatable() {
        return this.e_type == 1;
    }

    public boolean isSharedObject() {
        return this.e_type == 3;
    }

    public boolean isExecutable() {
        return this.e_type == 2;
    }

    public int e_version() {
        return this.e_version;
    }

    public ElfSectionHeader[] getSections() {
        return this.sectionHeaders;
    }

    public ElfSectionHeader[] getSections(int type) {
        ArrayList<ElfSectionHeader> list = new ArrayList<ElfSectionHeader>();
        for (ElfSectionHeader sectionHeader : this.sectionHeaders) {
            if (sectionHeader.getType() != type) continue;
            list.add(sectionHeader);
        }
        ElfSectionHeader[] sections = new ElfSectionHeader[list.size()];
        list.toArray(sections);
        return sections;
    }

    public ElfSectionHeader getSection(String name) {
        ArrayList<ElfSectionHeader> list = new ArrayList<ElfSectionHeader>();
        for (ElfSectionHeader sectionHeader : this.sectionHeaders) {
            if (name == null || !name.equals(sectionHeader.getNameAsString())) continue;
            list.add(sectionHeader);
        }
        if (list.size() == 0) {
            return null;
        }
        if (list.size() > 1) {
            throw new RuntimeException(">1 section with name of " + name);
        }
        return (ElfSectionHeader)list.get(0);
    }

    public ElfSectionHeader getSectionAt(long address) {
        for (ElfSectionHeader sectionHeader : this.sectionHeaders) {
            if (!sectionHeader.isAlloc() || sectionHeader.getAddress() != address) continue;
            return sectionHeader;
        }
        return null;
    }

    public ElfSectionHeader getSectionLoadHeaderContaining(long address) {
        for (ElfSectionHeader sectionHeader : this.sectionHeaders) {
            if (!sectionHeader.isAlloc()) continue;
            long start = sectionHeader.getAddress();
            long end = start + sectionHeader.getSize();
            if (start > address || address >= end) continue;
            return sectionHeader;
        }
        return null;
    }

    public ElfSectionHeader getSectionHeaderContainingFileRange(long fileOffset, long fileRangeLength) {
        long maxOffset = fileOffset + fileRangeLength - 1L;
        for (ElfSectionHeader section : this.sectionHeaders) {
            long size;
            if (section.getType() == 0 || section.isInvalidOffset() || (size = section.getSize()) == 0L) continue;
            long start = section.getOffset();
            long end = start + size - 1L;
            if (fileOffset < start || maxOffset > end) continue;
            return section;
        }
        return null;
    }

    public int getSectionIndex(ElfSectionHeader section) {
        for (int i = 0; i < this.sectionHeaders.length; ++i) {
            if (this.sectionHeaders[i] != section) continue;
            return i;
        }
        throw new RuntimeException("Section not located.");
    }

    public ElfProgramHeader[] getProgramHeaders() {
        return this.programHeaders;
    }

    public ElfProgramHeader[] getProgramHeaders(int type) {
        ArrayList<ElfProgramHeader> list = new ArrayList<ElfProgramHeader>();
        for (ElfProgramHeader programHeader : this.programHeaders) {
            if (programHeader.getType() != type) continue;
            list.add(programHeader);
        }
        ElfProgramHeader[] arr = new ElfProgramHeader[list.size()];
        list.toArray(arr);
        return arr;
    }

    public ElfDynamicTable getDynamicTable() {
        return this.dynamicTable;
    }

    public ElfProgramHeader getProgramHeaderProgramHeader() {
        ElfProgramHeader[] pharr = this.getProgramHeaders(6);
        if (pharr.length == 0 || pharr.length > 1) {
            return null;
        }
        return pharr[0];
    }

    public ElfProgramHeader getProgramHeaderAt(long virtualAddr) {
        for (ElfProgramHeader programHeader : this.programHeaders) {
            if (programHeader.getType() != 1 || programHeader.getVirtualAddress() != virtualAddr) continue;
            return programHeader;
        }
        return null;
    }

    public ElfProgramHeader getProgramLoadHeaderContaining(long virtualAddr) {
        for (ElfProgramHeader programHeader : this.programHeaders) {
            if (programHeader.getType() != 1) continue;
            long start = programHeader.getVirtualAddress();
            long end = programHeader.getAdjustedMemorySize() - 1L + start;
            if (virtualAddr < start || virtualAddr > end) continue;
            return programHeader;
        }
        return null;
    }

    public ElfProgramHeader getProgramLoadHeaderContainingFileOffset(long offset) {
        for (ElfProgramHeader programHeader : this.programHeaders) {
            if (programHeader == null || programHeader.getType() != 1 || programHeader.isInvalidOffset()) continue;
            long start = programHeader.getOffset();
            long end = start + (programHeader.getFileSize() - 1L);
            if (offset < start || offset > end) continue;
            return programHeader;
        }
        return null;
    }

    public String[] getDynamicLibraryNames() {
        return this.dynamicLibraryNames;
    }

    public ElfStringTable getDynamicStringTable() {
        return this.dynamicStringTable;
    }

    public ElfStringTable[] getStringTables() {
        return this.stringTables;
    }

    public ElfStringTable getStringTable(ElfSectionHeader section) {
        for (ElfStringTable stringTable : this.stringTables) {
            if (stringTable.getFileOffset() != section.getOffset()) continue;
            return stringTable;
        }
        return null;
    }

    public ElfSymbolTable getDynamicSymbolTable() {
        return this.dynamicSymbolTable;
    }

    public ElfSymbolTable[] getSymbolTables() {
        return this.symbolTables;
    }

    public ElfSymbolTable getSymbolTable(ElfSectionHeader symbolTableSection) {
        if (symbolTableSection == null) {
            return null;
        }
        for (ElfSymbolTable symbolTable : this.symbolTables) {
            if (symbolTable.getFileOffset() != symbolTableSection.getOffset()) continue;
            return symbolTable;
        }
        return null;
    }

    public ElfRelocationTable[] getRelocationTables() {
        return this.relocationTables;
    }

    public ElfRelocationTable getRelocationTable(ElfSectionHeader relocSection) {
        return this.getRelocationTableAtOffset(relocSection.getOffset());
    }

    public ElfRelocationTable getRelocationTableAtOffset(long fileOffset) {
        for (ElfRelocationTable relocationTable : this.relocationTables) {
            if (relocationTable.getFileOffset() != fileOffset) continue;
            return relocationTable;
        }
        return null;
    }

    public String getMachineName() {
        return Short.toString(this.e_machine);
    }

    public String getFlags() {
        return Integer.toString(this.e_flags);
    }

    @Override
    public DataType toDataType() {
        if (this.headerStructure != null) {
            return this.headerStructure;
        }
        String name = this.is32Bit() ? "Elf32_Ehdr" : "Elf64_Ehdr";
        this.headerStructure = new StructureDataType(new CategoryPath("/ELF"), name, 0);
        this.headerStructure.add(BYTE, "e_ident_magic_num", null);
        this.headerStructure.add(STRING, this.e_ident_magic_str.length(), "e_ident_magic_str", null);
        this.headerStructure.add(BYTE, "e_ident_class", null);
        this.headerStructure.add(BYTE, "e_ident_data", null);
        this.headerStructure.add(BYTE, "e_ident_version", null);
        this.headerStructure.add(BYTE, "e_ident_osabi", null);
        this.headerStructure.add(BYTE, "e_ident_abiversion", null);
        this.headerStructure.add((DataType)new ArrayDataType(BYTE, 7, 1), "e_ident_pad", null);
        this.headerStructure.add(WORD, "e_type", null);
        this.headerStructure.add(WORD, "e_machine", null);
        this.headerStructure.add(DWORD, "e_version", null);
        if (this.is32Bit()) {
            this.headerStructure.add(DWORD, "e_entry", null);
            this.headerStructure.add(DWORD, "e_phoff", null);
            this.headerStructure.add(DWORD, "e_shoff", null);
        } else {
            this.headerStructure.add(QWORD, "e_entry", null);
            this.headerStructure.add(QWORD, "e_phoff", null);
            this.headerStructure.add(QWORD, "e_shoff", null);
        }
        this.headerStructure.add(DWORD, "e_flags", null);
        this.headerStructure.add(WORD, "e_ehsize", null);
        this.headerStructure.add(WORD, "e_phentsize", null);
        this.headerStructure.add(WORD, "e_phnum", null);
        this.headerStructure.add(WORD, "e_shentsize", null);
        this.headerStructure.add(WORD, "e_shnum", null);
        this.headerStructure.add(WORD, "e_shstrndx", null);
        return this.headerStructure;
    }

    public int getEntryComponentOrdinal() {
        return 11;
    }

    public int getPhoffComponentOrdinal() {
        return 12;
    }

    public int getShoffComponentOrdinal() {
        return 13;
    }
}

