/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import type { AgentDefinition } from './types.js';
/**
 * DTO for TOML parsing - represents the raw structure of the TOML file.
 */
interface TomlAgentDefinition {
    name: string;
    description: string;
    display_name?: string;
    tools?: string[];
    prompts: {
        system_prompt: string;
        query?: string;
    };
    model?: {
        model?: string;
        temperature?: number;
    };
    run?: {
        max_turns?: number;
        timeout_mins?: number;
    };
}
/**
 * Error thrown when an agent definition is invalid or cannot be loaded.
 */
export declare class AgentLoadError extends Error {
    filePath: string;
    constructor(filePath: string, message: string);
}
/**
 * Result of loading agents from a directory.
 */
export interface AgentLoadResult {
    agents: AgentDefinition[];
    errors: AgentLoadError[];
}
/**
 * Parses and validates an agent TOML file.
 *
 * @param filePath Path to the TOML file.
 * @returns The parsed and validated TomlAgentDefinition.
 * @throws AgentLoadError if parsing or validation fails.
 */
export declare function parseAgentToml(filePath: string): Promise<TomlAgentDefinition>;
/**
 * Converts a TomlAgentDefinition DTO to the internal AgentDefinition structure.
 *
 * @param toml The parsed TOML definition.
 * @returns The internal AgentDefinition.
 */
export declare function tomlToAgentDefinition(toml: TomlAgentDefinition): AgentDefinition;
/**
 * Loads all agents from a specific directory.
 * Ignores non-TOML files and files starting with _.
 *
 * @param dir Directory path to scan.
 * @returns Object containing successfully loaded agents and any errors.
 */
export declare function loadAgentsFromDirectory(dir: string): Promise<AgentLoadResult>;
export {};
