/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { loadGlobalMemory, loadEnvironmentMemory, loadJitSubdirectoryMemory, concatenateInstructions, } from '../utils/memoryDiscovery.js';
export class ContextManager {
    loadedPaths = new Set();
    config;
    globalMemory = '';
    environmentMemory = '';
    constructor(config) {
        this.config = config;
    }
    /**
     * Loads the global memory (Tier 1) and returns the formatted content.
     */
    async loadGlobalMemory() {
        const result = await loadGlobalMemory(this.config.getDebugMode());
        this.markAsLoaded(result.files.map((f) => f.path));
        this.globalMemory = concatenateInstructions(result.files.map((f) => ({ filePath: f.path, content: f.content })), this.config.getWorkingDir());
        return this.globalMemory;
    }
    /**
     * Loads the environment memory (Tier 2) and returns the formatted content.
     */
    async loadEnvironmentMemory(trustedRoots, extensionLoader) {
        const result = await loadEnvironmentMemory(trustedRoots, extensionLoader, this.config.getDebugMode());
        this.markAsLoaded(result.files.map((f) => f.path));
        this.environmentMemory = concatenateInstructions(result.files.map((f) => ({ filePath: f.path, content: f.content })), this.config.getWorkingDir());
        return this.environmentMemory;
    }
    /**
     * Discovers and loads context for a specific accessed path (Tier 3 - JIT).
     * Traverses upwards from the accessed path to the project root.
     */
    async discoverContext(accessedPath, trustedRoots) {
        const result = await loadJitSubdirectoryMemory(accessedPath, trustedRoots, this.loadedPaths, this.config.getDebugMode());
        if (result.files.length === 0) {
            return '';
        }
        this.markAsLoaded(result.files.map((f) => f.path));
        return concatenateInstructions(result.files.map((f) => ({ filePath: f.path, content: f.content })), this.config.getWorkingDir());
    }
    getGlobalMemory() {
        return this.globalMemory;
    }
    getEnvironmentMemory() {
        return this.environmentMemory;
    }
    markAsLoaded(paths) {
        for (const p of paths) {
            this.loadedPaths.add(p);
        }
    }
    /**
     * Resets the loaded paths tracking and memory. Useful for testing or full reloads.
     */
    reset() {
        this.loadedPaths.clear();
        this.globalMemory = '';
        this.environmentMemory = '';
    }
    getLoadedPaths() {
        return this.loadedPaths;
    }
}
//# sourceMappingURL=contextManager.js.map