// Copyright 2014 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// This file contains 3011-digit string representations of fundamental constants.
// They are stored in 10000-bit Floats, almost sufficient to represent the values.
// When used in calculations, those Floats are truncated to the precision of the operands.

// In the unlikely event more digits are needed, it's easy to find the values online and
// lengthen the constants. Or compute them: By a magic of math, log 2 is not needed
// to compute log 2 by the algorithm here (the exponent is zero in the Taylor-Maclaurin
// series) so it is possible to bootstrap to huge precisions.

package value

import (
	"fmt"
	"math/big"

	"robpike.io/ivy/config"
)

const (
	// 10000*log(2)/log(10) == 3011
	constPrecisionInBits   = 10000
	constPrecisionInDigits = 3011
)

// Note: These are pre-allocated Values (interfaces) so that
// code that needs to return a Value can 'return zero'
// instead of 'return Int(0)' and avoid an allocation.
var (
	zero     Value = Int(0)
	one      Value = Int(1)
	minusOne Value = Int(-1)
)

var (
	empty = NewVector()

	bigIntZero     = bigInt64(0)
	bigIntOne      = bigInt64(1)
	bigIntTwo      = bigInt64(2)
	bigIntMinusOne = bigInt64(-1)
	bigIntTen      = big.NewInt(10)
	bigIntBillion  = big.NewInt(1e9)
	MaxBigInt63    = big.NewInt(int64(^uint64(0) >> 1)) // Used in ../parse/special.go

	bigRatOne      = big.NewRat(1, 1)
	bigRatMinusOne = big.NewRat(-1, 1)
	bigRatTen      = big.NewRat(10, 1)
	bigRatBillion  = big.NewRat(1e9, 1)

	complexOne       = NewComplex(one, zero)
	complexHalf      = NewComplex(BigRat{big.NewRat(1, 2)}, zero)
	minusOneOverTwoI Complex

	// set to constPrecision
	floatE          *big.Float
	floatPi         *big.Float
	floatPiBy2      *big.Float
	floatMinusPiBy2 *big.Float
	floatLog2       *big.Float
	floatLog10      *big.Float
	floatZero       *big.Float
	floatOne        *big.Float
	floatTwo        *big.Float
	floatHalf       *big.Float
	floatMinusOne   *big.Float
)

const strE = "2.71828182845904523536028747135266249775724709369995957496696762772407663035354759457138217852516642742746639193200305992181741359662904357290033429526059563073813232862794349076323382988075319525101901157383418793070215408914993488416750924476146066808226480016847741185374234544243710753907774499206955170276183860626133138458300075204493382656029760673711320070932870912744374704723069697720931014169283681902551510865746377211125238978442505695369677078544996996794686445490598793163688923009879312773617821542499922957635148220826989519366803318252886939849646510582093923982948879332036250944311730123819706841614039701983767932068328237646480429531180232878250981945581530175671736133206981125099618188159304169035159888851934580727386673858942287922849989208680582574927961048419844436346324496848756023362482704197862320900216099023530436994184914631409343173814364054625315209618369088870701676839642437814059271456354906130310720851038375051011574770417189861068739696552126715468895703503540212340784981933432106817012100562788023519303322474501585390473041995777709350366041699732972508868769664035557071622684471625607988265178713419512466520103059212366771943252786753985589448969709640975459185695638023637016211204774272283648961342251644507818244235294863637214174023889344124796357437026375529444833799801612549227850925778256209262264832627793338656648162772516401910590049164499828931505660472580277863186415519565324425869829469593080191529872117255634754639644791014590409058629849679128740687050489585867174798546677575732056812884592054133405392200011378630094556068816674001698420558040336379537645203040243225661352783695117788386387443966253224985065499588623428189970773327617178392803494650143455889707194258639877275471096295374152111513683506275260232648472870392076431005958411661205452970302364725492966693811513732275364509888903136020572481765851180630364428123149655070475102544650117272115551948668508003685322818315219600373562527944951582841882947876108526398139559900673764829224437528718462457803619298197139914756448826260390338144182326251509748279877799643730899703888677822713836057729788241256119071766394650706330452795466185509666618566470971134447401607046262156807174818778443714369882185596709591025968620023537185887485696522000503117343920732113908032936344797273559552773490717837934216370120500545132638354400018632399149070547977805669785335804896690629511943247309958765523681285904138324116072260299833053537087613893963917795745401613722361878936526053815584158718692553860616477983402543512843961294603529133259427949043372990857315802909586313826832914771163963370924003168945863606064584592512699465572483918656420975268508230754425459937691704197778008536273094171016343490769642372229435236612557250881477922315197477806056967253801718077636034624592787784658506560507808442115296975218908740196609066518035165017925046195013665854366327125496399085491442000145747608193022120660243300964127048943903971771951806990869986066365832322787093765022601492910115"

const strPi = "3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127372458700660631558817488152092096282925409171536436789259036001133053054882046652138414695194151160943305727036575959195309218611738193261179310511854807446237996274956735188575272489122793818301194912983367336244065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960917363717872146844090122495343014654958537105079227968925892354201995611212902196086403441815981362977477130996051870721134999999837297804995105973173281609631859502445945534690830264252230825334468503526193118817101000313783875288658753320838142061717766914730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586327886593615338182796823030195203530185296899577362259941389124972177528347913151557485724245415069595082953311686172785588907509838175463746493931925506040092770167113900984882401285836160356370766010471018194295559619894676783744944825537977472684710404753464620804668425906949129331367702898915210475216205696602405803815019351125338243003558764024749647326391419927260426992279678235478163600934172164121992458631503028618297455570674983850549458858692699569092721079750930295532116534498720275596023648066549911988183479775356636980742654252786255181841757467289097777279380008164706001614524919217321721477235014144197356854816136115735255213347574184946843852332390739414333454776241686251898356948556209921922218427255025425688767179049460165346680498862723279178608578438382796797668145410095388378636095068006422512520511739298489608412848862694560424196528502221066118630674427862203919494504712371378696095636437191728746776465757396241389086583264599581339047802759009946576407895126946839835259570982582262052248940772671947826848260147699090264013639443745530506820349625245174939965143142980919065925093722169646151570985838741059788595977297549893016175392846813826868386894277415599185592524595395943104997252468084598727364469584865383673622262609912460805124388439045124413654976278079771569143599770012961608944169486855584840635342207222582848864815845602850601684273945226746767889525213852254995466672782398645659611635488623057745649803559363456817432411251507606947945109659609402522887971089314566913686722874894056010150330861792868092087476091782493858900971490967598526136554978189312978482168299894872265880485756401427047755513237964145152374623436454285844479526586782105114135473573952311342716610213596953623144295248493718711014576540359027993440374200731057853906219838744780847848968332144571386875194350643021845319104848100537061468067491927819119793995206141966342875444064374512371819217999839101591956181467514269123974894090718649423196156794520809514655022"

const strLog2 = "0.69314718055994530941723212145817656807550013436025525412068000949339362196969471560586332699641868754200148102057068573368552023575813055703267075163507596193072757082837143519030703862389167347112335011536449795523912047517268157493206515552473413952588295045300709532636664265410423915781495204374043038550080194417064167151864471283996817178454695702627163106454615025720740248163777338963855069526066834113727387372292895649354702576265209885969320196505855476470330679365443254763274495125040606943814710468994650622016772042452452961268794654619316517468139267250410380254625965686914419287160829380317271436778265487756648508567407764845146443994046142260319309673540257444607030809608504748663852313818167675143866747664789088143714198549423151997354880375165861275352916610007105355824987941472950929311389715599820565439287170007218085761025236889213244971389320378439353088774825970171559107088236836275898425891853530243634214367061189236789192372314672321720534016492568727477823445353476481149418642386776774406069562657379600867076257199184734022651462837904883062033061144630073719489002743643965002580936519443041191150608094879306786515887090060520346842973619384128965255653968602219412292420757432175748909770675268711581705113700915894266547859596489065305846025866838294002283300538207400567705304678700184162404418833232798386349001563121889560650553151272199398332030751408426091479001265168243443893572472788205486271552741877243002489794540196187233980860831664811490930667519339312890431641370681397776498176974868903887789991296503619270710889264105230924783917373501229842420499568935992206602204654941510613918788574424557751020683703086661948089641218680779020818158858000168811597305618667619918739520076671921459223672060253959543654165531129517598994005600036651356756905124592682574394648316833262490180382424082423145230614096380570070255138770268178516306902551370323405380214501901537402950994226299577964742713815736380172987394070424217997226696297993931270693574724049338653087975872169964512944649188377115670167859880498183889678413493831401407316647276532763591923351123338933870951320905927218547132897547079789138444546667619270288553342342989932180376915497334026754675887323677834291619181043011609169526554785973289176354555674286387746398710191243175425588830120677921028034120687975914308128330723030088349470579249659100586001234156175741327246594306843546521113502154434153995538185652275022142456644000627618330320647272572197515290827856842132079598863896727711955221881904660395700977470651261950527893229608893140562543344255239206203034394177735794559212590199255911484402423901255425900312953705192206150643458378787300203541442178575801323645166070991438314500498589668857722214865288216941812704886075897220321666312837832915676307498729857463892826937350984077804939500493399876264755070316221613903484529942491724837340613662263834936811168416705692521475138393063845537186268779732889555887163442975624475539236636948887782389017498102735655240505185477306"

const strLog10 = "2.30258509299404568401799145468436420760110148862877297603332790096757260967735248023599720508959829834196778404228624863340952546508280675666628736909878168948290720832555468084379989482623319852839350530896537773262884616336622228769821988674654366747440424327436515504893431493939147961940440022210510171417480036880840126470806855677432162283552201148046637156591213734507478569476834636167921018064450706480002775026849167465505868569356734206705811364292245544057589257242082413146956890167589402567763113569192920333765871416602301057030896345720754403708474699401682692828084811842893148485249486448719278096762712757753970276686059524967166741834857044225071979650047149510504922147765676369386629769795221107182645497347726624257094293225827985025855097852653832076067263171643095059950878075237103331011978575473315414218084275438635917781170543098274823850456480190956102992918243182375253577097505395651876975103749708886921802051893395072385392051446341972652872869651108625714921988499787488737713456862091670584980782805975119385444500997813114691593466624107184669231010759843831919129223079250374729865092900988039194170265441681633572755570315159611356484654619089704281976336583698371632898217440736600916217785054177927636773114504178213766011101073104239783252189489881759792179866639431952393685591644711824675324563091252877833096360426298215304087456092776072664135478757661626292656829870495795491395491804920906943858079003276301794150311786686209240853794986126493347935487173745167580953708828106745244010589244497647968607512027572418187498939597164310551884819528833074669931781463493000032120032776565413047262188397059679445794346834321839530441484480370130575367426215367557981477045803141363779323629156012818533649846694226146520645994207291711937060244492935803700771898109736253322454836698850552828596619280509844717519850366668087497049698227322024482334309716911113681358841869654932371499694197968780300885040897961859875657989483644521204369821641529298781174297333258860791591251096718751092924847502393057266544627620092306879151813580347770129559364629841236649702335517458619556477246185771736936840467657704787431978057385327181093388349633881306994556939934610109074561603331224794936045536184912333306370475172487127637914092439833181016473782337969226563768207170693584639453161694941170184193811940541644946611127471281970581778329384174223140993002291150236219218672333726838568827353337192510341293070563254442661142976538830182238409102619858288843358745596045300454837078905257847316628370195339223104752756499811922874278971371571322831964100342212421008218067952527668985818095611920839176072108091992346151695259909947378278064812805879273199389345341532018596971102140754228279629823706894176474064222575721245539252617937365243444056059533659153916031252448014931323457245387952438903683923645050788173135971123814532370150841349112232439092768172474960795579915136398288105828574053800065337165555301419633224191808762101820491949265148389269229370789863527064"

func newF(conf *config.Config) *big.Float {
	return new(big.Float).SetPrec(conf.FloatPrec())
}

// addPrec returns the prec+extra, capped at big.MaxPrec
func addPrec(prec, extra uint) uint {
	return uint(min(uint64(prec+extra), big.MaxPrec))
}

// newFP returns a new big.Float with the requested precision.
func newFP(prec uint) *big.Float {
	return new(big.Float).SetPrec(prec)
}

func newFloat(c Context) *big.Float {
	return newF(c.Config())
}

func Consts(c Context) (e, pi BigFloat) {
	conf := c.Config()
	if conf.FloatPrec() > constPrecisionInBits {
		fmt.Fprintf(conf.ErrOutput(), "warning: precision too high; only have %d digits (%d bits) of precision for e and pi", constPrecisionInDigits, constPrecisionInBits)
	}
	// Use the default precision for interger constants.
	floatZero = new(big.Float).SetInt64(0)
	floatOne = new(big.Float).SetInt64(1)
	floatTwo = new(big.Float).SetInt64(2)
	floatHalf = new(big.Float).SetFloat64(0.5)
	floatMinusOne = new(big.Float).SetInt64(-1)
	var ok bool
	// Use 32 bits of extra precision for real constants. The rationale is that
	// most special functions based on Taylor series need ⌈log(prec)⌉ guard digits
	// (<= 22, rounding up to a big.Word size on 32 bits targets).
	prec := addPrec(conf.FloatPrec(), 32)
	floatE, ok = newFP(prec).SetString(strE)
	if !ok {
		panic("setting e")
	}
	floatPi, ok = newFP(prec).SetString(strPi)
	if !ok {
		panic("setting pi")
	}
	floatPiBy2, ok = newFP(prec).SetString(strPi)
	if !ok {
		panic("setting pi/2")
	}
	floatPiBy2.Quo(floatPiBy2, floatTwo)
	floatMinusPiBy2 = newFP(prec).Neg(floatPiBy2)
	floatLog2, ok = newFP(prec).SetString(strLog2)
	if !ok {
		panic("setting log(2)")
	}
	floatLog10, ok = newFP(prec).SetString(strLog10)
	if !ok {
		panic("setting log(10)")
	}
	return BigFloat{newFloat(c).Set(floatE)}, BigFloat{newFloat(c).Set(floatPi)}
}

// -1/2i is remarkably hard to build.
func init() {
	num, err := setBigRatFromFloatString("0.0")
	if err != nil {
		panic(err)
	}
	den, err := setBigRatFromFloatString("-0.5")
	if err != nil {
		panic(err)
	}
	minusOneOverTwoI = NewComplex(num, den)
}
