package repl

import (
	"context"
	"path/filepath"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap"
	"go.uber.org/zap/zaptest"

	// Import all subpackages to verify the directory structure is correct
	_ "github.com/atinylittleshell/gsh/internal/repl/completion"
	_ "github.com/atinylittleshell/gsh/internal/repl/config"
	_ "github.com/atinylittleshell/gsh/internal/repl/context"
	_ "github.com/atinylittleshell/gsh/internal/repl/executor"
	"github.com/atinylittleshell/gsh/internal/repl/input"
	_ "github.com/atinylittleshell/gsh/internal/repl/predict"
	"github.com/atinylittleshell/gsh/internal/script/interpreter"
)

func TestDirectoryStructure(t *testing.T) {
	// This test verifies that all subpackages in internal/repl/ can be imported.
	// The imports above will fail at compile time if any package is missing
	// or has incorrect package declarations.
	t.Log("All internal/repl subpackages are correctly structured and importable")
}

func TestNewREPL_DefaultOptions(t *testing.T) {
	// Create a temporary directory for history
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")

	// Create a non-existent config path to use defaults
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	logger := zaptest.NewLogger(t)

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      logger,
	})
	require.NoError(t, err)
	require.NotNil(t, repl)
	defer repl.Close()

	// Verify default config was loaded (Config now only holds declarations)
	assert.NotNil(t, repl.Config())

	// Verify executor was created
	assert.NotNil(t, repl.Executor())

	// Verify history manager was created
	assert.NotNil(t, repl.History())
}

func TestNewREPL_WithConfig(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")

	logger := zaptest.NewLogger(t)

	// Use DefaultConfigContent to set up SDK config
	defaultConfig := `
model testModel {
	provider: "openai",
	model: "gpt-4",
}
`

	repl, err := NewREPL(Options{
		DefaultConfigContent: defaultConfig,
		HistoryPath:          historyPath,
		Logger:               logger,
	})
	require.NoError(t, err)
	require.NotNil(t, repl)
	defer repl.Close()

	// Verify model was loaded
	assert.NotNil(t, repl.Config().GetModel("testModel"))
}

func TestNewREPL_NilLogger(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	// Should not panic with nil logger
	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      nil,
	})
	require.NoError(t, err)
	require.NotNil(t, repl)
	defer repl.Close()
}

func TestREPL_HandleBuiltinCommand_Exit(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Test that exit returns ErrExit
	handled, err := repl.handleBuiltinCommand("exit")
	assert.True(t, handled)
	assert.Equal(t, ErrExit, err)

	// Test unhandled command
	handled, err = repl.handleBuiltinCommand("ls")
	assert.False(t, handled)
	assert.NoError(t, err)
}

func TestREPL_ProcessCommand_Empty(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	ctx := context.Background()

	// Empty command should be no-op
	err = repl.processCommand(ctx, "")
	assert.NoError(t, err)

	// Whitespace-only command should be no-op
	err = repl.processCommand(ctx, "   ")
	assert.NoError(t, err)
}

func TestREPL_ProcessCommand_Echo(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	ctx := context.Background()

	// Execute a simple echo command
	err = repl.processCommand(ctx, "echo hello")
	assert.NoError(t, err)

	// Verify exit code was recorded
	assert.Equal(t, 0, repl.lastExitCode)
}

func TestREPL_ProcessCommand_RecordsHistory(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	ctx := context.Background()

	// Execute a command
	err = repl.processCommand(ctx, "echo test_history")
	assert.NoError(t, err)

	// Verify it was recorded in history
	entries, err := repl.History().GetRecentEntries("", 10)
	require.NoError(t, err)
	require.Len(t, entries, 1)
	assert.Equal(t, "echo test_history", entries[0].Command)
	assert.True(t, entries[0].ExitCode.Valid)
	assert.Equal(t, int32(0), entries[0].ExitCode.Int32)
}

func TestREPL_ProcessCommand_FailingCommand(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	ctx := context.Background()

	// Execute a failing command
	err = repl.processCommand(ctx, "exit 42")
	assert.NoError(t, err) // processCommand doesn't return error for non-zero exit

	// Verify exit code was recorded
	assert.Equal(t, 42, repl.lastExitCode)

	// Verify it was recorded in history with correct exit code
	entries, err := repl.History().GetRecentEntries("", 10)
	require.NoError(t, err)
	require.Len(t, entries, 1)
	assert.True(t, entries[0].ExitCode.Valid)
	assert.Equal(t, int32(42), entries[0].ExitCode.Int32)
}

func TestREPL_GetHistoryValues(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	ctx := context.Background()

	// Execute some commands
	_ = repl.processCommand(ctx, "echo first")
	_ = repl.processCommand(ctx, "echo second")
	_ = repl.processCommand(ctx, "echo third")

	// Get history values
	values := repl.getHistoryValues()
	require.Len(t, values, 3)

	// Most recent should be first
	assert.Equal(t, "echo third", values[0])
	assert.Equal(t, "echo second", values[1])
	assert.Equal(t, "echo first", values[2])
}

func TestREPL_GetPrompt(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")

	// Use DefaultConfigContent to set up SDK config with custom prompt via event handler
	defaultConfig := `
tool onPrompt(ctx) {
	gsh.prompt = "custom> "
}
gsh.on("repl.prompt", onPrompt)
`

	repl, err := NewREPL(Options{
		DefaultConfigContent: defaultConfig,
		HistoryPath:          historyPath,
		Logger:               zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Verify prompt is set by the event handler
	assert.Equal(t, "custom> ", repl.getPrompt())
}

func TestREPL_Close(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)

	// Close should not error
	err = repl.Close()
	assert.NoError(t, err)
}

func TestREPL_Run_ContextCancellation(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Create a context that's already cancelled
	ctx, cancel := context.WithCancel(context.Background())
	cancel()

	// Run should return immediately with context error
	err = repl.Run(ctx)
	assert.ErrorIs(t, err, context.Canceled)
}

func TestREPL_ProcessCommand_TracksDuration(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Mock time for testing
	callCount := 0
	originalTimeNow := timeNow
	timeNow = func() time.Time {
		callCount++
		if callCount == 1 {
			return time.Date(2024, 1, 1, 0, 0, 0, 0, time.UTC)
		}
		return time.Date(2024, 1, 1, 0, 0, 0, 100000000, time.UTC) // 100ms later
	}
	defer func() { timeNow = originalTimeNow }()

	ctx := context.Background()

	// Execute a command
	err = repl.processCommand(ctx, "echo hello")
	assert.NoError(t, err)

	// Verify duration was tracked
	assert.Equal(t, int64(100), repl.lastDurationMs)
}

func TestREPL_HandleBuiltinCommand_Clear(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Built-in commands are things like "exit"
	handled, err := repl.handleBuiltinCommand("exit")
	assert.True(t, handled)
	assert.Equal(t, ErrExit, err)
}

func TestREPL_HandleBuiltinCommand_UnknownCommand(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Unknown commands should not be handled
	handled, err := repl.handleBuiltinCommand("unknown")
	assert.False(t, handled)
	assert.NoError(t, err)

	handled, err = repl.handleBuiltinCommand("ls -la")
	assert.False(t, handled)
	assert.NoError(t, err)
}

func TestREPL_ContextProviderInitialized(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Verify context provider was created
	assert.NotNil(t, repl.contextProvider)

	// Get context and verify it contains expected keys
	contextMap := repl.contextProvider.GetContext()

	// Should have working_directory
	_, hasWorkingDir := contextMap["working_directory"]
	assert.True(t, hasWorkingDir, "context should include working_directory")

	// Should have system_info
	_, hasSystemInfo := contextMap["system_info"]
	assert.True(t, hasSystemInfo, "context should include system_info")

	// Should have git_status (might be "not in a git repository")
	_, hasGitStatus := contextMap["git_status"]
	assert.True(t, hasGitStatus, "context should include git_status")

	// Should have history_concise (since history manager was initialized)
	_, hasHistory := contextMap["history_concise"]
	assert.True(t, hasHistory, "context should include history_concise")
}

func TestREPL_UpdatePredictorContext_WithLazyModelResolver(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Predictor is now always created with lazy model resolution (SDKModelRef)
	// It will return empty predictions if gsh.models.lite is not configured
	assert.NotNil(t, repl.predictor)

	// updatePredictorContext should not panic
	repl.updatePredictorContext()
}

func TestREPL_UpdatePredictorContext_NilContextProvider(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Manually set contextProvider to nil to test edge case
	repl.contextProvider = nil

	// updatePredictorContext should not panic with nil contextProvider
	repl.updatePredictorContext()
}

func TestREPL_ContextProviderWithoutHistory(t *testing.T) {
	tmpDir := t.TempDir()
	// Use an invalid history path that will cause history initialization to fail
	historyPath := filepath.Join(tmpDir, "nonexistent_dir", "subdir", "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	logger := zaptest.NewLogger(t)

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      logger,
	})
	require.NoError(t, err)
	defer repl.Close()

	// Context provider should still be initialized
	assert.NotNil(t, repl.contextProvider)

	// Get context - should still have basic retrievers
	contextMap := repl.contextProvider.GetContext()

	// Should have working_directory
	_, hasWorkingDir := contextMap["working_directory"]
	assert.True(t, hasWorkingDir, "context should include working_directory")

	// Should have system_info
	_, hasSystemInfo := contextMap["system_info"]
	assert.True(t, hasSystemInfo, "context should include system_info")
}

func TestREPL_ContextContainsWorkingDirectory(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	contextMap := repl.contextProvider.GetContext()

	// Verify working_directory contains actual path
	workingDir := contextMap["working_directory"]
	assert.Contains(t, workingDir, "<working_dir>")
	assert.Contains(t, workingDir, "</working_dir>")
}

func TestREPL_ContextContainsSystemInfo(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	contextMap := repl.contextProvider.GetContext()

	// Verify system_info contains OS and arch
	systemInfo := contextMap["system_info"]
	assert.Contains(t, systemInfo, "<system_info>")
	assert.Contains(t, systemInfo, "OS:")
	assert.Contains(t, systemInfo, "Arch:")
}

func TestREPL_HistoryPredictionWithoutLLM(t *testing.T) {
	// This test verifies that history-based prediction works even when
	// no LLM prediction model is configured. The predictor now uses lazy
	// model resolution via SDKModelRef, so it's always created but will
	// return empty predictions if gsh.models.lite is not set.
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	logger := zaptest.NewLogger(t)

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      logger,
	})
	require.NoError(t, err)
	defer repl.Close()

	// Predictor is now always created with lazy model resolution (SDKModelRef)
	// It will return empty LLM predictions if gsh.models.lite is not configured
	assert.NotNil(t, repl.predictor, "predictor uses lazy model resolution")

	// Verify history is available
	require.NotNil(t, repl.history, "history manager should be initialized")

	ctx := context.Background()

	// Execute some commands to populate history
	err = repl.processCommand(ctx, "echo hello world")
	require.NoError(t, err)
	err = repl.processCommand(ctx, "echo testing prediction")
	require.NoError(t, err)

	// Verify commands are in history
	entries, err := repl.history.GetRecentEntriesByPrefix("echo", 10)
	require.NoError(t, err)
	require.GreaterOrEqual(t, len(entries), 2, "should have at least 2 history entries with 'echo' prefix")

	// Create a history provider from the history manager
	historyProvider := input.NewHistoryPredictionAdapter(repl.history)
	require.NotNil(t, historyProvider)

	// Create prediction state with history but WITHOUT LLM provider
	// This is the key test - it should not panic when llmProvider is nil
	predictionState := input.NewPredictionState(input.PredictionStateConfig{
		HistoryProvider: historyProvider,
		LLMProvider:     nil, // Explicitly nil - no LLM configured
		Logger:          logger,
	})
	require.NotNil(t, predictionState)

	// Trigger a prediction by simulating input change
	resultCh := predictionState.OnInputChanged("echo")
	require.NotNil(t, resultCh, "should return a result channel for prediction")

	// Wait for the prediction result (with timeout)
	select {
	case result := <-resultCh:
		// Should get a history-based prediction without error or panic
		assert.NoError(t, result.Error, "prediction should not return an error")
		assert.Equal(t, input.PredictionSourceHistory, result.Source, "prediction should come from history")
		assert.Contains(t, result.Prediction, "echo", "prediction should start with the input prefix")
	case <-time.After(1 * time.Second):
		t.Fatal("prediction timed out")
	}
}

// Tests for middleware integration in REPL

func TestREPL_ProcessCommand_NoMiddleware_FallsThroughToShell(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	ctx := context.Background()

	// Without middleware, commands should execute as shell commands
	err = repl.processCommand(ctx, "echo middleware_test")
	assert.NoError(t, err)
	assert.Equal(t, 0, repl.lastExitCode)
}

func TestREPL_ProcessCommand_MiddlewareHandlesInput(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Get interpreter and set up middleware
	interp := repl.executor.Interpreter()

	// Create a middleware that handles input starting with "#"
	code := `
tool testMiddleware(ctx, next) {
	if (ctx.input.startsWith("#")) {
		return { handled: true }
	}
	return next(ctx)
}
`
	_, err = interp.EvalString(code, nil)
	require.NoError(t, err)

	vars := interp.GetVariables()
	toolVal, ok := vars["testMiddleware"]
	require.True(t, ok)
	tool, ok := toolVal.(*interpreter.ToolValue)
	require.True(t, ok)

	// Set up REPL context with middleware manager
	replCtx := interp.SDKConfig().GetREPLContext()
	require.NotNil(t, replCtx)
	replCtx.MiddlewareManager = interpreter.NewMiddlewareManager()
	replCtx.MiddlewareManager.Use(tool, interp)

	ctx := context.Background()

	// Input starting with # should be handled by middleware (not executed as shell)
	err = repl.processCommand(ctx, "# this is a test")
	assert.NoError(t, err)
	// lastExitCode should be unchanged (default 0) since no shell command was executed
	// The key test is that no error occurred and middleware handled it

	// Verify that the command was still recorded in history even though middleware handled it
	entries, err := repl.History().GetRecentEntries("", 10)
	require.NoError(t, err)
	require.Len(t, entries, 1, "middleware-handled commands should be recorded in history")
	assert.Equal(t, "# this is a test", entries[0].Command)
	assert.True(t, entries[0].ExitCode.Valid)
	assert.Equal(t, int32(0), entries[0].ExitCode.Int32)
}

func TestREPL_ProcessCommand_MiddlewarePassesThrough(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Get interpreter and set up middleware
	interp := repl.executor.Interpreter()

	// Create a middleware that passes everything through
	code := `
tool passThroughMiddleware(ctx, next) {
	return next(ctx)
}
`
	_, err = interp.EvalString(code, nil)
	require.NoError(t, err)

	vars := interp.GetVariables()
	toolVal, ok := vars["passThroughMiddleware"]
	require.True(t, ok)
	tool, ok := toolVal.(*interpreter.ToolValue)
	require.True(t, ok)

	// Set up REPL context with middleware manager
	replCtx := interp.SDKConfig().GetREPLContext()
	require.NotNil(t, replCtx)
	replCtx.MiddlewareManager = interpreter.NewMiddlewareManager()
	replCtx.MiddlewareManager.Use(tool, interp)

	ctx := context.Background()

	// Command should pass through middleware and execute as shell command
	err = repl.processCommand(ctx, "echo pass_through_test")
	assert.NoError(t, err)
	assert.Equal(t, 0, repl.lastExitCode)
}

func TestREPL_ProcessCommand_MiddlewareModifiesInput(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Get interpreter and set up middleware
	interp := repl.executor.Interpreter()

	// Create a middleware that transforms "!" prefix to "echo"
	code := `
tool transformMiddleware(ctx, next) {
	if (ctx.input.startsWith("!")) {
		ctx.input = "echo " + ctx.input.substring(1)
	}
	return next(ctx)
}
`
	_, err = interp.EvalString(code, nil)
	require.NoError(t, err)

	vars := interp.GetVariables()
	toolVal, ok := vars["transformMiddleware"]
	require.True(t, ok)
	tool, ok := toolVal.(*interpreter.ToolValue)
	require.True(t, ok)

	// Set up REPL context with middleware manager
	replCtx := interp.SDKConfig().GetREPLContext()
	require.NotNil(t, replCtx)
	replCtx.MiddlewareManager = interpreter.NewMiddlewareManager()
	replCtx.MiddlewareManager.Use(tool, interp)

	ctx := context.Background()

	// "!hello" should be transformed to "echo hello" and executed
	err = repl.processCommand(ctx, "!hello")
	assert.NoError(t, err)
	assert.Equal(t, 0, repl.lastExitCode)
}

func TestREPL_ProcessCommand_MiddlewareChainOrder(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Get interpreter and set up middleware
	interp := repl.executor.Interpreter()

	// Create two middleware that append markers to verify order
	code := `
__middlewareOrder = ""

tool firstMiddleware(ctx, next) {
	__middlewareOrder = __middlewareOrder + "first,"
	return next(ctx)
}

tool secondMiddleware(ctx, next) {
	__middlewareOrder = __middlewareOrder + "second,"
	return next(ctx)
}
`
	_, err = interp.EvalString(code, nil)
	require.NoError(t, err)

	vars := interp.GetVariables()
	firstVal, _ := vars["firstMiddleware"]
	first, _ := firstVal.(*interpreter.ToolValue)
	secondVal, _ := vars["secondMiddleware"]
	second, _ := secondVal.(*interpreter.ToolValue)

	// Set up REPL context with middleware manager
	replCtx := interp.SDKConfig().GetREPLContext()
	require.NotNil(t, replCtx)
	replCtx.MiddlewareManager = interpreter.NewMiddlewareManager()
	replCtx.MiddlewareManager.Use(first, interp)
	replCtx.MiddlewareManager.Use(second, interp)

	ctx := context.Background()

	// Execute a command - both middleware should run in registration order
	err = repl.processCommand(ctx, "echo order_test")
	assert.NoError(t, err)

	// Verify order: first registered = first to run
	vars = interp.GetVariables()
	orderVal, ok := vars["__middlewareOrder"]
	require.True(t, ok)
	orderStr, ok := orderVal.(*interpreter.StringValue)
	require.True(t, ok)
	assert.Equal(t, "first,second,", orderStr.Value)
}

func TestREPL_ProcessCommand_BuiltinExitStillWorks(t *testing.T) {
	tmpDir := t.TempDir()
	historyPath := filepath.Join(tmpDir, "history.db")
	configPath := filepath.Join(tmpDir, "nonexistent.repl.gsh")

	repl, err := NewREPL(Options{
		ConfigPath:  configPath,
		HistoryPath: historyPath,
		Logger:      zap.NewNop(),
	})
	require.NoError(t, err)
	defer repl.Close()

	// Get interpreter and set up middleware that passes through
	interp := repl.executor.Interpreter()

	code := `
tool passThroughMiddleware(ctx, next) {
	return next(ctx)
}
`
	_, err = interp.EvalString(code, nil)
	require.NoError(t, err)

	vars := interp.GetVariables()
	toolVal, _ := vars["passThroughMiddleware"]
	tool, _ := toolVal.(*interpreter.ToolValue)

	// Set up REPL context with middleware manager
	replCtx := interp.SDKConfig().GetREPLContext()
	require.NotNil(t, replCtx)
	replCtx.MiddlewareManager = interpreter.NewMiddlewareManager()
	replCtx.MiddlewareManager.Use(tool, interp)

	ctx := context.Background()

	// "exit" should still work as a built-in command after middleware passes through
	err = repl.processCommand(ctx, "exit")
	assert.Equal(t, ErrExit, err)
}
