!-----------------------------------------------------------------------
! Support file for Table-Of-Contents in CLASS.
! Used by LIST /TOC and MRTCAL engines.
!-----------------------------------------------------------------------
!
module class_toc_parameters
  !---------------------------------------------------------------------
  ! Support module for command LAS\LIST /TOC
  !---------------------------------------------------------------------
  !
  ! Key codes (note: these codes can not be changed as they are written
  ! in the Mrtcal Data Format).
  integer(kind=4), parameter :: class_toc_key_entry     = 1
  integer(kind=4), parameter :: class_toc_key_bloc      = 2
  integer(kind=4), parameter :: class_toc_key_number    = 3
  integer(kind=4), parameter :: class_toc_key_version   = 4
  integer(kind=4), parameter :: class_toc_key_kind      = 5
  integer(kind=4), parameter :: class_toc_key_scan      = 6
  integer(kind=4), parameter :: class_toc_key_subscan   = 7
  integer(kind=4), parameter :: class_toc_key_observed  = 8
  integer(kind=4), parameter :: class_toc_key_source    = 9
  integer(kind=4), parameter :: class_toc_key_line      = 10
  integer(kind=4), parameter :: class_toc_key_telescope = 11
  integer(kind=4), parameter :: class_toc_key_offset1   = 12
  integer(kind=4), parameter :: class_toc_key_offset2   = 13
  integer(kind=4), parameter :: class_toc_key_quality   = 14
  integer(kind=4), parameter :: class_toc_key_drift     = 15
  integer(kind=4), parameter :: class_toc_key_direction = 16
  integer(kind=4), parameter :: class_toc_key_fesb      = 17
  integer(kind=4), parameter :: class_toc_key_frontend  = 18
  !
  ! Class types, i.e. for fields which needs special decoding (e.g. gag_date:
  ! we won't display the I*4 value, but its translation as a string)
  integer(kind=4), parameter :: class_ptype_null=0     ! No decoding, use value as is
  integer(kind=4), parameter :: class_ptype_gagdate=1  !
  integer(kind=4), parameter :: class_ptype_kind=2     ! Spectro, continuum, etc
  integer(kind=4), parameter :: class_ptype_angle=3    ! Radians printed with current SET ANGLE
  integer(kind=4), parameter :: class_ptype_deg=4      ! Radians printed in degrees
  integer(kind=4), parameter :: class_ptype_fesb=5     ! Telescope name but only frontend+sideband
  integer(kind=4), parameter :: class_ptype_frontend=6 ! Telescope name but only frontend
end module class_toc_parameters
!
module class_sort_var
  use toc_types
  !---------------------------------------------------------------------
  ! Support module for command LAS\LIST /TOC
  !---------------------------------------------------------------------
  !
  type(toc_t), target, save :: ltoc
  !
end module class_sort_var
!
subroutine class_list_toc_comm(set,line,idx,error)
  use gildas_def
  use classcore_dependencies_interfaces
  use classcore_interfaces, except_this=>class_list_toc_comm
  use class_types
  use class_toc_parameters
  use class_sort_var
  !---------------------------------------------------------------------
  ! @ private
  ! CLASS support routine for command
  !   LIST [IN|CURRENT] /TOC [Key1] ... [KeyN]  [/VARIABLE VarName]
  ! Main entry point
  !---------------------------------------------------------------------
  type(class_setup_t), intent(in)    :: set    !
  character(len=*),    intent(in)    :: line   ! Input command line
  type(optimize),      intent(in)    :: idx    !
  logical,             intent(inout) :: error  ! Logical error flag
  ! Local
  integer(kind=4), parameter :: opttoc=5  ! /TOC
  integer(kind=4), parameter :: optvar=6  ! /VARIABLE
  integer(kind=4) :: toc_keys(20)  ! Arbitrary number of args accepted in option /TOC
  character(len=varname_length) :: tocname
  integer(kind=4) :: nc
  !
  call class_toc_init(ltoc,error)
  if (error)  return
  !
  ! Default (unchanged in no arguments)
  toc_keys(:) = toc_key_null
  toc_keys(1) = class_toc_key_source
  toc_keys(2) = class_toc_key_line
  toc_keys(3) = class_toc_key_telescope
  call toc_getkeys(line,opttoc,ltoc,toc_keys,error)
  if (error)  return
  !
  tocname = 'TOC'  ! Default structure name
  call sic_ch(line,optvar,1,tocname,nc,.false.,error)
  if (error)  return
  !
  call class_list_toc(set,idx,toc_keys,tocname,error)
  if (error)  return
  !
end subroutine class_list_toc_comm
!
subroutine class_toc_init(toc,error)
  use gkernel_interfaces
  use classcore_dependencies_interfaces
  use toc_types
  use classcore_interfaces, except_this=>class_toc_init
  use class_toc_parameters
  !---------------------------------------------------------------------
  ! @ private
  ! CLASS Support routine for command LIST /TOC
  ! Initialization routine
  !---------------------------------------------------------------------
  type(toc_t), intent(inout) :: toc    !
  logical,     intent(inout) :: error  !
  ! Local
  character(len=*), parameter :: rname='TOC/INIT'
  integer(kind=4) :: ier
  !
  if (toc%initialized)  return
  !
  toc%nkey = 18
  allocate(toc%keys(toc%nkey),stat=ier)
  if (failed_allocate(rname,'keys array',ier,error)) return
  !
  ! Order below only affects the display to user (LIST /TOC ?)
  !
  toc%keys(1)%id           = class_toc_key_source
  toc%keys(1)%keyword      = 'SOUR'
  toc%keys(1)%sic_var_name = 'sour'
  toc%keys(1)%human_name   = 'SOURCE'
  toc%keys(1)%message      = 'Number of sources......'
  toc%keys(1)%ftype        = toc_ftype_c12_1d
  toc%keys(1)%ptype        = class_ptype_null
  !
  toc%keys(2)%id           = class_toc_key_line
  toc%keys(2)%keyword      = 'LINE'
  toc%keys(2)%sic_var_name = 'line'
  toc%keys(2)%human_name   = 'LINE'
  toc%keys(2)%message      = 'Number of lines........'
  toc%keys(2)%ftype        = toc_ftype_c12_1d
  toc%keys(2)%ptype        = class_ptype_null
  !
  toc%keys(3)%id           = class_toc_key_telescope
  toc%keys(3)%keyword      = 'TELE'
  toc%keys(3)%sic_var_name = 'tele'
  toc%keys(3)%human_name   = 'TELESCOPE'
  toc%keys(3)%message      = 'Number of backends.....'
  toc%keys(3)%ftype        = toc_ftype_c12_1d
  toc%keys(3)%ptype        = class_ptype_null
  !
  toc%keys(4)%id           = class_toc_key_offset1
  toc%keys(4)%keyword      = 'OFF1'
  toc%keys(4)%sic_var_name = 'off1'
  toc%keys(4)%human_name   = 'OFF1'
  toc%keys(4)%message      = 'Number of off1.........'
  toc%keys(4)%ftype        = toc_ftype_r4_1d
  toc%keys(4)%ptype        = class_ptype_angle
  !
  toc%keys(5)%id           = class_toc_key_offset2
  toc%keys(5)%keyword      = 'OFF2'
  toc%keys(5)%sic_var_name = 'off2'
  toc%keys(5)%human_name   = 'OFF2'
  toc%keys(5)%message      = 'Number of off2.........'
  toc%keys(5)%ftype        = toc_ftype_r4_1d
  toc%keys(5)%ptype        = class_ptype_angle
  !
  toc%keys(6)%id           = class_toc_key_entry
  toc%keys(6)%keyword      = 'ENT'
  toc%keys(6)%sic_var_name = 'ent'
  toc%keys(6)%human_name   = 'ENTRY'
  toc%keys(6)%message      = 'Number of entries......'
  toc%keys(6)%ftype        = toc_ftype_i8_1d
  toc%keys(6)%ptype        = class_ptype_null
  !
  toc%keys(7)%id           = class_toc_key_number
  toc%keys(7)%keyword      = 'NUM'
  toc%keys(7)%sic_var_name = 'num'
  toc%keys(7)%human_name   = 'NUMBER'
  toc%keys(7)%message      = 'Number of numbers......'
  toc%keys(7)%ftype        = toc_ftype_i8_1d
  toc%keys(7)%ptype        = class_ptype_null
  !
  toc%keys(8)%id           = class_toc_key_bloc
  toc%keys(8)%keyword      = 'BLOC'
  toc%keys(8)%sic_var_name = 'bloc'
  toc%keys(8)%human_name   = 'BLOCK'
  toc%keys(8)%message      = 'Number of blocks.......'
  toc%keys(8)%ftype        = toc_ftype_i8_1d
  toc%keys(8)%ptype        = class_ptype_null
  !
  toc%keys(9)%id           = class_toc_key_version
  toc%keys(9)%keyword      = 'VER'
  toc%keys(9)%sic_var_name = 'ver'
  toc%keys(9)%human_name   = 'VERSION'
  toc%keys(9)%message      = 'Number of versions.....'
  toc%keys(9)%ftype        = toc_ftype_i4_1d
  toc%keys(9)%ptype        = class_ptype_null
  !
  toc%keys(10)%id           = class_toc_key_kind
  toc%keys(10)%keyword      = 'KIND'
  toc%keys(10)%sic_var_name = 'kind'
  toc%keys(10)%human_name   = 'KIND'
  toc%keys(10)%message      = 'Number of kinds.......'
  toc%keys(10)%ftype        = toc_ftype_i4_1d
  toc%keys(10)%ptype        = class_ptype_kind
  !
  toc%keys(11)%id           = class_toc_key_quality
  toc%keys(11)%keyword      = 'QUAL'
  toc%keys(11)%sic_var_name = 'qual'
  toc%keys(11)%human_name   = 'QUALITY'
  toc%keys(11)%message      = 'Number of qualities...'
  toc%keys(11)%ftype        = toc_ftype_i4_1d
  toc%keys(11)%ptype        = class_ptype_null
  !
  toc%keys(12)%id           = class_toc_key_observed
  toc%keys(12)%keyword      = 'OBS'
  toc%keys(12)%sic_var_name = 'obs'
  toc%keys(12)%human_name   = 'OBSERVED' ! Observed date
  toc%keys(12)%message      = 'Number of observation dates'
  toc%keys(12)%ftype        = toc_ftype_i4_1d
  toc%keys(12)%ptype        = class_ptype_gagdate
  !
  toc%keys(13)%id           = class_toc_key_scan
  toc%keys(13)%keyword      = 'SCAN'
  toc%keys(13)%sic_var_name = 'scan'
  toc%keys(13)%human_name   = 'SCAN'
  toc%keys(13)%message      = 'Number of scans.......'
  toc%keys(13)%ftype        = toc_ftype_i8_1d
  toc%keys(13)%ptype        = class_ptype_null
  !
  toc%keys(14)%id           = class_toc_key_subscan
  toc%keys(14)%keyword      = 'SUBSCAN'
  toc%keys(14)%sic_var_name = 'subscan'
  toc%keys(14)%human_name   = 'SUBSCAN'
  toc%keys(14)%message      = 'Number of subscans....'
  toc%keys(14)%ftype        = toc_ftype_i4_1d
  toc%keys(14)%ptype        = class_ptype_null
  !
  ! Below this point are properties which are more technical (used by
  ! MRTCAL) and not exposed to users (need better names if needed)
  !
  ! 0 and 180 drift angles are equal, -90 and +90 are equal
  toc%keys(15)%id           = class_toc_key_drift
  toc%keys(15)%keyword      = 'DRIFT'
  toc%keys(15)%sic_var_name = 'drift'
  toc%keys(15)%human_name   = 'DRIFT'
  toc%keys(15)%message      = 'Number of drifts.......'
  toc%keys(15)%ftype        = toc_ftype_r4_1d
  toc%keys(15)%ptype        = class_ptype_deg  ! Should provide a function which shows 180 as 0, and -90 as 90
  toc%keys(15)%eq%r4       => class_toc_eq_drift
  !
  ! All drift angles are unique
  toc%keys(16)%id           = class_toc_key_direction
  toc%keys(16)%keyword      = 'DIRECTION'
  toc%keys(16)%sic_var_name = 'direction'
  toc%keys(16)%human_name   = 'DIRECTION'
  toc%keys(16)%message      = 'Number of directions...'
  toc%keys(16)%ftype        = toc_ftype_r4_1d
  toc%keys(16)%ptype        = class_ptype_deg
  toc%keys(16)%eq%r4       => class_toc_eq_direction
  !
  ! Equality for frontend+sideband
  toc%keys(17)%id           = class_toc_key_fesb
  toc%keys(17)%keyword      = 'FESB'
  toc%keys(17)%sic_var_name = 'fesb'
  toc%keys(17)%human_name   = 'FESB'
  toc%keys(17)%message      = 'Number of frontend-sb..'
  toc%keys(17)%ftype        = toc_ftype_c12_1d
  toc%keys(17)%ptype        = class_ptype_fesb
  toc%keys(17)%eq%c12      => class_toc_eq_fesb
  !
  ! Equality of frontends
  toc%keys(18)%id           = class_toc_key_frontend
  toc%keys(18)%keyword      = 'FRONTEND'
  toc%keys(18)%sic_var_name = 'frontend'
  toc%keys(18)%human_name   = 'FRONTEND'
  toc%keys(18)%message      = 'Number of frontend.....'
  toc%keys(18)%ftype        = toc_ftype_c12_1d
  toc%keys(18)%ptype        = class_ptype_frontend
  toc%keys(18)%eq%c12      => class_toc_eq_frontend
  !
  call toc_init_pointers(toc,error)
  if (error)  return
  !
  toc%initialized = .true.
end subroutine class_toc_init
!
function class_toc_eq_drift(ptr,l,m)
  use phys_const
  use gkernel_interfaces
  use classic_params
  use toc_types
  !-------------------------------------------------------------------
  ! @ private
  ! Compute the "equality" of drifts.
  ! The "drift value" is not directly available in the index. It is an
  ! information derived from r%head%dri%apos (the main "detail" is the
  ! modulo pi). Hence the specific equality routine.
  !-------------------------------------------------------------------
  logical :: class_toc_eq_drift
  class(pointer_typer4_t),    intent(in) :: ptr
  integer(kind=entry_length), intent(in) :: m,l
  ! Beware values can be e.g. 0+-epsilon, but the epsilon makes it
  ! difficult to modulo the value to a controled range.
  class_toc_eq_drift = nearly_equal(ptr%data1(m),ptr%data1(l),    1e-3) .or.  &
                       nearly_equal(ptr%data1(m),ptr%data1(l)-pis,1e-3) .or.  &
                       nearly_equal(ptr%data1(m),ptr%data1(l)+pis,1e-3)
end function class_toc_eq_drift
!
function class_toc_eq_direction(ptr,l,m)
  use phys_const
  use gkernel_interfaces
  use classic_params
  use toc_types
  !-------------------------------------------------------------------
  ! @ private
  ! Compute the "equality" of drift directions.
  ! Need some machine error, hence the specific equality routine.
  !-------------------------------------------------------------------
  logical :: class_toc_eq_direction
  class(pointer_typer4_t),    intent(in) :: ptr
  integer(kind=entry_length), intent(in) :: m,l
  ! Beware values can be e.g. 0+-epsilon, but the epsilon makes it
  ! difficult to modulo the value to a controled range.
  class_toc_eq_direction = nearly_equal(ptr%data1(m),ptr%data1(l),      1e-3) .or.  &
                           nearly_equal(ptr%data1(m),ptr%data1(l)-2*pis,1e-3) .or.  &
                           nearly_equal(ptr%data1(m),ptr%data1(l)+2*pis,1e-3)
end function class_toc_eq_direction
!
function class_toc_eq_fesb(ptr,l,m)
  use classic_params
  use toc_types
  !-------------------------------------------------------------------
  ! @ private
  ! Compute the "equality" of frontend-sideband.
  ! The "frontend-backend" value is not directly available in the
  ! index. It is an information derived from r%head%gen%teles. Hence
  ! the specific equality routine.
  !    30ME0HLS-B01 and
  !    30ME0VLS-B03 are equal (same frontend, same sideband, not the
  ! same polarizarion), and
  !    30ME0HUS-B02 and
  !    30ME0VUS-B04 are a different class.
  !-------------------------------------------------------------------
  logical :: class_toc_eq_fesb
  class(pointer_typec12_t),   intent(in) :: ptr
  integer(kind=entry_length), intent(in) :: m,l
  class_toc_eq_fesb = ptr%data1(m)(1:5).eq.ptr%data1(l)(1:5) .and.  &
                      ptr%data1(m)(7:8).eq.ptr%data1(l)(7:8)
end function class_toc_eq_fesb
!
function class_toc_eq_frontend(ptr,l,m)
  use classic_params
  use toc_types
  !-------------------------------------------------------------------
  ! @ private
  ! Compute the "equality" of frontends.
  ! The "frontend" value is not directly available in the index. It is
  ! an information derived from r%head%gen%teles. Hence the specific
  ! equality routine.
  !    30ME0HLS-B01 and
  !    30ME0HUS-B02 and
  !    30ME0VLS-B03 and
  !    30ME0VUS-B04 are all equal.
  !-------------------------------------------------------------------
  logical :: class_toc_eq_frontend
  class(pointer_typec12_t),   intent(in) :: ptr
  integer(kind=entry_length), intent(in) :: m,l
  class_toc_eq_frontend = ptr%data1(m)(1:5).eq.ptr%data1(l)(1:5)
end function class_toc_eq_frontend
!
function class_toc_drift_modulo(keys)
  use phys_const
  use class_toc_parameters
  !---------------------------------------------------------------------
  ! @ public for MRTCAL
  ! Ad-hoc function for MRTCAL. Given a list of TOC keys, return the
  ! modulo on the drift position angles which was used. In short, this
  ! tells how class_toc_eq_drift and class_toc_eq_direction behave.
  !---------------------------------------------------------------------
  real(kind=4) :: class_toc_drift_modulo  ! [rad] Function value on return
  integer(kind=4), intent(in) :: keys(:)  ! TOC key identifiers
  !
  integer(kind=4) :: ikey
  !
  ! If no drift-angle related keyword is found, this means that an
  ! angle might have been mixed with any other one. In other words
  ! this is "modulo 0"
  class_toc_drift_modulo = 0.
  !
  do ikey=1,size(keys)
    select case (keys(ikey))  ! Compare with class_toc_init keywords
    case (class_toc_key_drift)
      class_toc_drift_modulo = pi
    case (class_toc_key_direction)
      class_toc_drift_modulo = twopi
    end select
  enddo
end function class_toc_drift_modulo
!
subroutine class_toc_clean(error)
  use classcore_dependencies_interfaces
  use class_sort_var
  !---------------------------------------------------------------------
  ! @ public (for libclass only)
  ! CLASS Support routine for command LIST /TOC
  ! Global the global variable holding the TOC
  !---------------------------------------------------------------------
  logical, intent(inout) :: error  !
  !
  call toc_clean(ltoc,error)
  ! if (error)  continue
  !
end subroutine class_toc_clean
!
subroutine class_list_toc(set,idx,keywords,tocname,error)
  use gbl_message
  use classcore_dependencies_interfaces
  use classcore_interfaces, except_this=>class_list_toc
  use class_types
  use class_sort_var
  use output_header
  !---------------------------------------------------------------------
  ! @ private
  ! CLASS Support routine for command LIST /TOC
  ! Processing routine
  !---------------------------------------------------------------------
  type(class_setup_t), intent(in)    :: set          !
  type(optimize),      intent(in)    :: idx          ! The idx whose TOC is desired
  integer(kind=4),     intent(in)    :: keywords(:)  ! Selection (key identifiers)
  character(len=*),    intent(in)    :: tocname      ! Structure name
  logical,             intent(inout) :: error        ! Logical error flag
  ! Local
  character(len=*), parameter :: rname='LIST/TOC'
  !
  call class_toc_datasetup(ltoc,idx)
  !
  call toc_main(rname,ltoc,idx%next-1,keywords,tocname,p_lun,class_toc_format,error)
  if (error)  return
  !
contains
  !
  subroutine class_toc_format(key,ival,output)
    use phys_const
    use classcore_dependencies_interfaces
    use class_setup_new
    use toc_types
    use class_toc_parameters
    !---------------------------------------------------------------------
    ! Custom display routine (because there are custom types)
    !---------------------------------------------------------------------
    type(toc_descriptor_t),     intent(in)  :: key     !
    integer(kind=entry_length), intent(in)  :: ival    !
    character(len=*),           intent(out) :: output  !
    ! Local
    logical :: error
    !
    select case(key%ftype)
    case(toc_ftype_i4_1d) ! integer*4
      if (key%ptype.eq.class_ptype_gagdate) then
        call gag_todate(key%ptr%i4%data1(ival),output,error)
      elseif (key%ptype.eq.class_ptype_kind) then
        call class_toc_format_kind(key%ptr%i4%data1(ival),output)
      else
        write(output,'(i12)') key%ptr%i4%data1(ival)
      endif
      !
    case(toc_ftype_i8_1d)  ! integer*8
      write(output,'(i12)') key%ptr%i8%data1(ival)
      !
    case(toc_ftype_r4_1d)  ! real*4
      if (key%ptype.eq.class_ptype_angle) then
        write(output,'(f8.3)') key%ptr%r4%data1(ival) * class_setup_get_fangle()
      elseif (key%ptype.eq.class_ptype_deg) then
        write(output,'(f8.1)') key%ptr%r4%data1(ival) * deg_per_rad
      else
        write(output,'(f8.3)') key%ptr%r4%data1(ival)
      endif
      !
    case(toc_ftype_c12_1d) ! string*12
      if (key%ptype.eq.class_ptype_fesb) then
        output = key%ptr%c12%data1(ival)
        output(6:6) = '-'
        output(9:12) = '----'
      elseif (key%ptype.eq.class_ptype_frontend) then
        output = key%ptr%c12%data1(ival)
        output(6:12) = '-------'
      else
        output = key%ptr%c12%data1(ival)
      endif
      !
    end select
  end subroutine class_toc_format
  !
  subroutine class_toc_format_kind(kind,string)
    use gbl_constant
    !-------------------------------------------------------------------
    ! Format the 'kind' value (i4 value to string)
    !-------------------------------------------------------------------
    integer(kind=4),  intent(in)  :: kind
    character(len=*), intent(out) :: string
    select case(kind)
    case (kind_spec)
      string = 'SPECTRUM'
    case (kind_cont)
      string = 'CONTINUUM'
    case default
      write(string,'(A,I0)') 'CODE ',kind
    end select
  end subroutine class_toc_format_kind
  !
end subroutine class_list_toc
!
subroutine class_toc_datasetup(toc,idx)
  use class_types
  use toc_types
  !---------------------------------------------------------------------
  ! @ private
  ! Associate the TOC data pointers to the index whose TOC is desired
  !---------------------------------------------------------------------
  type(toc_t),    intent(inout) :: toc
  type(optimize), intent(in)    :: idx
  !
  toc%keys( 1)%ptr%c12%data1 => idx%csour
  toc%keys( 2)%ptr%c12%data1 => idx%cline
  toc%keys( 3)%ptr%c12%data1 => idx%ctele
  toc%keys( 4)%ptr%r4%data1  => idx%off1
  toc%keys( 5)%ptr%r4%data1  => idx%off2
  toc%keys( 6)%ptr%i8%data1  => idx%ind
  toc%keys( 7)%ptr%i8%data1  => idx%num
  toc%keys( 8)%ptr%i8%data1  => idx%bloc
  toc%keys( 9)%ptr%i4%data1  => idx%ver
  toc%keys(10)%ptr%i4%data1  => idx%kind
  toc%keys(11)%ptr%i4%data1  => idx%qual
  toc%keys(12)%ptr%i4%data1  => idx%dobs
  toc%keys(13)%ptr%i8%data1  => idx%scan
  toc%keys(14)%ptr%i4%data1  => idx%subscan
  ! Special ones. Note that we could have produced local copies with
  ! directly with the relevant values (which allows direct comparison
  ! afterwards). But this means computation cost for a marginal use
  ! case. Use instead dedicated comparison and printing routines.
  toc%keys(15)%ptr%r4%data1  => idx%posa
  toc%keys(16)%ptr%r4%data1  => idx%posa
  toc%keys(17)%ptr%c12%data1 => idx%ctele
  toc%keys(18)%ptr%c12%data1 => idx%ctele
end subroutine class_toc_datasetup
