/* #includes */ /*{{{C}}}*//*{{{*/
#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "getopt_.h"
#include "cpmfs.h"
/*}}}*/
/* #defines */ /*{{{*/
#ifndef O_BINARY
#define O_BINARY 0
#endif
/*}}}*/

/* mkfs -- make file system */ /*{{{*/
static int mkfs(struct cpmSuperBlock *drive, const char *devopts, const char *name, const char *format, const char *label, char *bootTracks, int timeStamps, int uppercase)
{
  /* variables */ /*{{{*/
  unsigned int i;
  char buf[128];
  char firstbuf[128];
  char *secbuf;
  unsigned int bytes;
  unsigned int trkbytes;
  int track, sector;
  int recs;
  /*}}}*/

  /* open image file */ /*{{{*/
  if ((boo=Device_open(&drive->dev, name, O_BINARY|O_CREAT|O_WRONLY, devopts)))
  {
    return -1;
  }
  track = 0;
  sector = 0;
  /*}}}*/
  /* write system tracks */ /*{{{*/
  /* this initialises only whole tracks, so it skew is not an issue */
  trkbytes=drive->secLength*drive->sectrk;
  for (i=0; i<trkbytes*drive->boottrk; i+=drive->secLength)
  {
    if ((boo=Device_writeSector(&drive->dev, track, sector, bootTracks+i)))
    {
      Device_close(&drive->dev);
      return -1;
    }
    ++sector;
    if (sector==drive->sectrk)
    {
      sector=0;
      ++track;
    }
  }
  /*}}}*/
  /* write directory */ /*{{{*/
  memset(buf,0xe5,128);
  bytes=drive->maxdir*32;
  /* write a full sector */
  if (bytes%drive->secLength) bytes=((bytes+drive->secLength)/drive->secLength)*drive->secLength;
  if (timeStamps && (drive->type==CPMFS_P2DOS || drive->type==CPMFS_DR3)) buf[3*32]=0x21;
  memcpy(firstbuf,buf,128);
  if (drive->type==CPMFS_DR3)
  {
    time_t now;
    struct tm *t;
    int min,hour,days;

    firstbuf[0]=0x20;
    for (i=0; i<11 && *label; ++i,++label) firstbuf[1+i]=toupper(*label&0x7f);
    while (i<11) firstbuf[1+i++]=' ';
    firstbuf[12]=timeStamps ? 0x11 : 0x01; /* label set and first time stamp is creation date */
    memset(&firstbuf[13],0,1+2+8);
    if (timeStamps)
    {
      int year;

      /* Stamp label. */
      time(&now);
      t=localtime(&now);
      min=((t->tm_min/10)<<4)|(t->tm_min%10);
      hour=((t->tm_hour/10)<<4)|(t->tm_hour%10);
      for (year=1978,days=0; year<1900+t->tm_year; ++year)
      {
        days+=365;
        if (year%4==0 && (year%100!=0 || year%400==0)) ++days;
      }
      days += t->tm_yday + 1;
      firstbuf[24]=firstbuf[28]=days&0xff; firstbuf[25]=firstbuf[29]=days>>8;
      firstbuf[26]=firstbuf[30]=hour;
      firstbuf[27]=firstbuf[31]=min;
    }
  }
  secbuf=malloc(drive->secLength);
  recs=drive->secLength/128;
  for (i=0; i<bytes; i+=drive->secLength)
  {
    int r;

    for (r=0; r<recs; ++r) memcpy(secbuf+r*128, i==0 && r==0 ? firstbuf : buf, 128);
    if ((boo=Device_writeSector(&drive->dev, track, drive->skewtab[sector], secbuf)))
    {
      Device_close(&drive->dev);
      free(secbuf);
      return -1;
    }
    ++sector;
    if (sector==drive->sectrk)
    {
      sector=0;
      ++track;
    }
  }
  free(secbuf);
  /*}}}*/
  /* close image file */ /*{{{*/
  if ((boo=Device_close(&drive->dev)))
  {
    return -1;
  }
  /*}}}*/
  if (timeStamps && !(drive->type==CPMFS_P2DOS || drive->type==CPMFS_DR3)) /*{{{*/
  {
    int offset,j;
    struct cpmInode ino, root;
    static const char sig[] = "!!!TIME";
    unsigned int records;
    struct dsDate *ds;
    struct cpmFile dsfile;
    const char *err;
    struct utimbuf ut;

    Device_close(&drive->dev);
    if ((err=Device_open(&drive->dev,name,O_RDWR,NULL)))
    {
      fprintf(stderr,"%s: can not open %s (%s)\n",cmd,name,err);
      exit(1);
    }
    cpmReadSuper(drive,&root,format,uppercase);

    records=(root.sb->maxdir+7)/8;
    if (!(ds=malloc(records*128)))
    {
      return -1;
    }
    memset(ds,0,records*128);
    offset=15;
    for (i=0; i<records; i++)
    {
      unsigned int cksum;

      for (j=0; j<7; j++,offset+=16)
      {
        *((char*)ds+offset) = sig[j];
      }
      /* skip checksum byte */
      offset+=16;

      for (cksum=0,j=0; j<127; ++j) cksum+=*((unsigned char*)ds+i*128+j);
      *((char*)ds+i*128+j)=cksum;
    }

    /* The filesystem does not know about datestamper yet, because it
     * was not there when it was mounted.
     */

    if (cpmCreat(&root,"00!!!TIME&.DAT",&ino,0222)==-1)
    {
      fprintf(stderr,"%s: Unable to create DateStamper file: %s\n",cmd,boo);
      return -1;
    }

    if (cpmOpen(&ino,&dsfile,O_WRONLY) == -1
        || cpmWrite(&dsfile,(char const*)ds,records*128) != records*128
        || cpmClose(&dsfile) == -1)
    {
      fprintf(stderr,"%s: Unable to write DateStamper file: %s\n",cmd,boo);
      return -1;
    }

    cpmChmod(&ino, 0);

    cpmUmount(drive);
    if ((err=Device_open(&drive->dev,name,O_RDWR,NULL)))
    {
      fprintf(stderr,"%s: can not open %s (%s)\n",cmd,name,err);
      exit(1);
    }
    cpmReadSuper(drive,&root,format,uppercase);

    cpmNamei(&root,"00!!!TIME&.DAT",&ino);
    time(&ut.actime);
    time(&ut.modtime);
    cpmUtime(&ino,&ut);
  }
  /*}}}*/

  return 0;
}
/*}}}*/

const char cmd[]="mkfs.cpm";

int main(int argc, char *argv[]) /*{{{*/
{
  char *image;
  const char *format;
  const char *devopts=NULL;
  int uppercase=0;
  int c,usage=0;
  struct cpmSuperBlock drive;
  struct cpmInode root;
  const char *label="unlabeled";
  int timeStamps=0;
  size_t bootTrackSize,used;
  char *bootTracks;
  const char *boot[4]={(const char*)0,(const char*)0,(const char*)0,(const char*)0};

  if (!(format=getenv("CPMTOOLSFMT"))) format=FORMAT;
  while ((c=getopt(argc,argv,"T:b:f:L:tuh?"))!=EOF) switch(c)
  {
    case 'T': devopts=optarg; break;
    case 'b':
    {
      if (boot[0]==(const char*)0) boot[0]=optarg;
      else if (boot[1]==(const char*)0) boot[1]=optarg;
      else if (boot[2]==(const char*)0) boot[2]=optarg;
      else if (boot[3]==(const char*)0) boot[3]=optarg;
      else usage=1;
      break;
    }
    case 'f': format=optarg; break;
    case 'L': label=optarg; break;
    case 't': timeStamps=1; break;
    case 'u': uppercase=1; break;
    case 'h':
    case '?': usage=1; break;
  }

  if (optind!=(argc-1)) usage=1;
  else image=argv[optind++];

  if (usage)
  {
    fprintf(stderr,"Usage: %s [-f format] [-b boot] [-L label] [-t] [-u] image\n",cmd);
    exit(1);
  }
  drive.dev.opened=0;
  cpmReadSuper(&drive,&root,format,uppercase);
  bootTrackSize=drive.boottrk*drive.secLength*drive.sectrk;
  if (bootTrackSize)
  {
    if ((bootTracks=malloc(bootTrackSize))==(void*)0)
    {
      fprintf(stderr,"%s: can not allocate boot track buffer: %s\n",cmd,strerror(errno));
      exit(1);
    }
    memset(bootTracks,0xe5,bootTrackSize);
    used=0; 
    for (c=0; c<4 && boot[c]; ++c)
    {
      int fd;
      size_t size;

      if ((fd=open(boot[c],O_BINARY|O_RDONLY))==-1)
      {
        fprintf(stderr,"%s: can not open %s: %s\n",cmd,boot[c],strerror(errno));
        exit(1);
      }
      size=read(fd,bootTracks+used,bootTrackSize-used);
#if 0
      fprintf(stderr,"%d %04x %s\n",c,used+0x800,boot[c]);
#endif
      if (size%drive.secLength) size=(size|(drive.secLength-1))+1;
      used+=size;
      close(fd);
    }
  }
  else bootTracks=(char*)0;
  if (mkfs(&drive,devopts,image,format,label,bootTracks,timeStamps,uppercase)==-1)
  {
    fprintf(stderr,"%s: can not make new file system: %s\n",cmd,boo);
    exit(1);
  }
  if (bootTracks) free(bootTracks);
  cpmUmount(&drive);
  exit(0);
}
/*}}}*/
