"""
This script:
1. Generates example files from examples/*.nbt
2. Generates function reference files via cargo run --example=inspect
3. Generates the units list
4. Runs zensical build
"""

import subprocess
from pathlib import Path
import urllib.parse
import os
import sys


SCRIPT_DIR = Path(__file__).parent.resolve()


def generate_example(
    filename, title, strip_asserts=True, insert_run_link=True, footer=None, icon=None
):
    """Generate a markdown file from a Numbat example file."""
    path_in = SCRIPT_DIR.parent / "examples" / f"{filename}.nbt"
    path_out = SCRIPT_DIR / "src" / "examples" / f"example-{filename}.md"
    print(f"Generating: {path_out}")

    code = []
    with open(path_in, "r", encoding="utf-8") as fin:
        for line in fin:
            if not (strip_asserts and "assert_eq" in line):
                code.append(line)

    url = f"https://numbat.dev/?q={urllib.parse.quote_plus(''.join(code))}"

    with open(path_out, "w", encoding="utf-8") as fout:
        if icon:
            fout.write("---\n")
            fout.write(f"icon: {icon}\n")
            fout.write("---\n")
            fout.write("\n")
        fout.write("<!-- This file is autogenerated! Do not modify it -->\n")
        fout.write("\n")
        fout.write(f"# {title}\n")
        if insert_run_link:
            fout.write("\n")
            fout.write(
                f"[:material-play-circle: Run this example]({url}){{ .md-button .md-button--primary }}\n"
            )
        fout.write("\n")
        fout.write("```numbat\n")
        fout.writelines(code)
        fout.write("```\n")

        if footer:
            fout.write("\n")
            fout.write(footer)
            fout.write("\n")


def xkcd_footer(number, img_name):
    """Generate footer HTML for XKCD comics."""
    footer = '<p align="center" style="margin-top: 2em">'
    footer += f'<a href="https://xkcd.com/{number}/"><img src="https://imgs.xkcd.com/comics/{img_name}.png" alt="XKCD {number}" style="max-width: 100%"></a>'
    footer += f'<br>Source: <a href="https://xkcd.com/{number}/">https://xkcd.com/{number}/</a>'
    footer += "</p>"
    return footer


def generate_all_examples():
    """Generate all example documentation files."""
    print("Generating example files...")

    generate_example("acidity", "Acidity")
    generate_example("barometric_formula", "Barometric formula")
    generate_example("body_mass_index", "Body mass index")
    generate_example("factorial", "Factorial", strip_asserts=False)
    generate_example("medication_dosage", "Medication dosage")
    generate_example("molarity", "Molarity")
    generate_example("musical_note_frequency", "Musical note frequency")
    generate_example("paper_size", "Paper sizes")
    generate_example("pipe_flow_rate", "Flow rate in a pipe")
    generate_example("population_growth", "Population growth")
    generate_example("recipe", "Recipe")
    generate_example("voyager", "Voyager")
    generate_example(
        "xkcd_681",
        "XKCD 681",
        footer=xkcd_footer(681, "gravity_wells"),
    )
    generate_example(
        "xkcd_687", "XKCD 687", footer=xkcd_footer(687, "dimensional_analysis")
    )
    generate_example("xkcd_2585", "XKCD 2585", footer=xkcd_footer(2585, "rounding"))
    generate_example(
        "xkcd_2812", "XKCD 2812", footer=xkcd_footer(2812, "solar_panel_placement")
    )
    generate_example(
        "numbat_syntax",
        "Syntax overview",
        strip_asserts=False,
        insert_run_link=False,
        icon="lucide/file-code",
    )


def generate_units_list():
    """Generate the list of units documentation."""
    path_units = SCRIPT_DIR / "src" / "prelude" / "list-units.md"
    print("Generating list of units...")

    with open(path_units, "w", encoding="utf-8") as f:
        f.write("---\n")
        f.write("icon: lucide/ruler\n")
        f.write("---\n")
        f.write("\n")
        f.flush()
        subprocess.run(
            ["cargo", "run", "--release", "--quiet", "--example=inspect", "units"],
            stdout=f,
            text=True,
            cwd=SCRIPT_DIR.parent,
        )


def list_of_functions(file_name, document):
    """Generate function reference documentation for a set of modules."""
    path = SCRIPT_DIR / "src" / "prelude" / "functions" / f"{file_name}.md"

    with open(path, "w", encoding="utf-8") as f:
        if icon := document.get("icon"):
            print("---", file=f)
            print(f"icon: {icon}", file=f)
            print("---\n", file=f, flush=True)

        print(f"# {document['title']}\n", file=f, flush=True)

        if introduction := document.get("introduction"):
            print(introduction + "\n", file=f, flush=True)

        sections = document["sections"]

        if len(sections) >= 3:
            links = []
            for section in sections:
                if title := section.get("title"):
                    link = title.lower().replace(" ", "-").replace(",", "")
                    links.append(f"[{title}](#{link})")
            print(f"{' · '.join(links)}\n", file=f, flush=True)

        for section in sections:
            modules = section["modules"]

            if title := section.get("title"):
                print(f"## {title}\n", file=f, flush=True)

            print(f"Defined in: `{'`, `'.join(modules)}`\n", file=f, flush=True)

            for module in modules:
                print(
                    f"Generating list of functions for module '{module}'...", flush=True
                )
                env = os.environ.copy()
                env["TZ"] = "UTC"

                try:
                    subprocess.run(
                        [
                            "cargo",
                            "run",
                            "--release",
                            "--quiet",
                            "--example=inspect",
                            "--",
                            "functions",
                            module,
                        ],
                        stdout=f,
                        text=True,
                        env=env,
                        check=True,
                        cwd=SCRIPT_DIR.parent,
                    )
                except subprocess.CalledProcessError as e:
                    sys.exit(e.returncode)


def generate_all_function_lists():
    """Generate all function reference documentation."""
    print("Generating function reference files...")

    # Ensure the functions directory exists
    (SCRIPT_DIR / "src" / "prelude" / "functions").mkdir(parents=True, exist_ok=True)

    list_of_functions(
        "math",
        {
            "title": "Mathematical functions",
            "icon": "lucide/sigma",
            "sections": [
                {"title": "Basics", "modules": ["core::functions"]},
                {
                    "title": "Transcendental functions",
                    "modules": ["math::transcendental"],
                },
                {"title": "Trigonometry", "modules": ["math::trigonometry"]},
                {"title": "Statistics", "modules": ["math::statistics"]},
                {"title": "Combinatorics", "modules": ["math::combinatorics"]},
                {
                    "title": "Random sampling, distributions",
                    "modules": ["core::random", "math::distributions"],
                },
                {"title": "Number theory", "modules": ["math::number_theory"]},
                {
                    "title": "Numerical methods",
                    "modules": [
                        "numerics::diff",
                        "numerics::solve",
                        "numerics::fixed_point",
                    ],
                },
                {
                    "title": "Percentage calculations",
                    "modules": ["math::percentage_calculations"],
                },
                {"title": "Geometry", "modules": ["math::geometry"]},
                {"title": "Algebra", "modules": ["extra::algebra"]},
                {
                    "title": "Trigonometry (extra)",
                    "modules": ["math::trigonometry_extra"],
                },
            ],
        },
    )

    list_of_functions(
        "lists",
        {
            "title": "List-related functions",
            "icon": "lucide/brackets",
            "sections": [{"modules": ["core::lists"]}],
        },
    )

    list_of_functions(
        "strings",
        {
            "title": "String-related functions",
            "icon": "lucide/quote",
            "sections": [{"modules": ["core::strings"]}],
        },
    )

    list_of_functions(
        "datetime",
        {
            "title": "Date and time",
            "icon": "lucide/clock",
            "introduction": "See [this page](../../basics/date-and-time.md) for a general introduction to date and time handling in Numbat.",
            "sections": [
                {
                    "modules": [
                        "datetime::functions",
                        "datetime::unixtime",
                        "datetime::human",
                        "datetime::julian_date",
                    ]
                }
            ],
        },
    )

    list_of_functions(
        "other",
        {
            "title": "Other functions",
            "icon": "lucide/ellipsis",
            "sections": [
                {"title": "Error handling", "modules": ["core::error"]},
                {"title": "Debugging", "modules": ["core::debug"]},
                {"title": "Floating point", "modules": ["core::numbers"]},
                {"title": "Quantities", "modules": ["core::quantities"]},
                {"title": "Chemical elements", "modules": ["chemistry::elements"]},
                {"title": "Mixed unit conversion", "modules": ["units::mixed"]},
                {
                    "title": "Temperature conversion",
                    "modules": ["physics::temperature_conversion"],
                },
                {"title": "Speed of sound", "modules": ["physics::speed_of_sound"]},
                {"title": "Color format conversion", "modules": ["extra::color"]},
                {"title": "Celestial calculations", "modules": ["extra::celestial"]},
            ],
        },
    )


def build_site():
    """Run zensical build."""
    print("Building documentation with zensical...")
    result = subprocess.run(
        ["zensical", "build", "--strict"],
        cwd=SCRIPT_DIR,
    )
    return result.returncode


def main():
    """Main entry point."""
    # Change to script directory
    os.chdir(SCRIPT_DIR)

    # Generate all content
    generate_all_examples()
    generate_units_list()
    generate_all_function_lists()

    # Build the site
    return build_site()


if __name__ == "__main__":
    sys.exit(main())
