// Copyright contributors to the openqasm-parser project
// SPDX-License-Identifier: Apache-2.0

//! Defines input for code generation process.

pub(crate) struct KindsSrc<'a> {
    pub(crate) punct: &'a [(&'a str, &'a str)],
    pub(crate) keywords: &'a [&'a str],
    pub(crate) literals: &'a [&'a str],
    pub(crate) scalar_types: &'a [&'a str],
    pub(crate) tokens: &'a [&'a str],
    pub(crate) nodes: &'a [&'a str],
}

pub(crate) const KINDS_SRC: KindsSrc<'_> = KindsSrc {
    punct: &[
        ("!", "BANG"),
        ("!=", "NEQ"),
        ("#", "POUND"),
        ("$", "DOLLAR"),
        ("%", "PERCENT"),
        ("%=", "PERCENTEQ"),
        ("&", "AMP"),
        ("&&", "AMP2"),
        ("&=", "AMPEQ"),
        ("(", "L_PAREN"),
        (")", "R_PAREN"),
        ("*", "STAR"),
        ("*=", "STAREQ"),
        ("+", "PLUS"),
        ("++", "DOUBLE_PLUS"),
        ("+=", "PLUSEQ"),
        (",", "COMMA"),
        ("-", "MINUS"),
        ("-=", "MINUSEQ"),
        ("->", "THIN_ARROW"),
        (".", "DOT"),
        ("..", "DOT2"),
        ("...", "DOT3"),
        ("..=", "DOT2EQ"),
        ("/", "SLASH"),
        ("/=", "SLASHEQ"),
        (":", "COLON"),
        ("::", "COLON2"),
        (";", "SEMICOLON"),
        ("<", "L_ANGLE"),
        ("<<", "SHL"),
        ("<<=", "SHLEQ"),
        ("<=", "LTEQ"),
        ("=", "EQ"),
        ("==", "EQ2"),
        ("=>", "FAT_ARROW"),
        (">", "R_ANGLE"),
        (">=", "GTEQ"),
        (">>", "SHR"),
        (">>=", "SHREQ"),
        ("?", "QUESTION"),
        ("@", "AT"),
        ("[", "L_BRACK"),
        ("]", "R_BRACK"),
        ("^", "CARET"),
        ("^=", "CARETEQ"),
        ("_", "UNDERSCORE"),
        ("{", "L_CURLY"),
        ("|", "PIPE"),
        ("|=", "PIPEEQ"),
        ("||", "PIPE2"),
        ("}", "R_CURLY"),
        ("~", "TILDE"),
    ],
    keywords: &[
        "OPENQASM",
        "barrier",
        "box",
        "cal",
        "const",
        "def",
        "defcal",
        "defcalgrammar",
        "delay",
        "extern",
        "gate",
        "gphase", // This is a slight hack because a `gphase` call has unique syntax.
        "include",
        "let",
        "measure",
        "pragma",
        "reset",
        // Flow control
        "break",
        "case",
        "continue",
        "default",
        "else",
        "end",
        "for",
        "if",
        "in",
        "return",
        "switch",
        "while",
        // Types
        "array",
        "creg",
        "input",
        "mutable",
        "output",
        "qreg",
        "qubit",
        "readonly",
        "void",
        // Gate modifiers
        "ctrl",
        "inv",
        "negctrl",
        "pow",
        // I suppose these are literals
        "false",
        "true",
    ],
    // GJL: try introducing scalar_types to help parse var declarations. May not be useful
    // sourcegen_ast.rs can convert these to upper snake case.
    scalar_types: &[
        "angle", "bit", "bool", "complex", "duration", "float", "int", "stretch", "uint",
    ],
    // These are already upper snake case.
    literals: &[
        "BIT_STRING",
        "BYTE",
        "CHAR",
        "FLOAT_NUMBER",
        "INT_NUMBER",
        "STRING",
    ],
    tokens: &[
        "ANNOTATION",
        "COMMENT",
        "ERROR",
        "HARDWAREIDENT",
        "IDENT",
        "PRAGMA",
        "WHITESPACE",
    ],
    nodes: &[
        "ANNOTATION_STATEMENT",
        "BARRIER",
        "CAL",
        "CONST",
        "DEF",
        "DEF_CAL",
        "DEF_CAL_GRAMMAR",
        "DELAY_STMT",
        "GATE",
        "MEASURE",
        "PRAGMA_STATEMENT",
        "RESET",
        "SOURCE_FILE",
        "TIMING_LITERAL",
        // atoms
        "ARRAY_EXPR",
        "BLOCK_EXPR",
        "BOX_EXPR",
        "BREAK_STMT",
        "CASE_EXPR",
        "CONTINUE_STMT",
        "END_STMT",
        "END_STMT",
        "FOR_ITERABLE",
        "FOR_STMT",
        "LET_STMT",
        "PAREN_EXPR",
        "RETURN_EXPR",
        "STMT_LIST",
        "SWITCH_CASE_STMT",
        "TUPLE_EXPR",
        "WHILE_STMT",
        // postfix
        "CALL_EXPR",
        "CAST_EXPRESSION",
        "GATE_CALL_EXPR",
        "G_PHASE_CALL_EXPR",
        "INDEX_EXPR",
        "MODIFIED_GATE_CALL_EXPR",
        // unary
        "ARG_LIST",
        "BIN_EXPR",
        "DECLARATION",
        "EXPR_STMT",
        "FILE_PATH",
        "INCLUDE",
        "LITERAL",
        "NAME",
        "PARAM",
        "PARAM_LIST",
        "PREFIX_EXPR",
        "QUBIT_LIST",
        "RANGE_EXPR",
        "TYPE",
        "TYPED_PARAM",
        "TYPED_PARAM_LIST",
        "VERSION",
        "VERSION_STRING",
        // From ANTLR grammar
        "ALIAS_DECLARATION_STATEMENT",
        "ARRAY_LITERAL",
        "ARRAY_TYPE",
        "ASSIGNMENT_STMT",
        "CLASSICAL_DECLARATION_STATEMENT",
        "DESIGNATOR",
        "EXPRESSION_LIST",
        "GATE_OPERAND",
        "HARDWARE_QUBIT",
        "IDENTIFIER",
        "INDEXED_IDENTIFIER",
        "INDEX_KIND",
        "INDEX_OPERATOR",
        "I_O_DECLARATION_STATEMENT",
        "MEASURE_EXPRESSION",
        "OLD_STYLE_DECLARATION_STATEMENT",
        "QUANTUM_DECLARATION_STATEMENT",
        "QUBIT_TYPE",
        "RETURN_SIGNATURE",
        "SCALAR_TYPE",
        "SET_EXPRESSION",
        // Gate modifiers
        "CTRL_MODIFIER",
        "INV_MODIFIER",
        "MODIFIER",
        "NEG_CTRL_MODIFIER",
        "POW_MODIFIER",
    ],
};

#[derive(Default, Debug)]
pub(crate) struct AstSrc {
    pub(crate) tokens: Vec<String>,
    pub(crate) nodes: Vec<AstNodeSrc>,
    pub(crate) enums: Vec<AstEnumSrc>,
}

#[derive(Debug)]
pub(crate) struct AstNodeSrc {
    pub(crate) doc: Vec<String>,
    pub(crate) name: String,
    pub(crate) traits: Vec<String>,
    pub(crate) fields: Vec<Field>,
}

#[derive(Debug, Eq, PartialEq)]
pub(crate) enum Field {
    Token(String),
    Node {
        name: String,
        ty: String,
        cardinality: Cardinality,
    },
}

#[derive(Debug, Eq, PartialEq)]
pub(crate) enum Cardinality {
    Optional,
    Many,
}

#[derive(Debug)]
pub(crate) struct AstEnumSrc {
    pub(crate) doc: Vec<String>,
    pub(crate) name: String,
    pub(crate) traits: Vec<String>,
    pub(crate) variants: Vec<String>,
}
