/**
 *    Copyright (C) 2022-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include "mongo/util/namespace_string_util.h"
#include "mongo/db/multitenancy_gen.h"
#include "mongo/db/namespace_string.h"
#include "mongo/db/server_feature_flags_gen.h"
#include "mongo/util/str.h"
#include <ostream>

namespace mongo {

std::string NamespaceStringUtil::serialize(const NamespaceString& ns,
                                           const SerializationContext& context,
                                           const SerializationOptions& options) {
    if (gMultitenancySupport) {
        const auto fcvSnapshot = serverGlobalParams.featureCompatibility.acquireFCVSnapshot();
        if (fcvSnapshot.isVersionInitialized() &&
            gFeatureFlagRequireTenantID.isEnabled(fcvSnapshot)) {
            return options.serializeIdentifier(ns.toString());
        }
        return options.serializeIdentifier(ns.toStringWithTenantId());
    }
    return options.serializeIdentifier(ns.toString());
}


NamespaceString NamespaceStringUtil::deserialize(boost::optional<TenantId> tenantId,
                                                 StringData ns,
                                                 const SerializationContext& context) {
    if (ns.empty()) {
        return NamespaceString();
    }

    if (!gMultitenancySupport) {
        massert(6972102,
                str::stream() << "TenantId must not be set, but it is " << tenantId->toString(),
                tenantId == boost::none);
        return NamespaceString(boost::none, ns);
    }

    const auto fcvSnapshot = serverGlobalParams.featureCompatibility.acquireFCVSnapshot();
    if (fcvSnapshot.isVersionInitialized() && gFeatureFlagRequireTenantID.isEnabled(fcvSnapshot)) {
        StringData dbName = ns.substr(0, ns.find('.'));
        if (!(dbName == DatabaseName::kAdmin.db()) && !(dbName == DatabaseName::kLocal.db()) &&
            !(dbName == DatabaseName::kConfig.db())) {
            massert(6972100,
                    str::stream() << "TenantId must be set on nss " << ns,
                    tenantId != boost::none);
        }
        return NamespaceString(std::move(tenantId), ns);
    }

    auto nss = NamespaceString::parseFromStringExpectTenantIdInMultitenancyMode(ns);
    // TenantId could be prefixed, or passed in separately (or both) and namespace is always
    // constructed with the tenantId separately.
    if (tenantId != boost::none) {
        if (!nss.tenantId()) {
            return NamespaceString(std::move(tenantId), ns);
        }
        massert(6972101,
                str::stream() << "TenantId must match the db prefix tenantId: "
                              << tenantId->toString() << " prefix " << nss.tenantId()->toString(),
                tenantId == nss.tenantId());
    }

    return nss;
}

NamespaceString NamespaceStringUtil::parseNamespaceFromRequest(
    const boost::optional<TenantId>& tenantId, StringData ns) {
    return deserialize(tenantId, ns);
}

NamespaceString NamespaceStringUtil::parseNamespaceFromRequest(
    const boost::optional<TenantId>& tenantId, StringData db, StringData coll) {
    if (!gMultitenancySupport) {
        return NamespaceString{tenantId, db, coll};
    }

    if (coll.empty())
        return deserialize(tenantId, db);

    uassert(ErrorCodes::InvalidNamespace,
            "Collection names cannot start with '.': " + coll,
            coll[0] != '.');

    return deserialize(tenantId, str::stream() << db << "." << coll);
}

NamespaceString NamespaceStringUtil::parseNamespaceFromRequest(const DatabaseName& dbName,
                                                               StringData coll) {
    if (!gMultitenancySupport) {
        return NamespaceString{dbName, coll};
    }

    if (coll.empty()) {
        return NamespaceString(dbName);
    }

    uassert(ErrorCodes::InvalidNamespace,
            "Collection names cannot start with '.': " + coll,
            coll[0] != '.');

    return deserialize(dbName.tenantId(), str::stream() << dbName.db() << "." << coll);
}

NamespaceString NamespaceStringUtil::parseNamespaceFromDoc(
    const boost::optional<TenantId>& tenantId, StringData ns) {
    return deserialize(tenantId, ns);
}

NamespaceString NamespaceStringUtil::parseNamespaceFromDoc(
    const boost::optional<TenantId>& tenantId, StringData db, StringData coll) {
    if (!gMultitenancySupport) {
        return NamespaceString{tenantId, db, coll};
    }

    if (coll.empty())
        return deserialize(tenantId, db);

    uassert(ErrorCodes::InvalidNamespace,
            "Collection names cannot start with '.': " + coll,
            coll[0] != '.');

    return deserialize(tenantId, str::stream() << db << "." << coll);
}

NamespaceString NamespaceStringUtil::parseNamespaceFromDoc(const DatabaseName& dbName,
                                                           StringData coll) {
    if (!gMultitenancySupport) {
        return NamespaceString{dbName, coll};
    }

    if (coll.empty())
        return NamespaceString(dbName);

    uassert(ErrorCodes::InvalidNamespace,
            "Collection names cannot start with '.': " + coll,
            coll[0] != '.');

    return deserialize(dbName.tenantId(), str::stream() << dbName.db() << "." << coll);
}

NamespaceString NamespaceStringUtil::parseNamespaceFromResponse(const DatabaseName& dbName,
                                                                StringData coll) {
    return parseNamespaceFromDoc(dbName, coll);
}

}  // namespace mongo
