/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.jts.shape;

import org.locationtech.jts.algorithm.Angle;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateList;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.util.GeometryMapper;

public class CubicBezierCurve {
    private double minSegmentLength = 0.0;
    private int numVerticesPerSegment = 16;
    private Geometry inputGeom;
    private double alpha = -1.0;
    private double skew = 0.0;
    private Geometry controlPoints = null;
    private final GeometryFactory geomFactory;
    private Coordinate[] bezierCurvePts;
    private double[][] interpolationParam;
    private int controlPointIndex = 0;
    private static final double CIRCLE_LEN_FACTOR = 0.375;

    public static Geometry bezierCurve(Geometry geom, double alpha) {
        CubicBezierCurve curve = new CubicBezierCurve(geom, alpha);
        return curve.getResult();
    }

    public static Geometry bezierCurve(Geometry geom, double alpha, double skew) {
        CubicBezierCurve curve = new CubicBezierCurve(geom, alpha, skew);
        return curve.getResult();
    }

    public static Geometry bezierCurve(Geometry geom, Geometry controlPoints) {
        CubicBezierCurve curve = new CubicBezierCurve(geom, controlPoints);
        return curve.getResult();
    }

    CubicBezierCurve(Geometry geom, double alpha) {
        this.inputGeom = geom;
        this.geomFactory = geom.getFactory();
        if (alpha < 0.0) {
            alpha = 0.0;
        }
        this.alpha = alpha;
    }

    CubicBezierCurve(Geometry geom, double alpha, double skew) {
        this.inputGeom = geom;
        this.geomFactory = geom.getFactory();
        if (alpha < 0.0) {
            alpha = 0.0;
        }
        this.alpha = alpha;
        this.skew = skew;
    }

    CubicBezierCurve(Geometry geom, Geometry controlPoints) {
        this.inputGeom = geom;
        this.geomFactory = geom.getFactory();
        this.controlPoints = controlPoints;
    }

    public Geometry getResult() {
        this.bezierCurvePts = new Coordinate[this.numVerticesPerSegment];
        this.interpolationParam = CubicBezierCurve.computeIterpolationParameters(this.numVerticesPerSegment);
        return GeometryMapper.flatMap(this.inputGeom, 1, new GeometryMapper.MapOp(){

            @Override
            public Geometry map(Geometry geom) {
                if (geom instanceof LineString) {
                    return CubicBezierCurve.this.bezierLine((LineString)geom);
                }
                if (geom instanceof Polygon) {
                    return CubicBezierCurve.this.bezierPolygon((Polygon)geom);
                }
                return geom.copy();
            }
        });
    }

    private LineString bezierLine(LineString ls) {
        Coordinate[] coords = ls.getCoordinates();
        CoordinateList curvePts = this.bezierCurve(coords, false);
        curvePts.add(coords[coords.length - 1].copy(), false);
        return this.geomFactory.createLineString(curvePts.toCoordinateArray());
    }

    private LinearRing bezierRing(LinearRing ring) {
        Coordinate[] coords = ring.getCoordinates();
        CoordinateList curvePts = this.bezierCurve(coords, true);
        curvePts.closeRing();
        return this.geomFactory.createLinearRing(curvePts.toCoordinateArray());
    }

    private Polygon bezierPolygon(Polygon poly) {
        LinearRing shell = this.bezierRing(poly.getExteriorRing());
        LinearRing[] holes = null;
        if (poly.getNumInteriorRing() > 0) {
            holes = new LinearRing[poly.getNumInteriorRing()];
            for (int i2 = 0; i2 < poly.getNumInteriorRing(); ++i2) {
                holes[i2] = this.bezierRing(poly.getInteriorRingN(i2));
            }
        }
        return this.geomFactory.createPolygon(shell, holes);
    }

    private CoordinateList bezierCurve(Coordinate[] coords, boolean isRing) {
        Coordinate[] control = this.controlPoints(coords, isRing);
        CoordinateList curvePts = new CoordinateList();
        for (int i2 = 0; i2 < coords.length - 1; ++i2) {
            int ctrlIndex = 2 * i2;
            this.addCurve(coords[i2], coords[i2 + 1], control[ctrlIndex], control[ctrlIndex + 1], curvePts);
        }
        return curvePts;
    }

    private Coordinate[] controlPoints(Coordinate[] coords, boolean isRing) {
        if (this.controlPoints != null) {
            int expectedNum2;
            if (this.controlPointIndex >= this.controlPoints.getNumGeometries()) {
                throw new IllegalArgumentException("Too few control point elements");
            }
            Geometry ctrlPtsGeom = this.controlPoints.getGeometryN(this.controlPointIndex++);
            Coordinate[] ctrlPts = ctrlPtsGeom.getCoordinates();
            int expectedNum1 = 2 * coords.length - 2;
            int n = expectedNum2 = isRing ? coords.length - 1 : coords.length;
            if (expectedNum1 != ctrlPts.length && expectedNum2 != ctrlPts.length) {
                throw new IllegalArgumentException(String.format("Wrong number of control points for element %d - expected %d or %d, found %d", this.controlPointIndex - 1, expectedNum1, expectedNum2, ctrlPts.length));
            }
            return ctrlPts;
        }
        return this.controlPoints(coords, isRing, this.alpha, this.skew);
    }

    private void addCurve(Coordinate p0, Coordinate p1, Coordinate ctrl0, Coordinate crtl1, CoordinateList curvePts) {
        double len = p0.distance(p1);
        if (len < this.minSegmentLength) {
            curvePts.add(new Coordinate(p0));
        } else {
            this.cubicBezier(p0, p1, ctrl0, crtl1, this.interpolationParam, this.bezierCurvePts);
            for (int i2 = 0; i2 < this.bezierCurvePts.length - 1; ++i2) {
                curvePts.add(this.bezierCurvePts[i2], false);
            }
        }
    }

    private Coordinate[] controlPoints(Coordinate[] coords, boolean isRing, double alpha, double skew) {
        int N = coords.length;
        int start2 = 1;
        int end = N - 1;
        if (isRing) {
            N = coords.length - 1;
            start2 = 0;
            end = N;
        }
        int nControl = 2 * coords.length - 2;
        Coordinate[] ctrl = new Coordinate[nControl];
        for (int i2 = start2; i2 < end; ++i2) {
            int iprev = i2 == 0 ? N - 1 : i2 - 1;
            Coordinate v0 = coords[iprev];
            Coordinate v1 = coords[i2];
            Coordinate v2 = coords[i2 + 1];
            double interiorAng = Angle.angleBetweenOriented(v0, v1, v2);
            double orient = Math.signum(interiorAng);
            double angBisect = Angle.bisector(v0, v1, v2);
            double ang0 = angBisect - orient * 1.5707963267948966;
            double ang1 = angBisect + orient * 1.5707963267948966;
            double dist0 = v1.distance(v0);
            double dist1 = v1.distance(v2);
            double lenBase = Math.min(dist0, dist1);
            double intAngAbs = Math.abs(interiorAng);
            double sharpnessFactor = intAngAbs >= 1.5707963267948966 ? 1.0 : intAngAbs / 1.5707963267948966;
            double len = alpha * 0.375 * sharpnessFactor * lenBase;
            double stretch0 = 1.0;
            double stretch1 = 1.0;
            if (skew != 0.0) {
                int skewIndex;
                double stretch = Math.abs(dist0 - dist1) / Math.max(dist0, dist1);
                int n = skewIndex = dist0 > dist1 ? 0 : 1;
                if (skew < 0.0) {
                    skewIndex = 1 - skewIndex;
                }
                if (skewIndex == 0) {
                    stretch0 += Math.abs(skew) * stretch;
                } else {
                    stretch1 += Math.abs(skew) * stretch;
                }
            }
            Coordinate ctl0 = Angle.project(v1, ang0, stretch0 * len);
            Coordinate ctl1 = Angle.project(v1, ang1, stretch1 * len);
            int index = 2 * i2 - 1;
            int i0 = index < 0 ? nControl - 1 : index;
            ctrl[i0] = ctl0;
            ctrl[index + 1] = ctl1;
        }
        if (!isRing) {
            this.setLineEndControlPoints(coords, ctrl);
        }
        return ctrl;
    }

    private void setLineEndControlPoints(Coordinate[] coords, Coordinate[] ctrl) {
        int N = ctrl.length;
        ctrl[0] = CubicBezierCurve.mirrorControlPoint(ctrl[1], coords[1], coords[0]);
        ctrl[N - 1] = CubicBezierCurve.mirrorControlPoint(ctrl[N - 2], coords[coords.length - 1], coords[coords.length - 2]);
    }

    private static Coordinate aimedControlPoint(Coordinate c2, Coordinate p1, Coordinate p0) {
        double len = p1.distance(c2);
        double ang = Angle.angle(p0, p1);
        return Angle.project(p0, ang, len);
    }

    private static Coordinate mirrorControlPoint(Coordinate c2, Coordinate p0, Coordinate p1) {
        double vlinex = p1.x - p0.x;
        double vliney = p1.y - p0.y;
        double vrotx = -vliney;
        double vroty = vlinex;
        double midx = (p0.x + p1.x) / 2.0;
        double midy = (p0.y + p1.y) / 2.0;
        return CubicBezierCurve.reflectPointInLine(c2, new Coordinate(midx, midy), new Coordinate(midx + vrotx, midy + vroty));
    }

    private static Coordinate reflectPointInLine(Coordinate p2, Coordinate p0, Coordinate p1) {
        double vx = p1.x - p0.x;
        double vy = p1.y - p0.y;
        double x = p0.x - p2.x;
        double y = p0.y - p2.y;
        double r = 1.0 / (vx * vx + vy * vy);
        double rx = p2.x + 2.0 * (x - x * vx * vx * r - y * vx * vy * r);
        double ry = p2.y + 2.0 * (y - y * vy * vy * r - x * vx * vy * r);
        return new Coordinate(rx, ry);
    }

    private void cubicBezier(Coordinate p0, Coordinate p1, Coordinate ctrl1, Coordinate ctrl2, double[][] param, Coordinate[] curve) {
        int n = curve.length;
        curve[0] = new Coordinate(p0);
        curve[n - 1] = new Coordinate(p1);
        for (int i2 = 1; i2 < n - 1; ++i2) {
            Coordinate c2 = new Coordinate();
            double sum = param[i2][0] + param[i2][1] + param[i2][2] + param[i2][3];
            c2.x = param[i2][0] * p0.x + param[i2][1] * ctrl1.x + param[i2][2] * ctrl2.x + param[i2][3] * p1.x;
            c2.x /= sum;
            c2.y = param[i2][0] * p0.y + param[i2][1] * ctrl1.y + param[i2][2] * ctrl2.y + param[i2][3] * p1.y;
            c2.y /= sum;
            curve[i2] = c2;
        }
    }

    private static double[][] computeIterpolationParameters(int n) {
        double[][] param = new double[n][4];
        for (int i2 = 0; i2 < n; ++i2) {
            double t = (double)i2 / (double)(n - 1);
            double tc = 1.0 - t;
            param[i2][0] = tc * tc * tc;
            param[i2][1] = 3.0 * tc * tc * t;
            param[i2][2] = 3.0 * tc * t * t;
            param[i2][3] = t * t * t;
        }
        return param;
    }
}

