// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief Add a short analysis description here
  class DASP_1979_I132045 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(DASP_1979_I132045);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(FinalState(), "FS");

      // find the hists based on beam energies
      if (inRange(sqrtS()/GeV,3.6,3.67))       _sqs = "0";
      else if (inRange(sqrtS()/GeV,3.98,4.1))  _sqs = "1";
      else if (inRange(sqrtS()/GeV,4.1,4.24))  _sqs = "2";
      else if (inRange(sqrtS()/GeV,4.24,4.36)) _sqs = "3";
      else if (inRange(sqrtS()/GeV,4.36,4.46)) _sqs = "4";
      else if (inRange(sqrtS()/GeV,4.46,4.98)) _sqs = "5";
      else if (isCompatibleWithSqrtS(5.0*GeV)) _sqs = "6";
      else if (isCompatibleWithSqrtS(5.2*GeV)) _sqs = "7";
      raiseBeamErrorIf(_sqs.empty());
      // Book histograms
      for (size_t ih=0; ih <= 7; ++ih) {
        const string pre = toString(ih);
        book(_h_p[pre+"pi"],      1,1,1+ih);
        book(_h_p[pre+"K"],       2+ih,1,1);
        book(_h_p[pre+"proton"], 10+ih,1,1);
        book(_h_x[pre+"pi"],     18,1,1+ih);
        book(_h_x[pre+"K"],      19+ih,1,1);
        book(_h_x[pre+"proton"], 27+ih,1,1);
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

      for (const Particle& p : apply<FinalState>(event, "FS").particles()) {
        const int id = p.abspid();
        const double modp = p.p3().mod();
        const double xp = 2.*modp/sqrtS();
        const double beta = modp / p.E();
        if (id==211) {
          _h_p[_sqs+"pi"]->fill(modp);
          _h_x[_sqs+"pi"]->fill(xp, 1./beta);
        }
        else if (id==321) {
          _h_p[_sqs+"K"]->fill(modp);
          _h_x[_sqs+"K"]->fill(xp, 1./beta);
        }
        else if (id==2212) {
          _h_p[_sqs+"proton"]->fill(modp);
          _h_x[_sqs+"proton"]->fill(xp, 1./beta);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      const double fact = crossSection()/nanobarn/sumOfWeights();
      scale(_h_p, fact);
      scale(_h_x, fact*sqr(sqrtS())); // < the energy is not defined when merging!
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h_p, _h_x;
    string _sqs = "";
    /// @}

  };

  RIVET_DECLARE_PLUGIN(DASP_1979_I132045);
}
