/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Zend Technologies
 *******************************************************************************/
package org.eclipse.php.internal.ui.editor.contentassist;

import java.util.function.Supplier;

import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.IPreferencesService;
import org.eclipse.dltk.core.IModelElement;
import org.eclipse.dltk.core.IScriptProject;
import org.eclipse.dltk.core.IType;
import org.eclipse.dltk.ui.text.ScriptTextTools;
import org.eclipse.dltk.ui.text.completion.ScriptCompletionProposal;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.php.core.compiler.ast.nodes.NamespaceReference;
import org.eclipse.php.internal.core.PHPCoreConstants;
import org.eclipse.php.internal.core.PHPCorePlugin;
import org.eclipse.php.internal.core.codeassist.ProposalExtraInfo;
import org.eclipse.php.internal.core.codeassist.strategies.IncludeStatementStrategy;
import org.eclipse.php.internal.core.typeinference.PHPModelUtils;
import org.eclipse.php.internal.ui.PHPUiPlugin;
import org.eclipse.swt.graphics.Image;

public class PHPCompletionProposal extends ScriptCompletionProposal implements IPHPCompletionProposalExtension {

	public PHPCompletionProposal(String replacementString, int replacementOffset, int replacementLength,
			Supplier<Image> image, StyledString displayString, int relevance) {
		super(replacementString, replacementOffset, replacementLength, image, displayString, relevance, false);
	}

	public PHPCompletionProposal(String replacementString, int replacementOffset, int replacementLength,
			Supplier<Image> image, StyledString displayString, int relevance, boolean indoc) {
		super(replacementString, replacementOffset, replacementLength, image, displayString, relevance, indoc);
	}

	@Override
	protected int getPatternMatchRule(String pattern, String string) {
		// XXX: avoid unnecessary StringIndexOutOfBoundsExceptions in
		// super.getPatternMatchRule(pattern, string) because
		// "pattern" and "string" may have nothing in common
		// when generated by class PHPCompletionProposal
		if (string.length() < pattern.length()) {
			return -1;
		}
		return super.getPatternMatchRule(pattern, string);
	}

	@Override
	protected boolean isValidPrefix(String prefix) {
		String word = getDisplayString();
		if (word.startsWith("$") && !prefix.startsWith("$")) { //$NON-NLS-1$ //$NON-NLS-2$
			word = word.substring(1);
		}
		boolean result = isPrefix(prefix, word);
		if (!result && prefix.length() > 0 && prefix.charAt(0) == NamespaceReference.NAMESPACE_SEPARATOR) {
			result = isPrefix(prefix.substring(1), word);
		}

		if (!result && ProposalExtraInfo.isMemberInNamespace(getExtraInfo()) && (getModelElement() instanceof IType)) {
			IType type = (IType) getModelElement();
			if (ProposalExtraInfo.isAbsoluteName(getExtraInfo())) {
				result = isPrefix(prefix.substring(1), PHPModelUtils.getFullName(type));
			} else {
				result = isPrefix(prefix, PHPModelUtils.getFullName(type));
			}
		}
		if (!result && ProposalExtraInfo.isFullName(getExtraInfo()) && getModelElement() != null) {
			if (ProposalExtraInfo.isAbsoluteName(getExtraInfo())) {
				result = isPrefix(prefix.substring(1), PHPModelUtils.getFullName(getModelElement()));
			} else {
				result = isPrefix(prefix, PHPModelUtils.getFullName(getModelElement()));
			}
		}
		// int index = word.indexOf(" - ");
		// if (!result && index >= 0) {
		// StringBuilder sb = new StringBuilder();
		// sb.append(word.substring(index + " - ".length()));
		// sb.append('\\');
		// sb.append(word.substring(0, index));
		// result = isPrefix(prefix, sb.toString());
		// }
		return result;
	}

	@Override
	protected boolean isSmartTrigger(char trigger) {
		return trigger == '$';
	}

	@Override
	public void apply(IDocument document, char trigger, int offset) {
		IModelElement modelElement = getModelElement();

		boolean activateCodeAssist = false;
		String replacementString = getReplacementString();
		if (modelElement instanceof IScriptProject
				&& replacementString.endsWith(IncludeStatementStrategy.FOLDER_SEPARATOR)) {
			// workaround for:
			// https://bugs.eclipse.org/bugs/show_bug.cgi?id=269634
			activateCodeAssist = true;
		} else {
			IPreferencesService preferencesService = Platform.getPreferencesService();
			boolean enableAutoactivation = preferencesService.getBoolean(PHPCorePlugin.ID,
					PHPCoreConstants.CODEASSIST_AUTOACTIVATION, false, null);
			if (enableAutoactivation) {
				char lastChar = replacementString.charAt(replacementString.length() - 1);
				for (char autoActivationChar : PHPCompletionProcessor.completionAutoActivationChars) {
					if (autoActivationChar == lastChar) {
						activateCodeAssist = true;
						break;
					}
				}
			}
		}
		if (activateCodeAssist) {
			AutoActivationTrigger.register(document);
		}

		UseStatementInjector injector = new UseStatementInjector(this);
		offset = injector.inject(document, getTextViewer(), offset);

		super.apply(document, trigger, offset);

		setCursorPosition(calcCursorPosition());
	}

	private int calcCursorPosition() {
		String replacementString = getReplacementString();
		int i = replacementString.lastIndexOf('(');
		if (i != -1) {
			return i + 1;
		}
		i = replacementString.lastIndexOf('\'');
		if (i != -1) {
			return i;
		}
		i = replacementString.lastIndexOf('\"');
		if (i != -1) {
			return i;
		}
		return replacementString.length();
	}

	@Override
	public IContextInformation getContextInformation() {
		String displayString = getDisplayString();
		if (displayString.indexOf('(') == -1) {
			return null;
		}
		return super.getContextInformation();
	}

	@Override
	protected boolean isCamelCaseMatching() {
		return true;
	}

	@Override
	protected boolean insertCompletion() {
		return Platform.getPreferencesService().getBoolean(PHPCorePlugin.ID,
				PHPCoreConstants.CODEASSIST_INSERT_COMPLETION, true, null);
	}

	@Override
	protected ScriptTextTools getTextTools() {
		return PHPUiPlugin.getDefault().getTextTools();
	}

	@Override
	public Object getExtraInfo() {
		return ProposalExtraInfo.DEFAULT;
	}
}