/* Copyright (C) 1993,1994 by the author(s).
 
 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with ShapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * AtFS -- Attribute Filesystem
 *
 * aflock.c -- Version locking
 *
 * Author: Andreas Lampen (Andreas.Lampen@cs.tu-berlin.de)
 *
 * $Header: aflock.c[7.0] Fri Jun 25 14:32:02 1993 andy@cs.tu-berlin.de frozen $
 */

#include "atfs.h"

/*================================================================
 *	af_lock
 *================================================================*/

EXPORT Af_user *af_lock (key, locker)
     Af_key  *key;
     Af_user *locker;
{
  int oldlockeruid, newlockeruid, calleruid;

  if (afAccessAso (key, AF_ATTRS))
    FAIL ("lock", "", AF_EINVKEY, NULL);
  
  if (!S_ISREG (VATTR(key).af_mode))
    FAIL ("lock", "", AF_ENOTREGULAR, NULL);

  if (VATTR(key).af_state == AF_DERIVED)
    FAIL ("lock", "", AF_EDERIVED, NULL);

  if (!locker->af_userhost[0])
    strcpy (locker->af_userhost, af_gethostname());
  if (!locker->af_userdomain[0])
    strcpy (locker->af_userdomain, af_getdomain());

  if ((newlockeruid = af_getuid (locker->af_username, locker->af_userhost, locker->af_userdomain)) == ERROR)
    FAIL ("lock", "invalid locker", AF_EMISC, NULL);
  calleruid = geteuid ();

  /* if caller == locker, everyone who has write access may set a lock, */
  /*                      if the object is not locked by someone else */
  /* else (caller != locker), only the owner may set a lock for someone else */

  if (calleruid == newlockeruid) {
    if (af_checkperm (key, AF_WORLD) == ERROR)
      return (NULL);
  }
  else {
    if (af_checkperm (key, AF_OWNER) == ERROR)
      return (NULL);
  }

  /* if version is not locked or already locked by caller */
  oldlockeruid = af_getuid (VATTR(key).af_lckname, VATTR(key).af_lckhost, VATTR(key).af_lckdomain);

  if ((oldlockeruid == (uid_t) ERROR) || (oldlockeruid == calleruid)) {
    VATTR(key).af_lckname = af_entersym (locker->af_username);
    VATTR(key).af_lckhost = af_enterhost (locker->af_userhost);
    VATTR(key).af_lckdomain = af_enterdomain (locker->af_userdomain);
    VATTR(key).af_ltime = af_acttime();
    if (afUpdateAso (key, AF_CHANGE) == ERROR)
      return (NULL);
    return (locker);
  }
  else
    FAIL ("lock", "", AF_ENOTLOCKED, NULL);
}

/*================================================================
 *	af_unlock
 *================================================================*/

EXPORT Af_user *af_unlock (key)
     Af_key *key;
{
  static Af_user locker;

  if (afAccessAso (key, AF_ATTRS))
    FAIL ("unlock", "", AF_EINVKEY, NULL);

  if (af_checkperm (key, AF_OWNER | AF_LOCKHOLDER) == ERROR)
    return (NULL);

  if (VATTR(key).af_lckname) {
    strcpy (locker.af_username, VATTR(key).af_lckname);
    strcpy (locker.af_userhost, VATTR(key).af_lckhost);
    strcpy (locker.af_userdomain, VATTR(key).af_lckdomain);
    VATTR(key).af_lckname = NULL;
    VATTR(key).af_lckhost = NULL;
    VATTR(key).af_lckdomain = NULL;
    VATTR(key).af_ltime = af_acttime();
    if (afUpdateAso (key, AF_CHANGE) == ERROR)
      return (NULL);
  }
  else {
    locker.af_username[0] = '\0';
    locker.af_userhost[0] = '\0';
    locker.af_userdomain[0] = '\0';
  }
  return (&locker);
}

/*================================================================
 *	af_testlock
 *================================================================*/

EXPORT Af_user *af_testlock (key)
     Af_key *key;
{
  static Af_user locker;

  if (afAccessAso (key, AF_ATTRS))
    FAIL ("testlock", "", AF_EINVKEY, NULL);

  if (VATTR(key).af_lckname) {
    strcpy (locker.af_username, VATTR(key).af_lckname);
    strcpy (locker.af_userhost, VATTR(key).af_lckhost);
    strcpy (locker.af_userdomain, VATTR(key).af_lckdomain);
  }
  else {
    locker.af_username[0] = '\0';
    locker.af_userhost[0] = '\0';
    locker.af_userdomain[0] = '\0';
  }

  return (&locker);
}


