/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

package hdf.hdf5lib;

import static org.hdfgroup.javahdf5.hdf5_h.*;

import java.lang.foreign.MemorySegment;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;

import hdf.hdf5lib.H5;
import hdf.hdf5lib.structs.H5O_token_t;

import org.hdfgroup.javahdf5.*;

/**
 * \page HDF5CONST Constants and Enumerated Types
 * This class contains C constants and enumerated types of HDF5 library. The
 * values of these constants are obtained from the library by calling
 * the JNI function jconstant, where jconstant is used for any of the private constants
 * which start their name with "H5" need to be converted.
 * <P>
 * <B>Do not edit this file!</b>
 *
 * @see @ref HDF5LIB
 */
public class HDF5Constants {

    static { System.err.println("OpenIDs = " + H5.getOpenIDCount()); }

    /** Special parameters for szip compression */
    public static final int H5_SZIP_MAX_PIXELS_PER_BLOCK = H5_SZIP_MAX_PIXELS_PER_BLOCK();
    /** Special parameters for szip compression */
    public static final int H5_SZIP_NN_OPTION_MASK = H5_SZIP_NN_OPTION_MASK();
    /** Special parameters for szip compression */
    public static final int H5_SZIP_EC_OPTION_MASK = H5_SZIP_EC_OPTION_MASK();
    /** Special parameters for szip compression */
    public static final int H5_SZIP_ALLOW_K13_OPTION_MASK = H5_SZIP_ALLOW_K13_OPTION_MASK();
    /** Special parameters for szip compression */
    public static final int H5_SZIP_CHIP_OPTION_MASK = H5_SZIP_CHIP_OPTION_MASK();
    /** indices on links, unknown index type */
    public static final int H5_INDEX_UNKNOWN = H5_INDEX_UNKNOWN();
    /** indices on links, index on names */
    public static final int H5_INDEX_NAME = H5_INDEX_NAME();
    /** indices on links, index on creation order */
    public static final int H5_INDEX_CRT_ORDER = H5_INDEX_CRT_ORDER();
    /** indices on links, number of indices defined */
    public static final int H5_INDEX_N = H5_INDEX_N();
    /** Common iteration orders, Unknown order */
    public static final int H5_ITER_UNKNOWN = H5_ITER_UNKNOWN();
    /** Common iteration orders, Increasing order */
    public static final int H5_ITER_INC = H5_ITER_INC();
    /** Common iteration orders, Decreasing order */
    public static final int H5_ITER_DEC = H5_ITER_DEC();
    /** Common iteration orders, No particular order, whatever is fastest */
    public static final int H5_ITER_NATIVE = H5_ITER_NATIVE();
    /** Common iteration orders, Number of iteration orders */
    public static final int H5_ITER_N = H5_ITER_N();
    /** The version of the H5AC_cache_config_t in use */
    public static final int H5AC_CURR_CACHE_CONFIG_VERSION = H5AC__CURR_CACHE_CONFIG_VERSION();
    /** The maximum length of the trace file path */
    public static final int H5AC_MAX_TRACE_FILE_NAME_LEN = H5AC__MAX_TRACE_FILE_NAME_LEN();
    /**
     *    When metadata_write_strategy is set to this value, only process
     *    zero is allowed to write dirty metadata to disk.  All other
     *    processes must retain dirty metadata until they are informed at
     *    a sync point that the dirty metadata in question has been written
     *    to disk.
     */
    public static final int H5AC_METADATA_WRITE_STRATEGY_PROCESS_0_ONLY =
        H5AC_METADATA_WRITE_STRATEGY__PROCESS_0_ONLY();
    /**
     *    In the distributed metadata write strategy, process zero still makes
     *    the decisions as to what entries should be flushed, but the actual
     *    flushes are distributed across the processes in the computation to
     *    the extent possible.
     */
    public static final int H5AC_METADATA_WRITE_STRATEGY_DISTRIBUTED =
        H5AC_METADATA_WRITE_STRATEGY__DISTRIBUTED();
    /** Don't attempt to increase the size of the cache automatically */
    public static final int H5C_incr_off = H5C_incr__off();
    /**
     *  Attempt to increase the size of the cache
     *              whenever the average hit rate over the last epoch drops
     *              below the value supplied in the lower_hr_threshold
     *              field
     */
    public static final int H5C_incr_threshold = H5C_incr__threshold();
    /** Don't perform flash increases in the size of the cache */
    public static final int H5C_flash_incr_off = H5C_flash_incr__off();
    /** increase the current maximum cache size by x * flash_multiple less any free space in the cache */
    public static final int H5C_flash_incr_add_space = H5C_flash_incr__add_space();
    /** Don't attempt to decrease the size of the cache automatically */
    public static final int H5C_decr_off = H5C_decr__off();
    /**
     *  Attempt to decrease the size of the cache
     *              whenever the average hit rate over the last epoch rises
     *              above the value supplied in the upper_hr_threshold
     *              field
     */
    public static final int H5C_decr_threshold = H5C_decr__threshold();
    /**
     *  At the end of each epoch, search the cache for
     *              entries that have not been accessed for at least the number
     *              of epochs specified in the epochs_before_eviction field, and
     *              evict these entries
     */
    public static final int H5C_decr_age_out = H5C_decr__age_out();
    /**
     *  Same as age_out, but we only
     *              attempt to reduce the cache size when the hit rate observed
     *              over the last epoch exceeds the value provided in the
     *              upper_hr_threshold field
     */
    public static final int H5C_decr_age_out_with_threshold = H5C_decr__age_out_with_threshold();
    /** */
    public static final int H5D_CHUNK_IDX_BTREE = H5D_CHUNK_IDX_BTREE();
    /** */
    public static final int H5D_ALLOC_TIME_DEFAULT = H5D_ALLOC_TIME_DEFAULT();
    /** */
    public static final int H5D_ALLOC_TIME_EARLY = H5D_ALLOC_TIME_EARLY();
    /** */
    public static final int H5D_ALLOC_TIME_ERROR = H5D_ALLOC_TIME_ERROR();
    /** */
    public static final int H5D_ALLOC_TIME_INCR = H5D_ALLOC_TIME_INCR();
    /** */
    public static final int H5D_ALLOC_TIME_LATE = H5D_ALLOC_TIME_LATE();
    /** */
    public static final int H5D_FILL_TIME_ERROR = H5D_FILL_TIME_ERROR();
    /** */
    public static final int H5D_FILL_TIME_ALLOC = H5D_FILL_TIME_ALLOC();
    /** */
    public static final int H5D_FILL_TIME_NEVER = H5D_FILL_TIME_NEVER();
    /** */
    public static final int H5D_FILL_TIME_IFSET = H5D_FILL_TIME_IFSET();
    /** */
    public static final int H5D_FILL_VALUE_DEFAULT = H5D_FILL_VALUE_DEFAULT();
    /** */
    public static final int H5D_FILL_VALUE_ERROR = H5D_FILL_VALUE_ERROR();
    /** */
    public static final int H5D_FILL_VALUE_UNDEFINED = H5D_FILL_VALUE_UNDEFINED();
    /** */
    public static final int H5D_FILL_VALUE_USER_DEFINED = H5D_FILL_VALUE_USER_DEFINED();
    /** */
    public static final int H5D_LAYOUT_ERROR = H5D_LAYOUT_ERROR();
    /** */
    public static final int H5D_CHUNKED = H5D_CHUNKED();
    /** */
    public static final int H5D_COMPACT = H5D_COMPACT();
    /** */
    public static final int H5D_CONTIGUOUS = H5D_CONTIGUOUS();
    /** */
    public static final int H5D_VIRTUAL = H5D_VIRTUAL();
    /** */
    public static final int H5D_NLAYOUTS = H5D_NLAYOUTS();
    /** */
    public static final int H5D_SPACE_STATUS_ALLOCATED = H5D_SPACE_STATUS_ALLOCATED();
    /** */
    public static final int H5D_SPACE_STATUS_ERROR = H5D_SPACE_STATUS_ERROR();
    /** */
    public static final int H5D_SPACE_STATUS_NOT_ALLOCATED = H5D_SPACE_STATUS_NOT_ALLOCATED();
    /** */
    public static final int H5D_SPACE_STATUS_PART_ALLOCATED = H5D_SPACE_STATUS_PART_ALLOCATED();
    /** */
    public static final int H5D_VDS_ERROR = H5D_VDS_ERROR();
    /** */
    public static final int H5D_VDS_FIRST_MISSING = H5D_VDS_FIRST_MISSING();
    /** */
    public static final int H5D_VDS_LAST_AVAILABLE = H5D_VDS_LAST_AVAILABLE();
    /** */
    public static final int H5D_CHUNK_DONT_FILTER_PARTIAL_CHUNKS = H5D_CHUNK_DONT_FILTER_PARTIAL_CHUNKS();

    /** Different kinds of error information - H5E_type_t */
    public static final int H5E_MAJOR = H5E_MAJOR();
    /** Different kinds of error information - H5E_type_t */
    public static final int H5E_MINOR = H5E_MINOR();
    /** Minor error codes - Object header related errors - Alignment error */
    public static final long H5E_ALIGNMENT = H5E_ALIGNMENT_g();
    /** Minor error codes - Resource errors - Object already exists */
    public static final long H5E_ALREADYEXISTS = H5E_ALREADYEXISTS_g();
    /** Minor error codes - Function entry/exit interface - Object already initialized */
    public static final long H5E_ALREADYINIT = H5E_ALREADYINIT_g();
    /** Major error codes - Invalid arguments to routine */
    public static final long H5E_ARGS = H5E_ARGS_g();
    /** Major error codes - Object ID */
    public static final long H5E_ID = H5E_ID_g();
    /** Major error codes - Attribute */
    public static final long H5E_ATTR = H5E_ATTR_g();
    /** Minor error codes - Object ID related errors - Unable to find ID information (already closed?) */
    public static final long H5E_BADID = H5E_BADID_g();
    /** Minor error codes - File accessibility errors - Bad file ID accessed */
    public static final long H5E_BADFILE = H5E_BADFILE_g();
    /** Minor error codes - Object ID related errors - Unable to find ID group information */
    public static final long H5E_BADGROUP = H5E_BADGROUP_g();
    /** Minor error codes - Object header related errors - Iteration failed */
    public static final long H5E_BADITER = H5E_BADITER_g();
    /** Minor error codes - Object header related errors - Unrecognized message */
    public static final long H5E_BADMESG = H5E_BADMESG_g();
    /** Minor error codes - Argument errors - Out of range */
    public static final long H5E_BADRANGE = H5E_BADRANGE_g();
    /** Minor error codes - Dataspace errors - Invalid selection */
    public static final long H5E_BADSELECT = H5E_BADSELECT_g();
    /** Datatype conversion errors - Bad size for object */
    public static final long H5E_BADSIZE = H5E_BADSIZE_g();
    /** Minor error codes - Argument errors - Inappropriate type */
    public static final long H5E_BADTYPE = H5E_BADTYPE_g();
    /** Minor error codes - Argument errors - Bad value */
    public static final long H5E_BADVALUE = H5E_BADVALUE_g();
    /** Major error codes - B-Tree node */
    public static final long H5E_BTREE = H5E_BTREE_g();
    /** Major error codes - Object cache */
    public static final long H5E_CACHE = H5E_CACHE_g();
    /** I/O pipeline errors - Callback failed */
    public static final long H5E_CALLBACK = H5E_CALLBACK_g();
    /** I/O pipeline errors - Error from filter 'can apply' callback */
    public static final long H5E_CANAPPLY = H5E_CANAPPLY_g();
    /** Minor error codes - Resource errors - Can't allocate space */
    public static final long H5E_CANTALLOC = H5E_CANTALLOC_g();
    /** Minor error codes - Dataspace errors - Can't append object */
    public static final long H5E_CANTAPPEND = H5E_CANTAPPEND_g();
    /** Minor error codes - Heap errors - Can't attach object */
    public static final long H5E_CANTATTACH = H5E_CANTATTACH_g();
    /** Minor error codes - Cache related errors - Unable to mark metadata as clean */
    public static final long H5E_CANTCLEAN = H5E_CANTCLEAN_g();
    /** Minor error codes - Dataspace errors - Can't clip hyperslab region */
    public static final long H5E_CANTCLIP = H5E_CANTCLIP_g();
    /** Minor error codes - File accessibility errors - Unable to close file */
    public static final long H5E_CANTCLOSEFILE = H5E_CANTCLOSEFILE_g();
    /** Minor error codes - Group related errors - Can't close object */
    public static final long H5E_CANTCLOSEOBJ = H5E_CANTCLOSEOBJ_g();
    /** Minor error codes - Dataspace errors - Can't compare objects */
    public static final long H5E_CANTCOMPARE = H5E_CANTCOMPARE_g();
    /** Minor error codes - Heap errors - Can't compute value */
    public static final long H5E_CANTCOMPUTE = H5E_CANTCOMPUTE_g();
    /** Datatype conversion errors - Can't convert datatypes */
    public static final long H5E_CANTCONVERT = H5E_CANTCONVERT_g();
    /** Minor error codes - Resource errors - Unable to copy object */
    public static final long H5E_CANTCOPY = H5E_CANTCOPY_g();
    /** Minor error codes - Cache related errors - Unable to cork an object */
    public static final long H5E_CANTCORK = H5E_CANTCORK_g();
    /** Minor error codes - Dataspace errors - Can't count elements */
    public static final long H5E_CANTCOUNT = H5E_CANTCOUNT_g();
    /** Minor error codes - File accessibility errors - Unable to create file */
    public static final long H5E_CANTCREATE = H5E_CANTCREATE_g();
    /** Minor error codes - Object ID related errors - Unable to decrement reference count */
    public static final long H5E_CANTDEC = H5E_CANTDEC_g();
    /** Minor error codes - B-tree related errors - Unable to decode value */
    public static final long H5E_CANTDECODE = H5E_CANTDECODE_g();
    /** Minor error codes - Object header related errors - Can't delete message */
    public static final long H5E_CANTDELETE = H5E_CANTDELETE_g();
    /** Minor error codes - File accessibility errors - Unable to delete file */
    public static final long H5E_CANTDELETEFILE = H5E_CANTDELETEFILE_g();
    /** Minor error codes - Cache related errors - Unable to create a flush dependency */
    public static final long H5E_CANTDEPEND = H5E_CANTDEPEND_g();
    /** Minor error codes - Cache related errors - Unable to mark metadata as dirty */
    public static final long H5E_CANTDIRTY = H5E_CANTDIRTY_g();
    /** Minor error codes - B-tree related errors - Unable to encode value */
    public static final long H5E_CANTENCODE = H5E_CANTENCODE_g();
    /** Minor error codes - Cache related errors - Unable to expunge a metadata cache entry */
    public static final long H5E_CANTEXPUNGE = H5E_CANTEXPUNGE_g();
    /** Minor error codes - Heap errors - Can't extend heap's space */
    public static final long H5E_CANTEXTEND = H5E_CANTEXTEND_g();
    /** I/O pipeline errors - Filter operation failed */
    public static final long H5E_CANTFILTER = H5E_CANTFILTER_g();
    /** Minor error codes - Cache related errors - Unable to flush data from cache */
    public static final long H5E_CANTFLUSH = H5E_CANTFLUSH_g();
    /** Minor error codes - Resource errors - Unable to free object */
    public static final long H5E_CANTFREE = H5E_CANTFREE_g();
    /** Minor error codes - Parallel MPI - Can't gather data */
    public static final long H5E_CANTGATHER = H5E_CANTGATHER_g();
    /** Minor error codes - Resource errors - Unable to garbage collect */
    public static final long H5E_CANTGC = H5E_CANTGC_g();
    /** Minor error codes - Property list errors - Can't get value */
    public static final long H5E_CANTGET = H5E_CANTGET_g();
    /** Minor error codes - Resource errors - Unable to compute size */
    public static final long H5E_CANTGETSIZE = H5E_CANTGETSIZE_g();
    /** Minor error codes - Object ID related errors - Unable to increment reference count */
    public static final long H5E_CANTINC = H5E_CANTINC_g();
    /** Minor error codes - Function entry/exit interface - Unable to initialize object */
    public static final long H5E_CANTINIT = H5E_CANTINIT_g();
    /** Minor error codes - Cache related errors - Unable to insert metadata into cache */
    public static final long H5E_CANTINS = H5E_CANTINS_g();
    /** Minor error codes - B-tree related errors - Unable to insert object */
    public static final long H5E_CANTINSERT = H5E_CANTINSERT_g();
    /** Minor error codes - B-tree related errors - Unable to list node */
    public static final long H5E_CANTLIST = H5E_CANTLIST_g();
    /** Minor error codes - Cache related errors - Unable to load metadata into cache */
    public static final long H5E_CANTLOAD = H5E_CANTLOAD_g();
    /** Minor error codes - Resource errors - Unable to lock object */
    public static final long H5E_CANTLOCK = H5E_CANTLOCK_g();
    /** Minor error codes - File accessibility errors  Unable to lock file */
    public static final long H5E_CANTLOCKFILE = H5E_CANTLOCKFILE_g();
    /** Minor error codes - Cache related errors - Unable to mark a pinned entry as clean */
    public static final long H5E_CANTMARKCLEAN = H5E_CANTMARKCLEAN_g();
    /** Minor error codes - Cache related errors - Unable to mark a pinned entry as dirty */
    public static final long H5E_CANTMARKDIRTY = H5E_CANTMARKDIRTY_g();
    /** Minor error codes - Cache related errors - Unable to mark an entry as unserialized */
    public static final long H5E_CANTMARKSERIALIZED = H5E_CANTMARKSERIALIZED_g();
    /** Minor error codes - Cache related errors - Unable to mark an entry as serialized */
    public static final long H5E_CANTMARKUNSERIALIZED = H5E_CANTMARKUNSERIALIZED_g();
    /** Minor error codes - Free space errors - Can't merge objects */
    public static final long H5E_CANTMERGE = H5E_CANTMERGE_g();
    /** Minor error codes - B-tree related errors - Unable to modify record */
    public static final long H5E_CANTMODIFY = H5E_CANTMODIFY_g();
    /** Minor error codes - Link related errors - Can't move object */
    public static final long H5E_CANTMOVE = H5E_CANTMOVE_g();
    /** Minor error codes - Dataspace errors - Can't move to next iterator location */
    public static final long H5E_CANTNEXT = H5E_CANTNEXT_g();
    /** Minor error codes - Cache related errors - Unable to notify object about action */
    public static final long H5E_CANTNOTIFY = H5E_CANTNOTIFY_g();
    /** Minor error codes - File accessibility errors - Unable to open file */
    public static final long H5E_CANTOPENFILE = H5E_CANTOPENFILE_g();
    /** Minor error codes - Group related errors - Can't open object */
    public static final long H5E_CANTOPENOBJ = H5E_CANTOPENOBJ_g();
    /** Minor error codes - Heap errors - Can't operate on object */
    public static final long H5E_CANTOPERATE = H5E_CANTOPERATE_g();
    /** Minor error codes - Object header related errors - Can't pack messages */
    public static final long H5E_CANTPACK = H5E_CANTPACK_g();
    /** Minor error codes - Cache related errors - Unable to pin cache entry */
    public static final long H5E_CANTPIN = H5E_CANTPIN_g();
    /** Minor error codes - Cache related errors - Unable to protect metadata */
    public static final long H5E_CANTPROTECT = H5E_CANTPROTECT_g();
    /** Minor error codes - Parallel MPI - Can't receive data */
    public static final long H5E_CANTRECV = H5E_CANTRECV_g();
    /** Minor error codes - B-tree related errors - Unable to redistribute records */
    public static final long H5E_CANTREDISTRIBUTE = H5E_CANTREDISTRIBUTE_g();
    /** Minor error codes - Object ID related errors - Unable to register new ID */
    public static final long H5E_CANTREGISTER = H5E_CANTREGISTER_g();
    /** Minor error codes - Function entry/exit interface - Unable to release object */
    public static final long H5E_CANTRELEASE = H5E_CANTRELEASE_g();
    /** Minor error codes - B-tree related errors - Unable to remove object */
    public static final long H5E_CANTREMOVE = H5E_CANTREMOVE_g();
    /** Minor error codes - Object header related errors - Unable to rename object */
    public static final long H5E_CANTRENAME = H5E_CANTRENAME_g();
    /** Minor error codes - Object header related errors - Can't reset object */
    public static final long H5E_CANTRESET = H5E_CANTRESET_g();
    /** Minor error codes - Cache related errors - Unable to resize a metadata cache entry */
    public static final long H5E_CANTRESIZE = H5E_CANTRESIZE_g();
    /** Minor error codes - Heap errors - Can't restore condition */
    public static final long H5E_CANTRESTORE = H5E_CANTRESTORE_g();
    /** Minor error codes - Free space errors - Can't revive object */
    public static final long H5E_CANTREVIVE = H5E_CANTREVIVE_g();
    /** Minor error codes - Free space errors - Can't shrink container */
    public static final long H5E_CANTSHRINK = H5E_CANTSHRINK_g();
    /** Minor error codes - Dataspace errors - Can't select hyperslab */
    public static final long H5E_CANTSELECT = H5E_CANTSELECT_g();
    /** Minor error codes - Cache related errors - Unable to serialize data from cache */
    public static final long H5E_CANTSERIALIZE = H5E_CANTSERIALIZE_g();
    /** Minor error codes - Property list errors - Can't set value */
    public static final long H5E_CANTSET = H5E_CANTSET_g();
    /** Minor error codes - Link related errors - Can't sort objects */
    public static final long H5E_CANTSORT = H5E_CANTSORT_g();
    /** Minor error codes - B-tree related errors - Unable to split node */
    public static final long H5E_CANTSPLIT = H5E_CANTSPLIT_g();
    /** Minor error codes - B-tree related errors - Unable to swap records */
    public static final long H5E_CANTSWAP = H5E_CANTSWAP_g();
    /** Minor error codes - Cache related errors - Unable to tag metadata in the cache */
    public static final long H5E_CANTTAG = H5E_CANTTAG_g();
    /** Minor error codes - Cache related errors - Unable to uncork an object */
    public static final long H5E_CANTUNCORK = H5E_CANTUNCORK_g();
    /** Minor error codes - Cache related errors - Unable to destroy a flush dependency */
    public static final long H5E_CANTUNDEPEND = H5E_CANTUNDEPEND_g();
    /** Minor error codes - Resource errors - Unable to unlock object */
    public static final long H5E_CANTUNLOCK = H5E_CANTUNLOCK_g();
    /** Minor error codes - File accessibility errors  Unable to unlock file */
    public static final long H5E_CANTUNLOCKFILE = H5E_CANTUNLOCKFILE_g();
    /** Minor error codes - Cache related errors - Unable to un-pin cache entry */
    public static final long H5E_CANTUNPIN = H5E_CANTUNPIN_g();
    /** Minor error codes - Cache related errors - Unable to unprotect metadata */
    public static final long H5E_CANTUNPROTECT = H5E_CANTUNPROTECT_g();
    /** Minor error codes - Cache related errors - Unable to mark metadata as unserialized */
    public static final long H5E_CANTUNSERIALIZE = H5E_CANTUNSERIALIZE_g();
    /** Minor error codes - Heap errors - Can't update object */
    public static final long H5E_CANTUPDATE = H5E_CANTUPDATE_g();
    /** Generic low-level file I/O errors - Close failed */
    public static final long H5E_CLOSEERROR = H5E_CLOSEERROR_g();
    /** Minor error codes - Group related errors - Name component is too long */
    public static final long H5E_COMPLEN = H5E_COMPLEN_g();
    /** Major error codes - API Context */
    public static final long H5E_CONTEXT = H5E_CONTEXT_g();
    /** Major error codes - Dataset */
    public static final long H5E_DATASET = H5E_DATASET_g();
    /** Major error codes - Dataspace */
    public static final long H5E_DATASPACE = H5E_DATASPACE_g();
    /** Major error codes - Datatype */
    public static final long H5E_DATATYPE = H5E_DATATYPE_g();
    /** Value for the default error stack */
    public static final long H5E_DEFAULT = H5E_DEFAULT();
    /** Minor error codes - Property list errors - Duplicate class name in parent class */
    public static final long H5E_DUPCLASS = H5E_DUPCLASS_g();
    /** Major error codes - Extensible Array */
    public static final long H5E_EARRAY = H5E_EARRAY_g();
    /** Major error codes - External file list */
    public static final long H5E_EFL = H5E_EFL_g();
    /** Major error codes - Error API */
    public static final long H5E_ERROR = H5E_ERROR_g();
    /** Minor error codes - B-tree related errors - Object already exists */
    public static final long H5E_EXISTS = H5E_EXISTS_g();
    /** Major error codes - Fixed Array */
    public static final long H5E_FARRAY = H5E_FARRAY_g();
    /** Generic low-level file I/O errors - File control (fcntl) failed */
    public static final long H5E_FCNTL = H5E_FCNTL_g();
    /** Major error codes - File accessibility */
    public static final long H5E_FILE = H5E_FILE_g();
    /** Minor error codes - File accessibility errors - File already exists */
    public static final long H5E_FILEEXISTS = H5E_FILEEXISTS_g();
    /** Minor error codes - File accessibility errors - File already open */
    public static final long H5E_FILEOPEN = H5E_FILEOPEN_g();
    /** Major error codes - Free Space Manager */
    public static final long H5E_FSPACE = H5E_FSPACE_g();
    /** Major error codes - Function entry/exit */
    public static final long H5E_FUNC = H5E_FUNC_g();
    /** Major error codes - Heap */
    public static final long H5E_HEAP = H5E_HEAP_g();
    /** Minor error codes - Dataspace errors - Internal states are inconsistent */
    public static final long H5E_INCONSISTENTSTATE = H5E_INCONSISTENTSTATE_g();
    /** Major error codes - Internal error (too specific to document in detail) */
    public static final long H5E_INTERNAL = H5E_INTERNAL_g();
    /** Major error codes - Low-level I/O */
    public static final long H5E_IO = H5E_IO_g();
    /** Major error codes - Links */
    public static final long H5E_LINK = H5E_LINK_g();
    /** Minor error codes - Object header related errors - Bad object header link count */
    public static final long H5E_LINKCOUNT = H5E_LINKCOUNT_g();
    /** Minor error codes - Cache related errors - Failure in the cache logging framework */
    public static final long H5E_LOGGING = H5E_LOGGING_g();
    /** Major error codes - Map */
    public static final long H5E_MAP = H5E_MAP_g();
    /** Minor error codes - File accessibility errors - File mount error */
    public static final long H5E_MOUNT = H5E_MOUNT_g();
    /** Minor error codes - Parallel MPI - Some MPI function failed */
    public static final long H5E_MPI = H5E_MPI_g();
    /** Minor error codes - Parallel MPI - MPI Error String */
    public static final long H5E_MPIERRSTR = H5E_MPIERRSTR_g();
    /** Minor error codes - Link related errors - Too many soft links in path */
    public static final long H5E_NLINKS = H5E_NLINKS_g();
    /** Minor error codes - Parallel MPI - Can't perform independent IO */
    public static final long H5E_NO_INDEPENDENT = H5E_NO_INDEPENDENT_g();
    /** I/O pipeline errors - Filter present but encoding disabled */
    public static final long H5E_NOENCODER = H5E_NOENCODER_g();
    /** I/O pipeline errors - Requested filter is not available */
    public static final long H5E_NOFILTER = H5E_NOFILTER_g();
    /** Minor error codes - Object ID related errors - Out of IDs for group */
    public static final long H5E_NOIDS = H5E_NOIDS_g();
    /** Major error codes - No error */
    public static final long H5E_NONE_MAJOR = H5E_NONE_MAJOR_g();
    /** No error */
    public static final long H5E_NONE_MINOR = H5E_NONE_MINOR_g();
    /** Minor error codes - Resource errors - No space available for allocation */
    public static final long H5E_NOSPACE = H5E_NOSPACE_g();
    /** Minor error codes - Cache related errors - Metadata not currently cached */
    public static final long H5E_NOTCACHED = H5E_NOTCACHED_g();
    /** Minor error codes - B-tree related errors - Object not found */
    public static final long H5E_NOTFOUND = H5E_NOTFOUND_g();
    /** Minor error codes - File accessibility errors - Not an HDF5 file */
    public static final long H5E_NOTHDF5 = H5E_NOTHDF5_g();
    /** Minor error codes - Link related errors - Link class not registered */
    public static final long H5E_NOTREGISTERED = H5E_NOTREGISTERED_g();
    /** Minor error codes - Resource errors - Object is already open */
    public static final long H5E_OBJOPEN = H5E_OBJOPEN_g();
    /** Major error codes - Object header */
    public static final long H5E_OHDR = H5E_OHDR_g();
    /** Minor error codes - Plugin errors - Can't open directory or file */
    public static final long H5E_OPENERROR = H5E_OPENERROR_g();
    /** Generic low-level file I/O errors - Address overflowed */
    public static final long H5E_OVERFLOW = H5E_OVERFLOW_g();
    /** Major error codes - Page Buffering */
    public static final long H5E_PAGEBUF = H5E_PAGEBUF_g();
    /** Minor error codes - Group related errors - Problem with path to object */
    public static final long H5E_PATH = H5E_PATH_g();
    /** Major error codes - Data filters */
    public static final long H5E_PLINE = H5E_PLINE_g();
    /** Major error codes - Property lists */
    public static final long H5E_PLIST = H5E_PLIST_g();
    /** Major error codes - Plugin for dynamically loaded library */
    public static final long H5E_PLUGIN = H5E_PLUGIN_g();
    /** Minor error codes - Cache related errors - Protected metadata error */
    public static final long H5E_PROTECT = H5E_PROTECT_g();
    /** Generic low-level file I/O errors - Read failed */
    public static final long H5E_READERROR = H5E_READERROR_g();
    /** Major error codes - References */
    public static final long H5E_REFERENCE = H5E_REFERENCE_g();
    /** Major error codes - Resource unavailable */
    public static final long H5E_RESOURCE = H5E_RESOURCE_g();
    /** Major error codes - Reference Counted Strings */
    public static final long H5E_RS = H5E_RS_g();
    /** Generic low-level file I/O errors - Seek failed */
    public static final long H5E_SEEKERROR = H5E_SEEKERROR_g();
    /** Minor error codes - Property list errors - Disallowed operation */
    public static final long H5E_SETDISALLOWED = H5E_SETDISALLOWED_g();
    /** I/O pipeline errors - Error from filter 'set local' callback */
    public static final long H5E_SETLOCAL = H5E_SETLOCAL_g();
    /** Major error codes - Skip Lists */
    public static final long H5E_SLIST = H5E_SLIST_g();
    /** Major error codes - Shared Object Header Messages */
    public static final long H5E_SOHM = H5E_SOHM_g();
    /** Major error codes - Data storage */
    public static final long H5E_STORAGE = H5E_STORAGE_g();
    /** Major error codes - Symbol table */
    public static final long H5E_SYM = H5E_SYM_g();
    /** Minor error codes - System level errors - System error message */
    public static final long H5E_SYSERRSTR = H5E_SYSERRSTR_g();
    /** Minor error codes - Cache related errors - Internal error detected */
    public static final long H5E_SYSTEM = H5E_SYSTEM_g();
    /** Minor error codes - Link related errors - Link traversal failure */
    public static final long H5E_TRAVERSE = H5E_TRAVERSE_g();
    /** Minor error codes - File accessibility errors - File has been truncated */
    public static final long H5E_TRUNCATED = H5E_TRUNCATED_g();
    /** Major error codes - Ternary Search Trees */
    public static final long H5E_TST = H5E_TST_g();
    /** Minor error codes - Argument errors - Information is uinitialized */
    public static final long H5E_UNINITIALIZED = H5E_UNINITIALIZED_g();
    /** Minor error codes - Argument errors - Feature is unsupported */
    public static final long H5E_UNSUPPORTED = H5E_UNSUPPORTED_g();
    /** Minor error codes - Object header related errors - Wrong version number */
    public static final long H5E_VERSION = H5E_VERSION_g();
    /** Major error codes - Virtual File Layer */
    public static final long H5E_VFL = H5E_VFL_g();
    /** Major error codes - Virtual Object Layer */
    public static final long H5E_VOL = H5E_VOL_g();
    /** Error stack traversal direction - begin at API function, end deep */
    public static final long H5E_WALK_DOWNWARD = H5E_WALK_DOWNWARD();
    /** Error stack traversal direction - begin deep, end at API function */
    public static final long H5E_WALK_UPWARD = H5E_WALK_UPWARD();
    /** Generic low-level file I/O errors - Write failed */
    public static final long H5E_WRITEERROR = H5E_WRITEERROR_g();

    /** */
    private static final int H5ES_STATUS_IN_PROGRESS = H5ES_STATUS_IN_PROGRESS();
    /** */
    private static final int H5ES_STATUS_SUCCEED = H5ES_STATUS_SUCCEED();
    /** */
    private static final int H5ES_STATUS_FAIL = H5ES_STATUS_FAIL();

    /** */
    public static final int H5F_ACC_CREAT = H5F_ACC_CREAT();
    /** */
    public static final int H5F_ACC_EXCL = H5F_ACC_EXCL();
    /** */
    public static final int H5F_ACC_RDONLY = H5F_ACC_RDONLY();
    /** */
    public static final int H5F_ACC_RDWR = H5F_ACC_RDWR();
    /** */
    public static final int H5F_ACC_TRUNC = H5F_ACC_TRUNC();
    /** */
    public static final int H5F_ACC_DEFAULT = H5F_ACC_DEFAULT();
    /** */
    public static final int H5F_ACC_SWMR_READ = H5F_ACC_SWMR_READ();
    /** */
    public static final int H5F_ACC_SWMR_WRITE = H5F_ACC_SWMR_WRITE();
    /** */
    public static final int H5F_CLOSE_DEFAULT = H5F_CLOSE_DEFAULT();
    /** */
    public static final int H5F_CLOSE_SEMI = H5F_CLOSE_SEMI();
    /** */
    public static final int H5F_CLOSE_STRONG = H5F_CLOSE_STRONG();
    /** */
    public static final int H5F_CLOSE_WEAK = H5F_CLOSE_WEAK();
    /** */
    public static final int H5F_LIBVER_ERROR = H5F_LIBVER_ERROR();
    /** */
    public static final int H5F_LIBVER_EARLIEST = H5F_LIBVER_EARLIEST();
    /** */
    public static final int H5F_LIBVER_V18 = H5F_LIBVER_V18();
    /** */
    public static final int H5F_LIBVER_V110 = H5F_LIBVER_V110();
    /** */
    public static final int H5F_LIBVER_V112 = H5F_LIBVER_V112();
    /** */
    public static final int H5F_LIBVER_V114 = H5F_LIBVER_V114();
    /** */
    public static final int H5F_LIBVER_V200 = H5F_LIBVER_V200();
    /** */
    public static final int H5F_LIBVER_LATEST = H5F_LIBVER_LATEST();
    /** */
    public static final int H5F_LIBVER_NBOUNDS = H5F_LIBVER_NBOUNDS();
    /** */
    public static final int H5F_OBJ_ALL = H5F_OBJ_ALL();
    /** */
    public static final int H5F_OBJ_ATTR = H5F_OBJ_ATTR();
    /** */
    public static final int H5F_OBJ_DATASET = H5F_OBJ_DATASET();
    /** */
    public static final int H5F_OBJ_DATATYPE = H5F_OBJ_DATATYPE();
    /** */
    public static final int H5F_OBJ_FILE = H5F_OBJ_FILE();
    /** */
    public static final int H5F_OBJ_GROUP = H5F_OBJ_GROUP();
    /** */
    public static final int H5F_OBJ_LOCAL = H5F_OBJ_LOCAL();
    /** */
    public static final int H5F_SCOPE_GLOBAL = H5F_SCOPE_GLOBAL();
    /** */
    public static final int H5F_SCOPE_LOCAL = H5F_SCOPE_LOCAL();
    /** */
    public static final long H5F_UNLIMITED = H5F_UNLIMITED();

    /** */
    public static final int H5F_FSPACE_STRATEGY_FSM_AGGR = H5F_FSPACE_STRATEGY_FSM_AGGR();
    /** */
    public static final int H5F_FSPACE_STRATEGY_AGGR = H5F_FSPACE_STRATEGY_AGGR();
    /** */
    public static final int H5F_FSPACE_STRATEGY_PAGE = H5F_FSPACE_STRATEGY_PAGE();
    /** */
    public static final int H5F_FSPACE_STRATEGY_NONE = H5F_FSPACE_STRATEGY_NONE();
    /** */
    public static final int H5F_FSPACE_STRATEGY_NTYPES = H5F_FSPACE_STRATEGY_NTYPES();

    /** */
    public static final long H5FD_CORE = H5FD_CORE_id_g();
    /** */
    public static final long H5FD_DIRECT = getH5FD_DIRECT();
    /** */
    public static final long H5FD_FAMILY = H5FD_FAMILY_id_g();
    /** */
    public static final long H5FD_LOG = H5FD_LOG_id_g();
    /** */
    public static final long H5FD_MPIO = getH5FD_MPIO();
    /** */
    public static final long H5FD_MULTI = H5FD_MULTI_id_g();
    /** */
    public static final long H5FD_ONION = H5FD_ONION_id_g();
    /** */
    public static final long H5FD_SEC2 = H5FD_SEC2_id_g();
    /** */
    public static final long H5FD_SPLITTER = H5FD_SPLITTER_id_g();
    /** */
    public static final long H5FD_STDIO = H5FD_STDIO_id_g();
    /** */
    public static final long H5FD_WINDOWS = H5FD_SEC2_id_g();
    /** */
    public static final long H5FD_ROS3 = getH5FD_ROS3();
    /** */
    public static final long H5FD_HDFS = getH5FD_HDFS();
    /** */
    public static final long H5FD_MIRROR = getH5FD_MIRROR();
    /** */
    public static final int H5FD_LOG_LOC_READ = H5FD_LOG_LOC_READ();
    /** */
    public static final int H5FD_LOG_LOC_WRITE = H5FD_LOG_LOC_WRITE();
    /** */
    public static final int H5FD_LOG_LOC_SEEK = H5FD_LOG_LOC_SEEK();
    /** */
    public static final int H5FD_LOG_LOC_IO = H5FD_LOG_LOC_IO();
    /** */
    public static final int H5FD_LOG_FILE_READ = H5FD_LOG_FILE_READ();
    /** */
    public static final int H5FD_LOG_FILE_WRITE = H5FD_LOG_FILE_WRITE();
    /** */
    public static final int H5FD_LOG_FILE_IO = H5FD_LOG_FILE_IO();
    /** */
    public static final int H5FD_LOG_FLAVOR = H5FD_LOG_FLAVOR();
    /** */
    public static final int H5FD_LOG_NUM_READ = H5FD_LOG_NUM_READ();
    /** */
    public static final int H5FD_LOG_NUM_WRITE = H5FD_LOG_NUM_WRITE();
    /** */
    public static final int H5FD_LOG_NUM_SEEK = H5FD_LOG_NUM_SEEK();
    /** */
    public static final int H5FD_LOG_NUM_TRUNCATE = H5FD_LOG_NUM_TRUNCATE();
    /** */
    public static final int H5FD_LOG_NUM_IO = H5FD_LOG_NUM_IO();
    /** */
    public static final int H5FD_LOG_TIME_OPEN = H5FD_LOG_TIME_OPEN();
    /** */
    public static final int H5FD_LOG_TIME_STAT = H5FD_LOG_TIME_STAT();
    /** */
    public static final int H5FD_LOG_TIME_READ = H5FD_LOG_TIME_READ();
    /** */
    public static final int H5FD_LOG_TIME_WRITE = H5FD_LOG_TIME_WRITE();
    /** */
    public static final int H5FD_LOG_TIME_SEEK = H5FD_LOG_TIME_SEEK();
    /** */
    public static final int H5FD_LOG_TIME_CLOSE = H5FD_LOG_TIME_CLOSE();
    /** */
    public static final int H5FD_LOG_TIME_IO = H5FD_LOG_TIME_IO();
    /** */
    public static final int H5FD_LOG_ALLOC = H5FD_LOG_ALLOC();
    /** */
    public static final int H5FD_LOG_ALL = H5FD_LOG_ALL();
    /** */
    public static final int H5FD_MEM_NOLIST = H5FD_MEM_NOLIST();
    /** */
    public static final int H5FD_MEM_DEFAULT = H5FD_MEM_DEFAULT();
    /** */
    public static final int H5FD_MEM_SUPER = H5FD_MEM_SUPER();
    /** */
    public static final int H5FD_MEM_BTREE = H5FD_MEM_BTREE();
    /** */
    public static final int H5FD_MEM_DRAW = H5FD_MEM_DRAW();
    /** */
    public static final int H5FD_MEM_GHEAP = H5FD_MEM_GHEAP();
    /** */
    public static final int H5FD_MEM_LHEAP = H5FD_MEM_LHEAP();
    /** */
    public static final int H5FD_MEM_OHDR = H5FD_MEM_OHDR();
    /** */
    public static final int H5FD_MEM_NTYPES = H5FD_MEM_NTYPES();
    /** */
    public static final BigInteger H5FD_BIG_MEM_NTYPES =
        new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(H5FD_MEM_NTYPES()).array());
    /** */
    public static final BigInteger H5FD_BIG_MEM_NTYPES_MINUS =
        new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(H5FD_MEM_NTYPES() - 1L).array());
    /** */
    public static final BigInteger H5FD_BIG_HADDR_MAX =
        new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(HADDR_MAX()).array());
    /** */
    public static final long H5FD_DEFAULT_HADDR_SIZE =
        H5FD_BIG_HADDR_MAX.divide(H5FD_BIG_MEM_NTYPES).longValue();
    /** */
    public static final long H5FD_MEM_DEFAULT_SIZE = 0L;
    /** */
    public static final long H5FD_MEM_DEFAULT_SUPER_SIZE = 0L;
    /** */
    public static final long H5FD_MEM_DEFAULT_BTREE_SIZE =
        H5FD_BIG_HADDR_MAX.divide(H5FD_BIG_MEM_NTYPES_MINUS)
            .multiply(new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(1).array()))
            .longValue();
    /** */
    public static final long H5FD_MEM_DEFAULT_DRAW_SIZE =
        H5FD_BIG_HADDR_MAX.divide(H5FD_BIG_MEM_NTYPES_MINUS)
            .multiply(new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(2).array()))
            .longValue();
    /** */
    public static final long H5FD_MEM_DEFAULT_GHEAP_SIZE =
        H5FD_BIG_HADDR_MAX.divide(H5FD_BIG_MEM_NTYPES_MINUS)
            .multiply(new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(3).array()))
            .longValue();
    /** */
    public static final long H5FD_MEM_DEFAULT_LHEAP_SIZE =
        H5FD_BIG_HADDR_MAX.divide(H5FD_BIG_MEM_NTYPES_MINUS)
            .multiply(new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(4).array()))
            .longValue();
    /** */
    public static final long H5FD_MEM_DEFAULT_OHDR_SIZE =
        H5FD_BIG_HADDR_MAX.divide(H5FD_BIG_MEM_NTYPES_MINUS)
            .multiply(new BigInteger(1, ByteBuffer.allocate(Long.SIZE / Byte.SIZE).putLong(5).array()))
            .longValue();

    //    public static final int H5G_DATASET = H5G_DATASET();
    //    public static final int H5G_GROUP = H5G_GROUP();
    //    public static final int H5G_LINK = H5G_LINK();
    //    public static final int H5G_UDLINK = H5G_UDLINK();
    //    public static final int H5G_LINK_ERROR = H5G_LINK_ERROR();
    //    public static final int H5G_LINK_HARD = H5G_LINK_HARD();
    //    public static final int H5G_LINK_SOFT = H5G_LINK_SOFT();
    //    public static final int H5G_NLIBTYPES = H5G_NLIBTYPES();
    //    public static final int H5G_NTYPES = H5G_NTYPES();
    //    public static final int H5G_NUSERTYPES = H5G_NUSERTYPES();
    //    public static final int H5G_RESERVED_5 = H5G_RESERVED_5();
    //    public static final int H5G_RESERVED_6 = H5G_RESERVED_6();
    //    public static final int H5G_RESERVED_7 = H5G_RESERVED_7();
    //    public static final int H5G_SAME_LOC = H5G_SAME_LOC();
    //    public static final int H5G_TYPE = H5G_TYPE();
    //    public static final int H5G_UNKNOWN = H5G_UNKNOWN();

    /** */
    public static final int H5G_STORAGE_TYPE_UNKNOWN = H5G_STORAGE_TYPE_UNKNOWN();
    /** */
    public static final int H5G_STORAGE_TYPE_SYMBOL_TABLE = H5G_STORAGE_TYPE_SYMBOL_TABLE();
    /** */
    public static final int H5G_STORAGE_TYPE_COMPACT = H5G_STORAGE_TYPE_COMPACT();
    /** */
    public static final int H5G_STORAGE_TYPE_DENSE = H5G_STORAGE_TYPE_DENSE();

    /** */
    public static final int H5I_ATTR = H5I_ATTR();
    /** */
    public static final int H5I_BADID = H5I_BADID();
    /** */
    public static final int H5I_DATASET = H5I_DATASET();
    /** */
    public static final int H5I_DATASPACE = H5I_DATASPACE();
    /** */
    public static final int H5I_DATATYPE = H5I_DATATYPE();
    /** */
    public static final int H5I_ERROR_CLASS = H5I_ERROR_CLASS();
    /** */
    public static final int H5I_ERROR_MSG = H5I_ERROR_MSG();
    /** */
    public static final int H5I_ERROR_STACK = H5I_ERROR_STACK();
    /** */
    public static final int H5I_FILE = H5I_FILE();
    /** */
    public static final int H5I_GENPROP_CLS = H5I_GENPROP_CLS();
    /** */
    public static final int H5I_GENPROP_LST = H5I_GENPROP_LST();
    /** */
    public static final int H5I_GROUP = H5I_GROUP();
    /** */
    public static final int H5I_INVALID_HID = H5I_INVALID_HID();
    /** */
    public static final int H5I_NTYPES = H5I_NTYPES();
    /** */
    public static final int H5I_UNINIT = H5I_UNINIT();
    /** */
    public static final int H5I_VFL = H5I_VFL();
    /** */
    public static final int H5I_VOL = H5I_VOL();

    /** */
    public static final int H5L_TYPE_ERROR = H5L_TYPE_ERROR();
    /** */
    public static final int H5L_TYPE_HARD = H5L_TYPE_HARD();
    /** */
    public static final int H5L_TYPE_SOFT = H5L_TYPE_SOFT();
    /** */
    public static final int H5L_TYPE_EXTERNAL = H5L_TYPE_EXTERNAL();
    /** */
    public static final int H5L_TYPE_MAX = H5L_TYPE_MAX();

    /** */
    public static final int H5O_COPY_SHALLOW_HIERARCHY_FLAG = H5O_COPY_SHALLOW_HIERARCHY_FLAG();
    /** */
    public static final int H5O_COPY_EXPAND_SOFT_LINK_FLAG = H5O_COPY_EXPAND_SOFT_LINK_FLAG();
    /** */
    public static final int H5O_COPY_EXPAND_EXT_LINK_FLAG = H5O_COPY_EXPAND_EXT_LINK_FLAG();
    /** */
    public static final int H5O_COPY_EXPAND_REFERENCE_FLAG = H5O_COPY_EXPAND_REFERENCE_FLAG();
    /** */
    public static final int H5O_COPY_WITHOUT_ATTR_FLAG = H5O_COPY_WITHOUT_ATTR_FLAG();
    /** */
    public static final int H5O_COPY_PRESERVE_NULL_FLAG = H5O_COPY_PRESERVE_NULL_FLAG();
    /** */
    public static final int H5O_INFO_BASIC = H5O_INFO_BASIC();
    /** */
    public static final int H5O_INFO_TIME = H5O_INFO_TIME();
    /** */
    public static final int H5O_INFO_NUM_ATTRS = H5O_INFO_NUM_ATTRS();
    /** */
    public static final int H5O_INFO_ALL = H5O_INFO_ALL();
    /** */
    public static final int H5O_NATIVE_INFO_HDR = H5O_NATIVE_INFO_HDR();
    /** */
    public static final int H5O_NATIVE_INFO_META_SIZE = H5O_NATIVE_INFO_META_SIZE();
    /** */
    public static final int H5O_NATIVE_INFO_ALL = H5O_NATIVE_INFO_ALL();
    /** */
    public static final int H5O_SHMESG_NONE_FLAG = H5O_SHMESG_NONE_FLAG();
    /** */
    public static final int H5O_SHMESG_SDSPACE_FLAG = H5O_SHMESG_SDSPACE_FLAG();
    /** */
    public static final int H5O_SHMESG_DTYPE_FLAG = H5O_SHMESG_DTYPE_FLAG();
    /** */
    public static final int H5O_SHMESG_FILL_FLAG = H5O_SHMESG_FILL_FLAG();
    /** */
    public static final int H5O_SHMESG_PLINE_FLAG = H5O_SHMESG_PLINE_FLAG();
    /** */
    public static final int H5O_SHMESG_ATTR_FLAG = H5O_SHMESG_ATTR_FLAG();
    /** */
    public static final int H5O_SHMESG_ALL_FLAG = H5O_SHMESG_ALL_FLAG();
    /** */
    public static final int H5O_TYPE_UNKNOWN = H5O_TYPE_UNKNOWN();
    /** */
    public static final int H5O_TYPE_GROUP = H5O_TYPE_GROUP();
    /** */
    public static final int H5O_TYPE_DATASET = H5O_TYPE_DATASET();
    /** */
    public static final int H5O_TYPE_NAMED_DATATYPE = H5O_TYPE_NAMED_DATATYPE();
    /** */
    public static final int H5O_TYPE_NTYPES = H5O_TYPE_NTYPES();
    /** */
    public static final int H5O_MAX_TOKEN_SIZE = H5O_MAX_TOKEN_SIZE();
    /** H5O_token_t is derived from the MemorySegment */
    public static final H5O_token_t H5O_TOKEN_UNDEF = new H5O_token_t(H5O_TOKEN_UNDEF_g());

    /** */
    public static final long H5P_ROOT = H5P_CLS_ROOT_ID_g();
    /** */
    public static final long H5P_OBJECT_CREATE = H5P_CLS_OBJECT_CREATE_ID_g();
    /** */
    public static final long H5P_FILE_CREATE = H5P_CLS_FILE_CREATE_ID_g();
    /** */
    public static final long H5P_FILE_ACCESS = H5P_CLS_FILE_ACCESS_ID_g();
    /** */
    public static final long H5P_DATASET_CREATE = H5P_CLS_DATASET_CREATE_ID_g();
    /** */
    public static final long H5P_DATASET_ACCESS = H5P_CLS_DATASET_ACCESS_ID_g();
    /** */
    public static final long H5P_DATASET_XFER = H5P_CLS_DATASET_XFER_ID_g();
    /** */
    public static final long H5P_FILE_MOUNT = H5P_CLS_FILE_MOUNT_ID_g();
    /** */
    public static final long H5P_GROUP_CREATE = H5P_CLS_GROUP_CREATE_ID_g();
    /** */
    public static final long H5P_GROUP_ACCESS = H5P_CLS_GROUP_ACCESS_ID_g();
    /** */
    public static final long H5P_DATATYPE_CREATE = H5P_CLS_DATATYPE_CREATE_ID_g();
    /** */
    public static final long H5P_DATATYPE_ACCESS = H5P_CLS_DATATYPE_ACCESS_ID_g();
    /** */
    public static final long H5P_MAP_CREATE = H5P_CLS_MAP_CREATE_ID_g();
    /** */
    public static final long H5P_MAP_ACCESS = H5P_CLS_MAP_ACCESS_ID_g();
    /** */
    public static final long H5P_STRING_CREATE = H5P_CLS_STRING_CREATE_ID_g();
    /** */
    public static final long H5P_ATTRIBUTE_CREATE = H5P_CLS_ATTRIBUTE_CREATE_ID_g();
    /** */
    public static final long H5P_ATTRIBUTE_ACCESS = H5P_CLS_ATTRIBUTE_ACCESS_ID_g();
    /** */
    public static final long H5P_OBJECT_COPY = H5P_CLS_OBJECT_COPY_ID_g();
    /** */
    public static final long H5P_LINK_CREATE = H5P_CLS_LINK_CREATE_ID_g();
    /** */
    public static final long H5P_LINK_ACCESS = H5P_CLS_LINK_ACCESS_ID_g();
    /** */
    public static final long H5P_VOL_INITIALIZE = H5P_CLS_VOL_INITIALIZE_ID_g();
    /** */
    public static final long H5P_REFERENCE_ACCESS = H5P_CLS_REFERENCE_ACCESS_ID_g();
    /** */
    public static final long H5P_FILE_CREATE_DEFAULT = H5P_LST_FILE_CREATE_ID_g();
    /** */
    public static final long H5P_FILE_ACCESS_DEFAULT = H5P_LST_FILE_ACCESS_ID_g();
    /** */
    public static final long H5P_DATASET_CREATE_DEFAULT = H5P_LST_DATASET_CREATE_ID_g();
    /** */
    public static final long H5P_DATASET_ACCESS_DEFAULT = H5P_LST_DATASET_ACCESS_ID_g();
    /** */
    public static final long H5P_DATASET_XFER_DEFAULT = H5P_LST_DATASET_XFER_ID_g();
    /** */
    public static final long H5P_FILE_MOUNT_DEFAULT = H5P_LST_FILE_MOUNT_ID_g();
    /** */
    public static final long H5P_GROUP_CREATE_DEFAULT = H5P_LST_GROUP_CREATE_ID_g();
    /** */
    public static final long H5P_GROUP_ACCESS_DEFAULT = H5P_LST_GROUP_ACCESS_ID_g();
    /** */
    public static final long H5P_DATATYPE_CREATE_DEFAULT = H5P_LST_DATATYPE_CREATE_ID_g();
    /** */
    public static final long H5P_DATATYPE_ACCESS_DEFAULT = H5P_LST_DATATYPE_ACCESS_ID_g();
    /** */
    public static final long H5P_MAP_CREATE_DEFAULT = H5P_LST_MAP_CREATE_ID_g();
    /** */
    public static final long H5P_MAP_ACCESS_DEFAULT = H5P_LST_MAP_ACCESS_ID_g();
    /** */
    public static final long H5P_ATTRIBUTE_CREATE_DEFAULT = H5P_LST_ATTRIBUTE_CREATE_ID_g();
    /** */
    public static final long H5P_ATTRIBUTE_ACCESS_DEFAULT = H5P_LST_ATTRIBUTE_ACCESS_ID_g();
    /** */
    public static final long H5P_OBJECT_COPY_DEFAULT = H5P_LST_OBJECT_COPY_ID_g();
    /** */
    public static final long H5P_LINK_CREATE_DEFAULT = H5P_LST_LINK_CREATE_ID_g();
    /** */
    public static final long H5P_LINK_ACCESS_DEFAULT = H5P_LST_LINK_ACCESS_ID_g();
    /** */
    public static final long H5P_VOL_INITIALIZE_DEFAULT = H5P_LST_VOL_INITIALIZE_ID_g();
    /** */
    public static final int H5P_CRT_ORDER_TRACKED = H5P_CRT_ORDER_TRACKED();
    /** */
    public static final int H5P_CRT_ORDER_INDEXED = H5P_CRT_ORDER_INDEXED();
    /** */
    public static final long H5P_DEFAULT = H5P_DEFAULT();

    /** */
    public static final int H5PL_TYPE_ERROR = H5PL_TYPE_ERROR();
    /** */
    public static final int H5PL_TYPE_FILTER = H5PL_TYPE_FILTER();
    /** */
    public static final int H5PL_TYPE_VOL = H5PL_TYPE_VOL();
    /** */
    public static final int H5PL_TYPE_NONE = H5PL_TYPE_NONE();
    /** */
    public static final int H5PL_FILTER_PLUGIN = H5PL_FILTER_PLUGIN();
    /** */
    public static final int H5PL_VOL_PLUGIN = H5PL_VOL_PLUGIN();
    /** */
    public static final int H5PL_ALL_PLUGIN = H5PL_ALL_PLUGIN();

    /** */
    public static final int H5R_ATTR = H5R_ATTR();
    /** */
    public static final int H5R_BADTYPE = H5R_BADTYPE();
    /** */
    public static final int H5R_DATASET_REGION = H5R_DATASET_REGION();
    /** */
    public static final int H5R_DATASET_REGION1 = H5R_DATASET_REGION1();
    /** */
    public static final int H5R_DATASET_REGION2 = H5R_DATASET_REGION2();
    /** */
    public static final int H5R_MAXTYPE = H5R_MAXTYPE();
    /** */
    public static final long H5R_DSET_REG_REF_BUF_SIZE = H5R_DSET_REG_REF_BUF_SIZE();
    /** */
    public static final long H5R_OBJ_REF_BUF_SIZE = H5R_OBJ_REF_BUF_SIZE();
    /** */
    public static final int H5R_REF_BUF_SIZE = H5R_REF_BUF_SIZE();
    /** */
    public static final int H5R_OBJECT = H5R_OBJECT();
    /** */
    public static final int H5R_OBJECT1 = H5R_OBJECT1();
    /** */
    public static final int H5R_OBJECT2 = H5R_OBJECT2();

    /** Define atomic datatypes */
    public static final int H5S_ALL = H5S_ALL();
    /** Define user-level maximum number of dimensions */
    public static final int H5S_MAX_RANK = H5S_MAX_RANK();
    /** Different types of dataspaces - error */
    public static final int H5S_NO_CLASS = H5S_NO_CLASS();
    /** Different types of dataspaces - null dataspace */
    public static final int H5S_NULL = H5S_NULL();
    /** Different types of dataspaces - scalar variable */
    public static final int H5S_SCALAR = H5S_SCALAR();
    /** Enumerated type for the type of selection - Entire extent selected */
    public static final int H5S_SEL_ALL = H5S_SEL_ALL();
    /** Enumerated type for the type of selection - Error */
    public static final int H5S_SEL_ERROR = H5S_SEL_ERROR();
    /** Enumerated type for the type of selection - Hyperslab selected */
    public static final int H5S_SEL_HYPERSLABS = H5S_SEL_HYPERSLABS();
    /** Enumerated type for the type of selection - LAST */
    public static final int H5S_SEL_N = H5S_SEL_N();
    /** Enumerated type for the type of selection - Nothing selected */
    public static final int H5S_SEL_NONE = H5S_SEL_NONE();
    /** Enumerated type for the type of selection - Points / elements selected */
    public static final int H5S_SEL_POINTS = H5S_SEL_POINTS();
    /** Different ways of combining selections - Binary "and" operation for hyperslabs */
    public static final int H5S_SELECT_AND = H5S_SELECT_AND();
    /** Different ways of combining selections - Append elements to end of point selection */
    public static final int H5S_SELECT_APPEND = H5S_SELECT_APPEND();
    /** Different ways of combining selections - Invalid upper bound on selection operations */
    public static final int H5S_SELECT_INVALID = H5S_SELECT_INVALID();
    /** Different ways of combining selections - error */
    public static final int H5S_SELECT_NOOP = H5S_SELECT_NOOP();
    /** Different ways of combining selections - Binary "not" operation for hyperslabs */
    public static final int H5S_SELECT_NOTA = H5S_SELECT_NOTA();
    /** Different ways of combining selections - Binary "not" operation for hyperslabs */
    public static final int H5S_SELECT_NOTB = H5S_SELECT_NOTB();
    /** Different ways of combining selections - Binary "or" operation for hyperslabs */
    public static final int H5S_SELECT_OR = H5S_SELECT_OR();
    /** Different ways of combining selections - Prepend elements to beginning of point selection */
    public static final int H5S_SELECT_PREPEND = H5S_SELECT_PREPEND();
    /** Different ways of combining selections - Select "set" operation */
    public static final int H5S_SELECT_SET = H5S_SELECT_SET();
    /** Different ways of combining selections - Binary "xor" operation for hyperslabs */
    public static final int H5S_SELECT_XOR = H5S_SELECT_XOR();
    /** Different types of dataspaces - simple dataspace */
    public static final int H5S_SIMPLE = H5S_SIMPLE();
    /** Define atomic datatypes */
    public static final long H5S_UNLIMITED = H5S_UNLIMITED();

    /** */
    // public static final long H5T_ALPHA_B16 = H5T_ALPHA_B16;
    /** */
    // public static final long H5T_ALPHA_B32 = H5T_ALPHA_B32;
    /** */
    // public static final long H5T_ALPHA_B64 = H5T_ALPHA_B64;
    /** */
    // public static final long H5T_ALPHA_B8 = H5T_ALPHA_B8;
    /** */
    // public static final long H5T_ALPHA_F32 = H5T_ALPHA_F32;
    /** */
    // public static final long H5T_ALPHA_F64 = H5T_ALPHA_F64;
    /** */
    // public static final long H5T_ALPHA_I16 = H5T_ALPHA_I16;
    /** */
    // public static final long H5T_ALPHA_I32 = H5T_ALPHA_I32;
    /** */
    // public static final long H5T_ALPHA_I64 = H5T_ALPHA_I64;
    /** */
    // public static final long H5T_ALPHA_I8 = H5T_ALPHA_I8;
    /** */
    // public static final long H5T_ALPHA_U16 = H5T_ALPHA_U16;
    /** */
    // public static final long H5T_ALPHA_U32 = H5T_ALPHA_U32;
    /** */
    // public static final long H5T_ALPHA_U64 = H5T_ALPHA_U64;
    /** */
    // public static final long H5T_ALPHA_U8 = H5T_ALPHA_U8;
    /** */
    public static final int H5T_ARRAY = H5T_ARRAY();
    /** */
    public static final int H5T_BITFIELD = H5T_BITFIELD();
    /** */
    public static final int H5T_BKG_NO = H5T_BKG_NO();
    /** */
    public static final int H5T_BKG_YES = H5T_BKG_YES();
    /** */
    public static final long H5T_C_S1 = H5T_C_S1_g();
    /** */
    public static final int H5T_COMPLEX = H5T_COMPLEX();
    /** */
    public static final int H5T_COMPOUND = H5T_COMPOUND();
    /** */
    public static final int H5T_CONV_CONV = H5T_CONV_CONV();
    /** */
    public static final int H5T_CONV_FREE = H5T_CONV_FREE();
    /** */
    public static final int H5T_CONV_INIT = H5T_CONV_INIT();
    /** */
    public static final long H5T_COMPLEX_IEEE_F16BE = H5T_COMPLEX_IEEE_F16BE_g();
    /** */
    public static final long H5T_COMPLEX_IEEE_F16LE = H5T_COMPLEX_IEEE_F16LE_g();
    /** */
    public static final long H5T_COMPLEX_IEEE_F32BE = H5T_COMPLEX_IEEE_F32BE_g();
    /** */
    public static final long H5T_COMPLEX_IEEE_F32LE = H5T_COMPLEX_IEEE_F32LE_g();
    /** */
    public static final long H5T_COMPLEX_IEEE_F64BE = H5T_COMPLEX_IEEE_F64BE_g();
    /** */
    public static final long H5T_COMPLEX_IEEE_F64LE = H5T_COMPLEX_IEEE_F64LE_g();
    /** */
    public static final int H5T_CSET_ERROR = H5T_CSET_ERROR();
    /** */
    public static final int H5T_CSET_ASCII = H5T_CSET_ASCII();
    /** */
    public static final int H5T_CSET_UTF8 = H5T_CSET_UTF8();
    /** */
    public static final int H5T_CSET_RESERVED_10 = H5T_CSET_RESERVED_10();
    /** */
    public static final int H5T_CSET_RESERVED_11 = H5T_CSET_RESERVED_11();
    /** */
    public static final int H5T_CSET_RESERVED_12 = H5T_CSET_RESERVED_12();
    /** */
    public static final int H5T_CSET_RESERVED_13 = H5T_CSET_RESERVED_13();
    /** */
    public static final int H5T_CSET_RESERVED_14 = H5T_CSET_RESERVED_14();
    /** */
    public static final int H5T_CSET_RESERVED_15 = H5T_CSET_RESERVED_15();
    /** */
    public static final int H5T_CSET_RESERVED_2 = H5T_CSET_RESERVED_2();
    /** */
    public static final int H5T_CSET_RESERVED_3 = H5T_CSET_RESERVED_3();
    /** */
    public static final int H5T_CSET_RESERVED_4 = H5T_CSET_RESERVED_4();
    /** */
    public static final int H5T_CSET_RESERVED_5 = H5T_CSET_RESERVED_5();
    /** */
    public static final int H5T_CSET_RESERVED_6 = H5T_CSET_RESERVED_6();
    /** */
    public static final int H5T_CSET_RESERVED_7 = H5T_CSET_RESERVED_7();
    /** */
    public static final int H5T_CSET_RESERVED_8 = H5T_CSET_RESERVED_8();
    /** */
    public static final int H5T_CSET_RESERVED_9 = H5T_CSET_RESERVED_9();
    /** */
    public static final int H5T_DIR_ASCEND = H5T_DIR_ASCEND();
    /** */
    public static final int H5T_DIR_DEFAULT = H5T_DIR_DEFAULT();
    /** */
    public static final int H5T_DIR_DESCEND = H5T_DIR_DESCEND();
    /** */
    public static final int H5T_ENUM = H5T_ENUM();
    /** */
    public static final int H5T_FLOAT = H5T_FLOAT();
    /** */
    public static final long H5T_FORTRAN_S1 = H5T_FORTRAN_S1_g();
    /** */
    public static final long H5T_IEEE_F16BE = H5T_IEEE_F16BE_g();
    /** */
    public static final long H5T_IEEE_F16LE = H5T_IEEE_F16LE_g();
    /** */
    public static final long H5T_IEEE_F32BE = H5T_IEEE_F32BE_g();
    /** */
    public static final long H5T_IEEE_F32LE = H5T_IEEE_F32LE_g();
    /** */
    public static final long H5T_IEEE_F64BE = H5T_IEEE_F64BE_g();
    /** */
    public static final long H5T_IEEE_F64LE = H5T_IEEE_F64LE_g();
    /** */
    public static final int H5T_INTEGER = H5T_INTEGER();
    /** */
    // public static final long H5T_INTEL_B16 = H5T_INTEL_B16;
    /** */
    // public static final long H5T_INTEL_B32 = H5T_INTEL_B32;
    /** */
    // public static final long H5T_INTEL_B64 = H5T_INTEL_B64;
    /** */
    // public static final long H5T_INTEL_B8 = H5T_INTEL_B8;
    /** */
    // public static final long H5T_INTEL_F32 = H5T_INTEL_F32;
    /** */
    // public static final long H5T_INTEL_F64 = H5T_INTEL_F64;
    /** */
    // public static final long H5T_INTEL_I16 = H5T_INTEL_I16;
    /** */
    // public static final long H5T_INTEL_I32 = H5T_INTEL_I32;
    /** */
    // public static final long H5T_INTEL_I64 = H5T_INTEL_I64;
    /** */
    // public static final long H5T_INTEL_I8 = H5T_INTEL_I8;
    /** */
    // public static final long H5T_INTEL_U16 = H5T_INTEL_U16;
    /** */
    // public static final long H5T_INTEL_U32 = H5T_INTEL_U32;
    /** */
    // public static final long H5T_INTEL_U64 = H5T_INTEL_U64;
    /** */
    // public static final long H5T_INTEL_U8 = H5T_INTEL_U8;
    /** */
    // public static final long H5T_MIPS_B16 = H5T_MIPS_B16;
    /** */
    // public static final long H5T_MIPS_B32 = H5T_MIPS_B32;
    /** */
    // public static final long H5T_MIPS_B64 = H5T_MIPS_B64;
    /** */
    // public static final long H5T_MIPS_B8 = H5T_MIPS_B8;
    /** */
    // public static final long H5T_MIPS_F32 = H5T_MIPS_F32;
    /** */
    // public static final long H5T_MIPS_F64 = H5T_MIPS_F64;
    /** */
    // public static final long H5T_MIPS_I16 = H5T_MIPS_I16;
    /** */
    // public static final long H5T_MIPS_I32 = H5T_MIPS_I32;
    /** */
    // public static final long H5T_MIPS_I64 = H5T_MIPS_I64;
    /** */
    // public static final long H5T_MIPS_I8 = H5T_MIPS_I8;
    /** */
    // public static final long H5T_MIPS_U16 = H5T_MIPS_U16;
    /** */
    // public static final long H5T_MIPS_U32 = H5T_MIPS_U32;
    /** */
    // public static final long H5T_MIPS_U64 = H5T_MIPS_U64;
    /** */
    // public static final long H5T_MIPS_U8 = H5T_MIPS_U8;
    /** */
    public static final long H5T_NATIVE_B16 = H5T_NATIVE_B16_g();
    /** */
    public static final long H5T_NATIVE_B32 = H5T_NATIVE_B32_g();
    /** */
    public static final long H5T_NATIVE_B64 = H5T_NATIVE_B64_g();
    /** */
    public static final long H5T_NATIVE_B8 = H5T_NATIVE_B8_g();
    /** */
    public static final long H5T_NATIVE_CHAR = (CHAR_MIN() < 0 ? H5T_NATIVE_SCHAR_g() : H5T_NATIVE_UCHAR_g());
    /** */
    public static final long H5T_NATIVE_DOUBLE = H5T_NATIVE_DOUBLE_g();
    /** */
    public static final long H5T_NATIVE_DOUBLE_COMPLEX = H5T_NATIVE_DOUBLE_COMPLEX_g();
    /** */
    public static final long H5T_NATIVE_FLOAT = H5T_NATIVE_FLOAT_g();
    /** */
    public static final long H5T_NATIVE_FLOAT16 = H5T_NATIVE_FLOAT16_g();
    /** */
    public static final long H5T_NATIVE_FLOAT_COMPLEX = H5T_NATIVE_FLOAT_COMPLEX_g();
    /** */
    public static final long H5T_NATIVE_HADDR = H5T_NATIVE_HADDR_g();
    /** */
    public static final long H5T_NATIVE_HBOOL = H5T_NATIVE_HBOOL_g();
    /** */
    public static final long H5T_NATIVE_HERR = H5T_NATIVE_HERR_g();
    /** */
    public static final long H5T_NATIVE_HSIZE = H5T_NATIVE_HSIZE_g();
    /** */
    public static final long H5T_NATIVE_HSSIZE = H5T_NATIVE_HSSIZE_g();
    /** */
    public static final long H5T_NATIVE_INT = H5T_NATIVE_INT_g();
    /** */
    public static final long H5T_NATIVE_INT_FAST16 = H5T_NATIVE_INT_FAST16_g();
    /** */
    public static final long H5T_NATIVE_INT_FAST32 = H5T_NATIVE_INT_FAST32_g();
    /** */
    public static final long H5T_NATIVE_INT_FAST64 = H5T_NATIVE_INT_FAST64_g();
    /** */
    public static final long H5T_NATIVE_INT_FAST8 = H5T_NATIVE_INT_FAST8_g();
    /** */
    public static final long H5T_NATIVE_INT_LEAST16 = H5T_NATIVE_INT_LEAST16_g();
    /** */
    public static final long H5T_NATIVE_INT_LEAST32 = H5T_NATIVE_INT_LEAST32_g();
    /** */
    public static final long H5T_NATIVE_INT_LEAST64 = H5T_NATIVE_INT_LEAST64_g();
    /** */
    public static final long H5T_NATIVE_INT_LEAST8 = H5T_NATIVE_INT_LEAST8_g();
    /** */
    public static final long H5T_NATIVE_INT16 = H5T_NATIVE_INT16_g();
    /** */
    public static final long H5T_NATIVE_INT32 = H5T_NATIVE_INT32_g();
    /** */
    public static final long H5T_NATIVE_INT64 = H5T_NATIVE_INT64_g();
    /** */
    public static final long H5T_NATIVE_INT8 = H5T_NATIVE_INT8_g();
    /** */
    public static final long H5T_NATIVE_LDOUBLE = H5T_NATIVE_LDOUBLE_g();
    /** */
    public static final long H5T_NATIVE_LLONG = H5T_NATIVE_LLONG_g();
    /** */
    public static final long H5T_NATIVE_LONG = H5T_NATIVE_LONG_g();
    /** */
    public static final long H5T_NATIVE_LDOUBLE_COMPLEX = H5T_NATIVE_LDOUBLE_COMPLEX_g();
    /** */
    public static final long H5T_NATIVE_OPAQUE = H5T_NATIVE_OPAQUE_g();
    /** */
    public static final long H5T_NATIVE_SCHAR = H5T_NATIVE_SCHAR_g();
    /** */
    public static final long H5T_NATIVE_SHORT = H5T_NATIVE_SHORT_g();
    /** */
    public static final long H5T_NATIVE_UCHAR = H5T_NATIVE_UCHAR_g();
    /** */
    public static final long H5T_NATIVE_UINT = H5T_NATIVE_UINT_g();
    /** */
    public static final long H5T_NATIVE_UINT_FAST16 = H5T_NATIVE_UINT_FAST16_g();
    /** */
    public static final long H5T_NATIVE_UINT_FAST32 = H5T_NATIVE_UINT_FAST32_g();
    /** */
    public static final long H5T_NATIVE_UINT_FAST64 = H5T_NATIVE_UINT_FAST64_g();
    /** */
    public static final long H5T_NATIVE_UINT_FAST8 = H5T_NATIVE_UINT_FAST8_g();
    /** */
    public static final long H5T_NATIVE_UINT_LEAST16 = H5T_NATIVE_UINT_LEAST16_g();
    /** */
    public static final long H5T_NATIVE_UINT_LEAST32 = H5T_NATIVE_UINT_LEAST32_g();
    /** */
    public static final long H5T_NATIVE_UINT_LEAST64 = H5T_NATIVE_UINT_LEAST64_g();
    /** */
    public static final long H5T_NATIVE_UINT_LEAST8 = H5T_NATIVE_UINT_LEAST8_g();
    /** */
    public static final long H5T_NATIVE_UINT16 = H5T_NATIVE_UINT16_g();
    /** */
    public static final long H5T_NATIVE_UINT32 = H5T_NATIVE_UINT32_g();
    /** */
    public static final long H5T_NATIVE_UINT64 = H5T_NATIVE_UINT64_g();
    /** */
    public static final long H5T_NATIVE_UINT8 = H5T_NATIVE_UINT8_g();
    /** */
    public static final long H5T_NATIVE_ULLONG = H5T_NATIVE_ULLONG_g();
    /** */
    public static final long H5T_NATIVE_ULONG = H5T_NATIVE_ULONG_g();
    /** */
    public static final long H5T_NATIVE_USHORT = H5T_NATIVE_USHORT_g();
    /** */
    public static final int H5T_NCLASSES = H5T_NCLASSES();
    /** */
    public static final int H5T_NO_CLASS = H5T_NO_CLASS();
    /** */
    public static final int H5T_NORM_ERROR = H5T_NORM_ERROR();
    /** */
    public static final int H5T_NORM_IMPLIED = H5T_NORM_IMPLIED();
    /** */
    public static final int H5T_NORM_MSBSET = H5T_NORM_MSBSET();
    /** */
    public static final int H5T_NORM_NONE = H5T_NORM_NONE();
    /** */
    public static final int H5T_NPAD = H5T_NPAD();
    /** */
    public static final int H5T_NSGN = H5T_NSGN();
    /** */
    public static final int H5T_OPAQUE = H5T_OPAQUE();
    /** */
    public static final int H5T_OPAQUE_TAG_MAX = H5T_OPAQUE_TAG_MAX(); /* 1.6.5 */
    /** */
    public static final int H5T_ORDER_BE = H5T_ORDER_BE();
    /** */
    public static final int H5T_ORDER_ERROR = H5T_ORDER_ERROR();
    /** */
    public static final int H5T_ORDER_LE = H5T_ORDER_LE();
    /** */
    public static final int H5T_ORDER_NONE = H5T_ORDER_NONE();
    /** */
    public static final int H5T_ORDER_VAX = H5T_ORDER_VAX();
    /** */
    public static final int H5T_PAD_BACKGROUND = H5T_PAD_BACKGROUND();
    /** */
    public static final int H5T_PAD_ERROR = H5T_PAD_ERROR();
    /** */
    public static final int H5T_PAD_ONE = H5T_PAD_ONE();
    /** */
    public static final int H5T_PAD_ZERO = H5T_PAD_ZERO();
    /** */
    public static final int H5T_PERS_DONTCARE = H5T_PERS_DONTCARE();
    /** */
    public static final int H5T_PERS_HARD = H5T_PERS_HARD();
    /** */
    public static final int H5T_PERS_SOFT = H5T_PERS_SOFT();
    /** */
    public static final int H5T_REFERENCE = H5T_REFERENCE();
    /** */
    public static final int H5T_SGN_2 = H5T_SGN_2();
    /** */
    public static final int H5T_SGN_ERROR = H5T_SGN_ERROR();
    /** */
    public static final int H5T_SGN_NONE = H5T_SGN_NONE();
    /** */
    public static final long H5T_STD_B16BE = H5T_STD_B16BE_g();
    /** */
    public static final long H5T_STD_B16LE = H5T_STD_B16LE_g();
    /** */
    public static final long H5T_STD_B32BE = H5T_STD_B32BE_g();
    /** */
    public static final long H5T_STD_B32LE = H5T_STD_B32LE_g();
    /** */
    public static final long H5T_STD_B64BE = H5T_STD_B64BE_g();
    /** */
    public static final long H5T_STD_B64LE = H5T_STD_B64LE_g();
    /** */
    public static final long H5T_STD_B8BE = H5T_STD_B8BE_g();
    /** */
    public static final long H5T_STD_B8LE = H5T_STD_B8LE_g();
    /** */
    public static final long H5T_STD_I16BE = H5T_STD_I16BE_g();
    /** */
    public static final long H5T_STD_I16LE = H5T_STD_I16LE_g();
    /** */
    public static final long H5T_STD_I32BE = H5T_STD_I32BE_g();
    /** */
    public static final long H5T_STD_I32LE = H5T_STD_I32LE_g();
    /** */
    public static final long H5T_STD_I64BE = H5T_STD_I64BE_g();
    /** */
    public static final long H5T_STD_I64LE = H5T_STD_I64LE_g();
    /** */
    public static final long H5T_STD_I8BE = H5T_STD_I8BE_g();
    /** */
    public static final long H5T_STD_I8LE = H5T_STD_I8LE_g();
    /** */
    public static final long H5T_STD_REF_DSETREG = H5T_STD_REF_DSETREG_g();
    /** */
    public static final long H5T_STD_REF_OBJ = H5T_STD_REF_OBJ_g();
    /** */
    public static final long H5T_STD_REF = H5T_STD_REF_g();
    /** */
    public static final long H5T_STD_U16BE = H5T_STD_U16BE_g();
    /** */
    public static final long H5T_STD_U16LE = H5T_STD_U16LE_g();
    /** */
    public static final long H5T_STD_U32BE = H5T_STD_U32BE_g();
    /** */
    public static final long H5T_STD_U32LE = H5T_STD_U32LE_g();
    /** */
    public static final long H5T_STD_U64BE = H5T_STD_U64BE_g();
    /** */
    public static final long H5T_STD_U64LE = H5T_STD_U64LE_g();
    /** */
    public static final long H5T_STD_U8BE = H5T_STD_U8BE_g();
    /** */
    public static final long H5T_STD_U8LE = H5T_STD_U8LE_g();
    /** */
    public static final int H5T_STR_ERROR = H5T_STR_ERROR();
    /** */
    public static final int H5T_STR_NULLPAD = H5T_STR_NULLPAD();
    /** */
    public static final int H5T_STR_NULLTERM = H5T_STR_NULLTERM();
    /** */
    public static final int H5T_STR_RESERVED_10 = H5T_STR_RESERVED_10();
    /** */
    public static final int H5T_STR_RESERVED_11 = H5T_STR_RESERVED_11();
    /** */
    public static final int H5T_STR_RESERVED_12 = H5T_STR_RESERVED_12();
    /** */
    public static final int H5T_STR_RESERVED_13 = H5T_STR_RESERVED_13();
    /** */
    public static final int H5T_STR_RESERVED_14 = H5T_STR_RESERVED_14();
    /** */
    public static final int H5T_STR_RESERVED_15 = H5T_STR_RESERVED_15();
    /** */
    public static final int H5T_STR_RESERVED_3 = H5T_STR_RESERVED_3();
    /** */
    public static final int H5T_STR_RESERVED_4 = H5T_STR_RESERVED_4();
    /** */
    public static final int H5T_STR_RESERVED_5 = H5T_STR_RESERVED_5();
    /** */
    public static final int H5T_STR_RESERVED_6 = H5T_STR_RESERVED_6();
    /** */
    public static final int H5T_STR_RESERVED_7 = H5T_STR_RESERVED_7();
    /** */
    public static final int H5T_STR_RESERVED_8 = H5T_STR_RESERVED_8();
    /** */
    public static final int H5T_STR_RESERVED_9 = H5T_STR_RESERVED_9();
    /** */
    public static final int H5T_STR_SPACEPAD = H5T_STR_SPACEPAD();
    /** */
    public static final int H5T_STRING = H5T_STRING();
    /** */
    public static final int H5T_TIME = H5T_TIME();
    /** */
    public static final long H5T_UNIX_D32BE = H5T_UNIX_D32BE_g();
    /** */
    public static final long H5T_UNIX_D32LE = H5T_UNIX_D32LE_g();
    /** */
    public static final long H5T_UNIX_D64BE = H5T_UNIX_D64BE_g();
    /** */
    public static final long H5T_UNIX_D64LE = H5T_UNIX_D64LE_g();
    /** */
    public static final long H5T_VARIABLE = H5T_VARIABLE();
    /** */
    public static final int H5T_VLEN = H5T_VLEN();

    /** */
    public static final int H5VL_CAP_FLAG_NONE = H5VL_CAP_FLAG_NONE();
    /** */
    public static final int H5VL_CAP_FLAG_THREADSAFE = H5VL_CAP_FLAG_THREADSAFE();
    /** */
    public static final long H5VL_NATIVE = H5VL_NATIVE_g();
    /** */
    public static final String H5VL_NATIVE_NAME = H5VL_NATIVE_NAME().getString(0);
    /** */
    public static final int H5VL_NATIVE_VALUE = H5VL_NATIVE_VALUE();
    /** */
    public static final int H5VL_NATIVE_VERSION = H5VL_NATIVE_VERSION();
    /** */
    public static final int H5_VOL_INVALID = H5_VOL_INVALID();
    /** */
    public static final int H5_VOL_NATIVE = H5_VOL_NATIVE();
    /** */
    public static final int H5_VOL_RESERVED = H5_VOL_RESERVED();
    /** */
    public static final int H5_VOL_MAX = H5_VOL_MAX();

    /** Return values for filter callback function */
    public static final int H5Z_CB_CONT = H5Z_CB_CONT();
    /** Return values for filter callback function */
    public static final int H5Z_CB_ERROR = H5Z_CB_ERROR();
    /** Return values for filter callback function */
    public static final int H5Z_CB_FAIL = H5Z_CB_FAIL();
    /** Return values for filter callback function */
    public static final int H5Z_CB_NO = H5Z_CB_NO();
    /** Values to decide if EDC is enabled for reading data */
    public static final int H5Z_DISABLE_EDC = H5Z_DISABLE_EDC();
    /** Values to decide if EDC is enabled for reading data */
    public static final int H5Z_ENABLE_EDC = H5Z_ENABLE_EDC();
    /** Values to decide if EDC is enabled for reading data */
    public static final int H5Z_ERROR_EDC = H5Z_ERROR_EDC();
    /** Filter IDs - deflation like gzip */
    public static final int H5Z_FILTER_DEFLATE = H5Z_FILTER_DEFLATE();
    /** Filter IDs - no filter */
    public static final int H5Z_FILTER_ERROR = H5Z_FILTER_ERROR();
    /** Filter IDs - fletcher32 checksum of EDC */
    public static final int H5Z_FILTER_FLETCHER32 = H5Z_FILTER_FLETCHER32();
    /** Filter IDs - maximum filter id */
    public static final int H5Z_FILTER_MAX = H5Z_FILTER_MAX();
    /** Filter IDs - nbit compression */
    public static final int H5Z_FILTER_NBIT = H5Z_FILTER_NBIT();
    /** Filter IDs - reserved indefinitely */
    public static final int H5Z_FILTER_NONE = H5Z_FILTER_NONE();
    /** Filter IDs - filter ids below this value are reserved for library use */
    public static final int H5Z_FILTER_RESERVED = H5Z_FILTER_RESERVED();
    /** Filter IDs - scale+offset compression */
    public static final int H5Z_FILTER_SCALEOFFSET = H5Z_FILTER_SCALEOFFSET();
    /** Filter IDs - shuffle the data */
    public static final int H5Z_FILTER_SHUFFLE = H5Z_FILTER_SHUFFLE();
    /** Filter IDs - szip compression */
    public static final int H5Z_FILTER_SZIP = H5Z_FILTER_SZIP();
    /**
     * Flags for filter definition (stored)
     * definition flag mask
     */
    public static final int H5Z_FLAG_DEFMASK = H5Z_FLAG_DEFMASK();
    /**
     * Additional flags for filter invocation (not stored)
     * invocation flag mask
     */
    public static final int H5Z_FLAG_INVMASK = H5Z_FLAG_INVMASK();
    /**
     * Flags for filter definition (stored)
     * filter is mandatory
     */
    public static final int H5Z_FLAG_MANDATORY = H5Z_FLAG_MANDATORY();
    /**
     * Flags for filter definition (stored)
     * filter is optional
     */
    public static final int H5Z_FLAG_OPTIONAL = H5Z_FLAG_OPTIONAL();
    /**
     * Additional flags for filter invocation (not stored)
     * reverse direction; read
     */
    public static final int H5Z_FLAG_REVERSE = H5Z_FLAG_REVERSE();
    /**
     * Additional flags for filter invocation (not stored)
     * skip EDC filters for read
     */
    public static final int H5Z_FLAG_SKIP_EDC = H5Z_FLAG_SKIP_EDC();
    /** Symbol to remove all filters in H5Premove_filter */
    public static final int H5Z_FILTER_ALL = H5Z_FILTER_ALL();
    /** Maximum number of filters allowed in a pipeline */
    public static final int H5Z_MAX_NFILTERS = H5Z_MAX_NFILTERS();
    /** Values to decide if EDC is enabled for reading data */
    public static final int H5Z_NO_EDC = H5Z_NO_EDC();
    /** Bit flags for H5Zget_filter_info */
    public static final int H5Z_FILTER_CONFIG_ENCODE_ENABLED = H5Z_FILTER_CONFIG_ENCODE_ENABLED();
    /** Bit flags for H5Zget_filter_info */
    public static final int H5Z_FILTER_CONFIG_DECODE_ENABLED = H5Z_FILTER_CONFIG_DECODE_ENABLED();
    /** Special parameters for ScaleOffset filter*/
    public static final int H5Z_SO_INT_MINBITS_DEFAULT = H5Z_SO_INT_MINBITS_DEFAULT();
    /** Special parameters for ScaleOffset filter*/
    public static final int H5Z_SO_FLOAT_DSCALE = H5Z_SO_FLOAT_DSCALE();
    /** Special parameters for ScaleOffset filter*/
    public static final int H5Z_SO_FLOAT_ESCALE = H5Z_SO_FLOAT_ESCALE();
    /** Special parameters for ScaleOffset filter*/
    public static final int H5Z_SO_INT = H5Z_SO_INT();
    /** shuffle filter - Number of parameters that users can set */
    public static final int H5Z_SHUFFLE_USER_NPARMS = H5Z_SHUFFLE_USER_NPARMS();
    /** shuffle filter - Total number of parameters for filter */
    public static final int H5Z_SHUFFLE_TOTAL_NPARMS = H5Z_SHUFFLE_TOTAL_NPARMS();
    /** szip filter - Number of parameters that users can set */
    public static final int H5Z_SZIP_USER_NPARMS = H5Z_SZIP_USER_NPARMS();
    /** szip filter - Total number of parameters for filter */
    public static final int H5Z_SZIP_TOTAL_NPARMS = H5Z_SZIP_TOTAL_NPARMS();
    /** szip filter - "User" parameter for option mask */
    public static final int H5Z_SZIP_PARM_MASK = H5Z_SZIP_PARM_MASK();
    /** szip filter - "User" parameter for pixels-per-block */
    public static final int H5Z_SZIP_PARM_PPB = H5Z_SZIP_PARM_PPB();
    /** szip filter - "Local" parameter for bits-per-pixel */
    public static final int H5Z_SZIP_PARM_BPP = H5Z_SZIP_PARM_BPP();
    /** szip filter - "Local" parameter for pixels-per-scanline */
    public static final int H5Z_SZIP_PARM_PPS = H5Z_SZIP_PARM_PPS();
    /** nbit filter - Number of parameters that users can set */
    public static final int H5Z_NBIT_USER_NPARMS = H5Z_NBIT_USER_NPARMS();
    /** scale offset filter - Number of parameters that users can set */
    public static final int H5Z_SCALEOFFSET_USER_NPARMS = H5Z_SCALEOFFSET_USER_NPARMS();

    /**
     * Helper method to get H5FD_DIRECT VFD identifier using reflection.
     * Returns H5I_INVALID_HID if Direct VFD is not available (e.g., on Windows or when H5_HAVE_DIRECT is not
     * defined).
     *
     * @return the H5FD_DIRECT VFD identifier, or H5I_INVALID_HID if not available
     */
    private static long getH5FD_DIRECT()
    {
        try {
            // Use reflection to call H5FD_DIRECT_id_g() if it exists
            // This method only exists if H5_HAVE_DIRECT is defined (Linux with Direct I/O)
            java.lang.reflect.Method method =
                org.hdfgroup.javahdf5.hdf5_h.class.getMethod("H5FD_DIRECT_id_g");
            return (long)method.invoke(null);
        }
        catch (NoSuchMethodException e) {
            // Method doesn't exist - Direct VFD not available on this platform
            return H5I_INVALID_HID();
        }
        catch (Exception e) {
            // Other error (shouldn't happen)
            return H5I_INVALID_HID();
        }
    }

    /**
     * Helper method to get H5FD_MPIO VFD identifier using reflection.
     * Returns H5I_INVALID_HID if MPIO VFD is not available (e.g., when parallel/MPI support is not enabled).
     *
     * @return the H5FD_MPIO VFD identifier, or H5I_INVALID_HID if not available
     */
    private static long getH5FD_MPIO()
    {
        try {
            // Use reflection to call H5FD_MPIO_id_g() if it exists
            // This method only exists if parallel/MPI support is enabled
            java.lang.reflect.Method method = org.hdfgroup.javahdf5.hdf5_h.class.getMethod("H5FD_MPIO_id_g");
            return (long)method.invoke(null);
        }
        catch (NoSuchMethodException e) {
            // Method doesn't exist - MPIO VFD not available (parallel not enabled)
            return H5I_INVALID_HID();
        }
        catch (Exception e) {
            // Other error (shouldn't happen)
            return H5I_INVALID_HID();
        }
    }

    /**
     * Helper method to get H5FD_ROS3 VFD identifier using reflection.
     * Returns H5I_INVALID_HID if ROS3 VFD is not available (e.g., when HDF5_ENABLE_ROS3_VFD is not enabled).
     *
     * @return the H5FD_ROS3 VFD identifier, or H5I_INVALID_HID if not available
     */
    private static long getH5FD_ROS3()
    {
        try {
            // Use reflection to call H5FD_ROS3_id_g() if it exists
            // This method only exists if ROS3 VFD support is enabled (H5_HAVE_ROS3_VFD)
            java.lang.reflect.Method method = org.hdfgroup.javahdf5.hdf5_h.class.getMethod("H5FD_ROS3_id_g");
            return (long)method.invoke(null);
        }
        catch (NoSuchMethodException e) {
            // Method doesn't exist - ROS3 VFD not available
            return H5I_INVALID_HID();
        }
        catch (Exception e) {
            // Other error (shouldn't happen)
            return H5I_INVALID_HID();
        }
    }

    /**
     * Helper method to get H5FD_HDFS VFD identifier using reflection.
     * Returns H5I_INVALID_HID if HDFS VFD is not available (e.g., when H5_HAVE_LIBHDFS is not defined).
     *
     * @return the H5FD_HDFS VFD identifier, or H5I_INVALID_HID if not available
     */
    private static long getH5FD_HDFS()
    {
        try {
            // Use reflection to call H5FD_HDFS_id_g() if it exists
            // This method only exists if HDFS support is enabled (H5_HAVE_LIBHDFS)
            java.lang.reflect.Method method = org.hdfgroup.javahdf5.hdf5_h.class.getMethod("H5FD_HDFS_id_g");
            return (long)method.invoke(null);
        }
        catch (NoSuchMethodException e) {
            // Method doesn't exist - HDFS VFD not available
            return H5I_INVALID_HID();
        }
        catch (Exception e) {
            // Other error (shouldn't happen)
            return H5I_INVALID_HID();
        }
    }

    /**
     * Helper method to get H5FD_MIRROR VFD identifier using reflection.
     * Returns H5I_INVALID_HID if Mirror VFD is not available (e.g., when H5_HAVE_MIRROR_VFD is not defined).
     *
     * @return the H5FD_MIRROR VFD identifier, or H5I_INVALID_HID if not available
     */
    private static long getH5FD_MIRROR()
    {
        try {
            // Use reflection to call H5FD_MIRROR_id_g() if it exists
            // This method only exists if Mirror VFD support is enabled (H5_HAVE_MIRROR_VFD)
            java.lang.reflect.Method method =
                org.hdfgroup.javahdf5.hdf5_h.class.getMethod("H5FD_MIRROR_id_g");
            return (long)method.invoke(null);
        }
        catch (NoSuchMethodException e) {
            // Method doesn't exist - Mirror VFD not available
            return H5I_INVALID_HID();
        }
        catch (Exception e) {
            // Other error (shouldn't happen)
            return H5I_INVALID_HID();
        }
    }
}
