% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/precision_profile.R
\name{precision_profile}
\alias{precision_profile}
\title{Precision Profile Analysis}
\usage{
precision_profile(
  x,
  concentration = "concentration",
  cv = "cv_pct",
  model = c("hyperbolic", "linear"),
  cv_targets = c(10, 20),
  conf_level = 0.95,
  bootstrap = FALSE,
  boot_n = 1999
)
}
\arguments{
\item{x}{An object of class \code{precision_study} with multiple concentration
levels, OR a data frame with columns for concentration and CV values.}

\item{concentration}{Character string specifying the column name for
concentration values (only used if \code{x} is a data frame). Default is
\code{"concentration"}.}

\item{cv}{Character string specifying the column name for CV values (only
used if \code{x} is a data frame). Default is \code{"cv_pct"}.}

\item{model}{Regression model for CV-concentration relationship:
\code{"hyperbolic"} (default) fits CV = sqrt(a^2 + (b/x)^2),
\code{"linear"} fits CV = a + b/x.}

\item{cv_targets}{Numeric vector of target CV percentages for functional
sensitivity estimation. Default is \code{c(10, 20)}.}

\item{conf_level}{Confidence level for prediction intervals (default: 0.95).}

\item{bootstrap}{Logical; if \code{TRUE}, uses bootstrap resampling for
confidence intervals on functional sensitivity estimates. Default is \code{FALSE}.}

\item{boot_n}{Number of bootstrap resamples when \code{bootstrap = TRUE}
(default: 1999).}
}
\value{
An object of class \code{c("precision_profile", "valytics_precision", "valytics_result")},
which is a list containing:

\describe{
\item{input}{List with original data:
\itemize{
\item \code{concentration}: Numeric vector of concentrations
\item \code{cv}: Numeric vector of CV values (percent)
\item \code{n_levels}: Number of concentration levels
\item \code{conc_range}: Concentration range (min, max)
\item \code{conc_span}: Fold-difference (max/min)
}
}
\item{model}{List with fitted model information:
\itemize{
\item \code{type}: Model type ("hyperbolic" or "linear")
\item \code{parameters}: Named vector of fitted parameters
\item \code{equation}: Character string describing the fitted equation
}
}
\item{fitted}{Data frame with fitted values:
\itemize{
\item \code{concentration}: Concentration values
\item \code{cv_observed}: Observed CV values
\item \code{cv_fitted}: Model-fitted CV values
\item \code{residual}: Residuals (observed - fitted)
\item \code{ci_lower}: Lower prediction interval
\item \code{ci_upper}: Upper prediction interval
}
}
\item{fit_quality}{List with goodness-of-fit statistics:
\itemize{
\item \code{r_squared}: Coefficient of determination
\item \code{adj_r_squared}: Adjusted R-squared
\item \code{rmse}: Root mean squared error
\item \code{mae}: Mean absolute error
}
}
\item{functional_sensitivity}{Data frame with functional sensitivity estimates:
\itemize{
\item \code{cv_target}: Target CV percentage
\item \code{concentration}: Estimated concentration at target CV
\item \code{ci_lower}: Lower confidence limit (if bootstrap)
\item \code{ci_upper}: Upper confidence limit (if bootstrap)
\item \code{achievable}: Logical; TRUE if target CV is achievable
}
}
\item{settings}{List with analysis settings}
\item{call}{The matched function call}
}
}
\description{
Constructs a precision profile (CV vs concentration relationship) from
precision study results and estimates functional sensitivity. The precision
profile characterizes how measurement imprecision changes across the
analytical measurement interval.
}
\details{
\strong{Precision Profile:}

The precision profile describes how analytical imprecision (CV) varies
across the analytical measurement interval. Typically, CV decreases as
concentration increases, following a hyperbolic relationship.

\strong{Hyperbolic Model:}

The hyperbolic model is:
\deqn{CV = \sqrt{a^2 + (b/x)^2}}

where:
\itemize{
\item \code{a} represents the asymptotic CV at high concentrations
\item \code{b} represents the concentration-dependent component
\item \code{x} is the analyte concentration
}

This model captures the characteristic behavior where CV approaches a
constant value at high concentrations and increases hyperbolically at
low concentrations.

\strong{Linear Model:}

The linear model is:
\deqn{CV = a + b/x}

This is a simpler alternative that may be appropriate when the relationship
is approximately linear when plotted as CV vs 1/concentration.

\strong{Functional Sensitivity:}

Functional sensitivity is defined as the lowest concentration at which a
measurement procedure achieves a specified level of precision (CV). Common
thresholds are:
\itemize{
\item \strong{10\% CV}: Modern standard for high-sensitivity assays (e.g., cardiac troponin)
\item \strong{20\% CV}: Traditional standard (originally defined for TSH assays)
}

The functional sensitivity is calculated by solving the fitted model equation
for the concentration that yields the target CV.
}
\section{Minimum Requirements}{

\itemize{
\item At least 4 concentration levels
\item Concentration span of at least 2-fold (warning if less)
\item Valid CV estimates at each level (from precision study)
}
}

\examples{
# Example with simulated multi-level precision data
set.seed(42)

# Generate data for 6 concentration levels
conc_levels <- c(5, 10, 25, 50, 100, 200)
n_levels <- length(conc_levels)

prec_data <- data.frame()
for (i in seq_along(conc_levels)) {
  level_data <- expand.grid(
    level = conc_levels[i],
    day = 1:5,
    replicate = 1:5
  )
  
  # Simulate CV that decreases with concentration
  true_cv <- sqrt(3^2 + (20/conc_levels[i])^2)
  level_data$value <- conc_levels[i] * rnorm(
    nrow(level_data),
    mean = 1,
    sd = true_cv/100
  )
  
  prec_data <- rbind(prec_data, level_data)
}

# Run precision study
prec <- precision_study(
  data = prec_data,
  value = "value",
  sample = "level",
  day = "day"
)

# Generate precision profile
profile <- precision_profile(prec)
print(profile)
summary(profile)

# Hyperbolic model with bootstrap CIs
profile_boot <- precision_profile(
  prec,
  model = "hyperbolic",
  cv_targets = c(10, 20),
  bootstrap = TRUE,
  boot_n = 499
)

# Linear model
profile_linear <- precision_profile(prec, model = "linear")

}
\references{
Armbruster DA, Pry T (2008). Limit of blank, limit of detection and limit of
quantitation. \emph{Clinical Biochemist Reviews}, 29(Suppl 1):S49-S52.

CLSI EP17-A2 (2012). Evaluation of Detection Capability for Clinical
Laboratory Measurement Procedures; Approved Guideline - Second Edition.
Clinical and Laboratory Standards Institute, Wayne, PA.

Kroll MH, Emancipator K (1993). A theoretical evaluation of linearity.
\emph{Clinical Chemistry}, 39(3):405-413.
}
\seealso{
\code{\link[=precision_study]{precision_study()}} for variance component analysis,
\code{\link[=plot.precision_profile]{plot.precision_profile()}} for visualization
}
