% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/irt_trt.R
\name{irt_trt}
\alias{irt_trt}
\title{Joint Item Response Theory and Testlet Response Theory Estimation (Dichotomous & Polytomous)}
\usage{
irt_trt(data, item_spec, method = "EM", control = list())
}
\arguments{
\item{data}{A \code{data.frame} or \code{matrix} containing item responses.
Responses should be 0-indexed integers. Missing values should be coded as \code{NA}.}

\item{item_spec}{A \code{data.frame} providing item metadata. Must include
columns \code{"item"} (matching \code{colnames(data)}) and \code{"model"}.
Optionally includes a \code{"testlet"} column for TRT specifications.}

\item{method}{A character string specifying the estimation method. Currently
supports \code{"EM"} (Expectation-Maximization). Defaults to \code{"EM"}.}

\item{control}{A \code{list} of control parameters for the estimation algorithm:
\itemize{
\item \code{max_iter}: Maximum number of EM iterations (default = 100).
\item \code{converge_tol}: Convergence criterion for parameter change (default = 1e-4).
\item \code{theta_range}: Numeric vector of length 2 specifying the integration
grid bounds (default = c(-4, 4)).
\item \code{quad_points}: Number of quadrature points (default = 21).
\item \code{verbose}: Logical; if \code{TRUE}, prints progress to console.
\item \code{fix_discrimination}: Logical; default=FALSE
}}
}
\value{
A \code{list} containing three components:
\item{item_params}{A data frame of estimated item slopes (discrimination),
difficulties/thresholds, and guessing parameters with associated standard errors.}
\item{person_params}{A data frame of EAP-based ability estimates (\eqn{\theta})
and testlet effect estimates (\eqn{\gamma}).}
\item{model_fit}{A data frame containing Log-Likelihood, AIC, and BIC indices.}
}
\description{
Provides a unified marginal maximum likelihood estimation framework for
a broad class of item response theory and testlet response theory models. The function automatically detects data
structures to apply appropriate models, along with their testlet-effect extensions (Bradlow et al., 1999).
}
\details{
The estimation utilizes a robust Newton-Raphson update within the M-step. For
testlet models, dimension reduction is achieved through the integration of
the nuisance testlet effect (Li et al., 2006). The function automatically
corrects model specifications if the data levels (binary vs. polytomous)
do not align with the requested model string.
}
\examples{
  # --- Example: Simulation (Binary + Poly + Testlets) ---
  set.seed(2025)
  N <- 100; J <- 20

  # 1. Generate Parameters
  theta <- rnorm(N, 0, 1)
  gamma_1 <- rnorm(N, 0, 0.5) # Testlet 1 effect
  gamma_2 <- rnorm(N, 0, 0.6) # Testlet 2 effect

  a_true <- runif(J, 0.8, 1.5)
  b_true <- seq(-1.5, 1.5, length.out = J)

  resp_matrix <- matrix(NA, N, J)
  colnames(resp_matrix) <- paste0("Item_", 1:J)

  # 2. Simulate Responses
  # Items 1-10: Binary Independent (Model: 2PL)
  for(j in 1:10) {
    p <- 1 / (1 + exp(-a_true[j] * (theta - b_true[j])))
    resp_matrix[,j] <- rbinom(N, 1, p)
  }

  # Items 11-15: Poly Independent (Model: GRM)
  for(j in 11:15) {
    thresh <- sort(c(b_true[j] - 0.7, b_true[j] + 0.7))
    p1 <- 1 / (1 + exp(-a_true[j] * (theta - thresh[1])))
    p2 <- 1 / (1 + exp(-a_true[j] * (theta - thresh[2])))
    probs <- cbind(1-p1, p1-p2, p2)
    resp_matrix[,j] <- apply(probs, 1, function(p) sample(0:2, 1, prob=p))
  }

  # Items 16-17: Binary Testlet 1 (Model: 2PLT)
  for(j in 16:17) {
    eff_theta <- theta + gamma_1
    p <- 1 / (1 + exp(-a_true[j] * (eff_theta - b_true[j])))
    resp_matrix[,j] <- rbinom(N, 1, p)
  }

  # Items 18-20: Poly Testlet 2 (Model: GRT)
  for(j in 18:20) {
    eff_theta <- theta + gamma_2
    thresh <- sort(c(b_true[j] - 0.5, b_true[j] + 0.5))
    p1 <- 1 / (1 + exp(-a_true[j] * (eff_theta - thresh[1])))
    p2 <- 1 / (1 + exp(-a_true[j] * (eff_theta - thresh[2])))
    probs <- cbind(1-p1, p1-p2, p2)
    resp_matrix[,j] <- apply(probs, 1, function(p) sample(0:2, 1, prob=p))
  }

  df_sim <- as.data.frame(resp_matrix)

  # 3. Create Item Specification
  # STRICT naming: Independent=2PL/GRM, Testlet=2PLT/GRT
  spec <- data.frame(
    item = colnames(df_sim),
    model = c(rep("2PL", 10), rep("GRM", 5), rep("2PLT", 2), rep("GRT", 3)),
    testlet = c(rep(NA, 15), rep("T1", 2), rep("T2", 3)),
    stringsAsFactors = FALSE
  )

  # 4. Run Estimation
  res <- irt_trt(df_sim, spec, method = "EM",
                 control = list(max_iter = 20, verbose = FALSE))

  head(res$item_params)
  head(res$person_params)
}
\references{
Bradlow, E. T., Wainer, H., & Wang, X. (1999). A testlet response model for
multidimensionality in item response theory. \emph{Psychometrika, 64}(2), 147-168.

Li, Y., Bolt, D. M., & Fu, J. (2006). A comparison of methods for estimating
secondary dimensions in testlet-based data. \emph{Applied Psychological
Measurement, 30}(3), 203-223.
}
