% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/group_imp.R
\name{group_imp}
\alias{group_imp}
\title{Grouped K-NN or PCA Imputation}
\usage{
group_imp(
  obj,
  group,
  k = NULL,
  colmax = NULL,
  knn_method = NULL,
  post_imp = NULL,
  dist_pow = NULL,
  tree = NULL,
  cores = 1,
  ncp = NULL,
  scale = NULL,
  pca_method = NULL,
  coeff.ridge = NULL,
  threshold = NULL,
  row.w = NULL,
  seed = NULL,
  nb.init = NULL,
  maxiter = NULL,
  miniter = NULL,
  .progress = TRUE
)
}
\arguments{
\item{obj}{A numeric matrix with \strong{samples in rows} and \strong{features in columns}.}

\item{group}{Preferably created by \code{\link[=group_features]{group_features()}}. A data.frame with columns:
\itemize{
\item \strong{features}: A list-column containing character vectors of feature column
names to impute
\item \strong{aux}: (Optional) A list-column containing character vectors of auxiliary
column names used for imputation but not imputed themselves
\item \strong{parameters}: (Optional) A list-column containing group-specific parameters
}}

\item{k}{Number of nearest neighbors for imputation. 10 is a good starting point.}

\item{colmax}{A number from 0 to 1. Threshold of missing data above which K-NN imputation is skipped.}

\item{knn_method}{Either "euclidean" (default) or "manhattan". Distance metric for nearest neighbor calculation.}

\item{post_imp}{Whether to impute remaining missing values (those that failed K-NN imputation)
using column means (default = \code{TRUE}).}

\item{dist_pow}{The amount of penalization for further away nearest neighbors in the weighted average.
\code{dist_pow = 0} (default) is the simple average of the nearest neighbors.}

\item{tree}{Either \code{NULL} (default, brute-force K-NN), "ball", or "kd" to find nearest neighbors using the \code{{mlpack}} ball-tree or kd-tree algorithms.}

\item{cores}{Controls the number of cores to parallelize over for K-NN imputation only.
To setup parallelization for PCA imputation, use \code{mirai::daemons()}.}

\item{ncp}{integer corresponding to the number of components used to to predict the missing entries}

\item{scale}{boolean. By default TRUE leading to a same weight for each variable}

\item{pca_method}{"regularized" by default or "EM".}

\item{coeff.ridge}{1 by default to perform the regularized pca_imp (imputePCA) algorithm; useful only if method="Regularized". Other regularization terms can be implemented by setting the value to less than 1 in order to regularized less (to get closer to the results of the EM method) or more than 1 to regularized more (to get closer to the results of the mean imputation)}

\item{threshold}{the threshold for assessing convergence}

\item{row.w}{Row weights. Can be one of:
\itemize{
\item \code{NULL} (default): all rows weighted equally.
\item A numeric vector of length \code{nrow(obj)}: custom positive weights.
\item \code{"n_miss"}: rows with more missing values receive lower weight.
}

Weights are normalized to sum to 1.}

\item{seed}{integer, by default seed = NULL implies that missing values are initially imputed by the mean of each variable. Other values leads to a random initialization}

\item{nb.init}{integer corresponding to the number of random initializations; the first initialization is the initialization with the mean imputation}

\item{maxiter}{integer, maximum number of iteration for the algorithm}

\item{miniter}{integer, minimum number of iteration for the algorithm}

\item{.progress}{Show imputation progress (default = FALSE)}
}
\value{
A numeric matrix of the same dimensions as \code{obj} with missing values imputed.
}
\description{
K-NN or PCA imputation by groups, such as chromosomes, flanking columns, or clusters
identified by column clustering techniques.
}
\details{
This function performs K-NN or PCA imputation on groups of features independently,
which significantly reduce imputation time for large datasets.

Specify \code{k} and related arguments to use K-NN, \code{ncp} and related arguments for PCA imputation.
If \code{k} and \code{ncp} are both \code{NULL}, then the group-wise parameters column i.e., \code{group$parameters}
must be specified and must contains either \code{k} or \code{ncp} for all groups of group-wise parameters.

Strategies for grouping may include:
\itemize{
\item Breaking down search space by chromosomes
\item Grouping features with their flanking values/neighbors (e.g., 1000 bp down/up stream of a CpG)
\item Using clusters identified by column clustering techniques
}

Only features in each group (each row of the data.frame) will be imputed, using
the search space defined as the union of the features and optional aux columns
of that group. Columns that are in aux or in the object but not in any features
will be left unchanged.
}
\examples{
# Generate example data with missing values
set.seed(1234)
to_test <- sim_mat(
  m = 20,
  n = 50,
  perc_NA = 0.3,
  perc_col_NA = 1,
  nchr = 2
)
# t() to put features in columns
obj <- t(to_test$input)
head(to_test$group_feature) # which group each feature belongs to

# Use group_features() to create the group tibble. By setting `k = 5` in
# group_features(), we are doing K-NN imputation in group_imp(). To make use
# of the `subset` argument in knn_imp(), we specify subset in group_features().
# For demonstration of different group-wise parameters we set `k = 10` for the
# second group.
subset_features <- sample(to_test$group_feature$feature_id, size = 10)
head(subset_features)
knn_df <- group_features(obj, to_test$group_feature, k = 5, subset = subset_features)
knn_df
knn_df$parameters[[2]]$k <- 10
knn_df$parameters

# Run grouped imputation. `k` for K-NN has been specified in `knn_df`.
knn_grouped <- group_imp(obj, group = knn_df, cores = 2)
knn_grouped # only features in subset are imputed

# Specify `ncp` for PCA directly in the group_imp() function (instead of in
# group_features()). We run in parallel with `mirai::daemons(2)`.

mirai::daemons(2) # Set up 2 cores for parallelization
pca_df <- group_features(obj, to_test$group_feature)
pca_grouped <- group_imp(obj, group = pca_df, ncp = 2)
mirai::daemons(0)
pca_grouped
}
\seealso{
\code{\link[=group_features]{group_features()}}
}
