#' Annotate pathogenic mutation to coding regions
#'
#' @param pathog.mut.obj A pathogenic mutation object
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' @param splice.region A positive integer specifying the length of splicing junction region in introns
#' 
#' @return A list of pathogenic mutation annotation including CDS region type, mutation start position in CDS, mutation end position in CDS, exon id  
#' 
#' @importFrom IRanges IRanges
#' @importFrom IRanges ranges
#' @importFrom GenomicRanges GRanges
#' @importFrom GenomicRanges findOverlaps
#' @importFrom GenomicRanges promoters
#' @importFrom S4Vectors subjectHits
#' @importFrom BiocGenerics start
#' @importFrom BiocGenerics end
#' @importFrom BiocGenerics strand
#' 
#' @examples
#' \dontrun{
#'     annoPathogMutToCds()
#' }
#'
#' @noRd
annoPathogMutToCds <- function(pathog.mut.obj, ensdb, splice.region){
    
    chr.id <- pathog.mut.obj$chr.id
    pathog.mut.start <- pathog.mut.obj$mut.start
    pathog.mut.end <- pathog.mut.obj$mut.end
    strand <- pathog.mut.obj$strand
    tx.id <- pathog.mut.obj$tx.id
    
    pathog.mut.gr <- GenomicRanges::GRanges(chr.id, ranges = IRanges::IRanges(pathog.mut.start, pathog.mut.end), strand=strand)
    
    gr.list <- getCdsAndNoncds(ensdb, tx.id)
    cds.gr <- gr.list$cds
    noncds.gr <- gr.list$noncds
    
    tx.cds.range <- range(cds.gr)
    ovlp.mut.tx <- GenomicRanges::findOverlaps(pathog.mut.gr, tx.cds.range, type="within")
    pathog.mut.start.cds <- NA
    pathog.mut.end.cds <- NA
    pathog.mut.exon.id <- NA
    
    if( length(ovlp.mut.tx)==0 ){
        
        pathog.mut.cds.location <- "out_of_cds"
        
    }else{
        
        ovlp.mut.cds <- GenomicRanges::findOverlaps(pathog.mut.gr, cds.gr, type="within")
        
        if( length(ovlp.mut.cds)==1 ){
            
            pathog.mut.cds.location <- "exonic"
            left.3bp <- GenomicRanges::GRanges(chr.id, ranges = IRanges::IRanges(BiocGenerics::start(IRanges::ranges(tx.cds.range)), BiocGenerics::start(IRanges::ranges(tx.cds.range))+2), strand=strand)
            right.3bp <- GenomicRanges::GRanges(chr.id, ranges = IRanges::IRanges(BiocGenerics::end(IRanges::ranges(tx.cds.range))-2, BiocGenerics::end(IRanges::ranges(tx.cds.range))), strand=strand)
            ovlp.mut.l3 <- GenomicRanges::findOverlaps(pathog.mut.gr, left.3bp)
            ovlp.mut.r3 <- GenomicRanges::findOverlaps(pathog.mut.gr, right.3bp)
            if( length(ovlp.mut.l3)==1 & length(ovlp.mut.r3)==0 ){
                if(strand=="+"){
                    pathog.mut.cds.location <- "exonic.start"
                }else if(strand=="-"){
                    pathog.mut.cds.location <- "exonic.stop"
                }
            }
            if( length(ovlp.mut.l3)==0 & length(ovlp.mut.r3)==1 ){
                if(strand=="+"){
                    pathog.mut.cds.location <- "exonic.stop"
                }else if(strand=="-"){
                    pathog.mut.cds.location <- "exonic.start"
                }
            }
            
            pathog.mut.start.cds <- genomeToCds(pathog.mut.start, tx.id, ensdb)
            pathog.mut.end.cds <- genomeToCds(pathog.mut.end, tx.id, ensdb)
            pathog.mut.exon.id <- cds.gr[S4Vectors::subjectHits(ovlp.mut.cds)]$exon_id
            
        }else if( length(ovlp.mut.cds)==0 ){
            
            cds.gr.pos <- cds.gr
            BiocGenerics::strand(cds.gr.pos) <- "+"
            left.splice <- GenomicRanges::promoters(cds.gr.pos, upstream=splice.region, downstream=0, use.names=TRUE)
            left.splice <- left.splice[ setdiff( 1:length(left.splice), 1 ) ]
            BiocGenerics::strand(left.splice) <- strand
            left.splice2bp <- GenomicRanges::promoters(cds.gr.pos, upstream=2, downstream=0, use.names=TRUE)
            left.splice2bp <- left.splice2bp[ setdiff( 1:length(left.splice2bp), 1 ) ]
            BiocGenerics::strand(left.splice2bp) <- strand
            
            cds.gr.neg <- cds.gr
            BiocGenerics::strand(cds.gr.neg) <- "-"
            right.splice <- GenomicRanges::promoters(cds.gr.neg, upstream=splice.region, downstream=0, use.names=TRUE)
            right.splice <- right.splice[ setdiff( 1:length(right.splice), length(right.splice) ) ]
            BiocGenerics::strand(right.splice) <- strand
            right.splice2bp <- GenomicRanges::promoters(cds.gr.neg, upstream=2, downstream=0, use.names=TRUE)
            right.splice2bp <- right.splice2bp[ setdiff( 1:length(right.splice2bp), length(right.splice2bp) ) ]
            BiocGenerics::strand(right.splice2bp) <- strand
            
            ovlp.mut.ls <- GenomicRanges::findOverlaps(pathog.mut.gr, left.splice)
            ovlp.mut.rs <- GenomicRanges::findOverlaps(pathog.mut.gr, right.splice)
            ovlp.mut.ls2 <- GenomicRanges::findOverlaps(pathog.mut.gr, left.splice2bp)
            ovlp.mut.rs2 <- GenomicRanges::findOverlaps(pathog.mut.gr, right.splice2bp)
            
            ovlp.mut.cds.any <- GenomicRanges::findOverlaps(pathog.mut.gr, cds.gr)
            ovlp.mut.noncds.any <- GenomicRanges::findOverlaps(pathog.mut.gr, noncds.gr)
            
            if( length(ovlp.mut.cds.any)>1 | length(ovlp.mut.noncds.any)>1 ){
                
                pathog.mut.cds.location <- "LGR"
                
            }else{
                
                if( length(ovlp.mut.ls)==0 & length(ovlp.mut.rs)==0 ){
                    
                    pathog.mut.cds.location <- "intronic"
                    
                }else{
                    
                    pathog.mut.cds.location <- "splice"
                    if( length(ovlp.mut.ls2)==1 & length(ovlp.mut.rs2)==0 ){
                        if(strand=="+"){
                            pathog.mut.cds.location <- "splice.acceptor"
                        }else if(strand=="-"){
                            pathog.mut.cds.location <- "splice.donor"
                        }
                    }
                    if( length(ovlp.mut.ls2)==0 & length(ovlp.mut.rs2)==1 ){
                        if(strand=="+"){
                            pathog.mut.cds.location <- "splice.donor"
                        }else if(strand=="-"){
                            pathog.mut.cds.location <- "splice.acceptor"
                        }
                    }
                    
                    pathog.mut.start.gr <- GenomicRanges::GRanges(chr.id, IRanges::IRanges(pathog.mut.start, width=1), strand=strand)
                    ovlp.mutStart.cds <- GenomicRanges::findOverlaps(pathog.mut.start.gr, cds.gr, type="within")
                    if( length(ovlp.mutStart.cds)==0 ){
                        ovlp.mutStart.noncds <- GenomicRanges::findOverlaps(pathog.mut.start.gr, noncds.gr, type="within")
                        pathog.mut.start.adjSplice <- BiocGenerics::start(IRanges::ranges(noncds.gr[S4Vectors::subjectHits(ovlp.mutStart.noncds)]))-1
                    }else{
                        pathog.mut.start.adjSplice <- pathog.mut.start
                    }
                    pathog.mut.start.cds <- genomeToCds(pathog.mut.start.adjSplice, tx.id, ensdb)
                    
                    pathog.mut.end.gr <- GenomicRanges::GRanges(chr.id, IRanges::IRanges(pathog.mut.end, width=1), strand=strand)
                    ovlp.mutEnd.cds <- GenomicRanges::findOverlaps(pathog.mut.end.gr, cds.gr, type="within")
                    if( length(ovlp.mutEnd.cds)==0 ){
                        ovlp.mutEnd.noncds <- GenomicRanges::findOverlaps(pathog.mut.end.gr, noncds.gr, type="within")
                        pathog.mut.end.adjSplice <- BiocGenerics::end(IRanges::ranges(noncds.gr[S4Vectors::subjectHits(ovlp.mutEnd.noncds)]))+1
                    }else{
                        pathog.mut.end.adjSplice <- pathog.mut.end
                    }
                    pathog.mut.end.cds <- genomeToCds(pathog.mut.end.adjSplice, tx.id, ensdb)
                    
                }
                
            }
            
        }
        
    }
    
    cds.anno <- list(
        mut.cds.location = pathog.mut.cds.location,
        mut.start.to.cds = pathog.mut.start.cds,
        mut.end.to.cds = pathog.mut.end.cds,
        mut.exon.id = pathog.mut.exon.id
    )
    
    return(cds.anno)
}

