% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/report_spec.r
\name{page_footer}
\alias{page_footer}
\title{Adds a page footer}
\usage{
page_footer(
  x,
  left = "",
  center = "",
  right = "",
  blank_row = "above",
  width = NULL
)
}
\arguments{
\item{x}{The report spec object.}

\item{left}{The left page footer text.  May be a single string or a vector
of strings.}

\item{center}{The center page footer text.  May be a single string or a
vector of strings.}

\item{right}{The right page footer text.  May be a single string or a vector
of strings.}

\item{blank_row}{Whether to create a blank row above the page footer.
Valid values are 'above' and 'none'.  Default is 'above'.}

\item{width}{Widths for left, center, and right columns of the page footer, 
passed as a vector of double values in the report unit of measure.  If the vector
contains fewer than three widths, the widths passed will be interpreted from
left to right, and any unassigned column widths will be calculated from
the remaining width on the page.  For example, if two values are assigned, they will be
interpreted as left and center, and the right column width will be calculated.
Setting a cell width to zero (0) will remove that column from the page footer
entirely. To set the left and right widths only, pass a zero for the center
cell, i.e. \code{width = c(6, 0, 3)} for a total width of 9 inches.}
}
\value{
The modified report.
}
\description{
This function adds a page footer to the report.  The page footer will appear
on each page of the report, at the bottom of the page.  The page footer
contains three sections: left, center, and right.  Content for each section
may be specified with the appropriate parameter.
}
\details{
Only one page footer is allowed per report.  The page footer will appear 
on all pages of the report.  The page footer may contain text on the 
left, right, or center. Use the appropriate parameters to specify the 
desired text for each section. Multiple strings may be passed to each section 
as a vector of strings. 

If the width of the page header
string exceeds the available space, an error will be generated. In addition,
there is a limit of 5 strings for each page footer section.  

There are two special tokens to generate page numbers: [pg] and [tpg]. 
Use [pg] to indicate the current page number.  Use [tpg] to indicate the
total number of pages in the report.  These tokens may be placed anywhere 
in the page header or page footer. 

Use the \code{blank_row} parameter to control the blank space above the 
page footer.

Each footer string must fit within the available space.  The \strong{reporter}
package will not wrap footer.  If a footer string does not fit within the 
available space, an error will be generated.  In these situations, either
shorten the footer string or split it into multiple footers that each fit 
within the available space.
}
\examples{
library(reporter)
library(magrittr)

# Create temp file path
tmp <- file.path(tempdir(), "mtcars.txt")

dat <- data.frame(name = rownames(mtcars[1:10, ]), mtcars[1:10, 1:5], 
                  stringsAsFactors = FALSE)

# Create the report object
rpt <- create_report(tmp, orientation = "portrait") \%>\% 
  page_header("Client: Motor Trend", "Study: Cars") \%>\% 
  titles("MTCARS Sample Report") \%>\% 
  add_content(create_table(dat)) \%>\% 
  page_footer(Sys.time(), right = "Page [pg] of [tpg]")

# Write the report to the file system
write_report(rpt)

# Write report to console
writeLines(readLines(tmp, encoding = "UTF-8"))

# Client: Motor Trend                                                Study: Cars
#                              MTCARS Sample Report
# 
#                name                 mpg cyl   disp   hp  drat
#                ----------------------------------------------
#                Mazda RX4             21   6    160  110   3.9
#                Mazda RX4 Wag         21   6    160  110   3.9
#                Datsun 710          22.8   4    108   93  3.85
#                Hornet 4 Drive      21.4   6    258  110  3.08
#                Hornet Sportabout   18.7   8    360  175  3.15
#                Valiant             18.1   6    225  105  2.76
#                Duster 360          14.3   8    360  245  3.21
#                Merc 240D           24.4   4  146.7   62  3.69
#                Merc 230            22.8   4  140.8   95  3.92
#                Merc 280            19.2   6  167.6  123  3.92
# 
# ...
# 
# 2020-10-17 11:53:51                                                Page 1 of 1
}
\seealso{
Other report: 
\code{\link{add_content}()},
\code{\link{create_report}()},
\code{\link{footer_image}()},
\code{\link{footnotes}()},
\code{\link{header_image}()},
\code{\link{options_fixed}()},
\code{\link{page_by}()},
\code{\link{page_header}()},
\code{\link{print.report_spec}()},
\code{\link{set_margins}()},
\code{\link{title_header}()},
\code{\link{titles}()},
\code{\link{write_report}()}
}
\concept{report}
