% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/serially.R
\name{serially}
\alias{serially}
\alias{expect_serially}
\alias{test_serially}
\title{Run several tests and a final validation in a serial manner}
\usage{
serially(
  x,
  ...,
  .list = list2(...),
  preconditions = NULL,
  actions = NULL,
  step_id = NULL,
  label = NULL,
  brief = NULL,
  active = TRUE
)

expect_serially(
  object,
  ...,
  .list = list2(...),
  preconditions = NULL,
  threshold = 1
)

test_serially(
  object,
  ...,
  .list = list2(...),
  preconditions = NULL,
  threshold = 1
)
}
\arguments{
\item{x}{\emph{A pointblank agent or a data table}

\verb{obj:<ptblank_agent>|obj:<tbl_*>} // \strong{required}

A data frame, tibble (\code{tbl_df} or \code{tbl_dbi}), Spark DataFrame
(\code{tbl_spark}), or, an \emph{agent} object of class \code{ptblank_agent} that is
commonly created with \code{\link[=create_agent]{create_agent()}}.}

\item{...}{\emph{Test/validation expressions}

\verb{<test/validation expressions>} // \strong{required} (or, use \code{.list})

A collection one-sided formulas that consist of \verb{test_*()} function calls
(e.g., \code{\link[=test_col_vals_between]{test_col_vals_between()}}, etc.) arranged in sequence of intended
interrogation order. Typically, validations up until the final one would
have some \code{threshold} value set (default is \code{1}) for short circuiting
within the series. A finishing validation function call (e.g.,
\code{\link[=col_vals_increasing]{col_vals_increasing()}}, etc.) can optionally be inserted at the end of the
series, serving as a validation step that only undergoes interrogation if
the prior tests adequately pass. An example of this is
\verb{~ test_column_exists(., a), ~ col_vals_not_null(., a)}).}

\item{.list}{\emph{Alternative to \code{...}}

\verb{<list of multiple expressions>} // \strong{required} (or, use \code{...})

Allows for the use of a list as an input alternative to \code{...}.}

\item{preconditions}{\emph{Input table modification prior to validation}

\verb{<table mutation expression>} // \emph{default:} \code{NULL} (\code{optional})

An optional expression for mutating the input table before proceeding with
the validation. This can either be provided as a one-sided R formula using
a leading \code{~} (e.g., \code{~ . \%>\% dplyr::mutate(col = col + 10)} or as a
function (e.g., \code{function(x) dplyr::mutate(x, col = col + 10)}. See the
\emph{Preconditions} section for more information.}

\item{actions}{\emph{Thresholds and actions for different states}

\verb{obj:<action_levels>} // \emph{default:} \code{NULL} (\code{optional})

A list containing threshold levels so that the validation step can react
accordingly when exceeding the set levels for different states. This is to
be created with the \code{\link[=action_levels]{action_levels()}} helper function.}

\item{step_id}{\emph{Manual setting of the step ID value}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

One or more optional identifiers for the single or multiple validation
steps generated from calling a validation function. The use of step IDs
serves to distinguish validation steps from each other and provide an
opportunity for supplying a more meaningful label compared to the step
index. By default this is \code{NULL}, and \strong{pointblank} will automatically
generate the step ID value (based on the step index) in this case. One or
more values can be provided, and the exact number of ID values should (1)
match the number of validation steps that the validation function call will
produce (influenced by the number of \code{columns} provided), (2) be an ID
string not used in any previous validation step, and (3) be a vector with
unique values.}

\item{label}{\emph{Optional label for the validation step}

\verb{vector<character>} // \emph{default:} \code{NULL} (\code{optional})

Optional label for the validation step. This label appears in the \emph{agent}
report and, for the best appearance, it should be kept quite short. See
the \emph{Labels} section for more information.}

\item{brief}{\emph{Brief description for the validation step}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

A \emph{brief} is a short, text-based description for the validation step. If
nothing is provided here then an \emph{autobrief} is generated by the \emph{agent},
using the language provided in \code{\link[=create_agent]{create_agent()}}'s \code{lang} argument (which
defaults to \code{"en"} or English). The \emph{autobrief} incorporates details of the
validation step so it's often the preferred option in most cases (where a
\code{label} might be better suited to succinctly describe the validation).}

\item{active}{\emph{Is the validation step active?}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

A logical value indicating whether the validation step should be active. If
the validation function is working with an \emph{agent}, \code{FALSE} will make the
validation step inactive (still reporting its presence and keeping indexes
for the steps unchanged). If the validation function will be operating
directly on data (no \emph{agent} involvement), then any step with \code{active = FALSE} will simply pass the data through with no validation whatsoever.
Aside from a logical vector, a one-sided R formula using a leading \code{~} can
be used with \code{.} (serving as the input data table) to evaluate to a single
logical value. With this approach, the \strong{pointblank} function
\code{\link[=has_columns]{has_columns()}} can be used to determine whether to make a validation step
active on the basis of one or more columns existing in the table
(e.g., \code{~ . \%>\% has_columns(c(d, e))}).}

\item{object}{\emph{A data table for expectations or tests}

\verb{obj:<tbl_*>} // \strong{required}

A data frame, tibble (\code{tbl_df} or \code{tbl_dbi}), or Spark DataFrame
(\code{tbl_spark}) that serves as the target table for the expectation function
or the test function.}

\item{threshold}{\emph{The failure threshold}

\code{scalar<integer|numeric>(val>=0)} // \emph{default:} \code{1}

A simple failure threshold value for use with the expectation (\code{expect_})
and the test (\code{test_}) function variants. By default, this is set to \code{1}
meaning that any single unit of failure in data validation results in an
overall test failure. Whole numbers beyond \code{1} indicate that any failing
units up to that absolute threshold value will result in a succeeding
\strong{testthat} test or evaluate to \code{TRUE}. Likewise, fractional values
(between \code{0} and \code{1}) act as a proportional failure threshold, where \code{0.15}
means that 15 percent of failing test units results in an overall test
failure.}
}
\value{
For the validation function, the return value is either a
\code{ptblank_agent} object or a table object (depending on whether an agent
object or a table was passed to \code{x}). The expectation function invisibly
returns its input but, in the context of testing data, the function is
called primarily for its potential side-effects (e.g., signaling failure).
The test function returns a logical value.
}
\description{
The \code{serially()} validation function allows for a series of tests to run in
sequence before either culminating in a final validation step or simply
exiting the series. This construction allows for pre-testing that may make
sense before a validation step. For example, there may be situations where
it's vital to check a column type before performing a validation on the same
column (since having the wrong type can result in an evaluation error for the
subsequent validation). Another serial workflow might entail having a bundle
of checks in a prescribed order and, if all pass, then the goal of this
testing has been achieved (e.g., checking if a table matches another through
a series of increasingly specific tests).

A series as specified inside \code{serially()} is composed with a listing of
calls, and we would draw upon test functions (\strong{T}) to describe tests and
optionally provide a finalizing call with a validation function (\strong{V}).
The following constraints apply:
\itemize{
\item there must be at least one test function in the series (\strong{T} -> \strong{V} is
good, \strong{V} is \emph{not})
\item there can only be one validation function call, \strong{V}; it's optional but,
if included, it must be placed at the end (\strong{T} -> \strong{T} -> \strong{V} is good,
these sequences are bad: (1) \strong{T} -> \strong{V} -> \strong{T}, (2) \strong{T} -> \strong{T} ->
\strong{V} -> \strong{V})
\item a validation function call (\strong{V}), if included, mustn't itself yield
multiple validation steps (this may happen when providing multiple \code{columns}
or any \code{segments})
}

Here's an example of how to arrange expressions:

\if{html}{\out{<div class="sourceCode">}}\preformatted{~ test_col_exists(., columns = count),
~ test_col_is_numeric(., columns = count),
~ col_vals_gt(., columns = count, value = 2)
}\if{html}{\out{</div>}}

This series concentrates on the column called \code{count} and first checks
whether the column exists, then checks if that column is numeric, and then
finally validates whether all values in the column are greater than \code{2}.

Note that in the above listing of calls, the \code{.} stands in for the target
table and is always necessary here. Also important is that all \verb{test_*()}
functions have a \code{threshold} argument that is set to \code{1} by default. Should
you need to bump up the threshold value it can be changed to a different
integer value (as an absolute threshold of failing test units) or a
decimal value between \code{0} and \code{1} (serving as a fractional threshold of
failing test units).
}
\section{Supported Input Tables}{


The types of data tables that are officially supported are:
\itemize{
\item data frames (\code{data.frame}) and tibbles (\code{tbl_df})
\item Spark DataFrames (\code{tbl_spark})
\item the following database tables (\code{tbl_dbi}):
\itemize{
\item \emph{PostgreSQL} tables (using the \code{RPostgres::Postgres()} as driver)
\item \emph{MySQL} tables (with \code{RMySQL::MySQL()})
\item \emph{Microsoft SQL Server} tables (via \strong{odbc})
\item \emph{BigQuery} tables (using \code{bigrquery::bigquery()})
\item \emph{DuckDB} tables (through \code{duckdb::duckdb()})
\item \emph{SQLite} (with \code{RSQLite::SQLite()})
}
}

Other database tables may work to varying degrees but they haven't been
formally tested (so be mindful of this when using unsupported backends with
\strong{pointblank}).
}

\section{Column Names}{


\code{columns} may be a single column (as symbol \code{a} or string \code{"a"}) or a vector
of columns (\code{c(a, b, c)} or \code{c("a", "b", "c")}). \code{{tidyselect}} helpers
are also supported, such as \code{contains("date")} and \code{where(is.double)}. If
passing an \emph{external vector} of columns, it should be wrapped in \code{all_of()}.

When multiple columns are selected by \code{columns}, the result will be an
expansion of validation steps to that number of columns (e.g.,
\code{c(col_a, col_b)} will result in the entry of two validation steps).

Previously, columns could be specified in \code{vars()}. This continues to work,
but \code{c()} offers the same capability and supersedes \code{vars()} in \code{columns}.
}

\section{Preconditions}{


Providing expressions as \code{preconditions} means \strong{pointblank} will preprocess
the target table during interrogation as a preparatory step. It might happen
that a particular validation requires a calculated column, some filtering of
rows, or the addition of columns via a join, etc. Especially for an
\emph{agent}-based report this can be advantageous since we can develop a large
validation plan with a single target table and make minor adjustments to it,
as needed, along the way.

The table mutation is totally isolated in scope to the validation step(s)
where \code{preconditions} is used. Using \strong{dplyr} code is suggested here since
the statements can be translated to SQL if necessary (i.e., if the target
table resides in a database). The code is most easily supplied as a one-sided
\strong{R} formula (using a leading \code{~}). In the formula representation, the \code{.}
serves as the input data table to be transformed (e.g., \code{~ . \%>\% dplyr::mutate(col_b = col_a + 10)}). Alternatively, a function could instead
be supplied (e.g., \code{function(x) dplyr::mutate(x, col_b = col_a + 10)}).
}

\section{Actions}{


Often, we will want to specify \code{actions} for the validation. This argument,
present in every validation function, takes a specially-crafted list
object that is best produced by the \code{\link[=action_levels]{action_levels()}} function. Read that
function's documentation for the lowdown on how to create reactions to
above-threshold failure levels in validation. The basic gist is that you'll
want at least a single threshold level (specified as either the fraction of
test units failed, or, an absolute value), often using the \code{warn_at}
argument. This is especially true when \code{x} is a table object because,
otherwise, nothing happens. For the \verb{col_vals_*()}-type functions, using
\code{action_levels(warn_at = 0.25)} or \code{action_levels(stop_at = 0.25)} are good
choices depending on the situation (the first produces a warning when a
quarter of the total test units fails, the other \code{stop()}s at the same
threshold level).
}

\section{Labels}{


\code{label} may be a single string or a character vector that matches the number
of expanded steps. \code{label} also supports \code{{glue}} syntax and exposes the
following dynamic variables contextualized to the current step:
\itemize{
\item \code{"{.step}"}: The validation step name
}

The glue context also supports ordinary expressions for further flexibility
(e.g., \code{"{toupper(.step)}"}) as long as they return a length-1 string.
}

\section{Briefs}{


Want to describe this validation step in some detail? Keep in mind that this
is only useful if \code{x} is an \emph{agent}. If that's the case, \code{brief} the agent
with some text that fits. Don't worry if you don't want to do it. The
\emph{autobrief} protocol is kicked in when \code{brief = NULL} and a simple brief will
then be automatically generated.
}

\section{YAML}{


A \strong{pointblank} agent can be written to YAML with \code{\link[=yaml_write]{yaml_write()}} and the
resulting YAML can be used to regenerate an agent (with \code{\link[=yaml_read_agent]{yaml_read_agent()}})
or interrogate the target table (via \code{\link[=yaml_agent_interrogate]{yaml_agent_interrogate()}}). When
\code{serially()} is represented in YAML (under the top-level \code{steps} key as a
list member), the syntax closely follows the signature of the validation
function. Here is an example of how a complex call of \code{serially()} as a
validation step is expressed in R code and in the corresponding YAML
representation.

R statement:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent \%>\%
  serially(
    ~ test_col_vals_lt(., columns = a, value = 8),
    ~ test_col_vals_gt(., columns = c, value = vars(a)),
    ~ col_vals_not_null(., columns = b),
    preconditions = ~ . \%>\% dplyr::filter(a < 10),
    actions = action_levels(warn_at = 0.1, stop_at = 0.2),
    label = "The `serially()` step.",
    active = FALSE
  )
}\if{html}{\out{</div>}}

YAML representation:

\if{html}{\out{<div class="sourceCode yaml">}}\preformatted{steps:
- serially:
    fns:
    - ~test_col_vals_lt(., columns = a, value = 8)
    - ~test_col_vals_gt(., columns = c, value = vars(a))
    - ~col_vals_not_null(., columns = b)
    preconditions: ~. \%>\% dplyr::filter(a < 10)
    actions:
      warn_fraction: 0.1
      stop_fraction: 0.2
    label: The `serially()` step.
    active: false
}\if{html}{\out{</div>}}

In practice, both of these will often be shorter as only the expressions for
validation steps are necessary. Arguments with default values won't be
written to YAML when using \code{\link[=yaml_write]{yaml_write()}} (though it is acceptable to include
them with their default when generating the YAML by other means). It is also
possible to preview the transformation of an agent to YAML without any
writing to disk by using the \code{\link[=yaml_agent_string]{yaml_agent_string()}} function.
}

\section{Examples}{


For all examples here, we'll use a simple table with three numeric columns
(\code{a}, \code{b}, and \code{c}). This is a very basic table but it'll be more useful when
explaining things later.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl <-
  dplyr::tibble(
    a = c(5, 2, 6),
    b = c(6, 4, 9),
    c = c(1, 2, 3)
  )

tbl
#> # A tibble: 3 x 3
#>       a     b     c
#>   <dbl> <dbl> <dbl>
#> 1     5     6     1
#> 2     2     4     2
#> 3     6     9     3
}\if{html}{\out{</div>}}
\subsection{A: Using an \code{agent} with validation functions and then \code{interrogate()}}{

The \code{serially()} function can be set up to perform a series of tests and then
perform a validation (only if all tests pass). Here, we are going to (1) test
whether columns \code{a} and \code{b} are numeric, (2) check that both don't have any
\code{NA} values, and (3) perform a finalizing validation that checks whether
values in \code{b} are greater than values in \code{a}. We'll determine if this
validation has any failing test units (there are 4 tests and a final
validation).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent_1 <-
  create_agent(tbl = tbl) \%>\%
  serially(
    ~ test_col_is_numeric(., columns = c(a, b)),
    ~ test_col_vals_not_null(., columns = c(a, b)),
    ~ col_vals_gt(., columns = b, value = vars(a))
    ) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

Printing the \code{agent} in the console shows the validation report in the
Viewer. Here is an excerpt of validation report, showing the single entry
that corresponds to the validation step demonstrated here.

\if{html}{
\out{
<img src="https://raw.githubusercontent.com/rstudio/pointblank/main/images/man_serially_1.png" alt="This image was generated from the first code example in the `serially()` help file." style="width:100\%;">
}
}

What's going on? All four of the tests passed and so the final validation
occurred. There were no failing test units in that either!

The final validation is optional and so here is a variation where only the
serial tests are performed.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent_2 <-
  create_agent(tbl = tbl) \%>\%
  serially(
    ~ test_col_is_numeric(., columns = c(a, b)),
    ~ test_col_vals_not_null(., columns = c(a, b))
  ) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

Everything is good here too:

\if{html}{
\out{
<img src="https://raw.githubusercontent.com/rstudio/pointblank/main/images/man_serially_2.png" alt="This image was generated from the second code example in the `serially()` help file." style="width:100\%;">
}
}
}

\subsection{B: Using the validation function directly on the data (no \code{agent})}{

This way of using validation functions acts as a data filter. Data is passed
through but should \code{stop()} if there is a single test unit failing. The
behavior of side effects can be customized with the \code{actions} option.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\%
  serially(
    ~ test_col_is_numeric(., columns = c(a, b)),
    ~ test_col_vals_not_null(., columns = c(a, b)),
    ~ col_vals_gt(., columns = b, value = vars(a))
  )
#> # A tibble: 3 x 3
#>       a     b     c
#>   <dbl> <dbl> <dbl>
#> 1     5     6     1
#> 2     2     4     2
#> 3     6     9     3
}\if{html}{\out{</div>}}
}

\subsection{C: Using the expectation function}{

With the \verb{expect_*()} form, we would typically perform one validation at a
time. This is primarily used in \strong{testthat} tests.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{expect_serially(
  tbl,
  ~ test_col_is_numeric(., columns = c(a, b)),
  ~ test_col_vals_not_null(., columns = c(a, b)),
  ~ col_vals_gt(., columns = b, value = vars(a))
)
}\if{html}{\out{</div>}}
}

\subsection{D: Using the test function}{

With the \verb{test_*()} form, we should get a single logical value returned to
us.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl \%>\%
  test_serially(
    ~ test_col_is_numeric(., columns = c(a, b)),
    ~ test_col_vals_not_null(., columns = c(a, b)),
    ~ col_vals_gt(., columns = b, value = vars(a))
  )
#> [1] TRUE
}\if{html}{\out{</div>}}
}
}

\section{Function ID}{

2-35
}

\seealso{
Other validation functions: 
\code{\link{col_count_match}()},
\code{\link{col_exists}()},
\code{\link{col_is_character}()},
\code{\link{col_is_date}()},
\code{\link{col_is_factor}()},
\code{\link{col_is_integer}()},
\code{\link{col_is_logical}()},
\code{\link{col_is_numeric}()},
\code{\link{col_is_posix}()},
\code{\link{col_schema_match}()},
\code{\link{col_vals_between}()},
\code{\link{col_vals_decreasing}()},
\code{\link{col_vals_equal}()},
\code{\link{col_vals_expr}()},
\code{\link{col_vals_gt}()},
\code{\link{col_vals_gte}()},
\code{\link{col_vals_in_set}()},
\code{\link{col_vals_increasing}()},
\code{\link{col_vals_lt}()},
\code{\link{col_vals_lte}()},
\code{\link{col_vals_make_set}()},
\code{\link{col_vals_make_subset}()},
\code{\link{col_vals_not_between}()},
\code{\link{col_vals_not_equal}()},
\code{\link{col_vals_not_in_set}()},
\code{\link{col_vals_not_null}()},
\code{\link{col_vals_null}()},
\code{\link{col_vals_regex}()},
\code{\link{col_vals_within_spec}()},
\code{\link{conjointly}()},
\code{\link{row_count_match}()},
\code{\link{rows_complete}()},
\code{\link{rows_distinct}()},
\code{\link{specially}()},
\code{\link{tbl_match}()}
}
\concept{validation functions}
