% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ccc_repeated.R
\name{ccc_pairwise_u_stat}
\alias{ccc_pairwise_u_stat}
\title{Repeated-Measures Lin's Concordance Correlation Coefficient (CCC)}
\usage{
ccc_pairwise_u_stat(
  data,
  response,
  method,
  subject,
  time = NULL,
  Dmat = NULL,
  delta = 1,
  ci = FALSE,
  conf_level = 0.95,
  n_threads = getOption("matrixCorr.threads", 1L),
  verbose = FALSE
)
}
\arguments{
\item{data}{A data frame containing the repeated-measures dataset.}

\item{response}{Character. Name of the numeric outcome column.}

\item{method}{Character. Name of the method column (factor with L
\eqn{\geq} 2 levels).}

\item{subject}{Character. Column identifying subjects. Every subject must
have measurements from all methods (and times, when supplied); rows with
incomplete \{subject, time, method\} coverage are dropped per pair.}

\item{time}{Character or NULL. Name of the time/repetition column. If NULL,
one time point is assumed.}

\item{Dmat}{Optional numeric weight matrix (T \eqn{\times} T) for
timepoints. Defaults to identity.}

\item{delta}{Numeric. Power exponent used in the distance computations
between method trajectories
across time points. This controls the contribution of differences between
measurements:
\itemize{
\item \code{delta = 1} (default) uses \strong{absolute differences}.
\item \code{delta = 2} uses \strong{squared differences}, more sensitive to
larger deviations.
\item \code{delta = 0} reduces to a \strong{binary distance} (presence/absence
of disagreement), analogous to a repeated-measures version of the kappa
statistic.
}
The choice of \code{delta} should reflect the penalty you want to assign to
measurement disagreement.}

\item{ci}{Logical. If TRUE, returns confidence intervals (default FALSE).}

\item{conf_level}{Confidence level for CI (default 0.95).}

\item{n_threads}{Integer (\eqn{\geq} 1). Number of OpenMP threads to use for computation.
Defaults to \code{getOption("matrixCorr.threads", 1L)}.}

\item{verbose}{Logical. If TRUE, prints diagnostic output (default FALSE).}
}
\value{
If \code{ci = FALSE}, a symmetric matrix of class \code{"ccc"} (estimates only).
If \code{ci = TRUE}, a list of class \code{"ccc"}, \code{"ccc_ci"} with elements:
\itemize{
\item \code{est}: CCC estimate matrix
\item \code{lwr.ci}: Lower bound matrix
\item \code{upr.ci}: Upper bound matrix
}
}
\description{
Computes all pairwise Lin's Concordance Correlation Coefficients (CCC)
across multiple methods (L \eqn{\geq} 2) for repeated-measures data.
Each subject must be measured by all methods across the same set of time
points or replicates.

CCC measures both accuracy (how close measurements are to the line of
equality) and precision (Pearson correlation). Confidence intervals are
optionally computed using a U-statistics-based estimator with Fisher's Z
transformation
}
\details{
This function computes pairwise Lin's Concordance Correlation Coefficient
(CCC) between methods in a repeated-measures design using a
U-statistics-based nonparametric estimator proposed by
Carrasco et al. (2013). It is computationally efficient and robust,
particularly for large-scale or balanced longitudinal designs.

Lin's CCC is defined as
\deqn{
\rho_c = \frac{2 \cdot \mathrm{cov}(X, Y)}{\sigma_X^2 + \sigma_Y^2 +
(\mu_X - \mu_Y)^2}
}{
CCC = 2 * cov(X, Y) / [var(X) + var(Y) + (mean(X) - mean(Y))^2]
}
where:
\itemize{
\item \eqn{X} and \eqn{Y} are paired measurements from two methods.
\item \eqn{\mu_X}, \eqn{\mu_Y} are means, and \eqn{\sigma_X^2},
\eqn{\sigma_Y^2} are variances.
}
\subsection{U-statistics Estimation}{

For repeated measures across \eqn{T} time points and \eqn{n} subjects we
assume
\itemize{
\item all \eqn{n(n-1)} pairs of subjects are considered to compute a
U-statistic estimator for within-method and cross-method distances.
\item if \code{delta > 0}, pairwise distances are raised to a power before
applying a time-weighted kernel matrix \eqn{D}.
\item if \code{delta = 0}, the method reduces to a version similar to a
repeated-measures kappa.
}
}

\subsection{Confidence Intervals}{

Confidence intervals are constructed using a \strong{Fisher Z-transformation}
of the CCC. Specifically,
\itemize{
\item The CCC is transformed using
\eqn{Z = 0.5 \log((1 + \rho_c) / (1 - \rho_c))}.
\item Standard errors are computed from the asymptotic variance of the
U-statistic.
\item Normal-based intervals are computed on the Z-scale and then
back-transformed to the CCC scale.
}
}

\subsection{Assumptions}{

\itemize{
\item The design must be \strong{balanced}, where all subjects must have
complete observations for all methods and time points.
\item The method is \strong{nonparametric} and does not require assumptions
of normality or linear mixed effects.
\item Weights (\code{Dmat}) allow differential importance of time points.
}

For \strong{unbalanced} or \strong{complex hierarchical} data (e.g.,
missing timepoints, covariate adjustments), consider using
\code{\link{ccc_lmm_reml}}, which uses a variance components approach
via linear mixed models.
}
}
\examples{
set.seed(123)
df <- expand.grid(subject = 1:10,
                  time = 1:2,
                  method = c("A", "B", "C"))
df$y <- rnorm(nrow(df), mean = match(df$method, c("A", "B", "C")), sd = 1)

# CCC matrix (no CIs)
ccc1 <- ccc_pairwise_u_stat(df, response = "y", method = "method",
                            subject = "subject", time = "time")
print(ccc1)
summary(ccc1)
plot(ccc1)

# With confidence intervals
ccc2 <- ccc_pairwise_u_stat(df, response = "y", method = "method",
                            subject = "subject", time = "time", ci = TRUE)
print(ccc2)
summary(ccc2)
plot(ccc2)

#------------------------------------------------------------------------
# Choosing delta based on distance sensitivity
#------------------------------------------------------------------------
# Absolute distance (L1 norm) - robust
ccc_pairwise_u_stat(df, response = "y", method = "method",
                    subject = "subject", time = "time", delta = 1)

# Squared distance (L2 norm) - amplifies large deviations
ccc_pairwise_u_stat(df, response = "y", method = "method",
                    subject = "subject", time = "time", delta = 2)

# Presence/absence of disagreement (like kappa)
ccc_pairwise_u_stat(df, response = "y", method = "method",
                    subject = "subject", time = "time", delta = 0)

}
\references{
Lin L (1989). A concordance correlation coefficient to evaluate reproducibility.
\emph{Biometrics}, 45: 255-268.

Lin L (2000). A note on the concordance correlation coefficient.
\emph{Biometrics}, 56: 324-325.

Carrasco JL, Jover L (2003). Estimating the concordance correlation coefficient:
a new approach. \emph{Computational Statistics & Data Analysis}, 47(4): 519-539.
}
\seealso{
\code{\link{ccc}}, \code{\link{ccc_lmm_reml}},
\code{\link{plot.ccc}}, \code{\link{print.ccc}}
}
\author{
Thiago de Paula Oliveira
}
