#' Mark correlation functions for inhomogeneous point patterns on linear networks.
#'
#' Mark correlation functions for inhomogeneous point patterns on linear networks.
#'
#' @usage \method{mcorrinhom}{lpp}(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani",
#'  "beisbart", "isham", "stoyancov", "schlather"),
#' r = NULL,
#' lambda = NULL,
#' method_lambda = c("kernel", "Voronoi"),
#' bw = bw.scott.iso,
#' f = NULL,
#' method = c("density", "loess"),
#' normalise = TRUE,
#' tol = 0.01,
#' ...)
#'
#' @param X An object of class lpp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \code{r} at which the mark correlation function should be evaluated.
#' @param lambda Estimated intensity at data points. If not given, it will be estimated internally. See details.
#' @param method_lambda The method to be used for estimating intensity at data points, if \code{lambda = NULL}.
#' @param bw Bandwidth method to be used for estimating intensity at data points if \code{lambda = NULL} and \code{method_lambda = "kernel"}.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param method Type of smoothing, either \code{density} or \code{loess}.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param tol Tolerance used in the calculation of the conditional mean of marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' @details
#' 
#' Most of the details are given in \code{\link[markstat]{mcorrinhom.ppp}}. However, here distances are measured via the shortest-path distances, and intensities are estimated differently. If \code{method_lambda = "kernel"}, then the function calls
#' \code{\link[spatstat.linnet]{densityQuick.lpp}} in combination with the bandwidth chosen via \code{\link[spatstat.explore]{bw.scott.iso}}. If \code{method_lambda = "Voronoi"}, the functions calls
#' \code{\link[spatstat.linnet]{densityVoronoi.lpp}} with arguments \code{f=0.2, nrep = 400} which are recommended by Moradi et al. (2019).
#' 
#' In the case of point processes on linear networks, adding correction will slow down the calculations. Given that, it is argued that the type of correction does not have significant effect on the final output, we have here ignored correction terms.
#' We refer to Eckardt and Moradi (2024)  and Moradi and Eckardt (2025) for further details.
#' 
#' If the point patten \eqn{X} has multiple real-valued marks, the function estimates the mark correlation function for each mark individually. In such case, marks are given as a \code{data.frame} whose columns represents different marks. The functions checks which columns are numeric, and for those the mark correlation function will be computed.
#' 
#' @examples
#'  library(spatstat.linnet)
#'  library(spatstat.geom)
#'  library(spatstat.explore)
#'  X <- rpoislpp(10, simplenet)
#'  marks(X) <- runif(npoints(X), 1, 10)
#'  mcorrinhom.lpp(X,  ftype = "stoyan", method = "density",
#'                 method_lambda = "kernel", bw = bw.scott.iso)

#' @return a data.frame which gives the estimated mark correlation function and the distance vector \eqn{r} at which the mark correlation function is estimated. If the point patten \eqn{X} has multiple real-valued marks, the estimated mark correlation function will be given for each mark. Name of columns will be the name of marks.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt
#' @references
#' Moradi, M., Cronie, O., Rubak, E., Lachieze-Rey, R., Mateu, J., & Baddeley, A. (2019). Resample-smoothing of Voronoi intensity estimators. Statistics and computing, 29(5), 995-1010.
#'
#' Moradi, M., & Eckardt, M. (2025). Inhomogeneous mark correlation functions for general marked point processes. arXiv e-prints, arXiv-2505.

#' @seealso \code{\link[markstat]{mcorr.lpp}}, \code{\link[markstat]{mcorrinhom.ppp}}.

#' @import spatstat.univar
#' @import spatstat.linnet
#' @import spatstat.explore
#' @import spatstat.utils
#' @import spatstat.geom 
#' @import stats
#' @export

mcorrinhom.lpp <- function(X,
                           ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                           r = NULL,
                           lambda = NULL,
                           method_lambda = c("kernel", "Voronoi"),
                           bw = bw.scott.iso,
                           f = NULL,
                           method = c("density", "loess"),
                           normalise = TRUE,
                           tol = 0.01,
                           ...){

  if (all(class(X) != "lpp")) stop("object X should be of class lpp.")

  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")

  if (missing(method)) stop("smoothing method should be chosen.")
  
  lambda_given <- lambda
  
  if(is.null(lambda)){
    
    if(method_lambda=="kernel"){
      
      lambda  <- as.numeric(densityQuick.lpp(unmark(X), sigma = bw(X), at="points", diggle=T))
      
    }else if(method_lambda=="Voronoi"){
      
      lambda <- as.numeric(densityVoronoi.lpp(X, f=0.2, nrep = 400)[X])
      
    }else{
      
      stop("You need to pick a method for intensity estimation!")
      
    }
  }else{
    lambda <- lambda
  }
  
  n <- npoints(X)
  d <- pairdist(X)
  
  if(is.null(r)){
    L <- X$domain
    rmaxdefault <- 0.98 * boundingradius(L)
    if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
    W <- Window(L)
    breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
    r <- breaks$r
  }
  
  rmax <- max(r)
  

  
  m <- marks(X)

  if (any(class(m) == "hyperframe" | class(m) == "data.frame")){
    m <- as.data.frame(m)
    num_cols <- unlist(sapply(m, is.numeric))
    s <- which(num_cols)
    
    out <- list()
    for (i in 1:length(s)) {
      marks(X) <- as.numeric(m[,s[i]])
      out[[i]] <- mcorrinhom.lpp(X, ftype = ftype, r = r, lambda = lambda, method_lambda = method_lambda, bw = bw, f = f,
                                 method = method, normalise = normalise,  tol = tol, ...)
    }

    r <- out[[1]]$r
    emps <- sapply(out, function(df) df$est)
    colnames(emps) <- names(s)
    finalout <- data.frame(r = r, emps)
    
    class(finalout) <- "mc"
    attr(finalout, "mtype") <- "real-valued"
    attr(finalout, "type") <- "global"
    attr(finalout, "ftype") <- ftype
    attr(finalout, "method") <- method
    attr(finalout, "lambda") <- lambda_given
    attr(finalout, "normalise") <- normalise
    attr(finalout, "method_lambda") <- method_lambda
    attr(finalout, "bw") <- bw
    
    return(finalout)
  }

  if (is.null(f)) {
    if (ftype == "variogram") {
      f <- function(m1, m2, mu = NULL) 0.5 * ((m1 - m2)^2)
    } else if (ftype == "stoyan") {
      f <- function(m1, m2, mu = NULL) m1 * m2
    } else if (ftype == "rcorr") {
      f <- function(m1, m2, mu = NULL) m1
    } else if (ftype == "shimatani") {
      f <- function(m1, m2, mu = NULL) (m1 - mean(m)) * (m2 - mean(m))
    } else if (ftype == "beisbart") {
      f <- function(m1, m2, mu = NULL) m1 + m2
    } else if (ftype == "isham") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "stoyancov") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "schlather") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - mu * (m1 + m2) + mu^2
    } else {
      stop("Your ftype is not supported!")
    }
  } else {
    warning("Your given test function is not among the default ones; only unnormalised version will be calculated.")
  }

  

  df <- cbind(
      dist = as.vector(d),
    id.row = rep(c(1:n), each=n),
    id.col = rep(c(1:n), n),
     int_i = rep(lambda, each=n),
     int_j = rep(lambda, n)
  )

  df.filter <- df[df[,1]<= rmax & df[,1]>0,]
  m1 <- m[df.filter[,2]]
  m2 <- m[df.filter[,3]]


  if (ftype=="schlather"){
    df.filter <- cbind(df.filter,
                       mu = as.numeric(unlist(sapply(df.filter[,1], function(d) {
                         matched <- df.filter[,3][abs(df.filter[,1] - d) <= tol]
                         paste(mean(m[matched]), collapse = ",")
                       }))))
    mu <- df.filter[,6]
    dfvario <- data.frame(d = df.filter[,1],
                          ff = (f(m1, m2, mu)),
                          int = df.filter[,"int_i"]*df.filter[,"int_j"])
  }else{
    dfvario <- data.frame(d = df.filter[,1],
                          ff = (f(m1,m2)),
                          int = df.filter[,"int_i"]*df.filter[,"int_j"]
    )
  }


  if(method=="density"){

    Kf <- unnormdensity(dfvario$d, weights = dfvario$ff/dfvario$int,
                        from=min(r), to=max(r), n=length(r), ...)$y
    K1 <- unnormdensity(dfvario$d, weights=1/dfvario$int,
                        from=min(r), to=max(r), n=length(r), ...)$y
    Eff <- Kf/K1

  }else if(method=="loess"){

    lo <- loess(ff~d,data = dfvario,...)
    Eff <- predict(lo, newdata=data.frame(d=r))

  }else{
    stop("method should currently be either loess or density!!!")
  }


  if(normalise){
    if(ftype=="stoyan"){
      out <- Eff/(mean(m)^2)
    } else if(ftype=="variogram" | ftype=="isham" | ftype=="schlather" | ftype=="shimatani"){
      out <- Eff/var(m)
    }else if(ftype=="rcorr"){
      out <- Eff/mean(m)
    }else if(ftype=="Beisbart"){
      out <- Eff/(2*mean(m))
    }else if(ftype=="stoyancov"){
      out <- Eff
    }else{
      stop("your ftype is not supported!!")
    }
  }else{
    out <- Eff
  }

  out <- as.data.frame(cbind(r = r, est = out))
  
  if(ncol(out) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(out) <- "mc"
  attr(out, "mtype") <- "real-valued"
  attr(out, "type") <- type
  attr(out, "ftype") <- ftype
  attr(out, "method") <- method
  attr(out, "lambda") <- lambda_given
  attr(out, "normalise") <- normalise
  attr(out, "method_lambda") <- method_lambda
  attr(out, "bw") <- bw

  return(out)
}
