/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      

static void  *H5O__mtime_new_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                    size_t p_size, const uint8_t *p);
static herr_t H5O__mtime_new_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                    const void *_mesg);
static size_t H5O__mtime_new_size(const H5F_t *f, bool disable_shared, const void *_mesg);

static void  *H5O__mtime_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                size_t p_size, const uint8_t *p);
static herr_t H5O__mtime_encode(H5F_t *f, bool disable_shared, size_t p_size, uint8_t *p, const void *_mesg);
static void  *H5O__mtime_copy(const void *_mesg, void *_dest);
static size_t H5O__mtime_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__mtime_free(void *_mesg);
static herr_t H5O__mtime_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_MTIME[1] = {{
    H5O_MTIME_ID,      
    "mtime",           
    sizeof(time_t),    
    0,                 
    H5O__mtime_decode, 
    H5O__mtime_encode, 
    H5O__mtime_copy,   
    H5O__mtime_size,   
    NULL,              
    H5O__mtime_free,   
    NULL,              
    NULL,              
    NULL,              
    NULL,              
    NULL,              
    NULL,              
    NULL,              
    NULL,              
    NULL,              
    H5O__mtime_debug   
}};

const H5O_msg_class_t H5O_MSG_MTIME_NEW[1] = {{
    H5O_MTIME_NEW_ID,      
    "mtime_new",           
    sizeof(time_t),        
    0,                     
    H5O__mtime_new_decode, 
    H5O__mtime_new_encode, 
    H5O__mtime_copy,       
    H5O__mtime_new_size,   
    NULL,                  
    H5O__mtime_free,       
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    H5O__mtime_debug       
}};

#define H5O_MTIME_VERSION 1

H5FL_DEFINE(time_t);

static void *
H5O__mtime_new_decode(H5F_t H5_ATTR_NDEBUG_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh,
                      unsigned H5_ATTR_UNUSED mesg_flags, unsigned H5_ATTR_UNUSED *ioflags, size_t p_size,
                      const uint8_t *p)
{
    const uint8_t *p_end = p + p_size - 1; 
    time_t        *mesg  = NULL;
    uint32_t       tmp_time;         
    void          *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_MTIME_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for mtime message");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 3, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    p += 3;

    
    if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT32DECODE(p, tmp_time);

    
    if (NULL == (mesg = H5FL_MALLOC(time_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    *mesg = (time_t)tmp_time;

    
    ret_value = mesg;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__mtime_decode(H5F_t H5_ATTR_NDEBUG_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh,
                  unsigned H5_ATTR_UNUSED mesg_flags, unsigned H5_ATTR_UNUSED *ioflags, size_t p_size,
                  const uint8_t *p)
{
    const uint8_t *p_end = p + p_size - 1; 
    time_t        *mesg  = NULL;
    time_t         the_time;
    struct tm      tm;
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 16, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    for (int i = 0; i < 14; i++)
        if (!isdigit(p[i]))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "badly formatted modification time message");

    
    memset(&tm, 0, sizeof tm);
    tm.tm_year  = (p[0] - '0') * 1000 + (p[1] - '0') * 100 + (p[2] - '0') * 10 + (p[3] - '0') - 1900;
    tm.tm_mon   = (p[4] - '0') * 10 + (p[5] - '0') - 1;
    tm.tm_mday  = (p[6] - '0') * 10 + (p[7] - '0');
    tm.tm_hour  = (p[8] - '0') * 10 + (p[9] - '0');
    tm.tm_min   = (p[10] - '0') * 10 + (p[11] - '0');
    tm.tm_sec   = (p[12] - '0') * 10 + (p[13] - '0');
    tm.tm_isdst = -1; 
    if ((time_t)-1 == (the_time = H5_make_time(&tm)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "can't construct time info");

    
    if (NULL == (mesg = H5FL_MALLOC(time_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    *mesg = the_time;

    
    ret_value = mesg;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__mtime_new_encode(H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared,
                      size_t H5_ATTR_UNUSED p_size, uint8_t *p, const void *_mesg)
{
    const time_t *mesg = (const time_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(mesg);

    
    *p++ = H5O_MTIME_VERSION;

    
    *p++ = 0;
    *p++ = 0;
    *p++ = 0;

    
    UINT32ENCODE(p, *mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__mtime_encode(H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, size_t p_size, uint8_t *p,
                  const void *_mesg)
{
    const time_t *mesg = (const time_t *)_mesg;
    struct tm    *tm;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(mesg);

    
    tm = gmtime(mesg);
    snprintf((char *)p, p_size, "%04d%02d%02d%02d%02d%02d", 1900 + tm->tm_year, 1 + tm->tm_mon, tm->tm_mday,
             tm->tm_hour, tm->tm_min, tm->tm_sec);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__mtime_copy(const void *_mesg, void *_dest)
{
    const time_t *mesg      = (const time_t *)_mesg;
    time_t       *dest      = (time_t *)_dest;
    void         *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);
    if (!dest && NULL == (dest = H5FL_MALLOC(time_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *mesg;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__mtime_new_size(const H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared,
                    const void H5_ATTR_UNUSED *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);

    FUNC_LEAVE_NOAPI(8)
} 

static size_t
H5O__mtime_size(const H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared,
                const void H5_ATTR_UNUSED *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);

    FUNC_LEAVE_NOAPI(16)
} 

static herr_t
H5O__mtime_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(time_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__mtime_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const time_t *mesg = (const time_t *)_mesg;
    struct tm    *tm;
    char          buf[128];

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    tm = localtime(mesg);

    strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %Z", tm);
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Time:", buf);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
