/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5HLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

static void  *H5O__efl_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags, size_t p_size,
                              const uint8_t *p);
static herr_t H5O__efl_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                              const void *_mesg);
static void  *H5O__efl_copy(const void *_mesg, void *_dest);
static size_t H5O__efl_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__efl_reset(void *_mesg);
static void  *H5O__efl_copy_file(H5F_t *file_src, void *mesg_src, H5F_t *file_dst, bool *recompute_size,
                                 unsigned *mesg_flags, H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__efl_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_EFL[1] = {{
    H5O_EFL_ID,           
    "external file list", 
    sizeof(H5O_efl_t),    
    0,                    
    H5O__efl_decode,      
    H5O__efl_encode,      
    H5O__efl_copy,        
    H5O__efl_size,        
    H5O__efl_reset,       
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    NULL,                 
    H5O__efl_copy_file,   
    NULL,                 
    NULL,                 
    NULL,                 
    H5O__efl_debug        
}};

#define H5O_EFL_VERSION 1

static void *
H5O__efl_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_efl_t     *mesg = NULL;
    int            version;
    const uint8_t *p_end = p + p_size - 1; 
    const char    *s     = NULL;
    H5HL_t        *heap  = NULL;
    size_t         block_size;       
    void          *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(p);

    if (NULL == (mesg = (H5O_efl_t *)H5MM_calloc(sizeof(H5O_efl_t))))
        HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    version = *p++;
    if (version != H5O_EFL_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for external file list message");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 3, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    p += 3;

    
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, mesg->nalloc);
    if (mesg->nalloc <= 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad number of allocated slots when parsing efl msg");

    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, mesg->nused);
    if (mesg->nused > mesg->nalloc)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad number of in-use slots when parsing efl msg");

    
    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_addr_decode(f, &p, &(mesg->heap_addr));
    if (H5_addr_defined(mesg->heap_addr) == false)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad local heap address when parsing efl msg");

    
    mesg->slot = (H5O_efl_entry_t *)H5MM_calloc(mesg->nalloc * sizeof(H5O_efl_entry_t));
    if (NULL == mesg->slot)
        HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, NULL, "memory allocation failed");

    if (NULL == (heap = H5HL_protect(f, mesg->heap_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, NULL, "unable to protect local heap");

#ifdef H5O_DEBUG
    
    s = (const char *)H5HL_offset_into(heap, 0);
    if (s == NULL)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "could not obtain pointer into local heap");
    if (*s != '\0')
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "entry at offset 0 in local heap not an empty string");
#endif

    
    block_size = H5HL_heap_get_size(heap);

    for (size_t u = 0; u < mesg->nused; u++) {

        hsize_t offset = 0;

        
        if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_DECODE_LENGTH(f, p, mesg->slot[u].name_offset);

        if ((s = (const char *)H5HL_offset_into(heap, mesg->slot[u].name_offset)) == NULL)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "unable to get external file name");
        if (*s == '\0')
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "invalid external file name");
        mesg->slot[u].name = H5MM_strndup(s, (block_size - mesg->slot[u].name_offset));
        if (mesg->slot[u].name == NULL)
            HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, NULL, "string duplication failed");

        
        if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_DECODE_LENGTH(f, p, offset); 
        mesg->slot[u].offset = (HDoff_t)offset;

        
        if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_DECODE_LENGTH(f, p, mesg->slot[u].size);
    }

    if (H5HL_unprotect(heap) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, NULL, "unable to unprotect local heap");

    
    ret_value = mesg;

done:
    if (ret_value == NULL) {
        if (mesg != NULL) {
            if (mesg->slot != NULL) {
                for (size_t u = 0; u < mesg->nused; u++)
                    H5MM_xfree(mesg->slot[u].name);
                H5MM_xfree(mesg->slot);
            }
            H5MM_xfree(mesg);
        }
        if (heap != NULL)
            if (H5HL_unprotect(heap) < 0)
                HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, NULL, "unable to unprotect local heap");
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__efl_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                const void *_mesg)
{
    const H5O_efl_t *mesg = (const H5O_efl_t *)_mesg;
    size_t           u; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);
    assert(p);

    
    *p++ = H5O_EFL_VERSION;

    
    *p++ = 0;
    *p++ = 0;
    *p++ = 0;

    
    assert(mesg->nalloc > 0);
    UINT16ENCODE(p, mesg->nused); 
    assert(mesg->nused > 0 && mesg->nused <= mesg->nalloc);
    UINT16ENCODE(p, mesg->nused);

    
    assert(H5_addr_defined(mesg->heap_addr));
    H5F_addr_encode(f, &p, mesg->heap_addr);

    
    for (u = 0; u < mesg->nused; u++) {
        
        assert(mesg->slot[u].name_offset);
        H5F_ENCODE_LENGTH(f, p, mesg->slot[u].name_offset);
        H5F_ENCODE_LENGTH(f, p, (hsize_t)mesg->slot[u].offset);
        H5F_ENCODE_LENGTH(f, p, mesg->slot[u].size);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__efl_copy(const void *_mesg, void *_dest)
{
    const H5O_efl_t *mesg = (const H5O_efl_t *)_mesg;
    H5O_efl_t       *dest = (H5O_efl_t *)_dest;
    size_t           u;                      
    bool             slot_allocated = false; 
    void            *ret_value      = NULL;  

    FUNC_ENTER_PACKAGE

    
    assert(mesg);

    
    if (!dest && NULL == (dest = (H5O_efl_t *)H5MM_calloc(sizeof(H5O_efl_t))))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, NULL, "can't allocate efl message");

    
    *dest = *mesg;

    
    if (dest->nalloc > 0) {
        if (NULL == (dest->slot = (H5O_efl_entry_t *)H5MM_calloc(dest->nalloc * sizeof(H5O_efl_entry_t))))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, NULL, "can't allocate efl message slots");
        slot_allocated = true;
        for (u = 0; u < mesg->nused; u++) {
            dest->slot[u] = mesg->slot[u];
            if (NULL == (dest->slot[u].name = H5MM_xstrdup(mesg->slot[u].name)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, NULL, "can't allocate efl message slot name");
        } 
    }     

    
    ret_value = dest;

done:
    if (NULL == ret_value) {
        if (slot_allocated) {
            for (u = 0; u < dest->nused; u++)
                if (dest->slot[u].name != NULL && dest->slot[u].name != mesg->slot[u].name)
                    dest->slot[u].name = (char *)H5MM_xfree(dest->slot[u].name);
            dest->slot = (H5O_efl_entry_t *)H5MM_xfree(dest->slot);
        } 
        if (NULL == _dest)
            dest = (H5O_efl_t *)H5MM_xfree(dest);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__efl_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_efl_t *mesg      = (const H5O_efl_t *)_mesg;
    size_t           ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);

    ret_value = (size_t)H5F_SIZEOF_ADDR(f) +                
                2 +                                         
                2 +                                         
                4 +                                         
                mesg->nused * ((size_t)H5F_SIZEOF_SIZE(f) + 
                               (size_t)H5F_SIZEOF_SIZE(f) + 
                               (size_t)H5F_SIZEOF_SIZE(f)); 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__efl_reset(void *_mesg)
{
    H5O_efl_t *mesg = (H5O_efl_t *)_mesg;
    size_t     u; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(mesg);

    
    if (mesg->slot) {
        for (u = 0; u < mesg->nused; u++) {
            mesg->slot[u].name        = (char *)H5MM_xfree(mesg->slot[u].name);
            mesg->slot[u].name_offset = 0;
        } 
        mesg->slot = (H5O_efl_entry_t *)H5MM_xfree(mesg->slot);
    } 
    mesg->heap_addr = HADDR_UNDEF;
    mesg->nused = mesg->nalloc = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O_efl_total_size(const H5O_efl_t *efl, hsize_t *size)
{
    hsize_t total_size = 0, tmp;
    herr_t  ret_value  = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (efl->nused > 0 && H5O_EFL_UNLIMITED == efl->slot[efl->nused - 1].size)
        *size = H5O_EFL_UNLIMITED;
    else {
        size_t u; 

        for (u = 0; u < efl->nused; u++, total_size = tmp) {
            tmp = total_size + efl->slot[u].size;
            if (tmp < total_size)
                HGOTO_ERROR(H5E_EFL, H5E_OVERFLOW, FAIL, "total external storage size overflowed");
        } 

        
        *size = total_size;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__efl_copy_file(H5F_t H5_ATTR_UNUSED *file_src, void *mesg_src, H5F_t *file_dst,
                   bool H5_ATTR_UNUSED *recompute_size, unsigned H5_ATTR_UNUSED *mesg_flags,
                   H5O_copy_t H5_ATTR_UNUSED *cpy_info, void H5_ATTR_UNUSED *_udata)
{
    H5O_efl_t *efl_src = (H5O_efl_t *)mesg_src;
    H5O_efl_t *efl_dst = NULL;
    H5HL_t    *heap    = NULL; 
    size_t     idx, size, name_offset, heap_size;
    void      *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_TAG(H5AC__COPIED_TAG)

    
    assert(efl_src);
    assert(file_dst);

    
    if (NULL == (efl_dst = (H5O_efl_t *)H5MM_calloc(sizeof(H5O_efl_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    H5MM_memcpy(efl_dst, efl_src, sizeof(H5O_efl_t));

    
    heap_size = H5HL_ALIGN(1); 
    for (idx = 0; idx < efl_src->nused; idx++)
        heap_size += H5HL_ALIGN(strlen(efl_src->slot[idx].name) + 1);

    
    if (H5HL_create(file_dst, heap_size, &efl_dst->heap_addr ) < 0)
        HGOTO_ERROR(H5E_EFL, H5E_CANTINIT, NULL, "can't create heap");

    
    if (NULL == (heap = H5HL_protect(file_dst, efl_dst->heap_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_EFL, H5E_PROTECT, NULL, "unable to protect EFL file name heap");

    
    if (H5HL_insert(file_dst, heap, (size_t)1, "", &name_offset) < 0)
        HGOTO_ERROR(H5E_EFL, H5E_CANTINSERT, NULL, "can't insert file name into heap");
    assert(0 == name_offset);

    
    if (efl_src->nalloc > 0) {
        size = efl_src->nalloc * sizeof(H5O_efl_entry_t);
        if ((efl_dst->slot = (H5O_efl_entry_t *)H5MM_calloc(size)) == NULL)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

        
        H5MM_memcpy(efl_dst->slot, efl_src->slot, size);
    } 

    
    for (idx = 0; idx < efl_src->nused; idx++) {
        efl_dst->slot[idx].name = H5MM_xstrdup(efl_src->slot[idx].name);
        if (H5HL_insert(file_dst, heap, strlen(efl_dst->slot[idx].name) + 1, efl_dst->slot[idx].name,
                        &(efl_dst->slot[idx].name_offset)) < 0)
            HGOTO_ERROR(H5E_EFL, H5E_CANTINSERT, NULL, "can't insert file name into heap");
    }

    
    ret_value = efl_dst;

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_EFL, H5E_PROTECT, NULL, "unable to unprotect EFL file name heap");
    if (!ret_value)
        if (efl_dst)
            H5MM_xfree(efl_dst);

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static herr_t
H5O__efl_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_efl_t *mesg = (const H5O_efl_t *)_mesg;
    size_t           u;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth, "Heap address:", mesg->heap_addr);

    Rfprintf(stream, "%*s%-*s %llu/%llu\n", indent, "", fwidth, "Slots used/allocated:", (unsigned long long)mesg->nused,
            (unsigned long long)mesg->nalloc);

    for (u = 0; u < mesg->nused; u++) {
        char buf[64];

        snprintf(buf, sizeof(buf), "File %llu", (unsigned long long)u);
        Rfprintf(stream, "%*s%s:\n", indent, "", buf);

        Rfprintf(stream, "%*s%-*s \"%s\"\n", indent + 3, "", MAX(fwidth - 3, 0), "Name:", mesg->slot[u].name);

        Rfprintf(stream, "%*s%-*s %llu\n", indent + 3, "", MAX(fwidth - 3, 0),
                "Name offset:", (unsigned long long)mesg->slot[u].name_offset);

        Rfprintf(stream, "%*s%-*s %" PRIdMAX "\n", indent + 3, "", MAX(fwidth - 3, 0),
                "Offset of data in file:", (intmax_t)(mesg->slot[u].offset));

        Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent + 3, "", MAX(fwidth - 3, 0),
                "Bytes reserved for data:", (mesg->slot[u].size));
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
