/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Tpkg.h"      
#include "H5VLprivate.h" 

static herr_t H5T__vlen_mem_seq_getlen(H5VL_object_t *file, const void *_vl, size_t *len);
static void  *H5T__vlen_mem_seq_getptr(void *_vl);
static herr_t H5T__vlen_mem_seq_isnull(const H5VL_object_t *file, void *_vl, bool *isnull);
static herr_t H5T__vlen_mem_seq_setnull(H5VL_object_t *file, void *_vl, void *_bg);
static herr_t H5T__vlen_mem_seq_read(H5VL_object_t *file, void *_vl, void *_buf, size_t len);
static herr_t H5T__vlen_mem_seq_write(H5VL_object_t *file, const H5T_vlen_alloc_info_t *vl_alloc_info,
                                      void *_vl, void *_buf, void *_bg, size_t seq_len, size_t base_size);

static herr_t H5T__vlen_mem_str_getlen(H5VL_object_t *file, const void *_vl, size_t *len);
static void  *H5T__vlen_mem_str_getptr(void *_vl);
static herr_t H5T__vlen_mem_str_isnull(const H5VL_object_t *file, void *_vl, bool *isnull);
static herr_t H5T__vlen_mem_str_setnull(H5VL_object_t *file, void *_vl, void *_bg);
static herr_t H5T__vlen_mem_str_read(H5VL_object_t *file, void *_vl, void *_buf, size_t len);
static herr_t H5T__vlen_mem_str_write(H5VL_object_t *file, const H5T_vlen_alloc_info_t *vl_alloc_info,
                                      void *_vl, void *_buf, void *_bg, size_t seq_len, size_t base_size);

static herr_t H5T__vlen_disk_getlen(H5VL_object_t *file, const void *_vl, size_t *len);
static herr_t H5T__vlen_disk_isnull(const H5VL_object_t *file, void *_vl, bool *isnull);
static herr_t H5T__vlen_disk_setnull(H5VL_object_t *file, void *_vl, void *_bg);
static herr_t H5T__vlen_disk_read(H5VL_object_t *file, void *_vl, void *_buf, size_t len);
static herr_t H5T__vlen_disk_write(H5VL_object_t *file, const H5T_vlen_alloc_info_t *vl_alloc_info, void *_vl,
                                   void *_buf, void *_bg, size_t seq_len, size_t base_size);
static herr_t H5T__vlen_disk_delete(H5VL_object_t *file, void *_vl);

static const H5T_vlen_class_t H5T_vlen_mem_seq_g = {
    H5T__vlen_mem_seq_getlen,  
    H5T__vlen_mem_seq_getptr,  
    H5T__vlen_mem_seq_isnull,  
    H5T__vlen_mem_seq_setnull, 
    H5T__vlen_mem_seq_read,    
    H5T__vlen_mem_seq_write,   
    NULL                       
};

static const H5T_vlen_class_t H5T_vlen_mem_str_g = {
    H5T__vlen_mem_str_getlen,  
    H5T__vlen_mem_str_getptr,  
    H5T__vlen_mem_str_isnull,  
    H5T__vlen_mem_str_setnull, 
    H5T__vlen_mem_str_read,    
    H5T__vlen_mem_str_write,   
    NULL                       
};

static const H5T_vlen_class_t H5T_vlen_disk_g = {
    H5T__vlen_disk_getlen,  
    NULL,                   
    H5T__vlen_disk_isnull,  
    H5T__vlen_disk_setnull, 
    H5T__vlen_disk_read,    
    H5T__vlen_disk_write,   
    H5T__vlen_disk_delete   
};

hid_t
H5Tvlen_create(hid_t base_id)
{
    H5T_t *base = NULL; 
    H5T_t *dt   = NULL; 
    hid_t  ret_value;   

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (base = (H5T_t *)H5I_object_verify(base_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not an valid base datatype");

    
    if ((dt = H5T__vlen_create(base)) == NULL)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, H5I_INVALID_HID, "invalid VL location");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

H5T_t *
H5T__vlen_create(const H5T_t *base)
{
    H5T_t *dt        = NULL; 
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(base);

    
    if (NULL == (dt = H5T__alloc()))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, NULL, "memory allocation failed");
    dt->shared->type = H5T_VLEN;

    
    dt->shared->force_conv = true;
    if (NULL == (dt->shared->parent = H5T_copy(base, H5T_COPY_ALL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy base datatype");

    
    dt->shared->version = base->shared->version;

    
    dt->shared->u.vlen.type = H5T_VLEN_SEQUENCE;

    
    if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "invalid datatype location");

    
    ret_value = dt;

done:
    if (!ret_value)
        if (dt && H5T_close_real(dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, NULL, "unable to release datatype info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5T__vlen_set_loc(H5T_t *dt, H5VL_object_t *file, H5T_loc_t loc)
{
    htri_t ret_value = false; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);
    assert(loc >= H5T_LOC_BADLOC && loc < H5T_LOC_MAXLOC);

    
    if (loc != dt->shared->u.vlen.loc || file != dt->shared->u.vlen.file) {
        switch (loc) {
            case H5T_LOC_MEMORY: 
                assert(NULL == file);

                
                dt->shared->u.vlen.loc = H5T_LOC_MEMORY;

                if (dt->shared->u.vlen.type == H5T_VLEN_SEQUENCE) {
                    
                    dt->shared->size = sizeof(hvl_t);

                    
                    dt->shared->u.vlen.cls = &H5T_vlen_mem_seq_g;
                } 
                else if (dt->shared->u.vlen.type == H5T_VLEN_STRING) {
                    
                    dt->shared->size = sizeof(char *);

                    
                    dt->shared->u.vlen.cls = &H5T_vlen_mem_str_g;
                } 
                else
                    assert(0 && "Invalid VL type");

                
                if (dt->shared->owned_vol_obj) {
                    if (H5VL_free_object(dt->shared->owned_vol_obj) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close owned VOL object");
                    dt->shared->owned_vol_obj = NULL;
                } 

                
                dt->shared->u.vlen.file = NULL;
                break;

            
            case H5T_LOC_DISK: {
                H5VL_file_cont_info_t cont_info = {H5VL_CONTAINER_INFO_VERSION, 0, 0, 0};
                H5VL_file_get_args_t  vol_cb_args; 

                assert(file);

                
                dt->shared->u.vlen.loc = H5T_LOC_DISK;

                
                vol_cb_args.op_type                 = H5VL_FILE_GET_CONT_INFO;
                vol_cb_args.args.get_cont_info.info = &cont_info;

                
                if (H5VL_file_get(file, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to get container info");

                
                dt->shared->size = 4 + cont_info.blob_id_size;

                
                
                dt->shared->u.vlen.cls = &H5T_vlen_disk_g;

                
                dt->shared->u.vlen.file = file;

                
                if (H5T_own_vol_obj(dt, file) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't give ownership of VOL object");
                break;
            }

            case H5T_LOC_BADLOC:
                
                dt->shared->u.vlen.loc = H5T_LOC_BADLOC;

                
                dt->shared->u.vlen.cls = NULL;

                
                dt->shared->u.vlen.file = NULL;
                break;

            case H5T_LOC_MAXLOC:
                
            default:
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "invalid VL datatype location");
        } 

        
        ret_value = true;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__vlen_mem_seq_getlen(H5VL_object_t H5_ATTR_UNUSED *file, const void *_vl, size_t *len)
{
    hvl_t vl; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(_vl);
    assert(len);

    
    H5MM_memcpy(&vl, _vl, sizeof(hvl_t));

    *len = vl.len;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5T__vlen_mem_seq_getptr(void *_vl)
{
    hvl_t vl; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(_vl);

    
    H5MM_memcpy(&vl, _vl, sizeof(hvl_t));

    FUNC_LEAVE_NOAPI(vl.p)
} 

static herr_t
H5T__vlen_mem_seq_isnull(const H5VL_object_t H5_ATTR_UNUSED *file, void *_vl, bool *isnull)
{
    hvl_t vl; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(_vl);

    
    H5MM_memcpy(&vl, _vl, sizeof(hvl_t));

    *isnull = ((vl.len == 0 || vl.p == NULL) ? true : false);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5T__vlen_mem_seq_setnull(H5VL_object_t H5_ATTR_UNUSED *file, void *_vl, void H5_ATTR_UNUSED *_bg)
{
    hvl_t vl; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(_vl);

    
    vl.len = 0;
    vl.p   = NULL;

    
    H5MM_memcpy(_vl, &vl, sizeof(hvl_t));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5T__vlen_mem_seq_read(H5VL_object_t H5_ATTR_UNUSED *file, void *_vl, void *buf, size_t len)
{
    hvl_t vl; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(buf);
    assert(_vl);

    
    H5MM_memcpy(&vl, _vl, sizeof(hvl_t));
    assert(vl.p);

    H5MM_memcpy(buf, vl.p, len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5T__vlen_mem_seq_write(H5VL_object_t H5_ATTR_UNUSED *file, const H5T_vlen_alloc_info_t *vl_alloc_info,
                        void *_vl, void *buf, void H5_ATTR_UNUSED *_bg, size_t seq_len, size_t base_size)
{
    hvl_t  vl;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(_vl);
    assert(buf);

    if (seq_len) {
        size_t len = seq_len * base_size; 

        
        if (vl_alloc_info->alloc_func != NULL) {
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    vl.p = (vl_alloc_info->alloc_func)(len, vl_alloc_info->alloc_info);
                }
            H5_AFTER_USER_CB(FAIL)
            if (NULL == vl.p)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL,
                            "application memory allocation routine failed for VL data");
        }    
        else 
            if (NULL == (vl.p = malloc(len)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "memory allocation failed for VL data");

        
        H5MM_memcpy(vl.p, buf, len);
    } 
    else
        vl.p = NULL;

    
    vl.len = seq_len;

    
    H5MM_memcpy(_vl, &vl, sizeof(hvl_t));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__vlen_mem_str_getlen(H5VL_object_t H5_ATTR_UNUSED *file, const void *_vl, size_t *len)
{
    const char *s = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(_vl);

    
    H5MM_memcpy(&s, _vl, sizeof(char *));

    *len = strlen(s);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5T__vlen_mem_str_getptr(void *_vl)
{
    char *s = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(_vl);

    
    H5MM_memcpy(&s, _vl, sizeof(char *));

    FUNC_LEAVE_NOAPI(s)
} 

static herr_t
H5T__vlen_mem_str_isnull(const H5VL_object_t H5_ATTR_UNUSED *file, void *_vl, bool *isnull)
{
    char *s = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    H5MM_memcpy(&s, _vl, sizeof(char *));

    *isnull = (s == NULL ? true : false);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5T__vlen_mem_str_setnull(H5VL_object_t H5_ATTR_UNUSED *file, void *_vl, void H5_ATTR_UNUSED *_bg)
{
    char *t = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    H5MM_memcpy(_vl, &t, sizeof(char *));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5T__vlen_mem_str_read(H5VL_object_t H5_ATTR_UNUSED *file, void *_vl, void *buf, size_t len)
{
    char *s; 

    FUNC_ENTER_PACKAGE_NOERR

    if (len > 0) {
        assert(buf);
        assert(_vl);

        
        H5MM_memcpy(&s, _vl, sizeof(char *));
        H5MM_memcpy(buf, s, len);
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5T__vlen_mem_str_write(H5VL_object_t H5_ATTR_UNUSED *file, const H5T_vlen_alloc_info_t *vl_alloc_info,
                        void *_vl, void *buf, void H5_ATTR_UNUSED *_bg, size_t seq_len, size_t base_size)
{
    char  *t;                   
    size_t len;                 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(buf);

    
    if (vl_alloc_info->alloc_func != NULL) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                t = (vl_alloc_info->alloc_func)((seq_len + 1) * base_size, vl_alloc_info->alloc_info);
            }
        H5_AFTER_USER_CB(FAIL)
        if (NULL == t)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL,
                        "application memory allocation routine failed for VL data");
    }    
    else 
        if (NULL == (t = (char *)malloc((seq_len + 1) * base_size)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "memory allocation failed for VL data");

    
    len = (seq_len * base_size);
    H5MM_memcpy(t, buf, len);
    t[len] = '\0';

    
    H5MM_memcpy(_vl, &t, sizeof(char *));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__vlen_disk_getlen(H5VL_object_t H5_ATTR_UNUSED *file, const void *_vl, size_t *seq_len)
{
    const uint8_t *vl = (const uint8_t *)_vl; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(vl);
    assert(seq_len);

    
    UINT32DECODE(vl, *seq_len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5T__vlen_disk_isnull(const H5VL_object_t *file, void *_vl, bool *isnull)
{
    H5VL_blob_specific_args_t vol_cb_args;                
    uint8_t                  *vl        = (uint8_t *)_vl; 
    herr_t                    ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(vl);
    assert(isnull);

    
    vl += 4;

    
    vol_cb_args.op_type             = H5VL_BLOB_ISNULL;
    vol_cb_args.args.is_null.isnull = isnull;

    
    if (H5VL_blob_specific(file, vl, &vol_cb_args) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to check if a blob ID is 'nil'");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__vlen_disk_setnull(H5VL_object_t *file, void *_vl, void *bg)
{
    H5VL_blob_specific_args_t vol_cb_args;                
    uint8_t                  *vl        = (uint8_t *)_vl; 
    herr_t                    ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(vl);

    
    if (bg != NULL)
        
        if (H5T__vlen_disk_delete(file, bg) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREMOVE, FAIL, "unable to remove background heap object");

    
    UINT32ENCODE(vl, 0);

    
    vol_cb_args.op_type = H5VL_BLOB_SETNULL;

    
    if (H5VL_blob_specific(file, vl, &vol_cb_args) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "unable to set a blob ID to 'nil'");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__vlen_disk_read(H5VL_object_t *file, void *_vl, void *buf, size_t len)
{
    const uint8_t *vl        = (const uint8_t *)_vl; 
    herr_t         ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(vl);
    assert(buf);

    
    vl += 4;

    
    if (H5VL_blob_get(file, vl, buf, len, NULL) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to get blob");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__vlen_disk_write(H5VL_object_t *file, const H5T_vlen_alloc_info_t H5_ATTR_UNUSED *vl_alloc_info,
                     void *_vl, void *buf, void *_bg, size_t seq_len, size_t base_size)
{
    uint8_t *vl        = (uint8_t *)_vl; 
    uint8_t *bg        = (uint8_t *)_bg; 
    herr_t   ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(vl);
    assert(seq_len == 0 || buf);
    assert(file);

    
    if (bg != NULL)
        if (H5T__vlen_disk_delete(file, bg) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREMOVE, FAIL, "unable to remove background heap object");

    
    UINT32ENCODE(vl, seq_len);

    
    if (H5VL_blob_put(file, buf, (seq_len * base_size), vl, NULL) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "unable to put blob");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__vlen_disk_delete(H5VL_object_t *file, void *_vl)
{
    uint8_t *vl        = (uint8_t *)_vl; 
    herr_t   ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(file);

    
    if (vl != NULL) {
        size_t seq_len; 

        
        UINT32DECODE(vl, seq_len);

        
        if (seq_len > 0) {
            H5VL_blob_specific_args_t vol_cb_args; 

            
            vol_cb_args.op_type = H5VL_BLOB_DELETE;

            if (H5VL_blob_specific(file, vl, &vol_cb_args) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREMOVE, FAIL, "unable to delete blob");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__vlen_reclaim(void *elem, const H5T_t *dt, H5T_vlen_alloc_info_t *alloc_info)
{
    unsigned    u;                   
    H5MM_free_t free_func;           
    void       *free_info;           
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(elem);
    assert(dt);
    assert(alloc_info);

    free_func = alloc_info->free_func;
    free_info = alloc_info->free_info;

    
    switch (dt->shared->type) {
        case H5T_ARRAY:
            
            if (H5T_IS_COMPOSITE(dt->shared->parent->shared->type)) {
                void *off; 

                
                for (u = 0; u < dt->shared->u.array.nelem; u++) {
                    off = ((uint8_t *)elem) + u * (dt->shared->parent->shared->size);
                    if (H5T_reclaim_cb(off, dt->shared->parent, 0, NULL, alloc_info) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free array element");
                } 
            }     
            break;

        case H5T_COMPOUND:
            
            for (u = 0; u < dt->shared->u.compnd.nmembs; u++) {
                
                if (H5T_IS_COMPOSITE(dt->shared->u.compnd.memb[u].type->shared->type)) {
                    void *off; 

                    
                    off = ((uint8_t *)elem) + dt->shared->u.compnd.memb[u].offset;
                    if (H5T_reclaim_cb(off, dt->shared->u.compnd.memb[u].type, 0, NULL, alloc_info) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free compound field");
                } 
            }     
            break;

        case H5T_VLEN:
            
            if (dt->shared->u.vlen.type == H5T_VLEN_SEQUENCE) {
                hvl_t *vl = (hvl_t *)elem; 

                
                if (vl->len != 0) {
                    
                    if (H5T_IS_COMPOSITE(dt->shared->parent->shared->type)) {
                        void *off; 

                        
                        while (vl->len > 0) {
                            off = ((uint8_t *)vl->p) + (vl->len - 1) * dt->shared->parent->shared->size;
                            if (H5T_reclaim_cb(off, dt->shared->parent, 0, NULL, alloc_info) < 0)
                                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free VL element");
                            vl->len--;
                        } 
                    }     

                    
                    if (free_func != NULL)
                        (*free_func)(vl->p, free_info);
                    else
                        free(vl->p);
                } 
            }
            else if (dt->shared->u.vlen.type == H5T_VLEN_STRING) {
                
                char *s;

                
                H5MM_memcpy(&s, elem, sizeof(char *));

                if (free_func != NULL)
                    (*free_func)(s, free_info);
                else
                    free(s);
            }
            else {
                assert(0 && "Invalid VL type");
            } 
            break;

        
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_OPAQUE:
        case H5T_ENUM:
        case H5T_COMPLEX:
            break;

        
        case H5T_REFERENCE:
        case H5T_NO_CLASS:
        case H5T_NCLASSES:
        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "invalid VL datatype class");
            break;

    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_vlen_reclaim_elmt(void *elem, const H5T_t *dt)
{
    H5T_vlen_alloc_info_t vl_alloc_info;       
    herr_t                ret_value = SUCCEED; 

    assert(dt);
    assert(elem);

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5CX_get_vlen_alloc_info(&vl_alloc_info) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to retrieve VL allocation info");

    
    if (H5T__vlen_reclaim(elem, dt, &vl_alloc_info) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "can't reclaim vlen elements");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
