/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 
#include "H5Ppkg.h"      
#include "H5VMprivate.h" 

#define H5L_ACS_NLINKS_SIZE sizeof(size_t)
#define H5L_ACS_NLINKS_DEF  H5L_NUM_LINKS 
#define H5L_ACS_NLINKS_ENC  H5P__encode_size_t
#define H5L_ACS_NLINKS_DEC  H5P__decode_size_t

#define H5L_ACS_ELINK_PREFIX_SIZE  sizeof(char *)
#define H5L_ACS_ELINK_PREFIX_DEF   NULL 
#define H5L_ACS_ELINK_PREFIX_SET   H5P__lacc_elink_pref_set
#define H5L_ACS_ELINK_PREFIX_GET   H5P__lacc_elink_pref_get
#define H5L_ACS_ELINK_PREFIX_ENC   H5P__lacc_elink_pref_enc
#define H5L_ACS_ELINK_PREFIX_DEC   H5P__lacc_elink_pref_dec
#define H5L_ACS_ELINK_PREFIX_DEL   H5P__lacc_elink_pref_del
#define H5L_ACS_ELINK_PREFIX_COPY  H5P__lacc_elink_pref_copy
#define H5L_ACS_ELINK_PREFIX_CMP   H5P__lacc_elink_pref_cmp
#define H5L_ACS_ELINK_PREFIX_CLOSE H5P__lacc_elink_pref_close

#define H5L_ACS_ELINK_FAPL_SIZE  sizeof(hid_t)
#define H5L_ACS_ELINK_FAPL_DEF   H5P_DEFAULT
#define H5L_ACS_ELINK_FAPL_SET   H5P__lacc_elink_fapl_set
#define H5L_ACS_ELINK_FAPL_GET   H5P__lacc_elink_fapl_get
#define H5L_ACS_ELINK_FAPL_ENC   H5P__lacc_elink_fapl_enc
#define H5L_ACS_ELINK_FAPL_DEC   H5P__lacc_elink_fapl_dec
#define H5L_ACS_ELINK_FAPL_DEL   H5P__lacc_elink_fapl_del
#define H5L_ACS_ELINK_FAPL_COPY  H5P__lacc_elink_fapl_copy
#define H5L_ACS_ELINK_FAPL_CMP   H5P__lacc_elink_fapl_cmp
#define H5L_ACS_ELINK_FAPL_CLOSE H5P__lacc_elink_fapl_close

#define H5L_ACS_ELINK_FLAGS_SIZE sizeof(unsigned)
#define H5L_ACS_ELINK_FLAGS_DEF  H5F_ACC_DEFAULT
#define H5L_ACS_ELINK_FLAGS_ENC  H5P__encode_unsigned
#define H5L_ACS_ELINK_FLAGS_DEC  H5P__decode_unsigned

#define H5L_ACS_ELINK_CB_SIZE sizeof(H5L_elink_cb_t)
#define H5L_ACS_ELINK_CB_DEF                                                                                 \
    {                                                                                                        \
        NULL, NULL                                                                                           \
    }

#ifdef H5_HAVE_PARALLEL

#define H5L_ACS_COLL_MD_READ_SIZE sizeof(H5P_coll_md_read_flag_t)
#define H5L_ACS_COLL_MD_READ_DEF  H5P_USER_FALSE
#define H5L_ACS_COLL_MD_READ_ENC  H5P__encode_coll_md_read_flag_t
#define H5L_ACS_COLL_MD_READ_DEC  H5P__decode_coll_md_read_flag_t
#endif 

static herr_t H5P__lacc_reg_prop(H5P_genclass_t *pclass);

static herr_t H5P__lacc_elink_pref_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__lacc_elink_pref_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__lacc_elink_pref_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__lacc_elink_pref_dec(const void **_pp, void *value);
static herr_t H5P__lacc_elink_pref_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__lacc_elink_pref_copy(const char *name, size_t size, void *value);
static int    H5P__lacc_elink_pref_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__lacc_elink_pref_close(const char *name, size_t size, void *value);
static herr_t H5P__lacc_elink_fapl_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__lacc_elink_fapl_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__lacc_elink_fapl_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__lacc_elink_fapl_dec(const void **_pp, void *value);
static herr_t H5P__lacc_elink_fapl_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__lacc_elink_fapl_copy(const char *name, size_t size, void *value);
static int    H5P__lacc_elink_fapl_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__lacc_elink_fapl_close(const char *name, size_t size, void *value);

const H5P_libclass_t H5P_CLS_LACC[1] = {{
    "link access",        
    H5P_TYPE_LINK_ACCESS, 

    &H5P_CLS_ROOT_g,           
    &H5P_CLS_LINK_ACCESS_g,    
    &H5P_CLS_LINK_ACCESS_ID_g, 
    &H5P_LST_LINK_ACCESS_ID_g, 
    H5P__lacc_reg_prop,        

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

static const size_t H5L_def_nlinks_g = H5L_ACS_NLINKS_DEF; 
static const char  *H5L_def_elink_prefix_g =
    H5L_ACS_ELINK_PREFIX_DEF;                                     
static const hid_t    H5L_def_fapl_id_g = H5L_ACS_ELINK_FAPL_DEF; 
static const unsigned H5L_def_elink_flags_g =
    H5L_ACS_ELINK_FLAGS_DEF; 
static const H5L_elink_cb_t H5L_def_elink_cb_g =
    H5L_ACS_ELINK_CB_DEF; 
#ifdef H5_HAVE_PARALLEL
static const H5P_coll_md_read_flag_t H5L_def_coll_md_read_g =
    H5L_ACS_COLL_MD_READ_DEF; 
#endif                        

static herr_t
H5P__lacc_reg_prop(H5P_genclass_t *pclass)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5L_ACS_NLINKS_NAME, H5L_ACS_NLINKS_SIZE, &H5L_def_nlinks_g, NULL, NULL,
                           NULL, H5L_ACS_NLINKS_ENC, H5L_ACS_NLINKS_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5L_ACS_ELINK_PREFIX_NAME, H5L_ACS_ELINK_PREFIX_SIZE,
                           &H5L_def_elink_prefix_g, NULL, H5L_ACS_ELINK_PREFIX_SET, H5L_ACS_ELINK_PREFIX_GET,
                           H5L_ACS_ELINK_PREFIX_ENC, H5L_ACS_ELINK_PREFIX_DEC, H5L_ACS_ELINK_PREFIX_DEL,
                           H5L_ACS_ELINK_PREFIX_COPY, H5L_ACS_ELINK_PREFIX_CMP,
                           H5L_ACS_ELINK_PREFIX_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5L_ACS_ELINK_FAPL_NAME, H5L_ACS_ELINK_FAPL_SIZE, &H5L_def_fapl_id_g, NULL,
                           H5L_ACS_ELINK_FAPL_SET, H5L_ACS_ELINK_FAPL_GET, H5L_ACS_ELINK_FAPL_ENC,
                           H5L_ACS_ELINK_FAPL_DEC, H5L_ACS_ELINK_FAPL_DEL, H5L_ACS_ELINK_FAPL_COPY,
                           H5L_ACS_ELINK_FAPL_CMP, H5L_ACS_ELINK_FAPL_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5L_ACS_ELINK_FLAGS_NAME, H5L_ACS_ELINK_FLAGS_SIZE, &H5L_def_elink_flags_g,
                           NULL, NULL, NULL, H5L_ACS_ELINK_FLAGS_ENC, H5L_ACS_ELINK_FLAGS_DEC, NULL, NULL,
                           NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5L_ACS_ELINK_CB_NAME, H5L_ACS_ELINK_CB_SIZE, &H5L_def_elink_cb_g, NULL,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

#ifdef H5_HAVE_PARALLEL
    
    if (H5P__register_real(pclass, H5_COLL_MD_READ_FLAG_NAME, H5L_ACS_COLL_MD_READ_SIZE,
                           &H5L_def_coll_md_read_g, NULL, NULL, NULL, H5L_ACS_COLL_MD_READ_ENC,
                           H5L_ACS_COLL_MD_READ_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_fapl_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    hid_t  l_fapl_id;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    l_fapl_id = *(const hid_t *)value;

    
    if (l_fapl_id != H5P_DEFAULT) {
        H5P_genplist_t *l_fapl_plist;

        if (NULL == (l_fapl_plist = (H5P_genplist_t *)H5P_object_verify(l_fapl_id, H5P_FILE_ACCESS, true)))
            HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "can't get property list");
        if (((*(hid_t *)value) = H5P_copy_plist(l_fapl_plist, false)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy file access property list");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_fapl_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    hid_t  l_fapl_id;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    l_fapl_id = *(const hid_t *)value;

    
    if (l_fapl_id != H5P_DEFAULT) {
        H5P_genplist_t *l_fapl_plist;

        if (NULL == (l_fapl_plist = (H5P_genplist_t *)H5P_object_verify(l_fapl_id, H5P_FILE_ACCESS, true)))
            HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "can't get property list");
        if (((*(hid_t *)value) = H5P_copy_plist(l_fapl_plist, false)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy file access property list");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_fapl_enc(const void *value, void **_pp, size_t *size)
{
    const hid_t    *elink_fapl = (const hid_t *)value; 
    uint8_t       **pp         = (uint8_t **)_pp;
    H5P_genplist_t *fapl_plist;                 
    bool            non_default_fapl = false;   
    size_t          fapl_size        = 0;       
    herr_t          ret_value        = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (*elink_fapl != H5P_DEFAULT) {
        if (NULL == (fapl_plist = (H5P_genplist_t *)H5P_object_verify(*elink_fapl, H5P_FILE_ACCESS, true)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get property list");
        non_default_fapl = true;
    } 

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)non_default_fapl;
    } 

    
    
    if (non_default_fapl) {
        if (H5P__encode(fapl_plist, true, NULL, &fapl_size) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "can't encode property list");

        if (*pp) {
            uint64_t enc_value;
            unsigned enc_size;

            
            enc_value = (uint64_t)fapl_size;
            enc_size  = H5VM_limit_enc_size(enc_value);
            assert(enc_size < 256);
            *(*pp)++ = (uint8_t)enc_size;
            UINT64ENCODE_VAR(*pp, enc_value, enc_size);

            
            if (H5P__encode(fapl_plist, true, *pp, &fapl_size) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, FAIL, "can't encode property list");

            *pp += fapl_size;
        }
        fapl_size += (1 + H5VM_limit_enc_size((uint64_t)fapl_size));
    } 

    *size += (1 + fapl_size); 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_fapl_dec(const void **_pp, void *_value)
{
    hid_t          *elink_fapl = (hid_t *)_value; 
    const uint8_t **pp         = (const uint8_t **)_pp;
    bool            non_default_fapl;    
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(elink_fapl);
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    non_default_fapl = (bool)*(*pp)++;

    if (non_default_fapl) {
        size_t   fapl_size = 0; 
        unsigned enc_size;
        uint64_t enc_value;

        
        enc_size = *(*pp)++;
        assert(enc_size < 256);
        UINT64DECODE_VAR(*pp, enc_value, enc_size);
        fapl_size = (size_t)enc_value;

        
        if ((*elink_fapl = H5P__decode(*pp)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTDECODE, FAIL, "can't decode property");

        *pp += fapl_size;
    } 
    else
        *elink_fapl = H5P_DEFAULT;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_fapl_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    hid_t  l_fapl_id;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    l_fapl_id = (*(const hid_t *)value);

    
    if (l_fapl_id != H5P_DEFAULT && H5I_dec_ref(l_fapl_id) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRELEASE, FAIL, "unable to close ID for file access property list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_fapl_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    hid_t  l_fapl_id;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    l_fapl_id = (*(const hid_t *)value);

    
    if (l_fapl_id != H5P_DEFAULT) {
        H5P_genplist_t *l_fapl_plist;

        if (NULL == (l_fapl_plist = (H5P_genplist_t *)H5P_object_verify(l_fapl_id, H5P_FILE_ACCESS, true)))
            HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "can't get property list");
        if (((*(hid_t *)value) = H5P_copy_plist(l_fapl_plist, false)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "unable to copy file access property list");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__lacc_elink_fapl_cmp(const void *value1, const void *value2, size_t H5_ATTR_UNUSED size)
{
    const hid_t    *fapl1 = (const hid_t *)value1;
    const hid_t    *fapl2 = (const hid_t *)value2;
    H5P_genplist_t *obj1, *obj2; 
    int             ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    if (*fapl1 == 0 && *fapl2 > 0)
        HGOTO_DONE(1);
    if (*fapl1 > 0 && *fapl2 == 0)
        HGOTO_DONE(-1);

    
    obj1 = (H5P_genplist_t *)H5I_object(*fapl1);
    obj2 = (H5P_genplist_t *)H5I_object(*fapl2);

    
    if (obj1 == NULL && obj2 != NULL)
        HGOTO_DONE(1);
    if (obj1 != NULL && obj2 == NULL)
        HGOTO_DONE(-1);
    if (obj1 && obj2) {
        herr_t H5_ATTR_NDEBUG_UNUSED status;

        status = H5P__cmp_plist(obj1, obj2, &ret_value);
        assert(status >= 0);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_fapl_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    hid_t  l_fapl_id;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    l_fapl_id = (*(const hid_t *)value);

    
    if ((l_fapl_id > H5P_DEFAULT) && (H5I_dec_ref(l_fapl_id) < 0))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTRELEASE, FAIL, "unable to close ID for file access property list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_pref_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);

    
    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__lacc_elink_pref_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);

    
    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__lacc_elink_pref_enc(const void *value, void **_pp, size_t *size)
{
    const char *elink_pref = *(const char *const *)value;
    uint8_t   **pp         = (uint8_t **)_pp;
    size_t      len        = 0;
    uint64_t    enc_value;
    unsigned    enc_size;

    FUNC_ENTER_PACKAGE_NOERR

    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    if (NULL != elink_pref)
        len = strlen(elink_pref);

    enc_value = (uint64_t)len;
    enc_size  = H5VM_limit_enc_size(enc_value);
    assert(enc_size < 256);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)enc_size;
        UINT64ENCODE_VAR(*pp, enc_value, enc_size);

        
        if (NULL != elink_pref) {
            H5MM_memcpy(*(char **)pp, elink_pref, len);
            *pp += len;
        } 
    }     

    *size += (1 + enc_size);
    if (NULL != elink_pref)
        *size += len;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__lacc_elink_pref_dec(const void **_pp, void *_value)
{
    char          **elink_pref = (char **)_value;
    const uint8_t **pp         = (const uint8_t **)_pp;
    size_t          len;
    uint64_t        enc_value; 
    unsigned        enc_size;  
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(pp);
    assert(*pp);
    assert(elink_pref);
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);

    
    UINT64DECODE_VAR(*pp, enc_value, enc_size);
    len = (size_t)enc_value;

    if (0 != len) {
        
        if (NULL == (*elink_pref = (char *)H5MM_malloc(len + 1)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, FAIL, "memory allocation failed for prefix");
        strncpy(*elink_pref, *(const char **)pp, len);
        (*elink_pref)[len] = '\0';

        *pp += len;
    } 
    else
        *elink_pref = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_pref_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    H5MM_xfree(*(void **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__lacc_elink_pref_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5P__lacc_elink_pref_cmp(const void *value1, const void *value2, size_t H5_ATTR_UNUSED size)
{
    const char *pref1     = *(const char *const *)value1;
    const char *pref2     = *(const char *const *)value2;
    int         ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    if (NULL == pref1 && NULL != pref2)
        HGOTO_DONE(1);
    if (NULL != pref1 && NULL == pref2)
        HGOTO_DONE(-1);
    if (NULL != pref1 && NULL != pref2)
        ret_value = strcmp(pref1, pref2);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__lacc_elink_pref_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    H5MM_xfree(*(void **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_nlinks(hid_t plist_id, size_t nlinks)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (nlinks <= 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "number of links must be positive");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_LINK_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5L_ACS_NLINKS_NAME, &nlinks) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set nlink info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_nlinks(hid_t plist_id, size_t *nlinks )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (!nlinks)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid pointer passed in");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_LINK_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5L_ACS_NLINKS_NAME, nlinks) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get number of links");

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5Pset_elink_prefix(hid_t plist_id, const char *prefix)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_LINK_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5L_ACS_ELINK_PREFIX_NAME, &prefix) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set prefix info");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Pget_elink_prefix(hid_t plist_id, char *prefix , size_t size)
{
    H5P_genplist_t *plist;     
    char           *my_prefix; 
    size_t          len;       
    ssize_t         ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_LINK_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5L_ACS_ELINK_PREFIX_NAME, &my_prefix) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get external link prefix");

    
    if (my_prefix) {
        
        len = strlen(my_prefix);
        if (prefix) {
            strncpy(prefix, my_prefix, size);
            if (len >= size)
                prefix[size - 1] = '\0';
        } 
    }     
    else
        len = 0;

    
    ret_value = (ssize_t)len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_elink_fapl(hid_t lapl_id, hid_t fapl_id)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(lapl_id, H5P_LINK_ACCESS, false)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a link access property list");

    
    if (H5P_set(plist, H5L_ACS_ELINK_FAPL_NAME, &fapl_id) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set fapl for link");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Pget_elink_fapl(hid_t lapl_id)
{
    H5P_genplist_t *plist;     
    hid_t           ret_value; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (plist = H5P_object_verify(lapl_id, H5P_LINK_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, H5I_INVALID_HID, "can't find object for ID");

    if (H5P_get(plist, H5L_ACS_ELINK_FAPL_NAME, &ret_value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, H5I_INVALID_HID, "can't get fapl for links");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_elink_acc_flags(hid_t lapl_id, unsigned flags)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if ((flags != H5F_ACC_RDWR) && (flags != (H5F_ACC_RDWR | H5F_ACC_SWMR_WRITE)) &&
        (flags != H5F_ACC_RDONLY) && (flags != (H5F_ACC_RDONLY | H5F_ACC_SWMR_READ)) &&
        (flags != H5F_ACC_DEFAULT))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid file open flags");

    
    if (NULL == (plist = H5P_object_verify(lapl_id, H5P_LINK_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5L_ACS_ELINK_FLAGS_NAME, &flags) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set access flags");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_elink_acc_flags(hid_t lapl_id, unsigned *flags )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(lapl_id, H5P_LINK_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (flags)
        if (H5P_get(plist, H5L_ACS_ELINK_FLAGS_NAME, flags) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, 0, "can't get access flags");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_elink_cb(hid_t lapl_id, H5L_elink_traverse_t func, void *op_data)
{
    H5P_genplist_t *plist;               
    H5L_elink_cb_t  cb_info;             
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!func && op_data)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "callback is NULL while user data is not");

    
    if (NULL == (plist = H5P_object_verify(lapl_id, H5P_LINK_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    cb_info.func      = func;
    cb_info.user_data = op_data;

    
    if (H5P_set(plist, H5L_ACS_ELINK_CB_NAME, &cb_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set callback info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_elink_cb(hid_t lapl_id, H5L_elink_traverse_t *func , void **op_data )
{
    H5P_genplist_t *plist;               
    H5L_elink_cb_t  cb_info;             
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(lapl_id, H5P_LINK_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5L_ACS_ELINK_CB_NAME, &cb_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get callback info");

    if (func)
        *func = cb_info.func;
    if (op_data)
        *op_data = cb_info.user_data;

done:
    FUNC_LEAVE_API(ret_value)
} 
