/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5HFprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5SMprivate.h" 
#include "H5WBprivate.h" 

#define H5O_SHARED_VERSION_1 1

#define H5O_SHARED_VERSION_2 2

#define H5O_SHARED_VERSION_3      3
#define H5O_SHARED_VERSION_LATEST H5O_SHARED_VERSION_3

#define H5O_MESG_BUF_SIZE 128

static void  *H5O__shared_read(H5F_t *f, H5O_t *open_oh, unsigned *ioflags, const H5O_shared_t *shared,
                               const H5O_msg_class_t *type);
static herr_t H5O__shared_link_adj(H5F_t *f, H5O_t *open_oh, const H5O_msg_class_t *type,
                                   H5O_shared_t *shared, int adjust);

static void *
H5O__shared_read(H5F_t *f, H5O_t *open_oh, unsigned *ioflags, const H5O_shared_t *shared,
                 const H5O_msg_class_t *type)
{
    H5HF_t *fheap = NULL;
    H5WB_t *wb    = NULL;                
    uint8_t mesg_buf[H5O_MESG_BUF_SIZE]; 
    void   *ret_value = NULL;            

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(shared);
    assert(type);
    assert(type->share_flags & H5O_SHARE_IS_SHARABLE);

    
    assert(H5O_IS_STORED_SHARED(shared->type));

    
    if (shared->type == H5O_SHARE_TYPE_SOHM) {
        haddr_t  fheap_addr; 
        uint8_t *mesg_ptr;   
        size_t   mesg_size;  

        
        if (H5SM_get_fheap_addr(f, type->id, &fheap_addr) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "can't get fheap address for shared messages");

        
        if (NULL == (fheap = H5HF_open(f, fheap_addr)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, NULL, "unable to open fractal heap");

        
        if (H5HF_get_obj_len(fheap, &(shared->u.heap_id), &mesg_size) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "can't get message size from fractal heap.");

        
        if (NULL == (wb = H5WB_wrap(mesg_buf, sizeof(mesg_buf))))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "can't wrap buffer");

        
        if (NULL == (mesg_ptr = (uint8_t *)H5WB_actual(wb, mesg_size)))
            HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, NULL, "can't get actual buffer");

        
        if (H5HF_read(fheap, &(shared->u.heap_id), mesg_ptr) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "can't read message from fractal heap.");

        
        if (NULL == (ret_value = (type->decode)(f, open_oh, 0, ioflags, mesg_size, mesg_ptr)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, NULL, "can't decode shared message.");
    } 
    else {
        H5O_loc_t oloc; 

        assert(shared->type == H5O_SHARE_TYPE_COMMITTED);

        
        oloc.file         = f;
        oloc.addr         = shared->u.loc.oh_addr;
        oloc.holding_file = false;

        if (open_oh && oloc.addr == H5O_OH_GET_ADDR(open_oh)) {
            
            if (NULL == (ret_value = H5O_msg_read_oh(f, open_oh, type->id, NULL)))
                HGOTO_ERROR(H5E_OHDR, H5E_READERROR, NULL, "unable to read message");
        }
        else
            
            if (NULL == (ret_value = H5O_msg_read(&oloc, type->id, NULL)))
                HGOTO_ERROR(H5E_OHDR, H5E_READERROR, NULL, "unable to read message");
    } 

    
    if (H5O_msg_set_share(type->id, shared, ret_value) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "unable to set sharing information");

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, NULL, "can't close fractal heap");
    if (wb && H5WB_unwrap(wb) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CLOSEERROR, NULL, "can't close wrapped buffer");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__shared_link_adj(H5F_t *f, H5O_t *open_oh, const H5O_msg_class_t *type, H5O_shared_t *shared, int adjust)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(shared);

    
    if (shared->type == H5O_SHARE_TYPE_COMMITTED) {
        H5O_loc_t oloc; 

        
        
        
        

        
        oloc.file         = f;
        oloc.addr         = shared->u.loc.oh_addr;
        oloc.holding_file = false;

        if (open_oh && oloc.addr == H5O_OH_GET_ADDR(open_oh)) {
            
            bool deleted = false; 

            if (H5O__link_oh(f, adjust, open_oh, &deleted) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "unable to adjust shared object link count");

            assert(!deleted);
        }
        else
            
            if (H5O_link(&oloc, adjust) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "unable to adjust shared object link count");
    } 
    else {
        assert(shared->type == H5O_SHARE_TYPE_SOHM || shared->type == H5O_SHARE_TYPE_HERE);

        
        if (adjust < 0) {
            if (H5SM_delete(f, open_oh, shared) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTDEC, FAIL, "unable to delete message from SOHM table");
        } 
        
        else if (adjust > 0) {
            if (H5SM_try_share(f, open_oh, 0, type->id, shared, NULL) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINC, FAIL, "error trying to share message");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5O__shared_decode(H5F_t *f, H5O_t *open_oh, unsigned *ioflags, size_t buf_size, const uint8_t *buf,
                   const H5O_msg_class_t *type)
{
    const uint8_t *buf_end = buf + buf_size - 1; 
    H5O_shared_t   sh_mesg;                      
    unsigned       version;                      
    void          *ret_value = NULL;             

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(buf);
    assert(type);

    
    if (H5_IS_BUFFER_OVERFLOW(buf, 1, buf_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    version = *buf++;
    if (version < H5O_SHARED_VERSION_1 || version > H5O_SHARED_VERSION_LATEST)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for shared object message");

    
    if (H5_IS_BUFFER_OVERFLOW(buf, 1, buf_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (version >= H5O_SHARED_VERSION_2)
        sh_mesg.type = *buf++;
    else {
        sh_mesg.type = H5O_SHARE_TYPE_COMMITTED;
        buf++;
    } 

    
    if (version == H5O_SHARED_VERSION_1) {
        if (H5_IS_BUFFER_OVERFLOW(buf, 6, buf_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        buf += 6;
    }

    
    if (version == H5O_SHARED_VERSION_1) {
        
        sh_mesg.u.loc.index = 0;

        
        if (H5_IS_BUFFER_OVERFLOW(buf, H5F_SIZEOF_SIZE(f), buf_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        buf += H5F_SIZEOF_SIZE(f); 
        if (H5_IS_BUFFER_OVERFLOW(buf, H5F_sizeof_addr(f), buf_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_addr_decode(f, &buf, &(sh_mesg.u.loc.oh_addr));
    } 
    else if (version >= H5O_SHARED_VERSION_2) {
        
        if (sh_mesg.type == H5O_SHARE_TYPE_SOHM) {
            assert(version >= H5O_SHARED_VERSION_3);
            if (H5_IS_BUFFER_OVERFLOW(buf, sizeof(sh_mesg.u.heap_id), buf_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            H5MM_memcpy(&sh_mesg.u.heap_id, buf, sizeof(sh_mesg.u.heap_id));
        } 
        else {
            
            if (version < H5O_SHARED_VERSION_3)
                sh_mesg.type = H5O_SHARE_TYPE_COMMITTED;

            sh_mesg.u.loc.index = 0;
            if (H5_IS_BUFFER_OVERFLOW(buf, H5F_sizeof_addr(f), buf_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            H5F_addr_decode(f, &buf, &sh_mesg.u.loc.oh_addr);
        } 
    }     

    
    sh_mesg.file        = f;
    sh_mesg.msg_type_id = type->id;

    
    if (NULL == (ret_value = H5O__shared_read(f, open_oh, ioflags, &sh_mesg, type)))
        HGOTO_ERROR(H5E_OHDR, H5E_READERROR, NULL, "unable to retrieve native message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__shared_encode(const H5F_t *f, uint8_t *buf , const H5O_shared_t *sh_mesg)
{
    unsigned version;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(buf);
    assert(sh_mesg);

    
    if (sh_mesg->type == H5O_SHARE_TYPE_SOHM)
        version = H5O_SHARED_VERSION_LATEST;
    else {
        assert(sh_mesg->type == H5O_SHARE_TYPE_COMMITTED);
        version = H5O_SHARED_VERSION_2; 
    }                                   

    *buf++ = (uint8_t)version;
    *buf++ = (uint8_t)sh_mesg->type;

    
    if (sh_mesg->type == H5O_SHARE_TYPE_SOHM)
        H5MM_memcpy(buf, &(sh_mesg->u.heap_id), sizeof(sh_mesg->u.heap_id));
    else
        H5F_addr_encode(f, &buf, sh_mesg->u.loc.oh_addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O_set_shared(H5O_shared_t *dst, const H5O_shared_t *src)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(dst);
    assert(src);

    
    *dst = *src;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

size_t
H5O__shared_size(const H5F_t *f, const H5O_shared_t *sh_mesg)
{
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    if (sh_mesg->type == H5O_SHARE_TYPE_COMMITTED) {
        ret_value = (size_t)1 +                 
                    (size_t)1 +                 
                    (size_t)H5F_SIZEOF_ADDR(f); 
    }                                           
    else {
        assert(sh_mesg->type == H5O_SHARE_TYPE_SOHM);
        ret_value = 1 +               
                    1 +               
                    H5O_FHEAP_ID_LEN; 
    }                                 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__shared_delete(H5F_t *f, H5O_t *open_oh, const H5O_msg_class_t *type, H5O_shared_t *sh_mesg)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(sh_mesg);

    

    
    if (H5O__shared_link_adj(f, open_oh, type, sh_mesg, -1) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "unable to adjust shared object link count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__shared_link(H5F_t *f, H5O_t *open_oh, const H5O_msg_class_t *type, H5O_shared_t *sh_mesg)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(sh_mesg);

    
    if (H5O__shared_link_adj(f, open_oh, type, sh_mesg, 1) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "unable to adjust shared object link count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__shared_copy_file(H5F_t H5_ATTR_NDEBUG_UNUSED *file_src, H5F_t *file_dst,
                      const H5O_msg_class_t *mesg_type, const void *_native_src, void *_native_dst,
                      bool H5_ATTR_UNUSED *recompute_size, unsigned *mesg_flags,
                      H5O_copy_t H5_ATTR_NDEBUG_UNUSED *cpy_info, void H5_ATTR_UNUSED *udata)
{
    const H5O_shared_t *shared_src =
        (const H5O_shared_t *)_native_src; 
    H5O_shared_t *shared_dst =
        (H5O_shared_t *)_native_dst; 
    herr_t ret_value = SUCCEED;      

    FUNC_ENTER_PACKAGE

    
    assert(file_src);
    assert(file_dst);
    assert(mesg_type);
    assert(shared_src);
    assert(shared_dst);
    assert(recompute_size);
    assert(cpy_info);

    
    if (shared_src->type != H5O_SHARE_TYPE_COMMITTED) {
        
        
        H5_BEGIN_TAG(H5AC__COPIED_TAG)

        if (H5SM_try_share(file_dst, NULL, H5SM_DEFER, mesg_type->id, _native_dst, mesg_flags) < 0)
            HGOTO_ERROR_TAG(H5E_OHDR, H5E_WRITEERROR, FAIL,
                            "unable to determine if message should be shared");

        
        H5_END_TAG
    } 
    else {
        
        H5O_UPDATE_SHARED(shared_dst, H5O_SHARE_TYPE_COMMITTED, file_dst, mesg_type->id, 0, HADDR_UNDEF)
        *mesg_flags |= H5O_MSG_FLAG_SHARED;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__shared_post_copy_file(H5F_t *f, const H5O_msg_class_t *mesg_type, const H5O_shared_t *shared_src,
                           H5O_shared_t *shared_dst, unsigned *mesg_flags, H5O_copy_t *cpy_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(shared_src);
    assert(shared_dst);

    
    if (shared_src->type == H5O_SHARE_TYPE_COMMITTED) {
        H5O_loc_t dst_oloc;
        H5O_loc_t src_oloc;

        
        H5O_loc_reset(&dst_oloc);
        dst_oloc.file = f;
        src_oloc.file = shared_src->file;
        src_oloc.addr = shared_src->u.loc.oh_addr;
        if (H5O_copy_header_map(&src_oloc, &dst_oloc, cpy_info, false, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object");

        
        H5O_UPDATE_SHARED(shared_dst, H5O_SHARE_TYPE_COMMITTED, f, mesg_type->id, 0, dst_oloc.addr)
    } 
    else
        
        if (H5SM_try_share(f, NULL, H5SM_WAS_DEFERRED, mesg_type->id, shared_dst, mesg_flags) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "can't share message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__shared_debug(const H5O_shared_t *mesg, FILE *stream, int indent, int fwidth)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    switch (mesg->type) {
        case H5O_SHARE_TYPE_UNSHARED:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Shared Message type:", "Unshared");
            break;

        case H5O_SHARE_TYPE_COMMITTED:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Shared Message type:", "Obj Hdr");
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
                    "Object address:", mesg->u.loc.oh_addr);
            break;

        case H5O_SHARE_TYPE_SOHM:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Shared Message type:", "SOHM");
            Rfprintf(stream, "%*s%-*s %016llx\n", indent, "", fwidth,
                    "Heap ID:", (unsigned long long)mesg->u.heap_id.val);
            break;

        case H5O_SHARE_TYPE_HERE:
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Shared Message type:", "Here");
            break;

        default:
            Rfprintf(stream, "%*s%-*s %s (%u)\n", indent, "", fwidth, "Shared Message type:", "Unknown",
                    (unsigned)mesg->type);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
