/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 

#include "H5VLnative_private.h" 

typedef struct H5G_names_t {
    H5G_names_op_t op;              
    H5F_t         *src_file;        
    H5RS_str_t    *src_full_path_r; 
    H5F_t         *dst_file;        
    H5RS_str_t    *dst_full_path_r; 
} H5G_names_t;

typedef struct H5G_gnba_iter_t {
    
    const H5O_loc_t *loc; 

    
    char *path; 
} H5G_gnba_iter_t;

static htri_t      H5G__common_path(const H5RS_str_t *fullpath_r, const H5RS_str_t *prefix_r);
static H5RS_str_t *H5G__build_fullpath(const char *prefix, const char *name);
static herr_t      H5G__name_move_path(H5RS_str_t **path_r_ptr, const char *full_suffix, const char *src_path,
                                       const char *dst_path);
static int         H5G__name_replace_cb(void *obj_ptr, hid_t obj_id, void *key);

const char *
H5G__component(const char *name, size_t *size_p)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(name);

    while ('/' == *name)
        name++;
    if (size_p)
        *size_p = strcspn(name, "/");

    FUNC_LEAVE_NOAPI(name)
} 

char *
H5G_normalize(const char *name)
{
    char  *norm;             
    size_t s, d;             
    bool   last_slash;       
    char  *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(name);

    
    if (NULL == (norm = H5MM_strdup(name)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTALLOC, NULL, "memory allocation failed for normalized string");

    
    s = d      = 0;
    last_slash = false;
    while (name[s] != '\0') {
        if (name[s] == '/')
            if (last_slash)
                ;
            else {
                norm[d++]  = name[s];
                last_slash = true;
            } 
        else {
            norm[d++]  = name[s];
            last_slash = false;
        } 
        s++;
    } 

    
    norm[d] = '\0';

    
    if (d > 1 && last_slash)
        norm[d - 1] = '\0';

    
    ret_value = norm;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5G__common_path(const H5RS_str_t *fullpath_r, const H5RS_str_t *prefix_r)
{
    const char *fullpath;          
    const char *prefix;            
    size_t      nchars1, nchars2;  
    htri_t      ret_value = false; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (fullpath = H5RS_get_str(fullpath_r)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve full path");
    nchars1 = SIZE_MAX;
    if (NULL == (fullpath = H5G__component(fullpath, &nchars1)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve pointer to path component");
    if (SIZE_MAX == nchars1)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve component length");
    if (NULL == (prefix = H5RS_get_str(prefix_r)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve path prefix");
    nchars2 = SIZE_MAX;
    if (NULL == (prefix = H5G__component(prefix, &nchars2)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve pointer to path component");
    if (SIZE_MAX == nchars2)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve component length");

    
    while (*fullpath && *prefix) {
        
        if (nchars1 == nchars2) {
            
            if (strncmp(fullpath, prefix, nchars1) == 0) {
                
                fullpath += nchars1;
                prefix += nchars2;

                
                fullpath = H5G__component(fullpath, &nchars1);
                assert(fullpath);
                prefix = H5G__component(prefix, &nchars2);
                assert(prefix);
                if (NULL == (fullpath = H5G__component(fullpath, &nchars1)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve pointer to path component");
                if (SIZE_MAX == nchars1)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve component length");
                nchars2 = SIZE_MAX;
                if (NULL == (prefix = H5G__component(prefix, &nchars2)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve pointer to path component");
                if (SIZE_MAX == nchars2)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve component length");
            } 
            else
                HGOTO_DONE(false);
        } 
        else
            HGOTO_DONE(false);
    } 

    
    if (*prefix == '\0')
        ret_value = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5RS_str_t *
H5G__build_fullpath(const char *prefix, const char *name)
{
    H5RS_str_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(prefix);
    assert(name);

    
    if (NULL == (ret_value = H5RS_create(prefix)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, NULL, "can't create ref-counted string");
    if (prefix[strlen(prefix) - 1] != '/')
        H5RS_aputc(ret_value, '/'); 
    H5RS_acat(ret_value, name);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5RS_str_t *
H5G_build_fullpath_refstr_str(H5RS_str_t *prefix_r, const char *name)
{
    const char *prefix;           
    H5RS_str_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(prefix_r);
    assert(name);

    
    prefix = H5RS_get_str(prefix_r);
    assert(prefix);

    
    ret_value = H5G__build_fullpath(prefix, name);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__name_init(H5G_name_t *name, const char *path)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(name);

    
    name->full_path_r = H5RS_create(path);
    assert(name->full_path_r);
    name->user_path_r = H5RS_create(path);
    assert(name->user_path_r);
    name->obj_hidden = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G_name_set(const H5G_name_t *loc, H5G_name_t *obj, const char *name)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    assert(loc);
    assert(obj);
    assert(name);

    
    H5G_name_free(obj);

    
    if (loc->full_path_r) {
        
        if ((obj->full_path_r = H5G_build_fullpath_refstr_str(loc->full_path_r, name)) == NULL)
            HGOTO_ERROR(H5E_SYM, H5E_PATH, FAIL, "can't build user path name");
    } 

    
    if (loc->user_path_r) {
        
        if ((obj->user_path_r = H5G_build_fullpath_refstr_str(loc->user_path_r, name)) == NULL)
            HGOTO_ERROR(H5E_SYM, H5E_PATH, FAIL, "can't build user path name");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_name_copy(H5G_name_t *dst, const H5G_name_t *src, H5_copy_depth_t depth)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(src);
    assert(dst);
#if defined(H5_USING_MEMCHECKER) || !defined(NDEBUG)
    assert(dst->full_path_r == NULL);
    assert(dst->user_path_r == NULL);
#endif 
    assert(depth == H5_COPY_SHALLOW || depth == H5_COPY_DEEP);

    
    H5MM_memcpy(dst, src, sizeof(H5G_name_t));

    
    if (depth == H5_COPY_DEEP) {
        dst->full_path_r = H5RS_dup(src->full_path_r);
        dst->user_path_r = H5RS_dup(src->user_path_r);
    }
    else {
        H5_WARN_CAST_AWAY_CONST_OFF
        H5G_name_reset((H5G_name_t *)src);
        H5_WARN_CAST_AWAY_CONST_ON
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G_get_name(const H5G_loc_t *loc, char *name , size_t size, size_t *name_len, bool *cached)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(loc);

    
    if (loc->path->user_path_r != NULL && loc->path->obj_hidden == 0) {
        size_t len; 

        len = H5RS_len(loc->path->user_path_r);

        if (name) {
            strncpy(name, H5RS_get_str(loc->path->user_path_r), MIN((len + 1), size));
            if (len >= size)
                name[size - 1] = '\0';
        } 

        
        if (name_len)
            *name_len = len;

        
        
        if (cached)
            *cached = true;
    } 
    else if (!loc->path->obj_hidden) {
        
        if (H5G_get_name_by_addr(loc->oloc->file, loc->oloc, name, size, name_len) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't determine name");

        
        
        if (cached)
            *cached = false;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_name_reset(H5G_name_t *name)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(name);

    
    memset(name, 0, sizeof(H5G_name_t));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G_name_free(H5G_name_t *name)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(name);

    if (name->full_path_r) {
        H5RS_decr(name->full_path_r);
        name->full_path_r = NULL;
    } 
    if (name->user_path_r) {
        H5RS_decr(name->user_path_r);
        name->user_path_r = NULL;
    } 
    name->obj_hidden = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5G__name_move_path(H5RS_str_t **path_r_ptr, const char *full_suffix, const char *src_path,
                    const char *dst_path)
{
    const char *path;                
    size_t      path_len;            
    size_t      full_suffix_len;     
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(path_r_ptr && *path_r_ptr);
    assert(full_suffix);
    assert(src_path);
    assert(dst_path);

    
    path = H5RS_get_str(*path_r_ptr);
    assert(path);

    
    full_suffix_len = strlen(full_suffix);
    path_len        = strlen(path);
    if (full_suffix_len < path_len) {
        const char *dst_suffix;        
        const char *src_suffix;        
        size_t      path_prefix_len;   
        const char *path_prefix2;      
        size_t      path_prefix2_len;  
        size_t      common_prefix_len; 
        H5RS_str_t *rs;                

        
        path_prefix_len = path_len - full_suffix_len;

        
        common_prefix_len = 0;
        
        while (*(src_path + common_prefix_len) == *(dst_path + common_prefix_len))
            common_prefix_len++;
        
        while (*(src_path + common_prefix_len) != '/')
            common_prefix_len--;
        
        common_prefix_len++;

        
        src_suffix = src_path + (common_prefix_len - 1);

        
        dst_suffix = dst_path + (common_prefix_len - 1);

        
        path_prefix2     = path;
        path_prefix2_len = path_prefix_len - strlen(src_suffix);

        
        if (NULL == (rs = H5RS_create(NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, FAIL, "can't create ref-counted string");

        
        if (path_prefix2_len > 0)
            H5RS_ancat(rs, path_prefix2, path_prefix2_len);
        H5RS_acat(rs, dst_suffix);
        if (full_suffix_len > 0)
            H5RS_acat(rs, full_suffix);

        
        H5RS_decr(*path_r_ptr);

        
        *path_r_ptr = rs;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5G__name_replace_cb(void *obj_ptr, hid_t obj_id, void *key)
{
    const H5G_names_t *names = (const H5G_names_t *)key; 
    H5O_loc_t         *oloc;         
    H5G_name_t        *obj_path;     
    H5F_t             *top_obj_file; 
    bool   obj_in_child = false;     
    herr_t ret_value    = SUCCEED;   

    FUNC_ENTER_PACKAGE

    assert(obj_ptr);

    
    switch (H5I_get_type(obj_id)) {
        case H5I_GROUP:
            oloc     = H5G_oloc((H5G_t *)obj_ptr);
            obj_path = H5G_nameof((H5G_t *)obj_ptr);
            break;

        case H5I_DATASET:
            oloc     = H5D_oloc((H5D_t *)obj_ptr);
            obj_path = H5D_nameof((H5D_t *)obj_ptr);
            break;

        case H5I_DATATYPE:
            
            if (!H5T_is_named((H5T_t *)obj_ptr))
                HGOTO_DONE(SUCCEED); 

            oloc     = H5T_oloc((H5T_t *)obj_ptr);
            obj_path = H5T_nameof((H5T_t *)obj_ptr);
            break;

        case H5I_MAP:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "maps not supported in native VOL connector");

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_FILE:
        case H5I_DATASPACE:
        case H5I_ATTR:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unknown data object");
    } 
    assert(oloc);
    assert(obj_path);

    
    if (!obj_path->full_path_r)
        HGOTO_DONE(SUCCEED); 

    
    if (H5F_PARENT(oloc->file)) {
        
        if (names->dst_file && H5F_SAME_SHARED(oloc->file, names->dst_file))
            obj_in_child = true;

        
        top_obj_file = H5F_PARENT(oloc->file);
        while (H5F_PARENT(top_obj_file) != NULL) {
            
            if (names->dst_file && H5F_SAME_SHARED(top_obj_file, names->dst_file))
                obj_in_child = true;

            top_obj_file = H5F_PARENT(top_obj_file);
        } 
    }     
    else
        top_obj_file = oloc->file;

    
    if (names->dst_file && H5F_SAME_SHARED(top_obj_file, names->dst_file))
        obj_in_child = true;

    
    if (!H5F_SAME_SHARED(top_obj_file, names->src_file))
        HGOTO_DONE(SUCCEED); 

    switch (names->op) {
        
        case H5G_NAME_MOUNT:
            
            if (obj_in_child) {
                const char *full_path; 
                const char *src_path;  
                H5RS_str_t *rs;        

                
                full_path = H5RS_get_str(obj_path->full_path_r);
                src_path  = H5RS_get_str(names->src_full_path_r);

                
                if (NULL == (rs = H5RS_create(src_path)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, FAIL, "can't create ref-counted string");
                H5RS_acat(rs, full_path);

                
                H5RS_decr(obj_path->full_path_r);

                
                obj_path->full_path_r = rs;
            } 
            
            else {
                
                
                if (H5G__common_path(obj_path->full_path_r, names->src_full_path_r) &&
                    H5RS_cmp(obj_path->full_path_r, names->src_full_path_r)) {
                    
                    (obj_path->obj_hidden)++;
                } 
            }     
            break;

        
        case H5G_NAME_UNMOUNT:
            if (obj_in_child) {
                const char *full_path;   
                const char *full_suffix; 
                const char *src_path;    
                H5RS_str_t *rs;          

                
                full_path = H5RS_get_str(obj_path->full_path_r);
                src_path  = H5RS_get_str(names->src_full_path_r);

                
                full_suffix = full_path + strlen(src_path);

                
                if (NULL == (rs = H5RS_create(full_suffix)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, FAIL, "can't create ref-counted string");

                
                H5RS_decr(obj_path->full_path_r);

                
                obj_path->full_path_r = rs;

                
                if (obj_path->user_path_r && H5RS_len(rs) < H5RS_len(obj_path->user_path_r)) {
                    
                    H5RS_decr(obj_path->user_path_r);
                    obj_path->user_path_r = NULL;
                } 
            }     
            else {
                
                if (H5G__common_path(obj_path->full_path_r, names->src_full_path_r) &&
                    H5RS_cmp(obj_path->full_path_r, names->src_full_path_r)) {
                    
                    (obj_path->obj_hidden)--;
                } 
            }     
            break;

        
        case H5G_NAME_DELETE:
            
            if (H5G__common_path(obj_path->full_path_r, names->src_full_path_r)) {
                
                H5G_name_free(obj_path);
            } 
            break;

        
        case H5G_NAME_MOVE: 
            
            if (H5G__common_path(obj_path->full_path_r, names->src_full_path_r)) {
                const char *full_path;   
                const char *full_suffix; 
                const char *src_path;    
                const char *dst_path;    
                H5RS_str_t *rs;          

                
                assert(names->dst_full_path_r);

                
                full_path = H5RS_get_str(obj_path->full_path_r);
                src_path  = H5RS_get_str(names->src_full_path_r);
                dst_path  = H5RS_get_str(names->dst_full_path_r);

                
                assert(*src_path == '/');
                assert(*dst_path == '/');

                
                full_suffix = full_path + strlen(src_path);

                
                if (obj_path->user_path_r)
                    if (H5G__name_move_path(&(obj_path->user_path_r), full_suffix, src_path, dst_path) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_PATH, FAIL, "can't build user path name");

                
                if (NULL == (rs = H5RS_create(dst_path)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, FAIL, "can't create ref-counted string");
                H5RS_acat(rs, full_suffix);

                
                H5RS_decr(obj_path->full_path_r);

                
                obj_path->full_path_r = rs;
            } 
            break;

        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid operation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_name_replace(const H5O_link_t *lnk, H5G_names_op_t op, H5F_t *src_file, H5RS_str_t *src_full_path_r,
                 H5F_t *dst_file, H5RS_str_t *dst_full_path_r)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(src_file);

    
    if (src_full_path_r) {
        bool search_group    = false; 
        bool search_dataset  = false; 
        bool search_datatype = false; 

        
        if (lnk) {
            
            switch (lnk->type) {
                case H5L_TYPE_HARD: {
                    H5O_loc_t  tmp_oloc; 
                    H5O_type_t obj_type; 

                    
                    tmp_oloc.file = src_file;
                    tmp_oloc.addr = lnk->u.hard.addr;

                    
                    if (H5O_obj_type(&tmp_oloc, &obj_type) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get object type");

                    
                    switch (obj_type) {
                        case H5O_TYPE_GROUP:
                            
                            search_group = true;
                            break;

                        case H5O_TYPE_DATASET:
                            
                            search_dataset = true;
                            break;

                        case H5O_TYPE_NAMED_DATATYPE:
                            
                            search_datatype = true;
                            break;

                        case H5O_TYPE_MAP:
                            HGOTO_ERROR(H5E_SYM, H5E_BADTYPE, FAIL,
                                        "maps not supported in native VOL connector");

                        case H5O_TYPE_UNKNOWN:
                        case H5O_TYPE_NTYPES:
                            
                        default:
                            HGOTO_ERROR(H5E_SYM, H5E_BADTYPE, FAIL, "not valid object type");
                    } 
                }     
                break;

                case H5L_TYPE_SOFT:
                    
                    search_group = search_dataset = search_datatype = true;
                    break;

                case H5L_TYPE_ERROR:
                case H5L_TYPE_EXTERNAL:
                case H5L_TYPE_MAX:
                default: 
                    
                    if (lnk->type < H5L_TYPE_UD_MIN)
                        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unknown link type");

                    
                    break;
            } 
        }     
        else {
            
            search_group = search_dataset = search_datatype = true;
        }

        
        if (search_group || search_dataset || search_datatype) {
            H5G_names_t names; 

            
            while (H5F_PARENT(src_file))
                src_file = H5F_PARENT(src_file);

            
            names.src_file        = src_file;
            names.src_full_path_r = src_full_path_r;
            names.dst_file        = dst_file;
            names.dst_full_path_r = dst_full_path_r;
            names.op              = op;

            
            if (search_group)
                if (H5I_iterate(H5I_GROUP, H5G__name_replace_cb, &names, false) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "can't iterate over groups");

            
            if (search_dataset)
                if (H5I_iterate(H5I_DATASET, H5G__name_replace_cb, &names, false) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "can't iterate over datasets");

            
            if (search_datatype)
                if (H5I_iterate(H5I_DATATYPE, H5G__name_replace_cb, &names, false) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "can't iterate over datatypes");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__get_name_by_addr_cb(hid_t gid, const char *path, const H5L_info2_t *linfo, void *_udata)
{
    H5G_gnba_iter_t *udata = (H5G_gnba_iter_t *)_udata; 
    H5G_loc_t        obj_loc;                           
    H5G_name_t       obj_path;                          
    H5O_loc_t        obj_oloc;                          
    bool             obj_found = false;                 
    herr_t           ret_value = H5_ITER_CONT;          

    FUNC_ENTER_PACKAGE

    
    assert(path);
    assert(linfo);
    assert(udata->loc);
    assert(udata->path == NULL);

    
    if (linfo->type == H5L_TYPE_HARD) {
        haddr_t link_addr;

        
        if (H5VL_native_token_to_addr(udata->loc->file, H5I_FILE, linfo->u.token, &link_addr) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTUNSERIALIZE, FAIL, "can't deserialize object token into address");

        if (udata->loc->addr == link_addr) {
            H5G_loc_t grp_loc; 

            
            if (H5G_loc(gid, &grp_loc) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5_ITER_ERROR, "bad group location");

            
            obj_loc.oloc = &obj_oloc;
            obj_loc.path = &obj_path;
            H5G_loc_reset(&obj_loc);

            
            if (H5G_loc_find(&grp_loc, path, &obj_loc ) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, H5_ITER_ERROR, "object not found");
            obj_found = true;

            
            
            if (udata->loc->addr == obj_loc.oloc->addr && udata->loc->file == obj_loc.oloc->file) {
                if (NULL == (udata->path = H5MM_strdup(path)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTALLOC, H5_ITER_ERROR, "can't duplicate path string");

                
                HGOTO_DONE(H5_ITER_STOP);
            } 
        }     
    }         

done:
    if (obj_found && H5G_loc_free(&obj_loc) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, H5_ITER_ERROR, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_get_name_by_addr(H5F_t *f, const H5O_loc_t *loc, char *name, size_t size, size_t *name_len)
{
    H5G_gnba_iter_t udata;               
    size_t          len;                 
    H5G_loc_t       root_loc;            
    bool            found_obj = false;   
    herr_t          status;              
    herr_t          ret_value = SUCCEED; 

    
    memset(&udata, 0, sizeof(udata));

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5G_root_loc(f, &root_loc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get root group's location");

    
    if (root_loc.oloc->addr == loc->addr && root_loc.oloc->file == loc->file) {
        if (NULL == (udata.path = H5MM_strdup("")))
            HGOTO_ERROR(H5E_SYM, H5E_CANTALLOC, FAIL, "can't duplicate path string");
        found_obj = true;
    } 
    else {
        
        udata.loc  = loc;
        udata.path = NULL;

        
        if ((status = H5G_visit(&root_loc, "/", H5_INDEX_NAME, H5_ITER_NATIVE, H5G__get_name_by_addr_cb,
                                &udata)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "group traversal failed while looking for object name");
        else if (status > 0)
            found_obj = true;
    } 

    
    if (found_obj) {
        
        len = strlen(udata.path) + 1; 

        
        if (name) {
            
            strncpy(name, "/", (size_t)2);

            
            
            strncat(name, udata.path, (size - 2));
            if (len >= size)
                name[size - 1] = '\0';
        } 
    }     
    else
        len = 0;

    
    if (name_len)
        *name_len = len;

done:
    
    H5MM_xfree(udata.path);

    FUNC_LEAVE_NOAPI(ret_value)
} 
