/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5Gprivate.h"  
#include "H5HGprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Oprivate.h"  
#include "H5Pprivate.h"  
#include "H5RTprivate.h" 
#include "H5Sprivate.h"  
#include "H5VLprivate.h" 

#define H5D_VIRTUAL_DEF_SUB_DSET_SIZE 128

#define H5D_VIRTUAL_NOT_IN_TREE_INIT_SIZE 64

#define H5D_RTREE_SHOULD_INSERT(entry)                                                                       \
    !(((entry)->unlim_dim_virtual >= 0) || ((entry)->source_dset.virtual_select &&                           \
                                            H5S_GET_EXTENT_NDIMS((entry)->source_dset.virtual_select) < 1))

static herr_t H5D__virtual_construct(H5F_t *f, H5D_t *dset);
static herr_t H5D__virtual_init(H5F_t *f, H5D_t *dset, hid_t dapl_id, bool open_op);
static bool   H5D__virtual_is_space_alloc(const H5O_storage_t *storage);
static bool   H5D__virtual_is_data_cached(const H5D_shared_t *shared_dset);
static herr_t H5D__virtual_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__virtual_read(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__virtual_write(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__virtual_flush(H5D_t *dset);

static herr_t H5D__virtual_free_layout_mappings(H5O_storage_virtual_t *virt);
static herr_t H5D__virtual_open_source_dset(const H5D_t *vdset, H5O_storage_virtual_ent_t *virtual_ent,
                                            H5O_storage_virtual_srcdset_t *source_dset);
static herr_t H5D__virtual_reset_source_dset(H5O_storage_virtual_ent_t     *virtual_ent,
                                             H5O_storage_virtual_srcdset_t *source_dset);
static herr_t H5D__virtual_str_append(const char *src, size_t src_len, char **p, char **buf,
                                      size_t *buf_size);
static herr_t H5D__virtual_copy_parsed_name(H5O_storage_virtual_name_seg_t **dst,
                                            H5O_storage_virtual_name_seg_t  *src);
static herr_t H5D__virtual_build_source_name(char                                 *source_name,
                                             const H5O_storage_virtual_name_seg_t *parsed_name,
                                             size_t static_strlen, size_t nsubs, hsize_t blockno,
                                             char **built_name);
static herr_t H5D__virtual_init_all(const H5D_t *dset);
static herr_t H5D__virtual_pre_io(H5D_dset_io_info_t *dset_info, H5O_storage_virtual_t *storage,
                                  H5S_t *file_space, H5S_t *mem_space, hsize_t *tot_nelmts,
                                  H5RT_result_set_t *mappings);
static herr_t H5D__virtual_post_io(H5O_storage_virtual_t *storage, H5RT_result_set_t *mappings);
static herr_t H5D__virtual_close_mapping(H5O_storage_virtual_ent_t *mapping);
static herr_t H5D__virtual_read_one_mapping(H5D_dset_io_info_t        *dset_info,
                                            H5O_storage_virtual_ent_t *mapping);
static herr_t H5D__virtual_read_one_src(H5D_dset_io_info_t            *dset_info,
                                        H5O_storage_virtual_srcdset_t *source_dset);
static herr_t H5D__virtual_write_one_mapping(H5D_dset_io_info_t        *dset_info,
                                             H5O_storage_virtual_ent_t *mapping);
static herr_t H5D__virtual_write_one_src(H5D_dset_io_info_t            *dset_info,
                                         H5O_storage_virtual_srcdset_t *source_dset);

static herr_t H5D__virtual_build_tree(H5O_storage_virtual_t *virt, int rank);
static herr_t H5D__mappings_to_leaves(H5O_storage_virtual_ent_t *mappings, size_t num_mappings,
                                      H5RT_leaf_t **leaves_out, H5O_storage_virtual_ent_t ***not_in_tree_out,
                                      size_t *leaf_count, size_t *not_in_tree_count,
                                      size_t *not_in_tree_nalloc);
static herr_t H5D__should_build_tree(H5O_storage_virtual_t *storage, hid_t dapl_id, bool *should_build_tree);
static herr_t H5D__virtual_not_in_tree_grow(H5O_storage_virtual_ent_t ***list, size_t *nalloc);
static herr_t H5D__virtual_not_in_tree_add(H5O_storage_virtual_ent_t ***list, size_t *nused, size_t *nalloc,
                                           H5O_storage_virtual_ent_t *mapping);

const H5D_layout_ops_t H5D_LOPS_VIRTUAL[1] = {{
    H5D__virtual_construct,      
    H5D__virtual_init,           
    H5D__virtual_is_space_alloc, 
    H5D__virtual_is_data_cached, 
    H5D__virtual_io_init,        
    NULL,                        
    H5D__virtual_read,           
    H5D__virtual_write,          
    NULL,                        
    NULL,                        
    H5D__virtual_flush,          
    NULL,                        
    NULL                         
}};

H5FL_DEFINE(H5O_storage_virtual_name_seg_t);

H5FL_DEFINE_STATIC(H5D_virtual_held_file_t);

herr_t
H5D_virtual_check_mapping_pre(const H5S_t *vspace, const H5S_t *src_space,
                              H5O_virtual_space_status_t space_status)
{
    H5S_sel_type select_type;         
    hsize_t      nelmts_vs;           
    hsize_t      nelmts_ss;           
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5S_SEL_ERROR == (select_type = H5S_GET_SELECT_TYPE(vspace)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get selection type");
    if (select_type == H5S_SEL_POINTS)
        HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                    "point selections not currently supported with virtual datasets");
    if (H5S_SEL_ERROR == (select_type = H5S_GET_SELECT_TYPE(src_space)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get selection type");
    if (select_type == H5S_SEL_POINTS)
        HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                    "point selections not currently supported with virtual datasets");

    
    nelmts_vs = (hsize_t)H5S_GET_SELECT_NPOINTS(vspace);
    nelmts_ss = (hsize_t)H5S_GET_SELECT_NPOINTS(src_space);

    
    if (nelmts_vs == H5S_UNLIMITED) {
        
        if (nelmts_ss == H5S_UNLIMITED) {
            hsize_t nenu_vs; 
            hsize_t nenu_ss; 

            
            if (H5S_get_select_num_elem_non_unlim(vspace, &nenu_vs) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOUNT, FAIL,
                            "can't get number of elements in non-unlimited dimension");
            if (H5S_get_select_num_elem_non_unlim(src_space, &nenu_ss) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOUNT, FAIL,
                            "can't get number of elements in non-unlimited dimension");
            if (nenu_vs != nenu_ss)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                            "numbers of elements in the non-unlimited dimensions is different for source and "
                            "virtual spaces");
        } 
        
    } 
    else if (space_status != H5O_VIRTUAL_STATUS_INVALID)
        
        if (nelmts_vs != nelmts_ss)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "virtual and source space selections have different numbers of elements");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_virtual_check_mapping_post(const H5O_storage_virtual_ent_t *ent)
{
    hsize_t nelmts_vs;           
    hsize_t nelmts_ss;           
    H5S_t  *tmp_space = NULL;    
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    nelmts_vs = (hsize_t)H5S_GET_SELECT_NPOINTS(ent->source_dset.virtual_select);
    nelmts_ss = (hsize_t)H5S_GET_SELECT_NPOINTS(ent->source_select);

    
    if ((nelmts_vs == H5S_UNLIMITED) && (nelmts_ss != H5S_UNLIMITED)) {
        
        if ((ent->psfn_nsubs == 0) && (ent->psdn_nsubs == 0))
            HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL,
                        "unlimited virtual selection, limited source selection, and no Rprintf specifiers in "
                        "source names");

        
        if (H5S_GET_SELECT_TYPE(ent->source_dset.virtual_select) != H5S_SEL_HYPERSLABS)
            HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL,
                        "virtual selection with Rprintf mapping must be hyperslab");

        
        if (ent->source_space_status != H5O_VIRTUAL_STATUS_INVALID) {
            
            if (NULL == (tmp_space = H5S_hyper_get_unlim_block(ent->source_dset.virtual_select, (hsize_t)0)))
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get first block in virtual selection");

            
            nelmts_vs = (hsize_t)H5S_GET_SELECT_NPOINTS(tmp_space);
            if (nelmts_vs != nelmts_ss)
                HGOTO_ERROR(
                    H5E_ARGS, H5E_BADVALUE, FAIL,
                    "virtual (single block) and source space selections have different numbers of elements");
        } 
    }     
    else
        
        if ((ent->psfn_nsubs > 0) || (ent->psdn_nsubs > 0))
            HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL,
                        "Rprintf specifier(s) in source name(s) without an unlimited virtual selection and "
                        "limited source selection");

done:
    
    if (tmp_space)
        if (H5S_close(tmp_space) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "can't close dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_virtual_update_min_dims(H5O_layout_t *layout, size_t idx)
{
    H5O_storage_virtual_t     *virt = &layout->storage.u.virt;
    H5O_storage_virtual_ent_t *ent  = &virt->list[idx];
    H5S_sel_type               sel_type;
    int                        rank;
    hsize_t                    bounds_start[H5S_MAX_RANK];
    hsize_t                    bounds_end[H5S_MAX_RANK];
    int                        i;
    herr_t                     ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    assert(layout);
    assert(layout->type == H5D_VIRTUAL);
    assert(idx < virt->list_nalloc);

    
    if (H5S_SEL_ERROR == (sel_type = H5S_GET_SELECT_TYPE(ent->source_dset.virtual_select)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection type");

    
    if ((sel_type == H5S_SEL_ALL) || (sel_type == H5S_SEL_NONE))
        HGOTO_DONE(SUCCEED);

    
    if ((rank = H5S_GET_EXTENT_NDIMS(ent->source_dset.virtual_select)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get number of dimensions");

    
    if (H5S_SELECT_BOUNDS(ent->source_dset.virtual_select, bounds_start, bounds_end) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection bounds");

    
    for (i = 0; i < rank; i++)
        
        if ((i != ent->unlim_dim_virtual) && (bounds_end[i] >= virt->min_dims[i]))
            virt->min_dims[i] = bounds_end[i] + (hsize_t)1;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_virtual_check_min_dims(const H5D_t *dset)
{
    int     rank;
    hsize_t dims[H5S_MAX_RANK];
    int     i;
    herr_t  ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    assert(dset);
    assert(dset->shared);
    assert(dset->shared->layout.type == H5D_VIRTUAL);

    
    if ((rank = H5S_GET_EXTENT_NDIMS(dset->shared->space)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get number of dimensions");

    
    if (H5S_get_simple_extent_dims(dset->shared->space, dims, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get VDS dimensions");

    
    for (i = 0; i < rank; i++)
        if (dims[i] < dset->shared->layout.storage.u.virt.min_dims[i])
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                        "virtual dataset dimensions not large enough to contain all limited dimensions in "
                        "all selections");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_store_layout(H5F_t *f, H5O_layout_t *layout)
{
    H5O_storage_virtual_t *virt       = &layout->storage.u.virt;
    uint8_t               *heap_block = NULL; 
    size_t                *str_size   = NULL; 
    uint8_t               *heap_block_p;      
    size_t                 block_size;        
    hsize_t                tmp_hsize;         
    uint32_t               chksum;            
    uint8_t                max_version;       
    uint8_t                version = H5O_LAYOUT_VDS_GH_ENC_VERS_0; 
    size_t                 i;                                      
    herr_t                 ret_value = SUCCEED;                    

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(layout);
    assert(virt->serial_list_hobjid.addr == HADDR_UNDEF);

    
    if (virt->list_nused > 0) {

        
        H5CX_set_libver_bounds(f);

        
        max_version =
            H5F_LOW_BOUND(f) >= H5F_LIBVER_V200 ? H5O_LAYOUT_VDS_GH_ENC_VERS_1 : H5O_LAYOUT_VDS_GH_ENC_VERS_0;

        
        if (NULL == (str_size = (size_t *)H5MM_malloc(2 * virt->list_nused * sizeof(size_t))))
            HGOTO_ERROR(H5E_OHDR, H5E_RESOURCE, FAIL, "unable to allocate string length array");

        

        
        block_size = (size_t)1 + H5F_SIZEOF_SIZE(f);

        
        for (i = 0; i < virt->list_nused; i++) {
            H5O_storage_virtual_ent_t *ent = &virt->list[i];
            hssize_t                   select_serial_size; 

            assert(ent->source_file_name);
            assert(ent->source_dset_name);
            assert(ent->source_select);
            assert(ent->source_dset.virtual_select);

            
            str_size[2 * i] = strlen(ent->source_file_name) + (size_t)1;
            block_size += str_size[2 * i];

            
            str_size[(2 * i) + 1] = strlen(ent->source_dset_name) + (size_t)1;
            block_size += str_size[(2 * i) + 1];

            
            if ((select_serial_size = H5S_SELECT_SERIAL_SIZE(ent->source_select)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to check dataspace selection size");
            block_size += (size_t)select_serial_size;

            
            if ((select_serial_size = H5S_SELECT_SERIAL_SIZE(ent->source_dset.virtual_select)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to check dataspace selection size");
            block_size += (size_t)select_serial_size;
        }

        
        block_size += 4;

        
        if (max_version >= H5O_LAYOUT_VDS_GH_ENC_VERS_1) {
            size_t block_size_1; 
            
            block_size_1 = (size_t)1 + H5F_SIZEOF_SIZE(f);

            
            for (i = 0; i < virt->list_nused; i++) {
                H5O_storage_virtual_ent_t *ent = &virt->list[i];
                hssize_t                   select_serial_size; 

                
                block_size_1 += (size_t)1;

                
                if (strcmp(ent->source_file_name, ".")) {
                    if (ent->source_file_orig == SIZE_MAX)
                        block_size_1 += str_size[2 * i];
                    else
                        block_size_1 += MIN(str_size[2 * i], H5F_SIZEOF_SIZE(f));
                }

                
                if (ent->source_dset_orig == SIZE_MAX)
                    block_size_1 += str_size[(2 * i) + 1];
                else
                    block_size_1 += MIN(str_size[(2 * i) + 1], H5F_SIZEOF_SIZE(f));

                
                if ((select_serial_size = H5S_SELECT_SERIAL_SIZE(ent->source_select)) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to check dataspace selection size");
                block_size_1 += (size_t)select_serial_size;

                
                if ((select_serial_size = H5S_SELECT_SERIAL_SIZE(ent->source_dset.virtual_select)) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to check dataspace selection size");
                block_size_1 += (size_t)select_serial_size;
            }

            
            block_size_1 += 4;

            
            if (block_size_1 <= block_size) {
                version    = H5O_LAYOUT_VDS_GH_ENC_VERS_1;
                block_size = block_size_1;
            }
        }

        
        if (NULL == (heap_block = (uint8_t *)H5MM_malloc(block_size)))
            HGOTO_ERROR(H5E_OHDR, H5E_RESOURCE, FAIL, "unable to allocate heap block");

        
        heap_block_p = heap_block;

        
        *heap_block_p++ = version;

        
        H5_CHECK_OVERFLOW(virt->list_nused, size_t, hsize_t);
        tmp_hsize = (hsize_t)virt->list_nused;
        H5F_ENCODE_LENGTH(f, heap_block_p, tmp_hsize);

        
        for (i = 0; i < virt->list_nused; i++) {
            H5O_storage_virtual_ent_t *ent   = &virt->list[i];
            uint8_t                    flags = 0;

            
            if (version >= H5O_LAYOUT_VDS_GH_ENC_VERS_1) {
                if (!strcmp(ent->source_file_name, "."))
                    
                    flags |= H5O_LAYOUT_VDS_SOURCE_SAME_FILE;
                else if ((ent->source_file_orig != SIZE_MAX) && (str_size[2 * i] >= H5F_SIZEOF_SIZE(f)))
                    
                    flags |= H5O_LAYOUT_VDS_SOURCE_FILE_SHARED;

                if ((ent->source_dset_orig != SIZE_MAX) && (str_size[(2 * i) + 1] >= H5F_SIZEOF_SIZE(f)))
                    
                    flags |= H5O_LAYOUT_VDS_SOURCE_DSET_SHARED;

                *heap_block_p++ = flags;
            }

            
            if (!(flags & H5O_LAYOUT_VDS_SOURCE_SAME_FILE)) {
                if (flags & H5O_LAYOUT_VDS_SOURCE_FILE_SHARED) {
                    assert(ent->source_file_orig < i);
                    tmp_hsize = (hsize_t)ent->source_file_orig;
                    H5F_ENCODE_LENGTH(f, heap_block_p, tmp_hsize);
                }
                else {
                    H5MM_memcpy((char *)heap_block_p, ent->source_file_name, str_size[2 * i]);
                    heap_block_p += str_size[2 * i];
                }
            }

            
            if (flags & H5O_LAYOUT_VDS_SOURCE_DSET_SHARED) {
                assert(ent->source_dset_orig < i);
                tmp_hsize = (hsize_t)ent->source_dset_orig;
                H5F_ENCODE_LENGTH(f, heap_block_p, tmp_hsize);
            }
            else {
                H5MM_memcpy((char *)heap_block_p, ent->source_dset_name, str_size[(2 * i) + 1]);
                heap_block_p += str_size[(2 * i) + 1];
            }

            
            if (H5S_SELECT_SERIALIZE(ent->source_select, &heap_block_p) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to serialize source selection");

            
            if (H5S_SELECT_SERIALIZE(ent->source_dset.virtual_select, &heap_block_p) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to serialize virtual selection");
        }

        
        chksum = H5_checksum_metadata(heap_block, block_size - (size_t)4, 0);
        UINT32ENCODE(heap_block_p, chksum);

        
        if (H5HG_insert(f, block_size, heap_block, &(virt->serial_list_hobjid)) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "unable to insert virtual dataset heap block");
    }

done:
    heap_block = (uint8_t *)H5MM_xfree(heap_block);
    str_size   = (size_t *)H5MM_xfree(str_size);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_load_layout(H5F_t *f, H5O_layout_t *layout)
{
    uint8_t *heap_block = NULL;
    herr_t   ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (!layout->storage.u.virt.list && layout->storage.u.virt.serial_list_hobjid.addr != HADDR_UNDEF) {
        const uint8_t *heap_block_p;
        const uint8_t *heap_block_p_end;
        uint8_t        heap_vers;
        size_t         block_size = 0;
        size_t         tmp_size;
        hsize_t        tmp_hsize = 0;
        uint32_t       stored_chksum;
        uint32_t       computed_chksum;
        size_t         first_same_file       = SIZE_MAX;
        bool           clear_file_hash_table = false;

        
        if (NULL == (heap_block = (uint8_t *)H5HG_read(f, &(layout->storage.u.virt.serial_list_hobjid), NULL,
                                                       &block_size)))
            HGOTO_ERROR(H5E_OHDR, H5E_READERROR, FAIL, "Unable to read global heap block");

        heap_block_p     = (const uint8_t *)heap_block;
        heap_block_p_end = heap_block_p + block_size - 1;

        
        if (H5_IS_BUFFER_OVERFLOW(heap_block_p, 1, heap_block_p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        heap_vers = (uint8_t)*heap_block_p++;

        assert(H5O_LAYOUT_VDS_GH_ENC_VERS_0 == 0);
        if (heap_vers > (uint8_t)H5O_LAYOUT_VDS_GH_ENC_VERS_1)
            HGOTO_ERROR(H5E_OHDR, H5E_VERSION, FAIL,
                        "bad version # of encoded VDS heap information, expected %u or lower, got %u",
                        (unsigned)H5O_LAYOUT_VDS_GH_ENC_VERS_1, (unsigned)heap_vers);

        
        if (H5_IS_BUFFER_OVERFLOW(heap_block_p, H5F_sizeof_size(f), heap_block_p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        H5F_DECODE_LENGTH(f, heap_block_p, tmp_hsize);

        
        if (tmp_hsize > 0) {
            if (NULL == (layout->storage.u.virt.list = (H5O_storage_virtual_ent_t *)H5MM_calloc(
                             (size_t)tmp_hsize * sizeof(H5O_storage_virtual_ent_t))))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "unable to allocate heap block");
        }
        else {
            
            layout->storage.u.virt.list = NULL;
        }

        layout->storage.u.virt.list_nalloc = (size_t)tmp_hsize;
        layout->storage.u.virt.list_nused  = (size_t)tmp_hsize;

        
        for (size_t i = 0; i < layout->storage.u.virt.list_nused; i++) {
            H5O_storage_virtual_ent_t *ent = &layout->storage.u.virt.list[i]; 
            ptrdiff_t                  avail_buffer_space;
            uint8_t                    flags = 0;

            avail_buffer_space = heap_block_p_end - heap_block_p + 1;
            if (avail_buffer_space <= 0)
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");

            
            if (heap_vers >= H5O_LAYOUT_VDS_GH_ENC_VERS_1) {
                flags = *heap_block_p++;

                if (flags & ~H5O_LAYOUT_ALL_VDS_FLAGS)
                    HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "bad flag value for VDS mapping");
            }

            avail_buffer_space = heap_block_p_end - heap_block_p + 1;

            
            if (flags & H5O_LAYOUT_VDS_SOURCE_SAME_FILE) {
                
                if (first_same_file == SIZE_MAX) {
                    
                    if (NULL == (ent->source_file_name = (char *)H5MM_malloc(2)))
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL,
                                    "memory allocation failed for source file string");
                    ent->source_file_name[0] = '.';
                    ent->source_file_name[1] = '\0';
                    ent->source_file_orig    = SIZE_MAX;
                    first_same_file          = i;

                    
                    clear_file_hash_table = true;
                }
                else {
                    
                    assert(first_same_file < i);
                    ent->source_file_name = layout->storage.u.virt.list[first_same_file].source_file_name;
                    ent->source_file_orig = first_same_file;
                }
            }
            else {
                if (flags & H5O_LAYOUT_VDS_SOURCE_FILE_SHARED) {
                    if (avail_buffer_space < H5F_SIZEOF_SIZE(f))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");

                    
                    H5F_DECODE_LENGTH(f, heap_block_p, tmp_hsize);
                    H5_CHECK_OVERFLOW(tmp_hsize, hsize_t, size_t);
                    if ((size_t)tmp_hsize >= i)
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                                    "origin source file entry has higher index than current entry");
                    ent->source_file_orig = (size_t)tmp_hsize;

                    
                    ent->source_file_name = layout->storage.u.virt.list[tmp_hsize].source_file_name;
                }
                else {
                    tmp_size = strnlen((const char *)heap_block_p, (size_t)avail_buffer_space);
                    if (tmp_size == (size_t)avail_buffer_space)
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding - unterminated source file "
                                    "name string");
                    else
                        tmp_size += 1; 

                    
                    H5D_VIRTUAL_FIND_OR_ADD_NAME(file, layout, heap_block_p, tmp_size - 1, ent, FAIL);

                    
                    heap_block_p += tmp_size;
                }
            }

            avail_buffer_space = heap_block_p_end - heap_block_p + 1;

            
            if (flags & H5O_LAYOUT_VDS_SOURCE_DSET_SHARED) {
                if (avail_buffer_space < H5F_SIZEOF_SIZE(f))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");

                
                H5F_DECODE_LENGTH(f, heap_block_p, tmp_hsize);
                H5_CHECK_OVERFLOW(tmp_hsize, hsize_t, size_t);
                if ((size_t)tmp_hsize >= i)
                    HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                                "origin source dataset entry has higher index than current entry");
                ent->source_dset_orig = (size_t)tmp_hsize;

                
                ent->source_dset_name = layout->storage.u.virt.list[tmp_hsize].source_dset_name;
            }
            else {
                tmp_size = strnlen((const char *)heap_block_p, (size_t)avail_buffer_space);
                if (tmp_size == (size_t)avail_buffer_space)
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                "ran off end of input buffer while decoding - unterminated source dataset "
                                "name string");
                else
                    tmp_size += 1; 

                
                H5D_VIRTUAL_FIND_OR_ADD_NAME(dset, layout, heap_block_p, tmp_size - 1, ent, FAIL);

                
                heap_block_p += tmp_size;
            }

            
            avail_buffer_space = heap_block_p_end - heap_block_p + 1;

            if (avail_buffer_space <= 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_OVERFLOW, FAIL, "buffer overflow while decoding layout");

            if (H5S_SELECT_DESERIALIZE(&ent->source_select, &heap_block_p, (size_t)(avail_buffer_space)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, FAIL, "can't decode source space selection");

            

            
            avail_buffer_space = heap_block_p_end - heap_block_p + 1;

            if (avail_buffer_space <= 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_OVERFLOW, FAIL, "buffer overflow while decoding layout");

            if (H5S_SELECT_DESERIALIZE(&ent->source_dset.virtual_select, &heap_block_p,
                                       (size_t)(avail_buffer_space)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, FAIL, "can't decode virtual space selection");

            
            if (H5D_virtual_parse_source_name(ent->source_file_name, &ent->parsed_source_file_name,
                                              &ent->psfn_static_strlen, &ent->psfn_nsubs) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "can't parse source file name");
            if (H5D_virtual_parse_source_name(ent->source_dset_name, &ent->parsed_source_dset_name,
                                              &ent->psdn_static_strlen, &ent->psdn_nsubs) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "can't parse source dataset name");

            
            if ((ent->psfn_nsubs == 0) && (ent->psdn_nsubs == 0)) {
                if (ent->parsed_source_file_name)
                    ent->source_dset.file_name = ent->parsed_source_file_name->name_segment;
                else
                    ent->source_dset.file_name = ent->source_file_name;
                if (ent->parsed_source_dset_name)
                    ent->source_dset.dset_name = ent->parsed_source_dset_name->name_segment;
                else
                    ent->source_dset.dset_name = ent->source_dset_name;
            }

            
            ent->unlim_dim_source     = H5S_get_select_unlim_dim(ent->source_select);
            ent->unlim_dim_virtual    = H5S_get_select_unlim_dim(ent->source_dset.virtual_select);
            ent->unlim_extent_source  = HSIZE_UNDEF;
            ent->unlim_extent_virtual = HSIZE_UNDEF;
            ent->clip_size_source     = HSIZE_UNDEF;
            ent->clip_size_virtual    = HSIZE_UNDEF;

            
            if (ent->unlim_dim_virtual < 0) {
                ent->source_dset.clipped_source_select  = ent->source_select;
                ent->source_dset.clipped_virtual_select = ent->source_dset.virtual_select;
            }

            
            if (H5D_virtual_check_mapping_pre(ent->source_dset.virtual_select, ent->source_select,
                                              H5O_VIRTUAL_STATUS_INVALID) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "invalid mapping selections");
            if (H5D_virtual_check_mapping_post(ent) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid mapping entry");

            
            if (H5D_virtual_update_min_dims(layout, i) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL,
                            "unable to update virtual dataset minimum dimensions");
        }

        
        if (H5_IS_BUFFER_OVERFLOW(heap_block_p, 4, heap_block_p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        UINT32DECODE(heap_block_p, stored_chksum);

        
        computed_chksum = H5_checksum_metadata(heap_block, block_size - (size_t)4, 0);

        
        if (stored_chksum != computed_chksum)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "incorrect metadata checksum for global heap block");

        
        if ((size_t)(heap_block_p - heap_block) != block_size)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "incorrect heap block size");

        
        if (clear_file_hash_table)
            HASH_CLEAR(hh_source_file, layout->storage.u.virt.source_file_hash_table);
    } 

done:
    heap_block = (uint8_t *)H5MM_xfree(heap_block);

    
    if (ret_value < 0)
        if (H5D__virtual_free_layout_mappings(&layout->storage.u.virt) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release VDS mappings");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_copy_layout(H5O_layout_t *layout)
{
    H5O_storage_virtual_ent_t  *orig_list             = NULL;
    H5O_storage_virtual_ent_t **orig_not_in_tree_list = NULL;
    H5O_storage_virtual_t      *virt                  = &layout->storage.u.virt;
    hid_t                       orig_source_fapl;
    hid_t                       orig_source_dapl;
    H5P_genplist_t             *plist;
    size_t                      i;
    herr_t                      ret_value = SUCCEED;
    H5RT_t                     *new_tree  = NULL;

    FUNC_ENTER_PACKAGE

    assert(layout);
    assert(layout->type == H5D_VIRTUAL);

    
    virt->source_file_hash_table = NULL;
    virt->source_dset_hash_table = NULL;

    
    orig_source_fapl       = virt->source_fapl;
    virt->source_fapl      = -1;
    orig_source_dapl       = virt->source_dapl;
    virt->source_dapl      = -1;
    orig_list              = virt->list;
    virt->list             = NULL;
    orig_not_in_tree_list  = virt->not_in_tree_list;
    virt->not_in_tree_list = NULL;

    
    if (virt->list_nused > 0) {
        assert(orig_list);

        
        if (NULL == (virt->list = H5MM_calloc(virt->list_nused * sizeof(virt->list[0]))))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL,
                        "unable to allocate memory for virtual dataset entry list");
        virt->list_nalloc = virt->list_nused;

        
        for (i = 0; i < virt->list_nused; i++) {
            H5O_storage_virtual_ent_t *ent = &virt->list[i];

            
            if (NULL == (ent->source_dset.virtual_select =
                             H5S_copy(orig_list[i].source_dset.virtual_select, false, true)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy virtual selection");

            
            ent->source_file_orig = orig_list[i].source_file_orig;
            if (ent->source_file_orig == SIZE_MAX) {
                
                if (NULL == (ent->source_file_name = H5MM_strdup(orig_list[i].source_file_name)))
                    HGOTO_ERROR(H5E_DATASET, H5E_RESOURCE, FAIL, "unable to duplicate source file name");
            }
            else
                
                ent->source_file_name = virt->list[ent->source_file_orig].source_file_name;

            
            ent->source_dset_orig = orig_list[i].source_dset_orig;
            if (ent->source_dset_orig == SIZE_MAX) {
                if (NULL == (ent->source_dset_name = H5MM_strdup(orig_list[i].source_dset_name)))
                    HGOTO_ERROR(H5E_DATASET, H5E_RESOURCE, FAIL, "unable to duplicate source dataset name");
            }
            else
                
                ent->source_dset_name = virt->list[ent->source_dset_orig].source_dset_name;

            
            if (NULL == (ent->source_select = H5S_copy(orig_list[i].source_select, false, true)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy source selection");

            
            if (orig_list[i].unlim_dim_virtual < 0) {
                ent->source_dset.clipped_source_select  = ent->source_select;
                ent->source_dset.clipped_virtual_select = ent->source_dset.virtual_select;
            } 

            
            if (H5D__virtual_copy_parsed_name(&ent->parsed_source_file_name,
                                              orig_list[i].parsed_source_file_name) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy parsed source file name");
            ent->psfn_static_strlen = orig_list[i].psfn_static_strlen;
            ent->psfn_nsubs         = orig_list[i].psfn_nsubs;
            if (H5D__virtual_copy_parsed_name(&ent->parsed_source_dset_name,
                                              orig_list[i].parsed_source_dset_name) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy parsed source dataset name");
            ent->psdn_static_strlen = orig_list[i].psdn_static_strlen;
            ent->psdn_nsubs         = orig_list[i].psdn_nsubs;

            
            if (orig_list[i].source_dset.file_name) {
                if (orig_list[i].source_dset.file_name == orig_list[i].source_file_name)
                    ent->source_dset.file_name = ent->source_file_name;
                else if (orig_list[i].parsed_source_file_name &&
                         (orig_list[i].source_dset.file_name !=
                          orig_list[i].parsed_source_file_name->name_segment)) {
                    assert(ent->parsed_source_file_name);
                    assert(ent->parsed_source_file_name->name_segment);
                    ent->source_dset.file_name = ent->parsed_source_file_name->name_segment;
                } 
                else if (NULL ==
                         (ent->source_dset.file_name = H5MM_strdup(orig_list[i].source_dset.file_name)))
                    HGOTO_ERROR(H5E_DATASET, H5E_RESOURCE, FAIL, "unable to duplicate source file name");
            } 
            if (orig_list[i].source_dset.dset_name) {
                if (orig_list[i].source_dset.dset_name == orig_list[i].source_dset_name)
                    ent->source_dset.dset_name = ent->source_dset_name;
                else if (orig_list[i].parsed_source_dset_name &&
                         (orig_list[i].source_dset.dset_name !=
                          orig_list[i].parsed_source_dset_name->name_segment)) {
                    assert(ent->parsed_source_dset_name);
                    assert(ent->parsed_source_dset_name->name_segment);
                    ent->source_dset.dset_name = ent->parsed_source_dset_name->name_segment;
                } 
                else if (NULL ==
                         (ent->source_dset.dset_name = H5MM_strdup(orig_list[i].source_dset.dset_name)))
                    HGOTO_ERROR(H5E_DATASET, H5E_RESOURCE, FAIL, "unable to duplicate source dataset name");
            } 

            
            ent->unlim_dim_source     = orig_list[i].unlim_dim_source;
            ent->unlim_dim_virtual    = orig_list[i].unlim_dim_virtual;
            ent->unlim_extent_source  = orig_list[i].unlim_extent_source;
            ent->unlim_extent_virtual = orig_list[i].unlim_extent_virtual;
            ent->clip_size_source     = orig_list[i].clip_size_source;
            ent->clip_size_virtual    = orig_list[i].clip_size_virtual;
            ent->source_space_status  = orig_list[i].source_space_status;
            ent->virtual_space_status = orig_list[i].virtual_space_status;
        } 
    }     
    else {
        
        virt->list        = NULL;
        virt->list_nalloc = 0;
    } 

    
    if (virt->tree) {
        if ((new_tree = H5RT_copy(virt->tree)) == NULL)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy spatial tree");
        virt->tree = new_tree;

        
        if (virt->not_in_tree_nused > 0) {
            assert(orig_not_in_tree_list);

            
            if ((virt->not_in_tree_list = (H5O_storage_virtual_ent_t **)H5MM_calloc(
                     virt->not_in_tree_nused * sizeof(H5O_storage_virtual_ent_t *))) == NULL)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL,
                            "unable to allocate not_in_tree_list pointer array");

            virt->not_in_tree_nalloc = virt->not_in_tree_nused;

            
            for (i = 0; i < virt->not_in_tree_nused; i++) {
                ptrdiff_t offset = orig_not_in_tree_list[i] - orig_list;  
                assert(offset >= 0 && (size_t)offset < virt->list_nused); 
                virt->not_in_tree_list[i] = &virt->list[offset];          
            }
        }
    }
    else {
        virt->tree               = NULL;
        virt->not_in_tree_list   = NULL;
        virt->not_in_tree_nused  = 0;
        virt->not_in_tree_nalloc = 0;
    }

    
    if (orig_source_fapl >= 0) {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(orig_source_fapl, H5I_GENPROP_LST)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
        if ((virt->source_fapl = H5P_copy_plist(plist, false)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy fapl");
    } 
    if (orig_source_dapl >= 0) {
        if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(orig_source_dapl, H5I_GENPROP_LST)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
        if ((virt->source_dapl = H5P_copy_plist(plist, false)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy dapl");
    } 

    
    virt->init = false;

done:
    
    if (ret_value < 0)
        if (H5D__virtual_reset_layout(layout) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to reset virtual layout");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_free_layout_mappings(H5O_storage_virtual_t *virt)
{
    size_t i, j;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(virt);

    
    HASH_CLEAR(hh_source_file, virt->source_file_hash_table);
    HASH_CLEAR(hh_source_dset, virt->source_dset_hash_table);

    
    for (i = 0; i < virt->list_nused; i++) {
        H5O_storage_virtual_ent_t *ent = &virt->list[i];
        
        if (H5D__virtual_reset_source_dset(ent, &ent->source_dset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to reset source dataset");

        
        if (ent->source_file_orig == SIZE_MAX)
            (void)H5MM_xfree(ent->source_file_name);
        if (ent->source_dset_orig == SIZE_MAX)
            (void)H5MM_xfree(ent->source_dset_name);

        
        for (j = 0; j < ent->sub_dset_nalloc; j++)
            if (H5D__virtual_reset_source_dset(ent, &ent->sub_dset[j]) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to reset source dataset");
        ent->sub_dset = H5MM_xfree(ent->sub_dset);

        
        if (ent->source_select)
            if (H5S_close(ent->source_select) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release source selection");

        
        H5D_virtual_free_parsed_name(ent->parsed_source_file_name);

        
        H5D_virtual_free_parsed_name(ent->parsed_source_dset_name);
    }

    
    virt->list        = H5MM_xfree(virt->list);
    virt->list_nalloc = (size_t)0;
    virt->list_nused  = (size_t)0;
    (void)memset(virt->min_dims, 0, sizeof(virt->min_dims));

    
    if (virt->tree) {
        if (H5RT_free(virt->tree) < 0) {
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to destroy spatial tree");
        }
        virt->tree = NULL;
    }

    
    if (virt->not_in_tree_list) {
        
        virt->not_in_tree_list   = H5MM_xfree(virt->not_in_tree_list);
        virt->not_in_tree_nused  = 0;
        virt->not_in_tree_nalloc = 0;
    }

    
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_reset_layout(H5O_layout_t *layout)
{
    H5O_storage_virtual_t *virt      = &layout->storage.u.virt;
    herr_t                 ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(layout);
    assert(layout->type == H5D_VIRTUAL);

    
    if (H5D__virtual_free_layout_mappings(virt) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release VDS mappings");

    
    if (virt->source_fapl >= 0) {
        if (H5I_dec_ref(virt->source_fapl) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't close source fapl");
        virt->source_fapl = -1;
    }
    if (virt->source_dapl >= 0) {
        if (H5I_dec_ref(virt->source_dapl) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't close source dapl");
        virt->source_dapl = -1;
    }

    
    virt->init = false;

    
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_copy(H5F_t *f_dst, H5O_layout_t *layout_dst)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

#ifdef NOT_YET
    
    if (f_dst == f_src) {
        
        if ((heap_rc = H5HG_link(f_dst, (H5HG_t *)&(layout_dst->u.virt.serial_list_hobjid), 1)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTMODIFY, FAIL, "unable to adjust global heap reference count");
    } 
    else
#endif 
    {
        
        layout_dst->storage.u.virt.serial_list_hobjid.addr = HADDR_UNDEF;
        layout_dst->storage.u.virt.serial_list_hobjid.idx  = (size_t)0;

        
        if (H5D__virtual_store_layout(f_dst, layout_dst) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to store VDS info");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_delete(H5F_t *f, H5O_storage_t *storage)
{
#ifdef NOT_YET
    int heap_rc;                
#endif                          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(storage);
    assert(storage->type == H5D_VIRTUAL);

    
    if (storage->u.virt.serial_list_hobjid.addr != HADDR_UNDEF) {
#ifdef NOT_YET
        
        if ((heap_rc = H5HG_link(f, (H5HG_t *)&(storage->u.virt.serial_list_hobjid), -1)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTMODIFY, FAIL, "unable to adjust global heap reference count");
        if (heap_rc == 0)
#endif 
            
            if (H5HG_remove(f, (H5HG_t *)&(storage->u.virt.serial_list_hobjid)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTREMOVE, FAIL, "unable to remove heap object");
    } 

    
    storage->u.virt.serial_list_hobjid.addr = HADDR_UNDEF;
    storage->u.virt.serial_list_hobjid.idx  = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_open_source_dset(const H5D_t *vdset, H5O_storage_virtual_ent_t *virtual_ent,
                              H5O_storage_virtual_srcdset_t *source_dset)
{
    H5F_t *src_file      = NULL;    
    bool   src_file_open = false;   
    herr_t ret_value     = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(vdset);
    assert(source_dset);
    assert(!source_dset->dset);
    assert(source_dset->file_name);
    assert(source_dset->dset_name);

    
    if (strcmp(source_dset->file_name, ".") != 0) {
        unsigned intent; 

        
        intent = H5F_INTENT(vdset->oloc.file);

        
        if (H5F_prefix_open_file(true, &src_file, vdset->oloc.file, H5F_PREFIX_VDS, vdset->shared->vds_prefix,
                                 source_dset->file_name, intent,
                                 vdset->shared->layout.storage.u.virt.source_fapl) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENFILE, FAIL, "can't try opening file");

        
        if (src_file)
            src_file_open = true;
    } 
    else
        
        src_file = vdset->oloc.file;

    if (src_file) {
        H5G_loc_t src_root_loc; 
        bool      exists = false;

        
        if (NULL == (src_root_loc.oloc = H5G_oloc(H5G_rootof(src_file))))
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "unable to get object location for root group");
        if (NULL == (src_root_loc.path = H5G_nameof(H5G_rootof(src_file))))
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "unable to get path for root group");

        
        if (H5G_loc_exists(&src_root_loc, source_dset->dset_name, &exists ) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTFIND, FAIL, "can't check object's existence");

        
        if (exists) {
            
            if (NULL ==
                (source_dset->dset = H5D__open_name(&src_root_loc, source_dset->dset_name,
                                                    vdset->shared->layout.storage.u.virt.source_dapl)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open source dataset");

            
            source_dset->dset_exists = true;

            
            if (virtual_ent->source_space_status != H5O_VIRTUAL_STATUS_CORRECT) {
                if (H5S_extent_copy(virtual_ent->source_select, source_dset->dset->shared->space) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy source dataspace extent");
                virtual_ent->source_space_status = H5O_VIRTUAL_STATUS_CORRECT;
            } 
        }     
        else
            
            source_dset->dset_exists = false;
    } 

done:
    
    if (src_file_open)
        if (H5F_efc_close(vdset->oloc.file, src_file) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEFILE, FAIL, "can't close source file");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_reset_source_dset(H5O_storage_virtual_ent_t     *virtual_ent,
                               H5O_storage_virtual_srcdset_t *source_dset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(source_dset);

    
    if (source_dset->dset) {
        if (H5D_close(source_dset->dset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to close source dataset");
        source_dset->dset = NULL;
    } 

    
    if (virtual_ent->parsed_source_file_name &&
        (source_dset->file_name != virtual_ent->parsed_source_file_name->name_segment))
        source_dset->file_name = (char *)H5MM_xfree(source_dset->file_name);
    else
        assert((source_dset->file_name == virtual_ent->source_file_name) ||
               (virtual_ent->parsed_source_file_name &&
                (source_dset->file_name == virtual_ent->parsed_source_file_name->name_segment)) ||
               !source_dset->file_name);

    
    if (virtual_ent->parsed_source_dset_name &&
        (source_dset->dset_name != virtual_ent->parsed_source_dset_name->name_segment))
        source_dset->dset_name = (char *)H5MM_xfree(source_dset->dset_name);
    else
        assert((source_dset->dset_name == virtual_ent->source_dset_name) ||
               (virtual_ent->parsed_source_dset_name &&
                (source_dset->dset_name == virtual_ent->parsed_source_dset_name->name_segment)) ||
               !source_dset->dset_name);

    
    if (source_dset->clipped_virtual_select) {
        if (source_dset->clipped_virtual_select != source_dset->virtual_select)
            if (H5S_close(source_dset->clipped_virtual_select) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release clipped virtual selection");
        source_dset->clipped_virtual_select = NULL;
    } 

    
    if (source_dset->virtual_select) {
        if (H5S_close(source_dset->virtual_select) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release virtual selection");
        source_dset->virtual_select = NULL;
    } 

    
    if (source_dset->clipped_source_select) {
        if (source_dset->clipped_source_select != virtual_ent->source_select)
            if (H5S_close(source_dset->clipped_source_select) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release clipped source selection");
        source_dset->clipped_source_select = NULL;
    } 

    
    assert(!source_dset->projected_mem_space);

    
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_str_append(const char *src, size_t src_len, char **p, char **buf, size_t *buf_size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(src);
    assert(src_len > 0);
    assert(p);
    assert(buf);
    assert(*p >= *buf);
    assert(buf_size);

    
    if (!*buf) {
        assert(!*p);
        assert(*buf_size == 0);

        
        if (NULL == (*buf = (char *)H5MM_malloc(src_len + (size_t)1)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to allocate name segment struct");
        *buf_size = src_len + (size_t)1;
        *p        = *buf;
    } 
    else {
        size_t p_offset = (size_t)(*p - *buf); 

        
        if ((p_offset + src_len + (size_t)1) > *buf_size) {
            char  *tmp_buf;
            size_t tmp_buf_size;

            
            tmp_buf_size = MAX(p_offset + src_len + (size_t)1, *buf_size * (size_t)2);

            
            if (NULL == (tmp_buf = (char *)H5MM_realloc(*buf, tmp_buf_size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to reallocate name segment buffer");
            *buf      = tmp_buf;
            *buf_size = tmp_buf_size;
            *p        = *buf + p_offset;
        } 
    }     

    
    H5MM_memcpy(*p, src, src_len);

    
    *p += src_len;

    
    **p = '\0';

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_virtual_parse_source_name(const char *source_name, H5O_storage_virtual_name_seg_t **parsed_name,
                              size_t *static_strlen, size_t *nsubs)
{
    H5O_storage_virtual_name_seg_t  *tmp_parsed_name   = NULL;
    H5O_storage_virtual_name_seg_t **tmp_parsed_name_p = &tmp_parsed_name;
    size_t                           tmp_static_strlen;
    size_t                           tmp_strlen;
    size_t                           tmp_nsubs = 0;
    const char                      *p;
    const char                      *pct;
    char                            *name_seg_p    = NULL;
    size_t                           name_seg_size = 0;
    herr_t                           ret_value     = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(source_name);
    assert(parsed_name);
    assert(static_strlen);
    assert(nsubs);

    
    p                 = source_name;
    tmp_static_strlen = tmp_strlen = strlen(source_name);

    
    
    while ((pct = strchr(p, '%'))) {
        assert(pct >= p);

        
        if (!*tmp_parsed_name_p)
            if (NULL == (*tmp_parsed_name_p = H5FL_CALLOC(H5O_storage_virtual_name_seg_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to allocate name segment struct");

        
        if (pct[1] == 'b') {
            
            if (pct != p)
                
                if (H5D__virtual_str_append(p, (size_t)(pct - p), &name_seg_p,
                                            &(*tmp_parsed_name_p)->name_segment, &name_seg_size) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to append name segment");

            
            tmp_parsed_name_p = &(*tmp_parsed_name_p)->next;
            tmp_static_strlen -= 2;
            tmp_nsubs++;
            name_seg_p    = NULL;
            name_seg_size = 0;
        } 
        else if (pct[1] == '%') {
            
            if (H5D__virtual_str_append(p, (size_t)(pct - p) + (size_t)1, &name_seg_p,
                                        &(*tmp_parsed_name_p)->name_segment, &name_seg_size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to append name segment");

            
            tmp_static_strlen -= 1;
        } 
        else
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid format specifier");

        p = pct + 2;
    } 

    
    if (tmp_parsed_name) {
        assert(p >= source_name);
        if (*p == '\0')
            assert((size_t)(p - source_name) == tmp_strlen);
        else {
            assert((size_t)(p - source_name) < tmp_strlen);

            
            if (!*tmp_parsed_name_p)
                if (NULL == (*tmp_parsed_name_p = H5FL_CALLOC(H5O_storage_virtual_name_seg_t)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to allocate name segment struct");

            
            if (H5D__virtual_str_append(p, tmp_strlen - (size_t)(p - source_name), &name_seg_p,
                                        &(*tmp_parsed_name_p)->name_segment, &name_seg_size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to append name segment");
        } 
    }     

    
    *parsed_name    = tmp_parsed_name;
    tmp_parsed_name = NULL;
    *static_strlen  = tmp_static_strlen;
    *nsubs          = tmp_nsubs;

done:
    if (tmp_parsed_name) {
        assert(ret_value < 0);
        H5D_virtual_free_parsed_name(tmp_parsed_name);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_copy_parsed_name(H5O_storage_virtual_name_seg_t **dst, H5O_storage_virtual_name_seg_t *src)
{
    H5O_storage_virtual_name_seg_t  *tmp_dst   = NULL;
    H5O_storage_virtual_name_seg_t  *p_src     = src;
    H5O_storage_virtual_name_seg_t **p_dst     = &tmp_dst;
    herr_t                           ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dst);

    
    while (p_src) {
        
        if (NULL == (*p_dst = H5FL_CALLOC(H5O_storage_virtual_name_seg_t)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to allocate name segment struct");

        
        if (p_src->name_segment) {
            if (NULL == ((*p_dst)->name_segment = H5MM_strdup(p_src->name_segment)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to duplicate name segment");
        } 

        
        p_src = p_src->next;
        p_dst = &(*p_dst)->next;
    } 

    
    *dst    = tmp_dst;
    tmp_dst = NULL;

done:
    if (tmp_dst) {
        assert(ret_value < 0);
        H5D_virtual_free_parsed_name(tmp_dst);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_virtual_free_parsed_name(H5O_storage_virtual_name_seg_t *name_seg)
{
    H5O_storage_virtual_name_seg_t *next_seg;
    herr_t                          ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    while (name_seg) {
        (void)H5MM_xfree(name_seg->name_segment);
        next_seg = name_seg->next;
        (void)H5FL_FREE(H5O_storage_virtual_name_seg_t, name_seg);
        name_seg = next_seg;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_build_source_name(char *source_name, const H5O_storage_virtual_name_seg_t *parsed_name,
                               size_t static_strlen, size_t nsubs, hsize_t blockno, char **built_name)
{
    char  *tmp_name  = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(source_name);
    assert(built_name);

    
    if (nsubs == 0) {
        if (parsed_name)
            *built_name = parsed_name->name_segment;
        else
            *built_name = source_name;
    } 
    else {
        const H5O_storage_virtual_name_seg_t *name_seg = parsed_name;
        char                                 *p;
        hsize_t                               blockno_down = blockno;
        size_t                                blockno_len  = 1;
        size_t                                name_len;
        size_t                                name_len_rem;
        size_t                                seg_len;
        size_t                                nsubs_rem = nsubs;

        assert(parsed_name);

        
        do {
            blockno_down /= (hsize_t)10;
            if (blockno_down == 0)
                break;
            blockno_len++;
        } while (1);

        
        name_len_rem = name_len = static_strlen + (nsubs * blockno_len) + (size_t)1;

        
        if (NULL == (tmp_name = (char *)H5MM_malloc(name_len)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to allocate name buffer");
        p = tmp_name;

        
        do {
            
            if (name_seg->name_segment) {
                seg_len = strlen(name_seg->name_segment);
                assert(seg_len > 0);
                assert(seg_len < name_len_rem);
                strncpy(p, name_seg->name_segment, name_len_rem);
                name_len_rem -= seg_len;
                p += seg_len;
            } 

            
            if (nsubs_rem > 0) {
                assert(blockno_len < name_len_rem);
                if (snprintf(p, name_len_rem, "%llu", (long long unsigned)blockno) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to write block number to string");
                name_len_rem -= blockno_len;
                p += blockno_len;
                nsubs_rem--;
            } 

            
            name_seg = name_seg->next;
        } while (name_seg);

        
        *built_name = tmp_name;
        tmp_name    = NULL;
    } 

done:
    if (tmp_name) {
        assert(ret_value < 0);
        H5MM_free(tmp_name);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_set_extent_unlim(const H5D_t *dset)
{
    H5O_storage_virtual_t *storage;
    hsize_t                new_dims[H5S_MAX_RANK];
    hsize_t                curr_dims[H5S_MAX_RANK];
    hsize_t                clip_size;
    int                    rank;
    bool                   changed = false; 
    size_t                 i, j;
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->shared->layout.storage.type == H5D_VIRTUAL);
    storage = &dset->shared->layout.storage.u.virt;
    assert((storage->view == H5D_VDS_FIRST_MISSING) || (storage->view == H5D_VDS_LAST_AVAILABLE));

    
    if ((rank = H5S_GET_EXTENT_NDIMS(dset->shared->space)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get number of dimensions");

    
    for (i = 0; i < (size_t)rank; i++)
        new_dims[i] = HSIZE_UNDEF;

    
    for (i = 0; i < storage->list_nused; i++)
        
        if (storage->list[i].unlim_dim_virtual >= 0) {
            
            if (storage->list[i].unlim_dim_source >= 0) {
                
                
                if (!storage->list[i].source_dset.dset)
                    if (H5D__virtual_open_source_dset(dset, &storage->list[i],
                                                      &storage->list[i].source_dset) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open source dataset");

                
                if (storage->list[i].source_dset.dset) {
                    
                    if (H5S_extent_copy(storage->list[i].source_select,
                                        storage->list[i].source_dset.dset->shared->space) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy source dataspace extent");

                    
                    if (H5S_get_simple_extent_dims(storage->list[i].source_select, curr_dims, NULL) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get source space dimensions");

                    
                    if (curr_dims[storage->list[i].unlim_dim_source] == storage->list[i].unlim_extent_source)
                        
                        clip_size = storage->list[i].clip_size_virtual;
                    else {
                        
                        clip_size = H5S_hyper_get_clip_extent_match(
                            storage->list[i].source_dset.virtual_select, storage->list[i].source_select,
                            curr_dims[storage->list[i].unlim_dim_source],
                            storage->view == H5D_VDS_FIRST_MISSING);

                        
                        if (storage->view == H5D_VDS_LAST_AVAILABLE) {
                            if (clip_size != storage->list[i].clip_size_virtual) {
                                
                                if (storage->list[i].source_dset.clipped_virtual_select) {
                                    assert(storage->list[i].source_dset.clipped_virtual_select !=
                                           storage->list[i].source_dset.virtual_select);
                                    if (H5S_close(storage->list[i].source_dset.clipped_virtual_select) < 0)
                                        HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                                    "unable to release clipped virtual dataspace");
                                } 

                                
                                if (NULL == (storage->list[i].source_dset.clipped_virtual_select = H5S_copy(
                                                 storage->list[i].source_dset.virtual_select, false, true)))
                                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL,
                                                "unable to copy virtual selection");

                                
                                if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_virtual_select,
                                                         clip_size))
                                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                                "failed to clip unlimited selection");
                            } 

                            
                            if (storage->list[i].source_dset.clipped_source_select) {
                                assert(storage->list[i].source_dset.clipped_source_select !=
                                       storage->list[i].source_select);
                                if (H5S_close(storage->list[i].source_dset.clipped_source_select) < 0)
                                    HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                                "unable to release clipped source dataspace");
                            } 

                            
                            if (NULL == (storage->list[i].source_dset.clipped_source_select =
                                             H5S_copy(storage->list[i].source_select, false, true)))
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL,
                                            "unable to copy source selection");

                            
                            if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_source_select,
                                                     curr_dims[storage->list[i].unlim_dim_source]))
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                            "failed to clip unlimited selection");
                        } 

                        
                        storage->list[i].unlim_extent_source = curr_dims[storage->list[i].unlim_dim_source];
                        storage->list[i].clip_size_virtual   = clip_size;
                    } 
                }     
                else
                    clip_size = 0;
            } 
            else {
                
                hsize_t first_missing =
                    0; 

                
                assert(storage->printf_gap != HSIZE_UNDEF);
                for (j = 0; j <= (storage->printf_gap + first_missing); j++) {
                    
                    if (j >= (hsize_t)storage->list[i].sub_dset_nalloc) {
                        if (storage->list[i].sub_dset_nalloc == 0) {
                            
                            if (NULL ==
                                (storage->list[i].sub_dset = (H5O_storage_virtual_srcdset_t *)H5MM_calloc(
                                     H5D_VIRTUAL_DEF_SUB_DSET_SIZE * sizeof(H5O_storage_virtual_srcdset_t))))
                                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                            "unable to allocate sub dataset array");
                            storage->list[i].sub_dset_nalloc = H5D_VIRTUAL_DEF_SUB_DSET_SIZE;
                        } 
                        else {
                            H5O_storage_virtual_srcdset_t *tmp_sub_dset;

                            
                            if (NULL ==
                                (tmp_sub_dset = (H5O_storage_virtual_srcdset_t *)H5MM_realloc(
                                     storage->list[i].sub_dset, 2 * storage->list[i].sub_dset_nalloc *
                                                                    sizeof(H5O_storage_virtual_srcdset_t))))
                                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                            "unable to extend sub dataset array");
                            storage->list[i].sub_dset = tmp_sub_dset;

                            
                            (void)memset(&storage->list[i].sub_dset[storage->list[i].sub_dset_nalloc], 0,
                                         storage->list[i].sub_dset_nalloc *
                                             sizeof(H5O_storage_virtual_srcdset_t));

                            
                            storage->list[i].sub_dset_nalloc *= 2;
                        } 
                    }     

                    
                    if (storage->list[i].sub_dset[j].dset_exists)
                        first_missing = j + 1;
                    else {
                        
                        if (!storage->list[i].sub_dset[j].file_name)
                            if (H5D__virtual_build_source_name(storage->list[i].source_file_name,
                                                               storage->list[i].parsed_source_file_name,
                                                               storage->list[i].psfn_static_strlen,
                                                               storage->list[i].psfn_nsubs, j,
                                                               &storage->list[i].sub_dset[j].file_name) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                                            "unable to build source file name");

                        
                        if (!storage->list[i].sub_dset[j].dset_name)
                            if (H5D__virtual_build_source_name(storage->list[i].source_dset_name,
                                                               storage->list[i].parsed_source_dset_name,
                                                               storage->list[i].psdn_static_strlen,
                                                               storage->list[i].psdn_nsubs, j,
                                                               &storage->list[i].sub_dset[j].dset_name) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                                            "unable to build source dataset name");

                        
                        if (!storage->list[i].sub_dset[j].virtual_select)
                            if (NULL ==
                                (storage->list[i].sub_dset[j].virtual_select = H5S_hyper_get_unlim_block(
                                     storage->list[i].source_dset.virtual_select, j)))
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                                            "unable to get block in unlimited selection");

                        
                        if (!storage->list[i].sub_dset[j].clipped_source_select)
                            storage->list[i].sub_dset[j].clipped_source_select =
                                storage->list[i].source_select;
                        if (!storage->list[i].sub_dset[j].clipped_virtual_select)
                            storage->list[i].sub_dset[j].clipped_virtual_select =
                                storage->list[i].sub_dset[j].virtual_select;

                        
                        if (H5D__virtual_open_source_dset(dset, &storage->list[i],
                                                          &storage->list[i].sub_dset[j]) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open source dataset");

                        if (storage->list[i].sub_dset[j].dset) {
                            
                            first_missing = j + 1;

                            
                            if (H5D_close(storage->list[i].sub_dset[j].dset) < 0)
                                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                            "unable to close source dataset");
                            storage->list[i].sub_dset[j].dset = NULL;
                        } 
                    }     
                }         

                
                if ((first_missing == (hsize_t)storage->list[i].sub_dset_nused) &&
                    (storage->list[i].clip_size_virtual != HSIZE_UNDEF))
                    
                    clip_size = storage->list[i].clip_size_virtual;
                else {
                    
                    if (first_missing == 0)
                        
                        clip_size = (hsize_t)0;
                    else {
                        hsize_t bounds_start[H5S_MAX_RANK];
                        hsize_t bounds_end[H5S_MAX_RANK];

                        
                        if (storage->view == H5D_VDS_LAST_AVAILABLE) {
                            
                            if (H5S_SELECT_BOUNDS(
                                    storage->list[i].sub_dset[first_missing - (hsize_t)1].virtual_select,
                                    bounds_start, bounds_end) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection bounds");

                            
                            clip_size = bounds_end[storage->list[i].unlim_dim_virtual] + (hsize_t)1;
                        } 
                        else {
                            
                            if (H5S_SELECT_BOUNDS(storage->list[i].sub_dset[first_missing].virtual_select,
                                                  bounds_start, bounds_end) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection bounds");

                            
                            clip_size = bounds_start[storage->list[i].unlim_dim_virtual];
                        } 
                    }     

                    
                    storage->list[i].sub_dset_nused    = (size_t)first_missing;
                    storage->list[i].clip_size_virtual = clip_size;
                } 
            }     

            
            if ((new_dims[storage->list[i].unlim_dim_virtual] == HSIZE_UNDEF) ||
                (storage->view == H5D_VDS_FIRST_MISSING
                     ? (clip_size < (hsize_t)new_dims[storage->list[i].unlim_dim_virtual])
                     : (clip_size > (hsize_t)new_dims[storage->list[i].unlim_dim_virtual])))
                new_dims[storage->list[i].unlim_dim_virtual] = clip_size;
        } 

    
    if (H5S_get_simple_extent_dims(dset->shared->space, curr_dims, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get VDS dimensions");

    
    for (i = 0; i < (size_t)rank; i++) {
        if (new_dims[i] == HSIZE_UNDEF)
            new_dims[i] = curr_dims[i];
        else if (new_dims[i] < storage->min_dims[i])
            new_dims[i] = storage->min_dims[i];
        if (new_dims[i] != curr_dims[i])
            changed = true;
    } 

    
    if (changed) {
        
        if (H5S_set_extent(dset->shared->space, new_dims) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");

        
        if (H5F_INTENT(dset->oloc.file) & H5F_ACC_RDWR)
            if (H5D__mark(dset, H5D_MARK_SPACE) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to mark dataspace as dirty");
    } 

    
    if (changed || (!storage->init && (storage->view == H5D_VDS_FIRST_MISSING))) {
        
        for (i = 0; i < storage->list_nused; i++) {
            
            if ((storage->list[i].unlim_dim_virtual >= 0) && (storage->view == H5D_VDS_FIRST_MISSING) &&
                (new_dims[storage->list[i].unlim_dim_virtual] != storage->list[i].unlim_extent_virtual)) {
                
                if (storage->list[i].unlim_dim_source >= 0) {
                    
                    
                    if (storage->list[i].source_dset.clipped_virtual_select) {
                        assert(storage->list[i].source_dset.clipped_virtual_select !=
                               storage->list[i].source_dset.virtual_select);
                        if (H5S_close(storage->list[i].source_dset.clipped_virtual_select) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                        "unable to release clipped virtual dataspace");
                    } 

                    
                    if (NULL == (storage->list[i].source_dset.clipped_virtual_select =
                                     H5S_copy(storage->list[i].source_dset.virtual_select, false, true)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy virtual selection");

                    
                    if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_virtual_select,
                                             new_dims[storage->list[i].unlim_dim_source]))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "failed to clip unlimited selection");

                    
                    clip_size =
                        H5S_hyper_get_clip_extent(storage->list[i].source_select,
                                                  storage->list[i].source_dset.clipped_virtual_select, false);

                    
                    if (clip_size != storage->list[i].clip_size_source) {
                        
                        if (storage->list[i].source_dset.clipped_source_select) {
                            assert(storage->list[i].source_dset.clipped_source_select !=
                                   storage->list[i].source_select);
                            if (H5S_close(storage->list[i].source_dset.clipped_source_select) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                            "unable to release clipped source dataspace");
                        } 

                        
                        if (NULL == (storage->list[i].source_dset.clipped_source_select =
                                         H5S_copy(storage->list[i].source_select, false, true)))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy source selection");

                        
                        if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_source_select,
                                                 clip_size))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                        "failed to clip unlimited selection");

                        
                        storage->list[i].clip_size_source = clip_size;
                    } 
                }     
                else {
                    
                    hsize_t first_inc_block;
                    bool    partial_block;

                    
                    first_inc_block = H5S_hyper_get_first_inc_block(
                        storage->list[i].source_dset.virtual_select,
                        new_dims[storage->list[i].unlim_dim_virtual], &partial_block);

                    
                    for (j = 0; j < storage->list[i].sub_dset_nalloc; j++) {
                        
                        if (storage->list[i].sub_dset[j].clipped_source_select !=
                            storage->list[i].source_select) {
                            if (storage->list[i].sub_dset[j].clipped_source_select)
                                if (H5S_close(storage->list[i].sub_dset[j].clipped_source_select) < 0)
                                    HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                                "unable to release clipped source dataspace");

                            
                            storage->list[i].sub_dset[j].clipped_source_select =
                                storage->list[i].source_select;
                        } 

                        
                        if (storage->list[i].sub_dset[j].clipped_virtual_select !=
                            storage->list[i].sub_dset[j].virtual_select) {
                            if (storage->list[i].sub_dset[j].clipped_virtual_select)
                                if (H5S_close(storage->list[i].sub_dset[j].clipped_virtual_select) < 0)
                                    HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                                "unable to release clipped virtual dataspace");

                            
                            storage->list[i].sub_dset[j].clipped_virtual_select =
                                storage->list[i].sub_dset[j].virtual_select;
                        } 

                        
                        if (j >= (size_t)first_inc_block) {
                            
                            storage->list[i].sub_dset[j].clipped_source_select  = NULL;
                            storage->list[i].sub_dset[j].clipped_virtual_select = NULL;
                        } 
                    }     
                }         

                
                storage->list[i].unlim_extent_virtual = new_dims[storage->list[i].unlim_dim_virtual];
            } 

            
            if (H5S_set_extent(storage->list[i].source_dset.virtual_select, new_dims) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");
            if ((storage->list[i].source_dset.clipped_virtual_select !=
                 storage->list[i].source_dset.virtual_select) &&
                storage->list[i].source_dset.clipped_virtual_select)
                if (H5S_set_extent(storage->list[i].source_dset.clipped_virtual_select, new_dims) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");

            
            for (j = 0; j < storage->list[i].sub_dset_nalloc; j++)
                if (storage->list[i].sub_dset[j].virtual_select) {
                    if (H5S_set_extent(storage->list[i].sub_dset[j].virtual_select, new_dims) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");
                    if ((storage->list[i].sub_dset[j].clipped_virtual_select !=
                         storage->list[i].sub_dset[j].virtual_select) &&
                        storage->list[i].sub_dset[j].clipped_virtual_select)
                        if (H5S_set_extent(storage->list[i].sub_dset[j].clipped_virtual_select, new_dims) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                                        "unable to modify size of dataspace");
                } 
                else
                    assert(!storage->list[i].sub_dset[j].clipped_virtual_select);
        } 
    }     

    
    storage->init = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_init_all(const H5D_t *dset)
{
    H5O_storage_virtual_t *storage;
    hsize_t                virtual_dims[H5S_MAX_RANK];
    hsize_t                source_dims[H5S_MAX_RANK];
    hsize_t                clip_size;
    size_t                 i, j;
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->shared->layout.storage.type == H5D_VIRTUAL);
    storage = &dset->shared->layout.storage.u.virt;
    assert((storage->view == H5D_VDS_FIRST_MISSING) || (storage->view == H5D_VDS_LAST_AVAILABLE));

    
    if (H5S_get_simple_extent_dims(dset->shared->space, virtual_dims, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get VDS dimensions");

    
    for (i = 0; i < storage->list_nused; i++)
        
        if (storage->list[i].unlim_dim_virtual >= 0) {
            
            if (storage->list[i].unlim_dim_source >= 0) {
                
                
                if (!storage->list[i].source_dset.dset)
                    if (H5D__virtual_open_source_dset(dset, &storage->list[i],
                                                      &storage->list[i].source_dset) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open source dataset");

                
                if (storage->list[i].source_dset.dset) {
                    
                    if (H5S_extent_copy(storage->list[i].source_select,
                                        storage->list[i].source_dset.dset->shared->space) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy source dataspace extent");

                    
                    if (H5S_get_simple_extent_dims(storage->list[i].source_select, source_dims, NULL) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get source space dimensions");

                    
                    clip_size = H5S_hyper_get_clip_extent_match(
                        storage->list[i].source_select, storage->list[i].source_dset.virtual_select,
                        virtual_dims[storage->list[i].unlim_dim_virtual], false);

                    
                    if (storage->list[i].source_dset.clipped_virtual_select) {
                        assert(storage->list[i].source_dset.clipped_virtual_select !=
                               storage->list[i].source_dset.virtual_select);
                        if (H5S_close(storage->list[i].source_dset.clipped_virtual_select) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                        "unable to release clipped virtual dataspace");
                    } 

                    
                    if (NULL == (storage->list[i].source_dset.clipped_virtual_select =
                                     H5S_copy(storage->list[i].source_dset.virtual_select, false, true)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy virtual selection");

                    
                    if (storage->list[i].source_dset.clipped_source_select) {
                        assert(storage->list[i].source_dset.clipped_source_select !=
                               storage->list[i].source_select);
                        if (H5S_close(storage->list[i].source_dset.clipped_source_select) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                        "unable to release clipped source dataspace");
                    } 

                    
                    if (NULL == (storage->list[i].source_dset.clipped_source_select =
                                     H5S_copy(storage->list[i].source_select, false, true)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy source selection");

                    
                    if (clip_size <= source_dims[storage->list[i].unlim_dim_source]) {
                        
                        if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_virtual_select,
                                                 virtual_dims[storage->list[i].unlim_dim_virtual]))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                        "failed to clip unlimited selection");

                        
                        if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_source_select,
                                                 clip_size))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                        "failed to clip unlimited selection");
                    } 
                    else {
                        
                        clip_size = H5S_hyper_get_clip_extent_match(
                            storage->list[i].source_dset.virtual_select, storage->list[i].source_select,
                            source_dims[storage->list[i].unlim_dim_source], false);

                        
                        if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_virtual_select,
                                                 clip_size))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                        "failed to clip unlimited selection");

                        
                        if (H5S_hyper_clip_unlim(storage->list[i].source_dset.clipped_source_select,
                                                 source_dims[storage->list[i].unlim_dim_source]))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                        "failed to clip unlimited selection");
                    } 
                }     
                else {
                    assert(!storage->list[i].source_dset.clipped_virtual_select);
                    assert(!storage->list[i].source_dset.clipped_source_select);
                } 
            }     
            else {
                
                size_t sub_dset_max;
                bool   partial_block;

                
                sub_dset_max = (size_t)H5S_hyper_get_first_inc_block(
                    storage->list[i].source_dset.virtual_select,
                    virtual_dims[storage->list[i].unlim_dim_virtual], &partial_block);
                if (partial_block)
                    sub_dset_max++;

                
                if (!storage->list[i].sub_dset) {
                    
                    if (NULL == (storage->list[i].sub_dset = (H5O_storage_virtual_srcdset_t *)H5MM_calloc(
                                     sub_dset_max * sizeof(H5O_storage_virtual_srcdset_t))))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                    "unable to allocate sub dataset array");

                    
                    storage->list[i].sub_dset_nalloc = sub_dset_max;
                } 
                else if (sub_dset_max > storage->list[i].sub_dset_nalloc) {
                    H5O_storage_virtual_srcdset_t *tmp_sub_dset;

                    
                    if (NULL == (tmp_sub_dset = (H5O_storage_virtual_srcdset_t *)H5MM_realloc(
                                     storage->list[i].sub_dset,
                                     sub_dset_max * sizeof(H5O_storage_virtual_srcdset_t))))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "unable to extend sub dataset array");
                    storage->list[i].sub_dset = tmp_sub_dset;

                    
                    (void)memset(&storage->list[i].sub_dset[storage->list[i].sub_dset_nalloc], 0,
                                 (sub_dset_max - storage->list[i].sub_dset_nalloc) *
                                     sizeof(H5O_storage_virtual_srcdset_t));

                    
                    storage->list[i].sub_dset_nalloc = sub_dset_max;
                } 

                
                for (j = 0; j < sub_dset_max; j++) {
                    
                    if (!storage->list[i].sub_dset[j].file_name)
                        if (H5D__virtual_build_source_name(
                                storage->list[i].source_file_name, storage->list[i].parsed_source_file_name,
                                storage->list[i].psfn_static_strlen, storage->list[i].psfn_nsubs, j,
                                &storage->list[i].sub_dset[j].file_name) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to build source file name");

                    
                    if (!storage->list[i].sub_dset[j].dset_name)
                        if (H5D__virtual_build_source_name(
                                storage->list[i].source_dset_name, storage->list[i].parsed_source_dset_name,
                                storage->list[i].psdn_static_strlen, storage->list[i].psdn_nsubs, j,
                                &storage->list[i].sub_dset[j].dset_name) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                                        "unable to build source dataset name");

                    
                    if (!storage->list[i].sub_dset[j].virtual_select)
                        if (NULL == (storage->list[i].sub_dset[j].virtual_select = H5S_hyper_get_unlim_block(
                                         storage->list[i].source_dset.virtual_select, j)))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                                        "unable to get block in unlimited selection");

                    
                    if (storage->list[i].sub_dset[j].clipped_source_select !=
                        storage->list[i].source_select) {
                        if (storage->list[i].sub_dset[j].clipped_source_select)
                            if (H5S_close(storage->list[i].sub_dset[j].clipped_source_select) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                            "unable to release clipped source dataspace");

                        
                        storage->list[i].sub_dset[j].clipped_source_select = storage->list[i].source_select;
                    } 

                    
                    if (storage->list[i].sub_dset[j].clipped_virtual_select !=
                        storage->list[i].sub_dset[j].virtual_select) {
                        if (storage->list[i].sub_dset[j].clipped_virtual_select)
                            if (H5S_close(storage->list[i].sub_dset[j].clipped_virtual_select) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                            "unable to release clipped virtual dataspace");

                        
                        storage->list[i].sub_dset[j].clipped_virtual_select =
                            storage->list[i].sub_dset[j].virtual_select;
                    } 

                    
                    if ((j == (sub_dset_max - 1)) && partial_block) {
                        
                        storage->list[i].sub_dset[j].clipped_source_select  = NULL;
                        storage->list[i].sub_dset[j].clipped_virtual_select = NULL;
                    } 
                    
                } 

                
                storage->list[i].sub_dset_nused = sub_dset_max;
            } 
        }     
        else {
            
            assert(storage->list[i].source_dset.clipped_virtual_select);
            assert(storage->list[i].source_dset.clipped_source_select);
        } 

    
    storage->init = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_construct(H5F_t *f, H5D_t *dset)
{
    unsigned version;             
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(dset);
    assert(dset->shared);

    
    
    if (dset->shared->layout.version < H5O_LAYOUT_VERSION_4) {
        version = MAX(dset->shared->layout.version, H5O_LAYOUT_VERSION_4);

        
        if (version > H5O_layout_ver_bounds[H5F_HIGH_BOUND(f)])
            HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "layout version out of bounds");

        dset->shared->layout.version = version;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_init(H5F_t *f, H5D_t *dset, hid_t dapl_id, bool H5_ATTR_UNUSED open_op)
{
    H5O_storage_virtual_t *storage;                      
    H5P_genplist_t        *dapl;                         
    hssize_t               old_offset[H5O_LAYOUT_NDIMS]; 
    size_t                 i;                            
    herr_t                 ret_value = SUCCEED;          

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    storage = &dset->shared->layout.storage.u.virt;
    assert(storage->list || (storage->list_nused == 0));

    if (H5D__virtual_load_layout(f, &dset->shared->layout) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTLOAD, FAIL, "unable to load virtual layout information");

    
    if (H5D_virtual_check_min_dims(dset) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "virtual dataset dimensions not large enough to contain all limited dimensions in all "
                    "selections");

    
    for (i = 0; i < storage->list_nused; i++) {
        assert(storage->list[i].sub_dset_nalloc == 0);

        
        if (H5S_extent_copy(storage->list[i].source_dset.virtual_select, dset->shared->space) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy virtual dataspace extent");
        storage->list[i].virtual_space_status = H5O_VIRTUAL_STATUS_CORRECT;

        
        storage->list[i].source_space_status = H5O_VIRTUAL_STATUS_INVALID;

        
        if (H5S_hyper_normalize_offset(storage->list[i].source_dset.virtual_select, old_offset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADSELECT, FAIL, "unable to normalize dataspace by offset");
        if (H5S_hyper_normalize_offset(storage->list[i].source_select, old_offset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADSELECT, FAIL, "unable to normalize dataspace by offset");
    } 

    
    if (NULL == (dapl = (H5P_genplist_t *)H5I_object(dapl_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for dapl ID");

    
    if (H5P_get(dapl, H5D_ACS_VDS_VIEW_NAME, &storage->view) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get virtual view option");

    
    if (storage->view == H5D_VDS_LAST_AVAILABLE) {
        if (H5P_get(dapl, H5D_ACS_VDS_PRINTF_GAP_NAME, &storage->printf_gap) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get virtual Rprintf gap");
    } 
    else
        storage->printf_gap = (hsize_t)0;

    
    if (storage->source_fapl <= 0) {
        H5P_genplist_t    *source_fapl  = NULL;           
        H5F_close_degree_t close_degree = H5F_CLOSE_WEAK; 

        if ((storage->source_fapl = H5F_get_access_plist(f, false)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get fapl");

        
        if (NULL == (source_fapl = (H5P_genplist_t *)H5I_object(storage->source_fapl)))
            HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, H5I_INVALID_HID, "not a property list");

        
        if (H5P_set(source_fapl, H5F_ACS_CLOSE_DEGREE_NAME, &close_degree) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set file close degree");
    } 
#ifndef NDEBUG
    else {
        H5P_genplist_t    *source_fapl = NULL; 
        H5F_close_degree_t close_degree;       

        
        if (NULL == (source_fapl = (H5P_genplist_t *)H5I_object(storage->source_fapl)))
            HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, H5I_INVALID_HID, "not a property list");

        
        if (H5P_get(source_fapl, H5F_ACS_CLOSE_DEGREE_NAME, &close_degree) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get file close degree");

        assert(close_degree == H5F_CLOSE_WEAK);
    }  
#endif 

    
    if (storage->source_dapl <= 0)
        if ((storage->source_dapl = H5P_copy_plist(dapl, false)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy dapl");

    
    storage->init = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static bool
H5D__virtual_is_space_alloc(const H5O_storage_t H5_ATTR_UNUSED *storage)
{
    bool ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = true;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static bool
H5D__virtual_is_data_cached(const H5D_shared_t *shared_dset)
{
    const H5O_storage_virtual_t *storage;           
    size_t                       i, j;              
    bool                         ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(shared_dset);
    storage = &shared_dset->layout.storage.u.virt;

    
    for (i = 0; i < storage->list_nused; i++)
        
        if (storage->list[i].psfn_nsubs || storage->list[i].psdn_nsubs) {
            
            for (j = storage->list[i].sub_dset_io_start; j < storage->list[i].sub_dset_io_end; j++)
                
                if (storage->list[i].sub_dset[j].dset &&
                    storage->list[i].sub_dset[j].dset->shared->layout.ops->is_data_cached &&
                    storage->list[i].sub_dset[j].dset->shared->layout.ops->is_data_cached(
                        storage->list[i].sub_dset[j].dset->shared))
                    HGOTO_DONE(true);
        } 
        else if (storage->list[i].source_dset.dset &&
                 storage->list[i].source_dset.dset->shared->layout.ops->is_data_cached &&
                 storage->list[i].source_dset.dset->shared->layout.ops->is_data_cached(
                     storage->list[i].source_dset.dset->shared))
            HGOTO_DONE(true);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t H5_ATTR_UNUSED *dinfo)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
    io_info->no_selection_io_cause |= H5D_SEL_IO_NOT_CONTIGUOUS_OR_CHUNKED_DATASET;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__virtual_pre_io_process_mapping(H5D_dset_io_info_t *dset_info, H5S_t *file_space, H5S_t *mem_space,
                                    hsize_t *tot_nelmts, H5O_storage_virtual_ent_t *curr_mapping)
{
    const H5D_t *dset = dset_info->dset;     
    hssize_t     select_nelmts;              
    hsize_t      bounds_start[H5S_MAX_RANK]; 
    hsize_t      bounds_end[H5S_MAX_RANK];   
    int          rank        = 0;
    bool         bounds_init = false; 
    size_t       j, k;                
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(curr_mapping->virtual_space_status == H5O_VIRTUAL_STATUS_CORRECT);

    
    if (curr_mapping->psfn_nsubs || curr_mapping->psdn_nsubs) {
        bool partial_block;

        assert(curr_mapping->unlim_dim_virtual >= 0);

        
        if (!bounds_init) {
            
            if ((rank = H5S_GET_EXTENT_NDIMS(dset->shared->space)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get number of dimensions");

            
            if (H5S_SELECT_BOUNDS(file_space, bounds_start, bounds_end) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection bounds");

            
            for (j = 0; j < (size_t)rank; j++)
                bounds_end[j]++;

            
            bounds_init = true;
        } 

        
        curr_mapping->sub_dset_io_start = (size_t)H5S_hyper_get_first_inc_block(
            curr_mapping->source_dset.virtual_select, bounds_start[curr_mapping->unlim_dim_virtual], NULL);

        
        curr_mapping->sub_dset_io_end = (size_t)H5S_hyper_get_first_inc_block(
            curr_mapping->source_dset.virtual_select, bounds_end[curr_mapping->unlim_dim_virtual],
            &partial_block);
        if (partial_block)
            curr_mapping->sub_dset_io_end++;
        if (curr_mapping->sub_dset_io_end > curr_mapping->sub_dset_nused)
            curr_mapping->sub_dset_io_end = curr_mapping->sub_dset_nused;

        
        for (j = curr_mapping->sub_dset_io_start; j < curr_mapping->sub_dset_io_end; j++) {
            
            if (!curr_mapping->sub_dset[j].clipped_virtual_select) {
                hsize_t start[H5S_MAX_RANK];
                
                assert((j == (curr_mapping->sub_dset_io_end - 1)) && partial_block);

                
                if (curr_mapping->source_space_status != H5O_VIRTUAL_STATUS_CORRECT) {
                    assert(!curr_mapping->sub_dset[j].dset);
                    if (H5D__virtual_open_source_dset(dset, curr_mapping, &curr_mapping->sub_dset[j]) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open source dataset");
                } 

                
                if (curr_mapping->source_space_status == H5O_VIRTUAL_STATUS_CORRECT) {
                    hsize_t tmp_dims[H5S_MAX_RANK];
                    hsize_t vbounds_end[H5S_MAX_RANK];

                    
                    if (H5S_SELECT_BOUNDS(curr_mapping->sub_dset[j].virtual_select, tmp_dims, vbounds_end) <
                        0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection bounds");

                    assert(bounds_init);

                    
                    for (k = 0; k < (size_t)rank; k++)
                        vbounds_end[k]++;

                    
                    if (H5S_set_extent(curr_mapping->sub_dset[j].virtual_select, vbounds_end) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");

                    
                    if (H5S_get_simple_extent_dims(dset->shared->space, tmp_dims, NULL) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get VDS dimensions");

                    
                    if (NULL == (curr_mapping->sub_dset[j].clipped_virtual_select =
                                     H5S_copy(curr_mapping->sub_dset[j].virtual_select, false, true)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy virtual selection");

                    
                    (void)memset(start, 0, sizeof(start));
                    if (H5S_select_hyperslab(curr_mapping->sub_dset[j].clipped_virtual_select, H5S_SELECT_AND,
                                             start, NULL, tmp_dims, NULL) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "unable to clip hyperslab");

                    
                    if (H5S_select_project_intersection(
                            curr_mapping->sub_dset[j].virtual_select, curr_mapping->source_select,
                            curr_mapping->sub_dset[j].clipped_virtual_select,
                            &curr_mapping->sub_dset[j].clipped_source_select, true) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                    "can't project virtual intersection onto memory space");

                    
                    if (H5S_set_extent(curr_mapping->sub_dset[j].virtual_select, tmp_dims) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");
                    if (H5S_set_extent(curr_mapping->sub_dset[j].clipped_virtual_select, tmp_dims) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");
                } 
            }     

            
            if (curr_mapping->sub_dset[j].clipped_virtual_select) {
                
                if (H5S_select_project_intersection(file_space, mem_space,
                                                    curr_mapping->sub_dset[j].clipped_virtual_select,
                                                    &curr_mapping->sub_dset[j].projected_mem_space, true) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                                "can't project virtual intersection onto memory space");

                
                if ((select_nelmts =
                         (hssize_t)H5S_GET_SELECT_NPOINTS(curr_mapping->sub_dset[j].projected_mem_space)) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOUNT, FAIL,
                                "unable to get number of elements in selection");

                
                if (select_nelmts > (hssize_t)0) {
                    
                    if (!curr_mapping->sub_dset[j].dset)
                        
                        if (H5D__virtual_open_source_dset(dset, curr_mapping, &curr_mapping->sub_dset[j]) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open source dataset");

                    
                    if (!curr_mapping->sub_dset[j].dset)
                        select_nelmts = (hssize_t)0;
                } 

                
                if (select_nelmts == (hssize_t)0) {
                    if (H5S_close(curr_mapping->sub_dset[j].projected_mem_space) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close projected memory space");
                    curr_mapping->sub_dset[j].projected_mem_space = NULL;
                } 
                else
                    *tot_nelmts += (hsize_t)select_nelmts;
            } 
        }     
    }         
    else {
        if (curr_mapping->source_dset.clipped_virtual_select) {
            
            if (H5S_select_project_intersection(file_space, mem_space,
                                                curr_mapping->source_dset.clipped_virtual_select,
                                                &curr_mapping->source_dset.projected_mem_space, true) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                            "can't project virtual intersection onto memory space");

            
            if ((select_nelmts =
                     (hssize_t)H5S_GET_SELECT_NPOINTS(curr_mapping->source_dset.projected_mem_space)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOUNT, FAIL,
                            "unable to get number of elements in selection");

            
            if (select_nelmts > (hssize_t)0) {
                
                if (!curr_mapping->source_dset.dset)
                    
                    if (H5D__virtual_open_source_dset(dset, curr_mapping, &curr_mapping->source_dset) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open source dataset");

                
                if (!curr_mapping->source_dset.dset)
                    select_nelmts = (hssize_t)0;
            } 

            
            if (select_nelmts == (hssize_t)0) {
                if (H5S_close(curr_mapping->source_dset.projected_mem_space) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close projected memory space");
                curr_mapping->source_dset.projected_mem_space = NULL;
            } 
            else
                *tot_nelmts += (hsize_t)select_nelmts;
        } 
        else {
            
            assert(curr_mapping->unlim_dim_virtual >= 0);
            assert(!curr_mapping->source_dset.dset);
        } 
    }     
done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_pre_io(H5D_dset_io_info_t *dset_info, H5O_storage_virtual_t *storage, H5S_t *file_space,
                    H5S_t *mem_space, hsize_t *tot_nelmts, H5RT_result_set_t *mappings)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(storage);
    assert(mem_space);
    assert(file_space);
    assert(tot_nelmts);

    
    *tot_nelmts = 0;

    
    if (mappings) {
        
        for (size_t i = 0; i < mappings->count; i++) {
            H5RT_leaf_t *curr_leaf = mappings->results[i];
            assert(curr_leaf);

            if (H5D__virtual_pre_io_process_mapping(dset_info, file_space, mem_space, tot_nelmts,
                                                    (H5O_storage_virtual_ent_t *)curr_leaf->record) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "can't process mapping for pre I/O");
        }

        
        for (size_t i = 0; i < storage->not_in_tree_nused; i++) {
            if (H5D__virtual_pre_io_process_mapping(dset_info, file_space, mem_space, tot_nelmts,
                                                    storage->not_in_tree_list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "can't process mapping for pre I/O");
        }
    }
    else {
        
        for (size_t i = 0; i < storage->list_nused; i++) {
            if (H5D__virtual_pre_io_process_mapping(dset_info, file_space, mem_space, tot_nelmts,
                                                    &storage->list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "can't process mapping for pre I/O");
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_post_io(H5O_storage_virtual_t *storage, H5RT_result_set_t *mappings)
{
    size_t i;                   
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(storage);

    if (mappings) {
        
        for (i = 0; i < mappings->count; i++) {
            H5RT_leaf_t *curr_leaf = mappings->results[i];
            assert(curr_leaf);

            if (H5D__virtual_close_mapping(curr_leaf->record) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "can't process mapping for pre I/O");
        }

        
        for (i = 0; i < storage->not_in_tree_nused; i++) {
            if (H5D__virtual_close_mapping(storage->not_in_tree_list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "can't process mapping for pre I/O");
        }
    }
    else {
        
        for (i = 0; i < storage->list_nused; i++)
            if (H5D__virtual_close_mapping(&storage->list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "failed to close mapping");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_read_one_src(H5D_dset_io_info_t *dset_info, H5O_storage_virtual_srcdset_t *source_dset)
{
    H5S_t             *projected_src_space = NULL; 
    H5D_dset_io_info_t source_dinfo;               
    herr_t             ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    assert(source_dset);

    
    if (source_dset->projected_mem_space) {
        assert(source_dset->dset);
        assert(source_dset->clipped_source_select);

        
        if (H5S_select_project_intersection(source_dset->clipped_virtual_select,
                                            source_dset->clipped_source_select, dset_info->file_space,
                                            &projected_src_space, true) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                        "can't project virtual intersection onto source space");

        {
            
            source_dinfo.dset       = source_dset->dset;
            source_dinfo.mem_space  = source_dset->projected_mem_space;
            source_dinfo.file_space = projected_src_space;
            source_dinfo.buf.vp     = dset_info->buf.vp;
            source_dinfo.mem_type   = dset_info->type_info.dst_type;

            
            if (H5D__read(1, &source_dinfo) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read source dataset");
        }

        
        if (H5S_close(projected_src_space) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close projected source space");
        projected_src_space = NULL;
    } 

done:
    
    if (projected_src_space) {
        assert(ret_value < 0);
        if (H5S_close(projected_src_space) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close projected source space");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_read(H5D_io_info_t H5_ATTR_NDEBUG_UNUSED *io_info, H5D_dset_io_info_t *dset_info)
{
    H5O_storage_virtual_t *storage;                     
    hsize_t                tot_nelmts;                  
    H5S_t                 *fill_space = NULL;           
    size_t                 nelmts;                      
    size_t                 i, j;                        
    herr_t                 ret_value         = SUCCEED; 
    bool                   should_build_tree = false;   
    H5RT_result_set_t     *mappings          = NULL;    
    hsize_t                min[H5S_MAX_RANK];
    hsize_t                max[H5S_MAX_RANK];

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->buf.vp);
    assert(dset_info->mem_space);
    assert(dset_info->file_space);

    storage = &(dset_info->dset->shared->layout.storage.u.virt);
    assert((storage->view == H5D_VDS_FIRST_MISSING) || (storage->view == H5D_VDS_LAST_AVAILABLE));

    
    nelmts = H5S_GET_SELECT_NPOINTS(dset_info->file_space);

    memset(min, 0, sizeof(min));
    memset(max, 0, sizeof(max));
#ifdef H5_HAVE_PARALLEL
    
    if (H5F_HAS_FEATURE(dset_info->dset->oloc.file, H5FD_FEAT_HAS_MPI))
        HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "parallel reads not supported on virtual datasets");
#endif 

    
    if (!storage->init)
        if (H5D__virtual_init_all(dset_info->dset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize virtual layout");

    if (H5D__should_build_tree(storage, dset_info->dset->shared->dapl_id, &should_build_tree) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't determine if should build VDS tree");

    if (should_build_tree) {
        int rank = 0;

        
        if ((rank = H5S_GET_EXTENT_NDIMS(dset_info->dset->shared->space)) < 0 || rank >= H5S_MAX_RANK)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get dataset rank");

        if (rank == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "virtual dataset has no rank");

        if (H5D__virtual_build_tree(storage, rank) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't build virtual mapping tree");
    }

    if (storage->tree) {
        
        if (H5S_SELECT_BOUNDS(dset_info->file_space, min, max) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection bounds");

        if (H5RT_search(storage->tree, min, max, &mappings) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "R-tree search failed");
    }

    
    if (H5D__virtual_pre_io(dset_info, storage, dset_info->file_space, dset_info->mem_space, &tot_nelmts,
                            mappings) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "unable to prepare for I/O operation");

    
    if (mappings) {
        
        for (i = 0; i < mappings->count; i++) {
            H5RT_leaf_t *curr_leaf = mappings->results[i];
            assert(curr_leaf);

            if (H5D__virtual_read_one_mapping(dset_info, curr_leaf->record) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read source dataset");
        }

        
        for (i = 0; i < storage->not_in_tree_nused; i++) {
            if (H5D__virtual_read_one_mapping(dset_info, storage->not_in_tree_list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read source dataset");
        }
    }
    else {
        
        for (i = 0; i < storage->list_nused; i++) {
            if (H5D__virtual_read_one_mapping(dset_info, &storage->list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read source dataset");
        } 
    }

    
    if (tot_nelmts < nelmts) {
        H5D_fill_value_t fill_status; 

        
        if (H5P_is_fill_value_defined(&dset_info->dset->shared->dcpl_cache.fill, &fill_status) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't tell if fill value defined");

        
        if (fill_status != H5D_FILL_VALUE_UNDEFINED) {
            
            if (NULL == (fill_space = H5S_copy(dset_info->mem_space, false, true)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy memory selection");

            
            for (i = 0; i < storage->list_nused; i++)
                
                if (storage->list[i].psfn_nsubs || storage->list[i].psdn_nsubs) {
                    
                    for (j = storage->list[i].sub_dset_io_start; j < storage->list[i].sub_dset_io_end; j++)
                        if (storage->list[i].sub_dset[j].projected_mem_space)
                            if (H5S_select_subtract(fill_space,
                                                    storage->list[i].sub_dset[j].projected_mem_space) < 0)
                                HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "unable to clip fill selection");
                } 
                else if (storage->list[i].source_dset.projected_mem_space)
                    
                    if (H5S_select_subtract(fill_space, storage->list[i].source_dset.projected_mem_space) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "unable to clip fill selection");

            
            if (H5D__fill(dset_info->dset->shared->dcpl_cache.fill.buf, dset_info->dset->shared->type,
                          dset_info->buf.vp, dset_info->type_info.mem_type, fill_space) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "filling buf failed");

#ifndef NDEBUG
            
            {
                hssize_t select_nelmts; 

                
                if ((select_nelmts = (hssize_t)H5S_GET_SELECT_NPOINTS(fill_space)) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOUNT, FAIL,
                                "unable to get number of elements in selection");

                
                assert((tot_nelmts + (hsize_t)select_nelmts) >= nelmts);
            } 
#endif        
        }     
    }         

done:
    
    if (H5D__virtual_post_io(storage, mappings) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't cleanup I/O operation");

    if (mappings)
        if (H5RT_free_results(mappings) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't free R-tree search results");

    
    if (fill_space)
        if (H5S_close(fill_space) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close fill space");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_write_one_src(H5D_dset_io_info_t *dset_info, H5O_storage_virtual_srcdset_t *source_dset)
{
    H5S_t             *projected_src_space = NULL; 
    H5D_dset_io_info_t source_dinfo;               
    herr_t             ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    assert(source_dset);

    
    if (source_dset->projected_mem_space) {
        assert(source_dset->dset);
        assert(source_dset->clipped_source_select);

        
        
        if (H5S_select_project_intersection(source_dset->clipped_virtual_select,
                                            source_dset->clipped_source_select, dset_info->file_space,
                                            &projected_src_space, true) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL,
                        "can't project virtual intersection onto source space");

        {
            
            source_dinfo.dset       = source_dset->dset;
            source_dinfo.mem_space  = source_dset->projected_mem_space;
            source_dinfo.file_space = projected_src_space;
            source_dinfo.buf.cvp    = dset_info->buf.cvp;
            source_dinfo.mem_type   = dset_info->type_info.dst_type;

            
            if (H5D__write(1, &source_dinfo) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read source dataset");
        }

        
        if (H5S_close(projected_src_space) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close projected source space");
        projected_src_space = NULL;
    } 

done:
    
    if (projected_src_space) {
        assert(ret_value < 0);
        if (H5S_close(projected_src_space) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close projected source space");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_write(H5D_io_info_t H5_ATTR_NDEBUG_UNUSED *io_info, H5D_dset_io_info_t *dset_info)
{
    H5O_storage_virtual_t *storage;                     
    hsize_t                tot_nelmts;                  
    size_t                 nelmts;                      
    size_t                 i;                           
    herr_t                 ret_value         = SUCCEED; 
    bool                   should_build_tree = false;   
    H5RT_result_set_t     *mappings          = NULL;    
    hsize_t                min[H5S_MAX_RANK];
    hsize_t                max[H5S_MAX_RANK];

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->buf.cvp);
    assert(dset_info->mem_space);
    assert(dset_info->file_space);

    storage = &(dset_info->dset->shared->layout.storage.u.virt);
    assert((storage->view == H5D_VDS_FIRST_MISSING) || (storage->view == H5D_VDS_LAST_AVAILABLE));

    
    nelmts = H5S_GET_SELECT_NPOINTS(dset_info->file_space);

    memset(min, 0, sizeof(min));
    memset(max, 0, sizeof(max));
#ifdef H5_HAVE_PARALLEL
    
    if (H5F_HAS_FEATURE(dset_info->dset->oloc.file, H5FD_FEAT_HAS_MPI))
        HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "parallel writes not supported on virtual datasets");
#endif 

    
    if (!storage->init)
        if (H5D__virtual_init_all(dset_info->dset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize virtual layout");

    if (H5D__should_build_tree(storage, dset_info->dset->shared->dapl_id, &should_build_tree) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't determine if should build VDS tree");

    if (should_build_tree) {
        int rank = 0;

        
        if ((rank = H5S_GET_EXTENT_NDIMS(dset_info->dset->shared->space)) < 0 || rank >= H5S_MAX_RANK)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get dataset rank");

        if (rank == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "virtual dataset has no rank");

        if (H5D__virtual_build_tree(storage, rank) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't build virtual mapping tree");
    }

    if (storage->tree) {
        
        if (H5S_SELECT_BOUNDS(dset_info->file_space, min, max) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get selection bounds");

        if (H5RT_search(storage->tree, min, max, &mappings) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "R-tree search failed");
    }

    
    if (H5D__virtual_pre_io(dset_info, storage, dset_info->file_space, dset_info->mem_space, &tot_nelmts,
                            mappings) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCLIP, FAIL, "unable to prepare for I/O operation");

    
    if (tot_nelmts != nelmts)
        HGOTO_ERROR(H5E_DATASPACE, H5E_BADVALUE, FAIL,
                    "write requested to unmapped portion of virtual dataset");

    if (mappings) {
        
        for (i = 0; i < mappings->count; i++) {
            H5RT_leaf_t *curr_leaf = mappings->results[i];
            assert(curr_leaf);

            if (H5D__virtual_write_one_mapping(dset_info, curr_leaf->record) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read source dataset");
        }

        
        for (i = 0; i < storage->not_in_tree_nused; i++) {
            if (H5D__virtual_write_one_mapping(dset_info, storage->not_in_tree_list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read source dataset");
        }
    }
    else {
        
        for (i = 0; i < storage->list_nused; i++) {
            if (H5D__virtual_write_one_mapping(dset_info, &storage->list[i]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "can't write to virtual mapping");
        }
    }

done:
    
    if (H5D__virtual_post_io(storage, mappings) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't cleanup I/O operation");

    if (mappings)
        H5RT_free_results(mappings);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_flush(H5D_t *dset)
{
    H5O_storage_virtual_t *storage;             
    size_t                 i, j;                
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    storage = &dset->shared->layout.storage.u.virt;

    
    for (i = 0; i < storage->list_nused; i++)
        
        if (storage->list[i].psfn_nsubs || storage->list[i].psdn_nsubs) {
            
            for (j = 0; j < storage->list[i].sub_dset_nused; j++)
                if (storage->list[i].sub_dset[j].dset)
                    
                    if (H5D__flush_real(storage->list[i].sub_dset[j].dset) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to flush source dataset");
        } 
        else if (storage->list[i].source_dset.dset)
            
            if (H5D__flush_real(storage->list[i].source_dset.dset) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to flush source dataset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_hold_source_dset_files(const H5D_t *dset, H5D_virtual_held_file_t **head)
{
    H5O_storage_virtual_t   *storage;             
    H5D_virtual_held_file_t *tmp;                 
    size_t                   i;                   
    herr_t                   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(head && NULL == *head);

    
    storage = &dset->shared->layout.storage.u.virt;

    
    for (i = 0; i < storage->list_nused; i++)
        
        if (storage->list[i].psfn_nsubs || storage->list[i].psdn_nsubs) {
            size_t j; 

            
            for (j = 0; j < storage->list[i].sub_dset_nused; j++)
                if (storage->list[i].sub_dset[j].dset) {
                    
                    H5F_INCR_NOPEN_OBJS(storage->list[i].sub_dset[j].dset->oloc.file);

                    
                    if (NULL == (tmp = H5FL_MALLOC(H5D_virtual_held_file_t)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate held file node");

                    
                    tmp->file = storage->list[i].sub_dset[j].dset->oloc.file;
                    tmp->next = *head;
                    *head     = tmp;
                } 
        }         
        else if (storage->list[i].source_dset.dset) {
            
            H5F_INCR_NOPEN_OBJS(storage->list[i].source_dset.dset->oloc.file);

            
            if (NULL == (tmp = H5FL_MALLOC(H5D_virtual_held_file_t)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate held file node");

            
            tmp->file = storage->list[i].source_dset.dset->oloc.file;
            tmp->next = *head;
            *head     = tmp;
        } 

done:
    if (ret_value < 0)
        
        if (*head && H5D__virtual_release_source_dset_files(*head) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't release source datasets' files held open");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_refresh_source_dset(H5D_t **dset)
{
    hid_t          temp_id   = H5I_INVALID_HID; 
    H5VL_object_t *vol_obj   = NULL;            
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(dset && *dset);

    
    if ((temp_id = H5VL_wrap_register(H5I_DATASET, *dset, false)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTREGISTER, FAIL, "can't register (temporary) source dataset ID");

    
    if (H5D__refresh(*dset, temp_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to refresh source dataset");

    
    if (NULL == (vol_obj = (H5VL_object_t *)H5I_remove(temp_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTREMOVE, FAIL, "can't unregister source dataset ID");
    if (NULL == (*dset = (H5D_t *)H5VL_object_unwrap(vol_obj)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve library object from VOL object");
    H5VL_OBJ_DATA_RESET(vol_obj);

done:
    if (vol_obj && H5VL_free_object(vol_obj) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "unable to free VOL object");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_refresh_source_dsets(H5D_t *dset)
{
    H5O_storage_virtual_t *storage;             
    size_t                 i;                   
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    storage = &dset->shared->layout.storage.u.virt;

    
    for (i = 0; i < storage->list_nused; i++)
        
        if (storage->list[i].psfn_nsubs || storage->list[i].psdn_nsubs) {
            size_t j; 

            
            for (j = 0; j < storage->list[i].sub_dset_nused; j++)
                
                if (storage->list[i].sub_dset[j].dset)
                    
                    if (H5D__virtual_refresh_source_dset(&storage->list[i].sub_dset[j].dset) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to refresh source dataset");
        } 
        else
            
            if (storage->list[i].source_dset.dset)
                
                if (H5D__virtual_refresh_source_dset(&storage->list[i].source_dset.dset) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to refresh source dataset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__virtual_release_source_dset_files(H5D_virtual_held_file_t *head)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    while (head) {
        H5D_virtual_held_file_t *tmp = head->next; 

        
        H5F_DECR_NOPEN_OBJS(head->file);

        
        
        if (H5F_try_close(head->file, NULL) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCLOSEFILE, FAIL, "problem attempting file close");

        
        (void)H5FL_FREE(H5D_virtual_held_file_t, head);

        
        head = tmp;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__mappings_to_leaves(H5O_storage_virtual_ent_t *mappings, size_t num_mappings, H5RT_leaf_t **leaves_out,
                        H5O_storage_virtual_ent_t ***not_in_tree_out, size_t *leaf_count,
                        size_t *not_in_tree_count, size_t *not_in_tree_nalloc)
{
    herr_t ret_value = SUCCEED;

    H5RT_leaf_t                *leaves_temp = NULL;
    H5O_storage_virtual_ent_t **not_in_tree = NULL;

    H5O_storage_virtual_ent_t *curr_mapping         = NULL;
    H5RT_leaf_t               *curr_leaf            = NULL;
    size_t                     curr_leaf_count      = 0;
    size_t                     curr_not_tree_count  = 0;
    size_t                     not_in_tree_capacity = 0;
    H5S_t                     *curr_space           = NULL;

    int rank = 0;

    FUNC_ENTER_PACKAGE

    assert(mappings);
    assert(num_mappings > 0);
    assert(leaf_count);
    assert(leaves_out);
    assert(not_in_tree_out);
    assert(not_in_tree_count);
    assert(not_in_tree_nalloc);

    
    if ((curr_space = mappings[0].source_dset.virtual_select) == NULL)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "first mapping has no virtual space");

    if ((rank = H5S_GET_EXTENT_NDIMS(curr_space)) < 0 || rank > H5S_MAX_RANK)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get rank of dataspace");

    if (rank == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "mapping has zero-dimensional space");

    
    if ((leaves_temp = (H5RT_leaf_t *)calloc(num_mappings, sizeof(H5RT_leaf_t))) == NULL)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate leaves array");

    
    not_in_tree_capacity = H5D_VIRTUAL_NOT_IN_TREE_INIT_SIZE;

    if (NULL == (not_in_tree = (H5O_storage_virtual_ent_t **)H5MM_malloc(
                     not_in_tree_capacity * sizeof(H5O_storage_virtual_ent_t *))))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "failed to allocate not_in_tree_list");

    for (size_t i = 0; i < num_mappings; i++) {
        curr_mapping = &mappings[i];

        if (!(H5D_RTREE_SHOULD_INSERT(curr_mapping))) {
            
            if (H5D__virtual_not_in_tree_add(&not_in_tree, &curr_not_tree_count, &not_in_tree_capacity,
                                             curr_mapping) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "failed to add to not_in_tree_list");
            continue;
        }

        
        curr_leaf = &leaves_temp[curr_leaf_count];
        if (H5RT_leaf_init(curr_leaf, rank, (void *)curr_mapping) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't initialize R-tree leaf");

        
        assert(mappings[i].source_dset.virtual_select);
        curr_space = mappings[i].source_dset.virtual_select;

        
        if (H5S_SELECT_BOUNDS(curr_space, curr_leaf->min, curr_leaf->max) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get selection bounds");

        for (int d = 0; d < rank; d++) {
            
            if (curr_leaf->min[d] > curr_leaf->max[d])
                HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "invalid selection bounds: min > max");
            curr_leaf->mid[d] = curr_leaf->min[d] + (curr_leaf->max[d] - curr_leaf->min[d]) / 2;
        }

        curr_leaf_count++;
    }

    *leaves_out         = leaves_temp;
    *leaf_count         = curr_leaf_count;
    *not_in_tree_out    = not_in_tree;
    *not_in_tree_count  = curr_not_tree_count;
    *not_in_tree_nalloc = not_in_tree_capacity;
done:
    if (ret_value < 0) {
        if (leaves_temp) {
            
            for (size_t j = 0; j < curr_leaf_count; j++) {
                H5RT_leaf_cleanup(&leaves_temp[j]);
            }
            free(leaves_temp);
        }
        if (not_in_tree)
            H5MM_free(not_in_tree);
    }

    FUNC_LEAVE_NOAPI(ret_value);
} 

static herr_t
H5D__virtual_build_tree(H5O_storage_virtual_t *virt, int rank)
{
    H5O_storage_virtual_ent_t *mappings     = virt->list;
    size_t                     num_mappings = virt->list_nused;

    H5RT_leaf_t                *leaves               = NULL;
    size_t                      num_leaves           = 0;
    H5O_storage_virtual_ent_t **not_in_tree_mappings = NULL;
    size_t                      not_in_tree_count    = 0;
    size_t                      not_in_tree_nalloc   = 0;
    herr_t                      ret_value            = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(virt);

    if (H5D__mappings_to_leaves(mappings, num_mappings, &leaves, &not_in_tree_mappings, &num_leaves,
                                &not_in_tree_count, &not_in_tree_nalloc) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to get leaves from mappings");

    if (num_leaves == 0) {
        
        virt->tree = NULL;
        if (leaves) {
            free(leaves);
            leaves = NULL;
        }
    }
    else {
        
        if ((virt->tree = H5RT_create(rank, leaves, num_leaves)) == NULL)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create mapping tree");
        
        leaves = NULL;
    }

    
    if (not_in_tree_count > 0) {
        virt->not_in_tree_list   = not_in_tree_mappings;
        virt->not_in_tree_nused  = not_in_tree_count;
        virt->not_in_tree_nalloc = not_in_tree_nalloc;
        not_in_tree_mappings     = NULL; 
    }
    else {
        
        if (virt->not_in_tree_list) {
            H5MM_free(virt->not_in_tree_list);
        }
        virt->not_in_tree_list   = NULL;
        virt->not_in_tree_nused  = 0;
        virt->not_in_tree_nalloc = 0;
        if (not_in_tree_mappings) {
            H5MM_free(not_in_tree_mappings);
            not_in_tree_mappings = NULL;
        }
    }

done:
    if (ret_value < 0) {
        if (leaves) {
            
            for (size_t i = 0; i < num_leaves; i++) {
                H5RT_leaf_cleanup(&leaves[i]);
            }
            free(leaves);
        }
        if (not_in_tree_mappings)
            H5MM_free(not_in_tree_mappings);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_not_in_tree_grow(H5O_storage_virtual_ent_t ***list, size_t *nalloc)
{
    size_t                      new_capacity = 0;
    H5O_storage_virtual_ent_t **new_list     = NULL;
    herr_t                      ret_value    = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(list);
    assert(*list);
    assert(nalloc);

    new_capacity = *nalloc * 2;

    
    if (new_capacity < *nalloc || new_capacity > (SIZE_MAX / sizeof(H5O_storage_virtual_ent_t *)))
        HGOTO_ERROR(H5E_DATASET, H5E_OVERFLOW, FAIL, "not_in_tree_list capacity overflow");

    if (NULL == (new_list = (H5O_storage_virtual_ent_t **)H5MM_realloc(
                     *list, new_capacity * sizeof(H5O_storage_virtual_ent_t *))))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "failed to grow not_in_tree_list");

    *list   = new_list;
    *nalloc = new_capacity;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_not_in_tree_add(H5O_storage_virtual_ent_t ***list, size_t *nused, size_t *nalloc,
                             H5O_storage_virtual_ent_t *mapping)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(list);
    assert(*list);
    assert(nused);
    assert(nalloc);
    assert(mapping);

    
    if (*nused >= *nalloc) {
        if (H5D__virtual_not_in_tree_grow(list, nalloc) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "failed to grow not_in_tree_list");
    }

    (*list)[*nused] = mapping;
    (*nused)++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__should_build_tree(H5O_storage_virtual_t *storage, hid_t dapl_id, bool *should_build_tree)
{
    herr_t          ret_value         = SUCCEED;
    H5P_genplist_t *dapl_plist        = NULL;
    bool            tree_enabled_dapl = false;

    FUNC_ENTER_PACKAGE

    assert(storage);
    assert(should_build_tree);
    assert(dapl_id != H5I_INVALID_HID);

    
    if (storage->tree) {
        *should_build_tree = false;
        HGOTO_DONE(SUCCEED);
    }

    
    if (storage->list_nused < H5D_VIRTUAL_TREE_THRESHOLD) {
        *should_build_tree = false;
        HGOTO_DONE(SUCCEED);
    }

    
    if (NULL == (dapl_plist = (H5P_genplist_t *)H5I_object(dapl_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for dapl ID");

    if (H5P_get(dapl_plist, H5D_ACS_USE_TREE_NAME, &tree_enabled_dapl) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get virtual use tree flag");

    if (!tree_enabled_dapl) {
        *should_build_tree = false;
        HGOTO_DONE(SUCCEED);
    }

    *should_build_tree = true;
done:
    FUNC_LEAVE_NOAPI(ret_value);
} 

static herr_t
H5D__virtual_read_one_mapping(H5D_dset_io_info_t *dset_info, H5O_storage_virtual_ent_t *mapping)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(mapping->virtual_space_status == H5O_VIRTUAL_STATUS_CORRECT);

    
    if (mapping->psfn_nsubs || mapping->psdn_nsubs) {
        
        for (size_t j = mapping->sub_dset_io_start; j < mapping->sub_dset_io_end; j++)
            if (H5D__virtual_read_one_src(dset_info, &mapping->sub_dset[j]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read source dataset");
    } 
    else
        
        if (H5D__virtual_read_one_src(dset_info, &mapping->source_dset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read source dataset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_write_one_mapping(H5D_dset_io_info_t *dset_info, H5O_storage_virtual_ent_t *mapping)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(mapping->virtual_space_status == H5O_VIRTUAL_STATUS_CORRECT);

    
    if (mapping->psfn_nsubs || mapping->psdn_nsubs) {
        
        for (size_t j = mapping->sub_dset_io_start; j < mapping->sub_dset_io_end; j++)
            if (H5D__virtual_write_one_src(dset_info, &mapping->sub_dset[j]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to write to source dataset");
    }
    else
        
        if (H5D__virtual_write_one_src(dset_info, &mapping->source_dset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to write to source dataset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__virtual_close_mapping(H5O_storage_virtual_ent_t *mapping)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    if (mapping->psfn_nsubs || mapping->psdn_nsubs) {
        
        for (size_t j = mapping->sub_dset_io_start; j < mapping->sub_dset_io_end; j++)
            
            if (mapping->sub_dset[j].projected_mem_space) {
                
                if (H5S_close(mapping->sub_dset[j].projected_mem_space) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close temporary space");
                mapping->sub_dset[j].projected_mem_space = NULL;
            } 
    }         
    else
        
        if (mapping->source_dset.projected_mem_space) {
            if (H5S_close(mapping->source_dset.projected_mem_space) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close temporary space");
            mapping->source_dset.projected_mem_space = NULL;
        } 

    FUNC_LEAVE_NOAPI(ret_value);
} 
