/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 

#include "H5private.h"   
#include "H5Cpkg.h"      
#include "H5Clog.h"      
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 

#define H5C_MAX_JSON_LOG_MSG_SIZE 1024

typedef struct H5C_log_json_udata_t {
    FILE *outfile;
    char *message;
} H5C_log_json_udata_t;

static herr_t H5C__json_write_log_message(H5C_log_json_udata_t *json_udata);

static herr_t H5C__json_tear_down_logging(H5C_log_info_t *log_info);
static herr_t H5C__json_write_start_log_msg(void *udata);
static herr_t H5C__json_write_stop_log_msg(void *udata);
static herr_t H5C__json_write_create_cache_log_msg(void *udata, herr_t fxn_ret_value);
static herr_t H5C__json_write_destroy_cache_log_msg(void *udata);
static herr_t H5C__json_write_evict_cache_log_msg(void *udata, herr_t fxn_ret_value);
static herr_t H5C__json_write_expunge_entry_log_msg(void *udata, haddr_t address, int type_id,
                                                    herr_t fxn_ret_value);
static herr_t H5C__json_write_flush_cache_log_msg(void *udata, herr_t fxn_ret_value);
static herr_t H5C__json_write_insert_entry_log_msg(void *udata, haddr_t address, int type_id, unsigned flags,
                                                   size_t size, herr_t fxn_ret_value);
static herr_t H5C__json_write_mark_entry_dirty_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                       herr_t fxn_ret_value);
static herr_t H5C__json_write_mark_entry_clean_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                       herr_t fxn_ret_value);
static herr_t H5C__json_write_mark_unserialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                              herr_t fxn_ret_value);
static herr_t H5C__json_write_mark_serialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                            herr_t fxn_ret_value);
static herr_t H5C__json_write_move_entry_log_msg(void *udata, haddr_t old_addr, haddr_t new_addr, int type_id,
                                                 herr_t fxn_ret_value);
static herr_t H5C__json_write_pin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                herr_t fxn_ret_value);
static herr_t H5C__json_write_create_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                                const H5C_cache_entry_t *child, herr_t fxn_ret_value);
static herr_t H5C__json_write_protect_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, int type_id,
                                                    unsigned flags, herr_t fxn_ret_value);
static herr_t H5C__json_write_resize_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                   size_t new_size, herr_t fxn_ret_value);
static herr_t H5C__json_write_unpin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                  herr_t fxn_ret_value);
static herr_t H5C__json_write_destroy_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                                 const H5C_cache_entry_t *child, herr_t fxn_ret_value);
static herr_t H5C__json_write_unprotect_entry_log_msg(void *udata, haddr_t address, int type_id,
                                                      unsigned flags, herr_t fxn_ret_value);
static herr_t H5C__json_write_set_cache_config_log_msg(void *udata, const H5AC_cache_config_t *config,
                                                       herr_t fxn_ret_value);
static herr_t H5C__json_write_remove_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                   herr_t fxn_ret_value);

static const H5C_log_class_t H5C_json_log_class_g = {"json",
                                                     H5C__json_tear_down_logging,
                                                     NULL, 
                                                     NULL, 
                                                     H5C__json_write_start_log_msg,
                                                     H5C__json_write_stop_log_msg,
                                                     H5C__json_write_create_cache_log_msg,
                                                     H5C__json_write_destroy_cache_log_msg,
                                                     H5C__json_write_evict_cache_log_msg,
                                                     H5C__json_write_expunge_entry_log_msg,
                                                     H5C__json_write_flush_cache_log_msg,
                                                     H5C__json_write_insert_entry_log_msg,
                                                     H5C__json_write_mark_entry_dirty_log_msg,
                                                     H5C__json_write_mark_entry_clean_log_msg,
                                                     H5C__json_write_mark_unserialized_entry_log_msg,
                                                     H5C__json_write_mark_serialized_entry_log_msg,
                                                     H5C__json_write_move_entry_log_msg,
                                                     H5C__json_write_pin_entry_log_msg,
                                                     H5C__json_write_create_fd_log_msg,
                                                     H5C__json_write_protect_entry_log_msg,
                                                     H5C__json_write_resize_entry_log_msg,
                                                     H5C__json_write_unpin_entry_log_msg,
                                                     H5C__json_write_destroy_fd_log_msg,
                                                     H5C__json_write_unprotect_entry_log_msg,
                                                     H5C__json_write_set_cache_config_log_msg,
                                                     H5C__json_write_remove_entry_log_msg};

static herr_t
H5C__json_write_log_message(H5C_log_json_udata_t *json_udata)
{
    size_t n_chars;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->outfile);
    assert(json_udata->message);

    
    n_chars = strlen(json_udata->message);
    if ((int)n_chars != Rfprintf(json_udata->outfile, "%s", json_udata->message))
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "error writing log message");
    memset((void *)(json_udata->message), 0, (size_t)(n_chars * sizeof(char)));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__log_json_set_up(H5C_log_info_t *log_info, const char log_location[], int mpi_rank)
{
    H5C_log_json_udata_t *json_udata = NULL;
    char                 *file_name  = NULL;
    size_t                n_chars;
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(log_info);
    assert(log_location);

    
    log_info->cls = &H5C_json_log_class_g;

    
    if (NULL == (log_info->udata = H5MM_calloc(sizeof(H5C_log_json_udata_t))))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "memory allocation failed");
    json_udata = (H5C_log_json_udata_t *)(log_info->udata);

    
    if (NULL == (json_udata->message = (char *)H5MM_calloc(H5C_MAX_JSON_LOG_MSG_SIZE * sizeof(char))))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    n_chars = 5 + 39 + 1 + strlen(log_location) + 1;
    if (NULL == (file_name = (char *)H5MM_calloc(n_chars * sizeof(char))))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL,
                    "can't allocate memory for mdc log file name manipulation");

    
    if (-1 == mpi_rank)
        snprintf(file_name, n_chars, "%s", log_location);
    else
        snprintf(file_name, n_chars, "RANK_%d.%s", mpi_rank, log_location);

    
    if (NULL == (json_udata->outfile = fopen(file_name, "w")))
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "can't create mdc log file");
    setbuf(json_udata->outfile, NULL);

done:
    if (file_name)
        H5MM_xfree(file_name);

    
    if (FAIL == ret_value) {
        
        if (json_udata && json_udata->message)
            H5MM_xfree(json_udata->message);
        if (json_udata)
            H5MM_xfree(json_udata);

        
        log_info->udata = NULL;
        log_info->cls   = NULL;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_tear_down_logging(H5C_log_info_t *log_info)
{
    H5C_log_json_udata_t *json_udata = NULL;
    herr_t                ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(log_info);

    
    json_udata = (H5C_log_json_udata_t *)(log_info->udata);

    
    H5MM_xfree(json_udata->message);

    
    if (EOF == fclose(json_udata->outfile))
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "problem closing mdc log file");
    json_udata->outfile = NULL;

    
    H5MM_xfree(json_udata);

    
    log_info->cls   = NULL;
    log_info->udata = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_start_log_msg(void *udata)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\n\
\"HDF5 metadata cache log messages\" : [\n\
{\
\"timestamp\":%lld,\
\"action\":\"logging start\"\
},\n\
",
             (long long)time(NULL));

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_stop_log_msg(void *udata)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"logging stop\"\
}\n\
]}\n\
",
             (long long)time(NULL));

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_create_cache_log_msg(void *udata, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"create\",\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_destroy_cache_log_msg(void *udata)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"destroy\"\
},\n\
",
             (long long)time(NULL));

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_evict_cache_log_msg(void *udata, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"evict\",\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_expunge_entry_log_msg(void *udata, haddr_t address, int type_id, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"expunge\",\
\"address\":0x%lx,\
\"type_id\":%d,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)address, (int)type_id, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_flush_cache_log_msg(void *udata, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"flush\",\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_insert_entry_log_msg(void *udata, haddr_t address, int type_id, unsigned flags, size_t size,
                                     herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"insert\",\
\"address\":0x%lx,\
\"type_id\":%d,\
\"flags\":0x%x,\
\"size\":%d,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)address, type_id, flags, (int)size, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_mark_entry_dirty_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"dirty\",\
\"address\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_mark_entry_clean_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"clean\",\
\"address\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_mark_unserialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                                herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"unserialized\",\
\"address\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_mark_serialized_entry_log_msg(void *udata, const H5C_cache_entry_t *entry,
                                              herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"serialized\",\
\"address\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_move_entry_log_msg(void *udata, haddr_t old_addr, haddr_t new_addr, int type_id,
                                   herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"move\",\
\"old_address\":0x%lx,\
\"new_address\":0x%lx,\
\"type_id\":%d,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)old_addr, (unsigned long)new_addr, type_id,
             (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_pin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"pin\",\
\"address\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_create_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                  const H5C_cache_entry_t *child, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(parent);
    assert(child);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"create_fd\",\
\"parent_addr\":0x%lx,\
\"child_addr\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)parent->addr, (unsigned long)child->addr,
             (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_protect_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, int type_id,
                                      unsigned flags, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    char                  rw_s[16];
    herr_t                ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    if (H5C__READ_ONLY_FLAG == flags)
        strcpy(rw_s, "READ");
    else
        strcpy(rw_s, "WRITE");

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"protect\",\
\"address\":0x%lx,\
\"type_id\":%d,\
\"readwrite\":\"%s\",\
\"size\":%d,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, type_id, rw_s, (int)entry->size,
             (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_resize_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, size_t new_size,
                                     herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"resize\",\
\"address\":0x%lx,\
\"new_size\":%d,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)new_size, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_unpin_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"unpin\",\
\"address\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_destroy_fd_log_msg(void *udata, const H5C_cache_entry_t *parent,
                                   const H5C_cache_entry_t *child, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(parent);
    assert(child);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"destroy_fd\",\
\"parent_addr\":0x%lx,\
\"child_addr\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)parent->addr, (unsigned long)child->addr,
             (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_unprotect_entry_log_msg(void *udata, haddr_t address, int type_id, unsigned flags,
                                        herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"unprotect\",\
\"address\":0x%lx,\
\"id\":%d,\
\"flags\":%x,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)address, type_id, flags, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_set_cache_config_log_msg(void *udata, const H5AC_cache_config_t H5_ATTR_NDEBUG_UNUSED *config,
                                         herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(config);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"set_config\",\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__json_write_remove_entry_log_msg(void *udata, const H5C_cache_entry_t *entry, herr_t fxn_ret_value)
{
    H5C_log_json_udata_t *json_udata = (H5C_log_json_udata_t *)(udata);
    herr_t                ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(json_udata);
    assert(json_udata->message);
    assert(entry);

    
    snprintf(json_udata->message, H5C_MAX_JSON_LOG_MSG_SIZE, "\
{\
\"timestamp\":%lld,\
\"action\":\"remove\",\
\"address\":0x%lx,\
\"returned\":%d\
},\n\
",
             (long long)time(NULL), (unsigned long)entry->addr, (int)fxn_ret_value);

    
    if (H5C__json_write_log_message(json_udata) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
